# JsonDoc
<span class="source-link">[[Source]](src/json/json_doc.md#L4)</span>

Top level JSON type containing an entire document.
A JSON document consists of exactly 1 value.


```pony
class ref JsonDoc
```

## Constructors

### create
<span class="source-link">[[Source]](src/json/json_doc.md#L24)</span>


Default constructor building a document containing a single null.


```pony
new ref create()
: JsonDoc ref^
```

#### Returns

* [JsonDoc](json-JsonDoc.md) ref^

---

## Public fields

### var data: ([F64](builtin-F64.md) val | [I64](builtin-I64.md) val | [Bool](builtin-Bool.md) val | 
    [None](builtin-None.md) val | [String](builtin-String.md) val | [JsonArray](json-JsonArray.md) ref | 
    [JsonObject](json-JsonObject.md) ref)
<span class="source-link">[[Source]](src/json/json_doc.md#L9)</span>

The parsed JSON structure.

Will be a `None` if `parse(source: String)` has not been called yet.




---

## Public Functions

### string
<span class="source-link">[[Source]](src/json/json_doc.md#L30)</span>


Generate string representation of this document.


```pony
fun box string(
  indent: String val = "",
  pretty_print: Bool val = false)
: String val
```
#### Parameters

*   indent: [String](builtin-String.md) val = ""
*   pretty_print: [Bool](builtin-Bool.md) val = false

#### Returns

* [String](builtin-String.md) val

---

### parse
<span class="source-link">[[Source]](src/json/json_doc.md#L40)</span>


Parse the given string as a JSON file, building a document.
Raise error on invalid JSON in given source.


```pony
fun ref parse(
  source: String val)
: None val ?
```
#### Parameters

*   source: [String](builtin-String.md) val

#### Returns

* [None](builtin-None.md) val ?

---

### parse_report
<span class="source-link">[[Source]](src/json/json_doc.md#L62)</span>


Give details of the error that occurred last time we attempted to parse.
If parse was successful returns (0, "").


```pony
fun box parse_report()
: (USize val , String val)
```

#### Returns

* ([USize](builtin-USize.md) val , [String](builtin-String.md) val)

---

## Private Functions

### _parse_value
<span class="source-link">[[Source]](src/json/json_doc.md#L69)</span>


Parse a single JSON value of any type, which MUST be present.
Raise error on invalid or missing value.


```pony
fun ref _parse_value(
  context: String val)
: (F64 val | I64 val | Bool val | 
    None val | String val | JsonArray ref | 
    JsonObject ref) ?
```
#### Parameters

*   context: [String](builtin-String.md) val

#### Returns

* ([F64](builtin-F64.md) val | [I64](builtin-I64.md) val | [Bool](builtin-Bool.md) val | 
    [None](builtin-None.md) val | [String](builtin-String.md) val | [JsonArray](json-JsonArray.md) ref | 
    [JsonObject](json-JsonObject.md) ref) ?

---

### _parse_keyword
<span class="source-link">[[Source]](src/json/json_doc.md#L87)</span>


Parse a keyword, the first letter of which has already been peeked.


```pony
fun ref _parse_keyword()
: (Bool val | None val) ?
```

#### Returns

* ([Bool](builtin-Bool.md) val | [None](builtin-None.md) val) ?

---

### _parse_number
<span class="source-link">[[Source]](src/json/json_doc.md#L108)</span>


Parse a number, the leading character of which has already been peeked.


```pony
fun ref _parse_number()
: (F64 val | I64 val) ?
```

#### Returns

* ([F64](builtin-F64.md) val | [I64](builtin-I64.md) val) ?

---

### _parse_decimal
<span class="source-link">[[Source]](src/json/json_doc.md#L169)</span>


Parse a decimal integer which must appear immediately in the source.


```pony
fun ref _parse_decimal()
: (I64 val , U8 val) ?
```

#### Returns

* ([I64](builtin-I64.md) val , [U8](builtin-U8.md) val) ?

---

### _parse_object
<span class="source-link">[[Source]](src/json/json_doc.md#L191)</span>


Parse a JSON object, the leading { of which has already been peeked.


```pony
fun ref _parse_object()
: JsonObject ref ?
```

#### Returns

* [JsonObject](json-JsonObject.md) ref ?

---

### _parse_array
<span class="source-link">[[Source]](src/json/json_doc.md#L234)</span>


Parse an array, the leading [ of which has already been peeked.


```pony
fun ref _parse_array()
: JsonArray ref ?
```

#### Returns

* [JsonArray](json-JsonArray.md) ref ?

---

### _parse_string
<span class="source-link">[[Source]](src/json/json_doc.md#L267)</span>


Parse a string, which must be the next thing found, other than whitesapce.


```pony
fun ref _parse_string(
  context: String val)
: String val ?
```
#### Parameters

*   context: [String](builtin-String.md) val

#### Returns

* [String](builtin-String.md) val ?

---

### _parse_escape
<span class="source-link">[[Source]](src/json/json_doc.md#L291)</span>


Process a string escape sequence, the leading \ of which has already been
consumed.


```pony
fun ref _parse_escape()
: String val ?
```

#### Returns

* [String](builtin-String.md) val ?

---

### _parse_unicode_escape
<span class="source-link">[[Source]](src/json/json_doc.md#L311)</span>


Process a Unicode escape sequence, the leading \u of which has already been
consumed.


```pony
fun ref _parse_unicode_escape()
: String val ?
```

#### Returns

* [String](builtin-String.md) val ?

---

### _parse_unicode_digits
<span class="source-link">[[Source]](src/json/json_doc.md#L343)</span>


Parse the hex digits of a Unicode escape sequence, the leading \u of which
has already been consumed, and return the encoded character value.


```pony
fun ref _parse_unicode_digits()
: U32 val ?
```

#### Returns

* [U32](builtin-U32.md) val ?

---

### _dump_whitespace
<span class="source-link">[[Source]](src/json/json_doc.md#L369)</span>


Dump all whitespace at the current read location in source, if any.


```pony
fun ref _dump_whitespace()
: None val
```

#### Returns

* [None](builtin-None.md) val

---

### _peek_char
<span class="source-link">[[Source]](src/json/json_doc.md#L389)</span>


Peek the next char in the source, without consuming it.
If an eof_context is given then an error is thrown on eof, setting a
suitable message.
If eof_context is None then 0 is returned on EOF. It up to the caller to
handle this appropriately.


```pony
fun ref _peek_char(
  eof_context: (String val | None val) = reference)
: U8 val ?
```
#### Parameters

*   eof_context: ([String](builtin-String.md) val | [None](builtin-None.md) val) = reference

#### Returns

* [U8](builtin-U8.md) val ?

---

### _get_char
<span class="source-link">[[Source]](src/json/json_doc.md#L416)</span>


Get and consume the next char in the source.
If an eof_context is given then an error is thrown on eof, setting a
suitable message.
If eof_context is None then 0 is returned on EOF. It up to the caller to
handle this appropriately.


```pony
fun ref _get_char(
  eof_context: (String val | None val) = reference)
: U8 val ?
```
#### Parameters

*   eof_context: ([String](builtin-String.md) val | [None](builtin-None.md) val) = reference

#### Returns

* [U8](builtin-U8.md) val ?

---

### _last_char
<span class="source-link">[[Source]](src/json/json_doc.md#L433)</span>


Get the last character peeked or got from the source as a String.
For use generating error messages.


```pony
fun ref _last_char()
: String val
```

#### Returns

* [String](builtin-String.md) val

---

### _error
<span class="source-link">[[Source]](src/json/json_doc.md#L444)</span>


Record an error with the given message.


```pony
fun ref _error(
  msg: String val)
: None val
```
#### Parameters

*   msg: [String](builtin-String.md) val

#### Returns

* [None](builtin-None.md) val

---

