/// <reference types="node" />
import { AllPublishOptions, CancellationToken, PublishConfiguration, UpdateInfo, DownloadOptions } from "builder-util-runtime";
import { EventEmitter } from "events";
import { OutgoingHttpHeaders } from "http";
import { Lazy } from "lazy-val";
import { SemVer } from "semver";
import { AppAdapter } from "./AppAdapter";
import { DownloadedUpdateHelper } from "./DownloadedUpdateHelper";
import { Logger, Provider, ResolvedUpdateFileInfo, UpdateCheckResult, UpdateDownloadedEvent, UpdaterSignal } from "./main";
import { ProviderPlatform } from "./providers/Provider";
import Session = Electron.Session;
export declare abstract class AppUpdater extends EventEmitter {
    /**
     * Whether to automatically download an update when it is found.
     */
    autoDownload: boolean;
    /**
     * Whether to automatically install a downloaded update on app quit (if `quitAndInstall` was not called before).
     *
     * Applicable only on Windows and Linux.
     */
    autoInstallOnAppQuit: boolean;
    /**
     * *GitHub provider only.* Whether to allow update to pre-release versions. Defaults to `true` if application version contains prerelease components (e.g. `0.12.1-alpha.1`, here `alpha` is a prerelease component), otherwise `false`.
     *
     * If `true`, downgrade will be allowed (`allowDowngrade` will be set to `true`).
     */
    allowPrerelease: boolean;
    /**
     * *GitHub provider only.* Get all release notes (from current version to latest), not just the latest.
     * @default false
     */
    fullChangelog: boolean;
    /**
     * Whether to allow version downgrade (when a user from the beta channel wants to go back to the stable channel).
     *
     * Taken in account only if channel differs (pre-release version component in terms of semantic versioning).
     *
     * @default false
     */
    allowDowngrade: boolean;
    /**
     * The current application version.
     */
    readonly currentVersion: SemVer;
    private _channel;
    protected downloadedUpdateHelper: DownloadedUpdateHelper | null;
    /**
     * Get the update channel. Not applicable for GitHub. Doesn't return `channel` from the update configuration, only if was previously set.
     */
    get channel(): string | null;
    /**
     * Set the update channel. Not applicable for GitHub. Overrides `channel` in the update configuration.
     *
     * `allowDowngrade` will be automatically set to `true`. If this behavior is not suitable for you, simple set `allowDowngrade` explicitly after.
     */
    set channel(value: string | null);
    /**
     *  The request headers.
     */
    requestHeaders: OutgoingHttpHeaders | null;
    protected _logger: Logger;
    get netSession(): Session;
    /**
     * The logger. You can pass [electron-log](https://github.com/megahertz/electron-log), [winston](https://github.com/winstonjs/winston) or another logger with the following interface: `{ info(), warn(), error() }`.
     * Set it to `null` if you would like to disable a logging feature.
     */
    get logger(): Logger | null;
    set logger(value: Logger | null);
    /**
     * For type safety you can use signals, e.g. `autoUpdater.signals.updateDownloaded(() => {})` instead of `autoUpdater.on('update-available', () => {})`
     */
    readonly signals: UpdaterSignal;
    private _appUpdateConfigPath;
    /**
     * test only
     * @private
     */
    set updateConfigPath(value: string | null);
    private clientPromise;
    protected readonly stagingUserIdPromise: Lazy<string>;
    private checkForUpdatesPromise;
    protected readonly app: AppAdapter;
    protected updateInfoAndProvider: UpdateInfoAndProvider | null;
    protected constructor(options: AllPublishOptions | null | undefined, app?: AppAdapter);
    getFeedURL(): string | null | undefined;
    /**
     * Configure update provider. If value is `string`, [GenericServerOptions](/configuration/publish#genericserveroptions) will be set with value as `url`.
     * @param options If you want to override configuration in the `app-update.yml`.
     */
    setFeedURL(options: PublishConfiguration | AllPublishOptions | string): void;
    /**
     * Asks the server whether there is an update.
     */
    checkForUpdates(): Promise<UpdateCheckResult>;
    isUpdaterActive(): boolean;
    checkForUpdatesAndNotify(downloadNotification?: DownloadNotification): Promise<UpdateCheckResult | null>;
    private formatDownloadNotification;
    private isStagingMatch;
    private computeFinalHeaders;
    private isUpdateAvailable;
    protected getUpdateInfoAndProvider(): Promise<UpdateInfoAndProvider>;
    private createProviderRuntimeOptions;
    private doCheckForUpdates;
    protected onUpdateAvailable(updateInfo: UpdateInfo): void;
    /**
     * Start downloading update manually. You can use this method if `autoDownload` option is set to `false`.
     * @returns {Promise<string>} Path to downloaded file.
     */
    downloadUpdate(cancellationToken?: CancellationToken): Promise<any>;
    protected dispatchError(e: Error): void;
    protected dispatchUpdateDownloaded(event: UpdateDownloadedEvent): void;
    protected abstract doDownloadUpdate(downloadUpdateOptions: DownloadUpdateOptions): Promise<Array<string>>;
    /**
     * Restarts the app and installs the update after it has been downloaded.
     * It should only be called after `update-downloaded` has been emitted.
     *
     * **Note:** `autoUpdater.quitAndInstall()` will close all application windows first and only emit `before-quit` event on `app` after that.
     * This is different from the normal quit event sequence.
     *
     * @param isSilent *windows-only* Runs the installer in silent mode. Defaults to `false`.
     * @param isForceRunAfter Run the app after finish even on silent install. Not applicable for macOS. Ignored if `isSilent` is set to `false`.
     */
    abstract quitAndInstall(isSilent?: boolean, isForceRunAfter?: boolean): void;
    private loadUpdateConfig;
    private computeRequestHeaders;
    private getOrCreateStagingUserId;
    private getOrCreateDownloadHelper;
    protected executeDownload(taskOptions: DownloadExecutorTask): Promise<Array<string>>;
}
export interface DownloadUpdateOptions {
    readonly updateInfoAndProvider: UpdateInfoAndProvider;
    readonly requestHeaders: OutgoingHttpHeaders;
    readonly cancellationToken: CancellationToken;
}
/** @private */
export declare class NoOpLogger implements Logger {
    info(message?: any): void;
    warn(message?: any): void;
    error(message?: any): void;
}
export interface UpdateInfoAndProvider {
    info: UpdateInfo;
    provider: Provider<any>;
}
export interface DownloadExecutorTask {
    readonly fileExtension: string;
    readonly fileInfo: ResolvedUpdateFileInfo;
    readonly downloadUpdateOptions: DownloadUpdateOptions;
    readonly task: (destinationFile: string, downloadOptions: DownloadOptions, packageFile: string | null, removeTempDirIfAny: () => Promise<any>) => Promise<any>;
    readonly done?: (event: UpdateDownloadedEvent) => Promise<any>;
}
export interface DownloadNotification {
    body: string;
    title: string;
}
/** @private */
export interface TestOnlyUpdaterOptions {
    platform: ProviderPlatform;
    isUseDifferentialDownload?: boolean;
}
