package Lire::Config::CompoundSpec;

use strict;

=pod

=head1 Lire::Config::CompoundSpec

Describes an abstract type for configuration variables that can
contain subtypes.

=cut

use base qw/ Lire::Config::TypeSpec /;

use Lire::Utils qw/ check_param check_object_param /;
use Lire::DataTypes qw/check_xml_name/;
use Lire::Config::Value;

use Carp;

sub new {
    my $class = shift;
    my %args = @_;

    my $self = $class->SUPER::new( %args );

    $self->{'label'} = $args{'label'} if exists ( $args{'label'} );
    $self->{'components'} = [];
    $self->{'index'} = {};

    return $self;
}

=pod

=head2 has_component( $name )

Returns true if this type has a component named $name.

=cut

sub has_component {
    my ( $self, $name ) = @_;

    check_param( $name, 'name' );

    return exists $self->{'index'}{$name};
}

=pod

=head2 component_names()

Returns the name of all components defined in this spec.

=cut

sub component_names {
    return map { $_->name() } $_[0]->components();
}

=pod

=head2 components()

Returns an array containing the subtypes contained in this compound type.

=cut

sub components {
    return @{$_[0]{'components'}};
}

=pod

=head2 label_component()

Returns the component name that holds the label for identifying instances in GUI.

=cut

sub label_component {
    my $self = $_[0];

    return undef unless defined $self->{'label'};

    croak "'$self->{'label'}' component does not exist"
      unless $self->has_component( $self->{'label'} );

    return $self->{'label'};
}

=pod

=head2 add( $type )

This method adds $type as a component of this type.

=cut

sub add {
    my ($self, $type) = @_;

    check_object_param( $type, 'type', 'Lire::Config::TypeSpec' );

    my $type_name = $type->name();
    croak( "'" . $self->name()
           . "' already contains a component named '$type_name'" )
      if exists $self->{'index'}{ $type_name };

    $self->{'index'}{$type_name} =
      push ( @{$self->{'components'}}, $type ) - 1;

    return;
}

=pod

=head2 get( $name )

The get() method returns the component of this type with the
requested name.

=cut

sub get {
    my ( $self, $name ) = @_;

    check_param( $name, 'name', \&check_xml_name,
                 "invalid 'name' parameter" );

    croak "no component named '$name' in type $self->{'name'}"
      unless exists $self->{'index'}{$name};

    my $idx = $self->{'index'}{$name};

    return $self->{'components'}[$idx];
}

1;
__END__

=pod

=head1 VERSION

$Id: CompoundSpec.pm,v 1.9 2006/07/23 13:16:30 vanbaal Exp $

=head1 AUTHORS

Francis J. Lacoste <flacoste@logreport.org>
Wessel Dankers <wsl@logreport.org>
Wolfgang Sourdeau <wolfgang@logreport.org>

=head1 COPYRIGHT

Copyright (C) 2002-2004  Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html.

=cut
