"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.sendConversationList = void 0;
const electron_better_ipc_1 = require("electron-better-ipc");
const elementReady = require("element-ready");
const selectors_1 = require("./selectors");
const icon = {
    read: 'data-caprine-icon',
    unread: 'data-caprine-icon-unread'
};
const padding = {
    top: 3,
    right: 0,
    bottom: 3,
    left: 0
};
function drawIcon(size, img) {
    const canvas = document.createElement('canvas');
    if (img) {
        canvas.width = size + padding.left + padding.right;
        canvas.height = size + padding.top + padding.bottom;
        const ctx = canvas.getContext('2d');
        ctx.beginPath();
        ctx.arc((size / 2) + padding.left, (size / 2) + padding.top, (size / 2), 0, Math.PI * 2, true);
        ctx.closePath();
        ctx.clip();
        ctx.drawImage(img, padding.left, padding.top, size, size);
    }
    else {
        canvas.width = 0;
        canvas.height = 0;
    }
    return canvas;
}
// Return canvas with rounded image
async function urlToCanvas(url, size) {
    return new Promise(resolve => {
        const img = new Image();
        img.setAttribute('crossorigin', 'anonymous');
        img.addEventListener('load', () => {
            resolve(drawIcon(size, img));
        });
        img.addEventListener('error', () => {
            console.error('Image not found', url);
            resolve(drawIcon(size));
        });
        img.src = url;
    });
}
async function createIcons(element, url) {
    const canvas = await urlToCanvas(url, 50);
    element.setAttribute(icon.read, canvas.toDataURL());
    const markerSize = 8;
    const ctx = canvas.getContext('2d');
    ctx.fillStyle = '#f42020';
    ctx.beginPath();
    ctx.ellipse(canvas.width - markerSize, markerSize, markerSize, markerSize, 0, 0, 2 * Math.PI);
    ctx.closePath();
    ctx.fill();
    element.setAttribute(icon.unread, canvas.toDataURL());
}
async function discoverIcons(element) {
    const profilePicElement = element.querySelector('img:first-of-type');
    if (profilePicElement) {
        return createIcons(element, profilePicElement.src);
    }
    const groupPicElement = element.firstElementChild;
    if (groupPicElement) {
        const groupPicBackground = groupPicElement.style.backgroundImage;
        if (groupPicBackground) {
            return createIcons(element, groupPicBackground.replace(/^url\(["']?(.*?)["']?\)$/, '$1'));
        }
    }
    console.warn('Could not discover profile picture. Falling back to default image.');
    // Fall back to messenger favicon
    const messengerIcon = document.querySelector('link[rel~="icon"]');
    if (messengerIcon) {
        return createIcons(element, messengerIcon.getAttribute('href'));
    }
    // Fall back to facebook favicon
    return createIcons(element, 'https://facebook.com/favicon.ico');
}
async function getIcon(element, unread) {
    if (!element.getAttribute(icon.read)) {
        await discoverIcons(element);
    }
    return element.getAttribute(unread ? icon.unread : icon.read);
}
async function createConversation(element) {
    const conversation = {};
    const muted = element.classList.contains('_569x');
    conversation.selected = element.classList.contains('_1ht2');
    conversation.unread = !muted && element.getAttribute('aria-live') !== null;
    const profileElement = element.querySelector('div[data-tooltip-content]');
    conversation.label = profileElement.getAttribute('data-tooltip-content');
    conversation.icon = await getIcon(profileElement, conversation.unread);
    return conversation;
}
async function createConversationList() {
    const list = await elementReady(selectors_1.default.conversationList, {
        stopOnDomReady: false
    });
    if (!list) {
        console.error('Could not find conversation list', selectors_1.default.conversationList);
        return [];
    }
    const elements = [...list.children];
    const conversations = await Promise.all(elements.map(async (element) => createConversation(element)));
    return conversations;
}
async function sendConversationList() {
    const conversationsToRender = await createConversationList();
    electron_better_ipc_1.ipcRenderer.callMain('conversations', conversationsToRender);
}
exports.sendConversationList = sendConversationList;
window.addEventListener('load', async () => {
    const sidebar = document.querySelector('[role=navigation]');
    if (sidebar) {
        const conversationListObserver = new MutationObserver(sendConversationList);
        conversationListObserver.observe(sidebar, {
            subtree: true,
            childList: true,
            attributes: true,
            attributeFilter: ['class']
        });
    }
});
