'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});

var _extends = Object.assign || function (target) { for (var i = 1; i < arguments.length; i++) { var source = arguments[i]; for (var key in source) { if (Object.prototype.hasOwnProperty.call(source, key)) { target[key] = source[key]; } } } return target; }; /**
                                                                                                                                                                                                                                                                   *  Copyright (c) Facebook, Inc.
                                                                                                                                                                                                                                                                   *  All rights reserved.
                                                                                                                                                                                                                                                                   *
                                                                                                                                                                                                                                                                   *  This source code is licensed under the license found in the
                                                                                                                                                                                                                                                                   *  LICENSE file in the root directory of this source tree.
                                                                                                                                                                                                                                                                   *
                                                                                                                                                                                                                                                                   *  
                                                                                                                                                                                                                                                                   */

exports.getAutocompleteSuggestions = getAutocompleteSuggestions;
exports.getTokenAtPosition = getTokenAtPosition;
exports.getTypeInfo = getTypeInfo;

var _graphql = require('graphql');

var _graphqlLanguageServiceParser = require('graphql-language-service-parser');

var _autocompleteUtils = require('./autocompleteUtils');

/**
 * Given GraphQLSchema, queryText, and context of the current position within
 * the source text, provide a list of typeahead entries.
 */
function getAutocompleteSuggestions(schema, queryText, cursor, contextToken) {
  var token = contextToken || getTokenAtPosition(queryText, cursor);

  var state = token.state.kind === 'Invalid' ? token.state.prevState : token.state;

  // relieve flow errors by checking if `state` exists
  if (!state) {
    return [];
  }

  var kind = state.kind;
  var step = state.step;
  var typeInfo = getTypeInfo(schema, token.state);

  // Definition kinds
  if (kind === 'Document') {
    return (0, _autocompleteUtils.hintList)(token, [{ label: 'query' }, { label: 'mutation' }, { label: 'subscription' }, { label: 'fragment' }, { label: '{' }]);
  }

  // Field names
  if (kind === 'SelectionSet' || kind === 'Field' || kind === 'AliasedField') {
    return getSuggestionsForFieldNames(token, typeInfo, schema);
  }

  // Argument names
  if (kind === 'Arguments' || kind === 'Argument' && step === 0) {
    var argDefs = typeInfo.argDefs;
    if (argDefs) {
      return (0, _autocompleteUtils.hintList)(token, argDefs.map(function (argDef) {
        return {
          label: argDef.name,
          detail: String(argDef.type),
          documentation: argDef.description
        };
      }));
    }
  }

  // Input Object fields
  if (kind === 'ObjectValue' || kind === 'ObjectField' && step === 0) {
    if (typeInfo.objectFieldDefs) {
      var objectFields = (0, _autocompleteUtils.objectValues)(typeInfo.objectFieldDefs);
      return (0, _autocompleteUtils.hintList)(token, objectFields.map(function (field) {
        return {
          label: field.name,
          detail: String(field.type),
          documentation: field.description
        };
      }));
    }
  }

  // Input values: Enum and Boolean
  if (kind === 'EnumValue' || kind === 'ListValue' && step === 1 || kind === 'ObjectField' && step === 2 || kind === 'Argument' && step === 2) {
    return getSuggestionsForInputValues(token, typeInfo);
  }

  // Fragment type conditions
  if (kind === 'TypeCondition' && step === 1 || kind === 'NamedType' && state.prevState != null && state.prevState.kind === 'TypeCondition') {
    return getSuggestionsForFragmentTypeConditions(token, typeInfo, schema);
  }

  // Fragment spread names
  if (kind === 'FragmentSpread' && step === 1) {
    return getSuggestionsForFragmentSpread(token, typeInfo, schema, queryText);
  }

  // Variable definition types
  if (kind === 'VariableDefinition' && step === 2 || kind === 'ListType' && step === 1 || kind === 'NamedType' && state.prevState && (state.prevState.kind === 'VariableDefinition' || state.prevState.kind === 'ListType')) {
    return getSuggestionsForVariableDefinition(token, schema);
  }

  // Directive names
  if (kind === 'Directive') {
    return getSuggestionsForDirective(token, state, schema);
  }

  return [];
}

// Helper functions to get suggestions for each kinds
function getSuggestionsForFieldNames(token, typeInfo, schema) {
  if (typeInfo.parentType) {
    var parentType = typeInfo.parentType;
    var fields = parentType.getFields instanceof Function ? (0, _autocompleteUtils.objectValues)(parentType.getFields()) : [];
    if ((0, _graphql.isAbstractType)(parentType)) {
      fields.push(_graphql.TypeNameMetaFieldDef);
    }
    if (parentType === schema.getQueryType()) {
      fields.push(_graphql.SchemaMetaFieldDef, _graphql.TypeMetaFieldDef);
    }
    return (0, _autocompleteUtils.hintList)(token, fields.map(function (field) {
      return {
        label: field.name,
        detail: String(field.type),
        documentation: field.description,
        isDeprecated: field.isDeprecated,
        deprecationReason: field.deprecationReason
      };
    }));
  }
  return [];
}

function getSuggestionsForInputValues(token, typeInfo) {
  var namedInputType = (0, _graphql.getNamedType)(typeInfo.inputType);
  if (namedInputType instanceof _graphql.GraphQLEnumType) {
    var values = namedInputType.getValues();
    return (0, _autocompleteUtils.hintList)(token, values.map(function (value) {
      return {
        label: value.name,
        detail: String(namedInputType),
        documentation: value.description,
        isDeprecated: value.isDeprecated,
        deprecationReason: value.deprecationReason
      };
    }));
  } else if (namedInputType === _graphql.GraphQLBoolean) {
    return (0, _autocompleteUtils.hintList)(token, [{
      label: 'true',
      detail: String(_graphql.GraphQLBoolean),
      documentation: 'Not false.'
    }, {
      label: 'false',
      detail: String(_graphql.GraphQLBoolean),
      documentation: 'Not true.'
    }]);
  }

  return [];
}

function getSuggestionsForFragmentTypeConditions(token, typeInfo, schema) {
  var possibleTypes = void 0;
  if (typeInfo.parentType) {
    if ((0, _graphql.isAbstractType)(typeInfo.parentType)) {
      var abstractType = (0, _graphql.assertAbstractType)(typeInfo.parentType);
      // Collect both the possible Object types as well as the interfaces
      // they implement.
      var possibleObjTypes = schema.getPossibleTypes(abstractType);
      var possibleIfaceMap = Object.create(null);
      possibleObjTypes.forEach(function (type) {
        type.getInterfaces().forEach(function (iface) {
          possibleIfaceMap[iface.name] = iface;
        });
      });
      possibleTypes = possibleObjTypes.concat((0, _autocompleteUtils.objectValues)(possibleIfaceMap));
    } else {
      // The parent type is a non-abstract Object type, so the only possible
      // type that can be used is that same type.
      possibleTypes = [typeInfo.parentType];
    }
  } else {
    var typeMap = schema.getTypeMap();
    possibleTypes = (0, _autocompleteUtils.objectValues)(typeMap).filter(_graphql.isCompositeType);
  }
  return (0, _autocompleteUtils.hintList)(token, possibleTypes.map(function (type) {
    var namedType = (0, _graphql.getNamedType)(type);
    return {
      label: String(type),
      documentation: namedType && namedType.description || ''
    };
  }));
}

function getSuggestionsForFragmentSpread(token, typeInfo, schema, queryText) {
  var typeMap = schema.getTypeMap();
  var defState = (0, _autocompleteUtils.getDefinitionState)(token.state);
  var fragments = getFragmentDefinitions(queryText);

  // Filter down to only the fragments which may exist here.
  var relevantFrags = fragments.filter(function (frag) {
    return (
      // Only include fragments with known types.
      typeMap[frag.typeCondition.name.value] &&
      // Only include fragments which are not cyclic.
      !(defState && defState.kind === 'FragmentDefinition' && defState.name === frag.name.value) &&
      // Only include fragments which could possibly be spread here.
      (0, _graphql.isCompositeType)(typeInfo.parentType) && (0, _graphql.isCompositeType)(typeMap[frag.typeCondition.name.value]) && (0, _graphql.doTypesOverlap)(schema, typeInfo.parentType, typeMap[frag.typeCondition.name.value])
    );
  });

  return (0, _autocompleteUtils.hintList)(token, relevantFrags.map(function (frag) {
    return {
      label: frag.name.value,
      detail: String(typeMap[frag.typeCondition.name.value]),
      documentation: 'fragment ' + frag.name.value + ' on ' + frag.typeCondition.name.value
    };
  }));
}

function getFragmentDefinitions(queryText) {
  var fragmentDefs = [];
  runOnlineParser(queryText, function (_, state) {
    if (state.kind === 'FragmentDefinition' && state.name && state.type) {
      fragmentDefs.push({
        kind: 'FragmentDefinition',
        name: {
          kind: 'Name',
          value: state.name
        },
        selectionSet: {
          kind: 'SelectionSet',
          selections: []
        },
        typeCondition: {
          kind: 'NamedType',
          name: {
            kind: 'Name',
            value: state.type
          }
        }
      });
    }
  });

  return fragmentDefs;
}

function getSuggestionsForVariableDefinition(token, schema) {
  var inputTypeMap = schema.getTypeMap();
  var inputTypes = (0, _autocompleteUtils.objectValues)(inputTypeMap).filter(_graphql.isInputType);
  return (0, _autocompleteUtils.hintList)(token, inputTypes.map(function (type) {
    return {
      label: type.name,
      documentation: type.description
    };
  }));
}

function getSuggestionsForDirective(token, state, schema) {
  if (state.prevState && state.prevState.kind) {
    var directives = schema.getDirectives().filter(function (directive) {
      return canUseDirective(state.prevState, directive);
    });
    return (0, _autocompleteUtils.hintList)(token, directives.map(function (directive) {
      return {
        label: directive.name,
        documentation: directive.description || ''
      };
    }));
  }
  return [];
}

function getTokenAtPosition(queryText, cursor) {
  var styleAtCursor = null;
  var stateAtCursor = null;
  var stringAtCursor = null;
  var token = runOnlineParser(queryText, function (stream, state, style, index) {
    if (index === cursor.line) {
      if (stream.getCurrentPosition() >= cursor.character) {
        styleAtCursor = style;
        stateAtCursor = _extends({}, state);
        stringAtCursor = stream.current();
        return 'BREAK';
      }
    }
  });

  // Return the state/style of parsed token in case those at cursor aren't
  // available.
  return {
    start: token.start,
    end: token.end,
    string: stringAtCursor || token.string,
    state: stateAtCursor || token.state,
    style: styleAtCursor || token.style
  };
}

/**
 * Provides an utility function to parse a given query text and construct a
 * `token` context object.
 * A token context provides useful information about the token/style that
 * CharacterStream currently possesses, as well as the end state and style
 * of the token.
 */


function runOnlineParser(queryText, callback) {
  var lines = queryText.split('\n');
  var parser = (0, _graphqlLanguageServiceParser.onlineParser)();
  var state = parser.startState();
  var style = '';

  var stream = new _graphqlLanguageServiceParser.CharacterStream('');

  for (var i = 0; i < lines.length; i++) {
    stream = new _graphqlLanguageServiceParser.CharacterStream(lines[i]);
    while (!stream.eol()) {
      style = parser.token(stream, state);
      var code = callback(stream, state, style, i);
      if (code === 'BREAK') {
        break;
      }
    }

    // Above while loop won't run if there is an empty line.
    // Run the callback one more time to catch this.
    callback(stream, state, style, i);

    if (!state.kind) {
      state = parser.startState();
    }
  }

  return {
    start: stream.getStartOfToken(),
    end: stream.getCurrentPosition(),
    string: stream.current(),
    state: state,
    style: style
  };
}

function canUseDirective(state, directive) {
  if (!state || !state.kind) {
    return false;
  }
  var kind = state.kind;
  var locations = directive.locations;
  switch (kind) {
    case 'Query':
      return locations.indexOf('QUERY') !== -1;
    case 'Mutation':
      return locations.indexOf('MUTATION') !== -1;
    case 'Subscription':
      return locations.indexOf('SUBSCRIPTION') !== -1;
    case 'Field':
    case 'AliasedField':
      return locations.indexOf('FIELD') !== -1;
    case 'FragmentDefinition':
      return locations.indexOf('FRAGMENT_DEFINITION') !== -1;
    case 'FragmentSpread':
      return locations.indexOf('FRAGMENT_SPREAD') !== -1;
    case 'InlineFragment':
      return locations.indexOf('INLINE_FRAGMENT') !== -1;

    // Schema Definitions
    case 'SchemaDef':
      return locations.indexOf('SCHEMA') !== -1;
    case 'ScalarDef':
      return locations.indexOf('SCALAR') !== -1;
    case 'ObjectTypeDef':
      return locations.indexOf('OBJECT') !== -1;
    case 'FieldDef':
      return locations.indexOf('FIELD_DEFINITION') !== -1;
    case 'InterfaceDef':
      return locations.indexOf('INTERFACE') !== -1;
    case 'UnionDef':
      return locations.indexOf('UNION') !== -1;
    case 'EnumDef':
      return locations.indexOf('ENUM') !== -1;
    case 'EnumValue':
      return locations.indexOf('ENUM_VALUE') !== -1;
    case 'InputDef':
      return locations.indexOf('INPUT_OBJECT') !== -1;
    case 'InputValueDef':
      var prevStateKind = state.prevState && state.prevState.kind;
      switch (prevStateKind) {
        case 'ArgumentsDef':
          return locations.indexOf('ARGUMENT_DEFINITION') !== -1;
        case 'InputDef':
          return locations.indexOf('INPUT_FIELD_DEFINITION') !== -1;
      }
  }
  return false;
}

// Utility for collecting rich type information given any token's state
// from the graphql-mode parser.
function getTypeInfo(schema, tokenState) {
  var argDef = void 0;
  var argDefs = void 0;
  var directiveDef = void 0;
  var enumValue = void 0;
  var fieldDef = void 0;
  var inputType = void 0;
  var objectFieldDefs = void 0;
  var parentType = void 0;
  var type = void 0;

  (0, _autocompleteUtils.forEachState)(tokenState, function (state) {
    switch (state.kind) {
      case 'Query':
      case 'ShortQuery':
        type = schema.getQueryType();
        break;
      case 'Mutation':
        type = schema.getMutationType();
        break;
      case 'Subscription':
        type = schema.getSubscriptionType();
        break;
      case 'InlineFragment':
      case 'FragmentDefinition':
        if (state.type) {
          type = schema.getType(state.type);
        }
        break;
      case 'Field':
      case 'AliasedField':
        if (!type || !state.name) {
          fieldDef = null;
        } else {
          fieldDef = parentType ? (0, _autocompleteUtils.getFieldDef)(schema, parentType, state.name) : null;
          type = fieldDef ? fieldDef.type : null;
        }
        break;
      case 'SelectionSet':
        parentType = (0, _graphql.getNamedType)(type);
        break;
      case 'Directive':
        directiveDef = state.name ? schema.getDirective(state.name) : null;
        break;
      case 'Arguments':
        if (!state.prevState) {
          argDefs = null;
        } else {
          switch (state.prevState.kind) {
            case 'Field':
              argDefs = fieldDef && fieldDef.args;
              break;
            case 'Directive':
              argDefs = directiveDef && directiveDef.args;
              break;
            case 'AliasedField':
              var name = state.prevState && state.prevState.name;
              if (!name) {
                argDefs = null;
                break;
              }
              var field = parentType ? (0, _autocompleteUtils.getFieldDef)(schema, parentType, name) : null;
              if (!field) {
                argDefs = null;
                break;
              }
              argDefs = field.args;
              break;
            default:
              argDefs = null;
              break;
          }
        }
        break;
      case 'Argument':
        if (argDefs) {
          for (var i = 0; i < argDefs.length; i++) {
            if (argDefs[i].name === state.name) {
              argDef = argDefs[i];
              break;
            }
          }
        }
        inputType = argDef && argDef.type;
        break;
      case 'EnumValue':
        var enumType = (0, _graphql.getNamedType)(inputType);
        enumValue = enumType instanceof _graphql.GraphQLEnumType ? find(enumType.getValues(), function (val) {
          return val.value === state.name;
        }) : null;
        break;
      case 'ListValue':
        var nullableType = (0, _graphql.getNullableType)(inputType);
        inputType = nullableType instanceof _graphql.GraphQLList ? nullableType.ofType : null;
        break;
      case 'ObjectValue':
        var objectType = (0, _graphql.getNamedType)(inputType);
        objectFieldDefs = objectType instanceof _graphql.GraphQLInputObjectType ? objectType.getFields() : null;
        break;
      case 'ObjectField':
        var objectField = state.name && objectFieldDefs ? objectFieldDefs[state.name] : null;
        inputType = objectField && objectField.type;
        break;
      case 'NamedType':
        if (state.name) {
          type = schema.getType(state.name);
        }
        break;
    }
  });

  return {
    argDef: argDef,
    argDefs: argDefs,
    directiveDef: directiveDef,
    enumValue: enumValue,
    fieldDef: fieldDef,
    inputType: inputType,
    objectFieldDefs: objectFieldDefs,
    parentType: parentType,
    type: type
  };
}

// Returns the first item in the array which causes predicate to return truthy.
function find(array, predicate) {
  for (var i = 0; i < array.length; i++) {
    if (predicate(array[i])) {
      return array[i];
    }
  }
  return null;
}