/****************************************************************************
**
** Copyright (C) 2020 Klarälvdalens Datakonsult AB, a KDAB Group company, info@kdab.com, author Marc Mutz <marc.mutz@kdab.com>
** Contact: http://www.qt.io/licensing/
**
** This file is part of the QtCore module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see https://www.qt.io/terms-conditions. For further
** information use the contact form at https://www.qt.io/contact-us.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 3 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL3 included in the
** packaging of this file. Please review the following information to
** ensure the GNU Lesser General Public License version 3 requirements
** will be met: https://www.gnu.org/licenses/lgpl-3.0.html.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 2.0 or (at your option) the GNU General
** Public license version 3 or any later version approved by the KDE Free
** Qt Foundation. The licenses are as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL2 and LICENSE.GPL3
** included in the packaging of this file. Please review the following
** information to ensure the GNU General Public License requirements will
** be met: https://www.gnu.org/licenses/gpl-2.0.html and
** https://www.gnu.org/licenses/gpl-3.0.html.
**
** $QT_END_LICENSE$
**
****************************************************************************/
#ifndef QDUPLICATETRACKER_P_H
#define QDUPLICATETRACKER_P_H

//
//  W A R N I N G
//  -------------
//
// This file is not part of the Qt API.  It exists purely as an
// implementation detail.  This header file may change from version to
// version without notice, or even be removed.
//
// We mean it.
//

#include <qglobal.h>

#if QT_HAS_INCLUDE(<memory_resource>) && __cplusplus > 201402L
#  include <unordered_set>
#  include <memory_resource>
#else
#  include <qset.h>
#endif

QT_BEGIN_NAMESPACE

template <typename T, size_t Prealloc = 32>
class QDuplicateTracker {
#ifdef __cpp_lib_memory_resource
    char buffer[Prealloc * sizeof(T)];
    std::pmr::monotonic_buffer_resource res{buffer, sizeof buffer};
    std::pmr::unordered_set<T> set{&res};
#else
    QSet<T> set;
    int setSize = 0;
#endif
    Q_DISABLE_COPY_MOVE(QDuplicateTracker);
public:
    QDuplicateTracker() = default;
    void reserve(int n) { set.reserve(n); }
    [[nodiscard]] bool hasSeen(const T &s)
    {
        bool inserted;
#ifdef __cpp_lib_memory_resource
        inserted = set.insert(s).second;
#else
        set.insert(s);
        const int n = set.size();
        inserted = qExchange(setSize, n) != n;
#endif
        return !inserted;
    }
    [[nodiscard]] bool hasSeen(T &&s)
    {
        bool inserted;
#ifdef __cpp_lib_memory_resource
        inserted = set.insert(std::move(s)).second;
#else
        set.insert(std::move(s));
        const int n = set.size();
        inserted = qExchange(setSize, n) != n;
#endif
        return !inserted;
    }

    template <typename C>
    void appendTo(C &c) const &
    {
        for (const auto &e : set)
            c.push_back(e);
    }

    template <typename C>
    void appendTo(C &c) &&
    {
        for (auto &e : set)
            c.push_back(std::move(e));
    }
};

QT_END_NAMESPACE

#endif /* QDUPLICATETRACKER_P_H */
