/*
 * Decompiled with CFR 0.152.
 */
package jadx.api;

import jadx.api.CodePosition;
import jadx.api.ICodeInfo;
import jadx.api.JadxArgs;
import jadx.api.JadxArgsValidator;
import jadx.api.JavaClass;
import jadx.api.JavaField;
import jadx.api.JavaMethod;
import jadx.api.JavaNode;
import jadx.api.JavaPackage;
import jadx.api.ResourceFile;
import jadx.api.ResourceType;
import jadx.api.ResourcesLoader;
import jadx.api.plugins.JadxPlugin;
import jadx.api.plugins.JadxPluginManager;
import jadx.api.plugins.input.JadxInputPlugin;
import jadx.api.plugins.input.data.ILoadResult;
import jadx.core.Jadx;
import jadx.core.dex.attributes.AFlag;
import jadx.core.dex.attributes.nodes.LineAttrNode;
import jadx.core.dex.nodes.ClassNode;
import jadx.core.dex.nodes.FieldNode;
import jadx.core.dex.nodes.MethodNode;
import jadx.core.dex.nodes.RootNode;
import jadx.core.dex.visitors.SaveCode;
import jadx.core.export.ExportGradleProject;
import jadx.core.utils.Utils;
import jadx.core.utils.exceptions.JadxRuntimeException;
import jadx.core.xmlgen.BinaryXMLParser;
import jadx.core.xmlgen.ResourcesSaver;
import java.io.Closeable;
import java.io.File;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class JadxDecompiler
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(JadxDecompiler.class);
    private final JadxArgs args;
    private final JadxPluginManager pluginManager = new JadxPluginManager();
    private final List<ILoadResult> loadedInputs = new ArrayList<ILoadResult>();
    private RootNode root;
    private List<JavaClass> classes;
    private List<ResourceFile> resources;
    private BinaryXMLParser xmlParser;
    private final Map<ClassNode, JavaClass> classesMap = new ConcurrentHashMap<ClassNode, JavaClass>();
    private final Map<MethodNode, JavaMethod> methodsMap = new ConcurrentHashMap<MethodNode, JavaMethod>();
    private final Map<FieldNode, JavaField> fieldsMap = new ConcurrentHashMap<FieldNode, JavaField>();

    public JadxDecompiler() {
        this(new JadxArgs());
    }

    public JadxDecompiler(JadxArgs args) {
        this.args = args;
    }

    public void load() {
        this.reset();
        JadxArgsValidator.validate(this.args);
        LOG.info("loading ...");
        this.loadInputFiles();
        this.root = new RootNode(this.args);
        this.root.loadClasses(this.loadedInputs);
        this.root.initClassPath();
        this.root.loadResources(this.getResources());
        this.root.runPreDecompileStage();
        this.root.initPasses();
    }

    private void loadInputFiles() {
        this.loadedInputs.clear();
        List<Path> inputPaths = Utils.collectionMap(this.args.getInputFiles(), File::toPath);
        for (JadxInputPlugin inputPlugin : this.pluginManager.getInputPlugins()) {
            ILoadResult loadResult = inputPlugin.loadFiles(inputPaths);
            if (loadResult == null || loadResult.isEmpty()) continue;
            this.loadedInputs.add(loadResult);
        }
    }

    private void reset() {
        this.root = null;
        this.classes = null;
        this.resources = null;
        this.xmlParser = null;
        this.classesMap.clear();
        this.methodsMap.clear();
        this.fieldsMap.clear();
        this.closeInputs();
    }

    private void closeInputs() {
        this.loadedInputs.forEach(load -> {
            try {
                load.close();
            }
            catch (Exception e) {
                LOG.error("Failed to close input", (Throwable)e);
            }
        });
        this.loadedInputs.clear();
    }

    @Override
    public void close() {
        this.reset();
    }

    public void registerPlugin(JadxPlugin plugin) {
        this.pluginManager.register(plugin);
    }

    public static String getVersion() {
        return Jadx.getVersion();
    }

    public void save() {
        this.save(!this.args.isSkipSources(), !this.args.isSkipResources());
    }

    public void saveSources() {
        this.save(true, false);
    }

    public void saveResources() {
        this.save(false, true);
    }

    private void save(boolean saveSources, boolean saveResources) {
        ExecutorService ex = this.getSaveExecutor(saveSources, saveResources);
        ex.shutdown();
        try {
            ex.awaitTermination(1L, TimeUnit.DAYS);
        }
        catch (InterruptedException e) {
            LOG.error("Save interrupted", (Throwable)e);
            Thread.currentThread().interrupt();
        }
    }

    public ExecutorService getSaveExecutor() {
        return this.getSaveExecutor(!this.args.isSkipSources(), !this.args.isSkipResources());
    }

    private ExecutorService getSaveExecutor(boolean saveSources, boolean saveResources) {
        File resOutDir;
        File sourcesOutDir;
        if (this.root == null) {
            throw new JadxRuntimeException("No loaded files");
        }
        int threadsCount = this.args.getThreadsCount();
        LOG.debug("processing threads count: {}", (Object)threadsCount);
        LOG.info("processing ...");
        ExecutorService executor = Executors.newFixedThreadPool(threadsCount);
        if (this.args.isExportAsGradleProject()) {
            ExportGradleProject export = new ExportGradleProject(this.root, this.args.getOutDir());
            export.init();
            sourcesOutDir = export.getSrcOutDir();
            resOutDir = export.getResOutDir();
        } else {
            sourcesOutDir = this.args.getOutDirSrc();
            resOutDir = this.args.getOutDirRes();
        }
        if (saveResources) {
            this.appendResourcesSave(executor, resOutDir);
        }
        if (saveSources) {
            this.appendSourcesSave(executor, sourcesOutDir);
        }
        return executor;
    }

    private void appendResourcesSave(ExecutorService executor, File outDir) {
        Set inputFileNames = this.args.getInputFiles().stream().map(File::getAbsolutePath).collect(Collectors.toSet());
        for (ResourceFile resourceFile : this.getResources()) {
            if (resourceFile.getType() != ResourceType.ARSC && inputFileNames.contains(resourceFile.getOriginalName())) continue;
            executor.execute(new ResourcesSaver(outDir, resourceFile));
        }
    }

    private void appendSourcesSave(ExecutorService executor, File outDir) {
        Predicate<String> classFilter = this.args.getClassFilter();
        for (JavaClass cls : this.getClasses()) {
            if (cls.getClassNode().contains(AFlag.DONT_GENERATE) || classFilter != null && !classFilter.test(cls.getFullName())) continue;
            executor.execute(() -> {
                try {
                    ICodeInfo code = cls.getCodeInfo();
                    SaveCode.save(outDir, cls.getClassNode(), code);
                }
                catch (Exception e) {
                    LOG.error("Error saving class: {}", (Object)cls.getFullName(), (Object)e);
                }
            });
        }
    }

    public List<JavaClass> getClasses() {
        if (this.root == null) {
            return Collections.emptyList();
        }
        if (this.classes == null) {
            List<ClassNode> classNodeList = this.root.getClasses(false);
            ArrayList<JavaClass> clsList = new ArrayList<JavaClass>(classNodeList.size());
            this.classesMap.clear();
            for (ClassNode classNode : classNodeList) {
                if (classNode.contains(AFlag.DONT_GENERATE)) continue;
                JavaClass javaClass = new JavaClass(classNode, this);
                clsList.add(javaClass);
                this.classesMap.put(classNode, javaClass);
            }
            this.classes = Collections.unmodifiableList(clsList);
        }
        return this.classes;
    }

    public List<ResourceFile> getResources() {
        if (this.resources == null) {
            if (this.root == null) {
                return Collections.emptyList();
            }
            this.resources = new ResourcesLoader(this).load();
        }
        return this.resources;
    }

    public List<JavaPackage> getPackages() {
        List<JavaClass> classList = this.getClasses();
        if (classList.isEmpty()) {
            return Collections.emptyList();
        }
        HashMap<String, List> map = new HashMap<String, List>();
        for (JavaClass javaClass : classList) {
            String string = javaClass.getPackage();
            List clsList = map.computeIfAbsent(string, k -> new ArrayList());
            clsList.add(javaClass);
        }
        ArrayList<JavaPackage> packages = new ArrayList<JavaPackage>(map.size());
        for (Map.Entry entry : map.entrySet()) {
            packages.add(new JavaPackage((String)entry.getKey(), (List)entry.getValue()));
        }
        Collections.sort(packages);
        for (JavaPackage javaPackage : packages) {
            javaPackage.getClasses().sort(Comparator.comparing(JavaClass::getName, String.CASE_INSENSITIVE_ORDER));
        }
        return Collections.unmodifiableList(packages);
    }

    public int getErrorsCount() {
        if (this.root == null) {
            return 0;
        }
        return this.root.getErrorsCounter().getErrorCount();
    }

    public int getWarnsCount() {
        if (this.root == null) {
            return 0;
        }
        return this.root.getErrorsCounter().getWarnsCount();
    }

    public void printErrorsReport() {
        if (this.root == null) {
            return;
        }
        this.root.getClsp().printMissingClasses();
        this.root.getErrorsCounter().printReport();
    }

    public RootNode getRoot() {
        return this.root;
    }

    synchronized BinaryXMLParser getXmlParser() {
        if (this.xmlParser == null) {
            this.xmlParser = new BinaryXMLParser(this.root);
        }
        return this.xmlParser;
    }

    private void loadJavaClass(JavaClass javaClass) {
        javaClass.getMethods().forEach(mth -> this.methodsMap.put(mth.getMethodNode(), (JavaMethod)mth));
        javaClass.getFields().forEach(fld -> this.fieldsMap.put(fld.getFieldNode(), (JavaField)fld));
        for (JavaClass innerCls : javaClass.getInnerClasses()) {
            this.classesMap.put(innerCls.getClassNode(), innerCls);
            this.loadJavaClass(innerCls);
        }
    }

    @Nullable(value="For not generated classes")
    private @Nullable(value="For not generated classes") JavaClass getJavaClassByNode(ClassNode cls) {
        JavaClass javaClass = this.classesMap.get(cls);
        if (javaClass != null) {
            return javaClass;
        }
        ClassNode parentClass = cls.getTopParentClass();
        if (parentClass.contains(AFlag.DONT_GENERATE)) {
            return null;
        }
        if (parentClass != cls) {
            JavaClass parentJavaClass = this.classesMap.get(parentClass);
            if (parentJavaClass == null) {
                this.getClasses();
                parentJavaClass = this.classesMap.get(parentClass);
            }
            this.loadJavaClass(parentJavaClass);
            javaClass = this.classesMap.get(cls);
            if (javaClass != null) {
                return javaClass;
            }
        }
        if (cls.hasNotGeneratedParent()) {
            return null;
        }
        throw new JadxRuntimeException("JavaClass not found by ClassNode: " + cls);
    }

    @Nullable
    private JavaMethod getJavaMethodByNode(MethodNode mth) {
        JavaMethod javaMethod = this.methodsMap.get(mth);
        if (javaMethod != null) {
            return javaMethod;
        }
        JavaClass javaClass = this.getJavaClassByNode(mth.getParentClass().getTopParentClass());
        if (javaClass == null) {
            return null;
        }
        this.loadJavaClass(javaClass);
        javaMethod = this.methodsMap.get(mth);
        if (javaMethod != null) {
            return javaMethod;
        }
        if (mth.getParentClass().hasNotGeneratedParent()) {
            return null;
        }
        throw new JadxRuntimeException("JavaMethod not found by MethodNode: " + mth);
    }

    @Nullable
    private JavaField getJavaFieldByNode(FieldNode fld) {
        JavaField javaField = this.fieldsMap.get(fld);
        if (javaField != null) {
            return javaField;
        }
        JavaClass javaClass = this.getJavaClassByNode(fld.getParentClass().getTopParentClass());
        if (javaClass == null) {
            return null;
        }
        this.loadJavaClass(javaClass);
        javaField = this.fieldsMap.get(fld);
        if (javaField != null) {
            return javaField;
        }
        if (fld.getParentClass().hasNotGeneratedParent()) {
            return null;
        }
        throw new JadxRuntimeException("JavaField not found by FieldNode: " + fld);
    }

    @Nullable
    JavaNode convertNode(Object obj) {
        if (!(obj instanceof LineAttrNode)) {
            return null;
        }
        LineAttrNode node = (LineAttrNode)obj;
        if (node.contains(AFlag.DONT_GENERATE)) {
            return null;
        }
        if (obj instanceof ClassNode) {
            return this.getJavaClassByNode((ClassNode)obj);
        }
        if (obj instanceof MethodNode) {
            return this.getJavaMethodByNode((MethodNode)obj);
        }
        if (obj instanceof FieldNode) {
            return this.getJavaFieldByNode((FieldNode)obj);
        }
        throw new JadxRuntimeException("Unexpected node type: " + obj);
    }

    List<JavaNode> convertNodes(Collection<?> nodesList) {
        return nodesList.stream().map(this::convertNode).filter(Objects::nonNull).collect(Collectors.toList());
    }

    @Nullable
    public JavaNode getJavaNodeAtPosition(ICodeInfo codeInfo, int line, int offset) {
        Map<CodePosition, Object> map = codeInfo.getAnnotations();
        if (map.isEmpty()) {
            return null;
        }
        Object obj = map.get(new CodePosition(line, offset));
        if (obj == null) {
            return null;
        }
        return this.convertNode(obj);
    }

    @Nullable
    public CodePosition getDefinitionPosition(JavaNode javaNode) {
        JavaClass jCls = javaNode.getTopParentClass();
        jCls.decompile();
        int defLine = javaNode.getDecompiledLine();
        if (defLine == 0) {
            return null;
        }
        return new CodePosition(jCls, defLine, 0);
    }

    public JadxArgs getArgs() {
        return this.args;
    }

    public String toString() {
        return "jadx decompiler " + JadxDecompiler.getVersion();
    }
}

