#!/usr/bin/env perl
# BEGIN COPYRIGHT BLOCK
# Copyright (C) 2013 Red Hat, Inc.
# All rights reserved.
#
# License: GPL (version 3 or any later version).
# See LICENSE for details. 
# END COPYRIGHT BLOCK
#

use strict;

use lib qw(/usr/lib/dirsrv/perl);
use Getopt::Long;
use Resource;
use DSCreate qw(removeDSInstance);

# process command line options
Getopt::Long::Configure(qw(bundling)); # bundling allows -ddddd

my $res = new Resource("/usr/share/dirsrv/properties/setup-ds.res");

sub usage {
        print(STDERR "Usage: $0 [-a] [-f] [-d -d ... -d] -i instance\n\n");
        print(STDERR " Opts: -a            - remove all\n");
        print(STDERR "       -f            - force removal\n");
        print(STDERR "       -i instance   - instance name to remove (e.g. - slapd-example)\n");
        print(STDERR "       -d            - turn on debugging output\n");
}

my $force = "";
my $instname = "";
my $initconfig_dir = "";
my $all = "";

GetOptions('help|h|?' => sub { &usage; exit(0); },
           'debug|d+' => \$DSUtil::debuglevel,
           'instance|i=s' => \$instname,
           'initconfig_dir|c=s' => \$initconfig_dir,
           'force|f' => \$force,
           'all|a' => \$all
           );

# Make sure the instance name option was provided.
unless ($instname) {
    &usage; exit(1);
}

# Make sure a full instance name was provided.
my ($slapd, $inst) = split(/-/, $instname, 2);
unless ($inst) {
    print STDERR "Full instance name must be specified (e.g. - slapd-example)\n";
    exit 1;
}
unless ($slapd eq "slapd") {
    print STDERR "Error: Invalid instance name \"$instname\"\n";
    exit 1;
}

my @errs = removeDSInstance($inst, $force, $all, $initconfig_dir);
if (@errs) {
    print STDERR "The following errors occurred during removal:\n";
    for (@errs) {
        print STDERR $res->getText($_);
    }
    print STDERR "Error: could not remove directory server $inst\n";
    exit 1;
}

# if we got here, report success
print "Instance $instname removed.\n";
exit 0;

# emacs settings
# Local Variables:
# mode:perl
# indent-tabs-mode: nil
# tab-width: 4
# End:
