---
-- Library mapping for handling codepoints according to IDNA standard.
--
-- All the mappings are imported from Version 10.0.0 of UTS #46, Unicode IDNA
-- Compatibility Processing.
-- References:
-- * http://www.unicode.org/Public/idna/10.0.0/
--
-- @author Rewanth Cool
-- @copyright Same as Nmap--See https://nmap.org/book/man-legal.html

local stdnse = require "stdnse"

_ENV = stdnse.module("idnaMappings", stdnse.seeall)

-- The Unicode Consortium approved the mappings below.
tbl = {
 [0x002F] = {status="disallowed"}, -- SOLIDUS
 -- HYPHEN-MINUS..FULL STOP
 [0x002D] = {status="valid"},
 [0x002E] = {status="valid"},
 [0x0041] = 0x0061, -- LATIN CAPITAL LETTER A
 [0x0042] = 0x0062, -- LATIN CAPITAL LETTER B
 [0x0043] = 0x0063, -- LATIN CAPITAL LETTER C
 [0x0044] = 0x0064, -- LATIN CAPITAL LETTER D
 [0x0045] = 0x0065, -- LATIN CAPITAL LETTER E
 [0x0046] = 0x0066, -- LATIN CAPITAL LETTER F
 [0x0047] = 0x0067, -- LATIN CAPITAL LETTER G
 [0x0048] = 0x0068, -- LATIN CAPITAL LETTER H
 [0x0049] = 0x0069, -- LATIN CAPITAL LETTER I
 [0x004A] = 0x006A, -- LATIN CAPITAL LETTER J
 [0x004B] = 0x006B, -- LATIN CAPITAL LETTER K
 [0x004C] = 0x006C, -- LATIN CAPITAL LETTER L
 [0x004D] = 0x006D, -- LATIN CAPITAL LETTER M
 [0x004E] = 0x006E, -- LATIN CAPITAL LETTER N
 [0x004F] = 0x006F, -- LATIN CAPITAL LETTER O
 [0x0050] = 0x0070, -- LATIN CAPITAL LETTER P
 [0x0051] = 0x0071, -- LATIN CAPITAL LETTER Q
 [0x0052] = 0x0072, -- LATIN CAPITAL LETTER R
 [0x0053] = 0x0073, -- LATIN CAPITAL LETTER S
 [0x0054] = 0x0074, -- LATIN CAPITAL LETTER T
 [0x0055] = 0x0075, -- LATIN CAPITAL LETTER U
 [0x0056] = 0x0076, -- LATIN CAPITAL LETTER V
 [0x0057] = 0x0077, -- LATIN CAPITAL LETTER W
 [0x0058] = 0x0078, -- LATIN CAPITAL LETTER X
 [0x0059] = 0x0079, -- LATIN CAPITAL LETTER Y
 [0x005A] = 0x007A, -- LATIN CAPITAL LETTER Z
 -- LEFT CURLY BRACKET..<control-007F>
 [0x007B] = {status="disallowed"},
 [0x007C] = {status="disallowed"},
 [0x007D] = {status="disallowed"},
 [0x007E] = {status="disallowed"},
 [0x007F] = {status="disallowed"},
 [0x00A0] = {status="disallowed"}, -- NO-BREAK SPACE
 [0x00A8] = {status="disallowed"}, -- DIAERESIS
 [0x00A9] = {status="disallowed"}, -- COPYRIGHT SIGN
 [0x00AA] = 0x0061, -- FEMININE ORDINAL INDICATOR
 [0x00AD] = {status='ignored'}, -- SOFT HYPHEN
 [0x00AE] = {status="disallowed"}, -- REGISTERED SIGN
 [0x00AF] = {status="disallowed"}, -- MACRON
 -- LEFT-POINTING DOUBLE ANGLE QUOTATION MARK..NOT SIGN
 [0x00AB] = {status="disallowed"},
 [0x00AC] = {status="disallowed"},
 [0x00B2] = 0x0032, -- SUPERSCRIPT TWO
 [0x00B3] = 0x0033, -- SUPERSCRIPT THREE
 [0x00B4] = {status="disallowed"}, -- ACUTE ACCENT
 [0x00B5] = 0x03BC, -- MICRO SIGN
 [0x00B6] = {status="disallowed"}, -- PILCROW SIGN
 [0x00B7] = {status="valid"}, -- MIDDLE DOT
 [0x00B8] = {status="disallowed"}, -- CEDILLA
 [0x00B9] = 0x0031, -- SUPERSCRIPT ONE
 [0x00BA] = 0x006F, -- MASCULINE ORDINAL INDICATOR
 [0x00BB] = {status="disallowed"}, -- RIGHT-POINTING DOUBLE ANGLE QUOTATION MARK
 [0x00BC] = {0x0031, 0x2044, 0x0034}, -- VULGAR FRACTION ONE QUARTER
 [0x00BD] = {0x0031, 0x2044, 0x0032}, -- VULGAR FRACTION ONE HALF
 [0x00BE] = {0x0033, 0x2044, 0x0034}, -- VULGAR FRACTION THREE QUARTERS
 [0x00BF] = {status="disallowed"}, -- INVERTED QUESTION MARK
 [0x00C0] = 0x00E0, -- LATIN CAPITAL LETTER A WITH GRAVE
 [0x00C1] = 0x00E1, -- LATIN CAPITAL LETTER A WITH ACUTE
 [0x00C2] = 0x00E2, -- LATIN CAPITAL LETTER A WITH CIRCUMFLEX
 [0x00C3] = 0x00E3, -- LATIN CAPITAL LETTER A WITH TILDE
 [0x00C4] = 0x00E4, -- LATIN CAPITAL LETTER A WITH DIAERESIS
 [0x00C5] = 0x00E5, -- LATIN CAPITAL LETTER A WITH RING ABOVE
 [0x00C6] = 0x00E6, -- LATIN CAPITAL LETTER AE
 [0x00C7] = 0x00E7, -- LATIN CAPITAL LETTER C WITH CEDILLA
 [0x00C8] = 0x00E8, -- LATIN CAPITAL LETTER E WITH GRAVE
 [0x00C9] = 0x00E9, -- LATIN CAPITAL LETTER E WITH ACUTE
 [0x00CA] = 0x00EA, -- LATIN CAPITAL LETTER E WITH CIRCUMFLEX
 [0x00CB] = 0x00EB, -- LATIN CAPITAL LETTER E WITH DIAERESIS
 [0x00CC] = 0x00EC, -- LATIN CAPITAL LETTER I WITH GRAVE
 [0x00CD] = 0x00ED, -- LATIN CAPITAL LETTER I WITH ACUTE
 [0x00CE] = 0x00EE, -- LATIN CAPITAL LETTER I WITH CIRCUMFLEX
 [0x00CF] = 0x00EF, -- LATIN CAPITAL LETTER I WITH DIAERESIS
 [0x00D0] = 0x00F0, -- LATIN CAPITAL LETTER ETH
 [0x00D1] = 0x00F1, -- LATIN CAPITAL LETTER N WITH TILDE
 [0x00D2] = 0x00F2, -- LATIN CAPITAL LETTER O WITH GRAVE
 [0x00D3] = 0x00F3, -- LATIN CAPITAL LETTER O WITH ACUTE
 [0x00D4] = 0x00F4, -- LATIN CAPITAL LETTER O WITH CIRCUMFLEX
 [0x00D5] = 0x00F5, -- LATIN CAPITAL LETTER O WITH TILDE
 [0x00D6] = 0x00F6, -- LATIN CAPITAL LETTER O WITH DIAERESIS
 [0x00D7] = {status="disallowed"}, -- MULTIPLICATION SIGN
 [0x00D8] = 0x00F8, -- LATIN CAPITAL LETTER O WITH STROKE
 [0x00D9] = 0x00F9, -- LATIN CAPITAL LETTER U WITH GRAVE
 [0x00DA] = 0x00FA, -- LATIN CAPITAL LETTER U WITH ACUTE
 [0x00DB] = 0x00FB, -- LATIN CAPITAL LETTER U WITH CIRCUMFLEX
 [0x00DC] = 0x00FC, -- LATIN CAPITAL LETTER U WITH DIAERESIS
 [0x00DD] = 0x00FD, -- LATIN CAPITAL LETTER Y WITH ACUTE
 [0x00DE] = 0x00FE, -- LATIN CAPITAL LETTER THORN
 [0x00DF] = {status='deviation', {0x0073, 0x0073}}, -- LATIN SMALL LETTER SHARP S
 -- DEGREE SIGN..PLUS-MINUS SIGN
 [0x00B0] = {status="disallowed"},
 [0x00B1] = {status="disallowed"},
 [0x00F7] = {status="disallowed"}, -- DIVISION SIGN
 [0x0100] = 0x0101, -- LATIN CAPITAL LETTER A WITH MACRON
 [0x0101] = {status="valid"}, -- LATIN SMALL LETTER A WITH MACRON
 [0x0102] = 0x0103, -- LATIN CAPITAL LETTER A WITH BREVE
 [0x0103] = {status="valid"}, -- LATIN SMALL LETTER A WITH BREVE
 [0x0104] = 0x0105, -- LATIN CAPITAL LETTER A WITH OGONEK
 [0x0105] = {status="valid"}, -- LATIN SMALL LETTER A WITH OGONEK
 [0x0106] = 0x0107, -- LATIN CAPITAL LETTER C WITH ACUTE
 [0x0107] = {status="valid"}, -- LATIN SMALL LETTER C WITH ACUTE
 [0x0108] = 0x0109, -- LATIN CAPITAL LETTER C WITH CIRCUMFLEX
 [0x0109] = {status="valid"}, -- LATIN SMALL LETTER C WITH CIRCUMFLEX
 [0x010A] = 0x010B, -- LATIN CAPITAL LETTER C WITH DOT ABOVE
 [0x010B] = {status="valid"}, -- LATIN SMALL LETTER C WITH DOT ABOVE
 [0x010C] = 0x010D, -- LATIN CAPITAL LETTER C WITH CARON
 [0x010D] = {status="valid"}, -- LATIN SMALL LETTER C WITH CARON
 [0x010E] = 0x010F, -- LATIN CAPITAL LETTER D WITH CARON
 [0x010F] = {status="valid"}, -- LATIN SMALL LETTER D WITH CARON
 [0x0110] = 0x0111, -- LATIN CAPITAL LETTER D WITH STROKE
 [0x0111] = {status="valid"}, -- LATIN SMALL LETTER D WITH STROKE
 [0x0112] = 0x0113, -- LATIN CAPITAL LETTER E WITH MACRON
 [0x0113] = {status="valid"}, -- LATIN SMALL LETTER E WITH MACRON
 [0x0114] = 0x0115, -- LATIN CAPITAL LETTER E WITH BREVE
 [0x0115] = {status="valid"}, -- LATIN SMALL LETTER E WITH BREVE
 [0x0116] = 0x0117, -- LATIN CAPITAL LETTER E WITH DOT ABOVE
 [0x0117] = {status="valid"}, -- LATIN SMALL LETTER E WITH DOT ABOVE
 [0x0118] = 0x0119, -- LATIN CAPITAL LETTER E WITH OGONEK
 [0x0119] = {status="valid"}, -- LATIN SMALL LETTER E WITH OGONEK
 [0x011A] = 0x011B, -- LATIN CAPITAL LETTER E WITH CARON
 [0x011B] = {status="valid"}, -- LATIN SMALL LETTER E WITH CARON
 [0x011C] = 0x011D, -- LATIN CAPITAL LETTER G WITH CIRCUMFLEX
 [0x011D] = {status="valid"}, -- LATIN SMALL LETTER G WITH CIRCUMFLEX
 [0x011E] = 0x011F, -- LATIN CAPITAL LETTER G WITH BREVE
 [0x011F] = {status="valid"}, -- LATIN SMALL LETTER G WITH BREVE
 [0x0120] = 0x0121, -- LATIN CAPITAL LETTER G WITH DOT ABOVE
 [0x0121] = {status="valid"}, -- LATIN SMALL LETTER G WITH DOT ABOVE
 [0x0122] = 0x0123, -- LATIN CAPITAL LETTER G WITH CEDILLA
 [0x0123] = {status="valid"}, -- LATIN SMALL LETTER G WITH CEDILLA
 [0x0124] = 0x0125, -- LATIN CAPITAL LETTER H WITH CIRCUMFLEX
 [0x0125] = {status="valid"}, -- LATIN SMALL LETTER H WITH CIRCUMFLEX
 [0x0126] = 0x0127, -- LATIN CAPITAL LETTER H WITH STROKE
 [0x0127] = {status="valid"}, -- LATIN SMALL LETTER H WITH STROKE
 [0x0128] = 0x0129, -- LATIN CAPITAL LETTER I WITH TILDE
 [0x0129] = {status="valid"}, -- LATIN SMALL LETTER I WITH TILDE
 [0x012A] = 0x012B, -- LATIN CAPITAL LETTER I WITH MACRON
 [0x012B] = {status="valid"}, -- LATIN SMALL LETTER I WITH MACRON
 [0x012C] = 0x012D, -- LATIN CAPITAL LETTER I WITH BREVE
 [0x012D] = {status="valid"}, -- LATIN SMALL LETTER I WITH BREVE
 [0x012E] = 0x012F, -- LATIN CAPITAL LETTER I WITH OGONEK
 [0x012F] = {status="valid"}, -- LATIN SMALL LETTER I WITH OGONEK
 [0x0130] = {0x0069, 0x0307}, -- LATIN CAPITAL LETTER I WITH DOT ABOVE
 [0x0131] = {status="valid"}, -- LATIN SMALL LETTER DOTLESS I
 [0x0134] = 0x0135, -- LATIN CAPITAL LETTER J WITH CIRCUMFLEX
 [0x0135] = {status="valid"}, -- LATIN SMALL LETTER J WITH CIRCUMFLEX
 [0x0136] = 0x0137, -- LATIN CAPITAL LETTER K WITH CEDILLA
 -- LATIN CAPITAL LIGATURE IJ..LATIN SMALL LIGATURE IJ
 [0x0132] = {0x0069, 0x006A},
 [0x0133] = {0x0069, 0x006A},
 [0x0139] = 0x013A, -- LATIN CAPITAL LETTER L WITH ACUTE
 [0x013A] = {status="valid"}, -- LATIN SMALL LETTER L WITH ACUTE
 [0x013B] = 0x013C, -- LATIN CAPITAL LETTER L WITH CEDILLA
 [0x013C] = {status="valid"}, -- LATIN SMALL LETTER L WITH CEDILLA
 [0x013D] = 0x013E, -- LATIN CAPITAL LETTER L WITH CARON
 [0x013E] = {status="valid"}, -- LATIN SMALL LETTER L WITH CARON
 -- LATIN SMALL LETTER K WITH CEDILLA..LATIN SMALL LETTER KRA
 [0x0137] = {status="valid"},
 [0x0138] = {status="valid"},
 [0x0141] = 0x0142, -- LATIN CAPITAL LETTER L WITH STROKE
 [0x0142] = {status="valid"}, -- LATIN SMALL LETTER L WITH STROKE
 [0x0143] = 0x0144, -- LATIN CAPITAL LETTER N WITH ACUTE
 [0x0144] = {status="valid"}, -- LATIN SMALL LETTER N WITH ACUTE
 [0x0145] = 0x0146, -- LATIN CAPITAL LETTER N WITH CEDILLA
 [0x0146] = {status="valid"}, -- LATIN SMALL LETTER N WITH CEDILLA
 [0x0147] = 0x0148, -- LATIN CAPITAL LETTER N WITH CARON
 [0x0148] = {status="valid"}, -- LATIN SMALL LETTER N WITH CARON
 [0x0149] = {0x02BC, 0x006E}, -- LATIN SMALL LETTER N PRECEDED BY APOSTROPHE
 [0x014A] = 0x014B, -- LATIN CAPITAL LETTER ENG
 [0x014B] = {status="valid"}, -- LATIN SMALL LETTER ENG
 [0x014C] = 0x014D, -- LATIN CAPITAL LETTER O WITH MACRON
 [0x014D] = {status="valid"}, -- LATIN SMALL LETTER O WITH MACRON
 [0x014E] = 0x014F, -- LATIN CAPITAL LETTER O WITH BREVE
 [0x014F] = {status="valid"}, -- LATIN SMALL LETTER O WITH BREVE
 [0x0150] = 0x0151, -- LATIN CAPITAL LETTER O WITH DOUBLE ACUTE
 [0x0151] = {status="valid"}, -- LATIN SMALL LETTER O WITH DOUBLE ACUTE
 [0x0152] = 0x0153, -- LATIN CAPITAL LIGATURE OE
 [0x0153] = {status="valid"}, -- LATIN SMALL LIGATURE OE
 [0x0154] = 0x0155, -- LATIN CAPITAL LETTER R WITH ACUTE
 [0x0155] = {status="valid"}, -- LATIN SMALL LETTER R WITH ACUTE
 [0x0156] = 0x0157, -- LATIN CAPITAL LETTER R WITH CEDILLA
 [0x0157] = {status="valid"}, -- LATIN SMALL LETTER R WITH CEDILLA
 [0x0158] = 0x0159, -- LATIN CAPITAL LETTER R WITH CARON
 [0x0159] = {status="valid"}, -- LATIN SMALL LETTER R WITH CARON
 [0x015A] = 0x015B, -- LATIN CAPITAL LETTER S WITH ACUTE
 [0x015B] = {status="valid"}, -- LATIN SMALL LETTER S WITH ACUTE
 [0x015C] = 0x015D, -- LATIN CAPITAL LETTER S WITH CIRCUMFLEX
 [0x015D] = {status="valid"}, -- LATIN SMALL LETTER S WITH CIRCUMFLEX
 [0x015E] = 0x015F, -- LATIN CAPITAL LETTER S WITH CEDILLA
 [0x015F] = {status="valid"}, -- LATIN SMALL LETTER S WITH CEDILLA
 [0x0160] = 0x0161, -- LATIN CAPITAL LETTER S WITH CARON
 [0x0161] = {status="valid"}, -- LATIN SMALL LETTER S WITH CARON
 [0x0162] = 0x0163, -- LATIN CAPITAL LETTER T WITH CEDILLA
 [0x0163] = {status="valid"}, -- LATIN SMALL LETTER T WITH CEDILLA
 [0x0164] = 0x0165, -- LATIN CAPITAL LETTER T WITH CARON
 [0x0165] = {status="valid"}, -- LATIN SMALL LETTER T WITH CARON
 [0x0166] = 0x0167, -- LATIN CAPITAL LETTER T WITH STROKE
 [0x0167] = {status="valid"}, -- LATIN SMALL LETTER T WITH STROKE
 [0x0168] = 0x0169, -- LATIN CAPITAL LETTER U WITH TILDE
 [0x0169] = {status="valid"}, -- LATIN SMALL LETTER U WITH TILDE
 [0x016A] = 0x016B, -- LATIN CAPITAL LETTER U WITH MACRON
 [0x016B] = {status="valid"}, -- LATIN SMALL LETTER U WITH MACRON
 [0x016C] = 0x016D, -- LATIN CAPITAL LETTER U WITH BREVE
 [0x016D] = {status="valid"}, -- LATIN SMALL LETTER U WITH BREVE
 [0x016E] = 0x016F, -- LATIN CAPITAL LETTER U WITH RING ABOVE
 [0x016F] = {status="valid"}, -- LATIN SMALL LETTER U WITH RING ABOVE
 [0x0170] = 0x0171, -- LATIN CAPITAL LETTER U WITH DOUBLE ACUTE
 [0x0171] = {status="valid"}, -- LATIN SMALL LETTER U WITH DOUBLE ACUTE
 [0x0172] = 0x0173, -- LATIN CAPITAL LETTER U WITH OGONEK
 [0x0173] = {status="valid"}, -- LATIN SMALL LETTER U WITH OGONEK
 [0x0174] = 0x0175, -- LATIN CAPITAL LETTER W WITH CIRCUMFLEX
 [0x0175] = {status="valid"}, -- LATIN SMALL LETTER W WITH CIRCUMFLEX
 [0x0176] = 0x0177, -- LATIN CAPITAL LETTER Y WITH CIRCUMFLEX
 [0x0177] = {status="valid"}, -- LATIN SMALL LETTER Y WITH CIRCUMFLEX
 [0x0178] = 0x00FF, -- LATIN CAPITAL LETTER Y WITH DIAERESIS
 [0x0179] = 0x017A, -- LATIN CAPITAL LETTER Z WITH ACUTE
 [0x017A] = {status="valid"}, -- LATIN SMALL LETTER Z WITH ACUTE
 [0x017B] = 0x017C, -- LATIN CAPITAL LETTER Z WITH DOT ABOVE
 [0x017C] = {status="valid"}, -- LATIN SMALL LETTER Z WITH DOT ABOVE
 [0x017D] = 0x017E, -- LATIN CAPITAL LETTER Z WITH CARON
 [0x017E] = {status="valid"}, -- LATIN SMALL LETTER Z WITH CARON
 [0x017F] = 0x0073, -- LATIN SMALL LETTER LONG S
 [0x0180] = {status="valid"}, -- LATIN SMALL LETTER B WITH STROKE
 [0x0181] = 0x0253, -- LATIN CAPITAL LETTER B WITH HOOK
 [0x0182] = 0x0183, -- LATIN CAPITAL LETTER B WITH TOPBAR
 [0x0183] = {status="valid"}, -- LATIN SMALL LETTER B WITH TOPBAR
 [0x0184] = 0x0185, -- LATIN CAPITAL LETTER TONE SIX
 [0x0185] = {status="valid"}, -- LATIN SMALL LETTER TONE SIX
 [0x0186] = 0x0254, -- LATIN CAPITAL LETTER OPEN O
 [0x0187] = 0x0188, -- LATIN CAPITAL LETTER C WITH HOOK
 [0x0188] = {status="valid"}, -- LATIN SMALL LETTER C WITH HOOK
 [0x0189] = 0x0256, -- LATIN CAPITAL LETTER AFRICAN D
 [0x018A] = 0x0257, -- LATIN CAPITAL LETTER D WITH HOOK
 [0x018B] = 0x018C, -- LATIN CAPITAL LETTER D WITH TOPBAR
 -- LATIN CAPITAL LETTER L WITH MIDDLE DOT..LATIN SMALL LETTER L WITH MIDDLE DOT
 [0x013F] = {0x006C, 0x00B7},
 [0x0140] = {0x006C, 0x00B7},
 [0x018E] = 0x01DD, -- LATIN CAPITAL LETTER REVERSED E
 [0x018F] = 0x0259, -- LATIN CAPITAL LETTER SCHWA
 [0x0190] = 0x025B, -- LATIN CAPITAL LETTER OPEN E
 [0x0191] = 0x0192, -- LATIN CAPITAL LETTER F WITH HOOK
 [0x0192] = {status="valid"}, -- LATIN SMALL LETTER F WITH HOOK
 [0x0193] = 0x0260, -- LATIN CAPITAL LETTER G WITH HOOK
 [0x0194] = 0x0263, -- LATIN CAPITAL LETTER GAMMA
 [0x0195] = {status="valid"}, -- LATIN SMALL LETTER HV
 [0x0196] = 0x0269, -- LATIN CAPITAL LETTER IOTA
 [0x0197] = 0x0268, -- LATIN CAPITAL LETTER I WITH STROKE
 [0x0198] = 0x0199, -- LATIN CAPITAL LETTER K WITH HOOK
 -- LATIN SMALL LETTER D WITH TOPBAR..LATIN SMALL LETTER TURNED DELTA
 [0x018C] = {status="valid"},
 [0x018D] = {status="valid"},
 [0x019C] = 0x026F, -- LATIN CAPITAL LETTER TURNED M
 [0x019D] = 0x0272, -- LATIN CAPITAL LETTER N WITH LEFT HOOK
 [0x019E] = {status="valid"}, -- LATIN SMALL LETTER N WITH LONG RIGHT LEG
 [0x019F] = 0x0275, -- LATIN CAPITAL LETTER O WITH MIDDLE TILDE
 [0x01A0] = 0x01A1, -- LATIN CAPITAL LETTER O WITH HORN
 [0x01A1] = {status="valid"}, -- LATIN SMALL LETTER O WITH HORN
 [0x01A2] = 0x01A3, -- LATIN CAPITAL LETTER OI
 [0x01A3] = {status="valid"}, -- LATIN SMALL LETTER OI
 [0x01A4] = 0x01A5, -- LATIN CAPITAL LETTER P WITH HOOK
 [0x01A5] = {status="valid"}, -- LATIN SMALL LETTER P WITH HOOK
 [0x01A6] = 0x0280, -- LATIN LETTER YR
 [0x01A7] = 0x01A8, -- LATIN CAPITAL LETTER TONE TWO
 [0x01A8] = {status="valid"}, -- LATIN SMALL LETTER TONE TWO
 [0x01A9] = 0x0283, -- LATIN CAPITAL LETTER ESH
 -- LATIN SMALL LETTER K WITH HOOK..LATIN SMALL LETTER LAMBDA WITH STROKE
 [0x0199] = {status="valid"},
 [0x019A] = {status="valid"},
 [0x019B] = {status="valid"},
 [0x01AC] = 0x01AD, -- LATIN CAPITAL LETTER T WITH HOOK
 [0x01AD] = {status="valid"}, -- LATIN SMALL LETTER T WITH HOOK
 [0x01AE] = 0x0288, -- LATIN CAPITAL LETTER T WITH RETROFLEX HOOK
 [0x01AF] = 0x01B0, -- LATIN CAPITAL LETTER U WITH HORN
 [0x01B0] = {status="valid"}, -- LATIN SMALL LETTER U WITH HORN
 [0x01B1] = 0x028A, -- LATIN CAPITAL LETTER UPSILON
 [0x01B2] = 0x028B, -- LATIN CAPITAL LETTER V WITH HOOK
 [0x01B3] = 0x01B4, -- LATIN CAPITAL LETTER Y WITH HOOK
 [0x01B4] = {status="valid"}, -- LATIN SMALL LETTER Y WITH HOOK
 [0x01B5] = 0x01B6, -- LATIN CAPITAL LETTER Z WITH STROKE
 [0x01B6] = {status="valid"}, -- LATIN SMALL LETTER Z WITH STROKE
 [0x01B7] = 0x0292, -- LATIN CAPITAL LETTER EZH
 [0x01B8] = 0x01B9, -- LATIN CAPITAL LETTER EZH REVERSED
 -- LATIN LETTER REVERSED ESH LOOP..LATIN SMALL LETTER T WITH PALATAL HOOK
 [0x01AA] = {status="valid"},
 [0x01AB] = {status="valid"},
 [0x01BC] = 0x01BD, -- LATIN CAPITAL LETTER TONE FIVE
 -- LATIN SMALL LETTER EZH REVERSED..LATIN LETTER TWO WITH STROKE
 [0x01B9] = {status="valid"},
 [0x01BA] = {status="valid"},
 [0x01BB] = {status="valid"},
 -- LATIN CAPITAL LETTER DZ WITH CARON..LATIN SMALL LETTER DZ WITH CARON
 [0x01C4] = {0x0064, 0x017E},
 [0x01C5] = {0x0064, 0x017E},
 [0x01C6] = {0x0064, 0x017E},
 -- LATIN CAPITAL LETTER LJ..LATIN SMALL LETTER LJ
 [0x01C7] = {0x006C, 0x006A},
 [0x01C8] = {0x006C, 0x006A},
 [0x01C9] = {0x006C, 0x006A},
 [0x01CD] = 0x01CE, -- LATIN CAPITAL LETTER A WITH CARON
 [0x01CE] = {status="valid"}, -- LATIN SMALL LETTER A WITH CARON
 [0x01CF] = 0x01D0, -- LATIN CAPITAL LETTER I WITH CARON
 [0x01D0] = {status="valid"}, -- LATIN SMALL LETTER I WITH CARON
 [0x01D1] = 0x01D2, -- LATIN CAPITAL LETTER O WITH CARON
 [0x01D2] = {status="valid"}, -- LATIN SMALL LETTER O WITH CARON
 [0x01D3] = 0x01D4, -- LATIN CAPITAL LETTER U WITH CARON
 [0x01D4] = {status="valid"}, -- LATIN SMALL LETTER U WITH CARON
 [0x01D5] = 0x01D6, -- LATIN CAPITAL LETTER U WITH DIAERESIS AND MACRON
 [0x01D6] = {status="valid"}, -- LATIN SMALL LETTER U WITH DIAERESIS AND MACRON
 [0x01D7] = 0x01D8, -- LATIN CAPITAL LETTER U WITH DIAERESIS AND ACUTE
 [0x01D8] = {status="valid"}, -- LATIN SMALL LETTER U WITH DIAERESIS AND ACUTE
 [0x01D9] = 0x01DA, -- LATIN CAPITAL LETTER U WITH DIAERESIS AND CARON
 [0x01DA] = {status="valid"}, -- LATIN SMALL LETTER U WITH DIAERESIS AND CARON
 [0x01DB] = 0x01DC, -- LATIN CAPITAL LETTER U WITH DIAERESIS AND GRAVE
 -- LATIN CAPITAL LETTER NJ..LATIN SMALL LETTER NJ
 [0x01CA] = {0x006E, 0x006A},
 [0x01CB] = {0x006E, 0x006A},
 [0x01CC] = {0x006E, 0x006A},
 [0x01DE] = 0x01DF, -- LATIN CAPITAL LETTER A WITH DIAERESIS AND MACRON
 [0x01DF] = {status="valid"}, -- LATIN SMALL LETTER A WITH DIAERESIS AND MACRON
 [0x01E0] = 0x01E1, -- LATIN CAPITAL LETTER A WITH DOT ABOVE AND MACRON
 [0x01E1] = {status="valid"}, -- LATIN SMALL LETTER A WITH DOT ABOVE AND MACRON
 [0x01E2] = 0x01E3, -- LATIN CAPITAL LETTER AE WITH MACRON
 [0x01E3] = {status="valid"}, -- LATIN SMALL LETTER AE WITH MACRON
 [0x01E4] = 0x01E5, -- LATIN CAPITAL LETTER G WITH STROKE
 [0x01E5] = {status="valid"}, -- LATIN SMALL LETTER G WITH STROKE
 [0x01E6] = 0x01E7, -- LATIN CAPITAL LETTER G WITH CARON
 [0x01E7] = {status="valid"}, -- LATIN SMALL LETTER G WITH CARON
 [0x01E8] = 0x01E9, -- LATIN CAPITAL LETTER K WITH CARON
 [0x01E9] = {status="valid"}, -- LATIN SMALL LETTER K WITH CARON
 [0x01EA] = 0x01EB, -- LATIN CAPITAL LETTER O WITH OGONEK
 [0x01EB] = {status="valid"}, -- LATIN SMALL LETTER O WITH OGONEK
 [0x01EC] = 0x01ED, -- LATIN CAPITAL LETTER O WITH OGONEK AND MACRON
 [0x01ED] = {status="valid"}, -- LATIN SMALL LETTER O WITH OGONEK AND MACRON
 [0x01EE] = 0x01EF, -- LATIN CAPITAL LETTER EZH WITH CARON
 -- LATIN SMALL LETTER U WITH DIAERESIS AND GRAVE..LATIN SMALL LETTER TURNED E
 [0x01DC] = {status="valid"},
 [0x01DD] = {status="valid"},
 -- LATIN SMALL LETTER EZH WITH CARON..LATIN SMALL LETTER J WITH CARON
 [0x01EF] = {status="valid"},
 [0x01F0] = {status="valid"},
 [0x01F4] = 0x01F5, -- LATIN CAPITAL LETTER G WITH ACUTE
 [0x01F5] = {status="valid"}, -- LATIN SMALL LETTER G WITH ACUTE
 [0x01F6] = 0x0195, -- LATIN CAPITAL LETTER HWAIR
 [0x01F7] = 0x01BF, -- LATIN CAPITAL LETTER WYNN
 [0x01F8] = 0x01F9, -- LATIN CAPITAL LETTER N WITH GRAVE
 [0x01F9] = {status="valid"}, -- LATIN SMALL LETTER N WITH GRAVE
 [0x01FA] = 0x01FB, -- LATIN CAPITAL LETTER A WITH RING ABOVE AND ACUTE
 [0x01FB] = {status="valid"}, -- LATIN SMALL LETTER A WITH RING ABOVE AND ACUTE
 [0x01FC] = 0x01FD, -- LATIN CAPITAL LETTER AE WITH ACUTE
 [0x01FD] = {status="valid"}, -- LATIN SMALL LETTER AE WITH ACUTE
 [0x01FE] = 0x01FF, -- LATIN CAPITAL LETTER O WITH STROKE AND ACUTE
 [0x01FF] = {status="valid"}, -- LATIN SMALL LETTER O WITH STROKE AND ACUTE
 [0x0200] = 0x0201, -- LATIN CAPITAL LETTER A WITH DOUBLE GRAVE
 [0x0201] = {status="valid"}, -- LATIN SMALL LETTER A WITH DOUBLE GRAVE
 [0x0202] = 0x0203, -- LATIN CAPITAL LETTER A WITH INVERTED BREVE
 [0x0203] = {status="valid"}, -- LATIN SMALL LETTER A WITH INVERTED BREVE
 [0x0204] = 0x0205, -- LATIN CAPITAL LETTER E WITH DOUBLE GRAVE
 [0x0205] = {status="valid"}, -- LATIN SMALL LETTER E WITH DOUBLE GRAVE
 [0x0206] = 0x0207, -- LATIN CAPITAL LETTER E WITH INVERTED BREVE
 [0x0207] = {status="valid"}, -- LATIN SMALL LETTER E WITH INVERTED BREVE
 [0x0208] = 0x0209, -- LATIN CAPITAL LETTER I WITH DOUBLE GRAVE
 [0x0209] = {status="valid"}, -- LATIN SMALL LETTER I WITH DOUBLE GRAVE
 [0x020A] = 0x020B, -- LATIN CAPITAL LETTER I WITH INVERTED BREVE
 [0x020B] = {status="valid"}, -- LATIN SMALL LETTER I WITH INVERTED BREVE
 [0x020C] = 0x020D, -- LATIN CAPITAL LETTER O WITH DOUBLE GRAVE
 [0x020D] = {status="valid"}, -- LATIN SMALL LETTER O WITH DOUBLE GRAVE
 [0x020E] = 0x020F, -- LATIN CAPITAL LETTER O WITH INVERTED BREVE
 [0x020F] = {status="valid"}, -- LATIN SMALL LETTER O WITH INVERTED BREVE
 [0x0210] = 0x0211, -- LATIN CAPITAL LETTER R WITH DOUBLE GRAVE
 [0x0211] = {status="valid"}, -- LATIN SMALL LETTER R WITH DOUBLE GRAVE
 [0x0212] = 0x0213, -- LATIN CAPITAL LETTER R WITH INVERTED BREVE
 [0x0213] = {status="valid"}, -- LATIN SMALL LETTER R WITH INVERTED BREVE
 [0x0214] = 0x0215, -- LATIN CAPITAL LETTER U WITH DOUBLE GRAVE
 [0x0215] = {status="valid"}, -- LATIN SMALL LETTER U WITH DOUBLE GRAVE
 [0x0216] = 0x0217, -- LATIN CAPITAL LETTER U WITH INVERTED BREVE
 [0x0217] = {status="valid"}, -- LATIN SMALL LETTER U WITH INVERTED BREVE
 [0x0218] = 0x0219, -- LATIN CAPITAL LETTER S WITH COMMA BELOW
 [0x0219] = {status="valid"}, -- LATIN SMALL LETTER S WITH COMMA BELOW
 [0x021A] = 0x021B, -- LATIN CAPITAL LETTER T WITH COMMA BELOW
 [0x021B] = {status="valid"}, -- LATIN SMALL LETTER T WITH COMMA BELOW
 [0x021C] = 0x021D, -- LATIN CAPITAL LETTER YOGH
 [0x021D] = {status="valid"}, -- LATIN SMALL LETTER YOGH
 [0x021E] = 0x021F, -- LATIN CAPITAL LETTER H WITH CARON
 [0x021F] = {status="valid"}, -- LATIN SMALL LETTER H WITH CARON
 [0x0220] = 0x019E, -- LATIN CAPITAL LETTER N WITH LONG RIGHT LEG
 [0x0221] = {status="valid"}, -- LATIN SMALL LETTER D WITH CURL
 [0x0222] = 0x0223, -- LATIN CAPITAL LETTER OU
 [0x0223] = {status="valid"}, -- LATIN SMALL LETTER OU
 [0x0224] = 0x0225, -- LATIN CAPITAL LETTER Z WITH HOOK
 [0x0225] = {status="valid"}, -- LATIN SMALL LETTER Z WITH HOOK
 [0x0226] = 0x0227, -- LATIN CAPITAL LETTER A WITH DOT ABOVE
 [0x0227] = {status="valid"}, -- LATIN SMALL LETTER A WITH DOT ABOVE
 [0x0228] = 0x0229, -- LATIN CAPITAL LETTER E WITH CEDILLA
 [0x0229] = {status="valid"}, -- LATIN SMALL LETTER E WITH CEDILLA
 [0x022A] = 0x022B, -- LATIN CAPITAL LETTER O WITH DIAERESIS AND MACRON
 [0x022B] = {status="valid"}, -- LATIN SMALL LETTER O WITH DIAERESIS AND MACRON
 [0x022C] = 0x022D, -- LATIN CAPITAL LETTER O WITH TILDE AND MACRON
 [0x022D] = {status="valid"}, -- LATIN SMALL LETTER O WITH TILDE AND MACRON
 [0x022E] = 0x022F, -- LATIN CAPITAL LETTER O WITH DOT ABOVE
 [0x022F] = {status="valid"}, -- LATIN SMALL LETTER O WITH DOT ABOVE
 [0x0230] = 0x0231, -- LATIN CAPITAL LETTER O WITH DOT ABOVE AND MACRON
 [0x0231] = {status="valid"}, -- LATIN SMALL LETTER O WITH DOT ABOVE AND MACRON
 [0x0232] = 0x0233, -- LATIN CAPITAL LETTER Y WITH MACRON
 [0x0233] = {status="valid"}, -- LATIN SMALL LETTER Y WITH MACRON
 -- LATIN CAPITAL LETTER DZ..LATIN SMALL LETTER DZ
 [0x01F1] = {0x0064, 0x007A},
 [0x01F2] = {0x0064, 0x007A},
 [0x01F3] = {0x0064, 0x007A},
 [0x023A] = 0x2C65, -- LATIN CAPITAL LETTER A WITH STROKE
 [0x023B] = 0x023C, -- LATIN CAPITAL LETTER C WITH STROKE
 [0x023C] = {status="valid"}, -- LATIN SMALL LETTER C WITH STROKE
 [0x023D] = 0x019A, -- LATIN CAPITAL LETTER L WITH BAR
 [0x023E] = 0x2C66, -- LATIN CAPITAL LETTER T WITH DIAGONAL STROKE
 [0x0241] = 0x0242, -- LATIN CAPITAL LETTER GLOTTAL STOP
 [0x0242] = {status="valid"}, -- LATIN SMALL LETTER GLOTTAL STOP
 [0x0243] = 0x0180, -- LATIN CAPITAL LETTER B WITH STROKE
 [0x0244] = 0x0289, -- LATIN CAPITAL LETTER U BAR
 [0x0245] = 0x028C, -- LATIN CAPITAL LETTER TURNED V
 [0x0246] = 0x0247, -- LATIN CAPITAL LETTER E WITH STROKE
 [0x0247] = {status="valid"}, -- LATIN SMALL LETTER E WITH STROKE
 [0x0248] = 0x0249, -- LATIN CAPITAL LETTER J WITH STROKE
 [0x0249] = {status="valid"}, -- LATIN SMALL LETTER J WITH STROKE
 [0x024A] = 0x024B, -- LATIN CAPITAL LETTER SMALL Q WITH HOOK TAIL
 [0x024B] = {status="valid"}, -- LATIN SMALL LETTER Q WITH HOOK TAIL
 [0x024C] = 0x024D, -- LATIN CAPITAL LETTER R WITH STROKE
 [0x024D] = {status="valid"}, -- LATIN SMALL LETTER R WITH STROKE
 [0x024E] = 0x024F, -- LATIN CAPITAL LETTER Y WITH STROKE
 [0x024F] = {status="valid"}, -- LATIN SMALL LETTER Y WITH STROKE
 -- LATIN SMALL LETTER S WITH SWASH TAIL..LATIN SMALL LETTER Z WITH SWASH TAIL
 [0x023F] = {status="valid"},
 [0x0240] = {status="valid"},
 [0x02B0] = 0x0068, -- MODIFIER LETTER SMALL H
 [0x02B1] = 0x0266, -- MODIFIER LETTER SMALL H WITH HOOK
 [0x02B2] = 0x006A, -- MODIFIER LETTER SMALL J
 [0x02B3] = 0x0072, -- MODIFIER LETTER SMALL R
 [0x02B4] = 0x0279, -- MODIFIER LETTER SMALL TURNED R
 [0x02B5] = 0x027B, -- MODIFIER LETTER SMALL TURNED R WITH HOOK
 [0x02B6] = 0x0281, -- MODIFIER LETTER SMALL CAPITAL INVERTED R
 [0x02B7] = 0x0077, -- MODIFIER LETTER SMALL W
 [0x02B8] = 0x0079, -- MODIFIER LETTER SMALL Y
 -- MODIFIER LETTER LEFT ARROWHEAD..MODIFIER LETTER DOWN ARROWHEAD
 [0x02C2] = {status="disallowed"},
 [0x02C3] = {status="disallowed"},
 [0x02C4] = {status="disallowed"},
 [0x02C5] = {status="disallowed"},
 [0x02D8] = {status="disallowed"}, -- BREVE
 [0x02D9] = {status="disallowed"}, -- DOT ABOVE
 [0x02DA] = {status="disallowed"}, -- RING ABOVE
 [0x02DB] = {status="disallowed"}, -- OGONEK
 [0x02DC] = {status="disallowed"}, -- SMALL TILDE
 [0x02DD] = {status="disallowed"}, -- DOUBLE ACUTE ACCENT
 [0x02DE] = {status="disallowed"}, -- MODIFIER LETTER RHOTIC HOOK
 [0x02DF] = {status="disallowed"}, -- MODIFIER LETTER CROSS ACCENT
 [0x02E0] = 0x0263, -- MODIFIER LETTER SMALL GAMMA
 [0x02E1] = 0x006C, -- MODIFIER LETTER SMALL L
 [0x02E2] = 0x0073, -- MODIFIER LETTER SMALL S
 [0x02E3] = 0x0078, -- MODIFIER LETTER SMALL X
 [0x02E4] = 0x0295, -- MODIFIER LETTER SMALL REVERSED GLOTTAL STOP
 [0x02EC] = {status="valid"}, -- MODIFIER LETTER VOICING
 [0x02ED] = {status="disallowed"}, -- MODIFIER LETTER UNASPIRATED
 [0x02EE] = {status="valid"}, -- MODIFIER LETTER DOUBLE APOSTROPHE
 [0x0340] = 0x0300, -- COMBINING GRAVE TONE MARK
 [0x0341] = 0x0301, -- COMBINING ACUTE TONE MARK
 [0x0342] = {status="valid"}, -- COMBINING GREEK PERISPOMENI
 [0x0343] = 0x0313, -- COMBINING GREEK KORONIS
 [0x0344] = {0x0308, 0x0301}, -- COMBINING GREEK DIALYTIKA TONOS
 [0x0345] = 0x03B9, -- COMBINING GREEK YPOGEGRAMMENI
 [0x034F] = {status='ignored'}, -- COMBINING GRAPHEME JOINER
 [0x0362] = {status="valid"}, -- COMBINING DOUBLE RIGHTWARDS ARROW BELOW
 [0x0370] = 0x0371, -- GREEK CAPITAL LETTER HETA
 [0x0371] = {status="valid"}, -- GREEK SMALL LETTER HETA
 [0x0372] = 0x0373, -- GREEK CAPITAL LETTER ARCHAIC SAMPI
 [0x0373] = {status="valid"}, -- GREEK SMALL LETTER ARCHAIC SAMPI
 [0x0374] = 0x02B9, -- GREEK NUMERAL SIGN
 [0x0375] = {status="valid"}, -- GREEK LOWER NUMERAL SIGN
 [0x0376] = 0x0377, -- GREEK CAPITAL LETTER PAMPHYLIAN DIGAMMA
 [0x0377] = {status="valid"}, -- GREEK SMALL LETTER PAMPHYLIAN DIGAMMA
 [0x037A] = {status="disallowed"}, -- GREEK YPOGEGRAMMENI
 -- NA   <reserved-0378>..<reserved-0379>
 [0x0378] = {status="disallowed"},
 [0x0379] = {status="disallowed"},
 [0x037E] = {status="disallowed"}, -- GREEK QUESTION MARK
 [0x037F] = 0x03F3, -- GREEK CAPITAL LETTER YOT
 -- GREEK SMALL REVERSED LUNATE SIGMA SYMBOL..GREEK SMALL REVERSED DOTTED LUNATE SIGMA SYMBOL
 [0x037B] = {status="valid"},
 [0x037C] = {status="valid"},
 [0x037D] = {status="valid"},
 [0x0384] = {status="disallowed"}, -- GREEK TONOS
 [0x0385] = {status="disallowed"}, -- GREEK DIALYTIKA TONOS
 [0x0386] = 0x03AC, -- GREEK CAPITAL LETTER ALPHA WITH TONOS
 [0x0387] = 0x00B7, -- GREEK ANO TELEIA
 [0x0388] = 0x03AD, -- GREEK CAPITAL LETTER EPSILON WITH TONOS
 [0x0389] = 0x03AE, -- GREEK CAPITAL LETTER ETA WITH TONOS
 [0x038A] = 0x03AF, -- GREEK CAPITAL LETTER IOTA WITH TONOS
 [0x038B] = {status="disallowed"}, -- NA   <reserved-038B>
 [0x038C] = 0x03CC, -- GREEK CAPITAL LETTER OMICRON WITH TONOS
 [0x038D] = {status="disallowed"}, -- NA   <reserved-038D>
 [0x038E] = 0x03CD, -- GREEK CAPITAL LETTER UPSILON WITH TONOS
 [0x038F] = 0x03CE, -- GREEK CAPITAL LETTER OMEGA WITH TONOS
 [0x0390] = {status="valid"}, -- GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS
 [0x0391] = 0x03B1, -- GREEK CAPITAL LETTER ALPHA
 [0x0392] = 0x03B2, -- GREEK CAPITAL LETTER BETA
 [0x0393] = 0x03B3, -- GREEK CAPITAL LETTER GAMMA
 [0x0394] = 0x03B4, -- GREEK CAPITAL LETTER DELTA
 [0x0395] = 0x03B5, -- GREEK CAPITAL LETTER EPSILON
 [0x0396] = 0x03B6, -- GREEK CAPITAL LETTER ZETA
 [0x0397] = 0x03B7, -- GREEK CAPITAL LETTER ETA
 [0x0398] = 0x03B8, -- GREEK CAPITAL LETTER THETA
 [0x0399] = 0x03B9, -- GREEK CAPITAL LETTER IOTA
 [0x039A] = 0x03BA, -- GREEK CAPITAL LETTER KAPPA
 [0x039B] = 0x03BB, -- GREEK CAPITAL LETTER LAMDA
 [0x039C] = 0x03BC, -- GREEK CAPITAL LETTER MU
 [0x039D] = 0x03BD, -- GREEK CAPITAL LETTER NU
 [0x039E] = 0x03BE, -- GREEK CAPITAL LETTER XI
 [0x039F] = 0x03BF, -- GREEK CAPITAL LETTER OMICRON
 [0x03A0] = 0x03C0, -- GREEK CAPITAL LETTER PI
 [0x03A1] = 0x03C1, -- GREEK CAPITAL LETTER RHO
 [0x03A2] = {status="disallowed"}, -- NA   <reserved-03A2>
 [0x03A3] = 0x03C3, -- GREEK CAPITAL LETTER SIGMA
 [0x03A4] = 0x03C4, -- GREEK CAPITAL LETTER TAU
 [0x03A5] = 0x03C5, -- GREEK CAPITAL LETTER UPSILON
 [0x03A6] = 0x03C6, -- GREEK CAPITAL LETTER PHI
 [0x03A7] = 0x03C7, -- GREEK CAPITAL LETTER CHI
 [0x03A8] = 0x03C8, -- GREEK CAPITAL LETTER PSI
 [0x03A9] = 0x03C9, -- GREEK CAPITAL LETTER OMEGA
 [0x03AA] = 0x03CA, -- GREEK CAPITAL LETTER IOTA WITH DIALYTIKA
 [0x03AB] = 0x03CB, -- GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA
 -- NA   <reserved-0380>..<reserved-0383>
 [0x0380] = {status="disallowed"},
 [0x0381] = {status="disallowed"},
 [0x0382] = {status="disallowed"},
 [0x0383] = {status="disallowed"},
 [0x03C2] = {status='deviation', 0x03C3}, -- GREEK SMALL LETTER FINAL SIGMA
 [0x03CF] = 0x03D7, -- GREEK CAPITAL KAI SYMBOL
 [0x03D0] = 0x03B2, -- GREEK BETA SYMBOL
 [0x03D1] = 0x03B8, -- GREEK THETA SYMBOL
 [0x03D2] = 0x03C5, -- GREEK UPSILON WITH HOOK SYMBOL
 [0x03D3] = 0x03CD, -- GREEK UPSILON WITH ACUTE AND HOOK SYMBOL
 [0x03D4] = 0x03CB, -- GREEK UPSILON WITH DIAERESIS AND HOOK SYMBOL
 [0x03D5] = 0x03C6, -- GREEK PHI SYMBOL
 [0x03D6] = 0x03C0, -- GREEK PI SYMBOL
 [0x03D7] = {status="valid"}, -- GREEK KAI SYMBOL
 [0x03D8] = 0x03D9, -- GREEK LETTER ARCHAIC KOPPA
 [0x03D9] = {status="valid"}, -- GREEK SMALL LETTER ARCHAIC KOPPA
 [0x03DA] = 0x03DB, -- GREEK LETTER STIGMA
 [0x03DB] = {status="valid"}, -- GREEK SMALL LETTER STIGMA
 [0x03DC] = 0x03DD, -- GREEK LETTER DIGAMMA
 [0x03DD] = {status="valid"}, -- GREEK SMALL LETTER DIGAMMA
 [0x03DE] = 0x03DF, -- GREEK LETTER KOPPA
 [0x03DF] = {status="valid"}, -- GREEK SMALL LETTER KOPPA
 [0x03E0] = 0x03E1, -- GREEK LETTER SAMPI
 [0x03E1] = {status="valid"}, -- GREEK SMALL LETTER SAMPI
 [0x03E2] = 0x03E3, -- COPTIC CAPITAL LETTER SHEI
 [0x03E3] = {status="valid"}, -- COPTIC SMALL LETTER SHEI
 [0x03E4] = 0x03E5, -- COPTIC CAPITAL LETTER FEI
 [0x03E5] = {status="valid"}, -- COPTIC SMALL LETTER FEI
 [0x03E6] = 0x03E7, -- COPTIC CAPITAL LETTER KHEI
 [0x03E7] = {status="valid"}, -- COPTIC SMALL LETTER KHEI
 [0x03E8] = 0x03E9, -- COPTIC CAPITAL LETTER HORI
 [0x03E9] = {status="valid"}, -- COPTIC SMALL LETTER HORI
 [0x03EA] = 0x03EB, -- COPTIC CAPITAL LETTER GANGIA
 [0x03EB] = {status="valid"}, -- COPTIC SMALL LETTER GANGIA
 [0x03EC] = 0x03ED, -- COPTIC CAPITAL LETTER SHIMA
 [0x03ED] = {status="valid"}, -- COPTIC SMALL LETTER SHIMA
 [0x03EE] = 0x03EF, -- COPTIC CAPITAL LETTER DEI
 [0x03EF] = {status="valid"}, -- COPTIC SMALL LETTER DEI
 [0x03F0] = 0x03BA, -- GREEK KAPPA SYMBOL
 [0x03F1] = 0x03C1, -- GREEK RHO SYMBOL
 [0x03F2] = 0x03C3, -- GREEK LUNATE SIGMA SYMBOL
 [0x03F3] = {status="valid"}, -- GREEK LETTER YOT
 [0x03F4] = 0x03B8, -- GREEK CAPITAL THETA SYMBOL
 [0x03F5] = 0x03B5, -- GREEK LUNATE EPSILON SYMBOL
 [0x03F6] = {status="disallowed"}, -- GREEK REVERSED LUNATE EPSILON SYMBOL
 [0x03F7] = 0x03F8, -- GREEK CAPITAL LETTER SHO
 [0x03F8] = {status="valid"}, -- GREEK SMALL LETTER SHO
 [0x03F9] = 0x03C3, -- GREEK CAPITAL LUNATE SIGMA SYMBOL
 [0x03FA] = 0x03FB, -- GREEK CAPITAL LETTER SAN
 [0x03FB] = {status="valid"}, -- GREEK SMALL LETTER SAN
 [0x03FC] = {status="valid"}, -- GREEK RHO WITH STROKE SYMBOL
 [0x03FD] = 0x037B, -- GREEK CAPITAL REVERSED LUNATE SIGMA SYMBOL
 [0x03FE] = 0x037C, -- GREEK CAPITAL DOTTED LUNATE SIGMA SYMBOL
 [0x03FF] = 0x037D, -- GREEK CAPITAL REVERSED DOTTED LUNATE SIGMA SYMBOL
 [0x0400] = 0x0450, -- CYRILLIC CAPITAL LETTER IE WITH GRAVE
 [0x0401] = 0x0451, -- CYRILLIC CAPITAL LETTER IO
 [0x0402] = 0x0452, -- CYRILLIC CAPITAL LETTER DJE
 [0x0403] = 0x0453, -- CYRILLIC CAPITAL LETTER GJE
 [0x0404] = 0x0454, -- CYRILLIC CAPITAL LETTER UKRAINIAN IE
 [0x0405] = 0x0455, -- CYRILLIC CAPITAL LETTER DZE
 [0x0406] = 0x0456, -- CYRILLIC CAPITAL LETTER BYELORUSSIAN-UKRAINIAN I
 [0x0407] = 0x0457, -- CYRILLIC CAPITAL LETTER YI
 [0x0408] = 0x0458, -- CYRILLIC CAPITAL LETTER JE
 [0x0409] = 0x0459, -- CYRILLIC CAPITAL LETTER LJE
 [0x040A] = 0x045A, -- CYRILLIC CAPITAL LETTER NJE
 [0x040B] = 0x045B, -- CYRILLIC CAPITAL LETTER TSHE
 [0x040C] = 0x045C, -- CYRILLIC CAPITAL LETTER KJE
 [0x040D] = 0x045D, -- CYRILLIC CAPITAL LETTER I WITH GRAVE
 [0x040E] = 0x045E, -- CYRILLIC CAPITAL LETTER SHORT U
 [0x040F] = 0x045F, -- CYRILLIC CAPITAL LETTER DZHE
 [0x0410] = 0x0430, -- CYRILLIC CAPITAL LETTER A
 [0x0411] = 0x0431, -- CYRILLIC CAPITAL LETTER BE
 [0x0412] = 0x0432, -- CYRILLIC CAPITAL LETTER VE
 [0x0413] = 0x0433, -- CYRILLIC CAPITAL LETTER GHE
 [0x0414] = 0x0434, -- CYRILLIC CAPITAL LETTER DE
 [0x0415] = 0x0435, -- CYRILLIC CAPITAL LETTER IE
 [0x0416] = 0x0436, -- CYRILLIC CAPITAL LETTER ZHE
 [0x0417] = 0x0437, -- CYRILLIC CAPITAL LETTER ZE
 [0x0418] = 0x0438, -- CYRILLIC CAPITAL LETTER I
 [0x0419] = 0x0439, -- CYRILLIC CAPITAL LETTER SHORT I
 [0x041A] = 0x043A, -- CYRILLIC CAPITAL LETTER KA
 [0x041B] = 0x043B, -- CYRILLIC CAPITAL LETTER EL
 [0x041C] = 0x043C, -- CYRILLIC CAPITAL LETTER EM
 [0x041D] = 0x043D, -- CYRILLIC CAPITAL LETTER EN
 [0x041E] = 0x043E, -- CYRILLIC CAPITAL LETTER O
 [0x041F] = 0x043F, -- CYRILLIC CAPITAL LETTER PE
 [0x0420] = 0x0440, -- CYRILLIC CAPITAL LETTER ER
 [0x0421] = 0x0441, -- CYRILLIC CAPITAL LETTER ES
 [0x0422] = 0x0442, -- CYRILLIC CAPITAL LETTER TE
 [0x0423] = 0x0443, -- CYRILLIC CAPITAL LETTER U
 [0x0424] = 0x0444, -- CYRILLIC CAPITAL LETTER EF
 [0x0425] = 0x0445, -- CYRILLIC CAPITAL LETTER HA
 [0x0426] = 0x0446, -- CYRILLIC CAPITAL LETTER TSE
 [0x0427] = 0x0447, -- CYRILLIC CAPITAL LETTER CHE
 [0x0428] = 0x0448, -- CYRILLIC CAPITAL LETTER SHA
 [0x0429] = 0x0449, -- CYRILLIC CAPITAL LETTER SHCHA
 [0x042A] = 0x044A, -- CYRILLIC CAPITAL LETTER HARD SIGN
 [0x042B] = 0x044B, -- CYRILLIC CAPITAL LETTER YERU
 [0x042C] = 0x044C, -- CYRILLIC CAPITAL LETTER SOFT SIGN
 [0x042D] = 0x044D, -- CYRILLIC CAPITAL LETTER E
 [0x042E] = 0x044E, -- CYRILLIC CAPITAL LETTER YU
 [0x042F] = 0x044F, -- CYRILLIC CAPITAL LETTER YA
 [0x0450] = {status="valid"}, -- CYRILLIC SMALL LETTER IE WITH GRAVE
 [0x045D] = {status="valid"}, -- CYRILLIC SMALL LETTER I WITH GRAVE
 [0x0460] = 0x0461, -- CYRILLIC CAPITAL LETTER OMEGA
 [0x0461] = {status="valid"}, -- CYRILLIC SMALL LETTER OMEGA
 [0x0462] = 0x0463, -- CYRILLIC CAPITAL LETTER YAT
 [0x0463] = {status="valid"}, -- CYRILLIC SMALL LETTER YAT
 [0x0464] = 0x0465, -- CYRILLIC CAPITAL LETTER IOTIFIED E
 [0x0465] = {status="valid"}, -- CYRILLIC SMALL LETTER IOTIFIED E
 [0x0466] = 0x0467, -- CYRILLIC CAPITAL LETTER LITTLE YUS
 [0x0467] = {status="valid"}, -- CYRILLIC SMALL LETTER LITTLE YUS
 [0x0468] = 0x0469, -- CYRILLIC CAPITAL LETTER IOTIFIED LITTLE YUS
 [0x0469] = {status="valid"}, -- CYRILLIC SMALL LETTER IOTIFIED LITTLE YUS
 [0x046A] = 0x046B, -- CYRILLIC CAPITAL LETTER BIG YUS
 [0x046B] = {status="valid"}, -- CYRILLIC SMALL LETTER BIG YUS
 [0x046C] = 0x046D, -- CYRILLIC CAPITAL LETTER IOTIFIED BIG YUS
 [0x046D] = {status="valid"}, -- CYRILLIC SMALL LETTER IOTIFIED BIG YUS
 [0x046E] = 0x046F, -- CYRILLIC CAPITAL LETTER KSI
 [0x046F] = {status="valid"}, -- CYRILLIC SMALL LETTER KSI
 [0x0470] = 0x0471, -- CYRILLIC CAPITAL LETTER PSI
 [0x0471] = {status="valid"}, -- CYRILLIC SMALL LETTER PSI
 [0x0472] = 0x0473, -- CYRILLIC CAPITAL LETTER FITA
 [0x0473] = {status="valid"}, -- CYRILLIC SMALL LETTER FITA
 [0x0474] = 0x0475, -- CYRILLIC CAPITAL LETTER IZHITSA
 [0x0475] = {status="valid"}, -- CYRILLIC SMALL LETTER IZHITSA
 [0x0476] = 0x0477, -- CYRILLIC CAPITAL LETTER IZHITSA WITH DOUBLE GRAVE ACCENT
 [0x0477] = {status="valid"}, -- CYRILLIC SMALL LETTER IZHITSA WITH DOUBLE GRAVE ACCENT
 [0x0478] = 0x0479, -- CYRILLIC CAPITAL LETTER UK
 [0x0479] = {status="valid"}, -- CYRILLIC SMALL LETTER UK
 [0x047A] = 0x047B, -- CYRILLIC CAPITAL LETTER ROUND OMEGA
 [0x047B] = {status="valid"}, -- CYRILLIC SMALL LETTER ROUND OMEGA
 [0x047C] = 0x047D, -- CYRILLIC CAPITAL LETTER OMEGA WITH TITLO
 [0x047D] = {status="valid"}, -- CYRILLIC SMALL LETTER OMEGA WITH TITLO
 [0x047E] = 0x047F, -- CYRILLIC CAPITAL LETTER OT
 [0x047F] = {status="valid"}, -- CYRILLIC SMALL LETTER OT
 [0x0480] = 0x0481, -- CYRILLIC CAPITAL LETTER KOPPA
 [0x0481] = {status="valid"}, -- CYRILLIC SMALL LETTER KOPPA
 [0x0482] = {status="disallowed"}, -- CYRILLIC THOUSANDS SIGN
 -- CYRILLIC SMALL LETTER SHORT U..CYRILLIC SMALL LETTER DZHE
 [0x045E] = {status="valid"},
 [0x045F] = {status="valid"},
 [0x0487] = {status="valid"}, -- COMBINING CYRILLIC POKRYTIE
 -- COMBINING CYRILLIC TITLO..COMBINING CYRILLIC PSILI PNEUMATA
 [0x0483] = {status="valid"},
 [0x0484] = {status="valid"},
 [0x0485] = {status="valid"},
 [0x0486] = {status="valid"},
 [0x048A] = 0x048B, -- CYRILLIC CAPITAL LETTER SHORT I WITH TAIL
 [0x048B] = {status="valid"}, -- CYRILLIC SMALL LETTER SHORT I WITH TAIL
 [0x048C] = 0x048D, -- CYRILLIC CAPITAL LETTER SEMISOFT SIGN
 [0x048D] = {status="valid"}, -- CYRILLIC SMALL LETTER SEMISOFT SIGN
 [0x048E] = 0x048F, -- CYRILLIC CAPITAL LETTER ER WITH TICK
 [0x048F] = {status="valid"}, -- CYRILLIC SMALL LETTER ER WITH TICK
 [0x0490] = 0x0491, -- CYRILLIC CAPITAL LETTER GHE WITH UPTURN
 [0x0491] = {status="valid"}, -- CYRILLIC SMALL LETTER GHE WITH UPTURN
 [0x0492] = 0x0493, -- CYRILLIC CAPITAL LETTER GHE WITH STROKE
 [0x0493] = {status="valid"}, -- CYRILLIC SMALL LETTER GHE WITH STROKE
 [0x0494] = 0x0495, -- CYRILLIC CAPITAL LETTER GHE WITH MIDDLE HOOK
 [0x0495] = {status="valid"}, -- CYRILLIC SMALL LETTER GHE WITH MIDDLE HOOK
 [0x0496] = 0x0497, -- CYRILLIC CAPITAL LETTER ZHE WITH DESCENDER
 [0x0497] = {status="valid"}, -- CYRILLIC SMALL LETTER ZHE WITH DESCENDER
 [0x0498] = 0x0499, -- CYRILLIC CAPITAL LETTER ZE WITH DESCENDER
 [0x0499] = {status="valid"}, -- CYRILLIC SMALL LETTER ZE WITH DESCENDER
 [0x049A] = 0x049B, -- CYRILLIC CAPITAL LETTER KA WITH DESCENDER
 [0x049B] = {status="valid"}, -- CYRILLIC SMALL LETTER KA WITH DESCENDER
 [0x049C] = 0x049D, -- CYRILLIC CAPITAL LETTER KA WITH VERTICAL STROKE
 [0x049D] = {status="valid"}, -- CYRILLIC SMALL LETTER KA WITH VERTICAL STROKE
 [0x049E] = 0x049F, -- CYRILLIC CAPITAL LETTER KA WITH STROKE
 [0x049F] = {status="valid"}, -- CYRILLIC SMALL LETTER KA WITH STROKE
 [0x04A0] = 0x04A1, -- CYRILLIC CAPITAL LETTER BASHKIR KA
 [0x04A1] = {status="valid"}, -- CYRILLIC SMALL LETTER BASHKIR KA
 [0x04A2] = 0x04A3, -- CYRILLIC CAPITAL LETTER EN WITH DESCENDER
 [0x04A3] = {status="valid"}, -- CYRILLIC SMALL LETTER EN WITH DESCENDER
 [0x04A4] = 0x04A5, -- CYRILLIC CAPITAL LIGATURE EN GHE
 [0x04A5] = {status="valid"}, -- CYRILLIC SMALL LIGATURE EN GHE
 [0x04A6] = 0x04A7, -- CYRILLIC CAPITAL LETTER PE WITH MIDDLE HOOK
 [0x04A7] = {status="valid"}, -- CYRILLIC SMALL LETTER PE WITH MIDDLE HOOK
 [0x04A8] = 0x04A9, -- CYRILLIC CAPITAL LETTER ABKHASIAN HA
 [0x04A9] = {status="valid"}, -- CYRILLIC SMALL LETTER ABKHASIAN HA
 [0x04AA] = 0x04AB, -- CYRILLIC CAPITAL LETTER ES WITH DESCENDER
 [0x04AB] = {status="valid"}, -- CYRILLIC SMALL LETTER ES WITH DESCENDER
 [0x04AC] = 0x04AD, -- CYRILLIC CAPITAL LETTER TE WITH DESCENDER
 [0x04AD] = {status="valid"}, -- CYRILLIC SMALL LETTER TE WITH DESCENDER
 [0x04AE] = 0x04AF, -- CYRILLIC CAPITAL LETTER STRAIGHT U
 [0x04AF] = {status="valid"}, -- CYRILLIC SMALL LETTER STRAIGHT U
 [0x04B0] = 0x04B1, -- CYRILLIC CAPITAL LETTER STRAIGHT U WITH STROKE
 [0x04B1] = {status="valid"}, -- CYRILLIC SMALL LETTER STRAIGHT U WITH STROKE
 [0x04B2] = 0x04B3, -- CYRILLIC CAPITAL LETTER HA WITH DESCENDER
 [0x04B3] = {status="valid"}, -- CYRILLIC SMALL LETTER HA WITH DESCENDER
 [0x04B4] = 0x04B5, -- CYRILLIC CAPITAL LIGATURE TE TSE
 [0x04B5] = {status="valid"}, -- CYRILLIC SMALL LIGATURE TE TSE
 [0x04B6] = 0x04B7, -- CYRILLIC CAPITAL LETTER CHE WITH DESCENDER
 [0x04B7] = {status="valid"}, -- CYRILLIC SMALL LETTER CHE WITH DESCENDER
 [0x04B8] = 0x04B9, -- CYRILLIC CAPITAL LETTER CHE WITH VERTICAL STROKE
 [0x04B9] = {status="valid"}, -- CYRILLIC SMALL LETTER CHE WITH VERTICAL STROKE
 [0x04BA] = 0x04BB, -- CYRILLIC CAPITAL LETTER SHHA
 [0x04BB] = {status="valid"}, -- CYRILLIC SMALL LETTER SHHA
 [0x04BC] = 0x04BD, -- CYRILLIC CAPITAL LETTER ABKHASIAN CHE
 [0x04BD] = {status="valid"}, -- CYRILLIC SMALL LETTER ABKHASIAN CHE
 [0x04BE] = 0x04BF, -- CYRILLIC CAPITAL LETTER ABKHASIAN CHE WITH DESCENDER
 [0x04BF] = {status="valid"}, -- CYRILLIC SMALL LETTER ABKHASIAN CHE WITH DESCENDER
 [0x04C0] = {status="disallowed"}, -- CYRILLIC LETTER PALOCHKA
 [0x04C1] = 0x04C2, -- CYRILLIC CAPITAL LETTER ZHE WITH BREVE
 [0x04C2] = {status="valid"}, -- CYRILLIC SMALL LETTER ZHE WITH BREVE
 [0x04C3] = 0x04C4, -- CYRILLIC CAPITAL LETTER KA WITH HOOK
 [0x04C4] = {status="valid"}, -- CYRILLIC SMALL LETTER KA WITH HOOK
 [0x04C5] = 0x04C6, -- CYRILLIC CAPITAL LETTER EL WITH TAIL
 [0x04C6] = {status="valid"}, -- CYRILLIC SMALL LETTER EL WITH TAIL
 [0x04C7] = 0x04C8, -- CYRILLIC CAPITAL LETTER EN WITH HOOK
 [0x04C8] = {status="valid"}, -- CYRILLIC SMALL LETTER EN WITH HOOK
 [0x04C9] = 0x04CA, -- CYRILLIC CAPITAL LETTER EN WITH TAIL
 [0x04CA] = {status="valid"}, -- CYRILLIC SMALL LETTER EN WITH TAIL
 [0x04CB] = 0x04CC, -- CYRILLIC CAPITAL LETTER KHAKASSIAN CHE
 [0x04CC] = {status="valid"}, -- CYRILLIC SMALL LETTER KHAKASSIAN CHE
 [0x04CD] = 0x04CE, -- CYRILLIC CAPITAL LETTER EM WITH TAIL
 [0x04CE] = {status="valid"}, -- CYRILLIC SMALL LETTER EM WITH TAIL
 [0x04CF] = {status="valid"}, -- CYRILLIC SMALL LETTER PALOCHKA
 [0x04D0] = 0x04D1, -- CYRILLIC CAPITAL LETTER A WITH BREVE
 [0x04D1] = {status="valid"}, -- CYRILLIC SMALL LETTER A WITH BREVE
 [0x04D2] = 0x04D3, -- CYRILLIC CAPITAL LETTER A WITH DIAERESIS
 [0x04D3] = {status="valid"}, -- CYRILLIC SMALL LETTER A WITH DIAERESIS
 [0x04D4] = 0x04D5, -- CYRILLIC CAPITAL LIGATURE A IE
 [0x04D5] = {status="valid"}, -- CYRILLIC SMALL LIGATURE A IE
 [0x04D6] = 0x04D7, -- CYRILLIC CAPITAL LETTER IE WITH BREVE
 [0x04D7] = {status="valid"}, -- CYRILLIC SMALL LETTER IE WITH BREVE
 [0x04D8] = 0x04D9, -- CYRILLIC CAPITAL LETTER SCHWA
 [0x04D9] = {status="valid"}, -- CYRILLIC SMALL LETTER SCHWA
 [0x04DA] = 0x04DB, -- CYRILLIC CAPITAL LETTER SCHWA WITH DIAERESIS
 [0x04DB] = {status="valid"}, -- CYRILLIC SMALL LETTER SCHWA WITH DIAERESIS
 [0x04DC] = 0x04DD, -- CYRILLIC CAPITAL LETTER ZHE WITH DIAERESIS
 [0x04DD] = {status="valid"}, -- CYRILLIC SMALL LETTER ZHE WITH DIAERESIS
 [0x04DE] = 0x04DF, -- CYRILLIC CAPITAL LETTER ZE WITH DIAERESIS
 [0x04DF] = {status="valid"}, -- CYRILLIC SMALL LETTER ZE WITH DIAERESIS
 [0x04E0] = 0x04E1, -- CYRILLIC CAPITAL LETTER ABKHASIAN DZE
 [0x04E1] = {status="valid"}, -- CYRILLIC SMALL LETTER ABKHASIAN DZE
 [0x04E2] = 0x04E3, -- CYRILLIC CAPITAL LETTER I WITH MACRON
 [0x04E3] = {status="valid"}, -- CYRILLIC SMALL LETTER I WITH MACRON
 [0x04E4] = 0x04E5, -- CYRILLIC CAPITAL LETTER I WITH DIAERESIS
 [0x04E5] = {status="valid"}, -- CYRILLIC SMALL LETTER I WITH DIAERESIS
 [0x04E6] = 0x04E7, -- CYRILLIC CAPITAL LETTER O WITH DIAERESIS
 [0x04E7] = {status="valid"}, -- CYRILLIC SMALL LETTER O WITH DIAERESIS
 [0x04E8] = 0x04E9, -- CYRILLIC CAPITAL LETTER BARRED O
 [0x04E9] = {status="valid"}, -- CYRILLIC SMALL LETTER BARRED O
 [0x04EA] = 0x04EB, -- CYRILLIC CAPITAL LETTER BARRED O WITH DIAERESIS
 [0x04EB] = {status="valid"}, -- CYRILLIC SMALL LETTER BARRED O WITH DIAERESIS
 [0x04EC] = 0x04ED, -- CYRILLIC CAPITAL LETTER E WITH DIAERESIS
 [0x04ED] = {status="valid"}, -- CYRILLIC SMALL LETTER E WITH DIAERESIS
 [0x04EE] = 0x04EF, -- CYRILLIC CAPITAL LETTER U WITH MACRON
 [0x04EF] = {status="valid"}, -- CYRILLIC SMALL LETTER U WITH MACRON
 [0x04F0] = 0x04F1, -- CYRILLIC CAPITAL LETTER U WITH DIAERESIS
 [0x04F1] = {status="valid"}, -- CYRILLIC SMALL LETTER U WITH DIAERESIS
 [0x04F2] = 0x04F3, -- CYRILLIC CAPITAL LETTER U WITH DOUBLE ACUTE
 [0x04F3] = {status="valid"}, -- CYRILLIC SMALL LETTER U WITH DOUBLE ACUTE
 [0x04F4] = 0x04F5, -- CYRILLIC CAPITAL LETTER CHE WITH DIAERESIS
 [0x04F5] = {status="valid"}, -- CYRILLIC SMALL LETTER CHE WITH DIAERESIS
 [0x04F6] = 0x04F7, -- CYRILLIC CAPITAL LETTER GHE WITH DESCENDER
 [0x04F7] = {status="valid"}, -- CYRILLIC SMALL LETTER GHE WITH DESCENDER
 [0x04F8] = 0x04F9, -- CYRILLIC CAPITAL LETTER YERU WITH DIAERESIS
 [0x04F9] = {status="valid"}, -- CYRILLIC SMALL LETTER YERU WITH DIAERESIS
 [0x04FA] = 0x04FB, -- CYRILLIC CAPITAL LETTER GHE WITH STROKE AND HOOK
 [0x04FB] = {status="valid"}, -- CYRILLIC SMALL LETTER GHE WITH STROKE AND HOOK
 [0x04FC] = 0x04FD, -- CYRILLIC CAPITAL LETTER HA WITH HOOK
 [0x04FD] = {status="valid"}, -- CYRILLIC SMALL LETTER HA WITH HOOK
 [0x04FE] = 0x04FF, -- CYRILLIC CAPITAL LETTER HA WITH STROKE
 [0x04FF] = {status="valid"}, -- CYRILLIC SMALL LETTER HA WITH STROKE
 [0x0500] = 0x0501, -- CYRILLIC CAPITAL LETTER KOMI DE
 [0x0501] = {status="valid"}, -- CYRILLIC SMALL LETTER KOMI DE
 [0x0502] = 0x0503, -- CYRILLIC CAPITAL LETTER KOMI DJE
 [0x0503] = {status="valid"}, -- CYRILLIC SMALL LETTER KOMI DJE
 [0x0504] = 0x0505, -- CYRILLIC CAPITAL LETTER KOMI ZJE
 [0x0505] = {status="valid"}, -- CYRILLIC SMALL LETTER KOMI ZJE
 [0x0506] = 0x0507, -- CYRILLIC CAPITAL LETTER KOMI DZJE
 [0x0507] = {status="valid"}, -- CYRILLIC SMALL LETTER KOMI DZJE
 [0x0508] = 0x0509, -- CYRILLIC CAPITAL LETTER KOMI LJE
 [0x0509] = {status="valid"}, -- CYRILLIC SMALL LETTER KOMI LJE
 [0x050A] = 0x050B, -- CYRILLIC CAPITAL LETTER KOMI NJE
 [0x050B] = {status="valid"}, -- CYRILLIC SMALL LETTER KOMI NJE
 [0x050C] = 0x050D, -- CYRILLIC CAPITAL LETTER KOMI SJE
 [0x050D] = {status="valid"}, -- CYRILLIC SMALL LETTER KOMI SJE
 [0x050E] = 0x050F, -- CYRILLIC CAPITAL LETTER KOMI TJE
 [0x050F] = {status="valid"}, -- CYRILLIC SMALL LETTER KOMI TJE
 [0x0510] = 0x0511, -- CYRILLIC CAPITAL LETTER REVERSED ZE
 [0x0511] = {status="valid"}, -- CYRILLIC SMALL LETTER REVERSED ZE
 [0x0512] = 0x0513, -- CYRILLIC CAPITAL LETTER EL WITH HOOK
 [0x0513] = {status="valid"}, -- CYRILLIC SMALL LETTER EL WITH HOOK
 [0x0514] = 0x0515, -- CYRILLIC CAPITAL LETTER LHA
 [0x0515] = {status="valid"}, -- CYRILLIC SMALL LETTER LHA
 [0x0516] = 0x0517, -- CYRILLIC CAPITAL LETTER RHA
 [0x0517] = {status="valid"}, -- CYRILLIC SMALL LETTER RHA
 [0x0518] = 0x0519, -- CYRILLIC CAPITAL LETTER YAE
 [0x0519] = {status="valid"}, -- CYRILLIC SMALL LETTER YAE
 [0x051A] = 0x051B, -- CYRILLIC CAPITAL LETTER QA
 [0x051B] = {status="valid"}, -- CYRILLIC SMALL LETTER QA
 [0x051C] = 0x051D, -- CYRILLIC CAPITAL LETTER WE
 [0x051D] = {status="valid"}, -- CYRILLIC SMALL LETTER WE
 [0x051E] = 0x051F, -- CYRILLIC CAPITAL LETTER ALEUT KA
 [0x051F] = {status="valid"}, -- CYRILLIC SMALL LETTER ALEUT KA
 [0x0520] = 0x0521, -- CYRILLIC CAPITAL LETTER EL WITH MIDDLE HOOK
 [0x0521] = {status="valid"}, -- CYRILLIC SMALL LETTER EL WITH MIDDLE HOOK
 [0x0522] = 0x0523, -- CYRILLIC CAPITAL LETTER EN WITH MIDDLE HOOK
 [0x0523] = {status="valid"}, -- CYRILLIC SMALL LETTER EN WITH MIDDLE HOOK
 [0x0524] = 0x0525, -- CYRILLIC CAPITAL LETTER PE WITH DESCENDER
 [0x0525] = {status="valid"}, -- CYRILLIC SMALL LETTER PE WITH DESCENDER
 [0x0526] = 0x0527, -- CYRILLIC CAPITAL LETTER SHHA WITH DESCENDER
 [0x0527] = {status="valid"}, -- CYRILLIC SMALL LETTER SHHA WITH DESCENDER
 [0x0528] = 0x0529, -- CYRILLIC CAPITAL LETTER EN WITH LEFT HOOK
 [0x0529] = {status="valid"}, -- CYRILLIC SMALL LETTER EN WITH LEFT HOOK
 [0x052A] = 0x052B, -- CYRILLIC CAPITAL LETTER DZZHE
 [0x052B] = {status="valid"}, -- CYRILLIC SMALL LETTER DZZHE
 [0x052C] = 0x052D, -- CYRILLIC CAPITAL LETTER DCHE
 [0x052D] = {status="valid"}, -- CYRILLIC SMALL LETTER DCHE
 [0x052E] = 0x052F, -- CYRILLIC CAPITAL LETTER EL WITH DESCENDER
 [0x052F] = {status="valid"}, -- CYRILLIC SMALL LETTER EL WITH DESCENDER
 [0x0530] = {status="disallowed"}, -- NA   <reserved-0530>
 [0x0531] = 0x0561, -- ARMENIAN CAPITAL LETTER AYB
 [0x0532] = 0x0562, -- ARMENIAN CAPITAL LETTER BEN
 [0x0533] = 0x0563, -- ARMENIAN CAPITAL LETTER GIM
 [0x0534] = 0x0564, -- ARMENIAN CAPITAL LETTER DA
 [0x0535] = 0x0565, -- ARMENIAN CAPITAL LETTER ECH
 [0x0536] = 0x0566, -- ARMENIAN CAPITAL LETTER ZA
 [0x0537] = 0x0567, -- ARMENIAN CAPITAL LETTER EH
 [0x0538] = 0x0568, -- ARMENIAN CAPITAL LETTER ET
 [0x0539] = 0x0569, -- ARMENIAN CAPITAL LETTER TO
 [0x053A] = 0x056A, -- ARMENIAN CAPITAL LETTER ZHE
 [0x053B] = 0x056B, -- ARMENIAN CAPITAL LETTER INI
 [0x053C] = 0x056C, -- ARMENIAN CAPITAL LETTER LIWN
 [0x053D] = 0x056D, -- ARMENIAN CAPITAL LETTER XEH
 [0x053E] = 0x056E, -- ARMENIAN CAPITAL LETTER CA
 [0x053F] = 0x056F, -- ARMENIAN CAPITAL LETTER KEN
 [0x0540] = 0x0570, -- ARMENIAN CAPITAL LETTER HO
 [0x0541] = 0x0571, -- ARMENIAN CAPITAL LETTER JA
 [0x0542] = 0x0572, -- ARMENIAN CAPITAL LETTER GHAD
 [0x0543] = 0x0573, -- ARMENIAN CAPITAL LETTER CHEH
 [0x0544] = 0x0574, -- ARMENIAN CAPITAL LETTER MEN
 [0x0545] = 0x0575, -- ARMENIAN CAPITAL LETTER YI
 [0x0546] = 0x0576, -- ARMENIAN CAPITAL LETTER NOW
 [0x0547] = 0x0577, -- ARMENIAN CAPITAL LETTER SHA
 [0x0548] = 0x0578, -- ARMENIAN CAPITAL LETTER VO
 [0x0549] = 0x0579, -- ARMENIAN CAPITAL LETTER CHA
 [0x054A] = 0x057A, -- ARMENIAN CAPITAL LETTER PEH
 [0x054B] = 0x057B, -- ARMENIAN CAPITAL LETTER JHEH
 [0x054C] = 0x057C, -- ARMENIAN CAPITAL LETTER RA
 [0x054D] = 0x057D, -- ARMENIAN CAPITAL LETTER SEH
 [0x054E] = 0x057E, -- ARMENIAN CAPITAL LETTER VEW
 [0x054F] = 0x057F, -- ARMENIAN CAPITAL LETTER TIWN
 [0x0550] = 0x0580, -- ARMENIAN CAPITAL LETTER REH
 [0x0551] = 0x0581, -- ARMENIAN CAPITAL LETTER CO
 [0x0552] = 0x0582, -- ARMENIAN CAPITAL LETTER YIWN
 [0x0553] = 0x0583, -- ARMENIAN CAPITAL LETTER PIWR
 [0x0554] = 0x0584, -- ARMENIAN CAPITAL LETTER KEH
 [0x0555] = 0x0585, -- ARMENIAN CAPITAL LETTER OH
 [0x0556] = 0x0586, -- ARMENIAN CAPITAL LETTER FEH
 -- COMBINING CYRILLIC HUNDRED THOUSANDS SIGN..COMBINING CYRILLIC MILLIONS SIGN
 [0x0488] = {status="disallowed"},
 [0x0489] = {status="disallowed"},
 [0x0559] = {status="valid"}, -- ARMENIAN MODIFIER LETTER LEFT HALF RING
 -- NA   <reserved-0557>..<reserved-0558>
 [0x0557] = {status="disallowed"},
 [0x0558] = {status="disallowed"},
 [0x0560] = {status="disallowed"}, -- NA   <reserved-0560>
 [0x0587] = {0x0565, 0x0582}, -- ARMENIAN SMALL LIGATURE ECH YIWN
 [0x0588] = {status="disallowed"}, -- NA   <reserved-0588>
 [0x0589] = {status="disallowed"}, -- ARMENIAN FULL STOP
 [0x058A] = {status="disallowed"}, -- ARMENIAN HYPHEN
 -- NA   <reserved-058B>..<reserved-058C>
 [0x058B] = {status="disallowed"},
 [0x058C] = {status="disallowed"},
 [0x058F] = {status="disallowed"}, -- ARMENIAN DRAM SIGN
 [0x0590] = {status="disallowed"}, -- NA   <reserved-0590>
 -- RIGHT-FACING ARMENIAN ETERNITY SIGN..LEFT-FACING ARMENIAN ETERNITY SIGN
 [0x058D] = {status="disallowed"},
 [0x058E] = {status="disallowed"},
 [0x05A2] = {status="valid"}, -- HEBREW ACCENT ATNAH HAFUKH
 [0x05BA] = {status="valid"}, -- HEBREW POINT HOLAM HASER FOR VAV
 [0x05BE] = {status="disallowed"}, -- HEBREW PUNCTUATION MAQAF
 [0x05BF] = {status="valid"}, -- HEBREW POINT RAFE
 [0x05C0] = {status="disallowed"}, -- HEBREW PUNCTUATION PASEQ
 -- HEBREW POINT QUBUTS..HEBREW POINT METEG
 [0x05BB] = {status="valid"},
 [0x05BC] = {status="valid"},
 [0x05BD] = {status="valid"},
 [0x05C3] = {status="disallowed"}, -- HEBREW PUNCTUATION SOF PASUQ
 [0x05C4] = {status="valid"}, -- HEBREW MARK UPPER DOT
 [0x05C5] = {status="valid"}, -- HEBREW MARK LOWER DOT
 [0x05C6] = {status="disallowed"}, -- HEBREW PUNCTUATION NUN HAFUKHA
 [0x05C7] = {status="valid"}, -- HEBREW POINT QAMATS QATAN
 -- HEBREW POINT SHIN DOT..HEBREW POINT SIN DOT
 [0x05C1] = {status="valid"},
 [0x05C2] = {status="valid"},
 -- NA   <reserved-05EB>..<reserved-05EF>
 [0x05EB] = {status="disallowed"},
 [0x05EC] = {status="disallowed"},
 [0x05ED] = {status="disallowed"},
 [0x05EE] = {status="disallowed"},
 [0x05EF] = {status="disallowed"},
 -- HEBREW LIGATURE YIDDISH DOUBLE VAV..HEBREW PUNCTUATION GERSHAYIM
 [0x05F0] = {status="valid"},
 [0x05F1] = {status="valid"},
 [0x05F2] = {status="valid"},
 [0x05F3] = {status="valid"},
 [0x05F4] = {status="valid"},
 [0x0604] = {status="disallowed"}, -- ARABIC SIGN SAMVAT
 [0x0605] = {status="disallowed"}, -- ARABIC NUMBER MARK ABOVE
 [0x060B] = {status="disallowed"}, -- AFGHANI SIGN
 [0x060C] = {status="disallowed"}, -- ARABIC COMMA
 -- ARABIC-INDIC CUBE ROOT..ARABIC-INDIC PER TEN THOUSAND SIGN
 [0x0606] = {status="disallowed"},
 [0x0607] = {status="disallowed"},
 [0x0608] = {status="disallowed"},
 [0x0609] = {status="disallowed"},
 [0x060A] = {status="disallowed"},
 -- ARABIC DATE SEPARATOR..ARABIC SIGN MISRA
 [0x060D] = {status="disallowed"},
 [0x060E] = {status="disallowed"},
 [0x060F] = {status="disallowed"},
 [0x061B] = {status="disallowed"}, -- ARABIC SEMICOLON
 [0x061C] = {status="disallowed"}, -- ARABIC LETTER MARK
 [0x061D] = {status="disallowed"}, -- NA   <reserved-061D>
 [0x061E] = {status="disallowed"}, -- ARABIC TRIPLE DOT PUNCTUATION MARK
 [0x061F] = {status="disallowed"}, -- ARABIC QUESTION MARK
 [0x0620] = {status="valid"}, -- ARABIC LETTER KASHMIRI YEH
 [0x0640] = {status="disallowed"}, -- ARABIC TATWEEL
 [0x065F] = {status="valid"}, -- ARABIC WAVY HAMZA BELOW
 -- ARABIC PERCENT SIGN..ARABIC FIVE POINTED STAR
 [0x066A] = {status="disallowed"},
 [0x066B] = {status="disallowed"},
 [0x066C] = {status="disallowed"},
 [0x066D] = {status="disallowed"},
 [0x0675] = {0x0627, 0x0674}, -- ARABIC LETTER HIGH HAMZA ALEF
 [0x0676] = {0x0648, 0x0674}, -- ARABIC LETTER HIGH HAMZA WAW
 [0x0677] = {0x06C7, 0x0674}, -- ARABIC LETTER U WITH HAMZA ABOVE
 [0x0678] = {0x064A, 0x0674}, -- ARABIC LETTER HIGH HAMZA YEH
 [0x06BF] = {status="valid"}, -- ARABIC LETTER TCHEH WITH DOT ABOVE
 [0x06CF] = {status="valid"}, -- ARABIC LETTER WAW WITH DOT ABOVE
 [0x06D4] = {status="disallowed"}, -- ARABIC FULL STOP
 -- ARABIC LETTER E..ARABIC LETTER YEH BARREE WITH HAMZA ABOVE
 [0x06D0] = {status="valid"},
 [0x06D1] = {status="valid"},
 [0x06D2] = {status="valid"},
 [0x06D3] = {status="valid"},
 [0x06DD] = {status="disallowed"}, -- ARABIC END OF AYAH
 [0x06DE] = {status="disallowed"}, -- ARABIC START OF RUB EL HIZB
 [0x06E9] = {status="disallowed"}, -- ARABIC PLACE OF SAJDAH
 [0x06FF] = {status="valid"}, -- ARABIC LETTER HEH WITH INVERTED V
 [0x070E] = {status="disallowed"}, -- NA   <reserved-070E>
 [0x070F] = {status="disallowed"}, -- SYRIAC ABBREVIATION MARK
 -- NA   <reserved-074B>..<reserved-074C>
 [0x074B] = {status="disallowed"},
 [0x074C] = {status="disallowed"},
 [0x07B1] = {status="valid"}, -- THAANA LETTER NAA
 -- NKO SYMBOL OO DENNEN..NKO LAJANYALAN
 [0x07F6] = {status="disallowed"},
 [0x07F7] = {status="disallowed"},
 [0x07F8] = {status="disallowed"},
 [0x07F9] = {status="disallowed"},
 [0x07FA] = {status="disallowed"},
 -- NA   <reserved-07FB>..<reserved-07FF>
 [0x07FB] = {status="disallowed"},
 [0x07FC] = {status="disallowed"},
 [0x07FD] = {status="disallowed"},
 [0x07FE] = {status="disallowed"},
 [0x07FF] = {status="disallowed"},
 -- NA   <reserved-082E>..<reserved-082F>
 [0x082E] = {status="disallowed"},
 [0x082F] = {status="disallowed"},
 [0x083F] = {status="disallowed"}, -- NA   <reserved-083F>
 [0x085E] = {status="disallowed"}, -- MANDAIC PUNCTUATION
 [0x085F] = {status="disallowed"}, -- NA   <reserved-085F>
 -- NA   <reserved-085C>..<reserved-085D>
 [0x085C] = {status="disallowed"},
 [0x085D] = {status="disallowed"},
 [0x08A0] = {status="valid"}, -- ARABIC LETTER BEH WITH SMALL V BELOW
 [0x08A1] = {status="valid"}, -- ARABIC LETTER BEH WITH HAMZA ABOVE
 [0x08B5] = {status="disallowed"}, -- NA   <reserved-08B5>
 [0x08E2] = {status="disallowed"}, -- ARABIC DISPUTED END OF AYAH
 [0x08E3] = {status="valid"}, -- ARABIC TURNED DAMMA BELOW
 [0x08FF] = {status="valid"}, -- ARABIC MARK SIDEWAYS NOON GHUNNA
 [0x0900] = {status="valid"}, -- DEVANAGARI SIGN INVERTED CANDRABINDU
 [0x0904] = {status="valid"}, -- DEVANAGARI LETTER SHORT A
 -- DEVANAGARI SIGN CANDRABINDU..DEVANAGARI SIGN VISARGA
 [0x0901] = {status="valid"},
 [0x0902] = {status="valid"},
 [0x0903] = {status="valid"},
 [0x094E] = {status="valid"}, -- DEVANAGARI VOWEL SIGN PRISHTHAMATRA E
 [0x094F] = {status="valid"}, -- DEVANAGARI VOWEL SIGN AW
 [0x0955] = {status="valid"}, -- DEVANAGARI VOWEL SIGN CANDRA LONG E
 -- DEVANAGARI OM..DEVANAGARI ACUTE ACCENT
 [0x0950] = {status="valid"},
 [0x0951] = {status="valid"},
 [0x0952] = {status="valid"},
 [0x0953] = {status="valid"},
 [0x0954] = {status="valid"},
 [0x0958] = {0x0915, 0x093C}, -- DEVANAGARI LETTER QA
 [0x0959] = {0x0916, 0x093C}, -- DEVANAGARI LETTER KHHA
 [0x095A] = {0x0917, 0x093C}, -- DEVANAGARI LETTER GHHA
 [0x095B] = {0x091C, 0x093C}, -- DEVANAGARI LETTER ZA
 [0x095C] = {0x0921, 0x093C}, -- DEVANAGARI LETTER DDDHA
 [0x095D] = {0x0922, 0x093C}, -- DEVANAGARI LETTER RHA
 [0x095E] = {0x092B, 0x093C}, -- DEVANAGARI LETTER FA
 [0x095F] = {0x092F, 0x093C}, -- DEVANAGARI LETTER YYA
 -- DEVANAGARI VOWEL SIGN UE..DEVANAGARI VOWEL SIGN UUE
 [0x0956] = {status="valid"},
 [0x0957] = {status="valid"},
 -- DEVANAGARI LETTER VOCALIC RR..DEVANAGARI VOWEL SIGN VOCALIC LL
 [0x0960] = {status="valid"},
 [0x0961] = {status="valid"},
 [0x0962] = {status="valid"},
 [0x0963] = {status="valid"},
 -- DEVANAGARI DANDA..DEVANAGARI DOUBLE DANDA
 [0x0964] = {status="disallowed"},
 [0x0965] = {status="disallowed"},
 [0x0970] = {status="disallowed"}, -- DEVANAGARI ABBREVIATION SIGN
 [0x0978] = {status="valid"}, -- DEVANAGARI LETTER MARWARI DDA
 [0x097D] = {status="valid"}, -- DEVANAGARI LETTER GLOTTAL STOP
 -- DEVANAGARI LETTER ZHA..DEVANAGARI LETTER HEAVY YA
      -- DEVANAGARI LETTER GGA..DEVANAGARI LETTER JJA
 [0x0979] = {status="valid"},
 [0x097A] = {status="valid"},
 [0x097B] = {status="valid"},
 [0x097C] = {status="valid"},
 [0x0980] = {status="valid"}, -- BENGALI ANJI
 -- DEVANAGARI LETTER DDDA..DEVANAGARI LETTER BBA
 [0x097E] = {status="valid"},
 [0x097F] = {status="valid"},
 [0x0984] = {status="disallowed"}, -- NA   <reserved-0984>
 -- BENGALI SIGN CANDRABINDU..BENGALI SIGN VISARGA
 [0x0981] = {status="valid"},
 [0x0982] = {status="valid"},
 [0x0983] = {status="valid"},
 -- NA   <reserved-098D>..<reserved-098E>
 [0x098D] = {status="disallowed"},
 [0x098E] = {status="disallowed"},
 -- BENGALI LETTER E..BENGALI LETTER AI
 [0x098F] = {status="valid"},
 [0x0990] = {status="valid"},
 -- NA   <reserved-0991>..<reserved-0992>
 [0x0991] = {status="disallowed"},
 [0x0992] = {status="disallowed"},
 [0x09A9] = {status="disallowed"}, -- NA   <reserved-09A9>
 [0x09B1] = {status="disallowed"}, -- NA   <reserved-09B1>
 [0x09B2] = {status="valid"}, -- BENGALI LETTER LA
 -- NA   <reserved-09B3>..<reserved-09B5>
 [0x09B3] = {status="disallowed"},
 [0x09B4] = {status="disallowed"},
 [0x09B5] = {status="disallowed"},
 -- BENGALI LETTER SHA..BENGALI LETTER HA
 [0x09B6] = {status="valid"},
 [0x09B7] = {status="valid"},
 [0x09B8] = {status="valid"},
 [0x09B9] = {status="valid"},
 [0x09BC] = {status="valid"}, -- BENGALI SIGN NUKTA
 [0x09BD] = {status="valid"}, -- BENGALI SIGN AVAGRAHA
 -- NA   <reserved-09BA>..<reserved-09BB>
 [0x09BA] = {status="disallowed"},
 [0x09BB] = {status="disallowed"},
 -- NA   <reserved-09C5>..<reserved-09C6>
 [0x09C5] = {status="disallowed"},
 [0x09C6] = {status="disallowed"},
 -- BENGALI VOWEL SIGN E..BENGALI VOWEL SIGN AI
 [0x09C7] = {status="valid"},
 [0x09C8] = {status="valid"},
 -- NA   <reserved-09C9>..<reserved-09CA>
 [0x09C9] = {status="disallowed"},
 [0x09CA] = {status="disallowed"},
 [0x09CE] = {status="valid"}, -- BENGALI LETTER KHANDA TA
 -- BENGALI VOWEL SIGN O..BENGALI SIGN VIRAMA
 [0x09CB] = {status="valid"},
 [0x09CC] = {status="valid"},
 [0x09CD] = {status="valid"},
 [0x09D7] = {status="valid"}, -- BENGALI AU LENGTH MARK
 [0x09DC] = {0x09A1, 0x09BC}, -- BENGALI LETTER RRA
 [0x09DD] = {0x09A2, 0x09BC}, -- BENGALI LETTER RHA
 [0x09DE] = {status="disallowed"}, -- NA   <reserved-09DE>
 [0x09DF] = {0x09AF, 0x09BC}, -- BENGALI LETTER YYA
 -- NA   <reserved-09D8>..<reserved-09DB>
 [0x09D8] = {status="disallowed"},
 [0x09D9] = {status="disallowed"},
 [0x09DA] = {status="disallowed"},
 [0x09DB] = {status="disallowed"},
 -- BENGALI LETTER VOCALIC RR..BENGALI VOWEL SIGN VOCALIC LL
 [0x09E0] = {status="valid"},
 [0x09E1] = {status="valid"},
 [0x09E2] = {status="valid"},
 [0x09E3] = {status="valid"},
 -- NA   <reserved-09E4>..<reserved-09E5>
 [0x09E4] = {status="disallowed"},
 [0x09E5] = {status="disallowed"},
 [0x09FB] = {status="disallowed"}, -- BENGALI GANDA MARK
 [0x09FC] = {status="valid"}, -- BENGALI LETTER VEDIC ANUSVARA
 [0x09FD] = {status="disallowed"}, -- BENGALI ABBREVIATION SIGN
 [0x0A01] = {status="valid"}, -- GURMUKHI SIGN ADAK BINDI
 [0x0A02] = {status="valid"}, -- GURMUKHI SIGN BINDI
 [0x0A03] = {status="valid"}, -- GURMUKHI SIGN VISARGA
 [0x0A04] = {status="disallowed"}, -- NA   <reserved-0A04>
 -- NA   <reserved-09FE>..<reserved-0A00>
 [0x09FE] = {status="disallowed"},
 [0x09FF] = {status="disallowed"},
 [0x0A00] = {status="disallowed"},
 -- NA   <reserved-0A0B>..<reserved-0A0E>
 [0x0A0B] = {status="disallowed"},
 [0x0A0C] = {status="disallowed"},
 [0x0A0D] = {status="disallowed"},
 [0x0A0E] = {status="disallowed"},
 -- GURMUKHI LETTER EE..GURMUKHI LETTER AI
 [0x0A0F] = {status="valid"},
 [0x0A10] = {status="valid"},
 -- NA   <reserved-0A11>..<reserved-0A12>
 [0x0A11] = {status="disallowed"},
 [0x0A12] = {status="disallowed"},
 [0x0A29] = {status="disallowed"}, -- NA   <reserved-0A29>
 [0x0A31] = {status="disallowed"}, -- NA   <reserved-0A31>
 [0x0A32] = {status="valid"}, -- GURMUKHI LETTER LA
 [0x0A33] = {0x0A32, 0x0A3C}, -- GURMUKHI LETTER LLA
 [0x0A34] = {status="disallowed"}, -- NA   <reserved-0A34>
 [0x0A35] = {status="valid"}, -- GURMUKHI LETTER VA
 [0x0A36] = {0x0A38, 0x0A3C}, -- GURMUKHI LETTER SHA
 [0x0A37] = {status="disallowed"}, -- NA   <reserved-0A37>
 -- GURMUKHI LETTER SA..GURMUKHI LETTER HA
 [0x0A38] = {status="valid"},
 [0x0A39] = {status="valid"},
 [0x0A3C] = {status="valid"}, -- GURMUKHI SIGN NUKTA
 [0x0A3D] = {status="disallowed"}, -- NA   <reserved-0A3D>
 -- NA   <reserved-0A3A>..<reserved-0A3B>
 [0x0A3A] = {status="disallowed"},
 [0x0A3B] = {status="disallowed"},
 -- GURMUKHI VOWEL SIGN AA..GURMUKHI VOWEL SIGN UU
 [0x0A3E] = {status="valid"},
 [0x0A3F] = {status="valid"},
 [0x0A40] = {status="valid"},
 [0x0A41] = {status="valid"},
 [0x0A42] = {status="valid"},
 -- NA   <reserved-0A43>..<reserved-0A46>
 [0x0A43] = {status="disallowed"},
 [0x0A44] = {status="disallowed"},
 [0x0A45] = {status="disallowed"},
 [0x0A46] = {status="disallowed"},
 -- GURMUKHI VOWEL SIGN EE..GURMUKHI VOWEL SIGN AI
 [0x0A47] = {status="valid"},
 [0x0A48] = {status="valid"},
 -- NA   <reserved-0A49>..<reserved-0A4A>
 [0x0A49] = {status="disallowed"},
 [0x0A4A] = {status="disallowed"},
 -- GURMUKHI VOWEL SIGN OO..GURMUKHI SIGN VIRAMA
 [0x0A4B] = {status="valid"},
 [0x0A4C] = {status="valid"},
 [0x0A4D] = {status="valid"},
 [0x0A51] = {status="valid"}, -- GURMUKHI SIGN UDAAT
 -- NA   <reserved-0A4E>..<reserved-0A50>
 [0x0A4E] = {status="disallowed"},
 [0x0A4F] = {status="disallowed"},
 [0x0A50] = {status="disallowed"},
 [0x0A59] = {0x0A16, 0x0A3C}, -- GURMUKHI LETTER KHHA
 [0x0A5A] = {0x0A17, 0x0A3C}, -- GURMUKHI LETTER GHHA
 [0x0A5B] = {0x0A1C, 0x0A3C}, -- GURMUKHI LETTER ZA
 [0x0A5C] = {status="valid"}, -- GURMUKHI LETTER RRA
 [0x0A5D] = {status="disallowed"}, -- NA   <reserved-0A5D>
 [0x0A5E] = {0x0A2B, 0x0A3C}, -- GURMUKHI LETTER FA
 [0x0A75] = {status="valid"}, -- GURMUKHI SIGN YAKASH
 [0x0A84] = {status="disallowed"}, -- NA   <reserved-0A84>
 -- GUJARATI SIGN CANDRABINDU..GUJARATI SIGN VISARGA
 [0x0A81] = {status="valid"},
 [0x0A82] = {status="valid"},
 [0x0A83] = {status="valid"},
 [0x0A8C] = {status="valid"}, -- GUJARATI LETTER VOCALIC L
 [0x0A8D] = {status="valid"}, -- GUJARATI VOWEL CANDRA E
 [0x0A8E] = {status="disallowed"}, -- NA   <reserved-0A8E>
 [0x0A92] = {status="disallowed"}, -- NA   <reserved-0A92>
 -- GUJARATI LETTER E..GUJARATI VOWEL CANDRA O
 [0x0A8F] = {status="valid"},
 [0x0A90] = {status="valid"},
 [0x0A91] = {status="valid"},
 [0x0AA9] = {status="disallowed"}, -- NA   <reserved-0AA9>
 [0x0AB1] = {status="disallowed"}, -- NA   <reserved-0AB1>
 [0x0AB4] = {status="disallowed"}, -- NA   <reserved-0AB4>
 -- GUJARATI LETTER LA..GUJARATI LETTER LLA
 [0x0AB2] = {status="valid"},
 [0x0AB3] = {status="valid"},
 -- GUJARATI LETTER VA..GUJARATI LETTER HA
 [0x0AB5] = {status="valid"},
 [0x0AB6] = {status="valid"},
 [0x0AB7] = {status="valid"},
 [0x0AB8] = {status="valid"},
 [0x0AB9] = {status="valid"},
 -- NA   <reserved-0ABA>..<reserved-0ABB>
 [0x0ABA] = {status="disallowed"},
 [0x0ABB] = {status="disallowed"},
 [0x0AC6] = {status="disallowed"}, -- NA   <reserved-0AC6>
 [0x0ACA] = {status="disallowed"}, -- NA   <reserved-0ACA>
 -- GUJARATI VOWEL SIGN E..GUJARATI VOWEL SIGN CANDRA O
 [0x0AC7] = {status="valid"},
 [0x0AC8] = {status="valid"},
 [0x0AC9] = {status="valid"},
 -- GUJARATI VOWEL SIGN O..GUJARATI SIGN VIRAMA
 [0x0ACB] = {status="valid"},
 [0x0ACC] = {status="valid"},
 [0x0ACD] = {status="valid"},
 [0x0AD0] = {status="valid"}, -- GUJARATI OM
 -- NA   <reserved-0ACE>..<reserved-0ACF>
 [0x0ACE] = {status="disallowed"},
 [0x0ACF] = {status="disallowed"},
 [0x0AE0] = {status="valid"}, -- GUJARATI LETTER VOCALIC RR
 -- GUJARATI LETTER VOCALIC LL..GUJARATI VOWEL SIGN VOCALIC LL
 [0x0AE1] = {status="valid"},
 [0x0AE2] = {status="valid"},
 [0x0AE3] = {status="valid"},
 -- NA   <reserved-0AE4>..<reserved-0AE5>
 [0x0AE4] = {status="disallowed"},
 [0x0AE5] = {status="disallowed"},
 [0x0AF0] = {status="disallowed"}, -- GUJARATI ABBREVIATION SIGN
 [0x0AF1] = {status="disallowed"}, -- GUJARATI RUPEE SIGN
 [0x0AF9] = {status="valid"}, -- GUJARATI LETTER ZHA
 [0x0B00] = {status="disallowed"}, -- NA   <reserved-0B00>
 [0x0B04] = {status="disallowed"}, -- NA   <reserved-0B04>
 -- ORIYA SIGN CANDRABINDU..ORIYA SIGN VISARGA
 [0x0B01] = {status="valid"},
 [0x0B02] = {status="valid"},
 [0x0B03] = {status="valid"},
 -- NA   <reserved-0B0D>..<reserved-0B0E>
 [0x0B0D] = {status="disallowed"},
 [0x0B0E] = {status="disallowed"},
 -- ORIYA LETTER E..ORIYA LETTER AI
 [0x0B0F] = {status="valid"},
 [0x0B10] = {status="valid"},
 -- NA   <reserved-0B11>..<reserved-0B12>
 [0x0B11] = {status="disallowed"},
 [0x0B12] = {status="disallowed"},
 [0x0B29] = {status="disallowed"}, -- NA   <reserved-0B29>
 [0x0B31] = {status="disallowed"}, -- NA   <reserved-0B31>
 [0x0B34] = {status="disallowed"}, -- NA   <reserved-0B34>
 [0x0B35] = {status="valid"}, -- ORIYA LETTER VA
 -- ORIYA LETTER LA..ORIYA LETTER LLA
 [0x0B32] = {status="valid"},
 [0x0B33] = {status="valid"},
 -- ORIYA LETTER SHA..ORIYA LETTER HA
 [0x0B36] = {status="valid"},
 [0x0B37] = {status="valid"},
 [0x0B38] = {status="valid"},
 [0x0B39] = {status="valid"},
 -- NA   <reserved-0B3A>..<reserved-0B3B>
 [0x0B3A] = {status="disallowed"},
 [0x0B3B] = {status="disallowed"},
 [0x0B44] = {status="valid"}, -- ORIYA VOWEL SIGN VOCALIC RR
 -- NA   <reserved-0B45>..<reserved-0B46>
 [0x0B45] = {status="disallowed"},
 [0x0B46] = {status="disallowed"},
 -- ORIYA VOWEL SIGN E..ORIYA VOWEL SIGN AI
 [0x0B47] = {status="valid"},
 [0x0B48] = {status="valid"},
 -- NA   <reserved-0B49>..<reserved-0B4A>
 [0x0B49] = {status="disallowed"},
 [0x0B4A] = {status="disallowed"},
 -- ORIYA VOWEL SIGN O..ORIYA SIGN VIRAMA
 [0x0B4B] = {status="valid"},
 [0x0B4C] = {status="valid"},
 [0x0B4D] = {status="valid"},
 -- ORIYA AI LENGTH MARK..ORIYA AU LENGTH MARK
 [0x0B56] = {status="valid"},
 [0x0B57] = {status="valid"},
 [0x0B5C] = {0x0B21, 0x0B3C}, -- ORIYA LETTER RRA
 [0x0B5D] = {0x0B22, 0x0B3C}, -- ORIYA LETTER RHA
 [0x0B5E] = {status="disallowed"}, -- NA   <reserved-0B5E>
 -- NA   <reserved-0B58>..<reserved-0B5B>
 [0x0B58] = {status="disallowed"},
 [0x0B59] = {status="disallowed"},
 [0x0B5A] = {status="disallowed"},
 [0x0B5B] = {status="disallowed"},
 -- ORIYA LETTER YYA..ORIYA LETTER VOCALIC LL
      -- ORIYA VOWEL SIGN VOCALIC L..ORIYA VOWEL SIGN VOCALIC LL
 [0x0B5F] = {status="valid"},
 [0x0B60] = {status="valid"},
 [0x0B61] = {status="valid"},
 [0x0B62] = {status="valid"},
 [0x0B63] = {status="valid"},
 -- NA   <reserved-0B64>..<reserved-0B65>
 [0x0B64] = {status="disallowed"},
 [0x0B65] = {status="disallowed"},
 [0x0B70] = {status="disallowed"}, -- ORIYA ISSHAR
 [0x0B71] = {status="valid"}, -- ORIYA LETTER WA
 [0x0B84] = {status="disallowed"}, -- NA   <reserved-0B84>
 -- TAMIL SIGN ANUSVARA..TAMIL SIGN VISARGA
 [0x0B82] = {status="valid"},
 [0x0B83] = {status="valid"},
 -- NA   <reserved-0B8B>..<reserved-0B8D>
 [0x0B8B] = {status="disallowed"},
 [0x0B8C] = {status="disallowed"},
 [0x0B8D] = {status="disallowed"},
 [0x0B91] = {status="disallowed"}, -- NA   <reserved-0B91>
 -- TAMIL LETTER E..TAMIL LETTER AI
 [0x0B8E] = {status="valid"},
 [0x0B8F] = {status="valid"},
 [0x0B90] = {status="valid"},
 -- TAMIL LETTER O..TAMIL LETTER KA
 [0x0B92] = {status="valid"},
 [0x0B93] = {status="valid"},
 [0x0B94] = {status="valid"},
 [0x0B95] = {status="valid"},
 -- NA   <reserved-0B96>..<reserved-0B98>
 [0x0B96] = {status="disallowed"},
 [0x0B97] = {status="disallowed"},
 [0x0B98] = {status="disallowed"},
 [0x0B9B] = {status="disallowed"}, -- NA   <reserved-0B9B>
 [0x0B9C] = {status="valid"}, -- TAMIL LETTER JA
 [0x0B9D] = {status="disallowed"}, -- NA   <reserved-0B9D>
 -- TAMIL LETTER NGA..TAMIL LETTER CA
 [0x0B99] = {status="valid"},
 [0x0B9A] = {status="valid"},
 -- TAMIL LETTER NYA..TAMIL LETTER TTA
 [0x0B9E] = {status="valid"},
 [0x0B9F] = {status="valid"},
 -- NA   <reserved-0BA0>..<reserved-0BA2>
 [0x0BA0] = {status="disallowed"},
 [0x0BA1] = {status="disallowed"},
 [0x0BA2] = {status="disallowed"},
 -- TAMIL LETTER NNA..TAMIL LETTER TA
 [0x0BA3] = {status="valid"},
 [0x0BA4] = {status="valid"},
 -- NA   <reserved-0BA5>..<reserved-0BA7>
 [0x0BA5] = {status="disallowed"},
 [0x0BA6] = {status="disallowed"},
 [0x0BA7] = {status="disallowed"},
 -- TAMIL LETTER NA..TAMIL LETTER PA
 [0x0BA8] = {status="valid"},
 [0x0BA9] = {status="valid"},
 [0x0BAA] = {status="valid"},
 -- NA   <reserved-0BAB>..<reserved-0BAD>
 [0x0BAB] = {status="disallowed"},
 [0x0BAC] = {status="disallowed"},
 [0x0BAD] = {status="disallowed"},
 [0x0BB6] = {status="valid"}, -- TAMIL LETTER SHA
 -- TAMIL LETTER SSA..TAMIL LETTER HA
 [0x0BB7] = {status="valid"},
 [0x0BB8] = {status="valid"},
 [0x0BB9] = {status="valid"},
 -- NA   <reserved-0BBA>..<reserved-0BBD>
 [0x0BBA] = {status="disallowed"},
 [0x0BBB] = {status="disallowed"},
 [0x0BBC] = {status="disallowed"},
 [0x0BBD] = {status="disallowed"},
 -- TAMIL VOWEL SIGN AA..TAMIL VOWEL SIGN UU
 [0x0BBE] = {status="valid"},
 [0x0BBF] = {status="valid"},
 [0x0BC0] = {status="valid"},
 [0x0BC1] = {status="valid"},
 [0x0BC2] = {status="valid"},
 -- NA   <reserved-0BC3>..<reserved-0BC5>
 [0x0BC3] = {status="disallowed"},
 [0x0BC4] = {status="disallowed"},
 [0x0BC5] = {status="disallowed"},
 [0x0BC9] = {status="disallowed"}, -- NA   <reserved-0BC9>
 -- TAMIL VOWEL SIGN E..TAMIL VOWEL SIGN AI
 [0x0BC6] = {status="valid"},
 [0x0BC7] = {status="valid"},
 [0x0BC8] = {status="valid"},
 -- TAMIL VOWEL SIGN O..TAMIL SIGN VIRAMA
 [0x0BCA] = {status="valid"},
 [0x0BCB] = {status="valid"},
 [0x0BCC] = {status="valid"},
 [0x0BCD] = {status="valid"},
 [0x0BD0] = {status="valid"}, -- TAMIL OM
 -- NA   <reserved-0BCE>..<reserved-0BCF>
 [0x0BCE] = {status="disallowed"},
 [0x0BCF] = {status="disallowed"},
 [0x0BD7] = {status="valid"}, -- TAMIL AU LENGTH MARK
 [0x0BE6] = {status="valid"}, -- TAMIL DIGIT ZERO
 [0x0C00] = {status="valid"}, -- TELUGU SIGN COMBINING CANDRABINDU ABOVE
 -- NA   <reserved-0BFB>..<reserved-0BFF>
 [0x0BFB] = {status="disallowed"},
 [0x0BFC] = {status="disallowed"},
 [0x0BFD] = {status="disallowed"},
 [0x0BFE] = {status="disallowed"},
 [0x0BFF] = {status="disallowed"},
 [0x0C04] = {status="disallowed"}, -- NA   <reserved-0C04>
 -- TELUGU SIGN CANDRABINDU..TELUGU SIGN VISARGA
 [0x0C01] = {status="valid"},
 [0x0C02] = {status="valid"},
 [0x0C03] = {status="valid"},
 [0x0C0D] = {status="disallowed"}, -- NA   <reserved-0C0D>
 [0x0C11] = {status="disallowed"}, -- NA   <reserved-0C11>
 -- TELUGU LETTER E..TELUGU LETTER AI
 [0x0C0E] = {status="valid"},
 [0x0C0F] = {status="valid"},
 [0x0C10] = {status="valid"},
 [0x0C29] = {status="disallowed"}, -- NA   <reserved-0C29>
 [0x0C34] = {status="valid"}, -- TELUGU LETTER LLLA
 -- TELUGU LETTER VA..TELUGU LETTER HA
 [0x0C35] = {status="valid"},
 [0x0C36] = {status="valid"},
 [0x0C37] = {status="valid"},
 [0x0C38] = {status="valid"},
 [0x0C39] = {status="valid"},
 [0x0C3D] = {status="valid"}, -- TELUGU SIGN AVAGRAHA
 -- NA   <reserved-0C3A>..<reserved-0C3C>
 [0x0C3A] = {status="disallowed"},
 [0x0C3B] = {status="disallowed"},
 [0x0C3C] = {status="disallowed"},
 [0x0C45] = {status="disallowed"}, -- NA   <reserved-0C45>
 [0x0C49] = {status="disallowed"}, -- NA   <reserved-0C49>
 -- TELUGU VOWEL SIGN E..TELUGU VOWEL SIGN AI
 [0x0C46] = {status="valid"},
 [0x0C47] = {status="valid"},
 [0x0C48] = {status="valid"},
 -- TELUGU VOWEL SIGN O..TELUGU SIGN VIRAMA
 [0x0C4A] = {status="valid"},
 [0x0C4B] = {status="valid"},
 [0x0C4C] = {status="valid"},
 [0x0C4D] = {status="valid"},
 [0x0C57] = {status="disallowed"}, -- NA   <reserved-0C57>
 -- TELUGU LENGTH MARK..TELUGU AI LENGTH MARK
 [0x0C55] = {status="valid"},
 [0x0C56] = {status="valid"},
 [0x0C5A] = {status="valid"}, -- TELUGU LETTER RRRA
 -- TELUGU LETTER TSA..TELUGU LETTER DZA
 [0x0C58] = {status="valid"},
 [0x0C59] = {status="valid"},
 -- NA   <reserved-0C5B>..<reserved-0C5F>
 [0x0C5B] = {status="disallowed"},
 [0x0C5C] = {status="disallowed"},
 [0x0C5D] = {status="disallowed"},
 [0x0C5E] = {status="disallowed"},
 [0x0C5F] = {status="disallowed"},
 -- TELUGU LETTER VOCALIC RR..TELUGU LETTER VOCALIC LL
      -- TELUGU VOWEL SIGN VOCALIC L..TELUGU VOWEL SIGN VOCALIC LL
 [0x0C60] = {status="valid"},
 [0x0C61] = {status="valid"},
 [0x0C62] = {status="valid"},
 [0x0C63] = {status="valid"},
 -- NA   <reserved-0C64>..<reserved-0C65>
 [0x0C64] = {status="disallowed"},
 [0x0C65] = {status="disallowed"},
 [0x0C80] = {status="valid"}, -- KANNADA SIGN SPACING CANDRABINDU
 [0x0C81] = {status="valid"}, -- KANNADA SIGN CANDRABINDU
 [0x0C84] = {status="disallowed"}, -- NA   <reserved-0C84>
 -- KANNADA SIGN ANUSVARA..KANNADA SIGN VISARGA
 [0x0C82] = {status="valid"},
 [0x0C83] = {status="valid"},
 [0x0C8D] = {status="disallowed"}, -- NA   <reserved-0C8D>
 [0x0C91] = {status="disallowed"}, -- NA   <reserved-0C91>
 -- KANNADA LETTER E..KANNADA LETTER AI
 [0x0C8E] = {status="valid"},
 [0x0C8F] = {status="valid"},
 [0x0C90] = {status="valid"},
 [0x0CA9] = {status="disallowed"}, -- NA   <reserved-0CA9>
 [0x0CB4] = {status="disallowed"}, -- NA   <reserved-0CB4>
 -- KANNADA LETTER VA..KANNADA LETTER HA
 [0x0CB5] = {status="valid"},
 [0x0CB6] = {status="valid"},
 [0x0CB7] = {status="valid"},
 [0x0CB8] = {status="valid"},
 [0x0CB9] = {status="valid"},
 -- NA   <reserved-0CBA>..<reserved-0CBB>
 [0x0CBA] = {status="disallowed"},
 [0x0CBB] = {status="disallowed"},
 [0x0CC5] = {status="disallowed"}, -- NA   <reserved-0CC5>
 [0x0CC9] = {status="disallowed"}, -- NA   <reserved-0CC9>
 -- KANNADA VOWEL SIGN E..KANNADA VOWEL SIGN AI
 [0x0CC6] = {status="valid"},
 [0x0CC7] = {status="valid"},
 [0x0CC8] = {status="valid"},
 -- KANNADA VOWEL SIGN O..KANNADA SIGN VIRAMA
 [0x0CCA] = {status="valid"},
 [0x0CCB] = {status="valid"},
 [0x0CCC] = {status="valid"},
 [0x0CCD] = {status="valid"},
 -- KANNADA LENGTH MARK..KANNADA AI LENGTH MARK
 [0x0CD5] = {status="valid"},
 [0x0CD6] = {status="valid"},
 [0x0CDE] = {status="valid"}, -- KANNADA LETTER FA
 [0x0CDF] = {status="disallowed"}, -- NA   <reserved-0CDF>
 -- KANNADA LETTER VOCALIC RR..KANNADA LETTER VOCALIC LL
      -- KANNADA VOWEL SIGN VOCALIC L..KANNADA VOWEL SIGN VOCALIC LL
 [0x0CE0] = {status="valid"},
 [0x0CE1] = {status="valid"},
 [0x0CE2] = {status="valid"},
 [0x0CE3] = {status="valid"},
 -- NA   <reserved-0CE4>..<reserved-0CE5>
 [0x0CE4] = {status="disallowed"},
 [0x0CE5] = {status="disallowed"},
 [0x0CF0] = {status="disallowed"}, -- NA   <reserved-0CF0>
 -- KANNADA SIGN JIHVAMULIYA..KANNADA SIGN UPADHMANIYA
 [0x0CF1] = {status="valid"},
 [0x0CF2] = {status="valid"},
 [0x0D00] = {status="valid"}, -- MALAYALAM SIGN COMBINING ANUSVARA ABOVE
 [0x0D01] = {status="valid"}, -- MALAYALAM SIGN CANDRABINDU
 [0x0D04] = {status="disallowed"}, -- NA   <reserved-0D04>
 -- MALAYALAM SIGN ANUSVARA..MALAYALAM SIGN VISARGA
 [0x0D02] = {status="valid"},
 [0x0D03] = {status="valid"},
 [0x0D0D] = {status="disallowed"}, -- NA   <reserved-0D0D>
 [0x0D11] = {status="disallowed"}, -- NA   <reserved-0D11>
 -- MALAYALAM LETTER E..MALAYALAM LETTER AI
 [0x0D0E] = {status="valid"},
 [0x0D0F] = {status="valid"},
 [0x0D10] = {status="valid"},
 [0x0D29] = {status="valid"}, -- MALAYALAM LETTER NNNA
 [0x0D3A] = {status="valid"}, -- MALAYALAM LETTER TTTA
 [0x0D3D] = {status="valid"}, -- MALAYALAM SIGN AVAGRAHA
 -- MALAYALAM SIGN VERTICAL BAR VIRAMA..MALAYALAM SIGN CIRCULAR VIRAMA
 [0x0D3B] = {status="valid"},
 [0x0D3C] = {status="valid"},
 [0x0D44] = {status="valid"}, -- MALAYALAM VOWEL SIGN VOCALIC RR
 [0x0D45] = {status="disallowed"}, -- NA   <reserved-0D45>
 [0x0D49] = {status="disallowed"}, -- NA   <reserved-0D49>
 -- MALAYALAM VOWEL SIGN E..MALAYALAM VOWEL SIGN AI
 [0x0D46] = {status="valid"},
 [0x0D47] = {status="valid"},
 [0x0D48] = {status="valid"},
 [0x0D4E] = {status="valid"}, -- MALAYALAM LETTER DOT REPH
 [0x0D4F] = {status="disallowed"}, -- MALAYALAM SIGN PARA
 -- MALAYALAM VOWEL SIGN O..MALAYALAM SIGN VIRAMA
 [0x0D4A] = {status="valid"},
 [0x0D4B] = {status="valid"},
 [0x0D4C] = {status="valid"},
 [0x0D4D] = {status="valid"},
 -- NA   <reserved-0D50>..<reserved-0D53>
 [0x0D50] = {status="disallowed"},
 [0x0D51] = {status="disallowed"},
 [0x0D52] = {status="disallowed"},
 [0x0D53] = {status="disallowed"},
 [0x0D57] = {status="valid"}, -- MALAYALAM AU LENGTH MARK
 -- MALAYALAM LETTER CHILLU M..MALAYALAM LETTER CHILLU LLL
 [0x0D54] = {status="valid"},
 [0x0D55] = {status="valid"},
 [0x0D56] = {status="valid"},
 [0x0D5F] = {status="valid"}, -- MALAYALAM LETTER ARCHAIC II
 -- MALAYALAM LETTER VOCALIC RR..MALAYALAM LETTER VOCALIC LL
      -- MALAYALAM VOWEL SIGN VOCALIC L..MALAYALAM VOWEL SIGN VOCALIC LL
 [0x0D60] = {status="valid"},
 [0x0D61] = {status="valid"},
 [0x0D62] = {status="valid"},
 [0x0D63] = {status="valid"},
 -- NA   <reserved-0D64>..<reserved-0D65>
 [0x0D64] = {status="disallowed"},
 [0x0D65] = {status="disallowed"},
 [0x0D79] = {status="disallowed"}, -- MALAYALAM DATE MARK
 -- NA   <reserved-0D80>..<reserved-0D81>
 [0x0D80] = {status="disallowed"},
 [0x0D81] = {status="disallowed"},
 [0x0D84] = {status="disallowed"}, -- NA   <reserved-0D84>
 -- SINHALA SIGN ANUSVARAYA..SINHALA SIGN VISARGAYA
 [0x0D82] = {status="valid"},
 [0x0D83] = {status="valid"},
 -- NA   <reserved-0D97>..<reserved-0D99>
 [0x0D97] = {status="disallowed"},
 [0x0D98] = {status="disallowed"},
 [0x0D99] = {status="disallowed"},
 [0x0DB2] = {status="disallowed"}, -- NA   <reserved-0DB2>
 [0x0DBC] = {status="disallowed"}, -- NA   <reserved-0DBC>
 [0x0DBD] = {status="valid"}, -- SINHALA LETTER DANTAJA LAYANNA
 -- NA   <reserved-0DBE>..<reserved-0DBF>
 [0x0DBE] = {status="disallowed"},
 [0x0DBF] = {status="disallowed"},
 [0x0DCA] = {status="valid"}, -- SINHALA SIGN AL-LAKUNA
 -- NA   <reserved-0DC7>..<reserved-0DC9>
 [0x0DC7] = {status="disallowed"},
 [0x0DC8] = {status="disallowed"},
 [0x0DC9] = {status="disallowed"},
 -- NA   <reserved-0DCB>..<reserved-0DCE>
 [0x0DCB] = {status="disallowed"},
 [0x0DCC] = {status="disallowed"},
 [0x0DCD] = {status="disallowed"},
 [0x0DCE] = {status="disallowed"},
 [0x0DD5] = {status="disallowed"}, -- NA   <reserved-0DD5>
 [0x0DD6] = {status="valid"}, -- SINHALA VOWEL SIGN DIGA PAA-PILLA
 [0x0DD7] = {status="disallowed"}, -- NA   <reserved-0DD7>
 -- NA   <reserved-0DF0>..<reserved-0DF1>
 [0x0DF0] = {status="disallowed"},
 [0x0DF1] = {status="disallowed"},
 [0x0DF4] = {status="disallowed"}, -- SINHALA PUNCTUATION KUNDDALIYA
 -- SINHALA VOWEL SIGN DIGA GAETTA-PILLA..SINHALA VOWEL SIGN DIGA GAYANUKITTA
 [0x0DF2] = {status="valid"},
 [0x0DF3] = {status="valid"},
 [0x0E33] = {0x0E4D, 0x0E32}, -- THAI CHARACTER SARA AM
 [0x0E3F] = {status="disallowed"}, -- THAI CURRENCY SYMBOL BAHT
 -- NA   <reserved-0E3B>..<reserved-0E3E>
 [0x0E3B] = {status="disallowed"},
 [0x0E3C] = {status="disallowed"},
 [0x0E3D] = {status="disallowed"},
 [0x0E3E] = {status="disallowed"},
 [0x0E4F] = {status="disallowed"}, -- THAI CHARACTER FONGMAN
 -- THAI CHARACTER ANGKHANKHU..THAI CHARACTER KHOMUT
 [0x0E5A] = {status="disallowed"},
 [0x0E5B] = {status="disallowed"},
 [0x0E83] = {status="disallowed"}, -- NA   <reserved-0E83>
 [0x0E84] = {status="valid"}, -- LAO LETTER KHO TAM
 -- LAO LETTER KO..LAO LETTER KHO SUNG
 [0x0E81] = {status="valid"},
 [0x0E82] = {status="valid"},
 -- NA   <reserved-0E85>..<reserved-0E86>
 [0x0E85] = {status="disallowed"},
 [0x0E86] = {status="disallowed"},
 [0x0E89] = {status="disallowed"}, -- NA   <reserved-0E89>
 [0x0E8A] = {status="valid"}, -- LAO LETTER SO TAM
 -- LAO LETTER NGO..LAO LETTER CO
 [0x0E87] = {status="valid"},
 [0x0E88] = {status="valid"},
 [0x0E8D] = {status="valid"}, -- LAO LETTER NYO
 -- NA   <reserved-0E8B>..<reserved-0E8C>
 [0x0E8B] = {status="disallowed"},
 [0x0E8C] = {status="disallowed"},
 [0x0E98] = {status="disallowed"}, -- NA   <reserved-0E98>
 -- LAO LETTER DO..LAO LETTER THO TAM
 [0x0E94] = {status="valid"},
 [0x0E95] = {status="valid"},
 [0x0E96] = {status="valid"},
 [0x0E97] = {status="valid"},
 [0x0EA0] = {status="disallowed"}, -- NA   <reserved-0EA0>
 [0x0EA4] = {status="disallowed"}, -- NA   <reserved-0EA4>
 [0x0EA5] = {status="valid"}, -- LAO LETTER LO LOOT
 [0x0EA6] = {status="disallowed"}, -- NA   <reserved-0EA6>
 [0x0EA7] = {status="valid"}, -- LAO LETTER WO
 -- LAO LETTER MO..LAO LETTER LO LING
 [0x0EA1] = {status="valid"},
 [0x0EA2] = {status="valid"},
 [0x0EA3] = {status="valid"},
 -- NA   <reserved-0EA8>..<reserved-0EA9>
 [0x0EA8] = {status="disallowed"},
 [0x0EA9] = {status="disallowed"},
 [0x0EAC] = {status="disallowed"}, -- NA   <reserved-0EAC>
 -- LAO LETTER SO SUNG..LAO LETTER HO SUNG
 [0x0EAA] = {status="valid"},
 [0x0EAB] = {status="valid"},
 [0x0EB3] = {0x0ECD, 0x0EB2}, -- LAO VOWEL SIGN AM
 [0x0EBA] = {status="disallowed"}, -- NA   <reserved-0EBA>
 -- LAO VOWEL SIGN MAI KON..LAO SEMIVOWEL SIGN NYO
 [0x0EBB] = {status="valid"},
 [0x0EBC] = {status="valid"},
 [0x0EBD] = {status="valid"},
 -- NA   <reserved-0EBE>..<reserved-0EBF>
 [0x0EBE] = {status="disallowed"},
 [0x0EBF] = {status="disallowed"},
 [0x0EC5] = {status="disallowed"}, -- NA   <reserved-0EC5>
 [0x0EC6] = {status="valid"}, -- LAO KO LA
 [0x0EC7] = {status="disallowed"}, -- NA   <reserved-0EC7>
 -- LAO VOWEL SIGN E..LAO VOWEL SIGN AI
 [0x0EC0] = {status="valid"},
 [0x0EC1] = {status="valid"},
 [0x0EC2] = {status="valid"},
 [0x0EC3] = {status="valid"},
 [0x0EC4] = {status="valid"},
 -- NA   <reserved-0ECE>..<reserved-0ECF>
 [0x0ECE] = {status="disallowed"},
 [0x0ECF] = {status="disallowed"},
 [0x0EDC] = {0x0EAB, 0x0E99}, -- LAO HO NO
 [0x0EDD] = {0x0EAB, 0x0EA1}, -- LAO HO MO
 -- NA   <reserved-0EDA>..<reserved-0EDB>
 [0x0EDA] = {status="disallowed"},
 [0x0EDB] = {status="disallowed"},
 -- LAO LETTER KHMU GO..LAO LETTER KHMU NYO
 [0x0EDE] = {status="valid"},
 [0x0EDF] = {status="valid"},
 [0x0F00] = {status="valid"}, -- TIBETAN SYLLABLE OM
 [0x0F0B] = {status="valid"}, -- TIBETAN MARK INTERSYLLABIC TSHEG
 [0x0F0C] = 0x0F0B, -- TIBETAN MARK DELIMITER TSHEG BSTAR
 -- TIBETAN ASTROLOGICAL SIGN -KHYUD PA..TIBETAN ASTROLOGICAL SIGN SDONG TSHUGS
 [0x0F18] = {status="valid"},
 [0x0F19] = {status="valid"},
 [0x0F35] = {status="valid"}, -- TIBETAN MARK NGAS BZUNG NYI ZLA
 [0x0F36] = {status="disallowed"}, -- TIBETAN MARK CARET -DZUD RTAGS BZHI MIG CAN
 [0x0F37] = {status="valid"}, -- TIBETAN MARK NGAS BZUNG SGOR RTAGS
 [0x0F38] = {status="disallowed"}, -- TIBETAN MARK CHE MGO
 [0x0F39] = {status="valid"}, -- TIBETAN MARK TSA -PHRU
 -- TIBETAN MARK GUG RTAGS GYON..TIBETAN MARK ANG KHANG GYAS
 [0x0F3A] = {status="disallowed"},
 [0x0F3B] = {status="disallowed"},
 [0x0F3C] = {status="disallowed"},
 [0x0F3D] = {status="disallowed"},
 [0x0F43] = {0x0F42, 0x0FB7}, -- TIBETAN LETTER GHA
 -- TIBETAN SIGN YAR TSHES..TIBETAN LETTER GA
 [0x0F3E] = {status="valid"},
 [0x0F3F] = {status="valid"},
 [0x0F40] = {status="valid"},
 [0x0F41] = {status="valid"},
 [0x0F42] = {status="valid"},
 [0x0F48] = {status="disallowed"}, -- NA   <reserved-0F48>
 -- TIBETAN LETTER NGA..TIBETAN LETTER JA
 [0x0F44] = {status="valid"},
 [0x0F45] = {status="valid"},
 [0x0F46] = {status="valid"},
 [0x0F47] = {status="valid"},
 [0x0F4D] = {0x0F4C, 0x0FB7}, -- TIBETAN LETTER DDHA
 -- TIBETAN LETTER NYA..TIBETAN LETTER DDA
 [0x0F49] = {status="valid"},
 [0x0F4A] = {status="valid"},
 [0x0F4B] = {status="valid"},
 [0x0F4C] = {status="valid"},
 [0x0F52] = {0x0F51, 0x0FB7}, -- TIBETAN LETTER DHA
 -- TIBETAN LETTER NNA..TIBETAN LETTER DA
 [0x0F4E] = {status="valid"},
 [0x0F4F] = {status="valid"},
 [0x0F50] = {status="valid"},
 [0x0F51] = {status="valid"},
 [0x0F57] = {0x0F56, 0x0FB7}, -- TIBETAN LETTER BHA
 -- TIBETAN LETTER NA..TIBETAN LETTER BA
 [0x0F53] = {status="valid"},
 [0x0F54] = {status="valid"},
 [0x0F55] = {status="valid"},
 [0x0F56] = {status="valid"},
 [0x0F5C] = {0x0F5B, 0x0FB7}, -- TIBETAN LETTER DZHA
 -- TIBETAN LETTER MA..TIBETAN LETTER DZA
 [0x0F58] = {status="valid"},
 [0x0F59] = {status="valid"},
 [0x0F5A] = {status="valid"},
 [0x0F5B] = {status="valid"},
 [0x0F69] = {0x0F40, 0x0FB5}, -- TIBETAN LETTER KSSA
 [0x0F6A] = {status="valid"}, -- TIBETAN LETTER FIXED-FORM RA
 -- TIBETAN LETTER KKA..TIBETAN LETTER RRA
 [0x0F6B] = {status="valid"},
 [0x0F6C] = {status="valid"},
 -- NA   <reserved-0F6D>..<reserved-0F70>
 [0x0F6D] = {status="disallowed"},
 [0x0F6E] = {status="disallowed"},
 [0x0F6F] = {status="disallowed"},
 [0x0F70] = {status="disallowed"},
 [0x0F73] = {0x0F71, 0x0F72}, -- TIBETAN VOWEL SIGN II
 [0x0F74] = {status="valid"}, -- TIBETAN VOWEL SIGN U
 [0x0F75] = {0x0F71, 0x0F74}, -- TIBETAN VOWEL SIGN UU
 [0x0F76] = {0x0FB2, 0x0F80}, -- TIBETAN VOWEL SIGN VOCALIC R
 [0x0F77] = {0x0FB2, 0x0F71, 0x0F80}, -- TIBETAN VOWEL SIGN VOCALIC RR
 [0x0F78] = {0x0FB3, 0x0F80}, -- TIBETAN VOWEL SIGN VOCALIC L
 [0x0F79] = {0x0FB3, 0x0F71, 0x0F80}, -- TIBETAN VOWEL SIGN VOCALIC LL
 -- TIBETAN VOWEL SIGN AA..TIBETAN VOWEL SIGN I
 [0x0F71] = {status="valid"},
 [0x0F72] = {status="valid"},
 [0x0F81] = {0x0F71, 0x0F80}, -- TIBETAN VOWEL SIGN REVERSED II
 [0x0F85] = {status="disallowed"}, -- TIBETAN MARK PALUTA
 -- TIBETAN SIGN NYI ZLA NAA DA..TIBETAN MARK HALANTA
 [0x0F82] = {status="valid"},
 [0x0F83] = {status="valid"},
 [0x0F84] = {status="valid"},
 [0x0F93] = {0x0F92, 0x0FB7}, -- TIBETAN SUBJOINED LETTER GHA
 [0x0F96] = {status="valid"}, -- TIBETAN SUBJOINED LETTER CHA
 [0x0F97] = {status="valid"}, -- TIBETAN SUBJOINED LETTER JA
 [0x0F98] = {status="disallowed"}, -- NA   <reserved-0F98>
 -- TIBETAN SUBJOINED LETTER NGA..TIBETAN SUBJOINED LETTER CA
 [0x0F94] = {status="valid"},
 [0x0F95] = {status="valid"},
 [0x0F9D] = {0x0F9C, 0x0FB7}, -- TIBETAN SUBJOINED LETTER DDHA
 -- TIBETAN SUBJOINED LETTER NYA..TIBETAN SUBJOINED LETTER DDA
 [0x0F99] = {status="valid"},
 [0x0F9A] = {status="valid"},
 [0x0F9B] = {status="valid"},
 [0x0F9C] = {status="valid"},
 [0x0FA2] = {0x0FA1, 0x0FB7}, -- TIBETAN SUBJOINED LETTER DHA
 -- TIBETAN SUBJOINED LETTER NNA..TIBETAN SUBJOINED LETTER DA
 [0x0F9E] = {status="valid"},
 [0x0F9F] = {status="valid"},
 [0x0FA0] = {status="valid"},
 [0x0FA1] = {status="valid"},
 [0x0FA7] = {0x0FA6, 0x0FB7}, -- TIBETAN SUBJOINED LETTER BHA
 -- TIBETAN SUBJOINED LETTER NA..TIBETAN SUBJOINED LETTER BA
 [0x0FA3] = {status="valid"},
 [0x0FA4] = {status="valid"},
 [0x0FA5] = {status="valid"},
 [0x0FA6] = {status="valid"},
 [0x0FAC] = {0x0FAB, 0x0FB7}, -- TIBETAN SUBJOINED LETTER DZHA
 [0x0FAD] = {status="valid"}, -- TIBETAN SUBJOINED LETTER WA
 -- TIBETAN SUBJOINED LETTER MA..TIBETAN SUBJOINED LETTER DZA
 [0x0FA8] = {status="valid"},
 [0x0FA9] = {status="valid"},
 [0x0FAA] = {status="valid"},
 [0x0FAB] = {status="valid"},
 [0x0FB8] = {status="valid"}, -- TIBETAN SUBJOINED LETTER A
 [0x0FB9] = {0x0F90, 0x0FB5}, -- TIBETAN SUBJOINED LETTER KSSA
 [0x0FBD] = {status="disallowed"}, -- NA   <reserved-0FBD>
 -- TIBETAN SUBJOINED LETTER FIXED-FORM WA..TIBETAN SUBJOINED LETTER FIXED-FORM RA
 [0x0FBA] = {status="valid"},
 [0x0FBB] = {status="valid"},
 [0x0FBC] = {status="valid"},
 [0x0FC6] = {status="valid"}, -- TIBETAN SYMBOL PADMA GDAN
 [0x0FCD] = {status="disallowed"}, -- NA   <reserved-0FCD>
 [0x0FCE] = {status="disallowed"}, -- TIBETAN SIGN RDEL NAG RDEL DKAR
 [0x0FCF] = {status="disallowed"}, -- TIBETAN SIGN RDEL NAG GSUM
 [0x1022] = {status="valid"}, -- MYANMAR LETTER SHAN A
 [0x1028] = {status="valid"}, -- MYANMAR LETTER MON E
 -- MYANMAR LETTER I..MYANMAR LETTER E
 [0x1023] = {status="valid"},
 [0x1024] = {status="valid"},
 [0x1025] = {status="valid"},
 [0x1026] = {status="valid"},
 [0x1027] = {status="valid"},
 [0x102B] = {status="valid"}, -- MYANMAR VOWEL SIGN TALL AA
 -- MYANMAR LETTER O..MYANMAR LETTER AU
 [0x1029] = {status="valid"},
 [0x102A] = {status="valid"},
 -- MYANMAR SYMBOL SHAN ONE..MYANMAR SYMBOL SHAN EXCLAMATION
 [0x109E] = {status="disallowed"},
 [0x109F] = {status="disallowed"},
 [0x10C6] = {status="disallowed"}, -- NA   <reserved-10C6>
 [0x10C7] = 0x2D27, -- GEORGIAN CAPITAL LETTER YN
 [0x10CD] = 0x2D2D, -- GEORGIAN CAPITAL LETTER AEN
 -- NA   <reserved-10C8>..<reserved-10CC>
 [0x10C8] = {status="disallowed"},
 [0x10C9] = {status="disallowed"},
 [0x10CA] = {status="disallowed"},
 [0x10CB] = {status="disallowed"},
 [0x10CC] = {status="disallowed"},
 -- NA   <reserved-10CE>..<reserved-10CF>
 [0x10CE] = {status="disallowed"},
 [0x10CF] = {status="disallowed"},
 [0x10FB] = {status="disallowed"}, -- GEORGIAN PARAGRAPH SEPARATOR
 [0x10FC] = 0x10DC, -- MODIFIER LETTER GEORGIAN NAR
 -- GEORGIAN LETTER AEN..GEORGIAN LETTER LABIAL SIGN
 [0x10FD] = {status="valid"},
 [0x10FE] = {status="valid"},
 [0x10FF] = {status="valid"},
 -- HANGUL CHOSEONG FILLER..HANGUL JUNGSEONG FILLER
 [0x115F] = {status="disallowed"},
 [0x1160] = {status="disallowed"},
 [0x1207] = {status="valid"}, -- ETHIOPIC SYLLABLE HOA
 [0x1247] = {status="valid"}, -- ETHIOPIC SYLLABLE QOA
 [0x1248] = {status="valid"}, -- ETHIOPIC SYLLABLE QWA
 [0x1249] = {status="disallowed"}, -- NA   <reserved-1249>
 -- ETHIOPIC SYLLABLE QWI..ETHIOPIC SYLLABLE QWE
 [0x124A] = {status="valid"},
 [0x124B] = {status="valid"},
 [0x124C] = {status="valid"},
 [0x124D] = {status="valid"},
 -- NA   <reserved-124E>..<reserved-124F>
 [0x124E] = {status="disallowed"},
 [0x124F] = {status="disallowed"},
 [0x1257] = {status="disallowed"}, -- NA   <reserved-1257>
 [0x1258] = {status="valid"}, -- ETHIOPIC SYLLABLE QHWA
 [0x1259] = {status="disallowed"}, -- NA   <reserved-1259>
 -- ETHIOPIC SYLLABLE QHWI..ETHIOPIC SYLLABLE QHWE
 [0x125A] = {status="valid"},
 [0x125B] = {status="valid"},
 [0x125C] = {status="valid"},
 [0x125D] = {status="valid"},
 -- NA   <reserved-125E>..<reserved-125F>
 [0x125E] = {status="disallowed"},
 [0x125F] = {status="disallowed"},
 [0x1287] = {status="valid"}, -- ETHIOPIC SYLLABLE XOA
 [0x1288] = {status="valid"}, -- ETHIOPIC SYLLABLE XWA
 [0x1289] = {status="disallowed"}, -- NA   <reserved-1289>
 -- ETHIOPIC SYLLABLE XWI..ETHIOPIC SYLLABLE XWE
 [0x128A] = {status="valid"},
 [0x128B] = {status="valid"},
 [0x128C] = {status="valid"},
 [0x128D] = {status="valid"},
 -- NA   <reserved-128E>..<reserved-128F>
 [0x128E] = {status="disallowed"},
 [0x128F] = {status="disallowed"},
 [0x12AF] = {status="valid"}, -- ETHIOPIC SYLLABLE KOA
 [0x12B0] = {status="valid"}, -- ETHIOPIC SYLLABLE KWA
 [0x12B1] = {status="disallowed"}, -- NA   <reserved-12B1>
 -- ETHIOPIC SYLLABLE KWI..ETHIOPIC SYLLABLE KWE
 [0x12B2] = {status="valid"},
 [0x12B3] = {status="valid"},
 [0x12B4] = {status="valid"},
 [0x12B5] = {status="valid"},
 -- NA   <reserved-12B6>..<reserved-12B7>
 [0x12B6] = {status="disallowed"},
 [0x12B7] = {status="disallowed"},
 [0x12BF] = {status="disallowed"}, -- NA   <reserved-12BF>
 [0x12C0] = {status="valid"}, -- ETHIOPIC SYLLABLE KXWA
 [0x12C1] = {status="disallowed"}, -- NA   <reserved-12C1>
 -- ETHIOPIC SYLLABLE KXWI..ETHIOPIC SYLLABLE KXWE
 [0x12C2] = {status="valid"},
 [0x12C3] = {status="valid"},
 [0x12C4] = {status="valid"},
 [0x12C5] = {status="valid"},
 -- NA   <reserved-12C6>..<reserved-12C7>
 [0x12C6] = {status="disallowed"},
 [0x12C7] = {status="disallowed"},
 [0x12CF] = {status="valid"}, -- ETHIOPIC SYLLABLE WOA
 [0x12D7] = {status="disallowed"}, -- NA   <reserved-12D7>
 [0x12EF] = {status="valid"}, -- ETHIOPIC SYLLABLE YOA
 [0x130F] = {status="valid"}, -- ETHIOPIC SYLLABLE GOA
 [0x1310] = {status="valid"}, -- ETHIOPIC SYLLABLE GWA
 [0x1311] = {status="disallowed"}, -- NA   <reserved-1311>
 -- ETHIOPIC SYLLABLE GWI..ETHIOPIC SYLLABLE GWE
 [0x1312] = {status="valid"},
 [0x1313] = {status="valid"},
 [0x1314] = {status="valid"},
 [0x1315] = {status="valid"},
 -- NA   <reserved-1316>..<reserved-1317>
 [0x1316] = {status="disallowed"},
 [0x1317] = {status="disallowed"},
 [0x131F] = {status="valid"}, -- ETHIOPIC SYLLABLE GGWAA
 [0x1347] = {status="valid"}, -- ETHIOPIC SYLLABLE TZOA
 -- NA   <reserved-135B>..<reserved-135C>
 [0x135B] = {status="disallowed"},
 [0x135C] = {status="disallowed"},
 [0x135F] = {status="valid"}, -- ETHIOPIC COMBINING GEMINATION MARK
 [0x1360] = {status="disallowed"}, -- ETHIOPIC SECTION MARK
 -- ETHIOPIC COMBINING GEMINATION AND VOWEL LENGTH MARK..ETHIOPIC COMBINING VOWEL LENGTH MARK
 [0x135D] = {status="valid"},
 [0x135E] = {status="valid"},
 -- NA   <reserved-137D>..<reserved-137F>
 [0x137D] = {status="disallowed"},
 [0x137E] = {status="disallowed"},
 [0x137F] = {status="disallowed"},
 [0x13F5] = {status="valid"}, -- CHEROKEE LETTER MV
 [0x13F8] = 0x13F0, -- CHEROKEE SMALL LETTER YE
 [0x13F9] = 0x13F1, -- CHEROKEE SMALL LETTER YI
 [0x13FA] = 0x13F2, -- CHEROKEE SMALL LETTER YO
 [0x13FB] = 0x13F3, -- CHEROKEE SMALL LETTER YU
 [0x13FC] = 0x13F4, -- CHEROKEE SMALL LETTER YV
 [0x13FD] = 0x13F5, -- CHEROKEE SMALL LETTER MV
 -- NA   <reserved-13F6>..<reserved-13F7>
 [0x13F6] = {status="disallowed"},
 [0x13F7] = {status="disallowed"},
 [0x1400] = {status="disallowed"}, -- CANADIAN SYLLABICS HYPHEN
 -- NA   <reserved-13FE>..<reserved-13FF>
 [0x13FE] = {status="disallowed"},
 [0x13FF] = {status="disallowed"},
 -- CANADIAN SYLLABICS CHI SIGN..CANADIAN SYLLABICS FULL STOP
 [0x166D] = {status="disallowed"},
 [0x166E] = {status="disallowed"},
 [0x1680] = {status="disallowed"}, -- OGHAM SPACE MARK
 -- OGHAM FEATHER MARK..OGHAM REVERSED FEATHER MARK
 [0x169B] = {status="disallowed"},
 [0x169C] = {status="disallowed"},
 -- NA   <reserved-169D>..<reserved-169F>
 [0x169D] = {status="disallowed"},
 [0x169E] = {status="disallowed"},
 [0x169F] = {status="disallowed"},
 [0x170D] = {status="disallowed"}, -- NA   <reserved-170D>
 -- PHILIPPINE SINGLE PUNCTUATION..PHILIPPINE DOUBLE PUNCTUATION
 [0x1735] = {status="disallowed"},
 [0x1736] = {status="disallowed"},
 [0x176D] = {status="disallowed"}, -- NA   <reserved-176D>
 [0x1771] = {status="disallowed"}, -- NA   <reserved-1771>
 -- TAGBANWA LETTER LA..TAGBANWA LETTER SA
 [0x176E] = {status="valid"},
 [0x176F] = {status="valid"},
 [0x1770] = {status="valid"},
 -- TAGBANWA VOWEL SIGN I..TAGBANWA VOWEL SIGN U
 [0x1772] = {status="valid"},
 [0x1773] = {status="valid"},
 -- KHMER VOWEL INHERENT AQ..KHMER VOWEL INHERENT AA
 [0x17B4] = {status="disallowed"},
 [0x17B5] = {status="disallowed"},
 [0x17D7] = {status="valid"}, -- KHMER SIGN LEK TOO
 -- KHMER SIGN KHAN..KHMER SIGN CAMNUC PII KUUH
 [0x17D4] = {status="disallowed"},
 [0x17D5] = {status="disallowed"},
 [0x17D6] = {status="disallowed"},
 [0x17DC] = {status="valid"}, -- KHMER SIGN AVAKRAHASANYA
 [0x17DD] = {status="valid"}, -- KHMER SIGN ATTHACAN
 -- KHMER SIGN BEYYAL..KHMER CURRENCY SYMBOL RIEL
 [0x17D8] = {status="disallowed"},
 [0x17D9] = {status="disallowed"},
 [0x17DA] = {status="disallowed"},
 [0x17DB] = {status="disallowed"},
 -- NA   <reserved-17DE>..<reserved-17DF>
 [0x17DE] = {status="disallowed"},
 [0x17DF] = {status="disallowed"},
 [0x1806] = {status="disallowed"}, -- MONGOLIAN TODO SOFT HYPHEN
 -- MONGOLIAN SIBE SYLLABLE BOUNDARY MARKER..MONGOLIAN NIRUGU
 [0x1807] = {status="disallowed"},
 [0x1808] = {status="disallowed"},
 [0x1809] = {status="disallowed"},
 [0x180A] = {status="disallowed"},
 [0x180E] = {status="disallowed"}, -- MONGOLIAN VOWEL SEPARATOR
 [0x180F] = {status="disallowed"}, -- NA   <reserved-180F>
 -- MONGOLIAN FREE VARIATION SELECTOR ONE..MONGOLIAN FREE VARIATION SELECTOR THREE
 [0x180B] = {status='ignored'},
 [0x180C] = {status='ignored'},
 [0x180D] = {status='ignored'},
 [0x18AA] = {status="valid"}, -- MONGOLIAN LETTER MANCHU ALI GALI LHA
 -- NA   <reserved-18AB>..<reserved-18AF>
 [0x18AB] = {status="disallowed"},
 [0x18AC] = {status="disallowed"},
 [0x18AD] = {status="disallowed"},
 [0x18AE] = {status="disallowed"},
 [0x18AF] = {status="disallowed"},
 [0x191F] = {status="disallowed"}, -- NA   <reserved-191F>
 -- NA   <reserved-192C>..<reserved-192F>
 [0x192C] = {status="disallowed"},
 [0x192D] = {status="disallowed"},
 [0x192E] = {status="disallowed"},
 [0x192F] = {status="disallowed"},
 [0x1940] = {status="disallowed"}, -- LIMBU SIGN LOO
 -- NA   <reserved-193C>..<reserved-193F>
 [0x193C] = {status="disallowed"},
 [0x193D] = {status="disallowed"},
 [0x193E] = {status="disallowed"},
 [0x193F] = {status="disallowed"},
 -- NA   <reserved-1941>..<reserved-1943>
 [0x1941] = {status="disallowed"},
 [0x1942] = {status="disallowed"},
 [0x1943] = {status="disallowed"},
 -- LIMBU EXCLAMATION MARK..LIMBU QUESTION MARK
 [0x1944] = {status="disallowed"},
 [0x1945] = {status="disallowed"},
 -- NA   <reserved-196E>..<reserved-196F>
 [0x196E] = {status="disallowed"},
 [0x196F] = {status="disallowed"},
 -- TAI LE LETTER TONE-2..TAI LE LETTER TONE-6
 [0x1970] = {status="valid"},
 [0x1971] = {status="valid"},
 [0x1972] = {status="valid"},
 [0x1973] = {status="valid"},
 [0x1974] = {status="valid"},
 -- NA   <reserved-19AC>..<reserved-19AF>
 [0x19AC] = {status="disallowed"},
 [0x19AD] = {status="disallowed"},
 [0x19AE] = {status="disallowed"},
 [0x19AF] = {status="disallowed"},
 [0x19DA] = {status="disallowed"}, -- NEW TAI LUE THAM DIGIT ONE
 -- NA   <reserved-19DB>..<reserved-19DD>
 [0x19DB] = {status="disallowed"},
 [0x19DC] = {status="disallowed"},
 [0x19DD] = {status="disallowed"},
 -- NA   <reserved-1A1C>..<reserved-1A1D>
 [0x1A1C] = {status="disallowed"},
 [0x1A1D] = {status="disallowed"},
 -- BUGINESE PALLAWA..BUGINESE END OF SECTION
 [0x1A1E] = {status="disallowed"},
 [0x1A1F] = {status="disallowed"},
 [0x1A5F] = {status="disallowed"}, -- NA   <reserved-1A5F>
 -- NA   <reserved-1A7D>..<reserved-1A7E>
 [0x1A7D] = {status="disallowed"},
 [0x1A7E] = {status="disallowed"},
 [0x1AA7] = {status="valid"}, -- TAI THAM SIGN MAI YAMOK
 -- NA   <reserved-1AAE>..<reserved-1AAF>
 [0x1AAE] = {status="disallowed"},
 [0x1AAF] = {status="disallowed"},
 [0x1ABE] = {status="disallowed"}, -- COMBINING PARENTHESES OVERLAY
 -- NA   <reserved-1B4C>..<reserved-1B4F>
 [0x1B4C] = {status="disallowed"},
 [0x1B4D] = {status="disallowed"},
 [0x1B4E] = {status="disallowed"},
 [0x1B4F] = {status="disallowed"},
 -- NA   <reserved-1B7D>..<reserved-1B7F>
 [0x1B7D] = {status="disallowed"},
 [0x1B7E] = {status="disallowed"},
 [0x1B7F] = {status="disallowed"},
 -- BATAK SYMBOL BINDU NA METEK..BATAK SYMBOL BINDU PANGOLAT
 [0x1BFC] = {status="disallowed"},
 [0x1BFD] = {status="disallowed"},
 [0x1BFE] = {status="disallowed"},
 [0x1BFF] = {status="disallowed"},
 -- NA   <reserved-1C38>..<reserved-1C3A>
 [0x1C38] = {status="disallowed"},
 [0x1C39] = {status="disallowed"},
 [0x1C3A] = {status="disallowed"},
 -- LEPCHA PUNCTUATION TA-ROL..LEPCHA PUNCTUATION TSHOOK
 [0x1C3B] = {status="disallowed"},
 [0x1C3C] = {status="disallowed"},
 [0x1C3D] = {status="disallowed"},
 [0x1C3E] = {status="disallowed"},
 [0x1C3F] = {status="disallowed"},
 -- NA   <reserved-1C4A>..<reserved-1C4C>
 [0x1C4A] = {status="disallowed"},
 [0x1C4B] = {status="disallowed"},
 [0x1C4C] = {status="disallowed"},
 [0x1C80] = 0x0432, -- CYRILLIC SMALL LETTER ROUNDED VE
 [0x1C81] = 0x0434, -- CYRILLIC SMALL LETTER LONG-LEGGED DE
 [0x1C82] = 0x043E, -- CYRILLIC SMALL LETTER NARROW O
 [0x1C83] = 0x0441, -- CYRILLIC SMALL LETTER WIDE ES
 -- OL CHIKI PUNCTUATION MUCAAD..OL CHIKI PUNCTUATION DOUBLE MUCAAD
 [0x1C7E] = {status="disallowed"},
 [0x1C7F] = {status="disallowed"},
 [0x1C86] = 0x044A, -- CYRILLIC SMALL LETTER TALL HARD SIGN
 [0x1C87] = 0x0463, -- CYRILLIC SMALL LETTER TALL YAT
 [0x1C88] = 0xA64B, -- CYRILLIC SMALL LETTER UNBLENDED UK
 -- CYRILLIC SMALL LETTER TALL TE..CYRILLIC SMALL LETTER THREE-LEGGED TE
 [0x1C84] = 0x0442,
 [0x1C85] = 0x0442,
 [0x1CD3] = {status="disallowed"}, -- VEDIC SIGN NIHSHVASA
 -- VEDIC TONE KARSHANA..VEDIC TONE PRENKHA
 [0x1CD0] = {status="valid"},
 [0x1CD1] = {status="valid"},
 [0x1CD2] = {status="valid"},
 [0x1CF7] = {status="valid"}, -- VEDIC SIGN ATIKRAMA
 -- VEDIC TONE RING ABOVE..VEDIC TONE DOUBLE RING ABOVE
 [0x1CF8] = {status="valid"},
 [0x1CF9] = {status="valid"},
 [0x1D2C] = 0x0061, -- MODIFIER LETTER CAPITAL A
 [0x1D2D] = 0x00E6, -- MODIFIER LETTER CAPITAL AE
 [0x1D2E] = 0x0062, -- MODIFIER LETTER CAPITAL B
 [0x1D2F] = {status="valid"}, -- MODIFIER LETTER CAPITAL BARRED B
 [0x1D30] = 0x0064, -- MODIFIER LETTER CAPITAL D
 [0x1D31] = 0x0065, -- MODIFIER LETTER CAPITAL E
 [0x1D32] = 0x01DD, -- MODIFIER LETTER CAPITAL REVERSED E
 [0x1D33] = 0x0067, -- MODIFIER LETTER CAPITAL G
 [0x1D34] = 0x0068, -- MODIFIER LETTER CAPITAL H
 [0x1D35] = 0x0069, -- MODIFIER LETTER CAPITAL I
 [0x1D36] = 0x006A, -- MODIFIER LETTER CAPITAL J
 [0x1D37] = 0x006B, -- MODIFIER LETTER CAPITAL K
 [0x1D38] = 0x006C, -- MODIFIER LETTER CAPITAL L
 [0x1D39] = 0x006D, -- MODIFIER LETTER CAPITAL M
 [0x1D3A] = 0x006E, -- MODIFIER LETTER CAPITAL N
 [0x1D3B] = {status="valid"}, -- MODIFIER LETTER CAPITAL REVERSED N
 [0x1D3C] = 0x006F, -- MODIFIER LETTER CAPITAL O
 [0x1D3D] = 0x0223, -- MODIFIER LETTER CAPITAL OU
 [0x1D3E] = 0x0070, -- MODIFIER LETTER CAPITAL P
 [0x1D3F] = 0x0072, -- MODIFIER LETTER CAPITAL R
 [0x1D40] = 0x0074, -- MODIFIER LETTER CAPITAL T
 [0x1D41] = 0x0075, -- MODIFIER LETTER CAPITAL U
 [0x1D42] = 0x0077, -- MODIFIER LETTER CAPITAL W
 [0x1D43] = 0x0061, -- MODIFIER LETTER SMALL A
 [0x1D44] = 0x0250, -- MODIFIER LETTER SMALL TURNED A
 [0x1D45] = 0x0251, -- MODIFIER LETTER SMALL ALPHA
 [0x1D46] = 0x1D02, -- MODIFIER LETTER SMALL TURNED AE
 [0x1D47] = 0x0062, -- MODIFIER LETTER SMALL B
 [0x1D48] = 0x0064, -- MODIFIER LETTER SMALL D
 [0x1D49] = 0x0065, -- MODIFIER LETTER SMALL E
 [0x1D4A] = 0x0259, -- MODIFIER LETTER SMALL SCHWA
 [0x1D4B] = 0x025B, -- MODIFIER LETTER SMALL OPEN E
 [0x1D4C] = 0x025C, -- MODIFIER LETTER SMALL TURNED OPEN E
 [0x1D4D] = 0x0067, -- MODIFIER LETTER SMALL G
 [0x1D4E] = {status="valid"}, -- MODIFIER LETTER SMALL TURNED I
 [0x1D4F] = 0x006B, -- MODIFIER LETTER SMALL K
 [0x1D50] = 0x006D, -- MODIFIER LETTER SMALL M
 [0x1D51] = 0x014B, -- MODIFIER LETTER SMALL ENG
 [0x1D52] = 0x006F, -- MODIFIER LETTER SMALL O
 [0x1D53] = 0x0254, -- MODIFIER LETTER SMALL OPEN O
 [0x1D54] = 0x1D16, -- MODIFIER LETTER SMALL TOP HALF O
 [0x1D55] = 0x1D17, -- MODIFIER LETTER SMALL BOTTOM HALF O
 [0x1D56] = 0x0070, -- MODIFIER LETTER SMALL P
 [0x1D57] = 0x0074, -- MODIFIER LETTER SMALL T
 [0x1D58] = 0x0075, -- MODIFIER LETTER SMALL U
 [0x1D59] = 0x1D1D, -- MODIFIER LETTER SMALL SIDEWAYS U
 [0x1D5A] = 0x026F, -- MODIFIER LETTER SMALL TURNED M
 [0x1D5B] = 0x0076, -- MODIFIER LETTER SMALL V
 [0x1D5C] = 0x1D25, -- MODIFIER LETTER SMALL AIN
 [0x1D5D] = 0x03B2, -- MODIFIER LETTER SMALL BETA
 [0x1D5E] = 0x03B3, -- MODIFIER LETTER SMALL GREEK GAMMA
 [0x1D5F] = 0x03B4, -- MODIFIER LETTER SMALL DELTA
 [0x1D60] = 0x03C6, -- MODIFIER LETTER SMALL GREEK PHI
 [0x1D61] = 0x03C7, -- MODIFIER LETTER SMALL CHI
 [0x1D62] = 0x0069, -- LATIN SUBSCRIPT SMALL LETTER I
 [0x1D63] = 0x0072, -- LATIN SUBSCRIPT SMALL LETTER R
 [0x1D64] = 0x0075, -- LATIN SUBSCRIPT SMALL LETTER U
 [0x1D65] = 0x0076, -- LATIN SUBSCRIPT SMALL LETTER V
 [0x1D66] = 0x03B2, -- GREEK SUBSCRIPT SMALL LETTER BETA
 [0x1D67] = 0x03B3, -- GREEK SUBSCRIPT SMALL LETTER GAMMA
 [0x1D68] = 0x03C1, -- GREEK SUBSCRIPT SMALL LETTER RHO
 [0x1D69] = 0x03C6, -- GREEK SUBSCRIPT SMALL LETTER PHI
 [0x1D6A] = 0x03C7, -- GREEK SUBSCRIPT SMALL LETTER CHI
 [0x1D6B] = {status="valid"}, -- LATIN SMALL LETTER UE
 [0x1D78] = 0x043D, -- MODIFIER LETTER CYRILLIC EN
 [0x1D9B] = 0x0252, -- MODIFIER LETTER SMALL TURNED ALPHA
 [0x1D9C] = 0x0063, -- MODIFIER LETTER SMALL C
 [0x1D9D] = 0x0255, -- MODIFIER LETTER SMALL C WITH CURL
 [0x1D9E] = 0x00F0, -- MODIFIER LETTER SMALL ETH
 [0x1D9F] = 0x025C, -- MODIFIER LETTER SMALL REVERSED OPEN E
 [0x1DA0] = 0x0066, -- MODIFIER LETTER SMALL F
 [0x1DA1] = 0x025F, -- MODIFIER LETTER SMALL DOTLESS J WITH STROKE
 [0x1DA2] = 0x0261, -- MODIFIER LETTER SMALL SCRIPT G
 [0x1DA3] = 0x0265, -- MODIFIER LETTER SMALL TURNED H
 [0x1DA4] = 0x0268, -- MODIFIER LETTER SMALL I WITH STROKE
 [0x1DA5] = 0x0269, -- MODIFIER LETTER SMALL IOTA
 [0x1DA6] = 0x026A, -- MODIFIER LETTER SMALL CAPITAL I
 [0x1DA7] = 0x1D7B, -- MODIFIER LETTER SMALL CAPITAL I WITH STROKE
 [0x1DA8] = 0x029D, -- MODIFIER LETTER SMALL J WITH CROSSED-TAIL
 [0x1DA9] = 0x026D, -- MODIFIER LETTER SMALL L WITH RETROFLEX HOOK
 [0x1DAA] = 0x1D85, -- MODIFIER LETTER SMALL L WITH PALATAL HOOK
 [0x1DAB] = 0x029F, -- MODIFIER LETTER SMALL CAPITAL L
 [0x1DAC] = 0x0271, -- MODIFIER LETTER SMALL M WITH HOOK
 [0x1DAD] = 0x0270, -- MODIFIER LETTER SMALL TURNED M WITH LONG LEG
 [0x1DAE] = 0x0272, -- MODIFIER LETTER SMALL N WITH LEFT HOOK
 [0x1DAF] = 0x0273, -- MODIFIER LETTER SMALL N WITH RETROFLEX HOOK
 [0x1DB0] = 0x0274, -- MODIFIER LETTER SMALL CAPITAL N
 [0x1DB1] = 0x0275, -- MODIFIER LETTER SMALL BARRED O
 [0x1DB2] = 0x0278, -- MODIFIER LETTER SMALL PHI
 [0x1DB3] = 0x0282, -- MODIFIER LETTER SMALL S WITH HOOK
 [0x1DB4] = 0x0283, -- MODIFIER LETTER SMALL ESH
 [0x1DB5] = 0x01AB, -- MODIFIER LETTER SMALL T WITH PALATAL HOOK
 [0x1DB6] = 0x0289, -- MODIFIER LETTER SMALL U BAR
 [0x1DB7] = 0x028A, -- MODIFIER LETTER SMALL UPSILON
 [0x1DB8] = 0x1D1C, -- MODIFIER LETTER SMALL CAPITAL U
 [0x1DB9] = 0x028B, -- MODIFIER LETTER SMALL V WITH HOOK
 [0x1DBA] = 0x028C, -- MODIFIER LETTER SMALL TURNED V
 [0x1DBB] = 0x007A, -- MODIFIER LETTER SMALL Z
 [0x1DBC] = 0x0290, -- MODIFIER LETTER SMALL Z WITH RETROFLEX HOOK
 [0x1DBD] = 0x0291, -- MODIFIER LETTER SMALL Z WITH CURL
 [0x1DBE] = 0x0292, -- MODIFIER LETTER SMALL EZH
 [0x1DBF] = 0x03B8, -- MODIFIER LETTER SMALL THETA
 [0x1DFA] = {status="disallowed"}, -- NA   <reserved-1DFA>
 [0x1DFB] = {status="valid"}, -- COMBINING DELETION MARK
 [0x1DFC] = {status="valid"}, -- COMBINING DOUBLE INVERTED BREVE BELOW
 [0x1DFD] = {status="valid"}, -- COMBINING ALMOST EQUAL TO BELOW
 [0x1E00] = 0x1E01, -- LATIN CAPITAL LETTER A WITH RING BELOW
 [0x1E01] = {status="valid"}, -- LATIN SMALL LETTER A WITH RING BELOW
 [0x1E02] = 0x1E03, -- LATIN CAPITAL LETTER B WITH DOT ABOVE
 [0x1E03] = {status="valid"}, -- LATIN SMALL LETTER B WITH DOT ABOVE
 [0x1E04] = 0x1E05, -- LATIN CAPITAL LETTER B WITH DOT BELOW
 [0x1E05] = {status="valid"}, -- LATIN SMALL LETTER B WITH DOT BELOW
 [0x1E06] = 0x1E07, -- LATIN CAPITAL LETTER B WITH LINE BELOW
 [0x1E07] = {status="valid"}, -- LATIN SMALL LETTER B WITH LINE BELOW
 [0x1E08] = 0x1E09, -- LATIN CAPITAL LETTER C WITH CEDILLA AND ACUTE
 [0x1E09] = {status="valid"}, -- LATIN SMALL LETTER C WITH CEDILLA AND ACUTE
 [0x1E0A] = 0x1E0B, -- LATIN CAPITAL LETTER D WITH DOT ABOVE
 [0x1E0B] = {status="valid"}, -- LATIN SMALL LETTER D WITH DOT ABOVE
 [0x1E0C] = 0x1E0D, -- LATIN CAPITAL LETTER D WITH DOT BELOW
 [0x1E0D] = {status="valid"}, -- LATIN SMALL LETTER D WITH DOT BELOW
 [0x1E0E] = 0x1E0F, -- LATIN CAPITAL LETTER D WITH LINE BELOW
 [0x1E0F] = {status="valid"}, -- LATIN SMALL LETTER D WITH LINE BELOW
 [0x1E10] = 0x1E11, -- LATIN CAPITAL LETTER D WITH CEDILLA
 [0x1E11] = {status="valid"}, -- LATIN SMALL LETTER D WITH CEDILLA
 [0x1E12] = 0x1E13, -- LATIN CAPITAL LETTER D WITH CIRCUMFLEX BELOW
 [0x1E13] = {status="valid"}, -- LATIN SMALL LETTER D WITH CIRCUMFLEX BELOW
 [0x1E14] = 0x1E15, -- LATIN CAPITAL LETTER E WITH MACRON AND GRAVE
 [0x1E15] = {status="valid"}, -- LATIN SMALL LETTER E WITH MACRON AND GRAVE
 [0x1E16] = 0x1E17, -- LATIN CAPITAL LETTER E WITH MACRON AND ACUTE
 [0x1E17] = {status="valid"}, -- LATIN SMALL LETTER E WITH MACRON AND ACUTE
 [0x1E18] = 0x1E19, -- LATIN CAPITAL LETTER E WITH CIRCUMFLEX BELOW
 [0x1E19] = {status="valid"}, -- LATIN SMALL LETTER E WITH CIRCUMFLEX BELOW
 [0x1E1A] = 0x1E1B, -- LATIN CAPITAL LETTER E WITH TILDE BELOW
 [0x1E1B] = {status="valid"}, -- LATIN SMALL LETTER E WITH TILDE BELOW
 [0x1E1C] = 0x1E1D, -- LATIN CAPITAL LETTER E WITH CEDILLA AND BREVE
 [0x1E1D] = {status="valid"}, -- LATIN SMALL LETTER E WITH CEDILLA AND BREVE
 [0x1E1E] = 0x1E1F, -- LATIN CAPITAL LETTER F WITH DOT ABOVE
 [0x1E1F] = {status="valid"}, -- LATIN SMALL LETTER F WITH DOT ABOVE
 [0x1E20] = 0x1E21, -- LATIN CAPITAL LETTER G WITH MACRON
 [0x1E21] = {status="valid"}, -- LATIN SMALL LETTER G WITH MACRON
 [0x1E22] = 0x1E23, -- LATIN CAPITAL LETTER H WITH DOT ABOVE
 [0x1E23] = {status="valid"}, -- LATIN SMALL LETTER H WITH DOT ABOVE
 [0x1E24] = 0x1E25, -- LATIN CAPITAL LETTER H WITH DOT BELOW
 [0x1E25] = {status="valid"}, -- LATIN SMALL LETTER H WITH DOT BELOW
 [0x1E26] = 0x1E27, -- LATIN CAPITAL LETTER H WITH DIAERESIS
 [0x1E27] = {status="valid"}, -- LATIN SMALL LETTER H WITH DIAERESIS
 [0x1E28] = 0x1E29, -- LATIN CAPITAL LETTER H WITH CEDILLA
 [0x1E29] = {status="valid"}, -- LATIN SMALL LETTER H WITH CEDILLA
 [0x1E2A] = 0x1E2B, -- LATIN CAPITAL LETTER H WITH BREVE BELOW
 [0x1E2B] = {status="valid"}, -- LATIN SMALL LETTER H WITH BREVE BELOW
 [0x1E2C] = 0x1E2D, -- LATIN CAPITAL LETTER I WITH TILDE BELOW
 [0x1E2D] = {status="valid"}, -- LATIN SMALL LETTER I WITH TILDE BELOW
 [0x1E2E] = 0x1E2F, -- LATIN CAPITAL LETTER I WITH DIAERESIS AND ACUTE
 [0x1E2F] = {status="valid"}, -- LATIN SMALL LETTER I WITH DIAERESIS AND ACUTE
 [0x1E30] = 0x1E31, -- LATIN CAPITAL LETTER K WITH ACUTE
 [0x1E31] = {status="valid"}, -- LATIN SMALL LETTER K WITH ACUTE
 [0x1E32] = 0x1E33, -- LATIN CAPITAL LETTER K WITH DOT BELOW
 [0x1E33] = {status="valid"}, -- LATIN SMALL LETTER K WITH DOT BELOW
 [0x1E34] = 0x1E35, -- LATIN CAPITAL LETTER K WITH LINE BELOW
 [0x1E35] = {status="valid"}, -- LATIN SMALL LETTER K WITH LINE BELOW
 [0x1E36] = 0x1E37, -- LATIN CAPITAL LETTER L WITH DOT BELOW
 [0x1E37] = {status="valid"}, -- LATIN SMALL LETTER L WITH DOT BELOW
 [0x1E38] = 0x1E39, -- LATIN CAPITAL LETTER L WITH DOT BELOW AND MACRON
 [0x1E39] = {status="valid"}, -- LATIN SMALL LETTER L WITH DOT BELOW AND MACRON
 [0x1E3A] = 0x1E3B, -- LATIN CAPITAL LETTER L WITH LINE BELOW
 [0x1E3B] = {status="valid"}, -- LATIN SMALL LETTER L WITH LINE BELOW
 [0x1E3C] = 0x1E3D, -- LATIN CAPITAL LETTER L WITH CIRCUMFLEX BELOW
 [0x1E3D] = {status="valid"}, -- LATIN SMALL LETTER L WITH CIRCUMFLEX BELOW
 [0x1E3E] = 0x1E3F, -- LATIN CAPITAL LETTER M WITH ACUTE
 [0x1E3F] = {status="valid"}, -- LATIN SMALL LETTER M WITH ACUTE
 [0x1E40] = 0x1E41, -- LATIN CAPITAL LETTER M WITH DOT ABOVE
 [0x1E41] = {status="valid"}, -- LATIN SMALL LETTER M WITH DOT ABOVE
 [0x1E42] = 0x1E43, -- LATIN CAPITAL LETTER M WITH DOT BELOW
 [0x1E43] = {status="valid"}, -- LATIN SMALL LETTER M WITH DOT BELOW
 [0x1E44] = 0x1E45, -- LATIN CAPITAL LETTER N WITH DOT ABOVE
 [0x1E45] = {status="valid"}, -- LATIN SMALL LETTER N WITH DOT ABOVE
 [0x1E46] = 0x1E47, -- LATIN CAPITAL LETTER N WITH DOT BELOW
 [0x1E47] = {status="valid"}, -- LATIN SMALL LETTER N WITH DOT BELOW
 [0x1E48] = 0x1E49, -- LATIN CAPITAL LETTER N WITH LINE BELOW
 [0x1E49] = {status="valid"}, -- LATIN SMALL LETTER N WITH LINE BELOW
 [0x1E4A] = 0x1E4B, -- LATIN CAPITAL LETTER N WITH CIRCUMFLEX BELOW
 [0x1E4B] = {status="valid"}, -- LATIN SMALL LETTER N WITH CIRCUMFLEX BELOW
 [0x1E4C] = 0x1E4D, -- LATIN CAPITAL LETTER O WITH TILDE AND ACUTE
 [0x1E4D] = {status="valid"}, -- LATIN SMALL LETTER O WITH TILDE AND ACUTE
 [0x1E4E] = 0x1E4F, -- LATIN CAPITAL LETTER O WITH TILDE AND DIAERESIS
 [0x1E4F] = {status="valid"}, -- LATIN SMALL LETTER O WITH TILDE AND DIAERESIS
 [0x1E50] = 0x1E51, -- LATIN CAPITAL LETTER O WITH MACRON AND GRAVE
 [0x1E51] = {status="valid"}, -- LATIN SMALL LETTER O WITH MACRON AND GRAVE
 [0x1E52] = 0x1E53, -- LATIN CAPITAL LETTER O WITH MACRON AND ACUTE
 [0x1E53] = {status="valid"}, -- LATIN SMALL LETTER O WITH MACRON AND ACUTE
 [0x1E54] = 0x1E55, -- LATIN CAPITAL LETTER P WITH ACUTE
 [0x1E55] = {status="valid"}, -- LATIN SMALL LETTER P WITH ACUTE
 [0x1E56] = 0x1E57, -- LATIN CAPITAL LETTER P WITH DOT ABOVE
 [0x1E57] = {status="valid"}, -- LATIN SMALL LETTER P WITH DOT ABOVE
 [0x1E58] = 0x1E59, -- LATIN CAPITAL LETTER R WITH DOT ABOVE
 [0x1E59] = {status="valid"}, -- LATIN SMALL LETTER R WITH DOT ABOVE
 [0x1E5A] = 0x1E5B, -- LATIN CAPITAL LETTER R WITH DOT BELOW
 [0x1E5B] = {status="valid"}, -- LATIN SMALL LETTER R WITH DOT BELOW
 [0x1E5C] = 0x1E5D, -- LATIN CAPITAL LETTER R WITH DOT BELOW AND MACRON
 [0x1E5D] = {status="valid"}, -- LATIN SMALL LETTER R WITH DOT BELOW AND MACRON
 [0x1E5E] = 0x1E5F, -- LATIN CAPITAL LETTER R WITH LINE BELOW
 [0x1E5F] = {status="valid"}, -- LATIN SMALL LETTER R WITH LINE BELOW
 [0x1E60] = 0x1E61, -- LATIN CAPITAL LETTER S WITH DOT ABOVE
 [0x1E61] = {status="valid"}, -- LATIN SMALL LETTER S WITH DOT ABOVE
 [0x1E62] = 0x1E63, -- LATIN CAPITAL LETTER S WITH DOT BELOW
 [0x1E63] = {status="valid"}, -- LATIN SMALL LETTER S WITH DOT BELOW
 [0x1E64] = 0x1E65, -- LATIN CAPITAL LETTER S WITH ACUTE AND DOT ABOVE
 [0x1E65] = {status="valid"}, -- LATIN SMALL LETTER S WITH ACUTE AND DOT ABOVE
 [0x1E66] = 0x1E67, -- LATIN CAPITAL LETTER S WITH CARON AND DOT ABOVE
 [0x1E67] = {status="valid"}, -- LATIN SMALL LETTER S WITH CARON AND DOT ABOVE
 [0x1E68] = 0x1E69, -- LATIN CAPITAL LETTER S WITH DOT BELOW AND DOT ABOVE
 [0x1E69] = {status="valid"}, -- LATIN SMALL LETTER S WITH DOT BELOW AND DOT ABOVE
 [0x1E6A] = 0x1E6B, -- LATIN CAPITAL LETTER T WITH DOT ABOVE
 [0x1E6B] = {status="valid"}, -- LATIN SMALL LETTER T WITH DOT ABOVE
 [0x1E6C] = 0x1E6D, -- LATIN CAPITAL LETTER T WITH DOT BELOW
 [0x1E6D] = {status="valid"}, -- LATIN SMALL LETTER T WITH DOT BELOW
 [0x1E6E] = 0x1E6F, -- LATIN CAPITAL LETTER T WITH LINE BELOW
 [0x1E6F] = {status="valid"}, -- LATIN SMALL LETTER T WITH LINE BELOW
 [0x1E70] = 0x1E71, -- LATIN CAPITAL LETTER T WITH CIRCUMFLEX BELOW
 [0x1E71] = {status="valid"}, -- LATIN SMALL LETTER T WITH CIRCUMFLEX BELOW
 [0x1E72] = 0x1E73, -- LATIN CAPITAL LETTER U WITH DIAERESIS BELOW
 [0x1E73] = {status="valid"}, -- LATIN SMALL LETTER U WITH DIAERESIS BELOW
 [0x1E74] = 0x1E75, -- LATIN CAPITAL LETTER U WITH TILDE BELOW
 [0x1E75] = {status="valid"}, -- LATIN SMALL LETTER U WITH TILDE BELOW
 [0x1E76] = 0x1E77, -- LATIN CAPITAL LETTER U WITH CIRCUMFLEX BELOW
 [0x1E77] = {status="valid"}, -- LATIN SMALL LETTER U WITH CIRCUMFLEX BELOW
 [0x1E78] = 0x1E79, -- LATIN CAPITAL LETTER U WITH TILDE AND ACUTE
 [0x1E79] = {status="valid"}, -- LATIN SMALL LETTER U WITH TILDE AND ACUTE
 [0x1E7A] = 0x1E7B, -- LATIN CAPITAL LETTER U WITH MACRON AND DIAERESIS
 [0x1E7B] = {status="valid"}, -- LATIN SMALL LETTER U WITH MACRON AND DIAERESIS
 [0x1E7C] = 0x1E7D, -- LATIN CAPITAL LETTER V WITH TILDE
 [0x1E7D] = {status="valid"}, -- LATIN SMALL LETTER V WITH TILDE
 [0x1E7E] = 0x1E7F, -- LATIN CAPITAL LETTER V WITH DOT BELOW
 [0x1E7F] = {status="valid"}, -- LATIN SMALL LETTER V WITH DOT BELOW
 [0x1E80] = 0x1E81, -- LATIN CAPITAL LETTER W WITH GRAVE
 [0x1E81] = {status="valid"}, -- LATIN SMALL LETTER W WITH GRAVE
 [0x1E82] = 0x1E83, -- LATIN CAPITAL LETTER W WITH ACUTE
 [0x1E83] = {status="valid"}, -- LATIN SMALL LETTER W WITH ACUTE
 [0x1E84] = 0x1E85, -- LATIN CAPITAL LETTER W WITH DIAERESIS
 [0x1E85] = {status="valid"}, -- LATIN SMALL LETTER W WITH DIAERESIS
 [0x1E86] = 0x1E87, -- LATIN CAPITAL LETTER W WITH DOT ABOVE
 [0x1E87] = {status="valid"}, -- LATIN SMALL LETTER W WITH DOT ABOVE
 [0x1E88] = 0x1E89, -- LATIN CAPITAL LETTER W WITH DOT BELOW
 [0x1E89] = {status="valid"}, -- LATIN SMALL LETTER W WITH DOT BELOW
 [0x1E8A] = 0x1E8B, -- LATIN CAPITAL LETTER X WITH DOT ABOVE
 [0x1E8B] = {status="valid"}, -- LATIN SMALL LETTER X WITH DOT ABOVE
 [0x1E8C] = 0x1E8D, -- LATIN CAPITAL LETTER X WITH DIAERESIS
 [0x1E8D] = {status="valid"}, -- LATIN SMALL LETTER X WITH DIAERESIS
 [0x1E8E] = 0x1E8F, -- LATIN CAPITAL LETTER Y WITH DOT ABOVE
 [0x1E8F] = {status="valid"}, -- LATIN SMALL LETTER Y WITH DOT ABOVE
 [0x1E90] = 0x1E91, -- LATIN CAPITAL LETTER Z WITH CIRCUMFLEX
 [0x1E91] = {status="valid"}, -- LATIN SMALL LETTER Z WITH CIRCUMFLEX
 [0x1E92] = 0x1E93, -- LATIN CAPITAL LETTER Z WITH DOT BELOW
 [0x1E93] = {status="valid"}, -- LATIN SMALL LETTER Z WITH DOT BELOW
 [0x1E94] = 0x1E95, -- LATIN CAPITAL LETTER Z WITH LINE BELOW
 -- COMBINING LEFT ARROWHEAD ABOVE..COMBINING RIGHT ARROWHEAD AND DOWN ARROWHEAD BELOW
 [0x1DFE] = {status="valid"},
 [0x1DFF] = {status="valid"},
 [0x1E9A] = {0x0061, 0x02BE}, -- LATIN SMALL LETTER A WITH RIGHT HALF RING
 [0x1E9B] = 0x1E61, -- LATIN SMALL LETTER LONG S WITH DOT ABOVE
 -- LATIN SMALL LETTER Z WITH LINE BELOW..LATIN SMALL LETTER Y WITH RING ABOVE
 [0x1E95] = {status="valid"},
 [0x1E96] = {status="valid"},
 [0x1E97] = {status="valid"},
 [0x1E98] = {status="valid"},
 [0x1E99] = {status="valid"},
 [0x1E9E] = {0x0073, 0x0073}, -- LATIN CAPITAL LETTER SHARP S
 [0x1E9F] = {status="valid"}, -- LATIN SMALL LETTER DELTA
 [0x1EA0] = 0x1EA1, -- LATIN CAPITAL LETTER A WITH DOT BELOW
 [0x1EA1] = {status="valid"}, -- LATIN SMALL LETTER A WITH DOT BELOW
 [0x1EA2] = 0x1EA3, -- LATIN CAPITAL LETTER A WITH HOOK ABOVE
 [0x1EA3] = {status="valid"}, -- LATIN SMALL LETTER A WITH HOOK ABOVE
 [0x1EA4] = 0x1EA5, -- LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND ACUTE
 [0x1EA5] = {status="valid"}, -- LATIN SMALL LETTER A WITH CIRCUMFLEX AND ACUTE
 [0x1EA6] = 0x1EA7, -- LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND GRAVE
 [0x1EA7] = {status="valid"}, -- LATIN SMALL LETTER A WITH CIRCUMFLEX AND GRAVE
 [0x1EA8] = 0x1EA9, -- LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE
 [0x1EA9] = {status="valid"}, -- LATIN SMALL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE
 [0x1EAA] = 0x1EAB, -- LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND TILDE
 [0x1EAB] = {status="valid"}, -- LATIN SMALL LETTER A WITH CIRCUMFLEX AND TILDE
 [0x1EAC] = 0x1EAD, -- LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND DOT BELOW
 [0x1EAD] = {status="valid"}, -- LATIN SMALL LETTER A WITH CIRCUMFLEX AND DOT BELOW
 [0x1EAE] = 0x1EAF, -- LATIN CAPITAL LETTER A WITH BREVE AND ACUTE
 [0x1EAF] = {status="valid"}, -- LATIN SMALL LETTER A WITH BREVE AND ACUTE
 [0x1EB0] = 0x1EB1, -- LATIN CAPITAL LETTER A WITH BREVE AND GRAVE
 [0x1EB1] = {status="valid"}, -- LATIN SMALL LETTER A WITH BREVE AND GRAVE
 [0x1EB2] = 0x1EB3, -- LATIN CAPITAL LETTER A WITH BREVE AND HOOK ABOVE
 [0x1EB3] = {status="valid"}, -- LATIN SMALL LETTER A WITH BREVE AND HOOK ABOVE
 [0x1EB4] = 0x1EB5, -- LATIN CAPITAL LETTER A WITH BREVE AND TILDE
 [0x1EB5] = {status="valid"}, -- LATIN SMALL LETTER A WITH BREVE AND TILDE
 [0x1EB6] = 0x1EB7, -- LATIN CAPITAL LETTER A WITH BREVE AND DOT BELOW
 [0x1EB7] = {status="valid"}, -- LATIN SMALL LETTER A WITH BREVE AND DOT BELOW
 [0x1EB8] = 0x1EB9, -- LATIN CAPITAL LETTER E WITH DOT BELOW
 [0x1EB9] = {status="valid"}, -- LATIN SMALL LETTER E WITH DOT BELOW
 [0x1EBA] = 0x1EBB, -- LATIN CAPITAL LETTER E WITH HOOK ABOVE
 [0x1EBB] = {status="valid"}, -- LATIN SMALL LETTER E WITH HOOK ABOVE
 [0x1EBC] = 0x1EBD, -- LATIN CAPITAL LETTER E WITH TILDE
 [0x1EBD] = {status="valid"}, -- LATIN SMALL LETTER E WITH TILDE
 [0x1EBE] = 0x1EBF, -- LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND ACUTE
 [0x1EBF] = {status="valid"}, -- LATIN SMALL LETTER E WITH CIRCUMFLEX AND ACUTE
 [0x1EC0] = 0x1EC1, -- LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND GRAVE
 [0x1EC1] = {status="valid"}, -- LATIN SMALL LETTER E WITH CIRCUMFLEX AND GRAVE
 [0x1EC2] = 0x1EC3, -- LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE
 [0x1EC3] = {status="valid"}, -- LATIN SMALL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE
 [0x1EC4] = 0x1EC5, -- LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND TILDE
 [0x1EC5] = {status="valid"}, -- LATIN SMALL LETTER E WITH CIRCUMFLEX AND TILDE
 [0x1EC6] = 0x1EC7, -- LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND DOT BELOW
 [0x1EC7] = {status="valid"}, -- LATIN SMALL LETTER E WITH CIRCUMFLEX AND DOT BELOW
 [0x1EC8] = 0x1EC9, -- LATIN CAPITAL LETTER I WITH HOOK ABOVE
 [0x1EC9] = {status="valid"}, -- LATIN SMALL LETTER I WITH HOOK ABOVE
 [0x1ECA] = 0x1ECB, -- LATIN CAPITAL LETTER I WITH DOT BELOW
 [0x1ECB] = {status="valid"}, -- LATIN SMALL LETTER I WITH DOT BELOW
 [0x1ECC] = 0x1ECD, -- LATIN CAPITAL LETTER O WITH DOT BELOW
 [0x1ECD] = {status="valid"}, -- LATIN SMALL LETTER O WITH DOT BELOW
 [0x1ECE] = 0x1ECF, -- LATIN CAPITAL LETTER O WITH HOOK ABOVE
 [0x1ECF] = {status="valid"}, -- LATIN SMALL LETTER O WITH HOOK ABOVE
 [0x1ED0] = 0x1ED1, -- LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND ACUTE
 [0x1ED1] = {status="valid"}, -- LATIN SMALL LETTER O WITH CIRCUMFLEX AND ACUTE
 [0x1ED2] = 0x1ED3, -- LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND GRAVE
 [0x1ED3] = {status="valid"}, -- LATIN SMALL LETTER O WITH CIRCUMFLEX AND GRAVE
 [0x1ED4] = 0x1ED5, -- LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE
 [0x1ED5] = {status="valid"}, -- LATIN SMALL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE
 [0x1ED6] = 0x1ED7, -- LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND TILDE
 [0x1ED7] = {status="valid"}, -- LATIN SMALL LETTER O WITH CIRCUMFLEX AND TILDE
 [0x1ED8] = 0x1ED9, -- LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND DOT BELOW
 [0x1ED9] = {status="valid"}, -- LATIN SMALL LETTER O WITH CIRCUMFLEX AND DOT BELOW
 [0x1EDA] = 0x1EDB, -- LATIN CAPITAL LETTER O WITH HORN AND ACUTE
 [0x1EDB] = {status="valid"}, -- LATIN SMALL LETTER O WITH HORN AND ACUTE
 [0x1EDC] = 0x1EDD, -- LATIN CAPITAL LETTER O WITH HORN AND GRAVE
 [0x1EDD] = {status="valid"}, -- LATIN SMALL LETTER O WITH HORN AND GRAVE
 [0x1EDE] = 0x1EDF, -- LATIN CAPITAL LETTER O WITH HORN AND HOOK ABOVE
 [0x1EDF] = {status="valid"}, -- LATIN SMALL LETTER O WITH HORN AND HOOK ABOVE
 [0x1EE0] = 0x1EE1, -- LATIN CAPITAL LETTER O WITH HORN AND TILDE
 [0x1EE1] = {status="valid"}, -- LATIN SMALL LETTER O WITH HORN AND TILDE
 [0x1EE2] = 0x1EE3, -- LATIN CAPITAL LETTER O WITH HORN AND DOT BELOW
 [0x1EE3] = {status="valid"}, -- LATIN SMALL LETTER O WITH HORN AND DOT BELOW
 [0x1EE4] = 0x1EE5, -- LATIN CAPITAL LETTER U WITH DOT BELOW
 [0x1EE5] = {status="valid"}, -- LATIN SMALL LETTER U WITH DOT BELOW
 [0x1EE6] = 0x1EE7, -- LATIN CAPITAL LETTER U WITH HOOK ABOVE
 [0x1EE7] = {status="valid"}, -- LATIN SMALL LETTER U WITH HOOK ABOVE
 [0x1EE8] = 0x1EE9, -- LATIN CAPITAL LETTER U WITH HORN AND ACUTE
 [0x1EE9] = {status="valid"}, -- LATIN SMALL LETTER U WITH HORN AND ACUTE
 [0x1EEA] = 0x1EEB, -- LATIN CAPITAL LETTER U WITH HORN AND GRAVE
 [0x1EEB] = {status="valid"}, -- LATIN SMALL LETTER U WITH HORN AND GRAVE
 [0x1EEC] = 0x1EED, -- LATIN CAPITAL LETTER U WITH HORN AND HOOK ABOVE
 [0x1EED] = {status="valid"}, -- LATIN SMALL LETTER U WITH HORN AND HOOK ABOVE
 [0x1EEE] = 0x1EEF, -- LATIN CAPITAL LETTER U WITH HORN AND TILDE
 [0x1EEF] = {status="valid"}, -- LATIN SMALL LETTER U WITH HORN AND TILDE
 [0x1EF0] = 0x1EF1, -- LATIN CAPITAL LETTER U WITH HORN AND DOT BELOW
 [0x1EF1] = {status="valid"}, -- LATIN SMALL LETTER U WITH HORN AND DOT BELOW
 [0x1EF2] = 0x1EF3, -- LATIN CAPITAL LETTER Y WITH GRAVE
 [0x1EF3] = {status="valid"}, -- LATIN SMALL LETTER Y WITH GRAVE
 [0x1EF4] = 0x1EF5, -- LATIN CAPITAL LETTER Y WITH DOT BELOW
 [0x1EF5] = {status="valid"}, -- LATIN SMALL LETTER Y WITH DOT BELOW
 [0x1EF6] = 0x1EF7, -- LATIN CAPITAL LETTER Y WITH HOOK ABOVE
 [0x1EF7] = {status="valid"}, -- LATIN SMALL LETTER Y WITH HOOK ABOVE
 [0x1EF8] = 0x1EF9, -- LATIN CAPITAL LETTER Y WITH TILDE
 [0x1EF9] = {status="valid"}, -- LATIN SMALL LETTER Y WITH TILDE
 [0x1EFA] = 0x1EFB, -- LATIN CAPITAL LETTER MIDDLE-WELSH LL
 [0x1EFB] = {status="valid"}, -- LATIN SMALL LETTER MIDDLE-WELSH LL
 [0x1EFC] = 0x1EFD, -- LATIN CAPITAL LETTER MIDDLE-WELSH V
 [0x1EFD] = {status="valid"}, -- LATIN SMALL LETTER MIDDLE-WELSH V
 [0x1EFE] = 0x1EFF, -- LATIN CAPITAL LETTER Y WITH LOOP
 [0x1EFF] = {status="valid"}, -- LATIN SMALL LETTER Y WITH LOOP
 -- LATIN SMALL LETTER LONG S WITH DIAGONAL STROKE..LATIN SMALL LETTER LONG S WITH HIGH STROKE
 [0x1E9C] = {status="valid"},
 [0x1E9D] = {status="valid"},
 [0x1F08] = 0x1F00, -- GREEK CAPITAL LETTER ALPHA WITH PSILI
 [0x1F09] = 0x1F01, -- GREEK CAPITAL LETTER ALPHA WITH DASIA
 [0x1F0A] = 0x1F02, -- GREEK CAPITAL LETTER ALPHA WITH PSILI AND VARIA
 [0x1F0B] = 0x1F03, -- GREEK CAPITAL LETTER ALPHA WITH DASIA AND VARIA
 [0x1F0C] = 0x1F04, -- GREEK CAPITAL LETTER ALPHA WITH PSILI AND OXIA
 [0x1F0D] = 0x1F05, -- GREEK CAPITAL LETTER ALPHA WITH DASIA AND OXIA
 [0x1F0E] = 0x1F06, -- GREEK CAPITAL LETTER ALPHA WITH PSILI AND PERISPOMENI
 [0x1F0F] = 0x1F07, -- GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI
 [0x1F18] = 0x1F10, -- GREEK CAPITAL LETTER EPSILON WITH PSILI
 [0x1F19] = 0x1F11, -- GREEK CAPITAL LETTER EPSILON WITH DASIA
 [0x1F1A] = 0x1F12, -- GREEK CAPITAL LETTER EPSILON WITH PSILI AND VARIA
 [0x1F1B] = 0x1F13, -- GREEK CAPITAL LETTER EPSILON WITH DASIA AND VARIA
 [0x1F1C] = 0x1F14, -- GREEK CAPITAL LETTER EPSILON WITH PSILI AND OXIA
 [0x1F1D] = 0x1F15, -- GREEK CAPITAL LETTER EPSILON WITH DASIA AND OXIA
 -- NA   <reserved-1F16>..<reserved-1F17>
 [0x1F16] = {status="disallowed"},
 [0x1F17] = {status="disallowed"},
 -- NA   <reserved-1F1E>..<reserved-1F1F>
 [0x1F1E] = {status="disallowed"},
 [0x1F1F] = {status="disallowed"},
 [0x1F28] = 0x1F20, -- GREEK CAPITAL LETTER ETA WITH PSILI
 [0x1F29] = 0x1F21, -- GREEK CAPITAL LETTER ETA WITH DASIA
 [0x1F2A] = 0x1F22, -- GREEK CAPITAL LETTER ETA WITH PSILI AND VARIA
 [0x1F2B] = 0x1F23, -- GREEK CAPITAL LETTER ETA WITH DASIA AND VARIA
 [0x1F2C] = 0x1F24, -- GREEK CAPITAL LETTER ETA WITH PSILI AND OXIA
 [0x1F2D] = 0x1F25, -- GREEK CAPITAL LETTER ETA WITH DASIA AND OXIA
 [0x1F2E] = 0x1F26, -- GREEK CAPITAL LETTER ETA WITH PSILI AND PERISPOMENI
 [0x1F2F] = 0x1F27, -- GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI
 [0x1F38] = 0x1F30, -- GREEK CAPITAL LETTER IOTA WITH PSILI
 [0x1F39] = 0x1F31, -- GREEK CAPITAL LETTER IOTA WITH DASIA
 [0x1F3A] = 0x1F32, -- GREEK CAPITAL LETTER IOTA WITH PSILI AND VARIA
 [0x1F3B] = 0x1F33, -- GREEK CAPITAL LETTER IOTA WITH DASIA AND VARIA
 [0x1F3C] = 0x1F34, -- GREEK CAPITAL LETTER IOTA WITH PSILI AND OXIA
 [0x1F3D] = 0x1F35, -- GREEK CAPITAL LETTER IOTA WITH DASIA AND OXIA
 [0x1F3E] = 0x1F36, -- GREEK CAPITAL LETTER IOTA WITH PSILI AND PERISPOMENI
 [0x1F3F] = 0x1F37, -- GREEK CAPITAL LETTER IOTA WITH DASIA AND PERISPOMENI
 [0x1F48] = 0x1F40, -- GREEK CAPITAL LETTER OMICRON WITH PSILI
 [0x1F49] = 0x1F41, -- GREEK CAPITAL LETTER OMICRON WITH DASIA
 [0x1F4A] = 0x1F42, -- GREEK CAPITAL LETTER OMICRON WITH PSILI AND VARIA
 [0x1F4B] = 0x1F43, -- GREEK CAPITAL LETTER OMICRON WITH DASIA AND VARIA
 [0x1F4C] = 0x1F44, -- GREEK CAPITAL LETTER OMICRON WITH PSILI AND OXIA
 [0x1F4D] = 0x1F45, -- GREEK CAPITAL LETTER OMICRON WITH DASIA AND OXIA
 -- NA   <reserved-1F46>..<reserved-1F47>
 [0x1F46] = {status="disallowed"},
 [0x1F47] = {status="disallowed"},
 -- NA   <reserved-1F4E>..<reserved-1F4F>
 [0x1F4E] = {status="disallowed"},
 [0x1F4F] = {status="disallowed"},
 [0x1F58] = {status="disallowed"}, -- NA   <reserved-1F58>
 [0x1F59] = 0x1F51, -- GREEK CAPITAL LETTER UPSILON WITH DASIA
 [0x1F5A] = {status="disallowed"}, -- NA   <reserved-1F5A>
 [0x1F5B] = 0x1F53, -- GREEK CAPITAL LETTER UPSILON WITH DASIA AND VARIA
 [0x1F5C] = {status="disallowed"}, -- NA   <reserved-1F5C>
 [0x1F5D] = 0x1F55, -- GREEK CAPITAL LETTER UPSILON WITH DASIA AND OXIA
 [0x1F5E] = {status="disallowed"}, -- NA   <reserved-1F5E>
 [0x1F5F] = 0x1F57, -- GREEK CAPITAL LETTER UPSILON WITH DASIA AND PERISPOMENI
 [0x1F68] = 0x1F60, -- GREEK CAPITAL LETTER OMEGA WITH PSILI
 [0x1F69] = 0x1F61, -- GREEK CAPITAL LETTER OMEGA WITH DASIA
 [0x1F6A] = 0x1F62, -- GREEK CAPITAL LETTER OMEGA WITH PSILI AND VARIA
 [0x1F6B] = 0x1F63, -- GREEK CAPITAL LETTER OMEGA WITH DASIA AND VARIA
 [0x1F6C] = 0x1F64, -- GREEK CAPITAL LETTER OMEGA WITH PSILI AND OXIA
 [0x1F6D] = 0x1F65, -- GREEK CAPITAL LETTER OMEGA WITH DASIA AND OXIA
 [0x1F6E] = 0x1F66, -- GREEK CAPITAL LETTER OMEGA WITH PSILI AND PERISPOMENI
 [0x1F6F] = 0x1F67, -- GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI
 [0x1F70] = {status="valid"}, -- GREEK SMALL LETTER ALPHA WITH VARIA
 [0x1F71] = 0x03AC, -- GREEK SMALL LETTER ALPHA WITH OXIA
 [0x1F72] = {status="valid"}, -- GREEK SMALL LETTER EPSILON WITH VARIA
 [0x1F73] = 0x03AD, -- GREEK SMALL LETTER EPSILON WITH OXIA
 [0x1F74] = {status="valid"}, -- GREEK SMALL LETTER ETA WITH VARIA
 [0x1F75] = 0x03AE, -- GREEK SMALL LETTER ETA WITH OXIA
 [0x1F76] = {status="valid"}, -- GREEK SMALL LETTER IOTA WITH VARIA
 [0x1F77] = 0x03AF, -- GREEK SMALL LETTER IOTA WITH OXIA
 [0x1F78] = {status="valid"}, -- GREEK SMALL LETTER OMICRON WITH VARIA
 [0x1F79] = 0x03CC, -- GREEK SMALL LETTER OMICRON WITH OXIA
 [0x1F7A] = {status="valid"}, -- GREEK SMALL LETTER UPSILON WITH VARIA
 [0x1F7B] = 0x03CD, -- GREEK SMALL LETTER UPSILON WITH OXIA
 [0x1F7C] = {status="valid"}, -- GREEK SMALL LETTER OMEGA WITH VARIA
 [0x1F7D] = 0x03CE, -- GREEK SMALL LETTER OMEGA WITH OXIA
 [0x1F80] = {0x1F00, 0x03B9}, -- GREEK SMALL LETTER ALPHA WITH PSILI AND YPOGEGRAMMENI
 [0x1F81] = {0x1F01, 0x03B9}, -- GREEK SMALL LETTER ALPHA WITH DASIA AND YPOGEGRAMMENI
 [0x1F82] = {0x1F02, 0x03B9}, -- GREEK SMALL LETTER ALPHA WITH PSILI AND VARIA AND YPOGEGRAMMENI
 [0x1F83] = {0x1F03, 0x03B9}, -- GREEK SMALL LETTER ALPHA WITH DASIA AND VARIA AND YPOGEGRAMMENI
 [0x1F84] = {0x1F04, 0x03B9}, -- GREEK SMALL LETTER ALPHA WITH PSILI AND OXIA AND YPOGEGRAMMENI
 [0x1F85] = {0x1F05, 0x03B9}, -- GREEK SMALL LETTER ALPHA WITH DASIA AND OXIA AND YPOGEGRAMMENI
 [0x1F86] = {0x1F06, 0x03B9}, -- GREEK SMALL LETTER ALPHA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI
 [0x1F87] = {0x1F07, 0x03B9}, -- GREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
 [0x1F88] = {0x1F00, 0x03B9}, -- GREEK CAPITAL LETTER ALPHA WITH PSILI AND PROSGEGRAMMENI
 [0x1F89] = {0x1F01, 0x03B9}, -- GREEK CAPITAL LETTER ALPHA WITH DASIA AND PROSGEGRAMMENI
 [0x1F8A] = {0x1F02, 0x03B9}, -- GREEK CAPITAL LETTER ALPHA WITH PSILI AND VARIA AND PROSGEGRAMMENI
 [0x1F8B] = {0x1F03, 0x03B9}, -- GREEK CAPITAL LETTER ALPHA WITH DASIA AND VARIA AND PROSGEGRAMMENI
 [0x1F8C] = {0x1F04, 0x03B9}, -- GREEK CAPITAL LETTER ALPHA WITH PSILI AND OXIA AND PROSGEGRAMMENI
 [0x1F8D] = {0x1F05, 0x03B9}, -- GREEK CAPITAL LETTER ALPHA WITH DASIA AND OXIA AND PROSGEGRAMMENI
 [0x1F8E] = {0x1F06, 0x03B9}, -- GREEK CAPITAL LETTER ALPHA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI
 [0x1F8F] = {0x1F07, 0x03B9}, -- GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
 [0x1F90] = {0x1F20, 0x03B9}, -- GREEK SMALL LETTER ETA WITH PSILI AND YPOGEGRAMMENI
 [0x1F91] = {0x1F21, 0x03B9}, -- GREEK SMALL LETTER ETA WITH DASIA AND YPOGEGRAMMENI
 [0x1F92] = {0x1F22, 0x03B9}, -- GREEK SMALL LETTER ETA WITH PSILI AND VARIA AND YPOGEGRAMMENI
 [0x1F93] = {0x1F23, 0x03B9}, -- GREEK SMALL LETTER ETA WITH DASIA AND VARIA AND YPOGEGRAMMENI
 [0x1F94] = {0x1F24, 0x03B9}, -- GREEK SMALL LETTER ETA WITH PSILI AND OXIA AND YPOGEGRAMMENI
 [0x1F95] = {0x1F25, 0x03B9}, -- GREEK SMALL LETTER ETA WITH DASIA AND OXIA AND YPOGEGRAMMENI
 [0x1F96] = {0x1F26, 0x03B9}, -- GREEK SMALL LETTER ETA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI
 [0x1F97] = {0x1F27, 0x03B9}, -- GREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
 [0x1F98] = {0x1F20, 0x03B9}, -- GREEK CAPITAL LETTER ETA WITH PSILI AND PROSGEGRAMMENI
 [0x1F99] = {0x1F21, 0x03B9}, -- GREEK CAPITAL LETTER ETA WITH DASIA AND PROSGEGRAMMENI
 [0x1F9A] = {0x1F22, 0x03B9}, -- GREEK CAPITAL LETTER ETA WITH PSILI AND VARIA AND PROSGEGRAMMENI
 [0x1F9B] = {0x1F23, 0x03B9}, -- GREEK CAPITAL LETTER ETA WITH DASIA AND VARIA AND PROSGEGRAMMENI
 [0x1F9C] = {0x1F24, 0x03B9}, -- GREEK CAPITAL LETTER ETA WITH PSILI AND OXIA AND PROSGEGRAMMENI
 [0x1F9D] = {0x1F25, 0x03B9}, -- GREEK CAPITAL LETTER ETA WITH DASIA AND OXIA AND PROSGEGRAMMENI
 [0x1F9E] = {0x1F26, 0x03B9}, -- GREEK CAPITAL LETTER ETA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI
 [0x1F9F] = {0x1F27, 0x03B9}, -- GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
 [0x1FA0] = {0x1F60, 0x03B9}, -- GREEK SMALL LETTER OMEGA WITH PSILI AND YPOGEGRAMMENI
 [0x1FA1] = {0x1F61, 0x03B9}, -- GREEK SMALL LETTER OMEGA WITH DASIA AND YPOGEGRAMMENI
 [0x1FA2] = {0x1F62, 0x03B9}, -- GREEK SMALL LETTER OMEGA WITH PSILI AND VARIA AND YPOGEGRAMMENI
 [0x1FA3] = {0x1F63, 0x03B9}, -- GREEK SMALL LETTER OMEGA WITH DASIA AND VARIA AND YPOGEGRAMMENI
 [0x1FA4] = {0x1F64, 0x03B9}, -- GREEK SMALL LETTER OMEGA WITH PSILI AND OXIA AND YPOGEGRAMMENI
 [0x1FA5] = {0x1F65, 0x03B9}, -- GREEK SMALL LETTER OMEGA WITH DASIA AND OXIA AND YPOGEGRAMMENI
 [0x1FA6] = {0x1F66, 0x03B9}, -- GREEK SMALL LETTER OMEGA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI
 [0x1FA7] = {0x1F67, 0x03B9}, -- GREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
 [0x1FA8] = {0x1F60, 0x03B9}, -- GREEK CAPITAL LETTER OMEGA WITH PSILI AND PROSGEGRAMMENI
 [0x1FA9] = {0x1F61, 0x03B9}, -- GREEK CAPITAL LETTER OMEGA WITH DASIA AND PROSGEGRAMMENI
 [0x1FAA] = {0x1F62, 0x03B9}, -- GREEK CAPITAL LETTER OMEGA WITH PSILI AND VARIA AND PROSGEGRAMMENI
 [0x1FAB] = {0x1F63, 0x03B9}, -- GREEK CAPITAL LETTER OMEGA WITH DASIA AND VARIA AND PROSGEGRAMMENI
 [0x1FAC] = {0x1F64, 0x03B9}, -- GREEK CAPITAL LETTER OMEGA WITH PSILI AND OXIA AND PROSGEGRAMMENI
 [0x1FAD] = {0x1F65, 0x03B9}, -- GREEK CAPITAL LETTER OMEGA WITH DASIA AND OXIA AND PROSGEGRAMMENI
 [0x1FAE] = {0x1F66, 0x03B9}, -- GREEK CAPITAL LETTER OMEGA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI
 [0x1FAF] = {0x1F67, 0x03B9}, -- GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
 -- NA   <reserved-1F7E>..<reserved-1F7F>
 [0x1F7E] = {status="disallowed"},
 [0x1F7F] = {status="disallowed"},
 [0x1FB2] = {0x1F70, 0x03B9}, -- GREEK SMALL LETTER ALPHA WITH VARIA AND YPOGEGRAMMENI
 [0x1FB3] = {0x03B1, 0x03B9}, -- GREEK SMALL LETTER ALPHA WITH YPOGEGRAMMENI
 [0x1FB4] = {0x03AC, 0x03B9}, -- GREEK SMALL LETTER ALPHA WITH OXIA AND YPOGEGRAMMENI
 [0x1FB5] = {status="disallowed"}, -- NA   <reserved-1FB5>
 [0x1FB6] = {status="valid"}, -- GREEK SMALL LETTER ALPHA WITH PERISPOMENI
 [0x1FB7] = {0x1FB6, 0x03B9}, -- GREEK SMALL LETTER ALPHA WITH PERISPOMENI AND YPOGEGRAMMENI
 [0x1FB8] = 0x1FB0, -- GREEK CAPITAL LETTER ALPHA WITH VRACHY
 [0x1FB9] = 0x1FB1, -- GREEK CAPITAL LETTER ALPHA WITH MACRON
 [0x1FBA] = 0x1F70, -- GREEK CAPITAL LETTER ALPHA WITH VARIA
 [0x1FBB] = 0x03AC, -- GREEK CAPITAL LETTER ALPHA WITH OXIA
 [0x1FBC] = {0x03B1, 0x03B9}, -- GREEK CAPITAL LETTER ALPHA WITH PROSGEGRAMMENI
 [0x1FBD] = {status="disallowed"}, -- GREEK KORONIS
 [0x1FBE] = 0x03B9, -- GREEK PROSGEGRAMMENI
 [0x1FBF] = {status="disallowed"}, -- GREEK PSILI
 [0x1FC0] = {status="disallowed"}, -- GREEK PERISPOMENI
 [0x1FC1] = {status="disallowed"}, -- GREEK DIALYTIKA AND PERISPOMENI
 [0x1FC2] = {0x1F74, 0x03B9}, -- GREEK SMALL LETTER ETA WITH VARIA AND YPOGEGRAMMENI
 [0x1FC3] = {0x03B7, 0x03B9}, -- GREEK SMALL LETTER ETA WITH YPOGEGRAMMENI
 [0x1FC4] = {0x03AE, 0x03B9}, -- GREEK SMALL LETTER ETA WITH OXIA AND YPOGEGRAMMENI
 [0x1FC5] = {status="disallowed"}, -- NA   <reserved-1FC5>
 [0x1FC6] = {status="valid"}, -- GREEK SMALL LETTER ETA WITH PERISPOMENI
 [0x1FC7] = {0x1FC6, 0x03B9}, -- GREEK SMALL LETTER ETA WITH PERISPOMENI AND YPOGEGRAMMENI
 [0x1FC8] = 0x1F72, -- GREEK CAPITAL LETTER EPSILON WITH VARIA
 [0x1FC9] = 0x03AD, -- GREEK CAPITAL LETTER EPSILON WITH OXIA
 [0x1FCA] = 0x1F74, -- GREEK CAPITAL LETTER ETA WITH VARIA
 [0x1FCB] = 0x03AE, -- GREEK CAPITAL LETTER ETA WITH OXIA
 [0x1FCC] = {0x03B7, 0x03B9}, -- GREEK CAPITAL LETTER ETA WITH PROSGEGRAMMENI
 [0x1FCD] = {status="disallowed"}, -- GREEK PSILI AND VARIA
 [0x1FCE] = {status="disallowed"}, -- GREEK PSILI AND OXIA
 [0x1FCF] = {status="disallowed"}, -- GREEK PSILI AND PERISPOMENI
 -- GREEK SMALL LETTER ALPHA WITH VRACHY..GREEK SMALL LETTER ALPHA WITH MACRON
 [0x1FB0] = {status="valid"},
 [0x1FB1] = {status="valid"},
 [0x1FD3] = 0x0390, -- GREEK SMALL LETTER IOTA WITH DIALYTIKA AND OXIA
 -- GREEK SMALL LETTER IOTA WITH VRACHY..GREEK SMALL LETTER IOTA WITH DIALYTIKA AND VARIA
 [0x1FD0] = {status="valid"},
 [0x1FD1] = {status="valid"},
 [0x1FD2] = {status="valid"},
 -- NA   <reserved-1FD4>..<reserved-1FD5>
 [0x1FD4] = {status="disallowed"},
 [0x1FD5] = {status="disallowed"},
 [0x1FD8] = 0x1FD0, -- GREEK CAPITAL LETTER IOTA WITH VRACHY
 [0x1FD9] = 0x1FD1, -- GREEK CAPITAL LETTER IOTA WITH MACRON
 [0x1FDA] = 0x1F76, -- GREEK CAPITAL LETTER IOTA WITH VARIA
 [0x1FDB] = 0x03AF, -- GREEK CAPITAL LETTER IOTA WITH OXIA
 [0x1FDC] = {status="disallowed"}, -- NA   <reserved-1FDC>
 [0x1FDD] = {status="disallowed"}, -- GREEK DASIA AND VARIA
 [0x1FDE] = {status="disallowed"}, -- GREEK DASIA AND OXIA
 [0x1FDF] = {status="disallowed"}, -- GREEK DASIA AND PERISPOMENI
 -- GREEK SMALL LETTER IOTA WITH PERISPOMENI..GREEK SMALL LETTER IOTA WITH DIALYTIKA AND PERISPOMENI
 [0x1FD6] = {status="valid"},
 [0x1FD7] = {status="valid"},
 [0x1FE3] = 0x03B0, -- GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND OXIA
 -- GREEK SMALL LETTER UPSILON WITH VRACHY..GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND VARIA
 [0x1FE0] = {status="valid"},
 [0x1FE1] = {status="valid"},
 [0x1FE2] = {status="valid"},
 [0x1FE8] = 0x1FE0, -- GREEK CAPITAL LETTER UPSILON WITH VRACHY
 [0x1FE9] = 0x1FE1, -- GREEK CAPITAL LETTER UPSILON WITH MACRON
 [0x1FEA] = 0x1F7A, -- GREEK CAPITAL LETTER UPSILON WITH VARIA
 [0x1FEB] = 0x03CD, -- GREEK CAPITAL LETTER UPSILON WITH OXIA
 [0x1FEC] = 0x1FE5, -- GREEK CAPITAL LETTER RHO WITH DASIA
 [0x1FED] = {status="disallowed"}, -- GREEK DIALYTIKA AND VARIA
 [0x1FEE] = {status="disallowed"}, -- GREEK DIALYTIKA AND OXIA
 [0x1FEF] = {status="disallowed"}, -- GREEK VARIA
 -- GREEK SMALL LETTER RHO WITH PSILI..GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND PERISPOMENI
 [0x1FE4] = {status="valid"},
 [0x1FE5] = {status="valid"},
 [0x1FE6] = {status="valid"},
 [0x1FE7] = {status="valid"},
 [0x1FF2] = {0x1F7C, 0x03B9}, -- GREEK SMALL LETTER OMEGA WITH VARIA AND YPOGEGRAMMENI
 [0x1FF3] = {0x03C9, 0x03B9}, -- GREEK SMALL LETTER OMEGA WITH YPOGEGRAMMENI
 [0x1FF4] = {0x03CE, 0x03B9}, -- GREEK SMALL LETTER OMEGA WITH OXIA AND YPOGEGRAMMENI
 [0x1FF5] = {status="disallowed"}, -- NA   <reserved-1FF5>
 [0x1FF6] = {status="valid"}, -- GREEK SMALL LETTER OMEGA WITH PERISPOMENI
 [0x1FF7] = {0x1FF6, 0x03B9}, -- GREEK SMALL LETTER OMEGA WITH PERISPOMENI AND YPOGEGRAMMENI
 [0x1FF8] = 0x1F78, -- GREEK CAPITAL LETTER OMICRON WITH VARIA
 [0x1FF9] = 0x03CC, -- GREEK CAPITAL LETTER OMICRON WITH OXIA
 [0x1FFA] = 0x1F7C, -- GREEK CAPITAL LETTER OMEGA WITH VARIA
 [0x1FFB] = 0x03CE, -- GREEK CAPITAL LETTER OMEGA WITH OXIA
 [0x1FFC] = {0x03C9, 0x03B9}, -- GREEK CAPITAL LETTER OMEGA WITH PROSGEGRAMMENI
 [0x1FFD] = {status="disallowed"}, -- GREEK OXIA
 [0x1FFE] = {status="disallowed"}, -- GREEK DASIA
 [0x1FFF] = {status="disallowed"}, -- NA   <reserved-1FFF>
 -- NA   <reserved-1FF0>..<reserved-1FF1>
 [0x1FF0] = {status="disallowed"},
 [0x1FF1] = {status="disallowed"},
 [0x200B] = {status='ignored'}, -- ZERO WIDTH SPACE
 -- ZERO WIDTH NON-JOINER..ZERO WIDTH JOINER
 [0x200C] = {status='deviation', {}},
 [0x200D] = {status='deviation', {}},
 [0x2010] = {status="disallowed"}, -- HYPHEN
 [0x2011] = 0x2010, -- NON-BREAKING HYPHEN
 -- LEFT-TO-RIGHT MARK..RIGHT-TO-LEFT MARK
 [0x200E] = {status="disallowed"},
 [0x200F] = {status="disallowed"},
 [0x2017] = {status="disallowed"}, -- DOUBLE LOW LINE
 -- FIGURE DASH..DOUBLE VERTICAL LINE
 [0x2012] = {status="disallowed"},
 [0x2013] = {status="disallowed"},
 [0x2014] = {status="disallowed"},
 [0x2015] = {status="disallowed"},
 [0x2016] = {status="disallowed"},
 [0x2027] = {status="disallowed"}, -- HYPHENATION POINT
 -- ONE DOT LEADER..HORIZONTAL ELLIPSIS
 [0x2024] = {status="disallowed"},
 [0x2025] = {status="disallowed"},
 [0x2026] = {status="disallowed"},
 [0x202F] = {status="disallowed"}, -- NARROW NO-BREAK SPACE
 [0x2033] = {0x2032, 0x2032}, -- DOUBLE PRIME
 [0x2034] = {0x2032, 0x2032, 0x2032}, -- TRIPLE PRIME
 [0x2035] = {status="disallowed"}, -- REVERSED PRIME
 [0x2036] = {0x2035, 0x2035}, -- REVERSED DOUBLE PRIME
 [0x2037] = {0x2035, 0x2035, 0x2035}, -- REVERSED TRIPLE PRIME
 -- PER MILLE SIGN..PRIME
 [0x2030] = {status="disallowed"},
 [0x2031] = {status="disallowed"},
 [0x2032] = {status="disallowed"},
 [0x203C] = {status="disallowed"}, -- DOUBLE EXCLAMATION MARK
 [0x203D] = {status="disallowed"}, -- INTERROBANG
 [0x203E] = {status="disallowed"}, -- OVERLINE
 -- CARET..REFERENCE MARK
 [0x2038] = {status="disallowed"},
 [0x2039] = {status="disallowed"},
 [0x203A] = {status="disallowed"},
 [0x203B] = {status="disallowed"},
 [0x2047] = {status="disallowed"}, -- DOUBLE QUESTION MARK
 [0x2048] = {status="disallowed"}, -- QUESTION EXCLAMATION MARK
 [0x2049] = {status="disallowed"}, -- EXCLAMATION QUESTION MARK
 [0x2057] = {0x2032, 0x2032, 0x2032, 0x2032}, -- QUADRUPLE PRIME
 [0x205F] = {status="disallowed"}, -- MEDIUM MATHEMATICAL SPACE
 [0x2060] = {status='ignored'}, -- WORD JOINER
 [0x2064] = {status='ignored'}, -- INVISIBLE PLUS
 [0x2065] = {status="disallowed"}, -- NA   <reserved-2065>
 -- FUNCTION APPLICATION..INVISIBLE SEPARATOR
 [0x2061] = {status="disallowed"},
 [0x2062] = {status="disallowed"},
 [0x2063] = {status="disallowed"},
 [0x2070] = 0x0030, -- SUPERSCRIPT ZERO
 [0x2071] = 0x0069, -- SUPERSCRIPT LATIN SMALL LETTER I
 [0x2074] = 0x0034, -- SUPERSCRIPT FOUR
 [0x2075] = 0x0035, -- SUPERSCRIPT FIVE
 [0x2076] = 0x0036, -- SUPERSCRIPT SIX
 [0x2077] = 0x0037, -- SUPERSCRIPT SEVEN
 [0x2078] = 0x0038, -- SUPERSCRIPT EIGHT
 [0x2079] = 0x0039, -- SUPERSCRIPT NINE
 [0x207A] = {status="disallowed"}, -- SUPERSCRIPT PLUS SIGN
 [0x207B] = 0x2212, -- SUPERSCRIPT MINUS
 [0x207C] = {status="disallowed"}, -- SUPERSCRIPT EQUALS SIGN
 [0x207D] = {status="disallowed"}, -- SUPERSCRIPT LEFT PARENTHESIS
 [0x207E] = {status="disallowed"}, -- SUPERSCRIPT RIGHT PARENTHESIS
 [0x207F] = 0x006E, -- SUPERSCRIPT LATIN SMALL LETTER N
 [0x2080] = 0x0030, -- SUBSCRIPT ZERO
 [0x2081] = 0x0031, -- SUBSCRIPT ONE
 [0x2082] = 0x0032, -- SUBSCRIPT TWO
 [0x2083] = 0x0033, -- SUBSCRIPT THREE
 [0x2084] = 0x0034, -- SUBSCRIPT FOUR
 [0x2085] = 0x0035, -- SUBSCRIPT FIVE
 [0x2086] = 0x0036, -- SUBSCRIPT SIX
 [0x2087] = 0x0037, -- SUBSCRIPT SEVEN
 [0x2088] = 0x0038, -- SUBSCRIPT EIGHT
 [0x2089] = 0x0039, -- SUBSCRIPT NINE
 [0x208A] = {status="disallowed"}, -- SUBSCRIPT PLUS SIGN
 [0x208B] = 0x2212, -- SUBSCRIPT MINUS
 [0x208C] = {status="disallowed"}, -- SUBSCRIPT EQUALS SIGN
 [0x208D] = {status="disallowed"}, -- SUBSCRIPT LEFT PARENTHESIS
 [0x208E] = {status="disallowed"}, -- SUBSCRIPT RIGHT PARENTHESIS
 [0x208F] = {status="disallowed"}, -- NA   <reserved-208F>
 [0x2090] = 0x0061, -- LATIN SUBSCRIPT SMALL LETTER A
 [0x2091] = 0x0065, -- LATIN SUBSCRIPT SMALL LETTER E
 [0x2092] = 0x006F, -- LATIN SUBSCRIPT SMALL LETTER O
 [0x2093] = 0x0078, -- LATIN SUBSCRIPT SMALL LETTER X
 [0x2094] = 0x0259, -- LATIN SUBSCRIPT SMALL LETTER SCHWA
 [0x2095] = 0x0068, -- LATIN SUBSCRIPT SMALL LETTER H
 [0x2096] = 0x006B, -- LATIN SUBSCRIPT SMALL LETTER K
 [0x2097] = 0x006C, -- LATIN SUBSCRIPT SMALL LETTER L
 [0x2098] = 0x006D, -- LATIN SUBSCRIPT SMALL LETTER M
 [0x2099] = 0x006E, -- LATIN SUBSCRIPT SMALL LETTER N
 [0x209A] = 0x0070, -- LATIN SUBSCRIPT SMALL LETTER P
 [0x209B] = 0x0073, -- LATIN SUBSCRIPT SMALL LETTER S
 [0x209C] = 0x0074, -- LATIN SUBSCRIPT SMALL LETTER T
 -- NA   <reserved-2072>..<reserved-2073>
 [0x2072] = {status="disallowed"},
 [0x2073] = {status="disallowed"},
 -- NA   <reserved-209D>..<reserved-209F>
 [0x209D] = {status="disallowed"},
 [0x209E] = {status="disallowed"},
 [0x209F] = {status="disallowed"},
 [0x20A8] = {0x0072, 0x0073}, -- RUPEE SIGN
 [0x20AB] = {status="disallowed"}, -- DONG SIGN
 [0x20AC] = {status="disallowed"}, -- EURO SIGN
 -- WON SIGN..NEW SHEQEL SIGN
 [0x20A9] = {status="disallowed"},
 [0x20AA] = {status="disallowed"},
 [0x20B9] = {status="disallowed"}, -- INDIAN RUPEE SIGN
 [0x20BA] = {status="disallowed"}, -- TURKISH LIRA SIGN
 [0x20BE] = {status="disallowed"}, -- LARI SIGN
 [0x20BF] = {status="disallowed"}, -- BITCOIN SIGN
 -- NORDIC MARK SIGN..RUBLE SIGN
 [0x20BB] = {status="disallowed"},
 [0x20BC] = {status="disallowed"},
 [0x20BD] = {status="disallowed"},
 [0x20EB] = {status="disallowed"}, -- COMBINING LONG DOUBLE SOLIDUS OVERLAY
 [0x20F0] = {status="disallowed"}, -- COMBINING ASTERISK ABOVE
 -- COMBINING RIGHTWARDS HARPOON WITH BARB DOWNWARDS..COMBINING RIGHT ARROW BELOW
 [0x20EC] = {status="disallowed"},
 [0x20ED] = {status="disallowed"},
 [0x20EE] = {status="disallowed"},
 [0x20EF] = {status="disallowed"},
 [0x2100] = {status="disallowed"}, -- ACCOUNT OF
 [0x2101] = {status="disallowed"}, -- ADDRESSED TO THE SUBJECT
 [0x2102] = 0x0063, -- DOUBLE-STRUCK CAPITAL C
 [0x2103] = {0x00B0, 0x0063}, -- DEGREE CELSIUS
 [0x2104] = {status="disallowed"}, -- CENTRE LINE SYMBOL
 [0x2105] = {status="disallowed"}, -- CARE OF
 [0x2106] = {status="disallowed"}, -- CADA UNA
 [0x2107] = 0x025B, -- EULER CONSTANT
 [0x2108] = {status="disallowed"}, -- SCRUPLE
 [0x2109] = {0x00B0, 0x0066}, -- DEGREE FAHRENHEIT
 [0x210A] = 0x0067, -- SCRIPT SMALL G
 [0x210F] = 0x0127, -- PLANCK CONSTANT OVER TWO PI
 -- SCRIPT CAPITAL H..PLANCK CONSTANT
 [0x210B] = 0x0068,
 [0x210C] = 0x0068,
 [0x210D] = 0x0068,
 [0x210E] = 0x0068,
 -- SCRIPT CAPITAL I..BLACK-LETTER CAPITAL I
 [0x2110] = 0x0069,
 [0x2111] = 0x0069,
 [0x2114] = {status="disallowed"}, -- L B BAR SYMBOL
 [0x2115] = 0x006E, -- DOUBLE-STRUCK CAPITAL N
 [0x2116] = {0x006E, 0x006F}, -- NUMERO SIGN
 -- SCRIPT CAPITAL L..SCRIPT SMALL L
 [0x2112] = 0x006C,
 [0x2113] = 0x006C,
 [0x2119] = 0x0070, -- DOUBLE-STRUCK CAPITAL P
 [0x211A] = 0x0071, -- DOUBLE-STRUCK CAPITAL Q
 -- SOUND RECORDING COPYRIGHT..SCRIPT CAPITAL P
 [0x2117] = {status="disallowed"},
 [0x2118] = {status="disallowed"},
 -- SCRIPT CAPITAL R..DOUBLE-STRUCK CAPITAL R
 [0x211B] = 0x0072,
 [0x211C] = 0x0072,
 [0x211D] = 0x0072,
 [0x2120] = {0x0073, 0x006D}, -- SERVICE MARK
 [0x2121] = {0x0074, 0x0065, 0x006C}, -- TELEPHONE SIGN
 [0x2122] = {0x0074, 0x006D}, -- TRADE MARK SIGN
 [0x2123] = {status="disallowed"}, -- VERSICLE
 [0x2124] = 0x007A, -- DOUBLE-STRUCK CAPITAL Z
 [0x2125] = {status="disallowed"}, -- OUNCE SIGN
 [0x2126] = 0x03C9, -- OHM SIGN
 [0x2127] = {status="disallowed"}, -- INVERTED OHM SIGN
 [0x2128] = 0x007A, -- BLACK-LETTER CAPITAL Z
 [0x2129] = {status="disallowed"}, -- TURNED GREEK SMALL LETTER IOTA
 [0x212A] = 0x006B, -- KELVIN SIGN
 [0x212B] = 0x00E5, -- ANGSTROM SIGN
 [0x212C] = 0x0062, -- SCRIPT CAPITAL B
 [0x212D] = 0x0063, -- BLACK-LETTER CAPITAL C
 [0x212E] = {status="disallowed"}, -- ESTIMATED SYMBOL
 -- PRESCRIPTION TAKE..RESPONSE
 [0x211E] = {status="disallowed"},
 [0x211F] = {status="disallowed"},
 [0x2131] = 0x0066, -- SCRIPT CAPITAL F
 [0x2132] = {status="disallowed"}, -- TURNED CAPITAL F
 [0x2133] = 0x006D, -- SCRIPT CAPITAL M
 [0x2134] = 0x006F, -- SCRIPT SMALL O
 [0x2135] = 0x05D0, -- ALEF SYMBOL
 [0x2136] = 0x05D1, -- BET SYMBOL
 [0x2137] = 0x05D2, -- GIMEL SYMBOL
 [0x2138] = 0x05D3, -- DALET SYMBOL
 [0x2139] = 0x0069, -- INFORMATION SOURCE
 [0x213A] = {status="disallowed"}, -- ROTATED CAPITAL Q
 [0x213B] = {0x0066, 0x0061, 0x0078}, -- FACSIMILE SIGN
 [0x213C] = 0x03C0, -- DOUBLE-STRUCK SMALL PI
 -- SCRIPT SMALL E..SCRIPT CAPITAL E
 [0x212F] = 0x0065,
 [0x2130] = 0x0065,
 [0x213F] = 0x03C0, -- DOUBLE-STRUCK CAPITAL PI
 [0x2140] = 0x2211, -- DOUBLE-STRUCK N-ARY SUMMATION
 -- DOUBLE-STRUCK SMALL GAMMA..DOUBLE-STRUCK CAPITAL GAMMA
 [0x213D] = 0x03B3,
 [0x213E] = 0x03B3,
 -- TURNED SANS-SERIF CAPITAL G..TURNED SANS-SERIF CAPITAL Y
 [0x2141] = {status="disallowed"},
 [0x2142] = {status="disallowed"},
 [0x2143] = {status="disallowed"},
 [0x2144] = {status="disallowed"},
 [0x2147] = 0x0065, -- DOUBLE-STRUCK ITALIC SMALL E
 [0x2148] = 0x0069, -- DOUBLE-STRUCK ITALIC SMALL I
 [0x2149] = 0x006A, -- DOUBLE-STRUCK ITALIC SMALL J
 -- DOUBLE-STRUCK ITALIC CAPITAL D..DOUBLE-STRUCK ITALIC SMALL D
 [0x2145] = 0x0064,
 [0x2146] = 0x0064,
 [0x214C] = {status="disallowed"}, -- PER SIGN
 [0x214D] = {status="disallowed"}, -- AKTIESELSKAB
 [0x214E] = {status="valid"}, -- TURNED SMALL F
 [0x214F] = {status="disallowed"}, -- SYMBOL FOR SAMARITAN SOURCE
 [0x2150] = {0x0031, 0x2044, 0x0037}, -- VULGAR FRACTION ONE SEVENTH
 [0x2151] = {0x0031, 0x2044, 0x0039}, -- VULGAR FRACTION ONE NINTH
 [0x2152] = {0x0031, 0x2044, 0x0031, 0x0030}, -- VULGAR FRACTION ONE TENTH
 [0x2153] = {0x0031, 0x2044, 0x0033}, -- VULGAR FRACTION ONE THIRD
 [0x2154] = {0x0032, 0x2044, 0x0033}, -- VULGAR FRACTION TWO THIRDS
 [0x2155] = {0x0031, 0x2044, 0x0035}, -- VULGAR FRACTION ONE FIFTH
 [0x2156] = {0x0032, 0x2044, 0x0035}, -- VULGAR FRACTION TWO FIFTHS
 [0x2157] = {0x0033, 0x2044, 0x0035}, -- VULGAR FRACTION THREE FIFTHS
 [0x2158] = {0x0034, 0x2044, 0x0035}, -- VULGAR FRACTION FOUR FIFTHS
 [0x2159] = {0x0031, 0x2044, 0x0036}, -- VULGAR FRACTION ONE SIXTH
 [0x215A] = {0x0035, 0x2044, 0x0036}, -- VULGAR FRACTION FIVE SIXTHS
 [0x215B] = {0x0031, 0x2044, 0x0038}, -- VULGAR FRACTION ONE EIGHTH
 [0x215C] = {0x0033, 0x2044, 0x0038}, -- VULGAR FRACTION THREE EIGHTHS
 [0x215D] = {0x0035, 0x2044, 0x0038}, -- VULGAR FRACTION FIVE EIGHTHS
 [0x215E] = {0x0037, 0x2044, 0x0038}, -- VULGAR FRACTION SEVEN EIGHTHS
 [0x215F] = {0x0031, 0x2044}, -- FRACTION NUMERATOR ONE
 [0x2160] = 0x0069, -- ROMAN NUMERAL ONE
 [0x2161] = {0x0069, 0x0069}, -- ROMAN NUMERAL TWO
 [0x2162] = {0x0069, 0x0069, 0x0069}, -- ROMAN NUMERAL THREE
 [0x2163] = {0x0069, 0x0076}, -- ROMAN NUMERAL FOUR
 [0x2164] = 0x0076, -- ROMAN NUMERAL FIVE
 [0x2165] = {0x0076, 0x0069}, -- ROMAN NUMERAL SIX
 [0x2166] = {0x0076, 0x0069, 0x0069}, -- ROMAN NUMERAL SEVEN
 [0x2167] = {0x0076, 0x0069, 0x0069, 0x0069}, -- ROMAN NUMERAL EIGHT
 [0x2168] = {0x0069, 0x0078}, -- ROMAN NUMERAL NINE
 [0x2169] = 0x0078, -- ROMAN NUMERAL TEN
 [0x216A] = {0x0078, 0x0069}, -- ROMAN NUMERAL ELEVEN
 [0x216B] = {0x0078, 0x0069, 0x0069}, -- ROMAN NUMERAL TWELVE
 [0x216C] = 0x006C, -- ROMAN NUMERAL FIFTY
 [0x216D] = 0x0063, -- ROMAN NUMERAL ONE HUNDRED
 [0x216E] = 0x0064, -- ROMAN NUMERAL FIVE HUNDRED
 [0x216F] = 0x006D, -- ROMAN NUMERAL ONE THOUSAND
 [0x2170] = 0x0069, -- SMALL ROMAN NUMERAL ONE
 [0x2171] = {0x0069, 0x0069}, -- SMALL ROMAN NUMERAL TWO
 [0x2172] = {0x0069, 0x0069, 0x0069}, -- SMALL ROMAN NUMERAL THREE
 [0x2173] = {0x0069, 0x0076}, -- SMALL ROMAN NUMERAL FOUR
 [0x2174] = 0x0076, -- SMALL ROMAN NUMERAL FIVE
 [0x2175] = {0x0076, 0x0069}, -- SMALL ROMAN NUMERAL SIX
 [0x2176] = {0x0076, 0x0069, 0x0069}, -- SMALL ROMAN NUMERAL SEVEN
 [0x2177] = {0x0076, 0x0069, 0x0069, 0x0069}, -- SMALL ROMAN NUMERAL EIGHT
 [0x2178] = {0x0069, 0x0078}, -- SMALL ROMAN NUMERAL NINE
 [0x2179] = 0x0078, -- SMALL ROMAN NUMERAL TEN
 [0x217A] = {0x0078, 0x0069}, -- SMALL ROMAN NUMERAL ELEVEN
 [0x217B] = {0x0078, 0x0069, 0x0069}, -- SMALL ROMAN NUMERAL TWELVE
 [0x217C] = 0x006C, -- SMALL ROMAN NUMERAL FIFTY
 [0x217D] = 0x0063, -- SMALL ROMAN NUMERAL ONE HUNDRED
 [0x217E] = 0x0064, -- SMALL ROMAN NUMERAL FIVE HUNDRED
 [0x217F] = 0x006D, -- SMALL ROMAN NUMERAL ONE THOUSAND
 -- PROPERTY LINE..TURNED AMPERSAND
 [0x214A] = {status="disallowed"},
 [0x214B] = {status="disallowed"},
 [0x2183] = {status="disallowed"}, -- ROMAN NUMERAL REVERSED ONE HUNDRED
 [0x2184] = {status="valid"}, -- LATIN SMALL LETTER REVERSED C
 -- ROMAN NUMERAL ONE THOUSAND C D..ROMAN NUMERAL TEN THOUSAND
 [0x2180] = {status="disallowed"},
 [0x2181] = {status="disallowed"},
 [0x2182] = {status="disallowed"},
 [0x2189] = {0x0030, 0x2044, 0x0033}, -- VULGAR FRACTION ZERO THIRDS
 -- ROMAN NUMERAL SIX LATE FORM..ROMAN NUMERAL ONE HUNDRED THOUSAND
 [0x2185] = {status="disallowed"},
 [0x2186] = {status="disallowed"},
 [0x2187] = {status="disallowed"},
 [0x2188] = {status="disallowed"},
 -- TURNED DIGIT TWO..TURNED DIGIT THREE
 [0x218A] = {status="disallowed"},
 [0x218B] = {status="disallowed"},
 -- NA   <reserved-218C>..<reserved-218F>
 [0x218C] = {status="disallowed"},
 [0x218D] = {status="disallowed"},
 [0x218E] = {status="disallowed"},
 [0x218F] = {status="disallowed"},
 [0x222C] = {0x222B, 0x222B}, -- DOUBLE INTEGRAL
 [0x222D] = {0x222B, 0x222B, 0x222B}, -- TRIPLE INTEGRAL
 [0x222E] = {status="disallowed"}, -- CONTOUR INTEGRAL
 [0x222F] = {0x222E, 0x222E}, -- SURFACE INTEGRAL
 [0x2230] = {0x222E, 0x222E, 0x222E}, -- VOLUME INTEGRAL
 [0x2260] = {status="disallowed"}, -- NOT EQUAL TO
 -- NOT LESS-THAN..NOT GREATER-THAN
 [0x226E] = {status="disallowed"},
 [0x226F] = {status="disallowed"},
 [0x2300] = {status="disallowed"}, -- DIAMETER SIGN
 [0x2301] = {status="disallowed"}, -- ELECTRIC ARROW
 [0x2329] = 0x3008, -- LEFT-POINTING ANGLE BRACKET
 [0x232A] = 0x3009, -- RIGHT-POINTING ANGLE BRACKET
 [0x237B] = {status="disallowed"}, -- NOT CHECK MARK
 [0x237C] = {status="disallowed"}, -- RIGHT ANGLE WITH DOWNWARDS ZIGZAG ARROW
 [0x23E8] = {status="disallowed"}, -- DECIMAL EXPONENT SYMBOL
 [0x23FF] = {status="disallowed"}, -- OBSERVER EYE SYMBOL
 [0x2460] = 0x0031, -- CIRCLED DIGIT ONE
 [0x2461] = 0x0032, -- CIRCLED DIGIT TWO
 [0x2462] = 0x0033, -- CIRCLED DIGIT THREE
 [0x2463] = 0x0034, -- CIRCLED DIGIT FOUR
 [0x2464] = 0x0035, -- CIRCLED DIGIT FIVE
 [0x2465] = 0x0036, -- CIRCLED DIGIT SIX
 [0x2466] = 0x0037, -- CIRCLED DIGIT SEVEN
 [0x2467] = 0x0038, -- CIRCLED DIGIT EIGHT
 [0x2468] = 0x0039, -- CIRCLED DIGIT NINE
 [0x2469] = {0x0031, 0x0030}, -- CIRCLED NUMBER TEN
 [0x246A] = {0x0031, 0x0031}, -- CIRCLED NUMBER ELEVEN
 [0x246B] = {0x0031, 0x0032}, -- CIRCLED NUMBER TWELVE
 [0x246C] = {0x0031, 0x0033}, -- CIRCLED NUMBER THIRTEEN
 [0x246D] = {0x0031, 0x0034}, -- CIRCLED NUMBER FOURTEEN
 [0x246E] = {0x0031, 0x0035}, -- CIRCLED NUMBER FIFTEEN
 [0x246F] = {0x0031, 0x0036}, -- CIRCLED NUMBER SIXTEEN
 [0x2470] = {0x0031, 0x0037}, -- CIRCLED NUMBER SEVENTEEN
 [0x2471] = {0x0031, 0x0038}, -- CIRCLED NUMBER EIGHTEEN
 [0x2472] = {0x0031, 0x0039}, -- CIRCLED NUMBER NINETEEN
 [0x2473] = {0x0032, 0x0030}, -- CIRCLED NUMBER TWENTY
 [0x2474] = {status="disallowed"}, -- PARENTHESIZED DIGIT ONE
 [0x2475] = {status="disallowed"}, -- PARENTHESIZED DIGIT TWO
 [0x2476] = {status="disallowed"}, -- PARENTHESIZED DIGIT THREE
 [0x2477] = {status="disallowed"}, -- PARENTHESIZED DIGIT FOUR
 [0x2478] = {status="disallowed"}, -- PARENTHESIZED DIGIT FIVE
 [0x2479] = {status="disallowed"}, -- PARENTHESIZED DIGIT SIX
 [0x247A] = {status="disallowed"}, -- PARENTHESIZED DIGIT SEVEN
 [0x247B] = {status="disallowed"}, -- PARENTHESIZED DIGIT EIGHT
 [0x247C] = {status="disallowed"}, -- PARENTHESIZED DIGIT NINE
 [0x247D] = {status="disallowed"}, -- PARENTHESIZED NUMBER TEN
 [0x247E] = {status="disallowed"}, -- PARENTHESIZED NUMBER ELEVEN
 [0x247F] = {status="disallowed"}, -- PARENTHESIZED NUMBER TWELVE
 [0x2480] = {status="disallowed"}, -- PARENTHESIZED NUMBER THIRTEEN
 [0x2481] = {status="disallowed"}, -- PARENTHESIZED NUMBER FOURTEEN
 [0x2482] = {status="disallowed"}, -- PARENTHESIZED NUMBER FIFTEEN
 [0x2483] = {status="disallowed"}, -- PARENTHESIZED NUMBER SIXTEEN
 [0x2484] = {status="disallowed"}, -- PARENTHESIZED NUMBER SEVENTEEN
 [0x2485] = {status="disallowed"}, -- PARENTHESIZED NUMBER EIGHTEEN
 [0x2486] = {status="disallowed"}, -- PARENTHESIZED NUMBER NINETEEN
 [0x2487] = {status="disallowed"}, -- PARENTHESIZED NUMBER TWENTY
 [0x249C] = {status="disallowed"}, -- PARENTHESIZED LATIN SMALL LETTER A
 [0x249D] = {status="disallowed"}, -- PARENTHESIZED LATIN SMALL LETTER B
 [0x249E] = {status="disallowed"}, -- PARENTHESIZED LATIN SMALL LETTER C
 [0x249F] = {status="disallowed"}, -- PARENTHESIZED LATIN SMALL LETTER D
 [0x24A0] = {status="disallowed"}, -- PARENTHESIZED LATIN SMALL LETTER E
 [0x24A1] = {status="disallowed"}, -- PARENTHESIZED LATIN SMALL LETTER F
 [0x24A2] = {status="disallowed"}, -- PARENTHESIZED LATIN SMALL LETTER G
 [0x24A3] = {status="disallowed"}, -- PARENTHESIZED LATIN SMALL LETTER H
 [0x24A4] = {status="disallowed"}, -- PARENTHESIZED LATIN SMALL LETTER I
 [0x24A5] = {status="disallowed"}, -- PARENTHESIZED LATIN SMALL LETTER J
 [0x24A6] = {status="disallowed"}, -- PARENTHESIZED LATIN SMALL LETTER K
 [0x24A7] = {status="disallowed"}, -- PARENTHESIZED LATIN SMALL LETTER L
 [0x24A8] = {status="disallowed"}, -- PARENTHESIZED LATIN SMALL LETTER M
 [0x24A9] = {status="disallowed"}, -- PARENTHESIZED LATIN SMALL LETTER N
 [0x24AA] = {status="disallowed"}, -- PARENTHESIZED LATIN SMALL LETTER O
 [0x24AB] = {status="disallowed"}, -- PARENTHESIZED LATIN SMALL LETTER P
 [0x24AC] = {status="disallowed"}, -- PARENTHESIZED LATIN SMALL LETTER Q
 [0x24AD] = {status="disallowed"}, -- PARENTHESIZED LATIN SMALL LETTER R
 [0x24AE] = {status="disallowed"}, -- PARENTHESIZED LATIN SMALL LETTER S
 [0x24AF] = {status="disallowed"}, -- PARENTHESIZED LATIN SMALL LETTER T
 [0x24B0] = {status="disallowed"}, -- PARENTHESIZED LATIN SMALL LETTER U
 [0x24B1] = {status="disallowed"}, -- PARENTHESIZED LATIN SMALL LETTER V
 [0x24B2] = {status="disallowed"}, -- PARENTHESIZED LATIN SMALL LETTER W
 [0x24B3] = {status="disallowed"}, -- PARENTHESIZED LATIN SMALL LETTER X
 [0x24B4] = {status="disallowed"}, -- PARENTHESIZED LATIN SMALL LETTER Y
 [0x24B5] = {status="disallowed"}, -- PARENTHESIZED LATIN SMALL LETTER Z
 [0x24B6] = 0x0061, -- CIRCLED LATIN CAPITAL LETTER A
 [0x24B7] = 0x0062, -- CIRCLED LATIN CAPITAL LETTER B
 [0x24B8] = 0x0063, -- CIRCLED LATIN CAPITAL LETTER C
 [0x24B9] = 0x0064, -- CIRCLED LATIN CAPITAL LETTER D
 [0x24BA] = 0x0065, -- CIRCLED LATIN CAPITAL LETTER E
 [0x24BB] = 0x0066, -- CIRCLED LATIN CAPITAL LETTER F
 [0x24BC] = 0x0067, -- CIRCLED LATIN CAPITAL LETTER G
 [0x24BD] = 0x0068, -- CIRCLED LATIN CAPITAL LETTER H
 [0x24BE] = 0x0069, -- CIRCLED LATIN CAPITAL LETTER I
 [0x24BF] = 0x006A, -- CIRCLED LATIN CAPITAL LETTER J
 [0x24C0] = 0x006B, -- CIRCLED LATIN CAPITAL LETTER K
 [0x24C1] = 0x006C, -- CIRCLED LATIN CAPITAL LETTER L
 [0x24C2] = 0x006D, -- CIRCLED LATIN CAPITAL LETTER M
 [0x24C3] = 0x006E, -- CIRCLED LATIN CAPITAL LETTER N
 [0x24C4] = 0x006F, -- CIRCLED LATIN CAPITAL LETTER O
 [0x24C5] = 0x0070, -- CIRCLED LATIN CAPITAL LETTER P
 [0x24C6] = 0x0071, -- CIRCLED LATIN CAPITAL LETTER Q
 [0x24C7] = 0x0072, -- CIRCLED LATIN CAPITAL LETTER R
 [0x24C8] = 0x0073, -- CIRCLED LATIN CAPITAL LETTER S
 [0x24C9] = 0x0074, -- CIRCLED LATIN CAPITAL LETTER T
 [0x24CA] = 0x0075, -- CIRCLED LATIN CAPITAL LETTER U
 [0x24CB] = 0x0076, -- CIRCLED LATIN CAPITAL LETTER V
 [0x24CC] = 0x0077, -- CIRCLED LATIN CAPITAL LETTER W
 [0x24CD] = 0x0078, -- CIRCLED LATIN CAPITAL LETTER X
 [0x24CE] = 0x0079, -- CIRCLED LATIN CAPITAL LETTER Y
 [0x24CF] = 0x007A, -- CIRCLED LATIN CAPITAL LETTER Z
 [0x24D0] = 0x0061, -- CIRCLED LATIN SMALL LETTER A
 [0x24D1] = 0x0062, -- CIRCLED LATIN SMALL LETTER B
 [0x24D2] = 0x0063, -- CIRCLED LATIN SMALL LETTER C
 [0x24D3] = 0x0064, -- CIRCLED LATIN SMALL LETTER D
 [0x24D4] = 0x0065, -- CIRCLED LATIN SMALL LETTER E
 [0x24D5] = 0x0066, -- CIRCLED LATIN SMALL LETTER F
 [0x24D6] = 0x0067, -- CIRCLED LATIN SMALL LETTER G
 [0x24D7] = 0x0068, -- CIRCLED LATIN SMALL LETTER H
 [0x24D8] = 0x0069, -- CIRCLED LATIN SMALL LETTER I
 [0x24D9] = 0x006A, -- CIRCLED LATIN SMALL LETTER J
 [0x24DA] = 0x006B, -- CIRCLED LATIN SMALL LETTER K
 [0x24DB] = 0x006C, -- CIRCLED LATIN SMALL LETTER L
 [0x24DC] = 0x006D, -- CIRCLED LATIN SMALL LETTER M
 [0x24DD] = 0x006E, -- CIRCLED LATIN SMALL LETTER N
 [0x24DE] = 0x006F, -- CIRCLED LATIN SMALL LETTER O
 [0x24DF] = 0x0070, -- CIRCLED LATIN SMALL LETTER P
 [0x24E0] = 0x0071, -- CIRCLED LATIN SMALL LETTER Q
 [0x24E1] = 0x0072, -- CIRCLED LATIN SMALL LETTER R
 [0x24E2] = 0x0073, -- CIRCLED LATIN SMALL LETTER S
 [0x24E3] = 0x0074, -- CIRCLED LATIN SMALL LETTER T
 [0x24E4] = 0x0075, -- CIRCLED LATIN SMALL LETTER U
 [0x24E5] = 0x0076, -- CIRCLED LATIN SMALL LETTER V
 [0x24E6] = 0x0077, -- CIRCLED LATIN SMALL LETTER W
 [0x24E7] = 0x0078, -- CIRCLED LATIN SMALL LETTER X
 [0x24E8] = 0x0079, -- CIRCLED LATIN SMALL LETTER Y
 [0x24E9] = 0x007A, -- CIRCLED LATIN SMALL LETTER Z
 [0x24EA] = 0x0030, -- CIRCLED DIGIT ZERO
 [0x24FF] = {status="disallowed"}, -- NEGATIVE CIRCLED DIGIT ZERO
 [0x2618] = {status="disallowed"}, -- SHAMROCK
 [0x2619] = {status="disallowed"}, -- REVERSED ROTATED FLORAL HEART BULLET
 [0x269D] = {status="disallowed"}, -- OUTLINED WHITE STAR
 [0x26B2] = {status="disallowed"}, -- NEUTER
 [0x26CE] = {status="disallowed"}, -- OPHIUCHUS
 [0x26E2] = {status="disallowed"}, -- ASTRONOMICAL SYMBOL FOR URANUS
 [0x26E3] = {status="disallowed"}, -- HEAVY CIRCLE WITH STROKE AND TWO DOTS ABOVE
 [0x2700] = {status="disallowed"}, -- BLACK SAFETY SCISSORS
 [0x2705] = {status="disallowed"}, -- WHITE HEAVY CHECK MARK
 -- UPPER BLADE SCISSORS..WHITE SCISSORS
 [0x2701] = {status="disallowed"},
 [0x2702] = {status="disallowed"},
 [0x2703] = {status="disallowed"},
 [0x2704] = {status="disallowed"},
 [0x2728] = {status="disallowed"}, -- SPARKLES
 [0x274C] = {status="disallowed"}, -- CROSS MARK
 [0x274D] = {status="disallowed"}, -- SHADOWED WHITE CIRCLE
 [0x274E] = {status="disallowed"}, -- NEGATIVE SQUARED CROSS MARK
 [0x2756] = {status="disallowed"}, -- BLACK DIAMOND MINUS WHITE X
 [0x2757] = {status="disallowed"}, -- HEAVY EXCLAMATION MARK SYMBOL
 [0x27B0] = {status="disallowed"}, -- CURLY LOOP
 [0x27BF] = {status="disallowed"}, -- DOUBLE CURLY LOOP
 [0x27CB] = {status="disallowed"}, -- MATHEMATICAL RISING DIAGONAL
 [0x27CC] = {status="disallowed"}, -- LONG DIVISION
 [0x27CD] = {status="disallowed"}, -- MATHEMATICAL FALLING DIAGONAL
 [0x2A0C] = {0x222B, 0x222B, 0x222B, 0x222B}, -- QUADRUPLE INTEGRAL OPERATOR
 [0x2A74] = {status="disallowed"}, -- DOUBLE COLON EQUAL
 [0x2A75] = {status="disallowed"}, -- TWO CONSECUTIVE EQUALS SIGNS
 [0x2A76] = {status="disallowed"}, -- THREE CONSECUTIVE EQUALS SIGNS
 [0x2ADC] = {0x2ADD, 0x0338}, -- FORKING
 -- NA   <reserved-2B74>..<reserved-2B75>
 [0x2B74] = {status="disallowed"},
 [0x2B75] = {status="disallowed"},
 -- NA   <reserved-2B96>..<reserved-2B97>
 [0x2B96] = {status="disallowed"},
 [0x2B97] = {status="disallowed"},
 -- NA   <reserved-2BBA>..<reserved-2BBC>
 [0x2BBA] = {status="disallowed"},
 [0x2BBB] = {status="disallowed"},
 [0x2BBC] = {status="disallowed"},
 [0x2BC9] = {status="disallowed"}, -- NA   <reserved-2BC9>
 [0x2BD2] = {status="disallowed"}, -- GROUP MARK
 -- LEFTWARDS TWO-HEADED ARROW WITH TRIANGLE ARROWHEADS..DOWNWARDS TWO-HEADED ARROW WITH TRIANGLE ARROWHEADS
 [0x2BEC] = {status="disallowed"},
 [0x2BED] = {status="disallowed"},
 [0x2BEE] = {status="disallowed"},
 [0x2BEF] = {status="disallowed"},
 [0x2C00] = 0x2C30, -- GLAGOLITIC CAPITAL LETTER AZU
 [0x2C01] = 0x2C31, -- GLAGOLITIC CAPITAL LETTER BUKY
 [0x2C02] = 0x2C32, -- GLAGOLITIC CAPITAL LETTER VEDE
 [0x2C03] = 0x2C33, -- GLAGOLITIC CAPITAL LETTER GLAGOLI
 [0x2C04] = 0x2C34, -- GLAGOLITIC CAPITAL LETTER DOBRO
 [0x2C05] = 0x2C35, -- GLAGOLITIC CAPITAL LETTER YESTU
 [0x2C06] = 0x2C36, -- GLAGOLITIC CAPITAL LETTER ZHIVETE
 [0x2C07] = 0x2C37, -- GLAGOLITIC CAPITAL LETTER DZELO
 [0x2C08] = 0x2C38, -- GLAGOLITIC CAPITAL LETTER ZEMLJA
 [0x2C09] = 0x2C39, -- GLAGOLITIC CAPITAL LETTER IZHE
 [0x2C0A] = 0x2C3A, -- GLAGOLITIC CAPITAL LETTER INITIAL IZHE
 [0x2C0B] = 0x2C3B, -- GLAGOLITIC CAPITAL LETTER I
 [0x2C0C] = 0x2C3C, -- GLAGOLITIC CAPITAL LETTER DJERVI
 [0x2C0D] = 0x2C3D, -- GLAGOLITIC CAPITAL LETTER KAKO
 [0x2C0E] = 0x2C3E, -- GLAGOLITIC CAPITAL LETTER LJUDIJE
 [0x2C0F] = 0x2C3F, -- GLAGOLITIC CAPITAL LETTER MYSLITE
 [0x2C10] = 0x2C40, -- GLAGOLITIC CAPITAL LETTER NASHI
 [0x2C11] = 0x2C41, -- GLAGOLITIC CAPITAL LETTER ONU
 [0x2C12] = 0x2C42, -- GLAGOLITIC CAPITAL LETTER POKOJI
 [0x2C13] = 0x2C43, -- GLAGOLITIC CAPITAL LETTER RITSI
 [0x2C14] = 0x2C44, -- GLAGOLITIC CAPITAL LETTER SLOVO
 [0x2C15] = 0x2C45, -- GLAGOLITIC CAPITAL LETTER TVRIDO
 [0x2C16] = 0x2C46, -- GLAGOLITIC CAPITAL LETTER UKU
 [0x2C17] = 0x2C47, -- GLAGOLITIC CAPITAL LETTER FRITU
 [0x2C18] = 0x2C48, -- GLAGOLITIC CAPITAL LETTER HERU
 [0x2C19] = 0x2C49, -- GLAGOLITIC CAPITAL LETTER OTU
 [0x2C1A] = 0x2C4A, -- GLAGOLITIC CAPITAL LETTER PE
 [0x2C1B] = 0x2C4B, -- GLAGOLITIC CAPITAL LETTER SHTA
 [0x2C1C] = 0x2C4C, -- GLAGOLITIC CAPITAL LETTER TSI
 [0x2C1D] = 0x2C4D, -- GLAGOLITIC CAPITAL LETTER CHRIVI
 [0x2C1E] = 0x2C4E, -- GLAGOLITIC CAPITAL LETTER SHA
 [0x2C1F] = 0x2C4F, -- GLAGOLITIC CAPITAL LETTER YERU
 [0x2C20] = 0x2C50, -- GLAGOLITIC CAPITAL LETTER YERI
 [0x2C21] = 0x2C51, -- GLAGOLITIC CAPITAL LETTER YATI
 [0x2C22] = 0x2C52, -- GLAGOLITIC CAPITAL LETTER SPIDERY HA
 [0x2C23] = 0x2C53, -- GLAGOLITIC CAPITAL LETTER YU
 [0x2C24] = 0x2C54, -- GLAGOLITIC CAPITAL LETTER SMALL YUS
 [0x2C25] = 0x2C55, -- GLAGOLITIC CAPITAL LETTER SMALL YUS WITH TAIL
 [0x2C26] = 0x2C56, -- GLAGOLITIC CAPITAL LETTER YO
 [0x2C27] = 0x2C57, -- GLAGOLITIC CAPITAL LETTER IOTATED SMALL YUS
 [0x2C28] = 0x2C58, -- GLAGOLITIC CAPITAL LETTER BIG YUS
 [0x2C29] = 0x2C59, -- GLAGOLITIC CAPITAL LETTER IOTATED BIG YUS
 [0x2C2A] = 0x2C5A, -- GLAGOLITIC CAPITAL LETTER FITA
 [0x2C2B] = 0x2C5B, -- GLAGOLITIC CAPITAL LETTER IZHITSA
 [0x2C2C] = 0x2C5C, -- GLAGOLITIC CAPITAL LETTER SHTAPIC
 [0x2C2D] = 0x2C5D, -- GLAGOLITIC CAPITAL LETTER TROKUTASTI A
 [0x2C2E] = 0x2C5E, -- GLAGOLITIC CAPITAL LETTER LATINATE MYSLITE
 [0x2C2F] = {status="disallowed"}, -- NA   <reserved-2C2F>
 [0x2C5F] = {status="disallowed"}, -- NA   <reserved-2C5F>
 [0x2C60] = 0x2C61, -- LATIN CAPITAL LETTER L WITH DOUBLE BAR
 [0x2C61] = {status="valid"}, -- LATIN SMALL LETTER L WITH DOUBLE BAR
 [0x2C62] = 0x026B, -- LATIN CAPITAL LETTER L WITH MIDDLE TILDE
 [0x2C63] = 0x1D7D, -- LATIN CAPITAL LETTER P WITH STROKE
 [0x2C64] = 0x027D, -- LATIN CAPITAL LETTER R WITH TAIL
 [0x2C67] = 0x2C68, -- LATIN CAPITAL LETTER H WITH DESCENDER
 [0x2C68] = {status="valid"}, -- LATIN SMALL LETTER H WITH DESCENDER
 [0x2C69] = 0x2C6A, -- LATIN CAPITAL LETTER K WITH DESCENDER
 [0x2C6A] = {status="valid"}, -- LATIN SMALL LETTER K WITH DESCENDER
 [0x2C6B] = 0x2C6C, -- LATIN CAPITAL LETTER Z WITH DESCENDER
 [0x2C6C] = {status="valid"}, -- LATIN SMALL LETTER Z WITH DESCENDER
 [0x2C6D] = 0x0251, -- LATIN CAPITAL LETTER ALPHA
 [0x2C6E] = 0x0271, -- LATIN CAPITAL LETTER M WITH HOOK
 [0x2C6F] = 0x0250, -- LATIN CAPITAL LETTER TURNED A
 [0x2C70] = 0x0252, -- LATIN CAPITAL LETTER TURNED ALPHA
 [0x2C71] = {status="valid"}, -- LATIN SMALL LETTER V WITH RIGHT HOOK
 [0x2C72] = 0x2C73, -- LATIN CAPITAL LETTER W WITH HOOK
 [0x2C73] = {status="valid"}, -- LATIN SMALL LETTER W WITH HOOK
 [0x2C74] = {status="valid"}, -- LATIN SMALL LETTER V WITH CURL
 [0x2C75] = 0x2C76, -- LATIN CAPITAL LETTER HALF H
 -- LATIN SMALL LETTER A WITH STROKE..LATIN SMALL LETTER T WITH DIAGONAL STROKE
 [0x2C65] = {status="valid"},
 [0x2C66] = {status="valid"},
 [0x2C7C] = 0x006A, -- LATIN SUBSCRIPT SMALL LETTER J
 [0x2C7D] = 0x0076, -- MODIFIER LETTER CAPITAL V
 [0x2C7E] = 0x023F, -- LATIN CAPITAL LETTER S WITH SWASH TAIL
 [0x2C7F] = 0x0240, -- LATIN CAPITAL LETTER Z WITH SWASH TAIL
 [0x2C80] = 0x2C81, -- COPTIC CAPITAL LETTER ALFA
 [0x2C81] = {status="valid"}, -- COPTIC SMALL LETTER ALFA
 [0x2C82] = 0x2C83, -- COPTIC CAPITAL LETTER VIDA
 [0x2C83] = {status="valid"}, -- COPTIC SMALL LETTER VIDA
 [0x2C84] = 0x2C85, -- COPTIC CAPITAL LETTER GAMMA
 [0x2C85] = {status="valid"}, -- COPTIC SMALL LETTER GAMMA
 [0x2C86] = 0x2C87, -- COPTIC CAPITAL LETTER DALDA
 [0x2C87] = {status="valid"}, -- COPTIC SMALL LETTER DALDA
 [0x2C88] = 0x2C89, -- COPTIC CAPITAL LETTER EIE
 [0x2C89] = {status="valid"}, -- COPTIC SMALL LETTER EIE
 [0x2C8A] = 0x2C8B, -- COPTIC CAPITAL LETTER SOU
 [0x2C8B] = {status="valid"}, -- COPTIC SMALL LETTER SOU
 [0x2C8C] = 0x2C8D, -- COPTIC CAPITAL LETTER ZATA
 [0x2C8D] = {status="valid"}, -- COPTIC SMALL LETTER ZATA
 [0x2C8E] = 0x2C8F, -- COPTIC CAPITAL LETTER HATE
 [0x2C8F] = {status="valid"}, -- COPTIC SMALL LETTER HATE
 [0x2C90] = 0x2C91, -- COPTIC CAPITAL LETTER THETHE
 [0x2C91] = {status="valid"}, -- COPTIC SMALL LETTER THETHE
 [0x2C92] = 0x2C93, -- COPTIC CAPITAL LETTER IAUDA
 [0x2C93] = {status="valid"}, -- COPTIC SMALL LETTER IAUDA
 [0x2C94] = 0x2C95, -- COPTIC CAPITAL LETTER KAPA
 [0x2C95] = {status="valid"}, -- COPTIC SMALL LETTER KAPA
 [0x2C96] = 0x2C97, -- COPTIC CAPITAL LETTER LAULA
 [0x2C97] = {status="valid"}, -- COPTIC SMALL LETTER LAULA
 [0x2C98] = 0x2C99, -- COPTIC CAPITAL LETTER MI
 [0x2C99] = {status="valid"}, -- COPTIC SMALL LETTER MI
 [0x2C9A] = 0x2C9B, -- COPTIC CAPITAL LETTER NI
 [0x2C9B] = {status="valid"}, -- COPTIC SMALL LETTER NI
 [0x2C9C] = 0x2C9D, -- COPTIC CAPITAL LETTER KSI
 [0x2C9D] = {status="valid"}, -- COPTIC SMALL LETTER KSI
 [0x2C9E] = 0x2C9F, -- COPTIC CAPITAL LETTER O
 [0x2C9F] = {status="valid"}, -- COPTIC SMALL LETTER O
 [0x2CA0] = 0x2CA1, -- COPTIC CAPITAL LETTER PI
 [0x2CA1] = {status="valid"}, -- COPTIC SMALL LETTER PI
 [0x2CA2] = 0x2CA3, -- COPTIC CAPITAL LETTER RO
 [0x2CA3] = {status="valid"}, -- COPTIC SMALL LETTER RO
 [0x2CA4] = 0x2CA5, -- COPTIC CAPITAL LETTER SIMA
 [0x2CA5] = {status="valid"}, -- COPTIC SMALL LETTER SIMA
 [0x2CA6] = 0x2CA7, -- COPTIC CAPITAL LETTER TAU
 [0x2CA7] = {status="valid"}, -- COPTIC SMALL LETTER TAU
 [0x2CA8] = 0x2CA9, -- COPTIC CAPITAL LETTER UA
 [0x2CA9] = {status="valid"}, -- COPTIC SMALL LETTER UA
 [0x2CAA] = 0x2CAB, -- COPTIC CAPITAL LETTER FI
 [0x2CAB] = {status="valid"}, -- COPTIC SMALL LETTER FI
 [0x2CAC] = 0x2CAD, -- COPTIC CAPITAL LETTER KHI
 [0x2CAD] = {status="valid"}, -- COPTIC SMALL LETTER KHI
 [0x2CAE] = 0x2CAF, -- COPTIC CAPITAL LETTER PSI
 [0x2CAF] = {status="valid"}, -- COPTIC SMALL LETTER PSI
 [0x2CB0] = 0x2CB1, -- COPTIC CAPITAL LETTER OOU
 [0x2CB1] = {status="valid"}, -- COPTIC SMALL LETTER OOU
 [0x2CB2] = 0x2CB3, -- COPTIC CAPITAL LETTER DIALECT-P ALEF
 [0x2CB3] = {status="valid"}, -- COPTIC SMALL LETTER DIALECT-P ALEF
 [0x2CB4] = 0x2CB5, -- COPTIC CAPITAL LETTER OLD COPTIC AIN
 [0x2CB5] = {status="valid"}, -- COPTIC SMALL LETTER OLD COPTIC AIN
 [0x2CB6] = 0x2CB7, -- COPTIC CAPITAL LETTER CRYPTOGRAMMIC EIE
 [0x2CB7] = {status="valid"}, -- COPTIC SMALL LETTER CRYPTOGRAMMIC EIE
 [0x2CB8] = 0x2CB9, -- COPTIC CAPITAL LETTER DIALECT-P KAPA
 [0x2CB9] = {status="valid"}, -- COPTIC SMALL LETTER DIALECT-P KAPA
 [0x2CBA] = 0x2CBB, -- COPTIC CAPITAL LETTER DIALECT-P NI
 [0x2CBB] = {status="valid"}, -- COPTIC SMALL LETTER DIALECT-P NI
 [0x2CBC] = 0x2CBD, -- COPTIC CAPITAL LETTER CRYPTOGRAMMIC NI
 [0x2CBD] = {status="valid"}, -- COPTIC SMALL LETTER CRYPTOGRAMMIC NI
 [0x2CBE] = 0x2CBF, -- COPTIC CAPITAL LETTER OLD COPTIC OOU
 [0x2CBF] = {status="valid"}, -- COPTIC SMALL LETTER OLD COPTIC OOU
 [0x2CC0] = 0x2CC1, -- COPTIC CAPITAL LETTER SAMPI
 [0x2CC1] = {status="valid"}, -- COPTIC SMALL LETTER SAMPI
 [0x2CC2] = 0x2CC3, -- COPTIC CAPITAL LETTER CROSSED SHEI
 [0x2CC3] = {status="valid"}, -- COPTIC SMALL LETTER CROSSED SHEI
 [0x2CC4] = 0x2CC5, -- COPTIC CAPITAL LETTER OLD COPTIC SHEI
 [0x2CC5] = {status="valid"}, -- COPTIC SMALL LETTER OLD COPTIC SHEI
 [0x2CC6] = 0x2CC7, -- COPTIC CAPITAL LETTER OLD COPTIC ESH
 [0x2CC7] = {status="valid"}, -- COPTIC SMALL LETTER OLD COPTIC ESH
 [0x2CC8] = 0x2CC9, -- COPTIC CAPITAL LETTER AKHMIMIC KHEI
 [0x2CC9] = {status="valid"}, -- COPTIC SMALL LETTER AKHMIMIC KHEI
 [0x2CCA] = 0x2CCB, -- COPTIC CAPITAL LETTER DIALECT-P HORI
 [0x2CCB] = {status="valid"}, -- COPTIC SMALL LETTER DIALECT-P HORI
 [0x2CCC] = 0x2CCD, -- COPTIC CAPITAL LETTER OLD COPTIC HORI
 [0x2CCD] = {status="valid"}, -- COPTIC SMALL LETTER OLD COPTIC HORI
 [0x2CCE] = 0x2CCF, -- COPTIC CAPITAL LETTER OLD COPTIC HA
 [0x2CCF] = {status="valid"}, -- COPTIC SMALL LETTER OLD COPTIC HA
 [0x2CD0] = 0x2CD1, -- COPTIC CAPITAL LETTER L-SHAPED HA
 [0x2CD1] = {status="valid"}, -- COPTIC SMALL LETTER L-SHAPED HA
 [0x2CD2] = 0x2CD3, -- COPTIC CAPITAL LETTER OLD COPTIC HEI
 [0x2CD3] = {status="valid"}, -- COPTIC SMALL LETTER OLD COPTIC HEI
 [0x2CD4] = 0x2CD5, -- COPTIC CAPITAL LETTER OLD COPTIC HAT
 [0x2CD5] = {status="valid"}, -- COPTIC SMALL LETTER OLD COPTIC HAT
 [0x2CD6] = 0x2CD7, -- COPTIC CAPITAL LETTER OLD COPTIC GANGIA
 [0x2CD7] = {status="valid"}, -- COPTIC SMALL LETTER OLD COPTIC GANGIA
 [0x2CD8] = 0x2CD9, -- COPTIC CAPITAL LETTER OLD COPTIC DJA
 [0x2CD9] = {status="valid"}, -- COPTIC SMALL LETTER OLD COPTIC DJA
 [0x2CDA] = 0x2CDB, -- COPTIC CAPITAL LETTER OLD COPTIC SHIMA
 [0x2CDB] = {status="valid"}, -- COPTIC SMALL LETTER OLD COPTIC SHIMA
 [0x2CDC] = 0x2CDD, -- COPTIC CAPITAL LETTER OLD NUBIAN SHIMA
 [0x2CDD] = {status="valid"}, -- COPTIC SMALL LETTER OLD NUBIAN SHIMA
 [0x2CDE] = 0x2CDF, -- COPTIC CAPITAL LETTER OLD NUBIAN NGI
 [0x2CDF] = {status="valid"}, -- COPTIC SMALL LETTER OLD NUBIAN NGI
 [0x2CE0] = 0x2CE1, -- COPTIC CAPITAL LETTER OLD NUBIAN NYI
 [0x2CE1] = {status="valid"}, -- COPTIC SMALL LETTER OLD NUBIAN NYI
 [0x2CE2] = 0x2CE3, -- COPTIC CAPITAL LETTER OLD NUBIAN WAU
 -- COPTIC SMALL LETTER OLD NUBIAN WAU..COPTIC SYMBOL KAI
 [0x2CE3] = {status="valid"},
 [0x2CE4] = {status="valid"},
 [0x2CEB] = 0x2CEC, -- COPTIC CAPITAL LETTER CRYPTOGRAMMIC SHEI
 [0x2CEC] = {status="valid"}, -- COPTIC SMALL LETTER CRYPTOGRAMMIC SHEI
 [0x2CED] = 0x2CEE, -- COPTIC CAPITAL LETTER CRYPTOGRAMMIC GANGIA
 [0x2CF2] = 0x2CF3, -- COPTIC CAPITAL LETTER BOHAIRIC KHEI
 [0x2CF3] = {status="valid"}, -- COPTIC SMALL LETTER BOHAIRIC KHEI
 -- COPTIC SMALL LETTER CRYPTOGRAMMIC GANGIA..COPTIC COMBINING SPIRITUS LENIS
 [0x2CEE] = {status="valid"},
 [0x2CEF] = {status="valid"},
 [0x2CF0] = {status="valid"},
 [0x2CF1] = {status="valid"},
 -- NA   <reserved-2CF4>..<reserved-2CF8>
 [0x2CF4] = {status="disallowed"},
 [0x2CF5] = {status="disallowed"},
 [0x2CF6] = {status="disallowed"},
 [0x2CF7] = {status="disallowed"},
 [0x2CF8] = {status="disallowed"},
 [0x2D26] = {status="disallowed"}, -- NA   <reserved-2D26>
 [0x2D27] = {status="valid"}, -- GEORGIAN SMALL LETTER YN
 [0x2D2D] = {status="valid"}, -- GEORGIAN SMALL LETTER AEN
 -- NA   <reserved-2D28>..<reserved-2D2C>
 [0x2D28] = {status="disallowed"},
 [0x2D29] = {status="disallowed"},
 [0x2D2A] = {status="disallowed"},
 [0x2D2B] = {status="disallowed"},
 [0x2D2C] = {status="disallowed"},
 -- NA   <reserved-2D2E>..<reserved-2D2F>
 [0x2D2E] = {status="disallowed"},
 [0x2D2F] = {status="disallowed"},
 [0x2D6F] = 0x2D61, -- TIFINAGH MODIFIER LETTER LABIALIZATION MARK
 [0x2D70] = {status="disallowed"}, -- TIFINAGH SEPARATOR MARK
 [0x2D7F] = {status="valid"}, -- TIFINAGH CONSONANT JOINER
 [0x2DA7] = {status="disallowed"}, -- NA   <reserved-2DA7>
 [0x2DAF] = {status="disallowed"}, -- NA   <reserved-2DAF>
 [0x2DB7] = {status="disallowed"}, -- NA   <reserved-2DB7>
 [0x2DBF] = {status="disallowed"}, -- NA   <reserved-2DBF>
 [0x2DC7] = {status="disallowed"}, -- NA   <reserved-2DC7>
 [0x2DCF] = {status="disallowed"}, -- NA   <reserved-2DCF>
 [0x2DD7] = {status="disallowed"}, -- NA   <reserved-2DD7>
 [0x2DDF] = {status="disallowed"}, -- NA   <reserved-2DDF>
 [0x2E2F] = {status="valid"}, -- VERTICAL TILDE
 [0x2E30] = {status="disallowed"}, -- RING POINT
 [0x2E31] = {status="disallowed"}, -- WORD SEPARATOR MIDDLE DOT
 [0x2E9A] = {status="disallowed"}, -- NA   <reserved-2E9A>
 [0x2E9F] = 0x6BCD, -- CJK RADICAL MOTHER
 -- CJK RADICAL CHOKE..CJK RADICAL DEATH
 [0x2E9B] = {status="disallowed"},
 [0x2E9C] = {status="disallowed"},
 [0x2E9D] = {status="disallowed"},
 [0x2E9E] = {status="disallowed"},
 [0x2EF3] = 0x9F9F, -- CJK RADICAL C-SIMPLIFIED TURTLE
 [0x2F00] = 0x4E00, -- KANGXI RADICAL ONE
 [0x2F01] = 0x4E28, -- KANGXI RADICAL LINE
 [0x2F02] = 0x4E36, -- KANGXI RADICAL DOT
 [0x2F03] = 0x4E3F, -- KANGXI RADICAL SLASH
 [0x2F04] = 0x4E59, -- KANGXI RADICAL SECOND
 [0x2F05] = 0x4E85, -- KANGXI RADICAL HOOK
 [0x2F06] = 0x4E8C, -- KANGXI RADICAL TWO
 [0x2F07] = 0x4EA0, -- KANGXI RADICAL LID
 [0x2F08] = 0x4EBA, -- KANGXI RADICAL MAN
 [0x2F09] = 0x513F, -- KANGXI RADICAL LEGS
 [0x2F0A] = 0x5165, -- KANGXI RADICAL ENTER
 [0x2F0B] = 0x516B, -- KANGXI RADICAL EIGHT
 [0x2F0C] = 0x5182, -- KANGXI RADICAL DOWN BOX
 [0x2F0D] = 0x5196, -- KANGXI RADICAL COVER
 [0x2F0E] = 0x51AB, -- KANGXI RADICAL ICE
 [0x2F0F] = 0x51E0, -- KANGXI RADICAL TABLE
 [0x2F10] = 0x51F5, -- KANGXI RADICAL OPEN BOX
 [0x2F11] = 0x5200, -- KANGXI RADICAL KNIFE
 [0x2F12] = 0x529B, -- KANGXI RADICAL POWER
 [0x2F13] = 0x52F9, -- KANGXI RADICAL WRAP
 [0x2F14] = 0x5315, -- KANGXI RADICAL SPOON
 [0x2F15] = 0x531A, -- KANGXI RADICAL RIGHT OPEN BOX
 [0x2F16] = 0x5338, -- KANGXI RADICAL HIDING ENCLOSURE
 [0x2F17] = 0x5341, -- KANGXI RADICAL TEN
 [0x2F18] = 0x535C, -- KANGXI RADICAL DIVINATION
 [0x2F19] = 0x5369, -- KANGXI RADICAL SEAL
 [0x2F1A] = 0x5382, -- KANGXI RADICAL CLIFF
 [0x2F1B] = 0x53B6, -- KANGXI RADICAL PRIVATE
 [0x2F1C] = 0x53C8, -- KANGXI RADICAL AGAIN
 [0x2F1D] = 0x53E3, -- KANGXI RADICAL MOUTH
 [0x2F1E] = 0x56D7, -- KANGXI RADICAL ENCLOSURE
 [0x2F1F] = 0x571F, -- KANGXI RADICAL EARTH
 [0x2F20] = 0x58EB, -- KANGXI RADICAL SCHOLAR
 [0x2F21] = 0x5902, -- KANGXI RADICAL GO
 [0x2F22] = 0x590A, -- KANGXI RADICAL GO SLOWLY
 [0x2F23] = 0x5915, -- KANGXI RADICAL EVENING
 [0x2F24] = 0x5927, -- KANGXI RADICAL BIG
 [0x2F25] = 0x5973, -- KANGXI RADICAL WOMAN
 [0x2F26] = 0x5B50, -- KANGXI RADICAL CHILD
 [0x2F27] = 0x5B80, -- KANGXI RADICAL ROOF
 [0x2F28] = 0x5BF8, -- KANGXI RADICAL INCH
 [0x2F29] = 0x5C0F, -- KANGXI RADICAL SMALL
 [0x2F2A] = 0x5C22, -- KANGXI RADICAL LAME
 [0x2F2B] = 0x5C38, -- KANGXI RADICAL CORPSE
 [0x2F2C] = 0x5C6E, -- KANGXI RADICAL SPROUT
 [0x2F2D] = 0x5C71, -- KANGXI RADICAL MOUNTAIN
 [0x2F2E] = 0x5DDB, -- KANGXI RADICAL RIVER
 [0x2F2F] = 0x5DE5, -- KANGXI RADICAL WORK
 [0x2F30] = 0x5DF1, -- KANGXI RADICAL ONESELF
 [0x2F31] = 0x5DFE, -- KANGXI RADICAL TURBAN
 [0x2F32] = 0x5E72, -- KANGXI RADICAL DRY
 [0x2F33] = 0x5E7A, -- KANGXI RADICAL SHORT THREAD
 [0x2F34] = 0x5E7F, -- KANGXI RADICAL DOTTED CLIFF
 [0x2F35] = 0x5EF4, -- KANGXI RADICAL LONG STRIDE
 [0x2F36] = 0x5EFE, -- KANGXI RADICAL TWO HANDS
 [0x2F37] = 0x5F0B, -- KANGXI RADICAL SHOOT
 [0x2F38] = 0x5F13, -- KANGXI RADICAL BOW
 [0x2F39] = 0x5F50, -- KANGXI RADICAL SNOUT
 [0x2F3A] = 0x5F61, -- KANGXI RADICAL BRISTLE
 [0x2F3B] = 0x5F73, -- KANGXI RADICAL STEP
 [0x2F3C] = 0x5FC3, -- KANGXI RADICAL HEART
 [0x2F3D] = 0x6208, -- KANGXI RADICAL HALBERD
 [0x2F3E] = 0x6236, -- KANGXI RADICAL DOOR
 [0x2F3F] = 0x624B, -- KANGXI RADICAL HAND
 [0x2F40] = 0x652F, -- KANGXI RADICAL BRANCH
 [0x2F41] = 0x6534, -- KANGXI RADICAL RAP
 [0x2F42] = 0x6587, -- KANGXI RADICAL SCRIPT
 [0x2F43] = 0x6597, -- KANGXI RADICAL DIPPER
 [0x2F44] = 0x65A4, -- KANGXI RADICAL AXE
 [0x2F45] = 0x65B9, -- KANGXI RADICAL SQUARE
 [0x2F46] = 0x65E0, -- KANGXI RADICAL NOT
 [0x2F47] = 0x65E5, -- KANGXI RADICAL SUN
 [0x2F48] = 0x66F0, -- KANGXI RADICAL SAY
 [0x2F49] = 0x6708, -- KANGXI RADICAL MOON
 [0x2F4A] = 0x6728, -- KANGXI RADICAL TREE
 [0x2F4B] = 0x6B20, -- KANGXI RADICAL LACK
 [0x2F4C] = 0x6B62, -- KANGXI RADICAL STOP
 [0x2F4D] = 0x6B79, -- KANGXI RADICAL DEATH
 [0x2F4E] = 0x6BB3, -- KANGXI RADICAL WEAPON
 [0x2F4F] = 0x6BCB, -- KANGXI RADICAL DO NOT
 [0x2F50] = 0x6BD4, -- KANGXI RADICAL COMPARE
 [0x2F51] = 0x6BDB, -- KANGXI RADICAL FUR
 [0x2F52] = 0x6C0F, -- KANGXI RADICAL CLAN
 [0x2F53] = 0x6C14, -- KANGXI RADICAL STEAM
 [0x2F54] = 0x6C34, -- KANGXI RADICAL WATER
 [0x2F55] = 0x706B, -- KANGXI RADICAL FIRE
 [0x2F56] = 0x722A, -- KANGXI RADICAL CLAW
 [0x2F57] = 0x7236, -- KANGXI RADICAL FATHER
 [0x2F58] = 0x723B, -- KANGXI RADICAL DOUBLE X
 [0x2F59] = 0x723F, -- KANGXI RADICAL HALF TREE TRUNK
 [0x2F5A] = 0x7247, -- KANGXI RADICAL SLICE
 [0x2F5B] = 0x7259, -- KANGXI RADICAL FANG
 [0x2F5C] = 0x725B, -- KANGXI RADICAL COW
 [0x2F5D] = 0x72AC, -- KANGXI RADICAL DOG
 [0x2F5E] = 0x7384, -- KANGXI RADICAL PROFOUND
 [0x2F5F] = 0x7389, -- KANGXI RADICAL JADE
 [0x2F60] = 0x74DC, -- KANGXI RADICAL MELON
 [0x2F61] = 0x74E6, -- KANGXI RADICAL TILE
 [0x2F62] = 0x7518, -- KANGXI RADICAL SWEET
 [0x2F63] = 0x751F, -- KANGXI RADICAL LIFE
 [0x2F64] = 0x7528, -- KANGXI RADICAL USE
 [0x2F65] = 0x7530, -- KANGXI RADICAL FIELD
 [0x2F66] = 0x758B, -- KANGXI RADICAL BOLT OF CLOTH
 [0x2F67] = 0x7592, -- KANGXI RADICAL SICKNESS
 [0x2F68] = 0x7676, -- KANGXI RADICAL DOTTED TENT
 [0x2F69] = 0x767D, -- KANGXI RADICAL WHITE
 [0x2F6A] = 0x76AE, -- KANGXI RADICAL SKIN
 [0x2F6B] = 0x76BF, -- KANGXI RADICAL DISH
 [0x2F6C] = 0x76EE, -- KANGXI RADICAL EYE
 [0x2F6D] = 0x77DB, -- KANGXI RADICAL SPEAR
 [0x2F6E] = 0x77E2, -- KANGXI RADICAL ARROW
 [0x2F6F] = 0x77F3, -- KANGXI RADICAL STONE
 [0x2F70] = 0x793A, -- KANGXI RADICAL SPIRIT
 [0x2F71] = 0x79B8, -- KANGXI RADICAL TRACK
 [0x2F72] = 0x79BE, -- KANGXI RADICAL GRAIN
 [0x2F73] = 0x7A74, -- KANGXI RADICAL CAVE
 [0x2F74] = 0x7ACB, -- KANGXI RADICAL STAND
 [0x2F75] = 0x7AF9, -- KANGXI RADICAL BAMBOO
 [0x2F76] = 0x7C73, -- KANGXI RADICAL RICE
 [0x2F77] = 0x7CF8, -- KANGXI RADICAL SILK
 [0x2F78] = 0x7F36, -- KANGXI RADICAL JAR
 [0x2F79] = 0x7F51, -- KANGXI RADICAL NET
 [0x2F7A] = 0x7F8A, -- KANGXI RADICAL SHEEP
 [0x2F7B] = 0x7FBD, -- KANGXI RADICAL FEATHER
 [0x2F7C] = 0x8001, -- KANGXI RADICAL OLD
 [0x2F7D] = 0x800C, -- KANGXI RADICAL AND
 [0x2F7E] = 0x8012, -- KANGXI RADICAL PLOW
 [0x2F7F] = 0x8033, -- KANGXI RADICAL EAR
 [0x2F80] = 0x807F, -- KANGXI RADICAL BRUSH
 [0x2F81] = 0x8089, -- KANGXI RADICAL MEAT
 [0x2F82] = 0x81E3, -- KANGXI RADICAL MINISTER
 [0x2F83] = 0x81EA, -- KANGXI RADICAL SELF
 [0x2F84] = 0x81F3, -- KANGXI RADICAL ARRIVE
 [0x2F85] = 0x81FC, -- KANGXI RADICAL MORTAR
 [0x2F86] = 0x820C, -- KANGXI RADICAL TONGUE
 [0x2F87] = 0x821B, -- KANGXI RADICAL OPPOSE
 [0x2F88] = 0x821F, -- KANGXI RADICAL BOAT
 [0x2F89] = 0x826E, -- KANGXI RADICAL STOPPING
 [0x2F8A] = 0x8272, -- KANGXI RADICAL COLOR
 [0x2F8B] = 0x8278, -- KANGXI RADICAL GRASS
 [0x2F8C] = 0x864D, -- KANGXI RADICAL TIGER
 [0x2F8D] = 0x866B, -- KANGXI RADICAL INSECT
 [0x2F8E] = 0x8840, -- KANGXI RADICAL BLOOD
 [0x2F8F] = 0x884C, -- KANGXI RADICAL WALK ENCLOSURE
 [0x2F90] = 0x8863, -- KANGXI RADICAL CLOTHES
 [0x2F91] = 0x897E, -- KANGXI RADICAL WEST
 [0x2F92] = 0x898B, -- KANGXI RADICAL SEE
 [0x2F93] = 0x89D2, -- KANGXI RADICAL HORN
 [0x2F94] = 0x8A00, -- KANGXI RADICAL SPEECH
 [0x2F95] = 0x8C37, -- KANGXI RADICAL VALLEY
 [0x2F96] = 0x8C46, -- KANGXI RADICAL BEAN
 [0x2F97] = 0x8C55, -- KANGXI RADICAL PIG
 [0x2F98] = 0x8C78, -- KANGXI RADICAL BADGER
 [0x2F99] = 0x8C9D, -- KANGXI RADICAL SHELL
 [0x2F9A] = 0x8D64, -- KANGXI RADICAL RED
 [0x2F9B] = 0x8D70, -- KANGXI RADICAL RUN
 [0x2F9C] = 0x8DB3, -- KANGXI RADICAL FOOT
 [0x2F9D] = 0x8EAB, -- KANGXI RADICAL BODY
 [0x2F9E] = 0x8ECA, -- KANGXI RADICAL CART
 [0x2F9F] = 0x8F9B, -- KANGXI RADICAL BITTER
 [0x2FA0] = 0x8FB0, -- KANGXI RADICAL MORNING
 [0x2FA1] = 0x8FB5, -- KANGXI RADICAL WALK
 [0x2FA2] = 0x9091, -- KANGXI RADICAL CITY
 [0x2FA3] = 0x9149, -- KANGXI RADICAL WINE
 [0x2FA4] = 0x91C6, -- KANGXI RADICAL DISTINGUISH
 [0x2FA5] = 0x91CC, -- KANGXI RADICAL VILLAGE
 [0x2FA6] = 0x91D1, -- KANGXI RADICAL GOLD
 [0x2FA7] = 0x9577, -- KANGXI RADICAL LONG
 [0x2FA8] = 0x9580, -- KANGXI RADICAL GATE
 [0x2FA9] = 0x961C, -- KANGXI RADICAL MOUND
 [0x2FAA] = 0x96B6, -- KANGXI RADICAL SLAVE
 [0x2FAB] = 0x96B9, -- KANGXI RADICAL SHORT TAILED BIRD
 [0x2FAC] = 0x96E8, -- KANGXI RADICAL RAIN
 [0x2FAD] = 0x9751, -- KANGXI RADICAL BLUE
 [0x2FAE] = 0x975E, -- KANGXI RADICAL WRONG
 [0x2FAF] = 0x9762, -- KANGXI RADICAL FACE
 [0x2FB0] = 0x9769, -- KANGXI RADICAL LEATHER
 [0x2FB1] = 0x97CB, -- KANGXI RADICAL TANNED LEATHER
 [0x2FB2] = 0x97ED, -- KANGXI RADICAL LEEK
 [0x2FB3] = 0x97F3, -- KANGXI RADICAL SOUND
 [0x2FB4] = 0x9801, -- KANGXI RADICAL LEAF
 [0x2FB5] = 0x98A8, -- KANGXI RADICAL WIND
 [0x2FB6] = 0x98DB, -- KANGXI RADICAL FLY
 [0x2FB7] = 0x98DF, -- KANGXI RADICAL EAT
 [0x2FB8] = 0x9996, -- KANGXI RADICAL HEAD
 [0x2FB9] = 0x9999, -- KANGXI RADICAL FRAGRANT
 [0x2FBA] = 0x99AC, -- KANGXI RADICAL HORSE
 [0x2FBB] = 0x9AA8, -- KANGXI RADICAL BONE
 [0x2FBC] = 0x9AD8, -- KANGXI RADICAL TALL
 [0x2FBD] = 0x9ADF, -- KANGXI RADICAL HAIR
 [0x2FBE] = 0x9B25, -- KANGXI RADICAL FIGHT
 [0x2FBF] = 0x9B2F, -- KANGXI RADICAL SACRIFICIAL WINE
 [0x2FC0] = 0x9B32, -- KANGXI RADICAL CAULDRON
 [0x2FC1] = 0x9B3C, -- KANGXI RADICAL GHOST
 [0x2FC2] = 0x9B5A, -- KANGXI RADICAL FISH
 [0x2FC3] = 0x9CE5, -- KANGXI RADICAL BIRD
 [0x2FC4] = 0x9E75, -- KANGXI RADICAL SALT
 [0x2FC5] = 0x9E7F, -- KANGXI RADICAL DEER
 [0x2FC6] = 0x9EA5, -- KANGXI RADICAL WHEAT
 [0x2FC7] = 0x9EBB, -- KANGXI RADICAL HEMP
 [0x2FC8] = 0x9EC3, -- KANGXI RADICAL YELLOW
 [0x2FC9] = 0x9ECD, -- KANGXI RADICAL MILLET
 [0x2FCA] = 0x9ED1, -- KANGXI RADICAL BLACK
 [0x2FCB] = 0x9EF9, -- KANGXI RADICAL EMBROIDERY
 [0x2FCC] = 0x9EFD, -- KANGXI RADICAL FROG
 [0x2FCD] = 0x9F0E, -- KANGXI RADICAL TRIPOD
 [0x2FCE] = 0x9F13, -- KANGXI RADICAL DRUM
 [0x2FCF] = 0x9F20, -- KANGXI RADICAL RAT
 [0x2FD0] = 0x9F3B, -- KANGXI RADICAL NOSE
 [0x2FD1] = 0x9F4A, -- KANGXI RADICAL EVEN
 [0x2FD2] = 0x9F52, -- KANGXI RADICAL TOOTH
 [0x2FD3] = 0x9F8D, -- KANGXI RADICAL DRAGON
 [0x2FD4] = 0x9F9C, -- KANGXI RADICAL TURTLE
 [0x2FD5] = 0x9FA0, -- KANGXI RADICAL FLUTE
 [0x3000] = {status="disallowed"}, -- IDEOGRAPHIC SPACE
 [0x3001] = {status="disallowed"}, -- IDEOGRAPHIC COMMA
 [0x3002] = 0x002E, -- IDEOGRAPHIC FULL STOP
 -- DITTO MARK..JAPANESE INDUSTRIAL STANDARD SYMBOL
 [0x3003] = {status="disallowed"},
 [0x3004] = {status="disallowed"},
 -- IDEOGRAPHIC ITERATION MARK..IDEOGRAPHIC NUMBER ZERO
 [0x3005] = {status="valid"},
 [0x3006] = {status="valid"},
 [0x3007] = {status="valid"},
 -- IDEOGRAPHIC LEVEL TONE MARK..IDEOGRAPHIC ENTERING TONE MARK
 [0x302A] = {status="valid"},
 [0x302B] = {status="valid"},
 [0x302C] = {status="valid"},
 [0x302D] = {status="valid"},
 [0x3036] = 0x3012, -- CIRCLED POSTAL MARK
 [0x3037] = {status="disallowed"}, -- IDEOGRAPHIC TELEGRAPH LINE FEED SEPARATOR SYMBOL
 [0x3038] = 0x5341, -- HANGZHOU NUMERAL TEN
 [0x3039] = 0x5344, -- HANGZHOU NUMERAL TWENTY
 [0x303A] = 0x5345, -- HANGZHOU NUMERAL THIRTY
 [0x303B] = {status="disallowed"}, -- VERTICAL IDEOGRAPHIC ITERATION MARK
 [0x303C] = {status="valid"}, -- MASU MARK
 [0x303D] = {status="disallowed"}, -- PART ALTERNATION MARK
 [0x303E] = {status="disallowed"}, -- IDEOGRAPHIC VARIATION INDICATOR
 [0x303F] = {status="disallowed"}, -- IDEOGRAPHIC HALF FILL SPACE
 [0x3040] = {status="disallowed"}, -- NA   <reserved-3040>
 -- NA   <reserved-3097>..<reserved-3098>
 [0x3097] = {status="disallowed"},
 [0x3098] = {status="disallowed"},
 [0x309B] = {status="disallowed"}, -- KATAKANA-HIRAGANA VOICED SOUND MARK
 [0x309C] = {status="disallowed"}, -- KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK
 -- COMBINING KATAKANA-HIRAGANA VOICED SOUND MARK..COMBINING KATAKANA-HIRAGANA SEMI-VOICED SOUND MARK
 [0x3099] = {status="valid"},
 [0x309A] = {status="valid"},
 [0x309F] = {0x3088, 0x308A}, -- HIRAGANA DIGRAPH YORI
 [0x30A0] = {status="disallowed"}, -- KATAKANA-HIRAGANA DOUBLE HYPHEN
 -- HIRAGANA ITERATION MARK..HIRAGANA VOICED ITERATION MARK
 [0x309D] = {status="valid"},
 [0x309E] = {status="valid"},
 [0x30FF] = {0x30B3, 0x30C8}, -- KATAKANA DIGRAPH KOTO
 -- NA   <reserved-3100>..<reserved-3104>
 [0x3100] = {status="disallowed"},
 [0x3101] = {status="disallowed"},
 [0x3102] = {status="disallowed"},
 [0x3103] = {status="disallowed"},
 [0x3104] = {status="disallowed"},
 [0x312D] = {status="valid"}, -- BOPOMOFO LETTER IH
 [0x312E] = {status="valid"}, -- BOPOMOFO LETTER O WITH DOT ABOVE
 [0x3131] = 0x1100, -- HANGUL LETTER KIYEOK
 [0x3132] = 0x1101, -- HANGUL LETTER SSANGKIYEOK
 [0x3133] = 0x11AA, -- HANGUL LETTER KIYEOK-SIOS
 [0x3134] = 0x1102, -- HANGUL LETTER NIEUN
 [0x3135] = 0x11AC, -- HANGUL LETTER NIEUN-CIEUC
 [0x3136] = 0x11AD, -- HANGUL LETTER NIEUN-HIEUH
 [0x3137] = 0x1103, -- HANGUL LETTER TIKEUT
 [0x3138] = 0x1104, -- HANGUL LETTER SSANGTIKEUT
 [0x3139] = 0x1105, -- HANGUL LETTER RIEUL
 [0x313A] = 0x11B0, -- HANGUL LETTER RIEUL-KIYEOK
 [0x313B] = 0x11B1, -- HANGUL LETTER RIEUL-MIEUM
 [0x313C] = 0x11B2, -- HANGUL LETTER RIEUL-PIEUP
 [0x313D] = 0x11B3, -- HANGUL LETTER RIEUL-SIOS
 [0x313E] = 0x11B4, -- HANGUL LETTER RIEUL-THIEUTH
 [0x313F] = 0x11B5, -- HANGUL LETTER RIEUL-PHIEUPH
 [0x3140] = 0x111A, -- HANGUL LETTER RIEUL-HIEUH
 [0x3141] = 0x1106, -- HANGUL LETTER MIEUM
 [0x3142] = 0x1107, -- HANGUL LETTER PIEUP
 [0x3143] = 0x1108, -- HANGUL LETTER SSANGPIEUP
 [0x3144] = 0x1121, -- HANGUL LETTER PIEUP-SIOS
 [0x3145] = 0x1109, -- HANGUL LETTER SIOS
 [0x3146] = 0x110A, -- HANGUL LETTER SSANGSIOS
 [0x3147] = 0x110B, -- HANGUL LETTER IEUNG
 [0x3148] = 0x110C, -- HANGUL LETTER CIEUC
 [0x3149] = 0x110D, -- HANGUL LETTER SSANGCIEUC
 [0x314A] = 0x110E, -- HANGUL LETTER CHIEUCH
 [0x314B] = 0x110F, -- HANGUL LETTER KHIEUKH
 [0x314C] = 0x1110, -- HANGUL LETTER THIEUTH
 [0x314D] = 0x1111, -- HANGUL LETTER PHIEUPH
 [0x314E] = 0x1112, -- HANGUL LETTER HIEUH
 [0x314F] = 0x1161, -- HANGUL LETTER A
 [0x3150] = 0x1162, -- HANGUL LETTER AE
 [0x3151] = 0x1163, -- HANGUL LETTER YA
 [0x3152] = 0x1164, -- HANGUL LETTER YAE
 [0x3153] = 0x1165, -- HANGUL LETTER EO
 [0x3154] = 0x1166, -- HANGUL LETTER E
 [0x3155] = 0x1167, -- HANGUL LETTER YEO
 [0x3156] = 0x1168, -- HANGUL LETTER YE
 [0x3157] = 0x1169, -- HANGUL LETTER O
 [0x3158] = 0x116A, -- HANGUL LETTER WA
 [0x3159] = 0x116B, -- HANGUL LETTER WAE
 [0x315A] = 0x116C, -- HANGUL LETTER OE
 [0x315B] = 0x116D, -- HANGUL LETTER YO
 [0x315C] = 0x116E, -- HANGUL LETTER U
 [0x315D] = 0x116F, -- HANGUL LETTER WEO
 [0x315E] = 0x1170, -- HANGUL LETTER WE
 [0x315F] = 0x1171, -- HANGUL LETTER WI
 [0x3160] = 0x1172, -- HANGUL LETTER YU
 [0x3161] = 0x1173, -- HANGUL LETTER EU
 [0x3162] = 0x1174, -- HANGUL LETTER YI
 [0x3163] = 0x1175, -- HANGUL LETTER I
 [0x3164] = {status="disallowed"}, -- HANGUL FILLER
 [0x3165] = 0x1114, -- HANGUL LETTER SSANGNIEUN
 [0x3166] = 0x1115, -- HANGUL LETTER NIEUN-TIKEUT
 [0x3167] = 0x11C7, -- HANGUL LETTER NIEUN-SIOS
 [0x3168] = 0x11C8, -- HANGUL LETTER NIEUN-PANSIOS
 [0x3169] = 0x11CC, -- HANGUL LETTER RIEUL-KIYEOK-SIOS
 [0x316A] = 0x11CE, -- HANGUL LETTER RIEUL-TIKEUT
 [0x316B] = 0x11D3, -- HANGUL LETTER RIEUL-PIEUP-SIOS
 [0x316C] = 0x11D7, -- HANGUL LETTER RIEUL-PANSIOS
 [0x316D] = 0x11D9, -- HANGUL LETTER RIEUL-YEORINHIEUH
 [0x316E] = 0x111C, -- HANGUL LETTER MIEUM-PIEUP
 [0x316F] = 0x11DD, -- HANGUL LETTER MIEUM-SIOS
 [0x3170] = 0x11DF, -- HANGUL LETTER MIEUM-PANSIOS
 [0x3171] = 0x111D, -- HANGUL LETTER KAPYEOUNMIEUM
 [0x3172] = 0x111E, -- HANGUL LETTER PIEUP-KIYEOK
 [0x3173] = 0x1120, -- HANGUL LETTER PIEUP-TIKEUT
 [0x3174] = 0x1122, -- HANGUL LETTER PIEUP-SIOS-KIYEOK
 [0x3175] = 0x1123, -- HANGUL LETTER PIEUP-SIOS-TIKEUT
 [0x3176] = 0x1127, -- HANGUL LETTER PIEUP-CIEUC
 [0x3177] = 0x1129, -- HANGUL LETTER PIEUP-THIEUTH
 [0x3178] = 0x112B, -- HANGUL LETTER KAPYEOUNPIEUP
 [0x3179] = 0x112C, -- HANGUL LETTER KAPYEOUNSSANGPIEUP
 [0x317A] = 0x112D, -- HANGUL LETTER SIOS-KIYEOK
 [0x317B] = 0x112E, -- HANGUL LETTER SIOS-NIEUN
 [0x317C] = 0x112F, -- HANGUL LETTER SIOS-TIKEUT
 [0x317D] = 0x1132, -- HANGUL LETTER SIOS-PIEUP
 [0x317E] = 0x1136, -- HANGUL LETTER SIOS-CIEUC
 [0x317F] = 0x1140, -- HANGUL LETTER PANSIOS
 [0x3180] = 0x1147, -- HANGUL LETTER SSANGIEUNG
 [0x3181] = 0x114C, -- HANGUL LETTER YESIEUNG
 [0x3182] = 0x11F1, -- HANGUL LETTER YESIEUNG-SIOS
 [0x3183] = 0x11F2, -- HANGUL LETTER YESIEUNG-PANSIOS
 [0x3184] = 0x1157, -- HANGUL LETTER KAPYEOUNPHIEUPH
 [0x3185] = 0x1158, -- HANGUL LETTER SSANGHIEUH
 [0x3186] = 0x1159, -- HANGUL LETTER YEORINHIEUH
 [0x3187] = 0x1184, -- HANGUL LETTER YO-YA
 [0x3188] = 0x1185, -- HANGUL LETTER YO-YAE
 [0x3189] = 0x1188, -- HANGUL LETTER YO-I
 [0x318A] = 0x1191, -- HANGUL LETTER YU-YEO
 [0x318B] = 0x1192, -- HANGUL LETTER YU-YE
 [0x318C] = 0x1194, -- HANGUL LETTER YU-I
 [0x318D] = 0x119E, -- HANGUL LETTER ARAEA
 [0x318E] = 0x11A1, -- HANGUL LETTER ARAEAE
 [0x318F] = {status="disallowed"}, -- NA   <reserved-318F>
 -- NA   <reserved-312F>..<reserved-3130>
 [0x312F] = {status="disallowed"},
 [0x3130] = {status="disallowed"},
 [0x3192] = 0x4E00, -- IDEOGRAPHIC ANNOTATION ONE MARK
 [0x3193] = 0x4E8C, -- IDEOGRAPHIC ANNOTATION TWO MARK
 [0x3194] = 0x4E09, -- IDEOGRAPHIC ANNOTATION THREE MARK
 [0x3195] = 0x56DB, -- IDEOGRAPHIC ANNOTATION FOUR MARK
 [0x3196] = 0x4E0A, -- IDEOGRAPHIC ANNOTATION TOP MARK
 [0x3197] = 0x4E2D, -- IDEOGRAPHIC ANNOTATION MIDDLE MARK
 [0x3198] = 0x4E0B, -- IDEOGRAPHIC ANNOTATION BOTTOM MARK
 [0x3199] = 0x7532, -- IDEOGRAPHIC ANNOTATION FIRST MARK
 [0x319A] = 0x4E59, -- IDEOGRAPHIC ANNOTATION SECOND MARK
 [0x319B] = 0x4E19, -- IDEOGRAPHIC ANNOTATION THIRD MARK
 [0x319C] = 0x4E01, -- IDEOGRAPHIC ANNOTATION FOURTH MARK
 [0x319D] = 0x5929, -- IDEOGRAPHIC ANNOTATION HEAVEN MARK
 [0x319E] = 0x5730, -- IDEOGRAPHIC ANNOTATION EARTH MARK
 [0x319F] = 0x4EBA, -- IDEOGRAPHIC ANNOTATION MAN MARK
 -- IDEOGRAPHIC ANNOTATION LINKING MARK..IDEOGRAPHIC ANNOTATION REVERSE MARK
 [0x3190] = {status="disallowed"},
 [0x3191] = {status="disallowed"},
 -- NA   <reserved-31BB>..<reserved-31BF>
 [0x31BB] = {status="disallowed"},
 [0x31BC] = {status="disallowed"},
 [0x31BD] = {status="disallowed"},
 [0x31BE] = {status="disallowed"},
 [0x31BF] = {status="disallowed"},
 [0x3200] = {status="disallowed"}, -- PARENTHESIZED HANGUL KIYEOK
 [0x3201] = {status="disallowed"}, -- PARENTHESIZED HANGUL NIEUN
 [0x3202] = {status="disallowed"}, -- PARENTHESIZED HANGUL TIKEUT
 [0x3203] = {status="disallowed"}, -- PARENTHESIZED HANGUL RIEUL
 [0x3204] = {status="disallowed"}, -- PARENTHESIZED HANGUL MIEUM
 [0x3205] = {status="disallowed"}, -- PARENTHESIZED HANGUL PIEUP
 [0x3206] = {status="disallowed"}, -- PARENTHESIZED HANGUL SIOS
 [0x3207] = {status="disallowed"}, -- PARENTHESIZED HANGUL IEUNG
 [0x3208] = {status="disallowed"}, -- PARENTHESIZED HANGUL CIEUC
 [0x3209] = {status="disallowed"}, -- PARENTHESIZED HANGUL CHIEUCH
 [0x320A] = {status="disallowed"}, -- PARENTHESIZED HANGUL KHIEUKH
 [0x320B] = {status="disallowed"}, -- PARENTHESIZED HANGUL THIEUTH
 [0x320C] = {status="disallowed"}, -- PARENTHESIZED HANGUL PHIEUPH
 [0x320D] = {status="disallowed"}, -- PARENTHESIZED HANGUL HIEUH
 [0x320E] = {status="disallowed"}, -- PARENTHESIZED HANGUL KIYEOK A
 [0x320F] = {status="disallowed"}, -- PARENTHESIZED HANGUL NIEUN A
 [0x3210] = {status="disallowed"}, -- PARENTHESIZED HANGUL TIKEUT A
 [0x3211] = {status="disallowed"}, -- PARENTHESIZED HANGUL RIEUL A
 [0x3212] = {status="disallowed"}, -- PARENTHESIZED HANGUL MIEUM A
 [0x3213] = {status="disallowed"}, -- PARENTHESIZED HANGUL PIEUP A
 [0x3214] = {status="disallowed"}, -- PARENTHESIZED HANGUL SIOS A
 [0x3215] = {status="disallowed"}, -- PARENTHESIZED HANGUL IEUNG A
 [0x3216] = {status="disallowed"}, -- PARENTHESIZED HANGUL CIEUC A
 [0x3217] = {status="disallowed"}, -- PARENTHESIZED HANGUL CHIEUCH A
 [0x3218] = {status="disallowed"}, -- PARENTHESIZED HANGUL KHIEUKH A
 [0x3219] = {status="disallowed"}, -- PARENTHESIZED HANGUL THIEUTH A
 [0x321A] = {status="disallowed"}, -- PARENTHESIZED HANGUL PHIEUPH A
 [0x321B] = {status="disallowed"}, -- PARENTHESIZED HANGUL HIEUH A
 [0x321C] = {status="disallowed"}, -- PARENTHESIZED HANGUL CIEUC U
 [0x321D] = {status="disallowed"}, -- PARENTHESIZED KOREAN CHARACTER OJEON
 [0x321E] = {status="disallowed"}, -- PARENTHESIZED KOREAN CHARACTER O HU
 [0x321F] = {status="disallowed"}, -- NA   <reserved-321F>
 [0x3220] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH ONE
 [0x3221] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH TWO
 [0x3222] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH THREE
 [0x3223] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH FOUR
 [0x3224] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH FIVE
 [0x3225] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH SIX
 [0x3226] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH SEVEN
 [0x3227] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH EIGHT
 [0x3228] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH NINE
 [0x3229] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH TEN
 [0x322A] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH MOON
 [0x322B] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH FIRE
 [0x322C] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH WATER
 [0x322D] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH WOOD
 [0x322E] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH METAL
 [0x322F] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH EARTH
 [0x3230] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH SUN
 [0x3231] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH STOCK
 [0x3232] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH HAVE
 [0x3233] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH SOCIETY
 [0x3234] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH NAME
 [0x3235] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH SPECIAL
 [0x3236] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH FINANCIAL
 [0x3237] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH CONGRATULATION
 [0x3238] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH LABOR
 [0x3239] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH REPRESENT
 [0x323A] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH CALL
 [0x323B] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH STUDY
 [0x323C] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH SUPERVISE
 [0x323D] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH ENTERPRISE
 [0x323E] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH RESOURCE
 [0x323F] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH ALLIANCE
 [0x3240] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH FESTIVAL
 [0x3241] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH REST
 [0x3242] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH SELF
 [0x3243] = {status="disallowed"}, -- PARENTHESIZED IDEOGRAPH REACH
 [0x3244] = 0x554F, -- CIRCLED IDEOGRAPH QUESTION
 [0x3245] = 0x5E7C, -- CIRCLED IDEOGRAPH KINDERGARTEN
 [0x3246] = 0x6587, -- CIRCLED IDEOGRAPH SCHOOL
 [0x3247] = 0x7B8F, -- CIRCLED IDEOGRAPH KOTO
 [0x3250] = {0x0070, 0x0074, 0x0065}, -- PARTNERSHIP SIGN
 [0x3251] = {0x0032, 0x0031}, -- CIRCLED NUMBER TWENTY ONE
 [0x3252] = {0x0032, 0x0032}, -- CIRCLED NUMBER TWENTY TWO
 [0x3253] = {0x0032, 0x0033}, -- CIRCLED NUMBER TWENTY THREE
 [0x3254] = {0x0032, 0x0034}, -- CIRCLED NUMBER TWENTY FOUR
 [0x3255] = {0x0032, 0x0035}, -- CIRCLED NUMBER TWENTY FIVE
 [0x3256] = {0x0032, 0x0036}, -- CIRCLED NUMBER TWENTY SIX
 [0x3257] = {0x0032, 0x0037}, -- CIRCLED NUMBER TWENTY SEVEN
 [0x3258] = {0x0032, 0x0038}, -- CIRCLED NUMBER TWENTY EIGHT
 [0x3259] = {0x0032, 0x0039}, -- CIRCLED NUMBER TWENTY NINE
 [0x325A] = {0x0033, 0x0030}, -- CIRCLED NUMBER THIRTY
 [0x325B] = {0x0033, 0x0031}, -- CIRCLED NUMBER THIRTY ONE
 [0x325C] = {0x0033, 0x0032}, -- CIRCLED NUMBER THIRTY TWO
 [0x325D] = {0x0033, 0x0033}, -- CIRCLED NUMBER THIRTY THREE
 [0x325E] = {0x0033, 0x0034}, -- CIRCLED NUMBER THIRTY FOUR
 [0x325F] = {0x0033, 0x0035}, -- CIRCLED NUMBER THIRTY FIVE
 [0x3260] = 0x1100, -- CIRCLED HANGUL KIYEOK
 [0x3261] = 0x1102, -- CIRCLED HANGUL NIEUN
 [0x3262] = 0x1103, -- CIRCLED HANGUL TIKEUT
 [0x3263] = 0x1105, -- CIRCLED HANGUL RIEUL
 [0x3264] = 0x1106, -- CIRCLED HANGUL MIEUM
 [0x3265] = 0x1107, -- CIRCLED HANGUL PIEUP
 [0x3266] = 0x1109, -- CIRCLED HANGUL SIOS
 [0x3267] = 0x110B, -- CIRCLED HANGUL IEUNG
 [0x3268] = 0x110C, -- CIRCLED HANGUL CIEUC
 [0x3269] = 0x110E, -- CIRCLED HANGUL CHIEUCH
 [0x326A] = 0x110F, -- CIRCLED HANGUL KHIEUKH
 [0x326B] = 0x1110, -- CIRCLED HANGUL THIEUTH
 [0x326C] = 0x1111, -- CIRCLED HANGUL PHIEUPH
 [0x326D] = 0x1112, -- CIRCLED HANGUL HIEUH
 [0x326E] = 0xAC00, -- CIRCLED HANGUL KIYEOK A
 [0x326F] = 0xB098, -- CIRCLED HANGUL NIEUN A
 [0x3270] = 0xB2E4, -- CIRCLED HANGUL TIKEUT A
 [0x3271] = 0xB77C, -- CIRCLED HANGUL RIEUL A
 [0x3272] = 0xB9C8, -- CIRCLED HANGUL MIEUM A
 [0x3273] = 0xBC14, -- CIRCLED HANGUL PIEUP A
 [0x3274] = 0xC0AC, -- CIRCLED HANGUL SIOS A
 [0x3275] = 0xC544, -- CIRCLED HANGUL IEUNG A
 [0x3276] = 0xC790, -- CIRCLED HANGUL CIEUC A
 [0x3277] = 0xCC28, -- CIRCLED HANGUL CHIEUCH A
 [0x3278] = 0xCE74, -- CIRCLED HANGUL KHIEUKH A
 [0x3279] = 0xD0C0, -- CIRCLED HANGUL THIEUTH A
 [0x327A] = 0xD30C, -- CIRCLED HANGUL PHIEUPH A
 [0x327B] = 0xD558, -- CIRCLED HANGUL HIEUH A
 [0x327C] = {0xCC38, 0xACE0}, -- CIRCLED KOREAN CHARACTER CHAMKO
 [0x327D] = {0xC8FC, 0xC758}, -- CIRCLED KOREAN CHARACTER JUEUI
 [0x327E] = 0xC6B0, -- CIRCLED HANGUL IEUNG U
 [0x327F] = {status="disallowed"}, -- KOREAN STANDARD SYMBOL
 [0x3280] = 0x4E00, -- CIRCLED IDEOGRAPH ONE
 [0x3281] = 0x4E8C, -- CIRCLED IDEOGRAPH TWO
 [0x3282] = 0x4E09, -- CIRCLED IDEOGRAPH THREE
 [0x3283] = 0x56DB, -- CIRCLED IDEOGRAPH FOUR
 [0x3284] = 0x4E94, -- CIRCLED IDEOGRAPH FIVE
 [0x3285] = 0x516D, -- CIRCLED IDEOGRAPH SIX
 [0x3286] = 0x4E03, -- CIRCLED IDEOGRAPH SEVEN
 [0x3287] = 0x516B, -- CIRCLED IDEOGRAPH EIGHT
 [0x3288] = 0x4E5D, -- CIRCLED IDEOGRAPH NINE
 [0x3289] = 0x5341, -- CIRCLED IDEOGRAPH TEN
 [0x328A] = 0x6708, -- CIRCLED IDEOGRAPH MOON
 [0x328B] = 0x706B, -- CIRCLED IDEOGRAPH FIRE
 [0x328C] = 0x6C34, -- CIRCLED IDEOGRAPH WATER
 [0x328D] = 0x6728, -- CIRCLED IDEOGRAPH WOOD
 [0x328E] = 0x91D1, -- CIRCLED IDEOGRAPH METAL
 [0x328F] = 0x571F, -- CIRCLED IDEOGRAPH EARTH
 [0x3290] = 0x65E5, -- CIRCLED IDEOGRAPH SUN
 [0x3291] = 0x682A, -- CIRCLED IDEOGRAPH STOCK
 [0x3292] = 0x6709, -- CIRCLED IDEOGRAPH HAVE
 [0x3293] = 0x793E, -- CIRCLED IDEOGRAPH SOCIETY
 [0x3294] = 0x540D, -- CIRCLED IDEOGRAPH NAME
 [0x3295] = 0x7279, -- CIRCLED IDEOGRAPH SPECIAL
 [0x3296] = 0x8CA1, -- CIRCLED IDEOGRAPH FINANCIAL
 [0x3297] = 0x795D, -- CIRCLED IDEOGRAPH CONGRATULATION
 [0x3298] = 0x52B4, -- CIRCLED IDEOGRAPH LABOR
 [0x3299] = 0x79D8, -- CIRCLED IDEOGRAPH SECRET
 [0x329A] = 0x7537, -- CIRCLED IDEOGRAPH MALE
 [0x329B] = 0x5973, -- CIRCLED IDEOGRAPH FEMALE
 [0x329C] = 0x9069, -- CIRCLED IDEOGRAPH SUITABLE
 [0x329D] = 0x512A, -- CIRCLED IDEOGRAPH EXCELLENT
 [0x329E] = 0x5370, -- CIRCLED IDEOGRAPH PRINT
 [0x329F] = 0x6CE8, -- CIRCLED IDEOGRAPH ATTENTION
 [0x32A0] = 0x9805, -- CIRCLED IDEOGRAPH ITEM
 [0x32A1] = 0x4F11, -- CIRCLED IDEOGRAPH REST
 [0x32A2] = 0x5199, -- CIRCLED IDEOGRAPH COPY
 [0x32A3] = 0x6B63, -- CIRCLED IDEOGRAPH CORRECT
 [0x32A4] = 0x4E0A, -- CIRCLED IDEOGRAPH HIGH
 [0x32A5] = 0x4E2D, -- CIRCLED IDEOGRAPH CENTRE
 [0x32A6] = 0x4E0B, -- CIRCLED IDEOGRAPH LOW
 [0x32A7] = 0x5DE6, -- CIRCLED IDEOGRAPH LEFT
 [0x32A8] = 0x53F3, -- CIRCLED IDEOGRAPH RIGHT
 [0x32A9] = 0x533B, -- CIRCLED IDEOGRAPH MEDICINE
 [0x32AA] = 0x5B97, -- CIRCLED IDEOGRAPH RELIGION
 [0x32AB] = 0x5B66, -- CIRCLED IDEOGRAPH STUDY
 [0x32AC] = 0x76E3, -- CIRCLED IDEOGRAPH SUPERVISE
 [0x32AD] = 0x4F01, -- CIRCLED IDEOGRAPH ENTERPRISE
 [0x32AE] = 0x8CC7, -- CIRCLED IDEOGRAPH RESOURCE
 [0x32AF] = 0x5354, -- CIRCLED IDEOGRAPH ALLIANCE
 [0x32B0] = 0x591C, -- CIRCLED IDEOGRAPH NIGHT
 [0x32B1] = {0x0033, 0x0036}, -- CIRCLED NUMBER THIRTY SIX
 [0x32B2] = {0x0033, 0x0037}, -- CIRCLED NUMBER THIRTY SEVEN
 [0x32B3] = {0x0033, 0x0038}, -- CIRCLED NUMBER THIRTY EIGHT
 [0x32B4] = {0x0033, 0x0039}, -- CIRCLED NUMBER THIRTY NINE
 [0x32B5] = {0x0034, 0x0030}, -- CIRCLED NUMBER FORTY
 [0x32B6] = {0x0034, 0x0031}, -- CIRCLED NUMBER FORTY ONE
 [0x32B7] = {0x0034, 0x0032}, -- CIRCLED NUMBER FORTY TWO
 [0x32B8] = {0x0034, 0x0033}, -- CIRCLED NUMBER FORTY THREE
 [0x32B9] = {0x0034, 0x0034}, -- CIRCLED NUMBER FORTY FOUR
 [0x32BA] = {0x0034, 0x0035}, -- CIRCLED NUMBER FORTY FIVE
 [0x32BB] = {0x0034, 0x0036}, -- CIRCLED NUMBER FORTY SIX
 [0x32BC] = {0x0034, 0x0037}, -- CIRCLED NUMBER FORTY SEVEN
 [0x32BD] = {0x0034, 0x0038}, -- CIRCLED NUMBER FORTY EIGHT
 [0x32BE] = {0x0034, 0x0039}, -- CIRCLED NUMBER FORTY NINE
 [0x32BF] = {0x0035, 0x0030}, -- CIRCLED NUMBER FIFTY
 [0x32C0] = {0x0031, 0x6708}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR JANUARY
 [0x32C1] = {0x0032, 0x6708}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR FEBRUARY
 [0x32C2] = {0x0033, 0x6708}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR MARCH
 [0x32C3] = {0x0034, 0x6708}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR APRIL
 [0x32C4] = {0x0035, 0x6708}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR MAY
 [0x32C5] = {0x0036, 0x6708}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR JUNE
 [0x32C6] = {0x0037, 0x6708}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR JULY
 [0x32C7] = {0x0038, 0x6708}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR AUGUST
 [0x32C8] = {0x0039, 0x6708}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR SEPTEMBER
 [0x32C9] = {0x0031, 0x0030, 0x6708}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR OCTOBER
 [0x32CA] = {0x0031, 0x0031, 0x6708}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR NOVEMBER
 [0x32CB] = {0x0031, 0x0032, 0x6708}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DECEMBER
 [0x32CC] = {0x0068, 0x0067}, -- SQUARE HG
 [0x32CD] = {0x0065, 0x0072, 0x0067}, -- SQUARE ERG
 [0x32CE] = {0x0065, 0x0076}, -- SQUARE EV
 [0x32CF] = {0x006C, 0x0074, 0x0064}, -- LIMITED LIABILITY SIGN
 [0x32D0] = 0x30A2, -- CIRCLED KATAKANA A
 [0x32D1] = 0x30A4, -- CIRCLED KATAKANA I
 [0x32D2] = 0x30A6, -- CIRCLED KATAKANA U
 [0x32D3] = 0x30A8, -- CIRCLED KATAKANA E
 [0x32D4] = 0x30AA, -- CIRCLED KATAKANA O
 [0x32D5] = 0x30AB, -- CIRCLED KATAKANA KA
 [0x32D6] = 0x30AD, -- CIRCLED KATAKANA KI
 [0x32D7] = 0x30AF, -- CIRCLED KATAKANA KU
 [0x32D8] = 0x30B1, -- CIRCLED KATAKANA KE
 [0x32D9] = 0x30B3, -- CIRCLED KATAKANA KO
 [0x32DA] = 0x30B5, -- CIRCLED KATAKANA SA
 [0x32DB] = 0x30B7, -- CIRCLED KATAKANA SI
 [0x32DC] = 0x30B9, -- CIRCLED KATAKANA SU
 [0x32DD] = 0x30BB, -- CIRCLED KATAKANA SE
 [0x32DE] = 0x30BD, -- CIRCLED KATAKANA SO
 [0x32DF] = 0x30BF, -- CIRCLED KATAKANA TA
 [0x32E0] = 0x30C1, -- CIRCLED KATAKANA TI
 [0x32E1] = 0x30C4, -- CIRCLED KATAKANA TU
 [0x32E2] = 0x30C6, -- CIRCLED KATAKANA TE
 [0x32E3] = 0x30C8, -- CIRCLED KATAKANA TO
 [0x32E4] = 0x30CA, -- CIRCLED KATAKANA NA
 [0x32E5] = 0x30CB, -- CIRCLED KATAKANA NI
 [0x32E6] = 0x30CC, -- CIRCLED KATAKANA NU
 [0x32E7] = 0x30CD, -- CIRCLED KATAKANA NE
 [0x32E8] = 0x30CE, -- CIRCLED KATAKANA NO
 [0x32E9] = 0x30CF, -- CIRCLED KATAKANA HA
 [0x32EA] = 0x30D2, -- CIRCLED KATAKANA HI
 [0x32EB] = 0x30D5, -- CIRCLED KATAKANA HU
 [0x32EC] = 0x30D8, -- CIRCLED KATAKANA HE
 [0x32ED] = 0x30DB, -- CIRCLED KATAKANA HO
 [0x32EE] = 0x30DE, -- CIRCLED KATAKANA MA
 [0x32EF] = 0x30DF, -- CIRCLED KATAKANA MI
 [0x32F0] = 0x30E0, -- CIRCLED KATAKANA MU
 [0x32F1] = 0x30E1, -- CIRCLED KATAKANA ME
 [0x32F2] = 0x30E2, -- CIRCLED KATAKANA MO
 [0x32F3] = 0x30E4, -- CIRCLED KATAKANA YA
 [0x32F4] = 0x30E6, -- CIRCLED KATAKANA YU
 [0x32F5] = 0x30E8, -- CIRCLED KATAKANA YO
 [0x32F6] = 0x30E9, -- CIRCLED KATAKANA RA
 [0x32F7] = 0x30EA, -- CIRCLED KATAKANA RI
 [0x32F8] = 0x30EB, -- CIRCLED KATAKANA RU
 [0x32F9] = 0x30EC, -- CIRCLED KATAKANA RE
 [0x32FA] = 0x30ED, -- CIRCLED KATAKANA RO
 [0x32FB] = 0x30EF, -- CIRCLED KATAKANA WA
 [0x32FC] = 0x30F0, -- CIRCLED KATAKANA WI
 [0x32FD] = 0x30F1, -- CIRCLED KATAKANA WE
 [0x32FE] = 0x30F2, -- CIRCLED KATAKANA WO
 [0x32FF] = {status="disallowed"}, -- NA   <reserved-32FF>
 [0x3300] = {0x30A2, 0x30D1, 0x30FC, 0x30C8}, -- SQUARE APAATO
 [0x3301] = {0x30A2, 0x30EB, 0x30D5, 0x30A1}, -- SQUARE ARUHUA
 [0x3302] = {0x30A2, 0x30F3, 0x30DA, 0x30A2}, -- SQUARE ANPEA
 [0x3303] = {0x30A2, 0x30FC, 0x30EB}, -- SQUARE AARU
 [0x3304] = {0x30A4, 0x30CB, 0x30F3, 0x30B0}, -- SQUARE ININGU
 [0x3305] = {0x30A4, 0x30F3, 0x30C1}, -- SQUARE INTI
 [0x3306] = {0x30A6, 0x30A9, 0x30F3}, -- SQUARE UON
 [0x3307] = {0x30A8, 0x30B9, 0x30AF, 0x30FC, 0x30C9}, -- SQUARE ESUKUUDO
 [0x3308] = {0x30A8, 0x30FC, 0x30AB, 0x30FC}, -- SQUARE EEKAA
 [0x3309] = {0x30AA, 0x30F3, 0x30B9}, -- SQUARE ONSU
 [0x330A] = {0x30AA, 0x30FC, 0x30E0}, -- SQUARE OOMU
 [0x330B] = {0x30AB, 0x30A4, 0x30EA}, -- SQUARE KAIRI
 [0x330C] = {0x30AB, 0x30E9, 0x30C3, 0x30C8}, -- SQUARE KARATTO
 [0x330D] = {0x30AB, 0x30ED, 0x30EA, 0x30FC}, -- SQUARE KARORII
 [0x330E] = {0x30AC, 0x30ED, 0x30F3}, -- SQUARE GARON
 [0x330F] = {0x30AC, 0x30F3, 0x30DE}, -- SQUARE GANMA
 [0x3310] = {0x30AE, 0x30AC}, -- SQUARE GIGA
 [0x3311] = {0x30AE, 0x30CB, 0x30FC}, -- SQUARE GINII
 [0x3312] = {0x30AD, 0x30E5, 0x30EA, 0x30FC}, -- SQUARE KYURII
 [0x3313] = {0x30AE, 0x30EB, 0x30C0, 0x30FC}, -- SQUARE GIRUDAA
 [0x3314] = {0x30AD, 0x30ED}, -- SQUARE KIRO
 [0x3315] = {0x30AD, 0x30ED, 0x30B0, 0x30E9, 0x30E0}, -- SQUARE KIROGURAMU
 [0x3316] = {0x30AD, 0x30ED, 0x30E1, 0x30FC, 0x30C8, 0x30EB}, -- SQUARE KIROMEETORU
 [0x3317] = {0x30AD, 0x30ED, 0x30EF, 0x30C3, 0x30C8}, -- SQUARE KIROWATTO
 [0x3318] = {0x30B0, 0x30E9, 0x30E0}, -- SQUARE GURAMU
 [0x3319] = {0x30B0, 0x30E9, 0x30E0, 0x30C8, 0x30F3}, -- SQUARE GURAMUTON
 [0x331A] = {0x30AF, 0x30EB, 0x30BC, 0x30A4, 0x30ED}, -- SQUARE KURUZEIRO
 [0x331B] = {0x30AF, 0x30ED, 0x30FC, 0x30CD}, -- SQUARE KUROONE
 [0x331C] = {0x30B1, 0x30FC, 0x30B9}, -- SQUARE KEESU
 [0x331D] = {0x30B3, 0x30EB, 0x30CA}, -- SQUARE KORUNA
 [0x331E] = {0x30B3, 0x30FC, 0x30DD}, -- SQUARE KOOPO
 [0x331F] = {0x30B5, 0x30A4, 0x30AF, 0x30EB}, -- SQUARE SAIKURU
 [0x3320] = {0x30B5, 0x30F3, 0x30C1, 0x30FC, 0x30E0}, -- SQUARE SANTIIMU
 [0x3321] = {0x30B7, 0x30EA, 0x30F3, 0x30B0}, -- SQUARE SIRINGU
 [0x3322] = {0x30BB, 0x30F3, 0x30C1}, -- SQUARE SENTI
 [0x3323] = {0x30BB, 0x30F3, 0x30C8}, -- SQUARE SENTO
 [0x3324] = {0x30C0, 0x30FC, 0x30B9}, -- SQUARE DAASU
 [0x3325] = {0x30C7, 0x30B7}, -- SQUARE DESI
 [0x3326] = {0x30C9, 0x30EB}, -- SQUARE DORU
 [0x3327] = {0x30C8, 0x30F3}, -- SQUARE TON
 [0x3328] = {0x30CA, 0x30CE}, -- SQUARE NANO
 [0x3329] = {0x30CE, 0x30C3, 0x30C8}, -- SQUARE NOTTO
 [0x332A] = {0x30CF, 0x30A4, 0x30C4}, -- SQUARE HAITU
 [0x332B] = {0x30D1, 0x30FC, 0x30BB, 0x30F3, 0x30C8}, -- SQUARE PAASENTO
 [0x332C] = {0x30D1, 0x30FC, 0x30C4}, -- SQUARE PAATU
 [0x332D] = {0x30D0, 0x30FC, 0x30EC, 0x30EB}, -- SQUARE BAARERU
 [0x332E] = {0x30D4, 0x30A2, 0x30B9, 0x30C8, 0x30EB}, -- SQUARE PIASUTORU
 [0x332F] = {0x30D4, 0x30AF, 0x30EB}, -- SQUARE PIKURU
 [0x3330] = {0x30D4, 0x30B3}, -- SQUARE PIKO
 [0x3331] = {0x30D3, 0x30EB}, -- SQUARE BIRU
 [0x3332] = {0x30D5, 0x30A1, 0x30E9, 0x30C3, 0x30C9}, -- SQUARE HUARADDO
 [0x3333] = {0x30D5, 0x30A3, 0x30FC, 0x30C8}, -- SQUARE HUIITO
 [0x3334] = {0x30D6, 0x30C3, 0x30B7, 0x30A7, 0x30EB}, -- SQUARE BUSSYERU
 [0x3335] = {0x30D5, 0x30E9, 0x30F3}, -- SQUARE HURAN
 [0x3336] = {0x30D8, 0x30AF, 0x30BF, 0x30FC, 0x30EB}, -- SQUARE HEKUTAARU
 [0x3337] = {0x30DA, 0x30BD}, -- SQUARE PESO
 [0x3338] = {0x30DA, 0x30CB, 0x30D2}, -- SQUARE PENIHI
 [0x3339] = {0x30D8, 0x30EB, 0x30C4}, -- SQUARE HERUTU
 [0x333A] = {0x30DA, 0x30F3, 0x30B9}, -- SQUARE PENSU
 [0x333B] = {0x30DA, 0x30FC, 0x30B8}, -- SQUARE PEEZI
 [0x333C] = {0x30D9, 0x30FC, 0x30BF}, -- SQUARE BEETA
 [0x333D] = {0x30DD, 0x30A4, 0x30F3, 0x30C8}, -- SQUARE POINTO
 [0x333E] = {0x30DC, 0x30EB, 0x30C8}, -- SQUARE BORUTO
 [0x333F] = {0x30DB, 0x30F3}, -- SQUARE HON
 [0x3340] = {0x30DD, 0x30F3, 0x30C9}, -- SQUARE PONDO
 [0x3341] = {0x30DB, 0x30FC, 0x30EB}, -- SQUARE HOORU
 [0x3342] = {0x30DB, 0x30FC, 0x30F3}, -- SQUARE HOON
 [0x3343] = {0x30DE, 0x30A4, 0x30AF, 0x30ED}, -- SQUARE MAIKURO
 [0x3344] = {0x30DE, 0x30A4, 0x30EB}, -- SQUARE MAIRU
 [0x3345] = {0x30DE, 0x30C3, 0x30CF}, -- SQUARE MAHHA
 [0x3346] = {0x30DE, 0x30EB, 0x30AF}, -- SQUARE MARUKU
 [0x3347] = {0x30DE, 0x30F3, 0x30B7, 0x30E7, 0x30F3}, -- SQUARE MANSYON
 [0x3348] = {0x30DF, 0x30AF, 0x30ED, 0x30F3}, -- SQUARE MIKURON
 [0x3349] = {0x30DF, 0x30EA}, -- SQUARE MIRI
 [0x334A] = {0x30DF, 0x30EA, 0x30D0, 0x30FC, 0x30EB}, -- SQUARE MIRIBAARU
 [0x334B] = {0x30E1, 0x30AC}, -- SQUARE MEGA
 [0x334C] = {0x30E1, 0x30AC, 0x30C8, 0x30F3}, -- SQUARE MEGATON
 [0x334D] = {0x30E1, 0x30FC, 0x30C8, 0x30EB}, -- SQUARE MEETORU
 [0x334E] = {0x30E4, 0x30FC, 0x30C9}, -- SQUARE YAADO
 [0x334F] = {0x30E4, 0x30FC, 0x30EB}, -- SQUARE YAARU
 [0x3350] = {0x30E6, 0x30A2, 0x30F3}, -- SQUARE YUAN
 [0x3351] = {0x30EA, 0x30C3, 0x30C8, 0x30EB}, -- SQUARE RITTORU
 [0x3352] = {0x30EA, 0x30E9}, -- SQUARE RIRA
 [0x3353] = {0x30EB, 0x30D4, 0x30FC}, -- SQUARE RUPII
 [0x3354] = {0x30EB, 0x30FC, 0x30D6, 0x30EB}, -- SQUARE RUUBURU
 [0x3355] = {0x30EC, 0x30E0}, -- SQUARE REMU
 [0x3356] = {0x30EC, 0x30F3, 0x30C8, 0x30B2, 0x30F3}, -- SQUARE RENTOGEN
 [0x3357] = {0x30EF, 0x30C3, 0x30C8}, -- SQUARE WATTO
 [0x3358] = {0x0030, 0x70B9}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR ZERO
 [0x3359] = {0x0031, 0x70B9}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR ONE
 [0x335A] = {0x0032, 0x70B9}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWO
 [0x335B] = {0x0033, 0x70B9}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR THREE
 [0x335C] = {0x0034, 0x70B9}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FOUR
 [0x335D] = {0x0035, 0x70B9}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FIVE
 [0x335E] = {0x0036, 0x70B9}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SIX
 [0x335F] = {0x0037, 0x70B9}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SEVEN
 [0x3360] = {0x0038, 0x70B9}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR EIGHT
 [0x3361] = {0x0039, 0x70B9}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR NINE
 [0x3362] = {0x0031, 0x0030, 0x70B9}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TEN
 [0x3363] = {0x0031, 0x0031, 0x70B9}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR ELEVEN
 [0x3364] = {0x0031, 0x0032, 0x70B9}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWELVE
 [0x3365] = {0x0031, 0x0033, 0x70B9}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR THIRTEEN
 [0x3366] = {0x0031, 0x0034, 0x70B9}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FOURTEEN
 [0x3367] = {0x0031, 0x0035, 0x70B9}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR FIFTEEN
 [0x3368] = {0x0031, 0x0036, 0x70B9}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SIXTEEN
 [0x3369] = {0x0031, 0x0037, 0x70B9}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR SEVENTEEN
 [0x336A] = {0x0031, 0x0038, 0x70B9}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR EIGHTEEN
 [0x336B] = {0x0031, 0x0039, 0x70B9}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR NINETEEN
 [0x336C] = {0x0032, 0x0030, 0x70B9}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY
 [0x336D] = {0x0032, 0x0031, 0x70B9}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-ONE
 [0x336E] = {0x0032, 0x0032, 0x70B9}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-TWO
 [0x336F] = {0x0032, 0x0033, 0x70B9}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-THREE
 [0x3370] = {0x0032, 0x0034, 0x70B9}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR HOUR TWENTY-FOUR
 [0x3371] = {0x0068, 0x0070, 0x0061}, -- SQUARE HPA
 [0x3372] = {0x0064, 0x0061}, -- SQUARE DA
 [0x3373] = {0x0061, 0x0075}, -- SQUARE AU
 [0x3374] = {0x0062, 0x0061, 0x0072}, -- SQUARE BAR
 [0x3375] = {0x006F, 0x0076}, -- SQUARE OV
 [0x3376] = {0x0070, 0x0063}, -- SQUARE PC
 [0x3377] = {0x0064, 0x006D}, -- SQUARE DM
 [0x3378] = {0x0064, 0x006D, 0x0032}, -- SQUARE DM SQUARED
 [0x3379] = {0x0064, 0x006D, 0x0033}, -- SQUARE DM CUBED
 [0x337A] = {0x0069, 0x0075}, -- SQUARE IU
 [0x337B] = {0x5E73, 0x6210}, -- SQUARE ERA NAME HEISEI
 [0x337C] = {0x662D, 0x548C}, -- SQUARE ERA NAME SYOUWA
 [0x337D] = {0x5927, 0x6B63}, -- SQUARE ERA NAME TAISYOU
 [0x337E] = {0x660E, 0x6CBB}, -- SQUARE ERA NAME MEIZI
 [0x337F] = {0x682A, 0x5F0F, 0x4F1A, 0x793E}, -- SQUARE CORPORATION
 [0x3380] = {0x0070, 0x0061}, -- SQUARE PA AMPS
 [0x3381] = {0x006E, 0x0061}, -- SQUARE NA
 [0x3382] = {0x03BC, 0x0061}, -- SQUARE MU A
 [0x3383] = {0x006D, 0x0061}, -- SQUARE MA
 [0x3384] = {0x006B, 0x0061}, -- SQUARE KA
 [0x3385] = {0x006B, 0x0062}, -- SQUARE KB
 [0x3386] = {0x006D, 0x0062}, -- SQUARE MB
 [0x3387] = {0x0067, 0x0062}, -- SQUARE GB
 [0x3388] = {0x0063, 0x0061, 0x006C}, -- SQUARE CAL
 [0x3389] = {0x006B, 0x0063, 0x0061, 0x006C}, -- SQUARE KCAL
 [0x338A] = {0x0070, 0x0066}, -- SQUARE PF
 [0x338B] = {0x006E, 0x0066}, -- SQUARE NF
 [0x338C] = {0x03BC, 0x0066}, -- SQUARE MU F
 [0x338D] = {0x03BC, 0x0067}, -- SQUARE MU G
 [0x338E] = {0x006D, 0x0067}, -- SQUARE MG
 [0x338F] = {0x006B, 0x0067}, -- SQUARE KG
 [0x3390] = {0x0068, 0x007A}, -- SQUARE HZ
 [0x3391] = {0x006B, 0x0068, 0x007A}, -- SQUARE KHZ
 [0x3392] = {0x006D, 0x0068, 0x007A}, -- SQUARE MHZ
 [0x3393] = {0x0067, 0x0068, 0x007A}, -- SQUARE GHZ
 [0x3394] = {0x0074, 0x0068, 0x007A}, -- SQUARE THZ
 [0x3395] = {0x03BC, 0x006C}, -- SQUARE MU L
 [0x3396] = {0x006D, 0x006C}, -- SQUARE ML
 [0x3397] = {0x0064, 0x006C}, -- SQUARE DL
 [0x3398] = {0x006B, 0x006C}, -- SQUARE KL
 [0x3399] = {0x0066, 0x006D}, -- SQUARE FM
 [0x339A] = {0x006E, 0x006D}, -- SQUARE NM
 [0x339B] = {0x03BC, 0x006D}, -- SQUARE MU M
 [0x339C] = {0x006D, 0x006D}, -- SQUARE MM
 [0x339D] = {0x0063, 0x006D}, -- SQUARE CM
 [0x339E] = {0x006B, 0x006D}, -- SQUARE KM
 [0x339F] = {0x006D, 0x006D, 0x0032}, -- SQUARE MM SQUARED
 [0x33A0] = {0x0063, 0x006D, 0x0032}, -- SQUARE CM SQUARED
 [0x33A1] = {0x006D, 0x0032}, -- SQUARE M SQUARED
 [0x33A2] = {0x006B, 0x006D, 0x0032}, -- SQUARE KM SQUARED
 [0x33A3] = {0x006D, 0x006D, 0x0033}, -- SQUARE MM CUBED
 [0x33A4] = {0x0063, 0x006D, 0x0033}, -- SQUARE CM CUBED
 [0x33A5] = {0x006D, 0x0033}, -- SQUARE M CUBED
 [0x33A6] = {0x006B, 0x006D, 0x0033}, -- SQUARE KM CUBED
 [0x33A7] = {0x006D, 0x2215, 0x0073}, -- SQUARE M OVER S
 [0x33A8] = {0x006D, 0x2215, 0x0073, 0x0032}, -- SQUARE M OVER S SQUARED
 [0x33A9] = {0x0070, 0x0061}, -- SQUARE PA
 [0x33AA] = {0x006B, 0x0070, 0x0061}, -- SQUARE KPA
 [0x33AB] = {0x006D, 0x0070, 0x0061}, -- SQUARE MPA
 [0x33AC] = {0x0067, 0x0070, 0x0061}, -- SQUARE GPA
 [0x33AD] = {0x0072, 0x0061, 0x0064}, -- SQUARE RAD
 [0x33AE] = {0x0072, 0x0061, 0x0064, 0x2215, 0x0073}, -- SQUARE RAD OVER S
 [0x33AF] = {0x0072, 0x0061, 0x0064, 0x2215, 0x0073, 0x0032}, -- SQUARE RAD OVER S SQUARED
 [0x33B0] = {0x0070, 0x0073}, -- SQUARE PS
 [0x33B1] = {0x006E, 0x0073}, -- SQUARE NS
 [0x33B2] = {0x03BC, 0x0073}, -- SQUARE MU S
 [0x33B3] = {0x006D, 0x0073}, -- SQUARE MS
 [0x33B4] = {0x0070, 0x0076}, -- SQUARE PV
 [0x33B5] = {0x006E, 0x0076}, -- SQUARE NV
 [0x33B6] = {0x03BC, 0x0076}, -- SQUARE MU V
 [0x33B7] = {0x006D, 0x0076}, -- SQUARE MV
 [0x33B8] = {0x006B, 0x0076}, -- SQUARE KV
 [0x33B9] = {0x006D, 0x0076}, -- SQUARE MV MEGA
 [0x33BA] = {0x0070, 0x0077}, -- SQUARE PW
 [0x33BB] = {0x006E, 0x0077}, -- SQUARE NW
 [0x33BC] = {0x03BC, 0x0077}, -- SQUARE MU W
 [0x33BD] = {0x006D, 0x0077}, -- SQUARE MW
 [0x33BE] = {0x006B, 0x0077}, -- SQUARE KW
 [0x33BF] = {0x006D, 0x0077}, -- SQUARE MW MEGA
 [0x33C0] = {0x006B, 0x03C9}, -- SQUARE K OHM
 [0x33C1] = {0x006D, 0x03C9}, -- SQUARE M OHM
 [0x33C2] = {status="disallowed"}, -- SQUARE AM
 [0x33C3] = {0x0062, 0x0071}, -- SQUARE BQ
 [0x33C4] = {0x0063, 0x0063}, -- SQUARE CC
 [0x33C5] = {0x0063, 0x0064}, -- SQUARE CD
 [0x33C6] = {0x0063, 0x2215, 0x006B, 0x0067}, -- SQUARE C OVER KG
 [0x33C7] = {status="disallowed"}, -- SQUARE CO
 [0x33C8] = {0x0064, 0x0062}, -- SQUARE DB
 [0x33C9] = {0x0067, 0x0079}, -- SQUARE GY
 [0x33CA] = {0x0068, 0x0061}, -- SQUARE HA
 [0x33CB] = {0x0068, 0x0070}, -- SQUARE HP
 [0x33CC] = {0x0069, 0x006E}, -- SQUARE IN
 [0x33CD] = {0x006B, 0x006B}, -- SQUARE KK
 [0x33CE] = {0x006B, 0x006D}, -- SQUARE KM CAPITAL
 [0x33CF] = {0x006B, 0x0074}, -- SQUARE KT
 [0x33D0] = {0x006C, 0x006D}, -- SQUARE LM
 [0x33D1] = {0x006C, 0x006E}, -- SQUARE LN
 [0x33D2] = {0x006C, 0x006F, 0x0067}, -- SQUARE LOG
 [0x33D3] = {0x006C, 0x0078}, -- SQUARE LX
 [0x33D4] = {0x006D, 0x0062}, -- SQUARE MB SMALL
 [0x33D5] = {0x006D, 0x0069, 0x006C}, -- SQUARE MIL
 [0x33D6] = {0x006D, 0x006F, 0x006C}, -- SQUARE MOL
 [0x33D7] = {0x0070, 0x0068}, -- SQUARE PH
 [0x33D8] = {status="disallowed"}, -- SQUARE PM
 [0x33D9] = {0x0070, 0x0070, 0x006D}, -- SQUARE PPM
 [0x33DA] = {0x0070, 0x0072}, -- SQUARE PR
 [0x33DB] = {0x0073, 0x0072}, -- SQUARE SR
 [0x33DC] = {0x0073, 0x0076}, -- SQUARE SV
 [0x33DD] = {0x0077, 0x0062}, -- SQUARE WB
 [0x33DE] = {0x0076, 0x2215, 0x006D}, -- SQUARE V OVER M
 [0x33DF] = {0x0061, 0x2215, 0x006D}, -- SQUARE A OVER M
 [0x33E0] = {0x0031, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY ONE
 [0x33E1] = {0x0032, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWO
 [0x33E2] = {0x0033, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THREE
 [0x33E3] = {0x0034, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FOUR
 [0x33E4] = {0x0035, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FIVE
 [0x33E5] = {0x0036, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SIX
 [0x33E6] = {0x0037, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SEVEN
 [0x33E7] = {0x0038, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY EIGHT
 [0x33E8] = {0x0039, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY NINE
 [0x33E9] = {0x0031, 0x0030, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TEN
 [0x33EA] = {0x0031, 0x0031, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY ELEVEN
 [0x33EB] = {0x0031, 0x0032, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWELVE
 [0x33EC] = {0x0031, 0x0033, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THIRTEEN
 [0x33ED] = {0x0031, 0x0034, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FOURTEEN
 [0x33EE] = {0x0031, 0x0035, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY FIFTEEN
 [0x33EF] = {0x0031, 0x0036, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SIXTEEN
 [0x33F0] = {0x0031, 0x0037, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY SEVENTEEN
 [0x33F1] = {0x0031, 0x0038, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY EIGHTEEN
 [0x33F2] = {0x0031, 0x0039, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY NINETEEN
 [0x33F3] = {0x0032, 0x0030, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY
 [0x33F4] = {0x0032, 0x0031, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-ONE
 [0x33F5] = {0x0032, 0x0032, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-TWO
 [0x33F6] = {0x0032, 0x0033, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-THREE
 [0x33F7] = {0x0032, 0x0034, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-FOUR
 [0x33F8] = {0x0032, 0x0035, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-FIVE
 [0x33F9] = {0x0032, 0x0036, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-SIX
 [0x33FA] = {0x0032, 0x0037, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-SEVEN
 [0x33FB] = {0x0032, 0x0038, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-EIGHT
 [0x33FC] = {0x0032, 0x0039, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY TWENTY-NINE
 [0x33FD] = {0x0033, 0x0030, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THIRTY
 [0x33FE] = {0x0033, 0x0031, 0x65E5}, -- IDEOGRAPHIC TELEGRAPH SYMBOL FOR DAY THIRTY-ONE
 [0x33FF] = {0x0067, 0x0061, 0x006C}, -- SQUARE GAL
 [0x9FCC] = {status="valid"}, -- CJK UNIFIED IDEOGRAPH-9FCC
 -- NA   <reserved-A48D>..<reserved-A48F>
 [0xA48D] = {status="disallowed"},
 [0xA48E] = {status="disallowed"},
 [0xA48F] = {status="disallowed"},
 [0xA4B4] = {status="disallowed"}, -- YI RADICAL NZUP
 [0xA4C1] = {status="disallowed"}, -- YI RADICAL ZUR
 [0xA4C5] = {status="disallowed"}, -- YI RADICAL NBIE
 [0xA4C6] = {status="disallowed"}, -- YI RADICAL KE
 -- YI RADICAL SHOP..YI RADICAL ZZIET
 [0xA4C2] = {status="disallowed"},
 [0xA4C3] = {status="disallowed"},
 [0xA4C4] = {status="disallowed"},
 -- LISU PUNCTUATION COMMA..LISU PUNCTUATION FULL STOP
 [0xA4FE] = {status="disallowed"},
 [0xA4FF] = {status="disallowed"},
 -- VAI COMMA..VAI QUESTION MARK
 [0xA60D] = {status="disallowed"},
 [0xA60E] = {status="disallowed"},
 [0xA60F] = {status="disallowed"},
 [0xA640] = 0xA641, -- CYRILLIC CAPITAL LETTER ZEMLYA
 [0xA641] = {status="valid"}, -- CYRILLIC SMALL LETTER ZEMLYA
 [0xA642] = 0xA643, -- CYRILLIC CAPITAL LETTER DZELO
 [0xA643] = {status="valid"}, -- CYRILLIC SMALL LETTER DZELO
 [0xA644] = 0xA645, -- CYRILLIC CAPITAL LETTER REVERSED DZE
 [0xA645] = {status="valid"}, -- CYRILLIC SMALL LETTER REVERSED DZE
 [0xA646] = 0xA647, -- CYRILLIC CAPITAL LETTER IOTA
 [0xA647] = {status="valid"}, -- CYRILLIC SMALL LETTER IOTA
 [0xA648] = 0xA649, -- CYRILLIC CAPITAL LETTER DJERV
 [0xA649] = {status="valid"}, -- CYRILLIC SMALL LETTER DJERV
 [0xA64A] = 0xA64B, -- CYRILLIC CAPITAL LETTER MONOGRAPH UK
 [0xA64B] = {status="valid"}, -- CYRILLIC SMALL LETTER MONOGRAPH UK
 [0xA64C] = 0xA64D, -- CYRILLIC CAPITAL LETTER BROAD OMEGA
 [0xA64D] = {status="valid"}, -- CYRILLIC SMALL LETTER BROAD OMEGA
 [0xA64E] = 0xA64F, -- CYRILLIC CAPITAL LETTER NEUTRAL YER
 [0xA64F] = {status="valid"}, -- CYRILLIC SMALL LETTER NEUTRAL YER
 [0xA650] = 0xA651, -- CYRILLIC CAPITAL LETTER YERU WITH BACK YER
 [0xA651] = {status="valid"}, -- CYRILLIC SMALL LETTER YERU WITH BACK YER
 [0xA652] = 0xA653, -- CYRILLIC CAPITAL LETTER IOTIFIED YAT
 [0xA653] = {status="valid"}, -- CYRILLIC SMALL LETTER IOTIFIED YAT
 [0xA654] = 0xA655, -- CYRILLIC CAPITAL LETTER REVERSED YU
 [0xA655] = {status="valid"}, -- CYRILLIC SMALL LETTER REVERSED YU
 [0xA656] = 0xA657, -- CYRILLIC CAPITAL LETTER IOTIFIED A
 [0xA657] = {status="valid"}, -- CYRILLIC SMALL LETTER IOTIFIED A
 [0xA658] = 0xA659, -- CYRILLIC CAPITAL LETTER CLOSED LITTLE YUS
 [0xA659] = {status="valid"}, -- CYRILLIC SMALL LETTER CLOSED LITTLE YUS
 [0xA65A] = 0xA65B, -- CYRILLIC CAPITAL LETTER BLENDED YUS
 [0xA65B] = {status="valid"}, -- CYRILLIC SMALL LETTER BLENDED YUS
 [0xA65C] = 0xA65D, -- CYRILLIC CAPITAL LETTER IOTIFIED CLOSED LITTLE YUS
 [0xA65D] = {status="valid"}, -- CYRILLIC SMALL LETTER IOTIFIED CLOSED LITTLE YUS
 [0xA65E] = 0xA65F, -- CYRILLIC CAPITAL LETTER YN
 [0xA65F] = {status="valid"}, -- CYRILLIC SMALL LETTER YN
 [0xA660] = 0xA661, -- CYRILLIC CAPITAL LETTER REVERSED TSE
 [0xA661] = {status="valid"}, -- CYRILLIC SMALL LETTER REVERSED TSE
 [0xA662] = 0xA663, -- CYRILLIC CAPITAL LETTER SOFT DE
 [0xA663] = {status="valid"}, -- CYRILLIC SMALL LETTER SOFT DE
 [0xA664] = 0xA665, -- CYRILLIC CAPITAL LETTER SOFT EL
 [0xA665] = {status="valid"}, -- CYRILLIC SMALL LETTER SOFT EL
 [0xA666] = 0xA667, -- CYRILLIC CAPITAL LETTER SOFT EM
 [0xA667] = {status="valid"}, -- CYRILLIC SMALL LETTER SOFT EM
 [0xA668] = 0xA669, -- CYRILLIC CAPITAL LETTER MONOCULAR O
 [0xA669] = {status="valid"}, -- CYRILLIC SMALL LETTER MONOCULAR O
 [0xA66A] = 0xA66B, -- CYRILLIC CAPITAL LETTER BINOCULAR O
 [0xA66B] = {status="valid"}, -- CYRILLIC SMALL LETTER BINOCULAR O
 [0xA66C] = 0xA66D, -- CYRILLIC CAPITAL LETTER DOUBLE MONOCULAR O
 -- CYRILLIC SMALL LETTER DOUBLE MONOCULAR O..COMBINING CYRILLIC VZMET
 [0xA66D] = {status="valid"},
 [0xA66E] = {status="valid"},
 [0xA66F] = {status="valid"},
 -- COMBINING CYRILLIC TEN MILLIONS SIGN..SLAVONIC ASTERISK
 [0xA670] = {status="disallowed"},
 [0xA671] = {status="disallowed"},
 [0xA672] = {status="disallowed"},
 [0xA673] = {status="disallowed"},
 [0xA67E] = {status="disallowed"}, -- CYRILLIC KAVYKA
 [0xA67F] = {status="valid"}, -- CYRILLIC PAYEROK
 [0xA680] = 0xA681, -- CYRILLIC CAPITAL LETTER DWE
 [0xA681] = {status="valid"}, -- CYRILLIC SMALL LETTER DWE
 [0xA682] = 0xA683, -- CYRILLIC CAPITAL LETTER DZWE
 [0xA683] = {status="valid"}, -- CYRILLIC SMALL LETTER DZWE
 [0xA684] = 0xA685, -- CYRILLIC CAPITAL LETTER ZHWE
 [0xA685] = {status="valid"}, -- CYRILLIC SMALL LETTER ZHWE
 [0xA686] = 0xA687, -- CYRILLIC CAPITAL LETTER CCHE
 [0xA687] = {status="valid"}, -- CYRILLIC SMALL LETTER CCHE
 [0xA688] = 0xA689, -- CYRILLIC CAPITAL LETTER DZZE
 [0xA689] = {status="valid"}, -- CYRILLIC SMALL LETTER DZZE
 [0xA68A] = 0xA68B, -- CYRILLIC CAPITAL LETTER TE WITH MIDDLE HOOK
 [0xA68B] = {status="valid"}, -- CYRILLIC SMALL LETTER TE WITH MIDDLE HOOK
 [0xA68C] = 0xA68D, -- CYRILLIC CAPITAL LETTER TWE
 [0xA68D] = {status="valid"}, -- CYRILLIC SMALL LETTER TWE
 [0xA68E] = 0xA68F, -- CYRILLIC CAPITAL LETTER TSWE
 [0xA68F] = {status="valid"}, -- CYRILLIC SMALL LETTER TSWE
 [0xA690] = 0xA691, -- CYRILLIC CAPITAL LETTER TSSE
 [0xA691] = {status="valid"}, -- CYRILLIC SMALL LETTER TSSE
 [0xA692] = 0xA693, -- CYRILLIC CAPITAL LETTER TCHE
 [0xA693] = {status="valid"}, -- CYRILLIC SMALL LETTER TCHE
 [0xA694] = 0xA695, -- CYRILLIC CAPITAL LETTER HWE
 [0xA695] = {status="valid"}, -- CYRILLIC SMALL LETTER HWE
 [0xA696] = 0xA697, -- CYRILLIC CAPITAL LETTER SHWE
 [0xA697] = {status="valid"}, -- CYRILLIC SMALL LETTER SHWE
 [0xA698] = 0xA699, -- CYRILLIC CAPITAL LETTER DOUBLE O
 [0xA699] = {status="valid"}, -- CYRILLIC SMALL LETTER DOUBLE O
 [0xA69A] = 0xA69B, -- CYRILLIC CAPITAL LETTER CROSSED O
 [0xA69B] = {status="valid"}, -- CYRILLIC SMALL LETTER CROSSED O
 [0xA69C] = 0x044A, -- MODIFIER LETTER CYRILLIC HARD SIGN
 [0xA69D] = 0x044C, -- MODIFIER LETTER CYRILLIC SOFT SIGN
 [0xA69E] = {status="valid"}, -- COMBINING CYRILLIC LETTER EF
 [0xA69F] = {status="valid"}, -- COMBINING CYRILLIC LETTER IOTIFIED E
 -- BAMUM COMBINING MARK KOQNDON..BAMUM COMBINING MARK TUKWENTIS
 [0xA6F0] = {status="valid"},
 [0xA6F1] = {status="valid"},
 [0xA722] = 0xA723, -- LATIN CAPITAL LETTER EGYPTOLOGICAL ALEF
 [0xA723] = {status="valid"}, -- LATIN SMALL LETTER EGYPTOLOGICAL ALEF
 [0xA724] = 0xA725, -- LATIN CAPITAL LETTER EGYPTOLOGICAL AIN
 [0xA725] = {status="valid"}, -- LATIN SMALL LETTER EGYPTOLOGICAL AIN
 [0xA726] = 0xA727, -- LATIN CAPITAL LETTER HENG
 [0xA727] = {status="valid"}, -- LATIN SMALL LETTER HENG
 [0xA728] = 0xA729, -- LATIN CAPITAL LETTER TZ
 [0xA729] = {status="valid"}, -- LATIN SMALL LETTER TZ
 [0xA72A] = 0xA72B, -- LATIN CAPITAL LETTER TRESILLO
 [0xA72B] = {status="valid"}, -- LATIN SMALL LETTER TRESILLO
 [0xA72C] = 0xA72D, -- LATIN CAPITAL LETTER CUATRILLO
 [0xA72D] = {status="valid"}, -- LATIN SMALL LETTER CUATRILLO
 [0xA72E] = 0xA72F, -- LATIN CAPITAL LETTER CUATRILLO WITH COMMA
 -- MODIFIER LETTER STRESS AND HIGH TONE..MODIFIER LETTER STRESS AND LOW TONE
 [0xA720] = {status="disallowed"},
 [0xA721] = {status="disallowed"},
 [0xA732] = 0xA733, -- LATIN CAPITAL LETTER AA
 [0xA733] = {status="valid"}, -- LATIN SMALL LETTER AA
 [0xA734] = 0xA735, -- LATIN CAPITAL LETTER AO
 [0xA735] = {status="valid"}, -- LATIN SMALL LETTER AO
 [0xA736] = 0xA737, -- LATIN CAPITAL LETTER AU
 [0xA737] = {status="valid"}, -- LATIN SMALL LETTER AU
 [0xA738] = 0xA739, -- LATIN CAPITAL LETTER AV
 [0xA739] = {status="valid"}, -- LATIN SMALL LETTER AV
 [0xA73A] = 0xA73B, -- LATIN CAPITAL LETTER AV WITH HORIZONTAL BAR
 [0xA73B] = {status="valid"}, -- LATIN SMALL LETTER AV WITH HORIZONTAL BAR
 [0xA73C] = 0xA73D, -- LATIN CAPITAL LETTER AY
 [0xA73D] = {status="valid"}, -- LATIN SMALL LETTER AY
 [0xA73E] = 0xA73F, -- LATIN CAPITAL LETTER REVERSED C WITH DOT
 [0xA73F] = {status="valid"}, -- LATIN SMALL LETTER REVERSED C WITH DOT
 [0xA740] = 0xA741, -- LATIN CAPITAL LETTER K WITH STROKE
 [0xA741] = {status="valid"}, -- LATIN SMALL LETTER K WITH STROKE
 [0xA742] = 0xA743, -- LATIN CAPITAL LETTER K WITH DIAGONAL STROKE
 [0xA743] = {status="valid"}, -- LATIN SMALL LETTER K WITH DIAGONAL STROKE
 [0xA744] = 0xA745, -- LATIN CAPITAL LETTER K WITH STROKE AND DIAGONAL STROKE
 [0xA745] = {status="valid"}, -- LATIN SMALL LETTER K WITH STROKE AND DIAGONAL STROKE
 [0xA746] = 0xA747, -- LATIN CAPITAL LETTER BROKEN L
 [0xA747] = {status="valid"}, -- LATIN SMALL LETTER BROKEN L
 [0xA748] = 0xA749, -- LATIN CAPITAL LETTER L WITH HIGH STROKE
 [0xA749] = {status="valid"}, -- LATIN SMALL LETTER L WITH HIGH STROKE
 [0xA74A] = 0xA74B, -- LATIN CAPITAL LETTER O WITH LONG STROKE OVERLAY
 [0xA74B] = {status="valid"}, -- LATIN SMALL LETTER O WITH LONG STROKE OVERLAY
 [0xA74C] = 0xA74D, -- LATIN CAPITAL LETTER O WITH LOOP
 [0xA74D] = {status="valid"}, -- LATIN SMALL LETTER O WITH LOOP
 [0xA74E] = 0xA74F, -- LATIN CAPITAL LETTER OO
 [0xA74F] = {status="valid"}, -- LATIN SMALL LETTER OO
 [0xA750] = 0xA751, -- LATIN CAPITAL LETTER P WITH STROKE THROUGH DESCENDER
 [0xA751] = {status="valid"}, -- LATIN SMALL LETTER P WITH STROKE THROUGH DESCENDER
 [0xA752] = 0xA753, -- LATIN CAPITAL LETTER P WITH FLOURISH
 [0xA753] = {status="valid"}, -- LATIN SMALL LETTER P WITH FLOURISH
 [0xA754] = 0xA755, -- LATIN CAPITAL LETTER P WITH SQUIRREL TAIL
 [0xA755] = {status="valid"}, -- LATIN SMALL LETTER P WITH SQUIRREL TAIL
 [0xA756] = 0xA757, -- LATIN CAPITAL LETTER Q WITH STROKE THROUGH DESCENDER
 [0xA757] = {status="valid"}, -- LATIN SMALL LETTER Q WITH STROKE THROUGH DESCENDER
 [0xA758] = 0xA759, -- LATIN CAPITAL LETTER Q WITH DIAGONAL STROKE
 [0xA759] = {status="valid"}, -- LATIN SMALL LETTER Q WITH DIAGONAL STROKE
 [0xA75A] = 0xA75B, -- LATIN CAPITAL LETTER R ROTUNDA
 [0xA75B] = {status="valid"}, -- LATIN SMALL LETTER R ROTUNDA
 [0xA75C] = 0xA75D, -- LATIN CAPITAL LETTER RUM ROTUNDA
 [0xA75D] = {status="valid"}, -- LATIN SMALL LETTER RUM ROTUNDA
 [0xA75E] = 0xA75F, -- LATIN CAPITAL LETTER V WITH DIAGONAL STROKE
 [0xA75F] = {status="valid"}, -- LATIN SMALL LETTER V WITH DIAGONAL STROKE
 [0xA760] = 0xA761, -- LATIN CAPITAL LETTER VY
 [0xA761] = {status="valid"}, -- LATIN SMALL LETTER VY
 [0xA762] = 0xA763, -- LATIN CAPITAL LETTER VISIGOTHIC Z
 [0xA763] = {status="valid"}, -- LATIN SMALL LETTER VISIGOTHIC Z
 [0xA764] = 0xA765, -- LATIN CAPITAL LETTER THORN WITH STROKE
 [0xA765] = {status="valid"}, -- LATIN SMALL LETTER THORN WITH STROKE
 [0xA766] = 0xA767, -- LATIN CAPITAL LETTER THORN WITH STROKE THROUGH DESCENDER
 [0xA767] = {status="valid"}, -- LATIN SMALL LETTER THORN WITH STROKE THROUGH DESCENDER
 [0xA768] = 0xA769, -- LATIN CAPITAL LETTER VEND
 [0xA769] = {status="valid"}, -- LATIN SMALL LETTER VEND
 [0xA76A] = 0xA76B, -- LATIN CAPITAL LETTER ET
 [0xA76B] = {status="valid"}, -- LATIN SMALL LETTER ET
 [0xA76C] = 0xA76D, -- LATIN CAPITAL LETTER IS
 [0xA76D] = {status="valid"}, -- LATIN SMALL LETTER IS
 [0xA76E] = 0xA76F, -- LATIN CAPITAL LETTER CON
 [0xA76F] = {status="valid"}, -- LATIN SMALL LETTER CON
 [0xA770] = 0xA76F, -- MODIFIER LETTER US
 -- LATIN SMALL LETTER CUATRILLO WITH COMMA..LATIN LETTER SMALL CAPITAL S
 [0xA72F] = {status="valid"},
 [0xA730] = {status="valid"},
 [0xA731] = {status="valid"},
 [0xA779] = 0xA77A, -- LATIN CAPITAL LETTER INSULAR D
 [0xA77A] = {status="valid"}, -- LATIN SMALL LETTER INSULAR D
 [0xA77B] = 0xA77C, -- LATIN CAPITAL LETTER INSULAR F
 [0xA77C] = {status="valid"}, -- LATIN SMALL LETTER INSULAR F
 [0xA77D] = 0x1D79, -- LATIN CAPITAL LETTER INSULAR G
 [0xA77E] = 0xA77F, -- LATIN CAPITAL LETTER TURNED INSULAR G
 [0xA77F] = {status="valid"}, -- LATIN SMALL LETTER TURNED INSULAR G
 [0xA780] = 0xA781, -- LATIN CAPITAL LETTER TURNED L
 [0xA781] = {status="valid"}, -- LATIN SMALL LETTER TURNED L
 [0xA782] = 0xA783, -- LATIN CAPITAL LETTER INSULAR R
 [0xA783] = {status="valid"}, -- LATIN SMALL LETTER INSULAR R
 [0xA784] = 0xA785, -- LATIN CAPITAL LETTER INSULAR S
 [0xA785] = {status="valid"}, -- LATIN SMALL LETTER INSULAR S
 [0xA786] = 0xA787, -- LATIN CAPITAL LETTER INSULAR T
 -- LATIN SMALL LETTER INSULAR T..MODIFIER LETTER LOW CIRCUMFLEX ACCENT
 [0xA787] = {status="valid"},
 [0xA788] = {status="valid"},
 [0xA78B] = 0xA78C, -- LATIN CAPITAL LETTER SALTILLO
 [0xA78C] = {status="valid"}, -- LATIN SMALL LETTER SALTILLO
 [0xA78D] = 0x0265, -- LATIN CAPITAL LETTER TURNED H
 [0xA78E] = {status="valid"}, -- LATIN SMALL LETTER L WITH RETROFLEX HOOK AND BELT
 [0xA78F] = {status="valid"}, -- LATIN LETTER SINOLOGICAL DOT
 [0xA790] = 0xA791, -- LATIN CAPITAL LETTER N WITH DESCENDER
 [0xA791] = {status="valid"}, -- LATIN SMALL LETTER N WITH DESCENDER
 [0xA792] = 0xA793, -- LATIN CAPITAL LETTER C WITH BAR
 [0xA793] = {status="valid"}, -- LATIN SMALL LETTER C WITH BAR
 -- MODIFIER LETTER COLON..MODIFIER LETTER SHORT EQUALS SIGN
 [0xA789] = {status="disallowed"},
 [0xA78A] = {status="disallowed"},
 [0xA796] = 0xA797, -- LATIN CAPITAL LETTER B WITH FLOURISH
 [0xA797] = {status="valid"}, -- LATIN SMALL LETTER B WITH FLOURISH
 [0xA798] = 0xA799, -- LATIN CAPITAL LETTER F WITH STROKE
 [0xA799] = {status="valid"}, -- LATIN SMALL LETTER F WITH STROKE
 [0xA79A] = 0xA79B, -- LATIN CAPITAL LETTER VOLAPUK AE
 [0xA79B] = {status="valid"}, -- LATIN SMALL LETTER VOLAPUK AE
 [0xA79C] = 0xA79D, -- LATIN CAPITAL LETTER VOLAPUK OE
 [0xA79D] = {status="valid"}, -- LATIN SMALL LETTER VOLAPUK OE
 [0xA79E] = 0xA79F, -- LATIN CAPITAL LETTER VOLAPUK UE
 [0xA79F] = {status="valid"}, -- LATIN SMALL LETTER VOLAPUK UE
 [0xA7A0] = 0xA7A1, -- LATIN CAPITAL LETTER G WITH OBLIQUE STROKE
 [0xA7A1] = {status="valid"}, -- LATIN SMALL LETTER G WITH OBLIQUE STROKE
 [0xA7A2] = 0xA7A3, -- LATIN CAPITAL LETTER K WITH OBLIQUE STROKE
 [0xA7A3] = {status="valid"}, -- LATIN SMALL LETTER K WITH OBLIQUE STROKE
 [0xA7A4] = 0xA7A5, -- LATIN CAPITAL LETTER N WITH OBLIQUE STROKE
 [0xA7A5] = {status="valid"}, -- LATIN SMALL LETTER N WITH OBLIQUE STROKE
 [0xA7A6] = 0xA7A7, -- LATIN CAPITAL LETTER R WITH OBLIQUE STROKE
 [0xA7A7] = {status="valid"}, -- LATIN SMALL LETTER R WITH OBLIQUE STROKE
 [0xA7A8] = 0xA7A9, -- LATIN CAPITAL LETTER S WITH OBLIQUE STROKE
 [0xA7A9] = {status="valid"}, -- LATIN SMALL LETTER S WITH OBLIQUE STROKE
 [0xA7AA] = 0x0266, -- LATIN CAPITAL LETTER H WITH HOOK
 [0xA7AB] = 0x025C, -- LATIN CAPITAL LETTER REVERSED OPEN E
 [0xA7AC] = 0x0261, -- LATIN CAPITAL LETTER SCRIPT G
 [0xA7AD] = 0x026C, -- LATIN CAPITAL LETTER L WITH BELT
 [0xA7AE] = 0x026A, -- LATIN CAPITAL LETTER SMALL CAPITAL I
 [0xA7AF] = {status="disallowed"}, -- NA   <reserved-A7AF>
 [0xA7B0] = 0x029E, -- LATIN CAPITAL LETTER TURNED K
 [0xA7B1] = 0x0287, -- LATIN CAPITAL LETTER TURNED T
 [0xA7B2] = 0x029D, -- LATIN CAPITAL LETTER J WITH CROSSED-TAIL
 [0xA7B3] = 0xAB53, -- LATIN CAPITAL LETTER CHI
 [0xA7B4] = 0xA7B5, -- LATIN CAPITAL LETTER BETA
 [0xA7B5] = {status="valid"}, -- LATIN SMALL LETTER BETA
 [0xA7B6] = 0xA7B7, -- LATIN CAPITAL LETTER OMEGA
 [0xA7B7] = {status="valid"}, -- LATIN SMALL LETTER OMEGA
 -- LATIN SMALL LETTER C WITH PALATAL HOOK..LATIN SMALL LETTER H WITH PALATAL HOOK
 [0xA794] = {status="valid"},
 [0xA795] = {status="valid"},
 [0xA7F7] = {status="valid"}, -- LATIN EPIGRAPHIC LETTER SIDEWAYS I
 [0xA7F8] = 0x0127, -- MODIFIER LETTER CAPITAL H WITH STROKE
 [0xA7F9] = 0x0153, -- MODIFIER LETTER SMALL LIGATURE OE
 [0xA7FA] = {status="valid"}, -- LATIN LETTER SMALL CAPITAL TURNED M
 -- SYLOTI NAGRI POETRY MARK-1..SYLOTI NAGRI POETRY MARK-4
 [0xA828] = {status="disallowed"},
 [0xA829] = {status="disallowed"},
 [0xA82A] = {status="disallowed"},
 [0xA82B] = {status="disallowed"},
 -- NA   <reserved-A82C>..<reserved-A82F>
 [0xA82C] = {status="disallowed"},
 [0xA82D] = {status="disallowed"},
 [0xA82E] = {status="disallowed"},
 [0xA82F] = {status="disallowed"},
 -- PHAGS-PA SINGLE HEAD MARK..PHAGS-PA MARK DOUBLE SHAD
 [0xA874] = {status="disallowed"},
 [0xA875] = {status="disallowed"},
 [0xA876] = {status="disallowed"},
 [0xA877] = {status="disallowed"},
 [0xA8C5] = {status="valid"}, -- SAURASHTRA SIGN CANDRABINDU
 -- SAURASHTRA DANDA..SAURASHTRA DOUBLE DANDA
 [0xA8CE] = {status="disallowed"},
 [0xA8CF] = {status="disallowed"},
 [0xA8FB] = {status="valid"}, -- DEVANAGARI HEADSTROKE
 [0xA8FC] = {status="disallowed"}, -- DEVANAGARI SIGN SIDDHAM
 [0xA8FD] = {status="valid"}, -- DEVANAGARI JAIN OM
 -- DEVANAGARI SIGN PUSHPIKA..DEVANAGARI CARET
 [0xA8F8] = {status="disallowed"},
 [0xA8F9] = {status="disallowed"},
 [0xA8FA] = {status="disallowed"},
 -- NA   <reserved-A8FE>..<reserved-A8FF>
 [0xA8FE] = {status="disallowed"},
 [0xA8FF] = {status="disallowed"},
 -- KAYAH LI SIGN CWI..KAYAH LI SIGN SHYA
 [0xA92E] = {status="disallowed"},
 [0xA92F] = {status="disallowed"},
 [0xA95F] = {status="disallowed"}, -- REJANG SECTION MARK
 -- NA   <reserved-A97D>..<reserved-A97F>
 [0xA97D] = {status="disallowed"},
 [0xA97E] = {status="disallowed"},
 [0xA97F] = {status="disallowed"},
 [0xA9CE] = {status="disallowed"}, -- NA   <reserved-A9CE>
 -- NA   <reserved-A9DA>..<reserved-A9DD>
 [0xA9DA] = {status="disallowed"},
 [0xA9DB] = {status="disallowed"},
 [0xA9DC] = {status="disallowed"},
 [0xA9DD] = {status="disallowed"},
 -- JAVANESE PADA TIRTA TUMETES..JAVANESE PADA ISEN-ISEN
 [0xA9DE] = {status="disallowed"},
 [0xA9DF] = {status="disallowed"},
 [0xA9FF] = {status="disallowed"}, -- NA   <reserved-A9FF>
 -- NA   <reserved-AA4E>..<reserved-AA4F>
 [0xAA4E] = {status="disallowed"},
 [0xAA4F] = {status="disallowed"},
 -- NA   <reserved-AA5A>..<reserved-AA5B>
 [0xAA5A] = {status="disallowed"},
 [0xAA5B] = {status="disallowed"},
 -- CHAM PUNCTUATION SPIRAL..CHAM PUNCTUATION TRIPLE DANDA
 [0xAA5C] = {status="disallowed"},
 [0xAA5D] = {status="disallowed"},
 [0xAA5E] = {status="disallowed"},
 [0xAA5F] = {status="disallowed"},
 -- MYANMAR SYMBOL AITON EXCLAMATION..MYANMAR SYMBOL AITON TWO
 [0xAA77] = {status="disallowed"},
 [0xAA78] = {status="disallowed"},
 [0xAA79] = {status="disallowed"},
 -- TAI VIET SYMBOL KON..TAI VIET SYMBOL SAM
 [0xAADB] = {status="valid"},
 [0xAADC] = {status="valid"},
 [0xAADD] = {status="valid"},
 -- TAI VIET SYMBOL HO HOI..TAI VIET SYMBOL KOI KOI
 [0xAADE] = {status="disallowed"},
 [0xAADF] = {status="disallowed"},
 -- MEETEI MAYEK CHEIKHAN..MEETEI MAYEK AHANG KHUDAM
 [0xAAF0] = {status="disallowed"},
 [0xAAF1] = {status="disallowed"},
 -- MEETEI MAYEK ANJI..MEETEI MAYEK VIRAMA
 [0xAAF2] = {status="valid"},
 [0xAAF3] = {status="valid"},
 [0xAAF4] = {status="valid"},
 [0xAAF5] = {status="valid"},
 [0xAAF6] = {status="valid"},
 -- NA   <reserved-AB07>..<reserved-AB08>
 [0xAB07] = {status="disallowed"},
 [0xAB08] = {status="disallowed"},
 -- NA   <reserved-AB0F>..<reserved-AB10>
 [0xAB0F] = {status="disallowed"},
 [0xAB10] = {status="disallowed"},
 [0xAB27] = {status="disallowed"}, -- NA   <reserved-AB27>
 [0xAB2F] = {status="disallowed"}, -- NA   <reserved-AB2F>
 [0xAB5B] = {status="disallowed"}, -- MODIFIER BREVE WITH INVERTED BREVE
 [0xAB5C] = 0xA727, -- MODIFIER LETTER SMALL HENG
 [0xAB5D] = 0xAB37, -- MODIFIER LETTER SMALL L WITH INVERTED LAZY S
 [0xAB5E] = 0x026B, -- MODIFIER LETTER SMALL L WITH MIDDLE TILDE
 [0xAB5F] = 0xAB52, -- MODIFIER LETTER SMALL U WITH LEFT HOOK
 [0xAB70] = 0x13A0, -- CHEROKEE SMALL LETTER A
 [0xAB71] = 0x13A1, -- CHEROKEE SMALL LETTER E
 [0xAB72] = 0x13A2, -- CHEROKEE SMALL LETTER I
 [0xAB73] = 0x13A3, -- CHEROKEE SMALL LETTER O
 [0xAB74] = 0x13A4, -- CHEROKEE SMALL LETTER U
 [0xAB75] = 0x13A5, -- CHEROKEE SMALL LETTER V
 [0xAB76] = 0x13A6, -- CHEROKEE SMALL LETTER GA
 [0xAB77] = 0x13A7, -- CHEROKEE SMALL LETTER KA
 [0xAB78] = 0x13A8, -- CHEROKEE SMALL LETTER GE
 [0xAB79] = 0x13A9, -- CHEROKEE SMALL LETTER GI
 [0xAB7A] = 0x13AA, -- CHEROKEE SMALL LETTER GO
 [0xAB7B] = 0x13AB, -- CHEROKEE SMALL LETTER GU
 [0xAB7C] = 0x13AC, -- CHEROKEE SMALL LETTER GV
 [0xAB7D] = 0x13AD, -- CHEROKEE SMALL LETTER HA
 [0xAB7E] = 0x13AE, -- CHEROKEE SMALL LETTER HE
 [0xAB7F] = 0x13AF, -- CHEROKEE SMALL LETTER HI
 [0xAB80] = 0x13B0, -- CHEROKEE SMALL LETTER HO
 [0xAB81] = 0x13B1, -- CHEROKEE SMALL LETTER HU
 [0xAB82] = 0x13B2, -- CHEROKEE SMALL LETTER HV
 [0xAB83] = 0x13B3, -- CHEROKEE SMALL LETTER LA
 [0xAB84] = 0x13B4, -- CHEROKEE SMALL LETTER LE
 [0xAB85] = 0x13B5, -- CHEROKEE SMALL LETTER LI
 [0xAB86] = 0x13B6, -- CHEROKEE SMALL LETTER LO
 [0xAB87] = 0x13B7, -- CHEROKEE SMALL LETTER LU
 [0xAB88] = 0x13B8, -- CHEROKEE SMALL LETTER LV
 [0xAB89] = 0x13B9, -- CHEROKEE SMALL LETTER MA
 [0xAB8A] = 0x13BA, -- CHEROKEE SMALL LETTER ME
 [0xAB8B] = 0x13BB, -- CHEROKEE SMALL LETTER MI
 [0xAB8C] = 0x13BC, -- CHEROKEE SMALL LETTER MO
 [0xAB8D] = 0x13BD, -- CHEROKEE SMALL LETTER MU
 [0xAB8E] = 0x13BE, -- CHEROKEE SMALL LETTER NA
 [0xAB8F] = 0x13BF, -- CHEROKEE SMALL LETTER HNA
 [0xAB90] = 0x13C0, -- CHEROKEE SMALL LETTER NAH
 [0xAB91] = 0x13C1, -- CHEROKEE SMALL LETTER NE
 [0xAB92] = 0x13C2, -- CHEROKEE SMALL LETTER NI
 [0xAB93] = 0x13C3, -- CHEROKEE SMALL LETTER NO
 [0xAB94] = 0x13C4, -- CHEROKEE SMALL LETTER NU
 [0xAB95] = 0x13C5, -- CHEROKEE SMALL LETTER NV
 [0xAB96] = 0x13C6, -- CHEROKEE SMALL LETTER QUA
 [0xAB97] = 0x13C7, -- CHEROKEE SMALL LETTER QUE
 [0xAB98] = 0x13C8, -- CHEROKEE SMALL LETTER QUI
 [0xAB99] = 0x13C9, -- CHEROKEE SMALL LETTER QUO
 [0xAB9A] = 0x13CA, -- CHEROKEE SMALL LETTER QUU
 [0xAB9B] = 0x13CB, -- CHEROKEE SMALL LETTER QUV
 [0xAB9C] = 0x13CC, -- CHEROKEE SMALL LETTER SA
 [0xAB9D] = 0x13CD, -- CHEROKEE SMALL LETTER S
 [0xAB9E] = 0x13CE, -- CHEROKEE SMALL LETTER SE
 [0xAB9F] = 0x13CF, -- CHEROKEE SMALL LETTER SI
 [0xABA0] = 0x13D0, -- CHEROKEE SMALL LETTER SO
 [0xABA1] = 0x13D1, -- CHEROKEE SMALL LETTER SU
 [0xABA2] = 0x13D2, -- CHEROKEE SMALL LETTER SV
 [0xABA3] = 0x13D3, -- CHEROKEE SMALL LETTER DA
 [0xABA4] = 0x13D4, -- CHEROKEE SMALL LETTER TA
 [0xABA5] = 0x13D5, -- CHEROKEE SMALL LETTER DE
 [0xABA6] = 0x13D6, -- CHEROKEE SMALL LETTER TE
 [0xABA7] = 0x13D7, -- CHEROKEE SMALL LETTER DI
 [0xABA8] = 0x13D8, -- CHEROKEE SMALL LETTER TI
 [0xABA9] = 0x13D9, -- CHEROKEE SMALL LETTER DO
 [0xABAA] = 0x13DA, -- CHEROKEE SMALL LETTER DU
 [0xABAB] = 0x13DB, -- CHEROKEE SMALL LETTER DV
 [0xABAC] = 0x13DC, -- CHEROKEE SMALL LETTER DLA
 [0xABAD] = 0x13DD, -- CHEROKEE SMALL LETTER TLA
 [0xABAE] = 0x13DE, -- CHEROKEE SMALL LETTER TLE
 [0xABAF] = 0x13DF, -- CHEROKEE SMALL LETTER TLI
 [0xABB0] = 0x13E0, -- CHEROKEE SMALL LETTER TLO
 [0xABB1] = 0x13E1, -- CHEROKEE SMALL LETTER TLU
 [0xABB2] = 0x13E2, -- CHEROKEE SMALL LETTER TLV
 [0xABB3] = 0x13E3, -- CHEROKEE SMALL LETTER TSA
 [0xABB4] = 0x13E4, -- CHEROKEE SMALL LETTER TSE
 [0xABB5] = 0x13E5, -- CHEROKEE SMALL LETTER TSI
 [0xABB6] = 0x13E6, -- CHEROKEE SMALL LETTER TSO
 [0xABB7] = 0x13E7, -- CHEROKEE SMALL LETTER TSU
 [0xABB8] = 0x13E8, -- CHEROKEE SMALL LETTER TSV
 [0xABB9] = 0x13E9, -- CHEROKEE SMALL LETTER WA
 [0xABBA] = 0x13EA, -- CHEROKEE SMALL LETTER WE
 [0xABBB] = 0x13EB, -- CHEROKEE SMALL LETTER WI
 [0xABBC] = 0x13EC, -- CHEROKEE SMALL LETTER WO
 [0xABBD] = 0x13ED, -- CHEROKEE SMALL LETTER WU
 [0xABBE] = 0x13EE, -- CHEROKEE SMALL LETTER WV
 [0xABBF] = 0x13EF, -- CHEROKEE SMALL LETTER YA
 [0xABEB] = {status="disallowed"}, -- MEETEI MAYEK CHEIKHEI
 -- MEETEI MAYEK LUM IYEK..MEETEI MAYEK APUN IYEK
 [0xABEC] = {status="valid"},
 [0xABED] = {status="valid"},
 -- NA   <reserved-ABEE>..<reserved-ABEF>
 [0xABEE] = {status="disallowed"},
 [0xABEF] = {status="disallowed"},
 -- NA   <reserved-D7C7>..<reserved-D7CA>
 [0xD7C7] = {status="disallowed"},
 [0xD7C8] = {status="disallowed"},
 [0xD7C9] = {status="disallowed"},
 [0xD7CA] = {status="disallowed"},
 [0xF900] = 0x8C48, -- CJK COMPATIBILITY IDEOGRAPH-F900
 [0xF901] = 0x66F4, -- CJK COMPATIBILITY IDEOGRAPH-F901
 [0xF902] = 0x8ECA, -- CJK COMPATIBILITY IDEOGRAPH-F902
 [0xF903] = 0x8CC8, -- CJK COMPATIBILITY IDEOGRAPH-F903
 [0xF904] = 0x6ED1, -- CJK COMPATIBILITY IDEOGRAPH-F904
 [0xF905] = 0x4E32, -- CJK COMPATIBILITY IDEOGRAPH-F905
 [0xF906] = 0x53E5, -- CJK COMPATIBILITY IDEOGRAPH-F906
 [0xF909] = 0x5951, -- CJK COMPATIBILITY IDEOGRAPH-F909
 [0xF90A] = 0x91D1, -- CJK COMPATIBILITY IDEOGRAPH-F90A
 [0xF90B] = 0x5587, -- CJK COMPATIBILITY IDEOGRAPH-F90B
 [0xF90C] = 0x5948, -- CJK COMPATIBILITY IDEOGRAPH-F90C
 [0xF90D] = 0x61F6, -- CJK COMPATIBILITY IDEOGRAPH-F90D
 [0xF90E] = 0x7669, -- CJK COMPATIBILITY IDEOGRAPH-F90E
 [0xF90F] = 0x7F85, -- CJK COMPATIBILITY IDEOGRAPH-F90F
 [0xF910] = 0x863F, -- CJK COMPATIBILITY IDEOGRAPH-F910
 [0xF911] = 0x87BA, -- CJK COMPATIBILITY IDEOGRAPH-F911
 [0xF912] = 0x88F8, -- CJK COMPATIBILITY IDEOGRAPH-F912
 [0xF913] = 0x908F, -- CJK COMPATIBILITY IDEOGRAPH-F913
 [0xF914] = 0x6A02, -- CJK COMPATIBILITY IDEOGRAPH-F914
 [0xF915] = 0x6D1B, -- CJK COMPATIBILITY IDEOGRAPH-F915
 [0xF916] = 0x70D9, -- CJK COMPATIBILITY IDEOGRAPH-F916
 [0xF917] = 0x73DE, -- CJK COMPATIBILITY IDEOGRAPH-F917
 [0xF918] = 0x843D, -- CJK COMPATIBILITY IDEOGRAPH-F918
 [0xF919] = 0x916A, -- CJK COMPATIBILITY IDEOGRAPH-F919
 [0xF91A] = 0x99F1, -- CJK COMPATIBILITY IDEOGRAPH-F91A
 [0xF91B] = 0x4E82, -- CJK COMPATIBILITY IDEOGRAPH-F91B
 [0xF91C] = 0x5375, -- CJK COMPATIBILITY IDEOGRAPH-F91C
 [0xF91D] = 0x6B04, -- CJK COMPATIBILITY IDEOGRAPH-F91D
 [0xF91E] = 0x721B, -- CJK COMPATIBILITY IDEOGRAPH-F91E
 [0xF91F] = 0x862D, -- CJK COMPATIBILITY IDEOGRAPH-F91F
 [0xF920] = 0x9E1E, -- CJK COMPATIBILITY IDEOGRAPH-F920
 [0xF921] = 0x5D50, -- CJK COMPATIBILITY IDEOGRAPH-F921
 [0xF922] = 0x6FEB, -- CJK COMPATIBILITY IDEOGRAPH-F922
 [0xF923] = 0x85CD, -- CJK COMPATIBILITY IDEOGRAPH-F923
 [0xF924] = 0x8964, -- CJK COMPATIBILITY IDEOGRAPH-F924
 [0xF925] = 0x62C9, -- CJK COMPATIBILITY IDEOGRAPH-F925
 [0xF926] = 0x81D8, -- CJK COMPATIBILITY IDEOGRAPH-F926
 [0xF927] = 0x881F, -- CJK COMPATIBILITY IDEOGRAPH-F927
 [0xF928] = 0x5ECA, -- CJK COMPATIBILITY IDEOGRAPH-F928
 [0xF929] = 0x6717, -- CJK COMPATIBILITY IDEOGRAPH-F929
 [0xF92A] = 0x6D6A, -- CJK COMPATIBILITY IDEOGRAPH-F92A
 [0xF92B] = 0x72FC, -- CJK COMPATIBILITY IDEOGRAPH-F92B
 [0xF92C] = 0x90CE, -- CJK COMPATIBILITY IDEOGRAPH-F92C
 [0xF92D] = 0x4F86, -- CJK COMPATIBILITY IDEOGRAPH-F92D
 [0xF92E] = 0x51B7, -- CJK COMPATIBILITY IDEOGRAPH-F92E
 [0xF92F] = 0x52DE, -- CJK COMPATIBILITY IDEOGRAPH-F92F
 [0xF930] = 0x64C4, -- CJK COMPATIBILITY IDEOGRAPH-F930
 [0xF931] = 0x6AD3, -- CJK COMPATIBILITY IDEOGRAPH-F931
 [0xF932] = 0x7210, -- CJK COMPATIBILITY IDEOGRAPH-F932
 [0xF933] = 0x76E7, -- CJK COMPATIBILITY IDEOGRAPH-F933
 [0xF934] = 0x8001, -- CJK COMPATIBILITY IDEOGRAPH-F934
 [0xF935] = 0x8606, -- CJK COMPATIBILITY IDEOGRAPH-F935
 [0xF936] = 0x865C, -- CJK COMPATIBILITY IDEOGRAPH-F936
 [0xF937] = 0x8DEF, -- CJK COMPATIBILITY IDEOGRAPH-F937
 [0xF938] = 0x9732, -- CJK COMPATIBILITY IDEOGRAPH-F938
 [0xF939] = 0x9B6F, -- CJK COMPATIBILITY IDEOGRAPH-F939
 [0xF93A] = 0x9DFA, -- CJK COMPATIBILITY IDEOGRAPH-F93A
 [0xF93B] = 0x788C, -- CJK COMPATIBILITY IDEOGRAPH-F93B
 [0xF93C] = 0x797F, -- CJK COMPATIBILITY IDEOGRAPH-F93C
 [0xF93D] = 0x7DA0, -- CJK COMPATIBILITY IDEOGRAPH-F93D
 [0xF93E] = 0x83C9, -- CJK COMPATIBILITY IDEOGRAPH-F93E
 [0xF93F] = 0x9304, -- CJK COMPATIBILITY IDEOGRAPH-F93F
 [0xF940] = 0x9E7F, -- CJK COMPATIBILITY IDEOGRAPH-F940
 [0xF941] = 0x8AD6, -- CJK COMPATIBILITY IDEOGRAPH-F941
 [0xF942] = 0x58DF, -- CJK COMPATIBILITY IDEOGRAPH-F942
 [0xF943] = 0x5F04, -- CJK COMPATIBILITY IDEOGRAPH-F943
 [0xF944] = 0x7C60, -- CJK COMPATIBILITY IDEOGRAPH-F944
 [0xF945] = 0x807E, -- CJK COMPATIBILITY IDEOGRAPH-F945
 [0xF946] = 0x7262, -- CJK COMPATIBILITY IDEOGRAPH-F946
 [0xF947] = 0x78CA, -- CJK COMPATIBILITY IDEOGRAPH-F947
 [0xF948] = 0x8CC2, -- CJK COMPATIBILITY IDEOGRAPH-F948
 [0xF949] = 0x96F7, -- CJK COMPATIBILITY IDEOGRAPH-F949
 [0xF94A] = 0x58D8, -- CJK COMPATIBILITY IDEOGRAPH-F94A
 [0xF94B] = 0x5C62, -- CJK COMPATIBILITY IDEOGRAPH-F94B
 [0xF94C] = 0x6A13, -- CJK COMPATIBILITY IDEOGRAPH-F94C
 [0xF94D] = 0x6DDA, -- CJK COMPATIBILITY IDEOGRAPH-F94D
 [0xF94E] = 0x6F0F, -- CJK COMPATIBILITY IDEOGRAPH-F94E
 [0xF94F] = 0x7D2F, -- CJK COMPATIBILITY IDEOGRAPH-F94F
 [0xF950] = 0x7E37, -- CJK COMPATIBILITY IDEOGRAPH-F950
 [0xF951] = 0x964B, -- CJK COMPATIBILITY IDEOGRAPH-F951
 [0xF952] = 0x52D2, -- CJK COMPATIBILITY IDEOGRAPH-F952
 [0xF953] = 0x808B, -- CJK COMPATIBILITY IDEOGRAPH-F953
 [0xF954] = 0x51DC, -- CJK COMPATIBILITY IDEOGRAPH-F954
 [0xF955] = 0x51CC, -- CJK COMPATIBILITY IDEOGRAPH-F955
 [0xF956] = 0x7A1C, -- CJK COMPATIBILITY IDEOGRAPH-F956
 [0xF957] = 0x7DBE, -- CJK COMPATIBILITY IDEOGRAPH-F957
 [0xF958] = 0x83F1, -- CJK COMPATIBILITY IDEOGRAPH-F958
 [0xF959] = 0x9675, -- CJK COMPATIBILITY IDEOGRAPH-F959
 [0xF95A] = 0x8B80, -- CJK COMPATIBILITY IDEOGRAPH-F95A
 [0xF95B] = 0x62CF, -- CJK COMPATIBILITY IDEOGRAPH-F95B
 [0xF95C] = 0x6A02, -- CJK COMPATIBILITY IDEOGRAPH-F95C
 [0xF95D] = 0x8AFE, -- CJK COMPATIBILITY IDEOGRAPH-F95D
 [0xF95E] = 0x4E39, -- CJK COMPATIBILITY IDEOGRAPH-F95E
 [0xF95F] = 0x5BE7, -- CJK COMPATIBILITY IDEOGRAPH-F95F
 [0xF960] = 0x6012, -- CJK COMPATIBILITY IDEOGRAPH-F960
 [0xF961] = 0x7387, -- CJK COMPATIBILITY IDEOGRAPH-F961
 [0xF962] = 0x7570, -- CJK COMPATIBILITY IDEOGRAPH-F962
 [0xF963] = 0x5317, -- CJK COMPATIBILITY IDEOGRAPH-F963
 [0xF964] = 0x78FB, -- CJK COMPATIBILITY IDEOGRAPH-F964
 [0xF965] = 0x4FBF, -- CJK COMPATIBILITY IDEOGRAPH-F965
 [0xF966] = 0x5FA9, -- CJK COMPATIBILITY IDEOGRAPH-F966
 [0xF967] = 0x4E0D, -- CJK COMPATIBILITY IDEOGRAPH-F967
 [0xF968] = 0x6CCC, -- CJK COMPATIBILITY IDEOGRAPH-F968
 [0xF969] = 0x6578, -- CJK COMPATIBILITY IDEOGRAPH-F969
 [0xF96A] = 0x7D22, -- CJK COMPATIBILITY IDEOGRAPH-F96A
 [0xF96B] = 0x53C3, -- CJK COMPATIBILITY IDEOGRAPH-F96B
 [0xF96C] = 0x585E, -- CJK COMPATIBILITY IDEOGRAPH-F96C
 [0xF96D] = 0x7701, -- CJK COMPATIBILITY IDEOGRAPH-F96D
 [0xF96E] = 0x8449, -- CJK COMPATIBILITY IDEOGRAPH-F96E
 [0xF96F] = 0x8AAA, -- CJK COMPATIBILITY IDEOGRAPH-F96F
 [0xF970] = 0x6BBA, -- CJK COMPATIBILITY IDEOGRAPH-F970
 [0xF971] = 0x8FB0, -- CJK COMPATIBILITY IDEOGRAPH-F971
 [0xF972] = 0x6C88, -- CJK COMPATIBILITY IDEOGRAPH-F972
 [0xF973] = 0x62FE, -- CJK COMPATIBILITY IDEOGRAPH-F973
 [0xF974] = 0x82E5, -- CJK COMPATIBILITY IDEOGRAPH-F974
 [0xF975] = 0x63A0, -- CJK COMPATIBILITY IDEOGRAPH-F975
 [0xF976] = 0x7565, -- CJK COMPATIBILITY IDEOGRAPH-F976
 [0xF977] = 0x4EAE, -- CJK COMPATIBILITY IDEOGRAPH-F977
 [0xF978] = 0x5169, -- CJK COMPATIBILITY IDEOGRAPH-F978
 [0xF979] = 0x51C9, -- CJK COMPATIBILITY IDEOGRAPH-F979
 [0xF97A] = 0x6881, -- CJK COMPATIBILITY IDEOGRAPH-F97A
 [0xF97B] = 0x7CE7, -- CJK COMPATIBILITY IDEOGRAPH-F97B
 [0xF97C] = 0x826F, -- CJK COMPATIBILITY IDEOGRAPH-F97C
 [0xF97D] = 0x8AD2, -- CJK COMPATIBILITY IDEOGRAPH-F97D
 [0xF97E] = 0x91CF, -- CJK COMPATIBILITY IDEOGRAPH-F97E
 [0xF97F] = 0x52F5, -- CJK COMPATIBILITY IDEOGRAPH-F97F
 [0xF980] = 0x5442, -- CJK COMPATIBILITY IDEOGRAPH-F980
 [0xF981] = 0x5973, -- CJK COMPATIBILITY IDEOGRAPH-F981
 [0xF982] = 0x5EEC, -- CJK COMPATIBILITY IDEOGRAPH-F982
 [0xF983] = 0x65C5, -- CJK COMPATIBILITY IDEOGRAPH-F983
 [0xF984] = 0x6FFE, -- CJK COMPATIBILITY IDEOGRAPH-F984
 [0xF985] = 0x792A, -- CJK COMPATIBILITY IDEOGRAPH-F985
 [0xF986] = 0x95AD, -- CJK COMPATIBILITY IDEOGRAPH-F986
 [0xF987] = 0x9A6A, -- CJK COMPATIBILITY IDEOGRAPH-F987
 [0xF988] = 0x9E97, -- CJK COMPATIBILITY IDEOGRAPH-F988
 [0xF989] = 0x9ECE, -- CJK COMPATIBILITY IDEOGRAPH-F989
 [0xF98A] = 0x529B, -- CJK COMPATIBILITY IDEOGRAPH-F98A
 [0xF98B] = 0x66C6, -- CJK COMPATIBILITY IDEOGRAPH-F98B
 [0xF98C] = 0x6B77, -- CJK COMPATIBILITY IDEOGRAPH-F98C
 [0xF98D] = 0x8F62, -- CJK COMPATIBILITY IDEOGRAPH-F98D
 [0xF98E] = 0x5E74, -- CJK COMPATIBILITY IDEOGRAPH-F98E
 [0xF98F] = 0x6190, -- CJK COMPATIBILITY IDEOGRAPH-F98F
 [0xF990] = 0x6200, -- CJK COMPATIBILITY IDEOGRAPH-F990
 [0xF991] = 0x649A, -- CJK COMPATIBILITY IDEOGRAPH-F991
 [0xF992] = 0x6F23, -- CJK COMPATIBILITY IDEOGRAPH-F992
 [0xF993] = 0x7149, -- CJK COMPATIBILITY IDEOGRAPH-F993
 [0xF994] = 0x7489, -- CJK COMPATIBILITY IDEOGRAPH-F994
 [0xF995] = 0x79CA, -- CJK COMPATIBILITY IDEOGRAPH-F995
 [0xF996] = 0x7DF4, -- CJK COMPATIBILITY IDEOGRAPH-F996
 [0xF997] = 0x806F, -- CJK COMPATIBILITY IDEOGRAPH-F997
 [0xF998] = 0x8F26, -- CJK COMPATIBILITY IDEOGRAPH-F998
 [0xF999] = 0x84EE, -- CJK COMPATIBILITY IDEOGRAPH-F999
 [0xF99A] = 0x9023, -- CJK COMPATIBILITY IDEOGRAPH-F99A
 [0xF99B] = 0x934A, -- CJK COMPATIBILITY IDEOGRAPH-F99B
 [0xF99C] = 0x5217, -- CJK COMPATIBILITY IDEOGRAPH-F99C
 [0xF99D] = 0x52A3, -- CJK COMPATIBILITY IDEOGRAPH-F99D
 [0xF99E] = 0x54BD, -- CJK COMPATIBILITY IDEOGRAPH-F99E
 [0xF99F] = 0x70C8, -- CJK COMPATIBILITY IDEOGRAPH-F99F
 [0xF9A0] = 0x88C2, -- CJK COMPATIBILITY IDEOGRAPH-F9A0
 [0xF9A1] = 0x8AAA, -- CJK COMPATIBILITY IDEOGRAPH-F9A1
 [0xF9A2] = 0x5EC9, -- CJK COMPATIBILITY IDEOGRAPH-F9A2
 [0xF9A3] = 0x5FF5, -- CJK COMPATIBILITY IDEOGRAPH-F9A3
 [0xF9A4] = 0x637B, -- CJK COMPATIBILITY IDEOGRAPH-F9A4
 [0xF9A5] = 0x6BAE, -- CJK COMPATIBILITY IDEOGRAPH-F9A5
 [0xF9A6] = 0x7C3E, -- CJK COMPATIBILITY IDEOGRAPH-F9A6
 [0xF9A7] = 0x7375, -- CJK COMPATIBILITY IDEOGRAPH-F9A7
 [0xF9A8] = 0x4EE4, -- CJK COMPATIBILITY IDEOGRAPH-F9A8
 [0xF9A9] = 0x56F9, -- CJK COMPATIBILITY IDEOGRAPH-F9A9
 [0xF9AA] = 0x5BE7, -- CJK COMPATIBILITY IDEOGRAPH-F9AA
 [0xF9AB] = 0x5DBA, -- CJK COMPATIBILITY IDEOGRAPH-F9AB
 [0xF9AC] = 0x601C, -- CJK COMPATIBILITY IDEOGRAPH-F9AC
 [0xF9AD] = 0x73B2, -- CJK COMPATIBILITY IDEOGRAPH-F9AD
 [0xF9AE] = 0x7469, -- CJK COMPATIBILITY IDEOGRAPH-F9AE
 [0xF9AF] = 0x7F9A, -- CJK COMPATIBILITY IDEOGRAPH-F9AF
 [0xF9B0] = 0x8046, -- CJK COMPATIBILITY IDEOGRAPH-F9B0
 [0xF9B1] = 0x9234, -- CJK COMPATIBILITY IDEOGRAPH-F9B1
 [0xF9B2] = 0x96F6, -- CJK COMPATIBILITY IDEOGRAPH-F9B2
 [0xF9B3] = 0x9748, -- CJK COMPATIBILITY IDEOGRAPH-F9B3
 [0xF9B4] = 0x9818, -- CJK COMPATIBILITY IDEOGRAPH-F9B4
 [0xF9B5] = 0x4F8B, -- CJK COMPATIBILITY IDEOGRAPH-F9B5
 [0xF9B6] = 0x79AE, -- CJK COMPATIBILITY IDEOGRAPH-F9B6
 [0xF9B7] = 0x91B4, -- CJK COMPATIBILITY IDEOGRAPH-F9B7
 [0xF9B8] = 0x96B8, -- CJK COMPATIBILITY IDEOGRAPH-F9B8
 [0xF9B9] = 0x60E1, -- CJK COMPATIBILITY IDEOGRAPH-F9B9
 [0xF9BA] = 0x4E86, -- CJK COMPATIBILITY IDEOGRAPH-F9BA
 [0xF9BB] = 0x50DA, -- CJK COMPATIBILITY IDEOGRAPH-F9BB
 [0xF9BC] = 0x5BEE, -- CJK COMPATIBILITY IDEOGRAPH-F9BC
 [0xF9BD] = 0x5C3F, -- CJK COMPATIBILITY IDEOGRAPH-F9BD
 [0xF9BE] = 0x6599, -- CJK COMPATIBILITY IDEOGRAPH-F9BE
 [0xF9BF] = 0x6A02, -- CJK COMPATIBILITY IDEOGRAPH-F9BF
 [0xF9C0] = 0x71CE, -- CJK COMPATIBILITY IDEOGRAPH-F9C0
 [0xF9C1] = 0x7642, -- CJK COMPATIBILITY IDEOGRAPH-F9C1
 [0xF9C2] = 0x84FC, -- CJK COMPATIBILITY IDEOGRAPH-F9C2
 [0xF9C3] = 0x907C, -- CJK COMPATIBILITY IDEOGRAPH-F9C3
 [0xF9C4] = 0x9F8D, -- CJK COMPATIBILITY IDEOGRAPH-F9C4
 [0xF9C5] = 0x6688, -- CJK COMPATIBILITY IDEOGRAPH-F9C5
 [0xF9C6] = 0x962E, -- CJK COMPATIBILITY IDEOGRAPH-F9C6
 [0xF9C7] = 0x5289, -- CJK COMPATIBILITY IDEOGRAPH-F9C7
 [0xF9C8] = 0x677B, -- CJK COMPATIBILITY IDEOGRAPH-F9C8
 [0xF9C9] = 0x67F3, -- CJK COMPATIBILITY IDEOGRAPH-F9C9
 [0xF9CA] = 0x6D41, -- CJK COMPATIBILITY IDEOGRAPH-F9CA
 [0xF9CB] = 0x6E9C, -- CJK COMPATIBILITY IDEOGRAPH-F9CB
 [0xF9CC] = 0x7409, -- CJK COMPATIBILITY IDEOGRAPH-F9CC
 [0xF9CD] = 0x7559, -- CJK COMPATIBILITY IDEOGRAPH-F9CD
 [0xF9CE] = 0x786B, -- CJK COMPATIBILITY IDEOGRAPH-F9CE
 [0xF9CF] = 0x7D10, -- CJK COMPATIBILITY IDEOGRAPH-F9CF
 [0xF9D0] = 0x985E, -- CJK COMPATIBILITY IDEOGRAPH-F9D0
 [0xF9D1] = 0x516D, -- CJK COMPATIBILITY IDEOGRAPH-F9D1
 [0xF9D2] = 0x622E, -- CJK COMPATIBILITY IDEOGRAPH-F9D2
 [0xF9D3] = 0x9678, -- CJK COMPATIBILITY IDEOGRAPH-F9D3
 [0xF9D4] = 0x502B, -- CJK COMPATIBILITY IDEOGRAPH-F9D4
 [0xF9D5] = 0x5D19, -- CJK COMPATIBILITY IDEOGRAPH-F9D5
 [0xF9D6] = 0x6DEA, -- CJK COMPATIBILITY IDEOGRAPH-F9D6
 [0xF9D7] = 0x8F2A, -- CJK COMPATIBILITY IDEOGRAPH-F9D7
 [0xF9D8] = 0x5F8B, -- CJK COMPATIBILITY IDEOGRAPH-F9D8
 [0xF9D9] = 0x6144, -- CJK COMPATIBILITY IDEOGRAPH-F9D9
 [0xF9DA] = 0x6817, -- CJK COMPATIBILITY IDEOGRAPH-F9DA
 [0xF9DB] = 0x7387, -- CJK COMPATIBILITY IDEOGRAPH-F9DB
 [0xF9DC] = 0x9686, -- CJK COMPATIBILITY IDEOGRAPH-F9DC
 [0xF9DD] = 0x5229, -- CJK COMPATIBILITY IDEOGRAPH-F9DD
 [0xF9DE] = 0x540F, -- CJK COMPATIBILITY IDEOGRAPH-F9DE
 [0xF9DF] = 0x5C65, -- CJK COMPATIBILITY IDEOGRAPH-F9DF
 [0xF9E0] = 0x6613, -- CJK COMPATIBILITY IDEOGRAPH-F9E0
 [0xF9E1] = 0x674E, -- CJK COMPATIBILITY IDEOGRAPH-F9E1
 [0xF9E2] = 0x68A8, -- CJK COMPATIBILITY IDEOGRAPH-F9E2
 [0xF9E3] = 0x6CE5, -- CJK COMPATIBILITY IDEOGRAPH-F9E3
 [0xF9E4] = 0x7406, -- CJK COMPATIBILITY IDEOGRAPH-F9E4
 [0xF9E5] = 0x75E2, -- CJK COMPATIBILITY IDEOGRAPH-F9E5
 [0xF9E6] = 0x7F79, -- CJK COMPATIBILITY IDEOGRAPH-F9E6
 [0xF9E7] = 0x88CF, -- CJK COMPATIBILITY IDEOGRAPH-F9E7
 [0xF9E8] = 0x88E1, -- CJK COMPATIBILITY IDEOGRAPH-F9E8
 [0xF9E9] = 0x91CC, -- CJK COMPATIBILITY IDEOGRAPH-F9E9
 [0xF9EA] = 0x96E2, -- CJK COMPATIBILITY IDEOGRAPH-F9EA
 [0xF9EB] = 0x533F, -- CJK COMPATIBILITY IDEOGRAPH-F9EB
 [0xF9EC] = 0x6EBA, -- CJK COMPATIBILITY IDEOGRAPH-F9EC
 [0xF9ED] = 0x541D, -- CJK COMPATIBILITY IDEOGRAPH-F9ED
 [0xF9EE] = 0x71D0, -- CJK COMPATIBILITY IDEOGRAPH-F9EE
 [0xF9EF] = 0x7498, -- CJK COMPATIBILITY IDEOGRAPH-F9EF
 [0xF9F0] = 0x85FA, -- CJK COMPATIBILITY IDEOGRAPH-F9F0
 [0xF9F1] = 0x96A3, -- CJK COMPATIBILITY IDEOGRAPH-F9F1
 [0xF9F2] = 0x9C57, -- CJK COMPATIBILITY IDEOGRAPH-F9F2
 [0xF9F3] = 0x9E9F, -- CJK COMPATIBILITY IDEOGRAPH-F9F3
 [0xF9F4] = 0x6797, -- CJK COMPATIBILITY IDEOGRAPH-F9F4
 [0xF9F5] = 0x6DCB, -- CJK COMPATIBILITY IDEOGRAPH-F9F5
 [0xF9F6] = 0x81E8, -- CJK COMPATIBILITY IDEOGRAPH-F9F6
 [0xF9F7] = 0x7ACB, -- CJK COMPATIBILITY IDEOGRAPH-F9F7
 [0xF9F8] = 0x7B20, -- CJK COMPATIBILITY IDEOGRAPH-F9F8
 [0xF9F9] = 0x7C92, -- CJK COMPATIBILITY IDEOGRAPH-F9F9
 [0xF9FA] = 0x72C0, -- CJK COMPATIBILITY IDEOGRAPH-F9FA
 [0xF9FB] = 0x7099, -- CJK COMPATIBILITY IDEOGRAPH-F9FB
 [0xF9FC] = 0x8B58, -- CJK COMPATIBILITY IDEOGRAPH-F9FC
 [0xF9FD] = 0x4EC0, -- CJK COMPATIBILITY IDEOGRAPH-F9FD
 [0xF9FE] = 0x8336, -- CJK COMPATIBILITY IDEOGRAPH-F9FE
 [0xF9FF] = 0x523A, -- CJK COMPATIBILITY IDEOGRAPH-F9FF
 [0xFA00] = 0x5207, -- CJK COMPATIBILITY IDEOGRAPH-FA00
 [0xFA01] = 0x5EA6, -- CJK COMPATIBILITY IDEOGRAPH-FA01
 [0xFA02] = 0x62D3, -- CJK COMPATIBILITY IDEOGRAPH-FA02
 [0xFA03] = 0x7CD6, -- CJK COMPATIBILITY IDEOGRAPH-FA03
 [0xFA04] = 0x5B85, -- CJK COMPATIBILITY IDEOGRAPH-FA04
 [0xFA05] = 0x6D1E, -- CJK COMPATIBILITY IDEOGRAPH-FA05
 [0xFA06] = 0x66B4, -- CJK COMPATIBILITY IDEOGRAPH-FA06
 [0xFA07] = 0x8F3B, -- CJK COMPATIBILITY IDEOGRAPH-FA07
 [0xFA08] = 0x884C, -- CJK COMPATIBILITY IDEOGRAPH-FA08
 [0xFA09] = 0x964D, -- CJK COMPATIBILITY IDEOGRAPH-FA09
 [0xFA0A] = 0x898B, -- CJK COMPATIBILITY IDEOGRAPH-FA0A
 [0xFA0B] = 0x5ED3, -- CJK COMPATIBILITY IDEOGRAPH-FA0B
 [0xFA0C] = 0x5140, -- CJK COMPATIBILITY IDEOGRAPH-FA0C
 [0xFA0D] = 0x55C0, -- CJK COMPATIBILITY IDEOGRAPH-FA0D
 -- CJK COMPATIBILITY IDEOGRAPH-F907..CJK COMPATIBILITY IDEOGRAPH-F908
 [0xF907] = 0x9F9C,
 [0xF908] = 0x9F9C,
 [0xFA10] = 0x585A, -- CJK COMPATIBILITY IDEOGRAPH-FA10
 [0xFA11] = {status="valid"}, -- CJK COMPATIBILITY IDEOGRAPH-FA11
 [0xFA12] = 0x6674, -- CJK COMPATIBILITY IDEOGRAPH-FA12
 -- CJK COMPATIBILITY IDEOGRAPH-FA0E..CJK COMPATIBILITY IDEOGRAPH-FA0F
 [0xFA0E] = {status="valid"},
 [0xFA0F] = {status="valid"},
 [0xFA15] = 0x51DE, -- CJK COMPATIBILITY IDEOGRAPH-FA15
 [0xFA16] = 0x732A, -- CJK COMPATIBILITY IDEOGRAPH-FA16
 [0xFA17] = 0x76CA, -- CJK COMPATIBILITY IDEOGRAPH-FA17
 [0xFA18] = 0x793C, -- CJK COMPATIBILITY IDEOGRAPH-FA18
 [0xFA19] = 0x795E, -- CJK COMPATIBILITY IDEOGRAPH-FA19
 [0xFA1A] = 0x7965, -- CJK COMPATIBILITY IDEOGRAPH-FA1A
 [0xFA1B] = 0x798F, -- CJK COMPATIBILITY IDEOGRAPH-FA1B
 [0xFA1C] = 0x9756, -- CJK COMPATIBILITY IDEOGRAPH-FA1C
 [0xFA1D] = 0x7CBE, -- CJK COMPATIBILITY IDEOGRAPH-FA1D
 [0xFA1E] = 0x7FBD, -- CJK COMPATIBILITY IDEOGRAPH-FA1E
 [0xFA1F] = {status="valid"}, -- CJK COMPATIBILITY IDEOGRAPH-FA1F
 [0xFA20] = 0x8612, -- CJK COMPATIBILITY IDEOGRAPH-FA20
 [0xFA21] = {status="valid"}, -- CJK COMPATIBILITY IDEOGRAPH-FA21
 [0xFA22] = 0x8AF8, -- CJK COMPATIBILITY IDEOGRAPH-FA22
 -- CJK COMPATIBILITY IDEOGRAPH-FA13..CJK COMPATIBILITY IDEOGRAPH-FA14
 [0xFA13] = {status="valid"},
 [0xFA14] = {status="valid"},
 [0xFA25] = 0x9038, -- CJK COMPATIBILITY IDEOGRAPH-FA25
 [0xFA26] = 0x90FD, -- CJK COMPATIBILITY IDEOGRAPH-FA26
 -- CJK COMPATIBILITY IDEOGRAPH-FA23..CJK COMPATIBILITY IDEOGRAPH-FA24
 [0xFA23] = {status="valid"},
 [0xFA24] = {status="valid"},
 [0xFA2A] = 0x98EF, -- CJK COMPATIBILITY IDEOGRAPH-FA2A
 [0xFA2B] = 0x98FC, -- CJK COMPATIBILITY IDEOGRAPH-FA2B
 [0xFA2C] = 0x9928, -- CJK COMPATIBILITY IDEOGRAPH-FA2C
 [0xFA2D] = 0x9DB4, -- CJK COMPATIBILITY IDEOGRAPH-FA2D
 [0xFA2E] = 0x90DE, -- CJK COMPATIBILITY IDEOGRAPH-FA2E
 [0xFA2F] = 0x96B7, -- CJK COMPATIBILITY IDEOGRAPH-FA2F
 [0xFA30] = 0x4FAE, -- CJK COMPATIBILITY IDEOGRAPH-FA30
 [0xFA31] = 0x50E7, -- CJK COMPATIBILITY IDEOGRAPH-FA31
 [0xFA32] = 0x514D, -- CJK COMPATIBILITY IDEOGRAPH-FA32
 [0xFA33] = 0x52C9, -- CJK COMPATIBILITY IDEOGRAPH-FA33
 [0xFA34] = 0x52E4, -- CJK COMPATIBILITY IDEOGRAPH-FA34
 [0xFA35] = 0x5351, -- CJK COMPATIBILITY IDEOGRAPH-FA35
 [0xFA36] = 0x559D, -- CJK COMPATIBILITY IDEOGRAPH-FA36
 [0xFA37] = 0x5606, -- CJK COMPATIBILITY IDEOGRAPH-FA37
 [0xFA38] = 0x5668, -- CJK COMPATIBILITY IDEOGRAPH-FA38
 [0xFA39] = 0x5840, -- CJK COMPATIBILITY IDEOGRAPH-FA39
 [0xFA3A] = 0x58A8, -- CJK COMPATIBILITY IDEOGRAPH-FA3A
 [0xFA3B] = 0x5C64, -- CJK COMPATIBILITY IDEOGRAPH-FA3B
 [0xFA3C] = 0x5C6E, -- CJK COMPATIBILITY IDEOGRAPH-FA3C
 [0xFA3D] = 0x6094, -- CJK COMPATIBILITY IDEOGRAPH-FA3D
 [0xFA3E] = 0x6168, -- CJK COMPATIBILITY IDEOGRAPH-FA3E
 [0xFA3F] = 0x618E, -- CJK COMPATIBILITY IDEOGRAPH-FA3F
 [0xFA40] = 0x61F2, -- CJK COMPATIBILITY IDEOGRAPH-FA40
 [0xFA41] = 0x654F, -- CJK COMPATIBILITY IDEOGRAPH-FA41
 [0xFA42] = 0x65E2, -- CJK COMPATIBILITY IDEOGRAPH-FA42
 [0xFA43] = 0x6691, -- CJK COMPATIBILITY IDEOGRAPH-FA43
 [0xFA44] = 0x6885, -- CJK COMPATIBILITY IDEOGRAPH-FA44
 [0xFA45] = 0x6D77, -- CJK COMPATIBILITY IDEOGRAPH-FA45
 [0xFA46] = 0x6E1A, -- CJK COMPATIBILITY IDEOGRAPH-FA46
 [0xFA47] = 0x6F22, -- CJK COMPATIBILITY IDEOGRAPH-FA47
 [0xFA48] = 0x716E, -- CJK COMPATIBILITY IDEOGRAPH-FA48
 [0xFA49] = 0x722B, -- CJK COMPATIBILITY IDEOGRAPH-FA49
 [0xFA4A] = 0x7422, -- CJK COMPATIBILITY IDEOGRAPH-FA4A
 [0xFA4B] = 0x7891, -- CJK COMPATIBILITY IDEOGRAPH-FA4B
 [0xFA4C] = 0x793E, -- CJK COMPATIBILITY IDEOGRAPH-FA4C
 [0xFA4D] = 0x7949, -- CJK COMPATIBILITY IDEOGRAPH-FA4D
 [0xFA4E] = 0x7948, -- CJK COMPATIBILITY IDEOGRAPH-FA4E
 [0xFA4F] = 0x7950, -- CJK COMPATIBILITY IDEOGRAPH-FA4F
 [0xFA50] = 0x7956, -- CJK COMPATIBILITY IDEOGRAPH-FA50
 [0xFA51] = 0x795D, -- CJK COMPATIBILITY IDEOGRAPH-FA51
 [0xFA52] = 0x798D, -- CJK COMPATIBILITY IDEOGRAPH-FA52
 [0xFA53] = 0x798E, -- CJK COMPATIBILITY IDEOGRAPH-FA53
 [0xFA54] = 0x7A40, -- CJK COMPATIBILITY IDEOGRAPH-FA54
 [0xFA55] = 0x7A81, -- CJK COMPATIBILITY IDEOGRAPH-FA55
 [0xFA56] = 0x7BC0, -- CJK COMPATIBILITY IDEOGRAPH-FA56
 [0xFA57] = 0x7DF4, -- CJK COMPATIBILITY IDEOGRAPH-FA57
 [0xFA58] = 0x7E09, -- CJK COMPATIBILITY IDEOGRAPH-FA58
 [0xFA59] = 0x7E41, -- CJK COMPATIBILITY IDEOGRAPH-FA59
 [0xFA5A] = 0x7F72, -- CJK COMPATIBILITY IDEOGRAPH-FA5A
 [0xFA5B] = 0x8005, -- CJK COMPATIBILITY IDEOGRAPH-FA5B
 [0xFA5C] = 0x81ED, -- CJK COMPATIBILITY IDEOGRAPH-FA5C
 -- CJK COMPATIBILITY IDEOGRAPH-FA27..CJK COMPATIBILITY IDEOGRAPH-FA29
 [0xFA27] = {status="valid"},
 [0xFA28] = {status="valid"},
 [0xFA29] = {status="valid"},
 [0xFA5F] = 0x8457, -- CJK COMPATIBILITY IDEOGRAPH-FA5F
 [0xFA60] = 0x8910, -- CJK COMPATIBILITY IDEOGRAPH-FA60
 [0xFA61] = 0x8996, -- CJK COMPATIBILITY IDEOGRAPH-FA61
 [0xFA62] = 0x8B01, -- CJK COMPATIBILITY IDEOGRAPH-FA62
 [0xFA63] = 0x8B39, -- CJK COMPATIBILITY IDEOGRAPH-FA63
 [0xFA64] = 0x8CD3, -- CJK COMPATIBILITY IDEOGRAPH-FA64
 [0xFA65] = 0x8D08, -- CJK COMPATIBILITY IDEOGRAPH-FA65
 [0xFA66] = 0x8FB6, -- CJK COMPATIBILITY IDEOGRAPH-FA66
 [0xFA67] = 0x9038, -- CJK COMPATIBILITY IDEOGRAPH-FA67
 [0xFA68] = 0x96E3, -- CJK COMPATIBILITY IDEOGRAPH-FA68
 [0xFA69] = 0x97FF, -- CJK COMPATIBILITY IDEOGRAPH-FA69
 [0xFA6A] = 0x983B, -- CJK COMPATIBILITY IDEOGRAPH-FA6A
 [0xFA6B] = 0x6075, -- CJK COMPATIBILITY IDEOGRAPH-FA6B
 [0xFA6C] = 0x242EE, -- CJK COMPATIBILITY IDEOGRAPH-FA6C
 [0xFA6D] = 0x8218, -- CJK COMPATIBILITY IDEOGRAPH-FA6D
 -- CJK COMPATIBILITY IDEOGRAPH-FA5D..CJK COMPATIBILITY IDEOGRAPH-FA5E
 [0xFA5D] = 0x8279,
 [0xFA5E] = 0x8279,
 [0xFA70] = 0x4E26, -- CJK COMPATIBILITY IDEOGRAPH-FA70
 [0xFA71] = 0x51B5, -- CJK COMPATIBILITY IDEOGRAPH-FA71
 [0xFA72] = 0x5168, -- CJK COMPATIBILITY IDEOGRAPH-FA72
 [0xFA73] = 0x4F80, -- CJK COMPATIBILITY IDEOGRAPH-FA73
 [0xFA74] = 0x5145, -- CJK COMPATIBILITY IDEOGRAPH-FA74
 [0xFA75] = 0x5180, -- CJK COMPATIBILITY IDEOGRAPH-FA75
 [0xFA76] = 0x52C7, -- CJK COMPATIBILITY IDEOGRAPH-FA76
 [0xFA77] = 0x52FA, -- CJK COMPATIBILITY IDEOGRAPH-FA77
 [0xFA78] = 0x559D, -- CJK COMPATIBILITY IDEOGRAPH-FA78
 [0xFA79] = 0x5555, -- CJK COMPATIBILITY IDEOGRAPH-FA79
 [0xFA7A] = 0x5599, -- CJK COMPATIBILITY IDEOGRAPH-FA7A
 [0xFA7B] = 0x55E2, -- CJK COMPATIBILITY IDEOGRAPH-FA7B
 [0xFA7C] = 0x585A, -- CJK COMPATIBILITY IDEOGRAPH-FA7C
 [0xFA7D] = 0x58B3, -- CJK COMPATIBILITY IDEOGRAPH-FA7D
 [0xFA7E] = 0x5944, -- CJK COMPATIBILITY IDEOGRAPH-FA7E
 [0xFA7F] = 0x5954, -- CJK COMPATIBILITY IDEOGRAPH-FA7F
 [0xFA80] = 0x5A62, -- CJK COMPATIBILITY IDEOGRAPH-FA80
 [0xFA81] = 0x5B28, -- CJK COMPATIBILITY IDEOGRAPH-FA81
 [0xFA82] = 0x5ED2, -- CJK COMPATIBILITY IDEOGRAPH-FA82
 [0xFA83] = 0x5ED9, -- CJK COMPATIBILITY IDEOGRAPH-FA83
 [0xFA84] = 0x5F69, -- CJK COMPATIBILITY IDEOGRAPH-FA84
 [0xFA85] = 0x5FAD, -- CJK COMPATIBILITY IDEOGRAPH-FA85
 [0xFA86] = 0x60D8, -- CJK COMPATIBILITY IDEOGRAPH-FA86
 [0xFA87] = 0x614E, -- CJK COMPATIBILITY IDEOGRAPH-FA87
 [0xFA88] = 0x6108, -- CJK COMPATIBILITY IDEOGRAPH-FA88
 [0xFA89] = 0x618E, -- CJK COMPATIBILITY IDEOGRAPH-FA89
 [0xFA8A] = 0x6160, -- CJK COMPATIBILITY IDEOGRAPH-FA8A
 [0xFA8B] = 0x61F2, -- CJK COMPATIBILITY IDEOGRAPH-FA8B
 [0xFA8C] = 0x6234, -- CJK COMPATIBILITY IDEOGRAPH-FA8C
 [0xFA8D] = 0x63C4, -- CJK COMPATIBILITY IDEOGRAPH-FA8D
 [0xFA8E] = 0x641C, -- CJK COMPATIBILITY IDEOGRAPH-FA8E
 [0xFA8F] = 0x6452, -- CJK COMPATIBILITY IDEOGRAPH-FA8F
 [0xFA90] = 0x6556, -- CJK COMPATIBILITY IDEOGRAPH-FA90
 [0xFA91] = 0x6674, -- CJK COMPATIBILITY IDEOGRAPH-FA91
 [0xFA92] = 0x6717, -- CJK COMPATIBILITY IDEOGRAPH-FA92
 [0xFA93] = 0x671B, -- CJK COMPATIBILITY IDEOGRAPH-FA93
 [0xFA94] = 0x6756, -- CJK COMPATIBILITY IDEOGRAPH-FA94
 [0xFA95] = 0x6B79, -- CJK COMPATIBILITY IDEOGRAPH-FA95
 [0xFA96] = 0x6BBA, -- CJK COMPATIBILITY IDEOGRAPH-FA96
 [0xFA97] = 0x6D41, -- CJK COMPATIBILITY IDEOGRAPH-FA97
 [0xFA98] = 0x6EDB, -- CJK COMPATIBILITY IDEOGRAPH-FA98
 [0xFA99] = 0x6ECB, -- CJK COMPATIBILITY IDEOGRAPH-FA99
 [0xFA9A] = 0x6F22, -- CJK COMPATIBILITY IDEOGRAPH-FA9A
 [0xFA9B] = 0x701E, -- CJK COMPATIBILITY IDEOGRAPH-FA9B
 [0xFA9C] = 0x716E, -- CJK COMPATIBILITY IDEOGRAPH-FA9C
 [0xFA9D] = 0x77A7, -- CJK COMPATIBILITY IDEOGRAPH-FA9D
 [0xFA9E] = 0x7235, -- CJK COMPATIBILITY IDEOGRAPH-FA9E
 [0xFA9F] = 0x72AF, -- CJK COMPATIBILITY IDEOGRAPH-FA9F
 [0xFAA0] = 0x732A, -- CJK COMPATIBILITY IDEOGRAPH-FAA0
 [0xFAA1] = 0x7471, -- CJK COMPATIBILITY IDEOGRAPH-FAA1
 [0xFAA2] = 0x7506, -- CJK COMPATIBILITY IDEOGRAPH-FAA2
 [0xFAA3] = 0x753B, -- CJK COMPATIBILITY IDEOGRAPH-FAA3
 [0xFAA4] = 0x761D, -- CJK COMPATIBILITY IDEOGRAPH-FAA4
 [0xFAA5] = 0x761F, -- CJK COMPATIBILITY IDEOGRAPH-FAA5
 [0xFAA6] = 0x76CA, -- CJK COMPATIBILITY IDEOGRAPH-FAA6
 [0xFAA7] = 0x76DB, -- CJK COMPATIBILITY IDEOGRAPH-FAA7
 [0xFAA8] = 0x76F4, -- CJK COMPATIBILITY IDEOGRAPH-FAA8
 [0xFAA9] = 0x774A, -- CJK COMPATIBILITY IDEOGRAPH-FAA9
 [0xFAAA] = 0x7740, -- CJK COMPATIBILITY IDEOGRAPH-FAAA
 [0xFAAB] = 0x78CC, -- CJK COMPATIBILITY IDEOGRAPH-FAAB
 [0xFAAC] = 0x7AB1, -- CJK COMPATIBILITY IDEOGRAPH-FAAC
 [0xFAAD] = 0x7BC0, -- CJK COMPATIBILITY IDEOGRAPH-FAAD
 [0xFAAE] = 0x7C7B, -- CJK COMPATIBILITY IDEOGRAPH-FAAE
 [0xFAAF] = 0x7D5B, -- CJK COMPATIBILITY IDEOGRAPH-FAAF
 [0xFAB0] = 0x7DF4, -- CJK COMPATIBILITY IDEOGRAPH-FAB0
 [0xFAB1] = 0x7F3E, -- CJK COMPATIBILITY IDEOGRAPH-FAB1
 [0xFAB2] = 0x8005, -- CJK COMPATIBILITY IDEOGRAPH-FAB2
 [0xFAB3] = 0x8352, -- CJK COMPATIBILITY IDEOGRAPH-FAB3
 [0xFAB4] = 0x83EF, -- CJK COMPATIBILITY IDEOGRAPH-FAB4
 [0xFAB5] = 0x8779, -- CJK COMPATIBILITY IDEOGRAPH-FAB5
 [0xFAB6] = 0x8941, -- CJK COMPATIBILITY IDEOGRAPH-FAB6
 [0xFAB7] = 0x8986, -- CJK COMPATIBILITY IDEOGRAPH-FAB7
 [0xFAB8] = 0x8996, -- CJK COMPATIBILITY IDEOGRAPH-FAB8
 [0xFAB9] = 0x8ABF, -- CJK COMPATIBILITY IDEOGRAPH-FAB9
 [0xFABA] = 0x8AF8, -- CJK COMPATIBILITY IDEOGRAPH-FABA
 [0xFABB] = 0x8ACB, -- CJK COMPATIBILITY IDEOGRAPH-FABB
 [0xFABC] = 0x8B01, -- CJK COMPATIBILITY IDEOGRAPH-FABC
 [0xFABD] = 0x8AFE, -- CJK COMPATIBILITY IDEOGRAPH-FABD
 [0xFABE] = 0x8AED, -- CJK COMPATIBILITY IDEOGRAPH-FABE
 [0xFABF] = 0x8B39, -- CJK COMPATIBILITY IDEOGRAPH-FABF
 [0xFAC0] = 0x8B8A, -- CJK COMPATIBILITY IDEOGRAPH-FAC0
 [0xFAC1] = 0x8D08, -- CJK COMPATIBILITY IDEOGRAPH-FAC1
 [0xFAC2] = 0x8F38, -- CJK COMPATIBILITY IDEOGRAPH-FAC2
 [0xFAC3] = 0x9072, -- CJK COMPATIBILITY IDEOGRAPH-FAC3
 [0xFAC4] = 0x9199, -- CJK COMPATIBILITY IDEOGRAPH-FAC4
 [0xFAC5] = 0x9276, -- CJK COMPATIBILITY IDEOGRAPH-FAC5
 [0xFAC6] = 0x967C, -- CJK COMPATIBILITY IDEOGRAPH-FAC6
 [0xFAC7] = 0x96E3, -- CJK COMPATIBILITY IDEOGRAPH-FAC7
 [0xFAC8] = 0x9756, -- CJK COMPATIBILITY IDEOGRAPH-FAC8
 [0xFAC9] = 0x97DB, -- CJK COMPATIBILITY IDEOGRAPH-FAC9
 [0xFACA] = 0x97FF, -- CJK COMPATIBILITY IDEOGRAPH-FACA
 [0xFACB] = 0x980B, -- CJK COMPATIBILITY IDEOGRAPH-FACB
 [0xFACC] = 0x983B, -- CJK COMPATIBILITY IDEOGRAPH-FACC
 [0xFACD] = 0x9B12, -- CJK COMPATIBILITY IDEOGRAPH-FACD
 [0xFACE] = 0x9F9C, -- CJK COMPATIBILITY IDEOGRAPH-FACE
 [0xFACF] = 0x2284A, -- CJK COMPATIBILITY IDEOGRAPH-FACF
 [0xFAD0] = 0x22844, -- CJK COMPATIBILITY IDEOGRAPH-FAD0
 [0xFAD1] = 0x233D5, -- CJK COMPATIBILITY IDEOGRAPH-FAD1
 [0xFAD2] = 0x3B9D, -- CJK COMPATIBILITY IDEOGRAPH-FAD2
 [0xFAD3] = 0x4018, -- CJK COMPATIBILITY IDEOGRAPH-FAD3
 [0xFAD4] = 0x4039, -- CJK COMPATIBILITY IDEOGRAPH-FAD4
 [0xFAD5] = 0x25249, -- CJK COMPATIBILITY IDEOGRAPH-FAD5
 [0xFAD6] = 0x25CD0, -- CJK COMPATIBILITY IDEOGRAPH-FAD6
 [0xFAD7] = 0x27ED3, -- CJK COMPATIBILITY IDEOGRAPH-FAD7
 [0xFAD8] = 0x9F43, -- CJK COMPATIBILITY IDEOGRAPH-FAD8
 [0xFAD9] = 0x9F8E, -- CJK COMPATIBILITY IDEOGRAPH-FAD9
 -- NA   <reserved-FA6E>..<reserved-FA6F>
 [0xFA6E] = {status="disallowed"},
 [0xFA6F] = {status="disallowed"},
 [0xFB00] = {0x0066, 0x0066}, -- LATIN SMALL LIGATURE FF
 [0xFB01] = {0x0066, 0x0069}, -- LATIN SMALL LIGATURE FI
 [0xFB02] = {0x0066, 0x006C}, -- LATIN SMALL LIGATURE FL
 [0xFB03] = {0x0066, 0x0066, 0x0069}, -- LATIN SMALL LIGATURE FFI
 [0xFB04] = {0x0066, 0x0066, 0x006C}, -- LATIN SMALL LIGATURE FFL
 -- LATIN SMALL LIGATURE LONG S T..LATIN SMALL LIGATURE ST
 [0xFB05] = {0x0073, 0x0074},
 [0xFB06] = {0x0073, 0x0074},
 [0xFB13] = {0x0574, 0x0576}, -- ARMENIAN SMALL LIGATURE MEN NOW
 [0xFB14] = {0x0574, 0x0565}, -- ARMENIAN SMALL LIGATURE MEN ECH
 [0xFB15] = {0x0574, 0x056B}, -- ARMENIAN SMALL LIGATURE MEN INI
 [0xFB16] = {0x057E, 0x0576}, -- ARMENIAN SMALL LIGATURE VEW NOW
 [0xFB17] = {0x0574, 0x056D}, -- ARMENIAN SMALL LIGATURE MEN XEH
 [0xFB1D] = {0x05D9, 0x05B4}, -- HEBREW LETTER YOD WITH HIRIQ
 [0xFB1E] = {status="valid"}, -- HEBREW POINT JUDEO-SPANISH VARIKA
 [0xFB1F] = {0x05F2, 0x05B7}, -- HEBREW LIGATURE YIDDISH YOD YOD PATAH
 [0xFB20] = 0x05E2, -- HEBREW LETTER ALTERNATIVE AYIN
 [0xFB21] = 0x05D0, -- HEBREW LETTER WIDE ALEF
 [0xFB22] = 0x05D3, -- HEBREW LETTER WIDE DALET
 [0xFB23] = 0x05D4, -- HEBREW LETTER WIDE HE
 [0xFB24] = 0x05DB, -- HEBREW LETTER WIDE KAF
 [0xFB25] = 0x05DC, -- HEBREW LETTER WIDE LAMED
 [0xFB26] = 0x05DD, -- HEBREW LETTER WIDE FINAL MEM
 [0xFB27] = 0x05E8, -- HEBREW LETTER WIDE RESH
 [0xFB28] = 0x05EA, -- HEBREW LETTER WIDE TAV
 [0xFB29] = {status="disallowed"}, -- HEBREW LETTER ALTERNATIVE PLUS SIGN
 [0xFB2A] = {0x05E9, 0x05C1}, -- HEBREW LETTER SHIN WITH SHIN DOT
 [0xFB2B] = {0x05E9, 0x05C2}, -- HEBREW LETTER SHIN WITH SIN DOT
 [0xFB2C] = {0x05E9, 0x05BC, 0x05C1}, -- HEBREW LETTER SHIN WITH DAGESH AND SHIN DOT
 [0xFB2D] = {0x05E9, 0x05BC, 0x05C2}, -- HEBREW LETTER SHIN WITH DAGESH AND SIN DOT
 [0xFB2E] = {0x05D0, 0x05B7}, -- HEBREW LETTER ALEF WITH PATAH
 [0xFB2F] = {0x05D0, 0x05B8}, -- HEBREW LETTER ALEF WITH QAMATS
 [0xFB30] = {0x05D0, 0x05BC}, -- HEBREW LETTER ALEF WITH MAPIQ
 [0xFB31] = {0x05D1, 0x05BC}, -- HEBREW LETTER BET WITH DAGESH
 [0xFB32] = {0x05D2, 0x05BC}, -- HEBREW LETTER GIMEL WITH DAGESH
 [0xFB33] = {0x05D3, 0x05BC}, -- HEBREW LETTER DALET WITH DAGESH
 [0xFB34] = {0x05D4, 0x05BC}, -- HEBREW LETTER HE WITH MAPIQ
 [0xFB35] = {0x05D5, 0x05BC}, -- HEBREW LETTER VAV WITH DAGESH
 [0xFB36] = {0x05D6, 0x05BC}, -- HEBREW LETTER ZAYIN WITH DAGESH
 [0xFB37] = {status="disallowed"}, -- NA   <reserved-FB37>
 [0xFB38] = {0x05D8, 0x05BC}, -- HEBREW LETTER TET WITH DAGESH
 [0xFB39] = {0x05D9, 0x05BC}, -- HEBREW LETTER YOD WITH DAGESH
 [0xFB3A] = {0x05DA, 0x05BC}, -- HEBREW LETTER FINAL KAF WITH DAGESH
 [0xFB3B] = {0x05DB, 0x05BC}, -- HEBREW LETTER KAF WITH DAGESH
 [0xFB3C] = {0x05DC, 0x05BC}, -- HEBREW LETTER LAMED WITH DAGESH
 [0xFB3D] = {status="disallowed"}, -- NA   <reserved-FB3D>
 [0xFB3E] = {0x05DE, 0x05BC}, -- HEBREW LETTER MEM WITH DAGESH
 [0xFB3F] = {status="disallowed"}, -- NA   <reserved-FB3F>
 [0xFB40] = {0x05E0, 0x05BC}, -- HEBREW LETTER NUN WITH DAGESH
 [0xFB41] = {0x05E1, 0x05BC}, -- HEBREW LETTER SAMEKH WITH DAGESH
 [0xFB42] = {status="disallowed"}, -- NA   <reserved-FB42>
 [0xFB43] = {0x05E3, 0x05BC}, -- HEBREW LETTER FINAL PE WITH DAGESH
 [0xFB44] = {0x05E4, 0x05BC}, -- HEBREW LETTER PE WITH DAGESH
 [0xFB45] = {status="disallowed"}, -- NA   <reserved-FB45>
 [0xFB46] = {0x05E6, 0x05BC}, -- HEBREW LETTER TSADI WITH DAGESH
 [0xFB47] = {0x05E7, 0x05BC}, -- HEBREW LETTER QOF WITH DAGESH
 [0xFB48] = {0x05E8, 0x05BC}, -- HEBREW LETTER RESH WITH DAGESH
 [0xFB49] = {0x05E9, 0x05BC}, -- HEBREW LETTER SHIN WITH DAGESH
 [0xFB4A] = {0x05EA, 0x05BC}, -- HEBREW LETTER TAV WITH DAGESH
 [0xFB4B] = {0x05D5, 0x05B9}, -- HEBREW LETTER VAV WITH HOLAM
 [0xFB4C] = {0x05D1, 0x05BF}, -- HEBREW LETTER BET WITH RAFE
 [0xFB4D] = {0x05DB, 0x05BF}, -- HEBREW LETTER KAF WITH RAFE
 [0xFB4E] = {0x05E4, 0x05BF}, -- HEBREW LETTER PE WITH RAFE
 [0xFB4F] = {0x05D0, 0x05DC}, -- HEBREW LIGATURE ALEF LAMED
 -- NA   <reserved-FB18>..<reserved-FB1C>
 [0xFB18] = {status="disallowed"},
 [0xFB19] = {status="disallowed"},
 [0xFB1A] = {status="disallowed"},
 [0xFB1B] = {status="disallowed"},
 [0xFB1C] = {status="disallowed"},
 -- ARABIC LETTER ALEF WASLA ISOLATED FORM..ARABIC LETTER ALEF WASLA FINAL FORM
 [0xFB50] = 0x0671,
 [0xFB51] = 0x0671,
 -- ARABIC LETTER BEEH ISOLATED FORM..ARABIC LETTER BEEH MEDIAL FORM
 [0xFB52] = 0x067B,
 [0xFB53] = 0x067B,
 [0xFB54] = 0x067B,
 [0xFB55] = 0x067B,
 -- ARABIC LETTER PEH ISOLATED FORM..ARABIC LETTER PEH MEDIAL FORM
 [0xFB56] = 0x067E,
 [0xFB57] = 0x067E,
 [0xFB58] = 0x067E,
 [0xFB59] = 0x067E,
 -- ARABIC LETTER BEHEH ISOLATED FORM..ARABIC LETTER BEHEH MEDIAL FORM
 [0xFB5A] = 0x0680,
 [0xFB5B] = 0x0680,
 [0xFB5C] = 0x0680,
 [0xFB5D] = 0x0680,
 -- ARABIC LETTER TTEHEH ISOLATED FORM..ARABIC LETTER TTEHEH MEDIAL FORM
 [0xFB5E] = 0x067A,
 [0xFB5F] = 0x067A,
 [0xFB60] = 0x067A,
 [0xFB61] = 0x067A,
 -- ARABIC LETTER TEHEH ISOLATED FORM..ARABIC LETTER TEHEH MEDIAL FORM
 [0xFB62] = 0x067F,
 [0xFB63] = 0x067F,
 [0xFB64] = 0x067F,
 [0xFB65] = 0x067F,
 -- ARABIC LETTER TTEH ISOLATED FORM..ARABIC LETTER TTEH MEDIAL FORM
 [0xFB66] = 0x0679,
 [0xFB67] = 0x0679,
 [0xFB68] = 0x0679,
 [0xFB69] = 0x0679,
 -- ARABIC LETTER VEH ISOLATED FORM..ARABIC LETTER VEH MEDIAL FORM
 [0xFB6A] = 0x06A4,
 [0xFB6B] = 0x06A4,
 [0xFB6C] = 0x06A4,
 [0xFB6D] = 0x06A4,
 -- ARABIC LETTER PEHEH ISOLATED FORM..ARABIC LETTER PEHEH MEDIAL FORM
 [0xFB6E] = 0x06A6,
 [0xFB6F] = 0x06A6,
 [0xFB70] = 0x06A6,
 [0xFB71] = 0x06A6,
 -- ARABIC LETTER DYEH ISOLATED FORM..ARABIC LETTER DYEH MEDIAL FORM
 [0xFB72] = 0x0684,
 [0xFB73] = 0x0684,
 [0xFB74] = 0x0684,
 [0xFB75] = 0x0684,
 -- ARABIC LETTER NYEH ISOLATED FORM..ARABIC LETTER NYEH MEDIAL FORM
 [0xFB76] = 0x0683,
 [0xFB77] = 0x0683,
 [0xFB78] = 0x0683,
 [0xFB79] = 0x0683,
 -- ARABIC LETTER TCHEH ISOLATED FORM..ARABIC LETTER TCHEH MEDIAL FORM
 [0xFB7A] = 0x0686,
 [0xFB7B] = 0x0686,
 [0xFB7C] = 0x0686,
 [0xFB7D] = 0x0686,
 -- ARABIC LETTER TCHEHEH ISOLATED FORM..ARABIC LETTER TCHEHEH MEDIAL FORM
 [0xFB7E] = 0x0687,
 [0xFB7F] = 0x0687,
 [0xFB80] = 0x0687,
 [0xFB81] = 0x0687,
 -- ARABIC LETTER DDAHAL ISOLATED FORM..ARABIC LETTER DDAHAL FINAL FORM
 [0xFB82] = 0x068D,
 [0xFB83] = 0x068D,
 -- ARABIC LETTER DAHAL ISOLATED FORM..ARABIC LETTER DAHAL FINAL FORM
 [0xFB84] = 0x068C,
 [0xFB85] = 0x068C,
 -- ARABIC LETTER DUL ISOLATED FORM..ARABIC LETTER DUL FINAL FORM
 [0xFB86] = 0x068E,
 [0xFB87] = 0x068E,
 -- ARABIC LETTER DDAL ISOLATED FORM..ARABIC LETTER DDAL FINAL FORM
 [0xFB88] = 0x0688,
 [0xFB89] = 0x0688,
 -- ARABIC LETTER JEH ISOLATED FORM..ARABIC LETTER JEH FINAL FORM
 [0xFB8A] = 0x0698,
 [0xFB8B] = 0x0698,
 -- ARABIC LETTER RREH ISOLATED FORM..ARABIC LETTER RREH FINAL FORM
 [0xFB8C] = 0x0691,
 [0xFB8D] = 0x0691,
 -- ARABIC LETTER KEHEH ISOLATED FORM..ARABIC LETTER KEHEH MEDIAL FORM
 [0xFB8E] = 0x06A9,
 [0xFB8F] = 0x06A9,
 [0xFB90] = 0x06A9,
 [0xFB91] = 0x06A9,
 -- ARABIC LETTER GAF ISOLATED FORM..ARABIC LETTER GAF MEDIAL FORM
 [0xFB92] = 0x06AF,
 [0xFB93] = 0x06AF,
 [0xFB94] = 0x06AF,
 [0xFB95] = 0x06AF,
 -- ARABIC LETTER GUEH ISOLATED FORM..ARABIC LETTER GUEH MEDIAL FORM
 [0xFB96] = 0x06B3,
 [0xFB97] = 0x06B3,
 [0xFB98] = 0x06B3,
 [0xFB99] = 0x06B3,
 -- ARABIC LETTER NGOEH ISOLATED FORM..ARABIC LETTER NGOEH MEDIAL FORM
 [0xFB9A] = 0x06B1,
 [0xFB9B] = 0x06B1,
 [0xFB9C] = 0x06B1,
 [0xFB9D] = 0x06B1,
 -- ARABIC LETTER NOON GHUNNA ISOLATED FORM..ARABIC LETTER NOON GHUNNA FINAL FORM
 [0xFB9E] = 0x06BA,
 [0xFB9F] = 0x06BA,
 -- ARABIC LETTER RNOON ISOLATED FORM..ARABIC LETTER RNOON MEDIAL FORM
 [0xFBA0] = 0x06BB,
 [0xFBA1] = 0x06BB,
 [0xFBA2] = 0x06BB,
 [0xFBA3] = 0x06BB,
 -- ARABIC LETTER HEH WITH YEH ABOVE ISOLATED FORM..ARABIC LETTER HEH WITH YEH ABOVE FINAL FORM
 [0xFBA4] = 0x06C0,
 [0xFBA5] = 0x06C0,
 -- ARABIC LETTER HEH GOAL ISOLATED FORM..ARABIC LETTER HEH GOAL MEDIAL FORM
 [0xFBA6] = 0x06C1,
 [0xFBA7] = 0x06C1,
 [0xFBA8] = 0x06C1,
 [0xFBA9] = 0x06C1,
 -- ARABIC LETTER HEH DOACHASHMEE ISOLATED FORM..ARABIC LETTER HEH DOACHASHMEE MEDIAL FORM
 [0xFBAA] = 0x06BE,
 [0xFBAB] = 0x06BE,
 [0xFBAC] = 0x06BE,
 [0xFBAD] = 0x06BE,
 -- ARABIC LETTER YEH BARREE ISOLATED FORM..ARABIC LETTER YEH BARREE FINAL FORM
 [0xFBAE] = 0x06D2,
 [0xFBAF] = 0x06D2,
 -- ARABIC LETTER YEH BARREE WITH HAMZA ABOVE ISOLATED FORM..ARABIC LETTER YEH BARREE WITH HAMZA ABOVE FINAL FORM
 [0xFBB0] = 0x06D3,
 [0xFBB1] = 0x06D3,
 -- ARABIC LETTER NG ISOLATED FORM..ARABIC LETTER NG MEDIAL FORM
 [0xFBD3] = 0x06AD,
 [0xFBD4] = 0x06AD,
 [0xFBD5] = 0x06AD,
 [0xFBD6] = 0x06AD,
 -- ARABIC LETTER U ISOLATED FORM..ARABIC LETTER U FINAL FORM
 [0xFBD7] = 0x06C7,
 [0xFBD8] = 0x06C7,
 -- ARABIC LETTER OE ISOLATED FORM..ARABIC LETTER OE FINAL FORM
 [0xFBD9] = 0x06C6,
 [0xFBDA] = 0x06C6,
 [0xFBDD] = {0x06C7, 0x0674}, -- ARABIC LETTER U WITH HAMZA ABOVE ISOLATED FORM
 -- ARABIC LETTER YU ISOLATED FORM..ARABIC LETTER YU FINAL FORM
 [0xFBDB] = 0x06C8,
 [0xFBDC] = 0x06C8,
 -- ARABIC LETTER VE ISOLATED FORM..ARABIC LETTER VE FINAL FORM
 [0xFBDE] = 0x06CB,
 [0xFBDF] = 0x06CB,
 -- ARABIC LETTER KIRGHIZ OE ISOLATED FORM..ARABIC LETTER KIRGHIZ OE FINAL FORM
 [0xFBE0] = 0x06C5,
 [0xFBE1] = 0x06C5,
 -- ARABIC LETTER KIRGHIZ YU ISOLATED FORM..ARABIC LETTER KIRGHIZ YU FINAL FORM
 [0xFBE2] = 0x06C9,
 [0xFBE3] = 0x06C9,
 -- ARABIC LETTER E ISOLATED FORM..ARABIC LETTER E MEDIAL FORM
 [0xFBE4] = 0x06D0,
 [0xFBE5] = 0x06D0,
 [0xFBE6] = 0x06D0,
 [0xFBE7] = 0x06D0,
 -- ARABIC LETTER UIGHUR KAZAKH KIRGHIZ ALEF MAKSURA INITIAL FORM..ARABIC LETTER UIGHUR KAZAKH KIRGHIZ ALEF MAKSURA MEDIAL FORM
 [0xFBE8] = 0x0649,
 [0xFBE9] = 0x0649,
 -- ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ALEF ISOLATED FORM..ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ALEF FINAL FORM
 [0xFBEA] = {0x0626, 0x0627},
 [0xFBEB] = {0x0626, 0x0627},
 -- ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH AE ISOLATED FORM..ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH AE FINAL FORM
 [0xFBEC] = {0x0626, 0x06D5},
 [0xFBED] = {0x0626, 0x06D5},
 -- ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH WAW ISOLATED FORM..ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH WAW FINAL FORM
 [0xFBEE] = {0x0626, 0x0648},
 [0xFBEF] = {0x0626, 0x0648},
 -- ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH U ISOLATED FORM..ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH U FINAL FORM
 [0xFBF0] = {0x0626, 0x06C7},
 [0xFBF1] = {0x0626, 0x06C7},
 -- ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH OE ISOLATED FORM..ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH OE FINAL FORM
 [0xFBF2] = {0x0626, 0x06C6},
 [0xFBF3] = {0x0626, 0x06C6},
 -- ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH YU ISOLATED FORM..ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH YU FINAL FORM
 [0xFBF4] = {0x0626, 0x06C8},
 [0xFBF5] = {0x0626, 0x06C8},
 -- ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH E ISOLATED FORM..ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH E INITIAL FORM
 [0xFBF6] = {0x0626, 0x06D0},
 [0xFBF7] = {0x0626, 0x06D0},
 [0xFBF8] = {0x0626, 0x06D0},
 -- ARABIC LIGATURE UIGHUR KIRGHIZ YEH WITH HAMZA ABOVE WITH ALEF MAKSURA ISOLATED FORM..ARABIC LIGATURE UIGHUR KIRGHIZ YEH WITH HAMZA ABOVE WITH ALEF MAKSURA INITIAL FORM
 [0xFBF9] = {0x0626, 0x0649},
 [0xFBFA] = {0x0626, 0x0649},
 [0xFBFB] = {0x0626, 0x0649},
 [0xFC00] = {0x0626, 0x062C}, -- ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH JEEM ISOLATED FORM
 [0xFC01] = {0x0626, 0x062D}, -- ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH HAH ISOLATED FORM
 [0xFC02] = {0x0626, 0x0645}, -- ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH MEEM ISOLATED FORM
 [0xFC03] = {0x0626, 0x0649}, -- ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ALEF MAKSURA ISOLATED FORM
 [0xFC04] = {0x0626, 0x064A}, -- ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH YEH ISOLATED FORM
 [0xFC05] = {0x0628, 0x062C}, -- ARABIC LIGATURE BEH WITH JEEM ISOLATED FORM
 [0xFC06] = {0x0628, 0x062D}, -- ARABIC LIGATURE BEH WITH HAH ISOLATED FORM
 [0xFC07] = {0x0628, 0x062E}, -- ARABIC LIGATURE BEH WITH KHAH ISOLATED FORM
 [0xFC08] = {0x0628, 0x0645}, -- ARABIC LIGATURE BEH WITH MEEM ISOLATED FORM
 [0xFC09] = {0x0628, 0x0649}, -- ARABIC LIGATURE BEH WITH ALEF MAKSURA ISOLATED FORM
 [0xFC0A] = {0x0628, 0x064A}, -- ARABIC LIGATURE BEH WITH YEH ISOLATED FORM
 [0xFC0B] = {0x062A, 0x062C}, -- ARABIC LIGATURE TEH WITH JEEM ISOLATED FORM
 [0xFC0C] = {0x062A, 0x062D}, -- ARABIC LIGATURE TEH WITH HAH ISOLATED FORM
 [0xFC0D] = {0x062A, 0x062E}, -- ARABIC LIGATURE TEH WITH KHAH ISOLATED FORM
 [0xFC0E] = {0x062A, 0x0645}, -- ARABIC LIGATURE TEH WITH MEEM ISOLATED FORM
 [0xFC0F] = {0x062A, 0x0649}, -- ARABIC LIGATURE TEH WITH ALEF MAKSURA ISOLATED FORM
 [0xFC10] = {0x062A, 0x064A}, -- ARABIC LIGATURE TEH WITH YEH ISOLATED FORM
 [0xFC11] = {0x062B, 0x062C}, -- ARABIC LIGATURE THEH WITH JEEM ISOLATED FORM
 [0xFC12] = {0x062B, 0x0645}, -- ARABIC LIGATURE THEH WITH MEEM ISOLATED FORM
 [0xFC13] = {0x062B, 0x0649}, -- ARABIC LIGATURE THEH WITH ALEF MAKSURA ISOLATED FORM
 [0xFC14] = {0x062B, 0x064A}, -- ARABIC LIGATURE THEH WITH YEH ISOLATED FORM
 [0xFC15] = {0x062C, 0x062D}, -- ARABIC LIGATURE JEEM WITH HAH ISOLATED FORM
 [0xFC16] = {0x062C, 0x0645}, -- ARABIC LIGATURE JEEM WITH MEEM ISOLATED FORM
 [0xFC17] = {0x062D, 0x062C}, -- ARABIC LIGATURE HAH WITH JEEM ISOLATED FORM
 [0xFC18] = {0x062D, 0x0645}, -- ARABIC LIGATURE HAH WITH MEEM ISOLATED FORM
 [0xFC19] = {0x062E, 0x062C}, -- ARABIC LIGATURE KHAH WITH JEEM ISOLATED FORM
 [0xFC1A] = {0x062E, 0x062D}, -- ARABIC LIGATURE KHAH WITH HAH ISOLATED FORM
 [0xFC1B] = {0x062E, 0x0645}, -- ARABIC LIGATURE KHAH WITH MEEM ISOLATED FORM
 [0xFC1C] = {0x0633, 0x062C}, -- ARABIC LIGATURE SEEN WITH JEEM ISOLATED FORM
 [0xFC1D] = {0x0633, 0x062D}, -- ARABIC LIGATURE SEEN WITH HAH ISOLATED FORM
 [0xFC1E] = {0x0633, 0x062E}, -- ARABIC LIGATURE SEEN WITH KHAH ISOLATED FORM
 [0xFC1F] = {0x0633, 0x0645}, -- ARABIC LIGATURE SEEN WITH MEEM ISOLATED FORM
 [0xFC20] = {0x0635, 0x062D}, -- ARABIC LIGATURE SAD WITH HAH ISOLATED FORM
 [0xFC21] = {0x0635, 0x0645}, -- ARABIC LIGATURE SAD WITH MEEM ISOLATED FORM
 [0xFC22] = {0x0636, 0x062C}, -- ARABIC LIGATURE DAD WITH JEEM ISOLATED FORM
 [0xFC23] = {0x0636, 0x062D}, -- ARABIC LIGATURE DAD WITH HAH ISOLATED FORM
 [0xFC24] = {0x0636, 0x062E}, -- ARABIC LIGATURE DAD WITH KHAH ISOLATED FORM
 [0xFC25] = {0x0636, 0x0645}, -- ARABIC LIGATURE DAD WITH MEEM ISOLATED FORM
 [0xFC26] = {0x0637, 0x062D}, -- ARABIC LIGATURE TAH WITH HAH ISOLATED FORM
 [0xFC27] = {0x0637, 0x0645}, -- ARABIC LIGATURE TAH WITH MEEM ISOLATED FORM
 [0xFC28] = {0x0638, 0x0645}, -- ARABIC LIGATURE ZAH WITH MEEM ISOLATED FORM
 [0xFC29] = {0x0639, 0x062C}, -- ARABIC LIGATURE AIN WITH JEEM ISOLATED FORM
 [0xFC2A] = {0x0639, 0x0645}, -- ARABIC LIGATURE AIN WITH MEEM ISOLATED FORM
 [0xFC2B] = {0x063A, 0x062C}, -- ARABIC LIGATURE GHAIN WITH JEEM ISOLATED FORM
 [0xFC2C] = {0x063A, 0x0645}, -- ARABIC LIGATURE GHAIN WITH MEEM ISOLATED FORM
 [0xFC2D] = {0x0641, 0x062C}, -- ARABIC LIGATURE FEH WITH JEEM ISOLATED FORM
 [0xFC2E] = {0x0641, 0x062D}, -- ARABIC LIGATURE FEH WITH HAH ISOLATED FORM
 [0xFC2F] = {0x0641, 0x062E}, -- ARABIC LIGATURE FEH WITH KHAH ISOLATED FORM
 [0xFC30] = {0x0641, 0x0645}, -- ARABIC LIGATURE FEH WITH MEEM ISOLATED FORM
 [0xFC31] = {0x0641, 0x0649}, -- ARABIC LIGATURE FEH WITH ALEF MAKSURA ISOLATED FORM
 [0xFC32] = {0x0641, 0x064A}, -- ARABIC LIGATURE FEH WITH YEH ISOLATED FORM
 [0xFC33] = {0x0642, 0x062D}, -- ARABIC LIGATURE QAF WITH HAH ISOLATED FORM
 [0xFC34] = {0x0642, 0x0645}, -- ARABIC LIGATURE QAF WITH MEEM ISOLATED FORM
 [0xFC35] = {0x0642, 0x0649}, -- ARABIC LIGATURE QAF WITH ALEF MAKSURA ISOLATED FORM
 [0xFC36] = {0x0642, 0x064A}, -- ARABIC LIGATURE QAF WITH YEH ISOLATED FORM
 [0xFC37] = {0x0643, 0x0627}, -- ARABIC LIGATURE KAF WITH ALEF ISOLATED FORM
 [0xFC38] = {0x0643, 0x062C}, -- ARABIC LIGATURE KAF WITH JEEM ISOLATED FORM
 [0xFC39] = {0x0643, 0x062D}, -- ARABIC LIGATURE KAF WITH HAH ISOLATED FORM
 [0xFC3A] = {0x0643, 0x062E}, -- ARABIC LIGATURE KAF WITH KHAH ISOLATED FORM
 [0xFC3B] = {0x0643, 0x0644}, -- ARABIC LIGATURE KAF WITH LAM ISOLATED FORM
 [0xFC3C] = {0x0643, 0x0645}, -- ARABIC LIGATURE KAF WITH MEEM ISOLATED FORM
 [0xFC3D] = {0x0643, 0x0649}, -- ARABIC LIGATURE KAF WITH ALEF MAKSURA ISOLATED FORM
 [0xFC3E] = {0x0643, 0x064A}, -- ARABIC LIGATURE KAF WITH YEH ISOLATED FORM
 [0xFC3F] = {0x0644, 0x062C}, -- ARABIC LIGATURE LAM WITH JEEM ISOLATED FORM
 [0xFC40] = {0x0644, 0x062D}, -- ARABIC LIGATURE LAM WITH HAH ISOLATED FORM
 [0xFC41] = {0x0644, 0x062E}, -- ARABIC LIGATURE LAM WITH KHAH ISOLATED FORM
 [0xFC42] = {0x0644, 0x0645}, -- ARABIC LIGATURE LAM WITH MEEM ISOLATED FORM
 [0xFC43] = {0x0644, 0x0649}, -- ARABIC LIGATURE LAM WITH ALEF MAKSURA ISOLATED FORM
 [0xFC44] = {0x0644, 0x064A}, -- ARABIC LIGATURE LAM WITH YEH ISOLATED FORM
 [0xFC45] = {0x0645, 0x062C}, -- ARABIC LIGATURE MEEM WITH JEEM ISOLATED FORM
 [0xFC46] = {0x0645, 0x062D}, -- ARABIC LIGATURE MEEM WITH HAH ISOLATED FORM
 [0xFC47] = {0x0645, 0x062E}, -- ARABIC LIGATURE MEEM WITH KHAH ISOLATED FORM
 [0xFC48] = {0x0645, 0x0645}, -- ARABIC LIGATURE MEEM WITH MEEM ISOLATED FORM
 [0xFC49] = {0x0645, 0x0649}, -- ARABIC LIGATURE MEEM WITH ALEF MAKSURA ISOLATED FORM
 [0xFC4A] = {0x0645, 0x064A}, -- ARABIC LIGATURE MEEM WITH YEH ISOLATED FORM
 [0xFC4B] = {0x0646, 0x062C}, -- ARABIC LIGATURE NOON WITH JEEM ISOLATED FORM
 [0xFC4C] = {0x0646, 0x062D}, -- ARABIC LIGATURE NOON WITH HAH ISOLATED FORM
 [0xFC4D] = {0x0646, 0x062E}, -- ARABIC LIGATURE NOON WITH KHAH ISOLATED FORM
 [0xFC4E] = {0x0646, 0x0645}, -- ARABIC LIGATURE NOON WITH MEEM ISOLATED FORM
 [0xFC4F] = {0x0646, 0x0649}, -- ARABIC LIGATURE NOON WITH ALEF MAKSURA ISOLATED FORM
 [0xFC50] = {0x0646, 0x064A}, -- ARABIC LIGATURE NOON WITH YEH ISOLATED FORM
 [0xFC51] = {0x0647, 0x062C}, -- ARABIC LIGATURE HEH WITH JEEM ISOLATED FORM
 [0xFC52] = {0x0647, 0x0645}, -- ARABIC LIGATURE HEH WITH MEEM ISOLATED FORM
 [0xFC53] = {0x0647, 0x0649}, -- ARABIC LIGATURE HEH WITH ALEF MAKSURA ISOLATED FORM
 [0xFC54] = {0x0647, 0x064A}, -- ARABIC LIGATURE HEH WITH YEH ISOLATED FORM
 [0xFC55] = {0x064A, 0x062C}, -- ARABIC LIGATURE YEH WITH JEEM ISOLATED FORM
 [0xFC56] = {0x064A, 0x062D}, -- ARABIC LIGATURE YEH WITH HAH ISOLATED FORM
 [0xFC57] = {0x064A, 0x062E}, -- ARABIC LIGATURE YEH WITH KHAH ISOLATED FORM
 [0xFC58] = {0x064A, 0x0645}, -- ARABIC LIGATURE YEH WITH MEEM ISOLATED FORM
 [0xFC59] = {0x064A, 0x0649}, -- ARABIC LIGATURE YEH WITH ALEF MAKSURA ISOLATED FORM
 [0xFC5A] = {0x064A, 0x064A}, -- ARABIC LIGATURE YEH WITH YEH ISOLATED FORM
 [0xFC5B] = {0x0630, 0x0670}, -- ARABIC LIGATURE THAL WITH SUPERSCRIPT ALEF ISOLATED FORM
 [0xFC5C] = {0x0631, 0x0670}, -- ARABIC LIGATURE REH WITH SUPERSCRIPT ALEF ISOLATED FORM
 [0xFC5D] = {0x0649, 0x0670}, -- ARABIC LIGATURE ALEF MAKSURA WITH SUPERSCRIPT ALEF ISOLATED FORM
 [0xFC5E] = {status="disallowed"}, -- ARABIC LIGATURE SHADDA WITH DAMMATAN ISOLATED FORM
 [0xFC5F] = {status="disallowed"}, -- ARABIC LIGATURE SHADDA WITH KASRATAN ISOLATED FORM
 [0xFC60] = {status="disallowed"}, -- ARABIC LIGATURE SHADDA WITH FATHA ISOLATED FORM
 [0xFC61] = {status="disallowed"}, -- ARABIC LIGATURE SHADDA WITH DAMMA ISOLATED FORM
 [0xFC62] = {status="disallowed"}, -- ARABIC LIGATURE SHADDA WITH KASRA ISOLATED FORM
 [0xFC63] = {status="disallowed"}, -- ARABIC LIGATURE SHADDA WITH SUPERSCRIPT ALEF ISOLATED FORM
 [0xFC64] = {0x0626, 0x0631}, -- ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH REH FINAL FORM
 [0xFC65] = {0x0626, 0x0632}, -- ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ZAIN FINAL FORM
 [0xFC66] = {0x0626, 0x0645}, -- ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH MEEM FINAL FORM
 [0xFC67] = {0x0626, 0x0646}, -- ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH NOON FINAL FORM
 [0xFC68] = {0x0626, 0x0649}, -- ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH ALEF MAKSURA FINAL FORM
 [0xFC69] = {0x0626, 0x064A}, -- ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH YEH FINAL FORM
 [0xFC6A] = {0x0628, 0x0631}, -- ARABIC LIGATURE BEH WITH REH FINAL FORM
 [0xFC6B] = {0x0628, 0x0632}, -- ARABIC LIGATURE BEH WITH ZAIN FINAL FORM
 [0xFC6C] = {0x0628, 0x0645}, -- ARABIC LIGATURE BEH WITH MEEM FINAL FORM
 [0xFC6D] = {0x0628, 0x0646}, -- ARABIC LIGATURE BEH WITH NOON FINAL FORM
 [0xFC6E] = {0x0628, 0x0649}, -- ARABIC LIGATURE BEH WITH ALEF MAKSURA FINAL FORM
 [0xFC6F] = {0x0628, 0x064A}, -- ARABIC LIGATURE BEH WITH YEH FINAL FORM
 [0xFC70] = {0x062A, 0x0631}, -- ARABIC LIGATURE TEH WITH REH FINAL FORM
 [0xFC71] = {0x062A, 0x0632}, -- ARABIC LIGATURE TEH WITH ZAIN FINAL FORM
 [0xFC72] = {0x062A, 0x0645}, -- ARABIC LIGATURE TEH WITH MEEM FINAL FORM
 [0xFC73] = {0x062A, 0x0646}, -- ARABIC LIGATURE TEH WITH NOON FINAL FORM
 [0xFC74] = {0x062A, 0x0649}, -- ARABIC LIGATURE TEH WITH ALEF MAKSURA FINAL FORM
 [0xFC75] = {0x062A, 0x064A}, -- ARABIC LIGATURE TEH WITH YEH FINAL FORM
 [0xFC76] = {0x062B, 0x0631}, -- ARABIC LIGATURE THEH WITH REH FINAL FORM
 [0xFC77] = {0x062B, 0x0632}, -- ARABIC LIGATURE THEH WITH ZAIN FINAL FORM
 [0xFC78] = {0x062B, 0x0645}, -- ARABIC LIGATURE THEH WITH MEEM FINAL FORM
 [0xFC79] = {0x062B, 0x0646}, -- ARABIC LIGATURE THEH WITH NOON FINAL FORM
 [0xFC7A] = {0x062B, 0x0649}, -- ARABIC LIGATURE THEH WITH ALEF MAKSURA FINAL FORM
 [0xFC7B] = {0x062B, 0x064A}, -- ARABIC LIGATURE THEH WITH YEH FINAL FORM
 [0xFC7C] = {0x0641, 0x0649}, -- ARABIC LIGATURE FEH WITH ALEF MAKSURA FINAL FORM
 [0xFC7D] = {0x0641, 0x064A}, -- ARABIC LIGATURE FEH WITH YEH FINAL FORM
 [0xFC7E] = {0x0642, 0x0649}, -- ARABIC LIGATURE QAF WITH ALEF MAKSURA FINAL FORM
 [0xFC7F] = {0x0642, 0x064A}, -- ARABIC LIGATURE QAF WITH YEH FINAL FORM
 [0xFC80] = {0x0643, 0x0627}, -- ARABIC LIGATURE KAF WITH ALEF FINAL FORM
 [0xFC81] = {0x0643, 0x0644}, -- ARABIC LIGATURE KAF WITH LAM FINAL FORM
 [0xFC82] = {0x0643, 0x0645}, -- ARABIC LIGATURE KAF WITH MEEM FINAL FORM
 [0xFC83] = {0x0643, 0x0649}, -- ARABIC LIGATURE KAF WITH ALEF MAKSURA FINAL FORM
 [0xFC84] = {0x0643, 0x064A}, -- ARABIC LIGATURE KAF WITH YEH FINAL FORM
 [0xFC85] = {0x0644, 0x0645}, -- ARABIC LIGATURE LAM WITH MEEM FINAL FORM
 [0xFC86] = {0x0644, 0x0649}, -- ARABIC LIGATURE LAM WITH ALEF MAKSURA FINAL FORM
 [0xFC87] = {0x0644, 0x064A}, -- ARABIC LIGATURE LAM WITH YEH FINAL FORM
 [0xFC88] = {0x0645, 0x0627}, -- ARABIC LIGATURE MEEM WITH ALEF FINAL FORM
 [0xFC89] = {0x0645, 0x0645}, -- ARABIC LIGATURE MEEM WITH MEEM FINAL FORM
 [0xFC8A] = {0x0646, 0x0631}, -- ARABIC LIGATURE NOON WITH REH FINAL FORM
 [0xFC8B] = {0x0646, 0x0632}, -- ARABIC LIGATURE NOON WITH ZAIN FINAL FORM
 [0xFC8C] = {0x0646, 0x0645}, -- ARABIC LIGATURE NOON WITH MEEM FINAL FORM
 [0xFC8D] = {0x0646, 0x0646}, -- ARABIC LIGATURE NOON WITH NOON FINAL FORM
 [0xFC8E] = {0x0646, 0x0649}, -- ARABIC LIGATURE NOON WITH ALEF MAKSURA FINAL FORM
 [0xFC8F] = {0x0646, 0x064A}, -- ARABIC LIGATURE NOON WITH YEH FINAL FORM
 [0xFC90] = {0x0649, 0x0670}, -- ARABIC LIGATURE ALEF MAKSURA WITH SUPERSCRIPT ALEF FINAL FORM
 [0xFC91] = {0x064A, 0x0631}, -- ARABIC LIGATURE YEH WITH REH FINAL FORM
 [0xFC92] = {0x064A, 0x0632}, -- ARABIC LIGATURE YEH WITH ZAIN FINAL FORM
 [0xFC93] = {0x064A, 0x0645}, -- ARABIC LIGATURE YEH WITH MEEM FINAL FORM
 [0xFC94] = {0x064A, 0x0646}, -- ARABIC LIGATURE YEH WITH NOON FINAL FORM
 [0xFC95] = {0x064A, 0x0649}, -- ARABIC LIGATURE YEH WITH ALEF MAKSURA FINAL FORM
 [0xFC96] = {0x064A, 0x064A}, -- ARABIC LIGATURE YEH WITH YEH FINAL FORM
 [0xFC97] = {0x0626, 0x062C}, -- ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH JEEM INITIAL FORM
 [0xFC98] = {0x0626, 0x062D}, -- ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH HAH INITIAL FORM
 [0xFC99] = {0x0626, 0x062E}, -- ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH KHAH INITIAL FORM
 [0xFC9A] = {0x0626, 0x0645}, -- ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH MEEM INITIAL FORM
 [0xFC9B] = {0x0626, 0x0647}, -- ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH HEH INITIAL FORM
 [0xFC9C] = {0x0628, 0x062C}, -- ARABIC LIGATURE BEH WITH JEEM INITIAL FORM
 [0xFC9D] = {0x0628, 0x062D}, -- ARABIC LIGATURE BEH WITH HAH INITIAL FORM
 [0xFC9E] = {0x0628, 0x062E}, -- ARABIC LIGATURE BEH WITH KHAH INITIAL FORM
 [0xFC9F] = {0x0628, 0x0645}, -- ARABIC LIGATURE BEH WITH MEEM INITIAL FORM
 [0xFCA0] = {0x0628, 0x0647}, -- ARABIC LIGATURE BEH WITH HEH INITIAL FORM
 [0xFCA1] = {0x062A, 0x062C}, -- ARABIC LIGATURE TEH WITH JEEM INITIAL FORM
 [0xFCA2] = {0x062A, 0x062D}, -- ARABIC LIGATURE TEH WITH HAH INITIAL FORM
 [0xFCA3] = {0x062A, 0x062E}, -- ARABIC LIGATURE TEH WITH KHAH INITIAL FORM
 [0xFCA4] = {0x062A, 0x0645}, -- ARABIC LIGATURE TEH WITH MEEM INITIAL FORM
 [0xFCA5] = {0x062A, 0x0647}, -- ARABIC LIGATURE TEH WITH HEH INITIAL FORM
 [0xFCA6] = {0x062B, 0x0645}, -- ARABIC LIGATURE THEH WITH MEEM INITIAL FORM
 [0xFCA7] = {0x062C, 0x062D}, -- ARABIC LIGATURE JEEM WITH HAH INITIAL FORM
 [0xFCA8] = {0x062C, 0x0645}, -- ARABIC LIGATURE JEEM WITH MEEM INITIAL FORM
 [0xFCA9] = {0x062D, 0x062C}, -- ARABIC LIGATURE HAH WITH JEEM INITIAL FORM
 [0xFCAA] = {0x062D, 0x0645}, -- ARABIC LIGATURE HAH WITH MEEM INITIAL FORM
 [0xFCAB] = {0x062E, 0x062C}, -- ARABIC LIGATURE KHAH WITH JEEM INITIAL FORM
 [0xFCAC] = {0x062E, 0x0645}, -- ARABIC LIGATURE KHAH WITH MEEM INITIAL FORM
 [0xFCAD] = {0x0633, 0x062C}, -- ARABIC LIGATURE SEEN WITH JEEM INITIAL FORM
 [0xFCAE] = {0x0633, 0x062D}, -- ARABIC LIGATURE SEEN WITH HAH INITIAL FORM
 [0xFCAF] = {0x0633, 0x062E}, -- ARABIC LIGATURE SEEN WITH KHAH INITIAL FORM
 [0xFCB0] = {0x0633, 0x0645}, -- ARABIC LIGATURE SEEN WITH MEEM INITIAL FORM
 [0xFCB1] = {0x0635, 0x062D}, -- ARABIC LIGATURE SAD WITH HAH INITIAL FORM
 [0xFCB2] = {0x0635, 0x062E}, -- ARABIC LIGATURE SAD WITH KHAH INITIAL FORM
 [0xFCB3] = {0x0635, 0x0645}, -- ARABIC LIGATURE SAD WITH MEEM INITIAL FORM
 [0xFCB4] = {0x0636, 0x062C}, -- ARABIC LIGATURE DAD WITH JEEM INITIAL FORM
 [0xFCB5] = {0x0636, 0x062D}, -- ARABIC LIGATURE DAD WITH HAH INITIAL FORM
 [0xFCB6] = {0x0636, 0x062E}, -- ARABIC LIGATURE DAD WITH KHAH INITIAL FORM
 [0xFCB7] = {0x0636, 0x0645}, -- ARABIC LIGATURE DAD WITH MEEM INITIAL FORM
 [0xFCB8] = {0x0637, 0x062D}, -- ARABIC LIGATURE TAH WITH HAH INITIAL FORM
 [0xFCB9] = {0x0638, 0x0645}, -- ARABIC LIGATURE ZAH WITH MEEM INITIAL FORM
 [0xFCBA] = {0x0639, 0x062C}, -- ARABIC LIGATURE AIN WITH JEEM INITIAL FORM
 [0xFCBB] = {0x0639, 0x0645}, -- ARABIC LIGATURE AIN WITH MEEM INITIAL FORM
 [0xFCBC] = {0x063A, 0x062C}, -- ARABIC LIGATURE GHAIN WITH JEEM INITIAL FORM
 [0xFCBD] = {0x063A, 0x0645}, -- ARABIC LIGATURE GHAIN WITH MEEM INITIAL FORM
 [0xFCBE] = {0x0641, 0x062C}, -- ARABIC LIGATURE FEH WITH JEEM INITIAL FORM
 [0xFCBF] = {0x0641, 0x062D}, -- ARABIC LIGATURE FEH WITH HAH INITIAL FORM
 [0xFCC0] = {0x0641, 0x062E}, -- ARABIC LIGATURE FEH WITH KHAH INITIAL FORM
 [0xFCC1] = {0x0641, 0x0645}, -- ARABIC LIGATURE FEH WITH MEEM INITIAL FORM
 [0xFCC2] = {0x0642, 0x062D}, -- ARABIC LIGATURE QAF WITH HAH INITIAL FORM
 [0xFCC3] = {0x0642, 0x0645}, -- ARABIC LIGATURE QAF WITH MEEM INITIAL FORM
 [0xFCC4] = {0x0643, 0x062C}, -- ARABIC LIGATURE KAF WITH JEEM INITIAL FORM
 [0xFCC5] = {0x0643, 0x062D}, -- ARABIC LIGATURE KAF WITH HAH INITIAL FORM
 [0xFCC6] = {0x0643, 0x062E}, -- ARABIC LIGATURE KAF WITH KHAH INITIAL FORM
 [0xFCC7] = {0x0643, 0x0644}, -- ARABIC LIGATURE KAF WITH LAM INITIAL FORM
 [0xFCC8] = {0x0643, 0x0645}, -- ARABIC LIGATURE KAF WITH MEEM INITIAL FORM
 [0xFCC9] = {0x0644, 0x062C}, -- ARABIC LIGATURE LAM WITH JEEM INITIAL FORM
 [0xFCCA] = {0x0644, 0x062D}, -- ARABIC LIGATURE LAM WITH HAH INITIAL FORM
 [0xFCCB] = {0x0644, 0x062E}, -- ARABIC LIGATURE LAM WITH KHAH INITIAL FORM
 [0xFCCC] = {0x0644, 0x0645}, -- ARABIC LIGATURE LAM WITH MEEM INITIAL FORM
 [0xFCCD] = {0x0644, 0x0647}, -- ARABIC LIGATURE LAM WITH HEH INITIAL FORM
 [0xFCCE] = {0x0645, 0x062C}, -- ARABIC LIGATURE MEEM WITH JEEM INITIAL FORM
 [0xFCCF] = {0x0645, 0x062D}, -- ARABIC LIGATURE MEEM WITH HAH INITIAL FORM
 [0xFCD0] = {0x0645, 0x062E}, -- ARABIC LIGATURE MEEM WITH KHAH INITIAL FORM
 [0xFCD1] = {0x0645, 0x0645}, -- ARABIC LIGATURE MEEM WITH MEEM INITIAL FORM
 [0xFCD2] = {0x0646, 0x062C}, -- ARABIC LIGATURE NOON WITH JEEM INITIAL FORM
 [0xFCD3] = {0x0646, 0x062D}, -- ARABIC LIGATURE NOON WITH HAH INITIAL FORM
 [0xFCD4] = {0x0646, 0x062E}, -- ARABIC LIGATURE NOON WITH KHAH INITIAL FORM
 [0xFCD5] = {0x0646, 0x0645}, -- ARABIC LIGATURE NOON WITH MEEM INITIAL FORM
 [0xFCD6] = {0x0646, 0x0647}, -- ARABIC LIGATURE NOON WITH HEH INITIAL FORM
 [0xFCD7] = {0x0647, 0x062C}, -- ARABIC LIGATURE HEH WITH JEEM INITIAL FORM
 [0xFCD8] = {0x0647, 0x0645}, -- ARABIC LIGATURE HEH WITH MEEM INITIAL FORM
 [0xFCD9] = {0x0647, 0x0670}, -- ARABIC LIGATURE HEH WITH SUPERSCRIPT ALEF INITIAL FORM
 [0xFCDA] = {0x064A, 0x062C}, -- ARABIC LIGATURE YEH WITH JEEM INITIAL FORM
 [0xFCDB] = {0x064A, 0x062D}, -- ARABIC LIGATURE YEH WITH HAH INITIAL FORM
 [0xFCDC] = {0x064A, 0x062E}, -- ARABIC LIGATURE YEH WITH KHAH INITIAL FORM
 [0xFCDD] = {0x064A, 0x0645}, -- ARABIC LIGATURE YEH WITH MEEM INITIAL FORM
 [0xFCDE] = {0x064A, 0x0647}, -- ARABIC LIGATURE YEH WITH HEH INITIAL FORM
 [0xFCDF] = {0x0626, 0x0645}, -- ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH MEEM MEDIAL FORM
 [0xFCE0] = {0x0626, 0x0647}, -- ARABIC LIGATURE YEH WITH HAMZA ABOVE WITH HEH MEDIAL FORM
 [0xFCE1] = {0x0628, 0x0645}, -- ARABIC LIGATURE BEH WITH MEEM MEDIAL FORM
 [0xFCE2] = {0x0628, 0x0647}, -- ARABIC LIGATURE BEH WITH HEH MEDIAL FORM
 [0xFCE3] = {0x062A, 0x0645}, -- ARABIC LIGATURE TEH WITH MEEM MEDIAL FORM
 [0xFCE4] = {0x062A, 0x0647}, -- ARABIC LIGATURE TEH WITH HEH MEDIAL FORM
 [0xFCE5] = {0x062B, 0x0645}, -- ARABIC LIGATURE THEH WITH MEEM MEDIAL FORM
 [0xFCE6] = {0x062B, 0x0647}, -- ARABIC LIGATURE THEH WITH HEH MEDIAL FORM
 [0xFCE7] = {0x0633, 0x0645}, -- ARABIC LIGATURE SEEN WITH MEEM MEDIAL FORM
 [0xFCE8] = {0x0633, 0x0647}, -- ARABIC LIGATURE SEEN WITH HEH MEDIAL FORM
 [0xFCE9] = {0x0634, 0x0645}, -- ARABIC LIGATURE SHEEN WITH MEEM MEDIAL FORM
 [0xFCEA] = {0x0634, 0x0647}, -- ARABIC LIGATURE SHEEN WITH HEH MEDIAL FORM
 [0xFCEB] = {0x0643, 0x0644}, -- ARABIC LIGATURE KAF WITH LAM MEDIAL FORM
 [0xFCEC] = {0x0643, 0x0645}, -- ARABIC LIGATURE KAF WITH MEEM MEDIAL FORM
 [0xFCED] = {0x0644, 0x0645}, -- ARABIC LIGATURE LAM WITH MEEM MEDIAL FORM
 [0xFCEE] = {0x0646, 0x0645}, -- ARABIC LIGATURE NOON WITH MEEM MEDIAL FORM
 [0xFCEF] = {0x0646, 0x0647}, -- ARABIC LIGATURE NOON WITH HEH MEDIAL FORM
 [0xFCF0] = {0x064A, 0x0645}, -- ARABIC LIGATURE YEH WITH MEEM MEDIAL FORM
 [0xFCF1] = {0x064A, 0x0647}, -- ARABIC LIGATURE YEH WITH HEH MEDIAL FORM
 [0xFCF2] = {0x0640, 0x064E, 0x0651}, -- ARABIC LIGATURE SHADDA WITH FATHA MEDIAL FORM
 [0xFCF3] = {0x0640, 0x064F, 0x0651}, -- ARABIC LIGATURE SHADDA WITH DAMMA MEDIAL FORM
 [0xFCF4] = {0x0640, 0x0650, 0x0651}, -- ARABIC LIGATURE SHADDA WITH KASRA MEDIAL FORM
 [0xFCF5] = {0x0637, 0x0649}, -- ARABIC LIGATURE TAH WITH ALEF MAKSURA ISOLATED FORM
 [0xFCF6] = {0x0637, 0x064A}, -- ARABIC LIGATURE TAH WITH YEH ISOLATED FORM
 [0xFCF7] = {0x0639, 0x0649}, -- ARABIC LIGATURE AIN WITH ALEF MAKSURA ISOLATED FORM
 [0xFCF8] = {0x0639, 0x064A}, -- ARABIC LIGATURE AIN WITH YEH ISOLATED FORM
 [0xFCF9] = {0x063A, 0x0649}, -- ARABIC LIGATURE GHAIN WITH ALEF MAKSURA ISOLATED FORM
 [0xFCFA] = {0x063A, 0x064A}, -- ARABIC LIGATURE GHAIN WITH YEH ISOLATED FORM
 [0xFCFB] = {0x0633, 0x0649}, -- ARABIC LIGATURE SEEN WITH ALEF MAKSURA ISOLATED FORM
 [0xFCFC] = {0x0633, 0x064A}, -- ARABIC LIGATURE SEEN WITH YEH ISOLATED FORM
 [0xFCFD] = {0x0634, 0x0649}, -- ARABIC LIGATURE SHEEN WITH ALEF MAKSURA ISOLATED FORM
 [0xFCFE] = {0x0634, 0x064A}, -- ARABIC LIGATURE SHEEN WITH YEH ISOLATED FORM
 [0xFCFF] = {0x062D, 0x0649}, -- ARABIC LIGATURE HAH WITH ALEF MAKSURA ISOLATED FORM
 [0xFD00] = {0x062D, 0x064A}, -- ARABIC LIGATURE HAH WITH YEH ISOLATED FORM
 [0xFD01] = {0x062C, 0x0649}, -- ARABIC LIGATURE JEEM WITH ALEF MAKSURA ISOLATED FORM
 [0xFD02] = {0x062C, 0x064A}, -- ARABIC LIGATURE JEEM WITH YEH ISOLATED FORM
 [0xFD03] = {0x062E, 0x0649}, -- ARABIC LIGATURE KHAH WITH ALEF MAKSURA ISOLATED FORM
 [0xFD04] = {0x062E, 0x064A}, -- ARABIC LIGATURE KHAH WITH YEH ISOLATED FORM
 [0xFD05] = {0x0635, 0x0649}, -- ARABIC LIGATURE SAD WITH ALEF MAKSURA ISOLATED FORM
 [0xFD06] = {0x0635, 0x064A}, -- ARABIC LIGATURE SAD WITH YEH ISOLATED FORM
 [0xFD07] = {0x0636, 0x0649}, -- ARABIC LIGATURE DAD WITH ALEF MAKSURA ISOLATED FORM
 [0xFD08] = {0x0636, 0x064A}, -- ARABIC LIGATURE DAD WITH YEH ISOLATED FORM
 [0xFD09] = {0x0634, 0x062C}, -- ARABIC LIGATURE SHEEN WITH JEEM ISOLATED FORM
 [0xFD0A] = {0x0634, 0x062D}, -- ARABIC LIGATURE SHEEN WITH HAH ISOLATED FORM
 [0xFD0B] = {0x0634, 0x062E}, -- ARABIC LIGATURE SHEEN WITH KHAH ISOLATED FORM
 [0xFD0C] = {0x0634, 0x0645}, -- ARABIC LIGATURE SHEEN WITH MEEM ISOLATED FORM
 [0xFD0D] = {0x0634, 0x0631}, -- ARABIC LIGATURE SHEEN WITH REH ISOLATED FORM
 [0xFD0E] = {0x0633, 0x0631}, -- ARABIC LIGATURE SEEN WITH REH ISOLATED FORM
 [0xFD0F] = {0x0635, 0x0631}, -- ARABIC LIGATURE SAD WITH REH ISOLATED FORM
 [0xFD10] = {0x0636, 0x0631}, -- ARABIC LIGATURE DAD WITH REH ISOLATED FORM
 [0xFD11] = {0x0637, 0x0649}, -- ARABIC LIGATURE TAH WITH ALEF MAKSURA FINAL FORM
 [0xFD12] = {0x0637, 0x064A}, -- ARABIC LIGATURE TAH WITH YEH FINAL FORM
 [0xFD13] = {0x0639, 0x0649}, -- ARABIC LIGATURE AIN WITH ALEF MAKSURA FINAL FORM
 [0xFD14] = {0x0639, 0x064A}, -- ARABIC LIGATURE AIN WITH YEH FINAL FORM
 [0xFD15] = {0x063A, 0x0649}, -- ARABIC LIGATURE GHAIN WITH ALEF MAKSURA FINAL FORM
 [0xFD16] = {0x063A, 0x064A}, -- ARABIC LIGATURE GHAIN WITH YEH FINAL FORM
 [0xFD17] = {0x0633, 0x0649}, -- ARABIC LIGATURE SEEN WITH ALEF MAKSURA FINAL FORM
 [0xFD18] = {0x0633, 0x064A}, -- ARABIC LIGATURE SEEN WITH YEH FINAL FORM
 [0xFD19] = {0x0634, 0x0649}, -- ARABIC LIGATURE SHEEN WITH ALEF MAKSURA FINAL FORM
 [0xFD1A] = {0x0634, 0x064A}, -- ARABIC LIGATURE SHEEN WITH YEH FINAL FORM
 [0xFD1B] = {0x062D, 0x0649}, -- ARABIC LIGATURE HAH WITH ALEF MAKSURA FINAL FORM
 [0xFD1C] = {0x062D, 0x064A}, -- ARABIC LIGATURE HAH WITH YEH FINAL FORM
 [0xFD1D] = {0x062C, 0x0649}, -- ARABIC LIGATURE JEEM WITH ALEF MAKSURA FINAL FORM
 [0xFD1E] = {0x062C, 0x064A}, -- ARABIC LIGATURE JEEM WITH YEH FINAL FORM
 [0xFD1F] = {0x062E, 0x0649}, -- ARABIC LIGATURE KHAH WITH ALEF MAKSURA FINAL FORM
 [0xFD20] = {0x062E, 0x064A}, -- ARABIC LIGATURE KHAH WITH YEH FINAL FORM
 [0xFD21] = {0x0635, 0x0649}, -- ARABIC LIGATURE SAD WITH ALEF MAKSURA FINAL FORM
 [0xFD22] = {0x0635, 0x064A}, -- ARABIC LIGATURE SAD WITH YEH FINAL FORM
 [0xFD23] = {0x0636, 0x0649}, -- ARABIC LIGATURE DAD WITH ALEF MAKSURA FINAL FORM
 [0xFD24] = {0x0636, 0x064A}, -- ARABIC LIGATURE DAD WITH YEH FINAL FORM
 [0xFD25] = {0x0634, 0x062C}, -- ARABIC LIGATURE SHEEN WITH JEEM FINAL FORM
 [0xFD26] = {0x0634, 0x062D}, -- ARABIC LIGATURE SHEEN WITH HAH FINAL FORM
 [0xFD27] = {0x0634, 0x062E}, -- ARABIC LIGATURE SHEEN WITH KHAH FINAL FORM
 [0xFD28] = {0x0634, 0x0645}, -- ARABIC LIGATURE SHEEN WITH MEEM FINAL FORM
 [0xFD29] = {0x0634, 0x0631}, -- ARABIC LIGATURE SHEEN WITH REH FINAL FORM
 [0xFD2A] = {0x0633, 0x0631}, -- ARABIC LIGATURE SEEN WITH REH FINAL FORM
 [0xFD2B] = {0x0635, 0x0631}, -- ARABIC LIGATURE SAD WITH REH FINAL FORM
 [0xFD2C] = {0x0636, 0x0631}, -- ARABIC LIGATURE DAD WITH REH FINAL FORM
 [0xFD2D] = {0x0634, 0x062C}, -- ARABIC LIGATURE SHEEN WITH JEEM INITIAL FORM
 [0xFD2E] = {0x0634, 0x062D}, -- ARABIC LIGATURE SHEEN WITH HAH INITIAL FORM
 [0xFD2F] = {0x0634, 0x062E}, -- ARABIC LIGATURE SHEEN WITH KHAH INITIAL FORM
 [0xFD30] = {0x0634, 0x0645}, -- ARABIC LIGATURE SHEEN WITH MEEM INITIAL FORM
 [0xFD31] = {0x0633, 0x0647}, -- ARABIC LIGATURE SEEN WITH HEH INITIAL FORM
 [0xFD32] = {0x0634, 0x0647}, -- ARABIC LIGATURE SHEEN WITH HEH INITIAL FORM
 [0xFD33] = {0x0637, 0x0645}, -- ARABIC LIGATURE TAH WITH MEEM INITIAL FORM
 [0xFD34] = {0x0633, 0x062C}, -- ARABIC LIGATURE SEEN WITH JEEM MEDIAL FORM
 [0xFD35] = {0x0633, 0x062D}, -- ARABIC LIGATURE SEEN WITH HAH MEDIAL FORM
 [0xFD36] = {0x0633, 0x062E}, -- ARABIC LIGATURE SEEN WITH KHAH MEDIAL FORM
 [0xFD37] = {0x0634, 0x062C}, -- ARABIC LIGATURE SHEEN WITH JEEM MEDIAL FORM
 [0xFD38] = {0x0634, 0x062D}, -- ARABIC LIGATURE SHEEN WITH HAH MEDIAL FORM
 [0xFD39] = {0x0634, 0x062E}, -- ARABIC LIGATURE SHEEN WITH KHAH MEDIAL FORM
 [0xFD3A] = {0x0637, 0x0645}, -- ARABIC LIGATURE TAH WITH MEEM MEDIAL FORM
 [0xFD3B] = {0x0638, 0x0645}, -- ARABIC LIGATURE ZAH WITH MEEM MEDIAL FORM
 -- ARABIC LETTER FARSI YEH ISOLATED FORM..ARABIC LETTER FARSI YEH MEDIAL FORM
 [0xFBFC] = 0x06CC,
 [0xFBFD] = 0x06CC,
 [0xFBFE] = 0x06CC,
 [0xFBFF] = 0x06CC,
 -- ARABIC LIGATURE ALEF WITH FATHATAN FINAL FORM..ARABIC LIGATURE ALEF WITH FATHATAN ISOLATED FORM
 [0xFD3C] = {0x0627, 0x064B},
 [0xFD3D] = {0x0627, 0x064B},
 -- ORNATE LEFT PARENTHESIS..ORNATE RIGHT PARENTHESIS
 [0xFD3E] = {status="disallowed"},
 [0xFD3F] = {status="disallowed"},
 [0xFD50] = {0x062A, 0x062C, 0x0645}, -- ARABIC LIGATURE TEH WITH JEEM WITH MEEM INITIAL FORM
 [0xFD53] = {0x062A, 0x062D, 0x0645}, -- ARABIC LIGATURE TEH WITH HAH WITH MEEM INITIAL FORM
 [0xFD54] = {0x062A, 0x062E, 0x0645}, -- ARABIC LIGATURE TEH WITH KHAH WITH MEEM INITIAL FORM
 [0xFD55] = {0x062A, 0x0645, 0x062C}, -- ARABIC LIGATURE TEH WITH MEEM WITH JEEM INITIAL FORM
 [0xFD56] = {0x062A, 0x0645, 0x062D}, -- ARABIC LIGATURE TEH WITH MEEM WITH HAH INITIAL FORM
 [0xFD57] = {0x062A, 0x0645, 0x062E}, -- ARABIC LIGATURE TEH WITH MEEM WITH KHAH INITIAL FORM
 -- ARABIC LIGATURE TEH WITH HAH WITH JEEM FINAL FORM..ARABIC LIGATURE TEH WITH HAH WITH JEEM INITIAL FORM
 [0xFD51] = {0x062A, 0x062D, 0x062C},
 [0xFD52] = {0x062A, 0x062D, 0x062C},
 [0xFD5A] = {0x062D, 0x0645, 0x064A}, -- ARABIC LIGATURE HAH WITH MEEM WITH YEH FINAL FORM
 [0xFD5B] = {0x062D, 0x0645, 0x0649}, -- ARABIC LIGATURE HAH WITH MEEM WITH ALEF MAKSURA FINAL FORM
 [0xFD5C] = {0x0633, 0x062D, 0x062C}, -- ARABIC LIGATURE SEEN WITH HAH WITH JEEM INITIAL FORM
 [0xFD5D] = {0x0633, 0x062C, 0x062D}, -- ARABIC LIGATURE SEEN WITH JEEM WITH HAH INITIAL FORM
 [0xFD5E] = {0x0633, 0x062C, 0x0649}, -- ARABIC LIGATURE SEEN WITH JEEM WITH ALEF MAKSURA FINAL FORM
 -- ARABIC LIGATURE JEEM WITH MEEM WITH HAH FINAL FORM..ARABIC LIGATURE JEEM WITH MEEM WITH HAH INITIAL FORM
 [0xFD58] = {0x062C, 0x0645, 0x062D},
 [0xFD59] = {0x062C, 0x0645, 0x062D},
 [0xFD61] = {0x0633, 0x0645, 0x062C}, -- ARABIC LIGATURE SEEN WITH MEEM WITH JEEM INITIAL FORM
 -- ARABIC LIGATURE SEEN WITH MEEM WITH HAH FINAL FORM..ARABIC LIGATURE SEEN WITH MEEM WITH HAH INITIAL FORM
 [0xFD5F] = {0x0633, 0x0645, 0x062D},
 [0xFD60] = {0x0633, 0x0645, 0x062D},
 -- ARABIC LIGATURE SEEN WITH MEEM WITH MEEM FINAL FORM..ARABIC LIGATURE SEEN WITH MEEM WITH MEEM INITIAL FORM
 [0xFD62] = {0x0633, 0x0645, 0x0645},
 [0xFD63] = {0x0633, 0x0645, 0x0645},
 [0xFD66] = {0x0635, 0x0645, 0x0645}, -- ARABIC LIGATURE SAD WITH MEEM WITH MEEM FINAL FORM
 -- ARABIC LIGATURE SAD WITH HAH WITH HAH FINAL FORM..ARABIC LIGATURE SAD WITH HAH WITH HAH INITIAL FORM
 [0xFD64] = {0x0635, 0x062D, 0x062D},
 [0xFD65] = {0x0635, 0x062D, 0x062D},
 [0xFD69] = {0x0634, 0x062C, 0x064A}, -- ARABIC LIGATURE SHEEN WITH JEEM WITH YEH FINAL FORM
 -- ARABIC LIGATURE SHEEN WITH HAH WITH MEEM FINAL FORM..ARABIC LIGATURE SHEEN WITH HAH WITH MEEM INITIAL FORM
 [0xFD67] = {0x0634, 0x062D, 0x0645},
 [0xFD68] = {0x0634, 0x062D, 0x0645},
 -- ARABIC LIGATURE SHEEN WITH MEEM WITH KHAH FINAL FORM..ARABIC LIGATURE SHEEN WITH MEEM WITH KHAH INITIAL FORM
 [0xFD6A] = {0x0634, 0x0645, 0x062E},
 [0xFD6B] = {0x0634, 0x0645, 0x062E},
 [0xFD6E] = {0x0636, 0x062D, 0x0649}, -- ARABIC LIGATURE DAD WITH HAH WITH ALEF MAKSURA FINAL FORM
 -- ARABIC LIGATURE SHEEN WITH MEEM WITH MEEM FINAL FORM..ARABIC LIGATURE SHEEN WITH MEEM WITH MEEM INITIAL FORM
 [0xFD6C] = {0x0634, 0x0645, 0x0645},
 [0xFD6D] = {0x0634, 0x0645, 0x0645},
 -- ARABIC LIGATURE DAD WITH KHAH WITH MEEM FINAL FORM..ARABIC LIGATURE DAD WITH KHAH WITH MEEM INITIAL FORM
 [0xFD6F] = {0x0636, 0x062E, 0x0645},
 [0xFD70] = {0x0636, 0x062E, 0x0645},
 [0xFD73] = {0x0637, 0x0645, 0x0645}, -- ARABIC LIGATURE TAH WITH MEEM WITH MEEM INITIAL FORM
 [0xFD74] = {0x0637, 0x0645, 0x064A}, -- ARABIC LIGATURE TAH WITH MEEM WITH YEH FINAL FORM
 [0xFD75] = {0x0639, 0x062C, 0x0645}, -- ARABIC LIGATURE AIN WITH JEEM WITH MEEM FINAL FORM
 -- ARABIC LIGATURE TAH WITH MEEM WITH HAH FINAL FORM..ARABIC LIGATURE TAH WITH MEEM WITH HAH INITIAL FORM
 [0xFD71] = {0x0637, 0x0645, 0x062D},
 [0xFD72] = {0x0637, 0x0645, 0x062D},
 [0xFD78] = {0x0639, 0x0645, 0x0649}, -- ARABIC LIGATURE AIN WITH MEEM WITH ALEF MAKSURA FINAL FORM
 [0xFD79] = {0x063A, 0x0645, 0x0645}, -- ARABIC LIGATURE GHAIN WITH MEEM WITH MEEM FINAL FORM
 [0xFD7A] = {0x063A, 0x0645, 0x064A}, -- ARABIC LIGATURE GHAIN WITH MEEM WITH YEH FINAL FORM
 [0xFD7B] = {0x063A, 0x0645, 0x0649}, -- ARABIC LIGATURE GHAIN WITH MEEM WITH ALEF MAKSURA FINAL FORM
 -- ARABIC LIGATURE AIN WITH MEEM WITH MEEM FINAL FORM..ARABIC LIGATURE AIN WITH MEEM WITH MEEM INITIAL FORM
 [0xFD76] = {0x0639, 0x0645, 0x0645},
 [0xFD77] = {0x0639, 0x0645, 0x0645},
 [0xFD7E] = {0x0642, 0x0645, 0x062D}, -- ARABIC LIGATURE QAF WITH MEEM WITH HAH FINAL FORM
 [0xFD7F] = {0x0642, 0x0645, 0x0645}, -- ARABIC LIGATURE QAF WITH MEEM WITH MEEM FINAL FORM
 [0xFD80] = {0x0644, 0x062D, 0x0645}, -- ARABIC LIGATURE LAM WITH HAH WITH MEEM FINAL FORM
 [0xFD81] = {0x0644, 0x062D, 0x064A}, -- ARABIC LIGATURE LAM WITH HAH WITH YEH FINAL FORM
 [0xFD82] = {0x0644, 0x062D, 0x0649}, -- ARABIC LIGATURE LAM WITH HAH WITH ALEF MAKSURA FINAL FORM
 -- ARABIC LIGATURE FEH WITH KHAH WITH MEEM FINAL FORM..ARABIC LIGATURE FEH WITH KHAH WITH MEEM INITIAL FORM
 [0xFD7C] = {0x0641, 0x062E, 0x0645},
 [0xFD7D] = {0x0641, 0x062E, 0x0645},
 -- ARABIC LIGATURE LAM WITH JEEM WITH JEEM INITIAL FORM..ARABIC LIGATURE LAM WITH JEEM WITH JEEM FINAL FORM
 [0xFD83] = {0x0644, 0x062C, 0x062C},
 [0xFD84] = {0x0644, 0x062C, 0x062C},
 -- ARABIC LIGATURE LAM WITH KHAH WITH MEEM FINAL FORM..ARABIC LIGATURE LAM WITH KHAH WITH MEEM INITIAL FORM
 [0xFD85] = {0x0644, 0x062E, 0x0645},
 [0xFD86] = {0x0644, 0x062E, 0x0645},
 [0xFD89] = {0x0645, 0x062D, 0x062C}, -- ARABIC LIGATURE MEEM WITH HAH WITH JEEM INITIAL FORM
 [0xFD8A] = {0x0645, 0x062D, 0x0645}, -- ARABIC LIGATURE MEEM WITH HAH WITH MEEM INITIAL FORM
 [0xFD8B] = {0x0645, 0x062D, 0x064A}, -- ARABIC LIGATURE MEEM WITH HAH WITH YEH FINAL FORM
 [0xFD8C] = {0x0645, 0x062C, 0x062D}, -- ARABIC LIGATURE MEEM WITH JEEM WITH HAH INITIAL FORM
 [0xFD8D] = {0x0645, 0x062C, 0x0645}, -- ARABIC LIGATURE MEEM WITH JEEM WITH MEEM INITIAL FORM
 [0xFD8E] = {0x0645, 0x062E, 0x062C}, -- ARABIC LIGATURE MEEM WITH KHAH WITH JEEM INITIAL FORM
 [0xFD8F] = {0x0645, 0x062E, 0x0645}, -- ARABIC LIGATURE MEEM WITH KHAH WITH MEEM INITIAL FORM
 -- ARABIC LIGATURE LAM WITH MEEM WITH HAH FINAL FORM..ARABIC LIGATURE LAM WITH MEEM WITH HAH INITIAL FORM
 [0xFD87] = {0x0644, 0x0645, 0x062D},
 [0xFD88] = {0x0644, 0x0645, 0x062D},
 [0xFD92] = {0x0645, 0x062C, 0x062E}, -- ARABIC LIGATURE MEEM WITH JEEM WITH KHAH INITIAL FORM
 [0xFD93] = {0x0647, 0x0645, 0x062C}, -- ARABIC LIGATURE HEH WITH MEEM WITH JEEM INITIAL FORM
 [0xFD94] = {0x0647, 0x0645, 0x0645}, -- ARABIC LIGATURE HEH WITH MEEM WITH MEEM INITIAL FORM
 [0xFD95] = {0x0646, 0x062D, 0x0645}, -- ARABIC LIGATURE NOON WITH HAH WITH MEEM INITIAL FORM
 [0xFD96] = {0x0646, 0x062D, 0x0649}, -- ARABIC LIGATURE NOON WITH HAH WITH ALEF MAKSURA FINAL FORM
 -- NA   <reserved-FD90>..<reserved-FD91>
 [0xFD90] = {status="disallowed"},
 [0xFD91] = {status="disallowed"},
 [0xFD99] = {0x0646, 0x062C, 0x0649}, -- ARABIC LIGATURE NOON WITH JEEM WITH ALEF MAKSURA FINAL FORM
 [0xFD9A] = {0x0646, 0x0645, 0x064A}, -- ARABIC LIGATURE NOON WITH MEEM WITH YEH FINAL FORM
 [0xFD9B] = {0x0646, 0x0645, 0x0649}, -- ARABIC LIGATURE NOON WITH MEEM WITH ALEF MAKSURA FINAL FORM
 -- ARABIC LIGATURE NOON WITH JEEM WITH MEEM FINAL FORM..ARABIC LIGATURE NOON WITH JEEM WITH MEEM INITIAL FORM
 [0xFD97] = {0x0646, 0x062C, 0x0645},
 [0xFD98] = {0x0646, 0x062C, 0x0645},
 [0xFD9E] = {0x0628, 0x062E, 0x064A}, -- ARABIC LIGATURE BEH WITH KHAH WITH YEH FINAL FORM
 [0xFD9F] = {0x062A, 0x062C, 0x064A}, -- ARABIC LIGATURE TEH WITH JEEM WITH YEH FINAL FORM
 [0xFDA0] = {0x062A, 0x062C, 0x0649}, -- ARABIC LIGATURE TEH WITH JEEM WITH ALEF MAKSURA FINAL FORM
 [0xFDA1] = {0x062A, 0x062E, 0x064A}, -- ARABIC LIGATURE TEH WITH KHAH WITH YEH FINAL FORM
 [0xFDA2] = {0x062A, 0x062E, 0x0649}, -- ARABIC LIGATURE TEH WITH KHAH WITH ALEF MAKSURA FINAL FORM
 [0xFDA3] = {0x062A, 0x0645, 0x064A}, -- ARABIC LIGATURE TEH WITH MEEM WITH YEH FINAL FORM
 [0xFDA4] = {0x062A, 0x0645, 0x0649}, -- ARABIC LIGATURE TEH WITH MEEM WITH ALEF MAKSURA FINAL FORM
 [0xFDA5] = {0x062C, 0x0645, 0x064A}, -- ARABIC LIGATURE JEEM WITH MEEM WITH YEH FINAL FORM
 [0xFDA6] = {0x062C, 0x062D, 0x0649}, -- ARABIC LIGATURE JEEM WITH HAH WITH ALEF MAKSURA FINAL FORM
 [0xFDA7] = {0x062C, 0x0645, 0x0649}, -- ARABIC LIGATURE JEEM WITH MEEM WITH ALEF MAKSURA FINAL FORM
 [0xFDA8] = {0x0633, 0x062E, 0x0649}, -- ARABIC LIGATURE SEEN WITH KHAH WITH ALEF MAKSURA FINAL FORM
 [0xFDA9] = {0x0635, 0x062D, 0x064A}, -- ARABIC LIGATURE SAD WITH HAH WITH YEH FINAL FORM
 [0xFDAA] = {0x0634, 0x062D, 0x064A}, -- ARABIC LIGATURE SHEEN WITH HAH WITH YEH FINAL FORM
 [0xFDAB] = {0x0636, 0x062D, 0x064A}, -- ARABIC LIGATURE DAD WITH HAH WITH YEH FINAL FORM
 [0xFDAC] = {0x0644, 0x062C, 0x064A}, -- ARABIC LIGATURE LAM WITH JEEM WITH YEH FINAL FORM
 [0xFDAD] = {0x0644, 0x0645, 0x064A}, -- ARABIC LIGATURE LAM WITH MEEM WITH YEH FINAL FORM
 [0xFDAE] = {0x064A, 0x062D, 0x064A}, -- ARABIC LIGATURE YEH WITH HAH WITH YEH FINAL FORM
 [0xFDAF] = {0x064A, 0x062C, 0x064A}, -- ARABIC LIGATURE YEH WITH JEEM WITH YEH FINAL FORM
 [0xFDB0] = {0x064A, 0x0645, 0x064A}, -- ARABIC LIGATURE YEH WITH MEEM WITH YEH FINAL FORM
 [0xFDB1] = {0x0645, 0x0645, 0x064A}, -- ARABIC LIGATURE MEEM WITH MEEM WITH YEH FINAL FORM
 [0xFDB2] = {0x0642, 0x0645, 0x064A}, -- ARABIC LIGATURE QAF WITH MEEM WITH YEH FINAL FORM
 [0xFDB3] = {0x0646, 0x062D, 0x064A}, -- ARABIC LIGATURE NOON WITH HAH WITH YEH FINAL FORM
 [0xFDB4] = {0x0642, 0x0645, 0x062D}, -- ARABIC LIGATURE QAF WITH MEEM WITH HAH INITIAL FORM
 [0xFDB5] = {0x0644, 0x062D, 0x0645}, -- ARABIC LIGATURE LAM WITH HAH WITH MEEM INITIAL FORM
 [0xFDB6] = {0x0639, 0x0645, 0x064A}, -- ARABIC LIGATURE AIN WITH MEEM WITH YEH FINAL FORM
 [0xFDB7] = {0x0643, 0x0645, 0x064A}, -- ARABIC LIGATURE KAF WITH MEEM WITH YEH FINAL FORM
 [0xFDB8] = {0x0646, 0x062C, 0x062D}, -- ARABIC LIGATURE NOON WITH JEEM WITH HAH INITIAL FORM
 [0xFDB9] = {0x0645, 0x062E, 0x064A}, -- ARABIC LIGATURE MEEM WITH KHAH WITH YEH FINAL FORM
 [0xFDBA] = {0x0644, 0x062C, 0x0645}, -- ARABIC LIGATURE LAM WITH JEEM WITH MEEM INITIAL FORM
 [0xFDBB] = {0x0643, 0x0645, 0x0645}, -- ARABIC LIGATURE KAF WITH MEEM WITH MEEM FINAL FORM
 [0xFDBC] = {0x0644, 0x062C, 0x0645}, -- ARABIC LIGATURE LAM WITH JEEM WITH MEEM FINAL FORM
 [0xFDBD] = {0x0646, 0x062C, 0x062D}, -- ARABIC LIGATURE NOON WITH JEEM WITH HAH FINAL FORM
 [0xFDBE] = {0x062C, 0x062D, 0x064A}, -- ARABIC LIGATURE JEEM WITH HAH WITH YEH FINAL FORM
 [0xFDBF] = {0x062D, 0x062C, 0x064A}, -- ARABIC LIGATURE HAH WITH JEEM WITH YEH FINAL FORM
 [0xFDC0] = {0x0645, 0x062C, 0x064A}, -- ARABIC LIGATURE MEEM WITH JEEM WITH YEH FINAL FORM
 [0xFDC1] = {0x0641, 0x0645, 0x064A}, -- ARABIC LIGATURE FEH WITH MEEM WITH YEH FINAL FORM
 [0xFDC2] = {0x0628, 0x062D, 0x064A}, -- ARABIC LIGATURE BEH WITH HAH WITH YEH FINAL FORM
 [0xFDC3] = {0x0643, 0x0645, 0x0645}, -- ARABIC LIGATURE KAF WITH MEEM WITH MEEM INITIAL FORM
 [0xFDC4] = {0x0639, 0x062C, 0x0645}, -- ARABIC LIGATURE AIN WITH JEEM WITH MEEM INITIAL FORM
 [0xFDC5] = {0x0635, 0x0645, 0x0645}, -- ARABIC LIGATURE SAD WITH MEEM WITH MEEM INITIAL FORM
 [0xFDC6] = {0x0633, 0x062E, 0x064A}, -- ARABIC LIGATURE SEEN WITH KHAH WITH YEH FINAL FORM
 [0xFDC7] = {0x0646, 0x062C, 0x064A}, -- ARABIC LIGATURE NOON WITH JEEM WITH YEH FINAL FORM
 -- ARABIC LIGATURE YEH WITH MEEM WITH MEEM FINAL FORM..ARABIC LIGATURE YEH WITH MEEM WITH MEEM INITIAL FORM
 [0xFD9C] = {0x064A, 0x0645, 0x0645},
 [0xFD9D] = {0x064A, 0x0645, 0x0645},
 [0xFDF0] = {0x0635, 0x0644, 0x06D2}, -- ARABIC LIGATURE SALLA USED AS KORANIC STOP SIGN ISOLATED FORM
 [0xFDF1] = {0x0642, 0x0644, 0x06D2}, -- ARABIC LIGATURE QALA USED AS KORANIC STOP SIGN ISOLATED FORM
 [0xFDF2] = {0x0627, 0x0644, 0x0644, 0x0647}, -- ARABIC LIGATURE ALLAH ISOLATED FORM
 [0xFDF3] = {0x0627, 0x0643, 0x0628, 0x0631}, -- ARABIC LIGATURE AKBAR ISOLATED FORM
 [0xFDF4] = {0x0645, 0x062D, 0x0645, 0x062F}, -- ARABIC LIGATURE MOHAMMAD ISOLATED FORM
 [0xFDF5] = {0x0635, 0x0644, 0x0639, 0x0645}, -- ARABIC LIGATURE SALAM ISOLATED FORM
 [0xFDF6] = {0x0631, 0x0633, 0x0648, 0x0644}, -- ARABIC LIGATURE RASOUL ISOLATED FORM
 [0xFDF7] = {0x0639, 0x0644, 0x064A, 0x0647}, -- ARABIC LIGATURE ALAYHE ISOLATED FORM
 [0xFDF8] = {0x0648, 0x0633, 0x0644, 0x0645}, -- ARABIC LIGATURE WASALLAM ISOLATED FORM
 [0xFDF9] = {0x0635, 0x0644, 0x0649}, -- ARABIC LIGATURE SALLA ISOLATED FORM
 [0xFDFA] = {status="disallowed"}, -- ARABIC LIGATURE SALLALLAHOU ALAYHE WASALLAM
 [0xFDFB] = {status="disallowed"}, -- ARABIC LIGATURE JALLAJALALOUHOU
 [0xFDFC] = {0x0631, 0x06CC, 0x0627, 0x0644}, -- RIAL SIGN
 [0xFDFD] = {status="disallowed"}, -- ARABIC LIGATURE BISMILLAH AR-RAHMAN AR-RAHEEM
 -- NA   <reserved-FDFE>..<reserved-FDFF>
 [0xFDFE] = {status="disallowed"},
 [0xFDFF] = {status="disallowed"},
 [0xFE10] = {status="disallowed"}, -- PRESENTATION FORM FOR VERTICAL COMMA
 [0xFE11] = 0x3001, -- PRESENTATION FORM FOR VERTICAL IDEOGRAPHIC COMMA
 [0xFE12] = {status="disallowed"}, -- PRESENTATION FORM FOR VERTICAL IDEOGRAPHIC FULL STOP
 [0xFE13] = {status="disallowed"}, -- PRESENTATION FORM FOR VERTICAL COLON
 [0xFE14] = {status="disallowed"}, -- PRESENTATION FORM FOR VERTICAL SEMICOLON
 [0xFE15] = {status="disallowed"}, -- PRESENTATION FORM FOR VERTICAL EXCLAMATION MARK
 [0xFE16] = {status="disallowed"}, -- PRESENTATION FORM FOR VERTICAL QUESTION MARK
 [0xFE17] = 0x3016, -- PRESENTATION FORM FOR VERTICAL LEFT WHITE LENTICULAR BRACKET
 [0xFE18] = 0x3017, -- PRESENTATION FORM FOR VERTICAL RIGHT WHITE LENTICULAR BRAKCET
 [0xFE19] = {status="disallowed"}, -- PRESENTATION FORM FOR VERTICAL HORIZONTAL ELLIPSIS
 [0xFE30] = {status="disallowed"}, -- PRESENTATION FORM FOR VERTICAL TWO DOT LEADER
 [0xFE31] = 0x2014, -- PRESENTATION FORM FOR VERTICAL EM DASH
 [0xFE32] = 0x2013, -- PRESENTATION FORM FOR VERTICAL EN DASH
 [0xFE35] = {status="disallowed"}, -- PRESENTATION FORM FOR VERTICAL LEFT PARENTHESIS
 [0xFE36] = {status="disallowed"}, -- PRESENTATION FORM FOR VERTICAL RIGHT PARENTHESIS
 [0xFE37] = {status="disallowed"}, -- PRESENTATION FORM FOR VERTICAL LEFT CURLY BRACKET
 [0xFE38] = {status="disallowed"}, -- PRESENTATION FORM FOR VERTICAL RIGHT CURLY BRACKET
 [0xFE39] = 0x3014, -- PRESENTATION FORM FOR VERTICAL LEFT TORTOISE SHELL BRACKET
 [0xFE3A] = 0x3015, -- PRESENTATION FORM FOR VERTICAL RIGHT TORTOISE SHELL BRACKET
 [0xFE3B] = 0x3010, -- PRESENTATION FORM FOR VERTICAL LEFT BLACK LENTICULAR BRACKET
 [0xFE3C] = 0x3011, -- PRESENTATION FORM FOR VERTICAL RIGHT BLACK LENTICULAR BRACKET
 [0xFE3D] = 0x300A, -- PRESENTATION FORM FOR VERTICAL LEFT DOUBLE ANGLE BRACKET
 [0xFE3E] = 0x300B, -- PRESENTATION FORM FOR VERTICAL RIGHT DOUBLE ANGLE BRACKET
 [0xFE3F] = 0x3008, -- PRESENTATION FORM FOR VERTICAL LEFT ANGLE BRACKET
 [0xFE40] = 0x3009, -- PRESENTATION FORM FOR VERTICAL RIGHT ANGLE BRACKET
 [0xFE41] = 0x300C, -- PRESENTATION FORM FOR VERTICAL LEFT CORNER BRACKET
 [0xFE42] = 0x300D, -- PRESENTATION FORM FOR VERTICAL RIGHT CORNER BRACKET
 [0xFE43] = 0x300E, -- PRESENTATION FORM FOR VERTICAL LEFT WHITE CORNER BRACKET
 [0xFE44] = 0x300F, -- PRESENTATION FORM FOR VERTICAL RIGHT WHITE CORNER BRACKET
 -- PRESENTATION FORM FOR VERTICAL LOW LINE..PRESENTATION FORM FOR VERTICAL WAVY LOW LINE
 [0xFE33] = {status="disallowed"},
 [0xFE34] = {status="disallowed"},
 [0xFE47] = {status="disallowed"}, -- PRESENTATION FORM FOR VERTICAL LEFT SQUARE BRACKET
 [0xFE48] = {status="disallowed"}, -- PRESENTATION FORM FOR VERTICAL RIGHT SQUARE BRACKET
 -- SESAME DOT..WHITE SESAME DOT
 [0xFE45] = {status="disallowed"},
 [0xFE46] = {status="disallowed"},
 -- DASHED OVERLINE..DOUBLE WAVY OVERLINE
 [0xFE49] = {status="disallowed"},
 [0xFE4A] = {status="disallowed"},
 [0xFE4B] = {status="disallowed"},
 [0xFE4C] = {status="disallowed"},
 [0xFE50] = {status="disallowed"}, -- SMALL COMMA
 [0xFE51] = 0x3001, -- SMALL IDEOGRAPHIC COMMA
 [0xFE52] = {status="disallowed"}, -- SMALL FULL STOP
 [0xFE53] = {status="disallowed"}, -- NA   <reserved-FE53>
 [0xFE54] = {status="disallowed"}, -- SMALL SEMICOLON
 [0xFE55] = {status="disallowed"}, -- SMALL COLON
 [0xFE56] = {status="disallowed"}, -- SMALL QUESTION MARK
 [0xFE57] = {status="disallowed"}, -- SMALL EXCLAMATION MARK
 [0xFE58] = 0x2014, -- SMALL EM DASH
 [0xFE59] = {status="disallowed"}, -- SMALL LEFT PARENTHESIS
 [0xFE5A] = {status="disallowed"}, -- SMALL RIGHT PARENTHESIS
 [0xFE5B] = {status="disallowed"}, -- SMALL LEFT CURLY BRACKET
 [0xFE5C] = {status="disallowed"}, -- SMALL RIGHT CURLY BRACKET
 [0xFE5D] = 0x3014, -- SMALL LEFT TORTOISE SHELL BRACKET
 [0xFE5E] = 0x3015, -- SMALL RIGHT TORTOISE SHELL BRACKET
 [0xFE5F] = {status="disallowed"}, -- SMALL NUMBER SIGN
 [0xFE60] = {status="disallowed"}, -- SMALL AMPERSAND
 [0xFE61] = {status="disallowed"}, -- SMALL ASTERISK
 [0xFE62] = {status="disallowed"}, -- SMALL PLUS SIGN
 [0xFE63] = 0x002D, -- SMALL HYPHEN-MINUS
 [0xFE64] = {status="disallowed"}, -- SMALL LESS-THAN SIGN
 [0xFE65] = {status="disallowed"}, -- SMALL GREATER-THAN SIGN
 [0xFE66] = {status="disallowed"}, -- SMALL EQUALS SIGN
 [0xFE67] = {status="disallowed"}, -- NA   <reserved-FE67>
 [0xFE68] = {status="disallowed"}, -- SMALL REVERSE SOLIDUS
 [0xFE69] = {status="disallowed"}, -- SMALL DOLLAR SIGN
 [0xFE6A] = {status="disallowed"}, -- SMALL PERCENT SIGN
 [0xFE6B] = {status="disallowed"}, -- SMALL COMMERCIAL AT
 -- DASHED LOW LINE..WAVY LOW LINE
 [0xFE4D] = {status="disallowed"},
 [0xFE4E] = {status="disallowed"},
 [0xFE4F] = {status="disallowed"},
 [0xFE70] = {status="disallowed"}, -- ARABIC FATHATAN ISOLATED FORM
 [0xFE71] = {0x0640, 0x064B}, -- ARABIC TATWEEL WITH FATHATAN ABOVE
 [0xFE72] = {status="disallowed"}, -- ARABIC DAMMATAN ISOLATED FORM
 [0xFE73] = {status="valid"}, -- ARABIC TAIL FRAGMENT
 [0xFE74] = {status="disallowed"}, -- ARABIC KASRATAN ISOLATED FORM
 [0xFE75] = {status="disallowed"}, -- NA   <reserved-FE75>
 [0xFE76] = {status="disallowed"}, -- ARABIC FATHA ISOLATED FORM
 [0xFE77] = {0x0640, 0x064E}, -- ARABIC FATHA MEDIAL FORM
 [0xFE78] = {status="disallowed"}, -- ARABIC DAMMA ISOLATED FORM
 [0xFE79] = {0x0640, 0x064F}, -- ARABIC DAMMA MEDIAL FORM
 [0xFE7A] = {status="disallowed"}, -- ARABIC KASRA ISOLATED FORM
 [0xFE7B] = {0x0640, 0x0650}, -- ARABIC KASRA MEDIAL FORM
 [0xFE7C] = {status="disallowed"}, -- ARABIC SHADDA ISOLATED FORM
 [0xFE7D] = {0x0640, 0x0651}, -- ARABIC SHADDA MEDIAL FORM
 [0xFE7E] = {status="disallowed"}, -- ARABIC SUKUN ISOLATED FORM
 [0xFE7F] = {0x0640, 0x0652}, -- ARABIC SUKUN MEDIAL FORM
 [0xFE80] = 0x0621, -- ARABIC LETTER HAMZA ISOLATED FORM
 -- NA   <reserved-FE6C>..<reserved-FE6F>
 [0xFE6C] = {status="disallowed"},
 [0xFE6D] = {status="disallowed"},
 [0xFE6E] = {status="disallowed"},
 [0xFE6F] = {status="disallowed"},
 -- ARABIC LETTER ALEF WITH MADDA ABOVE ISOLATED FORM..ARABIC LETTER ALEF WITH MADDA ABOVE FINAL FORM
 [0xFE81] = 0x0622,
 [0xFE82] = 0x0622,
 -- ARABIC LETTER ALEF WITH HAMZA ABOVE ISOLATED FORM..ARABIC LETTER ALEF WITH HAMZA ABOVE FINAL FORM
 [0xFE83] = 0x0623,
 [0xFE84] = 0x0623,
 -- ARABIC LETTER WAW WITH HAMZA ABOVE ISOLATED FORM..ARABIC LETTER WAW WITH HAMZA ABOVE FINAL FORM
 [0xFE85] = 0x0624,
 [0xFE86] = 0x0624,
 -- ARABIC LETTER ALEF WITH HAMZA BELOW ISOLATED FORM..ARABIC LETTER ALEF WITH HAMZA BELOW FINAL FORM
 [0xFE87] = 0x0625,
 [0xFE88] = 0x0625,
 -- ARABIC LETTER YEH WITH HAMZA ABOVE ISOLATED FORM..ARABIC LETTER YEH WITH HAMZA ABOVE MEDIAL FORM
 [0xFE89] = 0x0626,
 [0xFE8A] = 0x0626,
 [0xFE8B] = 0x0626,
 [0xFE8C] = 0x0626,
 -- ARABIC LETTER ALEF ISOLATED FORM..ARABIC LETTER ALEF FINAL FORM
 [0xFE8D] = 0x0627,
 [0xFE8E] = 0x0627,
 -- ARABIC LETTER BEH ISOLATED FORM..ARABIC LETTER BEH MEDIAL FORM
 [0xFE8F] = 0x0628,
 [0xFE90] = 0x0628,
 [0xFE91] = 0x0628,
 [0xFE92] = 0x0628,
 -- ARABIC LETTER TEH MARBUTA ISOLATED FORM..ARABIC LETTER TEH MARBUTA FINAL FORM
 [0xFE93] = 0x0629,
 [0xFE94] = 0x0629,
 -- ARABIC LETTER TEH ISOLATED FORM..ARABIC LETTER TEH MEDIAL FORM
 [0xFE95] = 0x062A,
 [0xFE96] = 0x062A,
 [0xFE97] = 0x062A,
 [0xFE98] = 0x062A,
 -- ARABIC LETTER THEH ISOLATED FORM..ARABIC LETTER THEH MEDIAL FORM
 [0xFE99] = 0x062B,
 [0xFE9A] = 0x062B,
 [0xFE9B] = 0x062B,
 [0xFE9C] = 0x062B,
 -- ARABIC LETTER JEEM ISOLATED FORM..ARABIC LETTER JEEM MEDIAL FORM
 [0xFE9D] = 0x062C,
 [0xFE9E] = 0x062C,
 [0xFE9F] = 0x062C,
 [0xFEA0] = 0x062C,
 -- ARABIC LETTER HAH ISOLATED FORM..ARABIC LETTER HAH MEDIAL FORM
 [0xFEA1] = 0x062D,
 [0xFEA2] = 0x062D,
 [0xFEA3] = 0x062D,
 [0xFEA4] = 0x062D,
 -- ARABIC LETTER KHAH ISOLATED FORM..ARABIC LETTER KHAH MEDIAL FORM
 [0xFEA5] = 0x062E,
 [0xFEA6] = 0x062E,
 [0xFEA7] = 0x062E,
 [0xFEA8] = 0x062E,
 -- ARABIC LETTER DAL ISOLATED FORM..ARABIC LETTER DAL FINAL FORM
 [0xFEA9] = 0x062F,
 [0xFEAA] = 0x062F,
 -- ARABIC LETTER THAL ISOLATED FORM..ARABIC LETTER THAL FINAL FORM
 [0xFEAB] = 0x0630,
 [0xFEAC] = 0x0630,
 -- ARABIC LETTER REH ISOLATED FORM..ARABIC LETTER REH FINAL FORM
 [0xFEAD] = 0x0631,
 [0xFEAE] = 0x0631,
 -- ARABIC LETTER ZAIN ISOLATED FORM..ARABIC LETTER ZAIN FINAL FORM
 [0xFEAF] = 0x0632,
 [0xFEB0] = 0x0632,
 -- ARABIC LETTER SEEN ISOLATED FORM..ARABIC LETTER SEEN MEDIAL FORM
 [0xFEB1] = 0x0633,
 [0xFEB2] = 0x0633,
 [0xFEB3] = 0x0633,
 [0xFEB4] = 0x0633,
 -- ARABIC LETTER SHEEN ISOLATED FORM..ARABIC LETTER SHEEN MEDIAL FORM
 [0xFEB5] = 0x0634,
 [0xFEB6] = 0x0634,
 [0xFEB7] = 0x0634,
 [0xFEB8] = 0x0634,
 -- ARABIC LETTER SAD ISOLATED FORM..ARABIC LETTER SAD MEDIAL FORM
 [0xFEB9] = 0x0635,
 [0xFEBA] = 0x0635,
 [0xFEBB] = 0x0635,
 [0xFEBC] = 0x0635,
 -- ARABIC LETTER DAD ISOLATED FORM..ARABIC LETTER DAD MEDIAL FORM
 [0xFEBD] = 0x0636,
 [0xFEBE] = 0x0636,
 [0xFEBF] = 0x0636,
 [0xFEC0] = 0x0636,
 -- ARABIC LETTER TAH ISOLATED FORM..ARABIC LETTER TAH MEDIAL FORM
 [0xFEC1] = 0x0637,
 [0xFEC2] = 0x0637,
 [0xFEC3] = 0x0637,
 [0xFEC4] = 0x0637,
 -- ARABIC LETTER ZAH ISOLATED FORM..ARABIC LETTER ZAH MEDIAL FORM
 [0xFEC5] = 0x0638,
 [0xFEC6] = 0x0638,
 [0xFEC7] = 0x0638,
 [0xFEC8] = 0x0638,
 -- ARABIC LETTER AIN ISOLATED FORM..ARABIC LETTER AIN MEDIAL FORM
 [0xFEC9] = 0x0639,
 [0xFECA] = 0x0639,
 [0xFECB] = 0x0639,
 [0xFECC] = 0x0639,
 -- ARABIC LETTER GHAIN ISOLATED FORM..ARABIC LETTER GHAIN MEDIAL FORM
 [0xFECD] = 0x063A,
 [0xFECE] = 0x063A,
 [0xFECF] = 0x063A,
 [0xFED0] = 0x063A,
 -- ARABIC LETTER FEH ISOLATED FORM..ARABIC LETTER FEH MEDIAL FORM
 [0xFED1] = 0x0641,
 [0xFED2] = 0x0641,
 [0xFED3] = 0x0641,
 [0xFED4] = 0x0641,
 -- ARABIC LETTER QAF ISOLATED FORM..ARABIC LETTER QAF MEDIAL FORM
 [0xFED5] = 0x0642,
 [0xFED6] = 0x0642,
 [0xFED7] = 0x0642,
 [0xFED8] = 0x0642,
 -- ARABIC LETTER KAF ISOLATED FORM..ARABIC LETTER KAF MEDIAL FORM
 [0xFED9] = 0x0643,
 [0xFEDA] = 0x0643,
 [0xFEDB] = 0x0643,
 [0xFEDC] = 0x0643,
 -- ARABIC LETTER LAM ISOLATED FORM..ARABIC LETTER LAM MEDIAL FORM
 [0xFEDD] = 0x0644,
 [0xFEDE] = 0x0644,
 [0xFEDF] = 0x0644,
 [0xFEE0] = 0x0644,
 -- ARABIC LETTER MEEM ISOLATED FORM..ARABIC LETTER MEEM MEDIAL FORM
 [0xFEE1] = 0x0645,
 [0xFEE2] = 0x0645,
 [0xFEE3] = 0x0645,
 [0xFEE4] = 0x0645,
 -- ARABIC LETTER NOON ISOLATED FORM..ARABIC LETTER NOON MEDIAL FORM
 [0xFEE5] = 0x0646,
 [0xFEE6] = 0x0646,
 [0xFEE7] = 0x0646,
 [0xFEE8] = 0x0646,
 -- ARABIC LETTER HEH ISOLATED FORM..ARABIC LETTER HEH MEDIAL FORM
 [0xFEE9] = 0x0647,
 [0xFEEA] = 0x0647,
 [0xFEEB] = 0x0647,
 [0xFEEC] = 0x0647,
 -- ARABIC LETTER WAW ISOLATED FORM..ARABIC LETTER WAW FINAL FORM
 [0xFEED] = 0x0648,
 [0xFEEE] = 0x0648,
 -- ARABIC LETTER ALEF MAKSURA ISOLATED FORM..ARABIC LETTER ALEF MAKSURA FINAL FORM
 [0xFEEF] = 0x0649,
 [0xFEF0] = 0x0649,
 -- ARABIC LETTER YEH ISOLATED FORM..ARABIC LETTER YEH MEDIAL FORM
 [0xFEF1] = 0x064A,
 [0xFEF2] = 0x064A,
 [0xFEF3] = 0x064A,
 [0xFEF4] = 0x064A,
 -- ARABIC LIGATURE LAM WITH ALEF WITH MADDA ABOVE ISOLATED FORM..ARABIC LIGATURE LAM WITH ALEF WITH MADDA ABOVE FINAL FORM
 [0xFEF5] = {0x0644, 0x0622},
 [0xFEF6] = {0x0644, 0x0622},
 -- ARABIC LIGATURE LAM WITH ALEF WITH HAMZA ABOVE ISOLATED FORM..ARABIC LIGATURE LAM WITH ALEF WITH HAMZA ABOVE FINAL FORM
 [0xFEF7] = {0x0644, 0x0623},
 [0xFEF8] = {0x0644, 0x0623},
 -- ARABIC LIGATURE LAM WITH ALEF WITH HAMZA BELOW ISOLATED FORM..ARABIC LIGATURE LAM WITH ALEF WITH HAMZA BELOW FINAL FORM
 [0xFEF9] = {0x0644, 0x0625},
 [0xFEFA] = {0x0644, 0x0625},
 -- ARABIC LIGATURE LAM WITH ALEF ISOLATED FORM..ARABIC LIGATURE LAM WITH ALEF FINAL FORM
 [0xFEFB] = {0x0644, 0x0627},
 [0xFEFC] = {0x0644, 0x0627},
 [0xFEFF] = {status='ignored'}, -- ZERO WIDTH NO-BREAK SPACE
 [0xFF00] = {status="disallowed"}, -- NA   <reserved-FF00>
 [0xFF01] = {status="disallowed"}, -- FULLWIDTH EXCLAMATION MARK
 [0xFF02] = {status="disallowed"}, -- FULLWIDTH QUOTATION MARK
 [0xFF03] = {status="disallowed"}, -- FULLWIDTH NUMBER SIGN
 [0xFF04] = {status="disallowed"}, -- FULLWIDTH DOLLAR SIGN
 [0xFF05] = {status="disallowed"}, -- FULLWIDTH PERCENT SIGN
 [0xFF06] = {status="disallowed"}, -- FULLWIDTH AMPERSAND
 [0xFF07] = {status="disallowed"}, -- FULLWIDTH APOSTROPHE
 [0xFF08] = {status="disallowed"}, -- FULLWIDTH LEFT PARENTHESIS
 [0xFF09] = {status="disallowed"}, -- FULLWIDTH RIGHT PARENTHESIS
 [0xFF0A] = {status="disallowed"}, -- FULLWIDTH ASTERISK
 [0xFF0B] = {status="disallowed"}, -- FULLWIDTH PLUS SIGN
 [0xFF0C] = {status="disallowed"}, -- FULLWIDTH COMMA
 [0xFF0D] = 0x002D, -- FULLWIDTH HYPHEN-MINUS
 [0xFF0E] = 0x002E, -- FULLWIDTH FULL STOP
 [0xFF0F] = {status="disallowed"}, -- FULLWIDTH SOLIDUS
 [0xFF10] = 0x0030, -- FULLWIDTH DIGIT ZERO
 [0xFF11] = 0x0031, -- FULLWIDTH DIGIT ONE
 [0xFF12] = 0x0032, -- FULLWIDTH DIGIT TWO
 [0xFF13] = 0x0033, -- FULLWIDTH DIGIT THREE
 [0xFF14] = 0x0034, -- FULLWIDTH DIGIT FOUR
 [0xFF15] = 0x0035, -- FULLWIDTH DIGIT FIVE
 [0xFF16] = 0x0036, -- FULLWIDTH DIGIT SIX
 [0xFF17] = 0x0037, -- FULLWIDTH DIGIT SEVEN
 [0xFF18] = 0x0038, -- FULLWIDTH DIGIT EIGHT
 [0xFF19] = 0x0039, -- FULLWIDTH DIGIT NINE
 [0xFF1A] = {status="disallowed"}, -- FULLWIDTH COLON
 [0xFF1B] = {status="disallowed"}, -- FULLWIDTH SEMICOLON
 [0xFF1C] = {status="disallowed"}, -- FULLWIDTH LESS-THAN SIGN
 [0xFF1D] = {status="disallowed"}, -- FULLWIDTH EQUALS SIGN
 [0xFF1E] = {status="disallowed"}, -- FULLWIDTH GREATER-THAN SIGN
 [0xFF1F] = {status="disallowed"}, -- FULLWIDTH QUESTION MARK
 [0xFF20] = {status="disallowed"}, -- FULLWIDTH COMMERCIAL AT
 [0xFF21] = 0x0061, -- FULLWIDTH LATIN CAPITAL LETTER A
 [0xFF22] = 0x0062, -- FULLWIDTH LATIN CAPITAL LETTER B
 [0xFF23] = 0x0063, -- FULLWIDTH LATIN CAPITAL LETTER C
 [0xFF24] = 0x0064, -- FULLWIDTH LATIN CAPITAL LETTER D
 [0xFF25] = 0x0065, -- FULLWIDTH LATIN CAPITAL LETTER E
 [0xFF26] = 0x0066, -- FULLWIDTH LATIN CAPITAL LETTER F
 [0xFF27] = 0x0067, -- FULLWIDTH LATIN CAPITAL LETTER G
 [0xFF28] = 0x0068, -- FULLWIDTH LATIN CAPITAL LETTER H
 [0xFF29] = 0x0069, -- FULLWIDTH LATIN CAPITAL LETTER I
 [0xFF2A] = 0x006A, -- FULLWIDTH LATIN CAPITAL LETTER J
 [0xFF2B] = 0x006B, -- FULLWIDTH LATIN CAPITAL LETTER K
 [0xFF2C] = 0x006C, -- FULLWIDTH LATIN CAPITAL LETTER L
 [0xFF2D] = 0x006D, -- FULLWIDTH LATIN CAPITAL LETTER M
 [0xFF2E] = 0x006E, -- FULLWIDTH LATIN CAPITAL LETTER N
 [0xFF2F] = 0x006F, -- FULLWIDTH LATIN CAPITAL LETTER O
 [0xFF30] = 0x0070, -- FULLWIDTH LATIN CAPITAL LETTER P
 [0xFF31] = 0x0071, -- FULLWIDTH LATIN CAPITAL LETTER Q
 [0xFF32] = 0x0072, -- FULLWIDTH LATIN CAPITAL LETTER R
 [0xFF33] = 0x0073, -- FULLWIDTH LATIN CAPITAL LETTER S
 [0xFF34] = 0x0074, -- FULLWIDTH LATIN CAPITAL LETTER T
 [0xFF35] = 0x0075, -- FULLWIDTH LATIN CAPITAL LETTER U
 [0xFF36] = 0x0076, -- FULLWIDTH LATIN CAPITAL LETTER V
 [0xFF37] = 0x0077, -- FULLWIDTH LATIN CAPITAL LETTER W
 [0xFF38] = 0x0078, -- FULLWIDTH LATIN CAPITAL LETTER X
 [0xFF39] = 0x0079, -- FULLWIDTH LATIN CAPITAL LETTER Y
 [0xFF3A] = 0x007A, -- FULLWIDTH LATIN CAPITAL LETTER Z
 [0xFF3B] = {status="disallowed"}, -- FULLWIDTH LEFT SQUARE BRACKET
 [0xFF3C] = {status="disallowed"}, -- FULLWIDTH REVERSE SOLIDUS
 [0xFF3D] = {status="disallowed"}, -- FULLWIDTH RIGHT SQUARE BRACKET
 [0xFF3E] = {status="disallowed"}, -- FULLWIDTH CIRCUMFLEX ACCENT
 [0xFF3F] = {status="disallowed"}, -- FULLWIDTH LOW LINE
 [0xFF40] = {status="disallowed"}, -- FULLWIDTH GRAVE ACCENT
 [0xFF41] = 0x0061, -- FULLWIDTH LATIN SMALL LETTER A
 [0xFF42] = 0x0062, -- FULLWIDTH LATIN SMALL LETTER B
 [0xFF43] = 0x0063, -- FULLWIDTH LATIN SMALL LETTER C
 [0xFF44] = 0x0064, -- FULLWIDTH LATIN SMALL LETTER D
 [0xFF45] = 0x0065, -- FULLWIDTH LATIN SMALL LETTER E
 [0xFF46] = 0x0066, -- FULLWIDTH LATIN SMALL LETTER F
 [0xFF47] = 0x0067, -- FULLWIDTH LATIN SMALL LETTER G
 [0xFF48] = 0x0068, -- FULLWIDTH LATIN SMALL LETTER H
 [0xFF49] = 0x0069, -- FULLWIDTH LATIN SMALL LETTER I
 [0xFF4A] = 0x006A, -- FULLWIDTH LATIN SMALL LETTER J
 [0xFF4B] = 0x006B, -- FULLWIDTH LATIN SMALL LETTER K
 [0xFF4C] = 0x006C, -- FULLWIDTH LATIN SMALL LETTER L
 [0xFF4D] = 0x006D, -- FULLWIDTH LATIN SMALL LETTER M
 [0xFF4E] = 0x006E, -- FULLWIDTH LATIN SMALL LETTER N
 [0xFF4F] = 0x006F, -- FULLWIDTH LATIN SMALL LETTER O
 [0xFF50] = 0x0070, -- FULLWIDTH LATIN SMALL LETTER P
 [0xFF51] = 0x0071, -- FULLWIDTH LATIN SMALL LETTER Q
 [0xFF52] = 0x0072, -- FULLWIDTH LATIN SMALL LETTER R
 [0xFF53] = 0x0073, -- FULLWIDTH LATIN SMALL LETTER S
 [0xFF54] = 0x0074, -- FULLWIDTH LATIN SMALL LETTER T
 [0xFF55] = 0x0075, -- FULLWIDTH LATIN SMALL LETTER U
 [0xFF56] = 0x0076, -- FULLWIDTH LATIN SMALL LETTER V
 [0xFF57] = 0x0077, -- FULLWIDTH LATIN SMALL LETTER W
 [0xFF58] = 0x0078, -- FULLWIDTH LATIN SMALL LETTER X
 [0xFF59] = 0x0079, -- FULLWIDTH LATIN SMALL LETTER Y
 [0xFF5A] = 0x007A, -- FULLWIDTH LATIN SMALL LETTER Z
 [0xFF5B] = {status="disallowed"}, -- FULLWIDTH LEFT CURLY BRACKET
 [0xFF5C] = {status="disallowed"}, -- FULLWIDTH VERTICAL LINE
 [0xFF5D] = {status="disallowed"}, -- FULLWIDTH RIGHT CURLY BRACKET
 [0xFF5E] = {status="disallowed"}, -- FULLWIDTH TILDE
 [0xFF5F] = 0x2985, -- FULLWIDTH LEFT WHITE PARENTHESIS
 [0xFF60] = 0x2986, -- FULLWIDTH RIGHT WHITE PARENTHESIS
 [0xFF61] = 0x002E, -- HALFWIDTH IDEOGRAPHIC FULL STOP
 [0xFF62] = 0x300C, -- HALFWIDTH LEFT CORNER BRACKET
 [0xFF63] = 0x300D, -- HALFWIDTH RIGHT CORNER BRACKET
 [0xFF64] = 0x3001, -- HALFWIDTH IDEOGRAPHIC COMMA
 [0xFF65] = 0x30FB, -- HALFWIDTH KATAKANA MIDDLE DOT
 [0xFF66] = 0x30F2, -- HALFWIDTH KATAKANA LETTER WO
 [0xFF67] = 0x30A1, -- HALFWIDTH KATAKANA LETTER SMALL A
 [0xFF68] = 0x30A3, -- HALFWIDTH KATAKANA LETTER SMALL I
 [0xFF69] = 0x30A5, -- HALFWIDTH KATAKANA LETTER SMALL U
 [0xFF6A] = 0x30A7, -- HALFWIDTH KATAKANA LETTER SMALL E
 [0xFF6B] = 0x30A9, -- HALFWIDTH KATAKANA LETTER SMALL O
 [0xFF6C] = 0x30E3, -- HALFWIDTH KATAKANA LETTER SMALL YA
 [0xFF6D] = 0x30E5, -- HALFWIDTH KATAKANA LETTER SMALL YU
 [0xFF6E] = 0x30E7, -- HALFWIDTH KATAKANA LETTER SMALL YO
 [0xFF6F] = 0x30C3, -- HALFWIDTH KATAKANA LETTER SMALL TU
 [0xFF70] = 0x30FC, -- HALFWIDTH KATAKANA-HIRAGANA PROLONGED SOUND MARK
 [0xFF71] = 0x30A2, -- HALFWIDTH KATAKANA LETTER A
 [0xFF72] = 0x30A4, -- HALFWIDTH KATAKANA LETTER I
 [0xFF73] = 0x30A6, -- HALFWIDTH KATAKANA LETTER U
 [0xFF74] = 0x30A8, -- HALFWIDTH KATAKANA LETTER E
 [0xFF75] = 0x30AA, -- HALFWIDTH KATAKANA LETTER O
 [0xFF76] = 0x30AB, -- HALFWIDTH KATAKANA LETTER KA
 [0xFF77] = 0x30AD, -- HALFWIDTH KATAKANA LETTER KI
 [0xFF78] = 0x30AF, -- HALFWIDTH KATAKANA LETTER KU
 [0xFF79] = 0x30B1, -- HALFWIDTH KATAKANA LETTER KE
 [0xFF7A] = 0x30B3, -- HALFWIDTH KATAKANA LETTER KO
 [0xFF7B] = 0x30B5, -- HALFWIDTH KATAKANA LETTER SA
 [0xFF7C] = 0x30B7, -- HALFWIDTH KATAKANA LETTER SI
 [0xFF7D] = 0x30B9, -- HALFWIDTH KATAKANA LETTER SU
 [0xFF7E] = 0x30BB, -- HALFWIDTH KATAKANA LETTER SE
 [0xFF7F] = 0x30BD, -- HALFWIDTH KATAKANA LETTER SO
 [0xFF80] = 0x30BF, -- HALFWIDTH KATAKANA LETTER TA
 [0xFF81] = 0x30C1, -- HALFWIDTH KATAKANA LETTER TI
 [0xFF82] = 0x30C4, -- HALFWIDTH KATAKANA LETTER TU
 [0xFF83] = 0x30C6, -- HALFWIDTH KATAKANA LETTER TE
 [0xFF84] = 0x30C8, -- HALFWIDTH KATAKANA LETTER TO
 [0xFF85] = 0x30CA, -- HALFWIDTH KATAKANA LETTER NA
 [0xFF86] = 0x30CB, -- HALFWIDTH KATAKANA LETTER NI
 [0xFF87] = 0x30CC, -- HALFWIDTH KATAKANA LETTER NU
 [0xFF88] = 0x30CD, -- HALFWIDTH KATAKANA LETTER NE
 [0xFF89] = 0x30CE, -- HALFWIDTH KATAKANA LETTER NO
 [0xFF8A] = 0x30CF, -- HALFWIDTH KATAKANA LETTER HA
 [0xFF8B] = 0x30D2, -- HALFWIDTH KATAKANA LETTER HI
 [0xFF8C] = 0x30D5, -- HALFWIDTH KATAKANA LETTER HU
 [0xFF8D] = 0x30D8, -- HALFWIDTH KATAKANA LETTER HE
 [0xFF8E] = 0x30DB, -- HALFWIDTH KATAKANA LETTER HO
 [0xFF8F] = 0x30DE, -- HALFWIDTH KATAKANA LETTER MA
 [0xFF90] = 0x30DF, -- HALFWIDTH KATAKANA LETTER MI
 [0xFF91] = 0x30E0, -- HALFWIDTH KATAKANA LETTER MU
 [0xFF92] = 0x30E1, -- HALFWIDTH KATAKANA LETTER ME
 [0xFF93] = 0x30E2, -- HALFWIDTH KATAKANA LETTER MO
 [0xFF94] = 0x30E4, -- HALFWIDTH KATAKANA LETTER YA
 [0xFF95] = 0x30E6, -- HALFWIDTH KATAKANA LETTER YU
 [0xFF96] = 0x30E8, -- HALFWIDTH KATAKANA LETTER YO
 [0xFF97] = 0x30E9, -- HALFWIDTH KATAKANA LETTER RA
 [0xFF98] = 0x30EA, -- HALFWIDTH KATAKANA LETTER RI
 [0xFF99] = 0x30EB, -- HALFWIDTH KATAKANA LETTER RU
 [0xFF9A] = 0x30EC, -- HALFWIDTH KATAKANA LETTER RE
 [0xFF9B] = 0x30ED, -- HALFWIDTH KATAKANA LETTER RO
 [0xFF9C] = 0x30EF, -- HALFWIDTH KATAKANA LETTER WA
 [0xFF9D] = 0x30F3, -- HALFWIDTH KATAKANA LETTER N
 [0xFF9E] = 0x3099, -- HALFWIDTH KATAKANA VOICED SOUND MARK
 [0xFF9F] = 0x309A, -- HALFWIDTH KATAKANA SEMI-VOICED SOUND MARK
 [0xFFA0] = {status="disallowed"}, -- HALFWIDTH HANGUL FILLER
 [0xFFA1] = 0x1100, -- HALFWIDTH HANGUL LETTER KIYEOK
 [0xFFA2] = 0x1101, -- HALFWIDTH HANGUL LETTER SSANGKIYEOK
 [0xFFA3] = 0x11AA, -- HALFWIDTH HANGUL LETTER KIYEOK-SIOS
 [0xFFA4] = 0x1102, -- HALFWIDTH HANGUL LETTER NIEUN
 [0xFFA5] = 0x11AC, -- HALFWIDTH HANGUL LETTER NIEUN-CIEUC
 [0xFFA6] = 0x11AD, -- HALFWIDTH HANGUL LETTER NIEUN-HIEUH
 [0xFFA7] = 0x1103, -- HALFWIDTH HANGUL LETTER TIKEUT
 [0xFFA8] = 0x1104, -- HALFWIDTH HANGUL LETTER SSANGTIKEUT
 [0xFFA9] = 0x1105, -- HALFWIDTH HANGUL LETTER RIEUL
 [0xFFAA] = 0x11B0, -- HALFWIDTH HANGUL LETTER RIEUL-KIYEOK
 [0xFFAB] = 0x11B1, -- HALFWIDTH HANGUL LETTER RIEUL-MIEUM
 [0xFFAC] = 0x11B2, -- HALFWIDTH HANGUL LETTER RIEUL-PIEUP
 [0xFFAD] = 0x11B3, -- HALFWIDTH HANGUL LETTER RIEUL-SIOS
 [0xFFAE] = 0x11B4, -- HALFWIDTH HANGUL LETTER RIEUL-THIEUTH
 [0xFFAF] = 0x11B5, -- HALFWIDTH HANGUL LETTER RIEUL-PHIEUPH
 [0xFFB0] = 0x111A, -- HALFWIDTH HANGUL LETTER RIEUL-HIEUH
 [0xFFB1] = 0x1106, -- HALFWIDTH HANGUL LETTER MIEUM
 [0xFFB2] = 0x1107, -- HALFWIDTH HANGUL LETTER PIEUP
 [0xFFB3] = 0x1108, -- HALFWIDTH HANGUL LETTER SSANGPIEUP
 [0xFFB4] = 0x1121, -- HALFWIDTH HANGUL LETTER PIEUP-SIOS
 [0xFFB5] = 0x1109, -- HALFWIDTH HANGUL LETTER SIOS
 [0xFFB6] = 0x110A, -- HALFWIDTH HANGUL LETTER SSANGSIOS
 [0xFFB7] = 0x110B, -- HALFWIDTH HANGUL LETTER IEUNG
 [0xFFB8] = 0x110C, -- HALFWIDTH HANGUL LETTER CIEUC
 [0xFFB9] = 0x110D, -- HALFWIDTH HANGUL LETTER SSANGCIEUC
 [0xFFBA] = 0x110E, -- HALFWIDTH HANGUL LETTER CHIEUCH
 [0xFFBB] = 0x110F, -- HALFWIDTH HANGUL LETTER KHIEUKH
 [0xFFBC] = 0x1110, -- HALFWIDTH HANGUL LETTER THIEUTH
 [0xFFBD] = 0x1111, -- HALFWIDTH HANGUL LETTER PHIEUPH
 [0xFFBE] = 0x1112, -- HALFWIDTH HANGUL LETTER HIEUH
 -- NA   <reserved-FEFD>..<reserved-FEFE>
 [0xFEFD] = {status="disallowed"},
 [0xFEFE] = {status="disallowed"},
 [0xFFC2] = 0x1161, -- HALFWIDTH HANGUL LETTER A
 [0xFFC3] = 0x1162, -- HALFWIDTH HANGUL LETTER AE
 [0xFFC4] = 0x1163, -- HALFWIDTH HANGUL LETTER YA
 [0xFFC5] = 0x1164, -- HALFWIDTH HANGUL LETTER YAE
 [0xFFC6] = 0x1165, -- HALFWIDTH HANGUL LETTER EO
 [0xFFC7] = 0x1166, -- HALFWIDTH HANGUL LETTER E
 -- NA   <reserved-FFBF>..<reserved-FFC1>
 [0xFFBF] = {status="disallowed"},
 [0xFFC0] = {status="disallowed"},
 [0xFFC1] = {status="disallowed"},
 [0xFFCA] = 0x1167, -- HALFWIDTH HANGUL LETTER YEO
 [0xFFCB] = 0x1168, -- HALFWIDTH HANGUL LETTER YE
 [0xFFCC] = 0x1169, -- HALFWIDTH HANGUL LETTER O
 [0xFFCD] = 0x116A, -- HALFWIDTH HANGUL LETTER WA
 [0xFFCE] = 0x116B, -- HALFWIDTH HANGUL LETTER WAE
 [0xFFCF] = 0x116C, -- HALFWIDTH HANGUL LETTER OE
 -- NA   <reserved-FFC8>..<reserved-FFC9>
 [0xFFC8] = {status="disallowed"},
 [0xFFC9] = {status="disallowed"},
 [0xFFD2] = 0x116D, -- HALFWIDTH HANGUL LETTER YO
 [0xFFD3] = 0x116E, -- HALFWIDTH HANGUL LETTER U
 [0xFFD4] = 0x116F, -- HALFWIDTH HANGUL LETTER WEO
 [0xFFD5] = 0x1170, -- HALFWIDTH HANGUL LETTER WE
 [0xFFD6] = 0x1171, -- HALFWIDTH HANGUL LETTER WI
 [0xFFD7] = 0x1172, -- HALFWIDTH HANGUL LETTER YU
 -- NA   <reserved-FFD0>..<reserved-FFD1>
 [0xFFD0] = {status="disallowed"},
 [0xFFD1] = {status="disallowed"},
 [0xFFDA] = 0x1173, -- HALFWIDTH HANGUL LETTER EU
 [0xFFDB] = 0x1174, -- HALFWIDTH HANGUL LETTER YI
 [0xFFDC] = 0x1175, -- HALFWIDTH HANGUL LETTER I
 -- NA   <reserved-FFD8>..<reserved-FFD9>
 [0xFFD8] = {status="disallowed"},
 [0xFFD9] = {status="disallowed"},
 [0xFFE0] = 0x00A2, -- FULLWIDTH CENT SIGN
 [0xFFE1] = 0x00A3, -- FULLWIDTH POUND SIGN
 [0xFFE2] = 0x00AC, -- FULLWIDTH NOT SIGN
 [0xFFE3] = {status="disallowed"}, -- FULLWIDTH MACRON
 [0xFFE4] = 0x00A6, -- FULLWIDTH BROKEN BAR
 [0xFFE5] = 0x00A5, -- FULLWIDTH YEN SIGN
 [0xFFE6] = 0x20A9, -- FULLWIDTH WON SIGN
 [0xFFE7] = {status="disallowed"}, -- NA   <reserved-FFE7>
 [0xFFE8] = 0x2502, -- HALFWIDTH FORMS LIGHT VERTICAL
 [0xFFE9] = 0x2190, -- HALFWIDTH LEFTWARDS ARROW
 [0xFFEA] = 0x2191, -- HALFWIDTH UPWARDS ARROW
 [0xFFEB] = 0x2192, -- HALFWIDTH RIGHTWARDS ARROW
 [0xFFEC] = 0x2193, -- HALFWIDTH DOWNWARDS ARROW
 [0xFFED] = 0x25A0, -- HALFWIDTH BLACK SQUARE
 [0xFFEE] = 0x25CB, -- HALFWIDTH WHITE CIRCLE
 -- NA   <reserved-FFDD>..<reserved-FFDF>
 [0xFFDD] = {status="disallowed"},
 [0xFFDE] = {status="disallowed"},
 [0xFFDF] = {status="disallowed"},
 [0xFFFC] = {status="disallowed"}, -- OBJECT REPLACEMENT CHARACTER
 [0xFFFD] = {status="disallowed"}, -- REPLACEMENT CHARACTER
 -- <noncharacter-FFFE>..<noncharacter-FFFF>
 [0xFFFE] = {status="disallowed"},
 [0xFFFF] = {status="disallowed"},
 [0x1000C] = {status="disallowed"}, -- NA   <reserved-1000C>
 [0x10027] = {status="disallowed"}, -- NA   <reserved-10027>
 [0x1003B] = {status="disallowed"}, -- NA   <reserved-1003B>
 [0x1003E] = {status="disallowed"}, -- NA   <reserved-1003E>
 -- LINEAR B SYLLABLE B017 ZA..LINEAR B SYLLABLE B074 ZE
 [0x1003C] = {status="valid"},
 [0x1003D] = {status="valid"},
 -- NA   <reserved-1004E>..<reserved-1004F>
 [0x1004E] = {status="disallowed"},
 [0x1004F] = {status="disallowed"},
 -- NA   <reserved-100FB>..<reserved-100FF>
 [0x100FB] = {status="disallowed"},
 [0x100FC] = {status="disallowed"},
 [0x100FD] = {status="disallowed"},
 [0x100FE] = {status="disallowed"},
 [0x100FF] = {status="disallowed"},
 -- AEGEAN WORD SEPARATOR LINE..AEGEAN CHECK MARK
 [0x10100] = {status="disallowed"},
 [0x10101] = {status="disallowed"},
 [0x10102] = {status="disallowed"},
 -- NA   <reserved-10103>..<reserved-10106>
 [0x10103] = {status="disallowed"},
 [0x10104] = {status="disallowed"},
 [0x10105] = {status="disallowed"},
 [0x10106] = {status="disallowed"},
 -- NA   <reserved-10134>..<reserved-10136>
 [0x10134] = {status="disallowed"},
 [0x10135] = {status="disallowed"},
 [0x10136] = {status="disallowed"},
 [0x1018F] = {status="disallowed"}, -- NA   <reserved-1018F>
 [0x101A0] = {status="disallowed"}, -- GREEK SYMBOL TAU RHO
 -- NA   <reserved-1019C>..<reserved-1019F>
 [0x1019C] = {status="disallowed"},
 [0x1019D] = {status="disallowed"},
 [0x1019E] = {status="disallowed"},
 [0x1019F] = {status="disallowed"},
 [0x101FD] = {status="valid"}, -- PHAISTOS DISC SIGN COMBINING OBLIQUE STROKE
 -- NA   <reserved-1029D>..<reserved-1029F>
 [0x1029D] = {status="disallowed"},
 [0x1029E] = {status="disallowed"},
 [0x1029F] = {status="disallowed"},
 [0x102E0] = {status="valid"}, -- COPTIC EPACT THOUSANDS MARK
 -- NA   <reserved-102FC>..<reserved-102FF>
 [0x102FC] = {status="disallowed"},
 [0x102FD] = {status="disallowed"},
 [0x102FE] = {status="disallowed"},
 [0x102FF] = {status="disallowed"},
 [0x1031F] = {status="valid"}, -- OLD ITALIC LETTER ESS
 -- OLD ITALIC NUMERAL ONE..OLD ITALIC NUMERAL FIFTY
 [0x10320] = {status="disallowed"},
 [0x10321] = {status="disallowed"},
 [0x10322] = {status="disallowed"},
 [0x10323] = {status="disallowed"},
 [0x10341] = {status="disallowed"}, -- GOTHIC LETTER NINETY
 [0x1034A] = {status="disallowed"}, -- GOTHIC LETTER NINE HUNDRED
 -- NA   <reserved-1034B>..<reserved-1034F>
 [0x1034B] = {status="disallowed"},
 [0x1034C] = {status="disallowed"},
 [0x1034D] = {status="disallowed"},
 [0x1034E] = {status="disallowed"},
 [0x1034F] = {status="disallowed"},
 -- NA   <reserved-1037B>..<reserved-1037F>
 [0x1037B] = {status="disallowed"},
 [0x1037C] = {status="disallowed"},
 [0x1037D] = {status="disallowed"},
 [0x1037E] = {status="disallowed"},
 [0x1037F] = {status="disallowed"},
 [0x1039E] = {status="disallowed"}, -- NA   <reserved-1039E>
 [0x1039F] = {status="disallowed"}, -- UGARITIC WORD DIVIDER
 -- NA   <reserved-103C4>..<reserved-103C7>
 [0x103C4] = {status="disallowed"},
 [0x103C5] = {status="disallowed"},
 [0x103C6] = {status="disallowed"},
 [0x103C7] = {status="disallowed"},
 [0x10400] = 0x10428, -- DESERET CAPITAL LETTER LONG I
 [0x10401] = 0x10429, -- DESERET CAPITAL LETTER LONG E
 [0x10402] = 0x1042A, -- DESERET CAPITAL LETTER LONG A
 [0x10403] = 0x1042B, -- DESERET CAPITAL LETTER LONG AH
 [0x10404] = 0x1042C, -- DESERET CAPITAL LETTER LONG O
 [0x10405] = 0x1042D, -- DESERET CAPITAL LETTER LONG OO
 [0x10406] = 0x1042E, -- DESERET CAPITAL LETTER SHORT I
 [0x10407] = 0x1042F, -- DESERET CAPITAL LETTER SHORT E
 [0x10408] = 0x10430, -- DESERET CAPITAL LETTER SHORT A
 [0x10409] = 0x10431, -- DESERET CAPITAL LETTER SHORT AH
 [0x1040A] = 0x10432, -- DESERET CAPITAL LETTER SHORT O
 [0x1040B] = 0x10433, -- DESERET CAPITAL LETTER SHORT OO
 [0x1040C] = 0x10434, -- DESERET CAPITAL LETTER AY
 [0x1040D] = 0x10435, -- DESERET CAPITAL LETTER OW
 [0x1040E] = 0x10436, -- DESERET CAPITAL LETTER WU
 [0x1040F] = 0x10437, -- DESERET CAPITAL LETTER YEE
 [0x10410] = 0x10438, -- DESERET CAPITAL LETTER H
 [0x10411] = 0x10439, -- DESERET CAPITAL LETTER PEE
 [0x10412] = 0x1043A, -- DESERET CAPITAL LETTER BEE
 [0x10413] = 0x1043B, -- DESERET CAPITAL LETTER TEE
 [0x10414] = 0x1043C, -- DESERET CAPITAL LETTER DEE
 [0x10415] = 0x1043D, -- DESERET CAPITAL LETTER CHEE
 [0x10416] = 0x1043E, -- DESERET CAPITAL LETTER JEE
 [0x10417] = 0x1043F, -- DESERET CAPITAL LETTER KAY
 [0x10418] = 0x10440, -- DESERET CAPITAL LETTER GAY
 [0x10419] = 0x10441, -- DESERET CAPITAL LETTER EF
 [0x1041A] = 0x10442, -- DESERET CAPITAL LETTER VEE
 [0x1041B] = 0x10443, -- DESERET CAPITAL LETTER ETH
 [0x1041C] = 0x10444, -- DESERET CAPITAL LETTER THEE
 [0x1041D] = 0x10445, -- DESERET CAPITAL LETTER ES
 [0x1041E] = 0x10446, -- DESERET CAPITAL LETTER ZEE
 [0x1041F] = 0x10447, -- DESERET CAPITAL LETTER ESH
 [0x10420] = 0x10448, -- DESERET CAPITAL LETTER ZHEE
 [0x10421] = 0x10449, -- DESERET CAPITAL LETTER ER
 [0x10422] = 0x1044A, -- DESERET CAPITAL LETTER EL
 [0x10423] = 0x1044B, -- DESERET CAPITAL LETTER EM
 [0x10424] = 0x1044C, -- DESERET CAPITAL LETTER EN
 [0x10425] = 0x1044D, -- DESERET CAPITAL LETTER ENG
 [0x10426] = 0x1044E, -- DESERET CAPITAL LETTER OI
 [0x10427] = 0x1044F, -- DESERET CAPITAL LETTER EW
 -- NA   <reserved-1049E>..<reserved-1049F>
 [0x1049E] = {status="disallowed"},
 [0x1049F] = {status="disallowed"},
 [0x104B0] = 0x104D8, -- OSAGE CAPITAL LETTER A
 [0x104B1] = 0x104D9, -- OSAGE CAPITAL LETTER AI
 [0x104B2] = 0x104DA, -- OSAGE CAPITAL LETTER AIN
 [0x104B3] = 0x104DB, -- OSAGE CAPITAL LETTER AH
 [0x104B4] = 0x104DC, -- OSAGE CAPITAL LETTER BRA
 [0x104B5] = 0x104DD, -- OSAGE CAPITAL LETTER CHA
 [0x104B6] = 0x104DE, -- OSAGE CAPITAL LETTER EHCHA
 [0x104B7] = 0x104DF, -- OSAGE CAPITAL LETTER E
 [0x104B8] = 0x104E0, -- OSAGE CAPITAL LETTER EIN
 [0x104B9] = 0x104E1, -- OSAGE CAPITAL LETTER HA
 [0x104BA] = 0x104E2, -- OSAGE CAPITAL LETTER HYA
 [0x104BB] = 0x104E3, -- OSAGE CAPITAL LETTER I
 [0x104BC] = 0x104E4, -- OSAGE CAPITAL LETTER KA
 [0x104BD] = 0x104E5, -- OSAGE CAPITAL LETTER EHKA
 [0x104BE] = 0x104E6, -- OSAGE CAPITAL LETTER KYA
 [0x104BF] = 0x104E7, -- OSAGE CAPITAL LETTER LA
 [0x104C0] = 0x104E8, -- OSAGE CAPITAL LETTER MA
 [0x104C1] = 0x104E9, -- OSAGE CAPITAL LETTER NA
 [0x104C2] = 0x104EA, -- OSAGE CAPITAL LETTER O
 [0x104C3] = 0x104EB, -- OSAGE CAPITAL LETTER OIN
 [0x104C4] = 0x104EC, -- OSAGE CAPITAL LETTER PA
 [0x104C5] = 0x104ED, -- OSAGE CAPITAL LETTER EHPA
 [0x104C6] = 0x104EE, -- OSAGE CAPITAL LETTER SA
 [0x104C7] = 0x104EF, -- OSAGE CAPITAL LETTER SHA
 [0x104C8] = 0x104F0, -- OSAGE CAPITAL LETTER TA
 [0x104C9] = 0x104F1, -- OSAGE CAPITAL LETTER EHTA
 [0x104CA] = 0x104F2, -- OSAGE CAPITAL LETTER TSA
 [0x104CB] = 0x104F3, -- OSAGE CAPITAL LETTER EHTSA
 [0x104CC] = 0x104F4, -- OSAGE CAPITAL LETTER TSHA
 [0x104CD] = 0x104F5, -- OSAGE CAPITAL LETTER DHA
 [0x104CE] = 0x104F6, -- OSAGE CAPITAL LETTER U
 [0x104CF] = 0x104F7, -- OSAGE CAPITAL LETTER WA
 [0x104D0] = 0x104F8, -- OSAGE CAPITAL LETTER KHA
 [0x104D1] = 0x104F9, -- OSAGE CAPITAL LETTER GHA
 [0x104D2] = 0x104FA, -- OSAGE CAPITAL LETTER ZA
 [0x104D3] = 0x104FB, -- OSAGE CAPITAL LETTER ZHA
 -- NA   <reserved-104D4>..<reserved-104D7>
 [0x104D4] = {status="disallowed"},
 [0x104D5] = {status="disallowed"},
 [0x104D6] = {status="disallowed"},
 [0x104D7] = {status="disallowed"},
 -- NA   <reserved-104FC>..<reserved-104FF>
 [0x104FC] = {status="disallowed"},
 [0x104FD] = {status="disallowed"},
 [0x104FE] = {status="disallowed"},
 [0x104FF] = {status="disallowed"},
 [0x1056F] = {status="disallowed"}, -- CAUCASIAN ALBANIAN CITATION MARK
 [0x10808] = {status="valid"}, -- CYPRIOT SYLLABLE JO
 [0x10809] = {status="disallowed"}, -- NA   <reserved-10809>
 -- NA   <reserved-10806>..<reserved-10807>
 [0x10806] = {status="disallowed"},
 [0x10807] = {status="disallowed"},
 [0x10836] = {status="disallowed"}, -- NA   <reserved-10836>
 -- CYPRIOT SYLLABLE XA..CYPRIOT SYLLABLE XE
 [0x10837] = {status="valid"},
 [0x10838] = {status="valid"},
 [0x1083C] = {status="valid"}, -- CYPRIOT SYLLABLE ZA
 -- NA   <reserved-10839>..<reserved-1083B>
 [0x10839] = {status="disallowed"},
 [0x1083A] = {status="disallowed"},
 [0x1083B] = {status="disallowed"},
 [0x1083F] = {status="valid"}, -- CYPRIOT SYLLABLE ZO
 -- NA   <reserved-1083D>..<reserved-1083E>
 [0x1083D] = {status="disallowed"},
 [0x1083E] = {status="disallowed"},
 [0x10856] = {status="disallowed"}, -- NA   <reserved-10856>
 [0x108F3] = {status="disallowed"}, -- NA   <reserved-108F3>
 -- HATRAN LETTER SHIN..HATRAN LETTER TAW
 [0x108F4] = {status="valid"},
 [0x108F5] = {status="valid"},
 -- NA   <reserved-108F6>..<reserved-108FA>
 [0x108F6] = {status="disallowed"},
 [0x108F7] = {status="disallowed"},
 [0x108F8] = {status="disallowed"},
 [0x108F9] = {status="disallowed"},
 [0x108FA] = {status="disallowed"},
 -- HATRAN NUMBER ONE..HATRAN NUMBER ONE HUNDRED
 [0x108FB] = {status="disallowed"},
 [0x108FC] = {status="disallowed"},
 [0x108FD] = {status="disallowed"},
 [0x108FE] = {status="disallowed"},
 [0x108FF] = {status="disallowed"},
 [0x1091F] = {status="disallowed"}, -- PHOENICIAN WORD SEPARATOR
 -- NA   <reserved-1091C>..<reserved-1091E>
 [0x1091C] = {status="disallowed"},
 [0x1091D] = {status="disallowed"},
 [0x1091E] = {status="disallowed"},
 [0x1093F] = {status="disallowed"}, -- LYDIAN TRIANGULAR MARK
 -- NA   <reserved-1093A>..<reserved-1093E>
 [0x1093A] = {status="disallowed"},
 [0x1093B] = {status="disallowed"},
 [0x1093C] = {status="disallowed"},
 [0x1093D] = {status="disallowed"},
 [0x1093E] = {status="disallowed"},
 -- NA   <reserved-109B8>..<reserved-109BB>
 [0x109B8] = {status="disallowed"},
 [0x109B9] = {status="disallowed"},
 [0x109BA] = {status="disallowed"},
 [0x109BB] = {status="disallowed"},
 -- MEROITIC CURSIVE FRACTION ELEVEN TWELFTHS..MEROITIC CURSIVE FRACTION ONE HALF
 [0x109BC] = {status="disallowed"},
 [0x109BD] = {status="disallowed"},
 -- MEROITIC CURSIVE LOGOGRAM RMT..MEROITIC CURSIVE LOGOGRAM IMN
 [0x109BE] = {status="valid"},
 [0x109BF] = {status="valid"},
 -- NA   <reserved-109D0>..<reserved-109D1>
 [0x109D0] = {status="disallowed"},
 [0x109D1] = {status="disallowed"},
 [0x10A04] = {status="disallowed"}, -- NA   <reserved-10A04>
 -- KHAROSHTHI LETTER A..KHAROSHTHI VOWEL SIGN VOCALIC R
 [0x10A00] = {status="valid"},
 [0x10A01] = {status="valid"},
 [0x10A02] = {status="valid"},
 [0x10A03] = {status="valid"},
 -- KHAROSHTHI VOWEL SIGN E..KHAROSHTHI VOWEL SIGN O
 [0x10A05] = {status="valid"},
 [0x10A06] = {status="valid"},
 -- NA   <reserved-10A07>..<reserved-10A0B>
 [0x10A07] = {status="disallowed"},
 [0x10A08] = {status="disallowed"},
 [0x10A09] = {status="disallowed"},
 [0x10A0A] = {status="disallowed"},
 [0x10A0B] = {status="disallowed"},
 [0x10A14] = {status="disallowed"}, -- NA   <reserved-10A14>
 [0x10A18] = {status="disallowed"}, -- NA   <reserved-10A18>
 -- KHAROSHTHI LETTER CA..KHAROSHTHI LETTER JA
 [0x10A15] = {status="valid"},
 [0x10A16] = {status="valid"},
 [0x10A17] = {status="valid"},
 -- NA   <reserved-10A34>..<reserved-10A37>
 [0x10A34] = {status="disallowed"},
 [0x10A35] = {status="disallowed"},
 [0x10A36] = {status="disallowed"},
 [0x10A37] = {status="disallowed"},
 -- KHAROSHTHI SIGN BAR ABOVE..KHAROSHTHI SIGN DOT BELOW
 [0x10A38] = {status="valid"},
 [0x10A39] = {status="valid"},
 [0x10A3A] = {status="valid"},
 [0x10A3F] = {status="valid"}, -- KHAROSHTHI VIRAMA
 -- NA   <reserved-10A3B>..<reserved-10A3E>
 [0x10A3B] = {status="disallowed"},
 [0x10A3C] = {status="disallowed"},
 [0x10A3D] = {status="disallowed"},
 [0x10A3E] = {status="disallowed"},
 -- OLD SOUTH ARABIAN NUMBER ONE..OLD SOUTH ARABIAN NUMERIC INDICATOR
 [0x10A7D] = {status="disallowed"},
 [0x10A7E] = {status="disallowed"},
 [0x10A7F] = {status="disallowed"},
 -- OLD NORTH ARABIAN NUMBER ONE..OLD NORTH ARABIAN NUMBER TWENTY
 [0x10A9D] = {status="disallowed"},
 [0x10A9E] = {status="disallowed"},
 [0x10A9F] = {status="disallowed"},
 [0x10AC8] = {status="disallowed"}, -- MANICHAEAN SIGN UD
 -- NA   <reserved-10AE7>..<reserved-10AEA>
 [0x10AE7] = {status="disallowed"},
 [0x10AE8] = {status="disallowed"},
 [0x10AE9] = {status="disallowed"},
 [0x10AEA] = {status="disallowed"},
 -- NA   <reserved-10B36>..<reserved-10B38>
 [0x10B36] = {status="disallowed"},
 [0x10B37] = {status="disallowed"},
 [0x10B38] = {status="disallowed"},
 -- NA   <reserved-10B56>..<reserved-10B57>
 [0x10B56] = {status="disallowed"},
 [0x10B57] = {status="disallowed"},
 -- NA   <reserved-10B73>..<reserved-10B77>
 [0x10B73] = {status="disallowed"},
 [0x10B74] = {status="disallowed"},
 [0x10B75] = {status="disallowed"},
 [0x10B76] = {status="disallowed"},
 [0x10B77] = {status="disallowed"},
 -- PSALTER PAHLAVI SECTION MARK..PSALTER PAHLAVI FOUR DOTS WITH DOT
 [0x10B99] = {status="disallowed"},
 [0x10B9A] = {status="disallowed"},
 [0x10B9B] = {status="disallowed"},
 [0x10B9C] = {status="disallowed"},
 [0x10C80] = 0x10CC0, -- OLD HUNGARIAN CAPITAL LETTER A
 [0x10C81] = 0x10CC1, -- OLD HUNGARIAN CAPITAL LETTER AA
 [0x10C82] = 0x10CC2, -- OLD HUNGARIAN CAPITAL LETTER EB
 [0x10C83] = 0x10CC3, -- OLD HUNGARIAN CAPITAL LETTER AMB
 [0x10C84] = 0x10CC4, -- OLD HUNGARIAN CAPITAL LETTER EC
 [0x10C85] = 0x10CC5, -- OLD HUNGARIAN CAPITAL LETTER ENC
 [0x10C86] = 0x10CC6, -- OLD HUNGARIAN CAPITAL LETTER ECS
 [0x10C87] = 0x10CC7, -- OLD HUNGARIAN CAPITAL LETTER ED
 [0x10C88] = 0x10CC8, -- OLD HUNGARIAN CAPITAL LETTER AND
 [0x10C89] = 0x10CC9, -- OLD HUNGARIAN CAPITAL LETTER E
 [0x10C8A] = 0x10CCA, -- OLD HUNGARIAN CAPITAL LETTER CLOSE E
 [0x10C8B] = 0x10CCB, -- OLD HUNGARIAN CAPITAL LETTER EE
 [0x10C8C] = 0x10CCC, -- OLD HUNGARIAN CAPITAL LETTER EF
 [0x10C8D] = 0x10CCD, -- OLD HUNGARIAN CAPITAL LETTER EG
 [0x10C8E] = 0x10CCE, -- OLD HUNGARIAN CAPITAL LETTER EGY
 [0x10C8F] = 0x10CCF, -- OLD HUNGARIAN CAPITAL LETTER EH
 [0x10C90] = 0x10CD0, -- OLD HUNGARIAN CAPITAL LETTER I
 [0x10C91] = 0x10CD1, -- OLD HUNGARIAN CAPITAL LETTER II
 [0x10C92] = 0x10CD2, -- OLD HUNGARIAN CAPITAL LETTER EJ
 [0x10C93] = 0x10CD3, -- OLD HUNGARIAN CAPITAL LETTER EK
 [0x10C94] = 0x10CD4, -- OLD HUNGARIAN CAPITAL LETTER AK
 [0x10C95] = 0x10CD5, -- OLD HUNGARIAN CAPITAL LETTER UNK
 [0x10C96] = 0x10CD6, -- OLD HUNGARIAN CAPITAL LETTER EL
 [0x10C97] = 0x10CD7, -- OLD HUNGARIAN CAPITAL LETTER ELY
 [0x10C98] = 0x10CD8, -- OLD HUNGARIAN CAPITAL LETTER EM
 [0x10C99] = 0x10CD9, -- OLD HUNGARIAN CAPITAL LETTER EN
 [0x10C9A] = 0x10CDA, -- OLD HUNGARIAN CAPITAL LETTER ENY
 [0x10C9B] = 0x10CDB, -- OLD HUNGARIAN CAPITAL LETTER O
 [0x10C9C] = 0x10CDC, -- OLD HUNGARIAN CAPITAL LETTER OO
 [0x10C9D] = 0x10CDD, -- OLD HUNGARIAN CAPITAL LETTER NIKOLSBURG OE
 [0x10C9E] = 0x10CDE, -- OLD HUNGARIAN CAPITAL LETTER RUDIMENTA OE
 [0x10C9F] = 0x10CDF, -- OLD HUNGARIAN CAPITAL LETTER OEE
 [0x10CA0] = 0x10CE0, -- OLD HUNGARIAN CAPITAL LETTER EP
 [0x10CA1] = 0x10CE1, -- OLD HUNGARIAN CAPITAL LETTER EMP
 [0x10CA2] = 0x10CE2, -- OLD HUNGARIAN CAPITAL LETTER ER
 [0x10CA3] = 0x10CE3, -- OLD HUNGARIAN CAPITAL LETTER SHORT ER
 [0x10CA4] = 0x10CE4, -- OLD HUNGARIAN CAPITAL LETTER ES
 [0x10CA5] = 0x10CE5, -- OLD HUNGARIAN CAPITAL LETTER ESZ
 [0x10CA6] = 0x10CE6, -- OLD HUNGARIAN CAPITAL LETTER ET
 [0x10CA7] = 0x10CE7, -- OLD HUNGARIAN CAPITAL LETTER ENT
 [0x10CA8] = 0x10CE8, -- OLD HUNGARIAN CAPITAL LETTER ETY
 [0x10CA9] = 0x10CE9, -- OLD HUNGARIAN CAPITAL LETTER ECH
 [0x10CAA] = 0x10CEA, -- OLD HUNGARIAN CAPITAL LETTER U
 [0x10CAB] = 0x10CEB, -- OLD HUNGARIAN CAPITAL LETTER UU
 [0x10CAC] = 0x10CEC, -- OLD HUNGARIAN CAPITAL LETTER NIKOLSBURG UE
 [0x10CAD] = 0x10CED, -- OLD HUNGARIAN CAPITAL LETTER RUDIMENTA UE
 [0x10CAE] = 0x10CEE, -- OLD HUNGARIAN CAPITAL LETTER EV
 [0x10CAF] = 0x10CEF, -- OLD HUNGARIAN CAPITAL LETTER EZ
 [0x10CB0] = 0x10CF0, -- OLD HUNGARIAN CAPITAL LETTER EZS
 [0x10CB1] = 0x10CF1, -- OLD HUNGARIAN CAPITAL LETTER ENT-SHAPED SIGN
 [0x10CB2] = 0x10CF2, -- OLD HUNGARIAN CAPITAL LETTER US
 -- NA   <reserved-1104E>..<reserved-11051>
 [0x1104E] = {status="disallowed"},
 [0x1104F] = {status="disallowed"},
 [0x11050] = {status="disallowed"},
 [0x11051] = {status="disallowed"},
 [0x1107F] = {status="valid"}, -- BRAHMI NUMBER JOINER
 [0x110BD] = {status="disallowed"}, -- KAITHI NUMBER SIGN
 -- KAITHI ABBREVIATION SIGN..KAITHI ENUMERATION SIGN
 [0x110BB] = {status="disallowed"},
 [0x110BC] = {status="disallowed"},
 -- KAITHI SECTION MARK..KAITHI DOUBLE DANDA
 [0x110BE] = {status="disallowed"},
 [0x110BF] = {status="disallowed"},
 [0x110C0] = {status="disallowed"},
 [0x110C1] = {status="disallowed"},
 [0x11135] = {status="disallowed"}, -- NA   <reserved-11135>
 -- CHAKMA SECTION MARK..CHAKMA QUESTION MARK
 [0x11140] = {status="disallowed"},
 [0x11141] = {status="disallowed"},
 [0x11142] = {status="disallowed"},
 [0x11143] = {status="disallowed"},
 [0x11176] = {status="valid"}, -- MAHAJANI LIGATURE SHRI
 -- MAHAJANI ABBREVIATION SIGN..MAHAJANI SECTION MARK
 [0x11174] = {status="disallowed"},
 [0x11175] = {status="disallowed"},
 [0x111C9] = {status="disallowed"}, -- SHARADA SANDHI MARK
 -- SHARADA DANDA..SHARADA SEPARATOR
 [0x111C5] = {status="disallowed"},
 [0x111C6] = {status="disallowed"},
 [0x111C7] = {status="disallowed"},
 [0x111C8] = {status="disallowed"},
 [0x111CD] = {status="disallowed"}, -- SHARADA SUTRA MARK
 -- SHARADA SIGN NUKTA..SHARADA EXTRA SHORT VOWEL MARK
 [0x111CA] = {status="valid"},
 [0x111CB] = {status="valid"},
 [0x111CC] = {status="valid"},
 -- NA   <reserved-111CE>..<reserved-111CF>
 [0x111CE] = {status="disallowed"},
 [0x111CF] = {status="disallowed"},
 [0x111DA] = {status="valid"}, -- SHARADA EKAM
 [0x111DB] = {status="disallowed"}, -- SHARADA SIGN SIDDHAM
 [0x111DC] = {status="valid"}, -- SHARADA HEADSTROKE
 [0x111E0] = {status="disallowed"}, -- NA   <reserved-111E0>
 -- SHARADA CONTINUATION SIGN..SHARADA SECTION MARK-2
 [0x111DD] = {status="disallowed"},
 [0x111DE] = {status="disallowed"},
 [0x111DF] = {status="disallowed"},
 [0x11212] = {status="disallowed"}, -- NA   <reserved-11212>
 [0x1123E] = {status="valid"}, -- KHOJKI SIGN SUKUN
 [0x11287] = {status="disallowed"}, -- NA   <reserved-11287>
 [0x11288] = {status="valid"}, -- MULTANI LETTER GHA
 [0x11289] = {status="disallowed"}, -- NA   <reserved-11289>
 [0x1128E] = {status="disallowed"}, -- NA   <reserved-1128E>
 -- MULTANI LETTER CA..MULTANI LETTER JJA
 [0x1128A] = {status="valid"},
 [0x1128B] = {status="valid"},
 [0x1128C] = {status="valid"},
 [0x1128D] = {status="valid"},
 [0x1129E] = {status="disallowed"}, -- NA   <reserved-1129E>
 [0x112A9] = {status="disallowed"}, -- MULTANI SECTION MARK
 -- NA   <reserved-112EB>..<reserved-112EF>
 [0x112EB] = {status="disallowed"},
 [0x112EC] = {status="disallowed"},
 [0x112ED] = {status="disallowed"},
 [0x112EE] = {status="disallowed"},
 [0x112EF] = {status="disallowed"},
 [0x11300] = {status="valid"}, -- GRANTHA SIGN COMBINING ANUSVARA ABOVE
 [0x11304] = {status="disallowed"}, -- NA   <reserved-11304>
 -- GRANTHA SIGN CANDRABINDU..GRANTHA SIGN VISARGA
 [0x11301] = {status="valid"},
 [0x11302] = {status="valid"},
 [0x11303] = {status="valid"},
 -- NA   <reserved-1130D>..<reserved-1130E>
 [0x1130D] = {status="disallowed"},
 [0x1130E] = {status="disallowed"},
 -- GRANTHA LETTER EE..GRANTHA LETTER AI
 [0x1130F] = {status="valid"},
 [0x11310] = {status="valid"},
 -- NA   <reserved-11311>..<reserved-11312>
 [0x11311] = {status="disallowed"},
 [0x11312] = {status="disallowed"},
 [0x11329] = {status="disallowed"}, -- NA   <reserved-11329>
 [0x11331] = {status="disallowed"}, -- NA   <reserved-11331>
 [0x11334] = {status="disallowed"}, -- NA   <reserved-11334>
 -- GRANTHA LETTER LA..GRANTHA LETTER LLA
 [0x11332] = {status="valid"},
 [0x11333] = {status="valid"},
 -- GRANTHA LETTER VA..GRANTHA LETTER HA
 [0x11335] = {status="valid"},
 [0x11336] = {status="valid"},
 [0x11337] = {status="valid"},
 [0x11338] = {status="valid"},
 [0x11339] = {status="valid"},
 -- NA   <reserved-1133A>..<reserved-1133B>
 [0x1133A] = {status="disallowed"},
 [0x1133B] = {status="disallowed"},
 -- NA   <reserved-11345>..<reserved-11346>
 [0x11345] = {status="disallowed"},
 [0x11346] = {status="disallowed"},
 -- GRANTHA VOWEL SIGN EE..GRANTHA VOWEL SIGN AI
 [0x11347] = {status="valid"},
 [0x11348] = {status="valid"},
 -- NA   <reserved-11349>..<reserved-1134A>
 [0x11349] = {status="disallowed"},
 [0x1134A] = {status="disallowed"},
 -- GRANTHA VOWEL SIGN OO..GRANTHA SIGN VIRAMA
 [0x1134B] = {status="valid"},
 [0x1134C] = {status="valid"},
 [0x1134D] = {status="valid"},
 [0x11350] = {status="valid"}, -- GRANTHA OM
 -- NA   <reserved-1134E>..<reserved-1134F>
 [0x1134E] = {status="disallowed"},
 [0x1134F] = {status="disallowed"},
 [0x11357] = {status="valid"}, -- GRANTHA AU LENGTH MARK
 -- NA   <reserved-11358>..<reserved-1135C>
 [0x11358] = {status="disallowed"},
 [0x11359] = {status="disallowed"},
 [0x1135A] = {status="disallowed"},
 [0x1135B] = {status="disallowed"},
 [0x1135C] = {status="disallowed"},
 -- NA   <reserved-11364>..<reserved-11365>
 [0x11364] = {status="disallowed"},
 [0x11365] = {status="disallowed"},
 -- NA   <reserved-1136D>..<reserved-1136F>
 [0x1136D] = {status="disallowed"},
 [0x1136E] = {status="disallowed"},
 [0x1136F] = {status="disallowed"},
 -- COMBINING GRANTHA LETTER A..COMBINING GRANTHA LETTER PA
 [0x11370] = {status="valid"},
 [0x11371] = {status="valid"},
 [0x11372] = {status="valid"},
 [0x11373] = {status="valid"},
 [0x11374] = {status="valid"},
 -- NEWA DANDA..NEWA ABBREVIATION SIGN
 [0x1144B] = {status="disallowed"},
 [0x1144C] = {status="disallowed"},
 [0x1144D] = {status="disallowed"},
 [0x1144E] = {status="disallowed"},
 [0x1144F] = {status="disallowed"},
 [0x1145A] = {status="disallowed"}, -- NA   <reserved-1145A>
 [0x1145B] = {status="disallowed"}, -- NEWA PLACEHOLDER MARK
 [0x1145C] = {status="disallowed"}, -- NA   <reserved-1145C>
 [0x1145D] = {status="disallowed"}, -- NEWA INSERTION SIGN
 [0x114C6] = {status="disallowed"}, -- TIRHUTA ABBREVIATION SIGN
 [0x114C7] = {status="valid"}, -- TIRHUTA OM
 -- NA   <reserved-115B6>..<reserved-115B7>
 [0x115B6] = {status="disallowed"},
 [0x115B7] = {status="disallowed"},
 [0x11644] = {status="valid"}, -- MODI SIGN HUVA
 -- MODI DANDA..MODI ABBREVIATION SIGN
 [0x11641] = {status="disallowed"},
 [0x11642] = {status="disallowed"},
 [0x11643] = {status="disallowed"},
 -- NA   <reserved-1171A>..<reserved-1171C>
 [0x1171A] = {status="disallowed"},
 [0x1171B] = {status="disallowed"},
 [0x1171C] = {status="disallowed"},
 -- NA   <reserved-1172C>..<reserved-1172F>
 [0x1172C] = {status="disallowed"},
 [0x1172D] = {status="disallowed"},
 [0x1172E] = {status="disallowed"},
 [0x1172F] = {status="disallowed"},
 [0x118A0] = 0x118C0, -- WARANG CITI CAPITAL LETTER NGAA
 [0x118A1] = 0x118C1, -- WARANG CITI CAPITAL LETTER A
 [0x118A2] = 0x118C2, -- WARANG CITI CAPITAL LETTER WI
 [0x118A3] = 0x118C3, -- WARANG CITI CAPITAL LETTER YU
 [0x118A4] = 0x118C4, -- WARANG CITI CAPITAL LETTER YA
 [0x118A5] = 0x118C5, -- WARANG CITI CAPITAL LETTER YO
 [0x118A6] = 0x118C6, -- WARANG CITI CAPITAL LETTER II
 [0x118A7] = 0x118C7, -- WARANG CITI CAPITAL LETTER UU
 [0x118A8] = 0x118C8, -- WARANG CITI CAPITAL LETTER E
 [0x118A9] = 0x118C9, -- WARANG CITI CAPITAL LETTER O
 [0x118AA] = 0x118CA, -- WARANG CITI CAPITAL LETTER ANG
 [0x118AB] = 0x118CB, -- WARANG CITI CAPITAL LETTER GA
 [0x118AC] = 0x118CC, -- WARANG CITI CAPITAL LETTER KO
 [0x118AD] = 0x118CD, -- WARANG CITI CAPITAL LETTER ENY
 [0x118AE] = 0x118CE, -- WARANG CITI CAPITAL LETTER YUJ
 [0x118AF] = 0x118CF, -- WARANG CITI CAPITAL LETTER UC
 [0x118B0] = 0x118D0, -- WARANG CITI CAPITAL LETTER ENN
 [0x118B1] = 0x118D1, -- WARANG CITI CAPITAL LETTER ODD
 [0x118B2] = 0x118D2, -- WARANG CITI CAPITAL LETTER TTE
 [0x118B3] = 0x118D3, -- WARANG CITI CAPITAL LETTER NUNG
 [0x118B4] = 0x118D4, -- WARANG CITI CAPITAL LETTER DA
 [0x118B5] = 0x118D5, -- WARANG CITI CAPITAL LETTER AT
 [0x118B6] = 0x118D6, -- WARANG CITI CAPITAL LETTER AM
 [0x118B7] = 0x118D7, -- WARANG CITI CAPITAL LETTER BU
 [0x118B8] = 0x118D8, -- WARANG CITI CAPITAL LETTER PU
 [0x118B9] = 0x118D9, -- WARANG CITI CAPITAL LETTER HIYO
 [0x118BA] = 0x118DA, -- WARANG CITI CAPITAL LETTER HOLO
 [0x118BB] = 0x118DB, -- WARANG CITI CAPITAL LETTER HORR
 [0x118BC] = 0x118DC, -- WARANG CITI CAPITAL LETTER HAR
 [0x118BD] = 0x118DD, -- WARANG CITI CAPITAL LETTER SSUU
 [0x118BE] = 0x118DE, -- WARANG CITI CAPITAL LETTER SII
 [0x118BF] = 0x118DF, -- WARANG CITI CAPITAL LETTER VIYO
 [0x118FF] = {status="valid"}, -- WARANG CITI OM
 [0x11A47] = {status="valid"}, -- ZANABAZAR SQUARE SUBJOINER
 -- NA   <reserved-11A84>..<reserved-11A85>
 [0x11A84] = {status="disallowed"},
 [0x11A85] = {status="disallowed"},
 [0x11A9D] = {status="disallowed"}, -- NA   <reserved-11A9D>
 -- SOYOMBO MARK TSHEG..SOYOMBO MARK DOUBLE SHAD
 [0x11A9A] = {status="disallowed"},
 [0x11A9B] = {status="disallowed"},
 [0x11A9C] = {status="disallowed"},
 -- SOYOMBO HEAD MARK WITH MOON AND SUN AND TRIPLE FLAME..SOYOMBO TERMINAL MARK-2
 [0x11A9E] = {status="disallowed"},
 [0x11A9F] = {status="disallowed"},
 [0x11AA0] = {status="disallowed"},
 [0x11AA1] = {status="disallowed"},
 [0x11AA2] = {status="disallowed"},
 [0x11C09] = {status="disallowed"}, -- NA   <reserved-11C09>
 [0x11C37] = {status="disallowed"}, -- NA   <reserved-11C37>
 -- BHAIKSUKI DANDA..BHAIKSUKI GAP FILLER-2
 [0x11C41] = {status="disallowed"},
 [0x11C42] = {status="disallowed"},
 [0x11C43] = {status="disallowed"},
 [0x11C44] = {status="disallowed"},
 [0x11C45] = {status="disallowed"},
 -- NA   <reserved-11C6D>..<reserved-11C6F>
 [0x11C6D] = {status="disallowed"},
 [0x11C6E] = {status="disallowed"},
 [0x11C6F] = {status="disallowed"},
 -- MARCHEN HEAD MARK..MARCHEN MARK SHAD
 [0x11C70] = {status="disallowed"},
 [0x11C71] = {status="disallowed"},
 -- NA   <reserved-11C90>..<reserved-11C91>
 [0x11C90] = {status="disallowed"},
 [0x11C91] = {status="disallowed"},
 [0x11CA8] = {status="disallowed"}, -- NA   <reserved-11CA8>
 [0x11D07] = {status="disallowed"}, -- NA   <reserved-11D07>
 [0x11D0A] = {status="disallowed"}, -- NA   <reserved-11D0A>
 -- MASARAM GONDI LETTER AI..MASARAM GONDI LETTER O
 [0x11D08] = {status="valid"},
 [0x11D09] = {status="valid"},
 [0x11D3A] = {status="valid"}, -- MASARAM GONDI VOWEL SIGN E
 [0x11D3B] = {status="disallowed"}, -- NA   <reserved-11D3B>
 -- NA   <reserved-11D37>..<reserved-11D39>
 [0x11D37] = {status="disallowed"},
 [0x11D38] = {status="disallowed"},
 [0x11D39] = {status="disallowed"},
 [0x11D3E] = {status="disallowed"}, -- NA   <reserved-11D3E>
 -- MASARAM GONDI VOWEL SIGN AI..MASARAM GONDI VOWEL SIGN O
 [0x11D3C] = {status="valid"},
 [0x11D3D] = {status="valid"},
 [0x12399] = {status="valid"}, -- CUNEIFORM SIGN U U
 [0x1246F] = {status="disallowed"}, -- NA   <reserved-1246F>
 [0x12474] = {status="disallowed"}, -- CUNEIFORM PUNCTUATION SIGN DIAGONAL QUADCOLON
 -- CUNEIFORM PUNCTUATION SIGN OLD ASSYRIAN WORD DIVIDER..CUNEIFORM PUNCTUATION SIGN DIAGONAL TRICOLON
 [0x12470] = {status="disallowed"},
 [0x12471] = {status="disallowed"},
 [0x12472] = {status="disallowed"},
 [0x12473] = {status="disallowed"},
 [0x16A5F] = {status="disallowed"}, -- NA   <reserved-16A5F>
 -- NA   <reserved-16A6A>..<reserved-16A6D>
 [0x16A6A] = {status="disallowed"},
 [0x16A6B] = {status="disallowed"},
 [0x16A6C] = {status="disallowed"},
 [0x16A6D] = {status="disallowed"},
 -- MRO DANDA..MRO DOUBLE DANDA
 [0x16A6E] = {status="disallowed"},
 [0x16A6F] = {status="disallowed"},
 -- NA   <reserved-16AEE>..<reserved-16AEF>
 [0x16AEE] = {status="disallowed"},
 [0x16AEF] = {status="disallowed"},
 [0x16AF5] = {status="disallowed"}, -- BASSA VAH FULL STOP
 -- BASSA VAH COMBINING HIGH TONE..BASSA VAH COMBINING HIGH-LOW TONE
 [0x16AF0] = {status="valid"},
 [0x16AF1] = {status="valid"},
 [0x16AF2] = {status="valid"},
 [0x16AF3] = {status="valid"},
 [0x16AF4] = {status="valid"},
 -- PAHAWH HMONG SIGN VOS SEEV..PAHAWH HMONG SIGN IB YAM
 [0x16B40] = {status="valid"},
 [0x16B41] = {status="valid"},
 [0x16B42] = {status="valid"},
 [0x16B43] = {status="valid"},
 -- PAHAWH HMONG SIGN XAUS..PAHAWH HMONG SIGN CIM TSOV ROG
 [0x16B44] = {status="disallowed"},
 [0x16B45] = {status="disallowed"},
 [0x16B5A] = {status="disallowed"}, -- NA   <reserved-16B5A>
 [0x16B62] = {status="disallowed"}, -- NA   <reserved-16B62>
 -- NA   <reserved-16B78>..<reserved-16B7C>
 [0x16B78] = {status="disallowed"},
 [0x16B79] = {status="disallowed"},
 [0x16B7A] = {status="disallowed"},
 [0x16B7B] = {status="disallowed"},
 [0x16B7C] = {status="disallowed"},
 [0x16FE0] = {status="valid"}, -- TANGUT ITERATION MARK
 [0x16FE1] = {status="valid"}, -- NUSHU ITERATION MARK
 -- NA   <reserved-1BC6B>..<reserved-1BC6F>
 [0x1BC6B] = {status="disallowed"},
 [0x1BC6C] = {status="disallowed"},
 [0x1BC6D] = {status="disallowed"},
 [0x1BC6E] = {status="disallowed"},
 [0x1BC6F] = {status="disallowed"},
 -- NA   <reserved-1BC7D>..<reserved-1BC7F>
 [0x1BC7D] = {status="disallowed"},
 [0x1BC7E] = {status="disallowed"},
 [0x1BC7F] = {status="disallowed"},
 [0x1BC9C] = {status="disallowed"}, -- DUPLOYAN SIGN O WITH CROSS
 -- NA   <reserved-1BC9A>..<reserved-1BC9B>
 [0x1BC9A] = {status="disallowed"},
 [0x1BC9B] = {status="disallowed"},
 [0x1BC9F] = {status="disallowed"}, -- DUPLOYAN PUNCTUATION CHINOOK FULL STOP
 -- DUPLOYAN THICK LETTER SELECTOR..DUPLOYAN DOUBLE MARK
 [0x1BC9D] = {status="valid"},
 [0x1BC9E] = {status="valid"},
 -- SHORTHAND FORMAT LETTER OVERLAP..SHORTHAND FORMAT UP STEP
 [0x1BCA0] = {status='ignored'},
 [0x1BCA1] = {status='ignored'},
 [0x1BCA2] = {status='ignored'},
 [0x1BCA3] = {status='ignored'},
 [0x1D129] = {status="disallowed"}, -- MUSICAL SYMBOL MULTIPLE MEASURE REST
 -- NA   <reserved-1D127>..<reserved-1D128>
 [0x1D127] = {status="disallowed"},
 [0x1D128] = {status="disallowed"},
 [0x1D15E] = {0x1D157, 0x1D165}, -- MUSICAL SYMBOL HALF NOTE
 [0x1D15F] = {0x1D158, 0x1D165}, -- MUSICAL SYMBOL QUARTER NOTE
 [0x1D160] = {0x1D158, 0x1D165, 0x1D16E}, -- MUSICAL SYMBOL EIGHTH NOTE
 [0x1D161] = {0x1D158, 0x1D165, 0x1D16F}, -- MUSICAL SYMBOL SIXTEENTH NOTE
 [0x1D162] = {0x1D158, 0x1D165, 0x1D170}, -- MUSICAL SYMBOL THIRTY-SECOND NOTE
 [0x1D163] = {0x1D158, 0x1D165, 0x1D171}, -- MUSICAL SYMBOL SIXTY-FOURTH NOTE
 [0x1D164] = {0x1D158, 0x1D165, 0x1D172}, -- MUSICAL SYMBOL ONE HUNDRED TWENTY-EIGHTH NOTE
 [0x1D1BB] = {0x1D1B9, 0x1D165}, -- MUSICAL SYMBOL MINIMA
 [0x1D1BC] = {0x1D1BA, 0x1D165}, -- MUSICAL SYMBOL MINIMA BLACK
 [0x1D1BD] = {0x1D1B9, 0x1D165, 0x1D16E}, -- MUSICAL SYMBOL SEMIMINIMA WHITE
 [0x1D1BE] = {0x1D1BA, 0x1D165, 0x1D16E}, -- MUSICAL SYMBOL SEMIMINIMA BLACK
 [0x1D1BF] = {0x1D1B9, 0x1D165, 0x1D16F}, -- MUSICAL SYMBOL FUSA WHITE
 [0x1D1C0] = {0x1D1BA, 0x1D165, 0x1D16F}, -- MUSICAL SYMBOL FUSA BLACK
 [0x1D400] = 0x0061, -- MATHEMATICAL BOLD CAPITAL A
 [0x1D401] = 0x0062, -- MATHEMATICAL BOLD CAPITAL B
 [0x1D402] = 0x0063, -- MATHEMATICAL BOLD CAPITAL C
 [0x1D403] = 0x0064, -- MATHEMATICAL BOLD CAPITAL D
 [0x1D404] = 0x0065, -- MATHEMATICAL BOLD CAPITAL E
 [0x1D405] = 0x0066, -- MATHEMATICAL BOLD CAPITAL F
 [0x1D406] = 0x0067, -- MATHEMATICAL BOLD CAPITAL G
 [0x1D407] = 0x0068, -- MATHEMATICAL BOLD CAPITAL H
 [0x1D408] = 0x0069, -- MATHEMATICAL BOLD CAPITAL I
 [0x1D409] = 0x006A, -- MATHEMATICAL BOLD CAPITAL J
 [0x1D40A] = 0x006B, -- MATHEMATICAL BOLD CAPITAL K
 [0x1D40B] = 0x006C, -- MATHEMATICAL BOLD CAPITAL L
 [0x1D40C] = 0x006D, -- MATHEMATICAL BOLD CAPITAL M
 [0x1D40D] = 0x006E, -- MATHEMATICAL BOLD CAPITAL N
 [0x1D40E] = 0x006F, -- MATHEMATICAL BOLD CAPITAL O
 [0x1D40F] = 0x0070, -- MATHEMATICAL BOLD CAPITAL P
 [0x1D410] = 0x0071, -- MATHEMATICAL BOLD CAPITAL Q
 [0x1D411] = 0x0072, -- MATHEMATICAL BOLD CAPITAL R
 [0x1D412] = 0x0073, -- MATHEMATICAL BOLD CAPITAL S
 [0x1D413] = 0x0074, -- MATHEMATICAL BOLD CAPITAL T
 [0x1D414] = 0x0075, -- MATHEMATICAL BOLD CAPITAL U
 [0x1D415] = 0x0076, -- MATHEMATICAL BOLD CAPITAL V
 [0x1D416] = 0x0077, -- MATHEMATICAL BOLD CAPITAL W
 [0x1D417] = 0x0078, -- MATHEMATICAL BOLD CAPITAL X
 [0x1D418] = 0x0079, -- MATHEMATICAL BOLD CAPITAL Y
 [0x1D419] = 0x007A, -- MATHEMATICAL BOLD CAPITAL Z
 [0x1D41A] = 0x0061, -- MATHEMATICAL BOLD SMALL A
 [0x1D41B] = 0x0062, -- MATHEMATICAL BOLD SMALL B
 [0x1D41C] = 0x0063, -- MATHEMATICAL BOLD SMALL C
 [0x1D41D] = 0x0064, -- MATHEMATICAL BOLD SMALL D
 [0x1D41E] = 0x0065, -- MATHEMATICAL BOLD SMALL E
 [0x1D41F] = 0x0066, -- MATHEMATICAL BOLD SMALL F
 [0x1D420] = 0x0067, -- MATHEMATICAL BOLD SMALL G
 [0x1D421] = 0x0068, -- MATHEMATICAL BOLD SMALL H
 [0x1D422] = 0x0069, -- MATHEMATICAL BOLD SMALL I
 [0x1D423] = 0x006A, -- MATHEMATICAL BOLD SMALL J
 [0x1D424] = 0x006B, -- MATHEMATICAL BOLD SMALL K
 [0x1D425] = 0x006C, -- MATHEMATICAL BOLD SMALL L
 [0x1D426] = 0x006D, -- MATHEMATICAL BOLD SMALL M
 [0x1D427] = 0x006E, -- MATHEMATICAL BOLD SMALL N
 [0x1D428] = 0x006F, -- MATHEMATICAL BOLD SMALL O
 [0x1D429] = 0x0070, -- MATHEMATICAL BOLD SMALL P
 [0x1D42A] = 0x0071, -- MATHEMATICAL BOLD SMALL Q
 [0x1D42B] = 0x0072, -- MATHEMATICAL BOLD SMALL R
 [0x1D42C] = 0x0073, -- MATHEMATICAL BOLD SMALL S
 [0x1D42D] = 0x0074, -- MATHEMATICAL BOLD SMALL T
 [0x1D42E] = 0x0075, -- MATHEMATICAL BOLD SMALL U
 [0x1D42F] = 0x0076, -- MATHEMATICAL BOLD SMALL V
 [0x1D430] = 0x0077, -- MATHEMATICAL BOLD SMALL W
 [0x1D431] = 0x0078, -- MATHEMATICAL BOLD SMALL X
 [0x1D432] = 0x0079, -- MATHEMATICAL BOLD SMALL Y
 [0x1D433] = 0x007A, -- MATHEMATICAL BOLD SMALL Z
 [0x1D434] = 0x0061, -- MATHEMATICAL ITALIC CAPITAL A
 [0x1D435] = 0x0062, -- MATHEMATICAL ITALIC CAPITAL B
 [0x1D436] = 0x0063, -- MATHEMATICAL ITALIC CAPITAL C
 [0x1D437] = 0x0064, -- MATHEMATICAL ITALIC CAPITAL D
 [0x1D438] = 0x0065, -- MATHEMATICAL ITALIC CAPITAL E
 [0x1D439] = 0x0066, -- MATHEMATICAL ITALIC CAPITAL F
 [0x1D43A] = 0x0067, -- MATHEMATICAL ITALIC CAPITAL G
 [0x1D43B] = 0x0068, -- MATHEMATICAL ITALIC CAPITAL H
 [0x1D43C] = 0x0069, -- MATHEMATICAL ITALIC CAPITAL I
 [0x1D43D] = 0x006A, -- MATHEMATICAL ITALIC CAPITAL J
 [0x1D43E] = 0x006B, -- MATHEMATICAL ITALIC CAPITAL K
 [0x1D43F] = 0x006C, -- MATHEMATICAL ITALIC CAPITAL L
 [0x1D440] = 0x006D, -- MATHEMATICAL ITALIC CAPITAL M
 [0x1D441] = 0x006E, -- MATHEMATICAL ITALIC CAPITAL N
 [0x1D442] = 0x006F, -- MATHEMATICAL ITALIC CAPITAL O
 [0x1D443] = 0x0070, -- MATHEMATICAL ITALIC CAPITAL P
 [0x1D444] = 0x0071, -- MATHEMATICAL ITALIC CAPITAL Q
 [0x1D445] = 0x0072, -- MATHEMATICAL ITALIC CAPITAL R
 [0x1D446] = 0x0073, -- MATHEMATICAL ITALIC CAPITAL S
 [0x1D447] = 0x0074, -- MATHEMATICAL ITALIC CAPITAL T
 [0x1D448] = 0x0075, -- MATHEMATICAL ITALIC CAPITAL U
 [0x1D449] = 0x0076, -- MATHEMATICAL ITALIC CAPITAL V
 [0x1D44A] = 0x0077, -- MATHEMATICAL ITALIC CAPITAL W
 [0x1D44B] = 0x0078, -- MATHEMATICAL ITALIC CAPITAL X
 [0x1D44C] = 0x0079, -- MATHEMATICAL ITALIC CAPITAL Y
 [0x1D44D] = 0x007A, -- MATHEMATICAL ITALIC CAPITAL Z
 [0x1D44E] = 0x0061, -- MATHEMATICAL ITALIC SMALL A
 [0x1D44F] = 0x0062, -- MATHEMATICAL ITALIC SMALL B
 [0x1D450] = 0x0063, -- MATHEMATICAL ITALIC SMALL C
 [0x1D451] = 0x0064, -- MATHEMATICAL ITALIC SMALL D
 [0x1D452] = 0x0065, -- MATHEMATICAL ITALIC SMALL E
 [0x1D453] = 0x0066, -- MATHEMATICAL ITALIC SMALL F
 [0x1D454] = 0x0067, -- MATHEMATICAL ITALIC SMALL G
 [0x1D455] = {status="disallowed"}, -- NA   <reserved-1D455>
 [0x1D456] = 0x0069, -- MATHEMATICAL ITALIC SMALL I
 [0x1D457] = 0x006A, -- MATHEMATICAL ITALIC SMALL J
 [0x1D458] = 0x006B, -- MATHEMATICAL ITALIC SMALL K
 [0x1D459] = 0x006C, -- MATHEMATICAL ITALIC SMALL L
 [0x1D45A] = 0x006D, -- MATHEMATICAL ITALIC SMALL M
 [0x1D45B] = 0x006E, -- MATHEMATICAL ITALIC SMALL N
 [0x1D45C] = 0x006F, -- MATHEMATICAL ITALIC SMALL O
 [0x1D45D] = 0x0070, -- MATHEMATICAL ITALIC SMALL P
 [0x1D45E] = 0x0071, -- MATHEMATICAL ITALIC SMALL Q
 [0x1D45F] = 0x0072, -- MATHEMATICAL ITALIC SMALL R
 [0x1D460] = 0x0073, -- MATHEMATICAL ITALIC SMALL S
 [0x1D461] = 0x0074, -- MATHEMATICAL ITALIC SMALL T
 [0x1D462] = 0x0075, -- MATHEMATICAL ITALIC SMALL U
 [0x1D463] = 0x0076, -- MATHEMATICAL ITALIC SMALL V
 [0x1D464] = 0x0077, -- MATHEMATICAL ITALIC SMALL W
 [0x1D465] = 0x0078, -- MATHEMATICAL ITALIC SMALL X
 [0x1D466] = 0x0079, -- MATHEMATICAL ITALIC SMALL Y
 [0x1D467] = 0x007A, -- MATHEMATICAL ITALIC SMALL Z
 [0x1D468] = 0x0061, -- MATHEMATICAL BOLD ITALIC CAPITAL A
 [0x1D469] = 0x0062, -- MATHEMATICAL BOLD ITALIC CAPITAL B
 [0x1D46A] = 0x0063, -- MATHEMATICAL BOLD ITALIC CAPITAL C
 [0x1D46B] = 0x0064, -- MATHEMATICAL BOLD ITALIC CAPITAL D
 [0x1D46C] = 0x0065, -- MATHEMATICAL BOLD ITALIC CAPITAL E
 [0x1D46D] = 0x0066, -- MATHEMATICAL BOLD ITALIC CAPITAL F
 [0x1D46E] = 0x0067, -- MATHEMATICAL BOLD ITALIC CAPITAL G
 [0x1D46F] = 0x0068, -- MATHEMATICAL BOLD ITALIC CAPITAL H
 [0x1D470] = 0x0069, -- MATHEMATICAL BOLD ITALIC CAPITAL I
 [0x1D471] = 0x006A, -- MATHEMATICAL BOLD ITALIC CAPITAL J
 [0x1D472] = 0x006B, -- MATHEMATICAL BOLD ITALIC CAPITAL K
 [0x1D473] = 0x006C, -- MATHEMATICAL BOLD ITALIC CAPITAL L
 [0x1D474] = 0x006D, -- MATHEMATICAL BOLD ITALIC CAPITAL M
 [0x1D475] = 0x006E, -- MATHEMATICAL BOLD ITALIC CAPITAL N
 [0x1D476] = 0x006F, -- MATHEMATICAL BOLD ITALIC CAPITAL O
 [0x1D477] = 0x0070, -- MATHEMATICAL BOLD ITALIC CAPITAL P
 [0x1D478] = 0x0071, -- MATHEMATICAL BOLD ITALIC CAPITAL Q
 [0x1D479] = 0x0072, -- MATHEMATICAL BOLD ITALIC CAPITAL R
 [0x1D47A] = 0x0073, -- MATHEMATICAL BOLD ITALIC CAPITAL S
 [0x1D47B] = 0x0074, -- MATHEMATICAL BOLD ITALIC CAPITAL T
 [0x1D47C] = 0x0075, -- MATHEMATICAL BOLD ITALIC CAPITAL U
 [0x1D47D] = 0x0076, -- MATHEMATICAL BOLD ITALIC CAPITAL V
 [0x1D47E] = 0x0077, -- MATHEMATICAL BOLD ITALIC CAPITAL W
 [0x1D47F] = 0x0078, -- MATHEMATICAL BOLD ITALIC CAPITAL X
 [0x1D480] = 0x0079, -- MATHEMATICAL BOLD ITALIC CAPITAL Y
 [0x1D481] = 0x007A, -- MATHEMATICAL BOLD ITALIC CAPITAL Z
 [0x1D482] = 0x0061, -- MATHEMATICAL BOLD ITALIC SMALL A
 [0x1D483] = 0x0062, -- MATHEMATICAL BOLD ITALIC SMALL B
 [0x1D484] = 0x0063, -- MATHEMATICAL BOLD ITALIC SMALL C
 [0x1D485] = 0x0064, -- MATHEMATICAL BOLD ITALIC SMALL D
 [0x1D486] = 0x0065, -- MATHEMATICAL BOLD ITALIC SMALL E
 [0x1D487] = 0x0066, -- MATHEMATICAL BOLD ITALIC SMALL F
 [0x1D488] = 0x0067, -- MATHEMATICAL BOLD ITALIC SMALL G
 [0x1D489] = 0x0068, -- MATHEMATICAL BOLD ITALIC SMALL H
 [0x1D48A] = 0x0069, -- MATHEMATICAL BOLD ITALIC SMALL I
 [0x1D48B] = 0x006A, -- MATHEMATICAL BOLD ITALIC SMALL J
 [0x1D48C] = 0x006B, -- MATHEMATICAL BOLD ITALIC SMALL K
 [0x1D48D] = 0x006C, -- MATHEMATICAL BOLD ITALIC SMALL L
 [0x1D48E] = 0x006D, -- MATHEMATICAL BOLD ITALIC SMALL M
 [0x1D48F] = 0x006E, -- MATHEMATICAL BOLD ITALIC SMALL N
 [0x1D490] = 0x006F, -- MATHEMATICAL BOLD ITALIC SMALL O
 [0x1D491] = 0x0070, -- MATHEMATICAL BOLD ITALIC SMALL P
 [0x1D492] = 0x0071, -- MATHEMATICAL BOLD ITALIC SMALL Q
 [0x1D493] = 0x0072, -- MATHEMATICAL BOLD ITALIC SMALL R
 [0x1D494] = 0x0073, -- MATHEMATICAL BOLD ITALIC SMALL S
 [0x1D495] = 0x0074, -- MATHEMATICAL BOLD ITALIC SMALL T
 [0x1D496] = 0x0075, -- MATHEMATICAL BOLD ITALIC SMALL U
 [0x1D497] = 0x0076, -- MATHEMATICAL BOLD ITALIC SMALL V
 [0x1D498] = 0x0077, -- MATHEMATICAL BOLD ITALIC SMALL W
 [0x1D499] = 0x0078, -- MATHEMATICAL BOLD ITALIC SMALL X
 [0x1D49A] = 0x0079, -- MATHEMATICAL BOLD ITALIC SMALL Y
 [0x1D49B] = 0x007A, -- MATHEMATICAL BOLD ITALIC SMALL Z
 [0x1D49C] = 0x0061, -- MATHEMATICAL SCRIPT CAPITAL A
 [0x1D49D] = {status="disallowed"}, -- NA   <reserved-1D49D>
 [0x1D49E] = 0x0063, -- MATHEMATICAL SCRIPT CAPITAL C
 [0x1D49F] = 0x0064, -- MATHEMATICAL SCRIPT CAPITAL D
 [0x1D4A2] = 0x0067, -- MATHEMATICAL SCRIPT CAPITAL G
 -- NA   <reserved-1D4A0>..<reserved-1D4A1>
 [0x1D4A0] = {status="disallowed"},
 [0x1D4A1] = {status="disallowed"},
 [0x1D4A5] = 0x006A, -- MATHEMATICAL SCRIPT CAPITAL J
 [0x1D4A6] = 0x006B, -- MATHEMATICAL SCRIPT CAPITAL K
 -- NA   <reserved-1D4A3>..<reserved-1D4A4>
 [0x1D4A3] = {status="disallowed"},
 [0x1D4A4] = {status="disallowed"},
 [0x1D4A9] = 0x006E, -- MATHEMATICAL SCRIPT CAPITAL N
 [0x1D4AA] = 0x006F, -- MATHEMATICAL SCRIPT CAPITAL O
 [0x1D4AB] = 0x0070, -- MATHEMATICAL SCRIPT CAPITAL P
 [0x1D4AC] = 0x0071, -- MATHEMATICAL SCRIPT CAPITAL Q
 [0x1D4AD] = {status="disallowed"}, -- NA   <reserved-1D4AD>
 [0x1D4AE] = 0x0073, -- MATHEMATICAL SCRIPT CAPITAL S
 [0x1D4AF] = 0x0074, -- MATHEMATICAL SCRIPT CAPITAL T
 [0x1D4B0] = 0x0075, -- MATHEMATICAL SCRIPT CAPITAL U
 [0x1D4B1] = 0x0076, -- MATHEMATICAL SCRIPT CAPITAL V
 [0x1D4B2] = 0x0077, -- MATHEMATICAL SCRIPT CAPITAL W
 [0x1D4B3] = 0x0078, -- MATHEMATICAL SCRIPT CAPITAL X
 [0x1D4B4] = 0x0079, -- MATHEMATICAL SCRIPT CAPITAL Y
 [0x1D4B5] = 0x007A, -- MATHEMATICAL SCRIPT CAPITAL Z
 [0x1D4B6] = 0x0061, -- MATHEMATICAL SCRIPT SMALL A
 [0x1D4B7] = 0x0062, -- MATHEMATICAL SCRIPT SMALL B
 [0x1D4B8] = 0x0063, -- MATHEMATICAL SCRIPT SMALL C
 [0x1D4B9] = 0x0064, -- MATHEMATICAL SCRIPT SMALL D
 [0x1D4BA] = {status="disallowed"}, -- NA   <reserved-1D4BA>
 [0x1D4BB] = 0x0066, -- MATHEMATICAL SCRIPT SMALL F
 [0x1D4BC] = {status="disallowed"}, -- NA   <reserved-1D4BC>
 [0x1D4BD] = 0x0068, -- MATHEMATICAL SCRIPT SMALL H
 [0x1D4BE] = 0x0069, -- MATHEMATICAL SCRIPT SMALL I
 [0x1D4BF] = 0x006A, -- MATHEMATICAL SCRIPT SMALL J
 [0x1D4C0] = 0x006B, -- MATHEMATICAL SCRIPT SMALL K
 [0x1D4C1] = 0x006C, -- MATHEMATICAL SCRIPT SMALL L
 [0x1D4C2] = 0x006D, -- MATHEMATICAL SCRIPT SMALL M
 [0x1D4C3] = 0x006E, -- MATHEMATICAL SCRIPT SMALL N
 [0x1D4C4] = {status="disallowed"}, -- NA   <reserved-1D4C4>
 [0x1D4C5] = 0x0070, -- MATHEMATICAL SCRIPT SMALL P
 [0x1D4C6] = 0x0071, -- MATHEMATICAL SCRIPT SMALL Q
 [0x1D4C7] = 0x0072, -- MATHEMATICAL SCRIPT SMALL R
 [0x1D4C8] = 0x0073, -- MATHEMATICAL SCRIPT SMALL S
 [0x1D4C9] = 0x0074, -- MATHEMATICAL SCRIPT SMALL T
 [0x1D4CA] = 0x0075, -- MATHEMATICAL SCRIPT SMALL U
 [0x1D4CB] = 0x0076, -- MATHEMATICAL SCRIPT SMALL V
 [0x1D4CC] = 0x0077, -- MATHEMATICAL SCRIPT SMALL W
 [0x1D4CD] = 0x0078, -- MATHEMATICAL SCRIPT SMALL X
 [0x1D4CE] = 0x0079, -- MATHEMATICAL SCRIPT SMALL Y
 [0x1D4CF] = 0x007A, -- MATHEMATICAL SCRIPT SMALL Z
 [0x1D4D0] = 0x0061, -- MATHEMATICAL BOLD SCRIPT CAPITAL A
 [0x1D4D1] = 0x0062, -- MATHEMATICAL BOLD SCRIPT CAPITAL B
 [0x1D4D2] = 0x0063, -- MATHEMATICAL BOLD SCRIPT CAPITAL C
 [0x1D4D3] = 0x0064, -- MATHEMATICAL BOLD SCRIPT CAPITAL D
 [0x1D4D4] = 0x0065, -- MATHEMATICAL BOLD SCRIPT CAPITAL E
 [0x1D4D5] = 0x0066, -- MATHEMATICAL BOLD SCRIPT CAPITAL F
 [0x1D4D6] = 0x0067, -- MATHEMATICAL BOLD SCRIPT CAPITAL G
 [0x1D4D7] = 0x0068, -- MATHEMATICAL BOLD SCRIPT CAPITAL H
 [0x1D4D8] = 0x0069, -- MATHEMATICAL BOLD SCRIPT CAPITAL I
 [0x1D4D9] = 0x006A, -- MATHEMATICAL BOLD SCRIPT CAPITAL J
 [0x1D4DA] = 0x006B, -- MATHEMATICAL BOLD SCRIPT CAPITAL K
 [0x1D4DB] = 0x006C, -- MATHEMATICAL BOLD SCRIPT CAPITAL L
 [0x1D4DC] = 0x006D, -- MATHEMATICAL BOLD SCRIPT CAPITAL M
 [0x1D4DD] = 0x006E, -- MATHEMATICAL BOLD SCRIPT CAPITAL N
 [0x1D4DE] = 0x006F, -- MATHEMATICAL BOLD SCRIPT CAPITAL O
 [0x1D4DF] = 0x0070, -- MATHEMATICAL BOLD SCRIPT CAPITAL P
 [0x1D4E0] = 0x0071, -- MATHEMATICAL BOLD SCRIPT CAPITAL Q
 [0x1D4E1] = 0x0072, -- MATHEMATICAL BOLD SCRIPT CAPITAL R
 [0x1D4E2] = 0x0073, -- MATHEMATICAL BOLD SCRIPT CAPITAL S
 [0x1D4E3] = 0x0074, -- MATHEMATICAL BOLD SCRIPT CAPITAL T
 [0x1D4E4] = 0x0075, -- MATHEMATICAL BOLD SCRIPT CAPITAL U
 [0x1D4E5] = 0x0076, -- MATHEMATICAL BOLD SCRIPT CAPITAL V
 [0x1D4E6] = 0x0077, -- MATHEMATICAL BOLD SCRIPT CAPITAL W
 [0x1D4E7] = 0x0078, -- MATHEMATICAL BOLD SCRIPT CAPITAL X
 [0x1D4E8] = 0x0079, -- MATHEMATICAL BOLD SCRIPT CAPITAL Y
 [0x1D4E9] = 0x007A, -- MATHEMATICAL BOLD SCRIPT CAPITAL Z
 [0x1D4EA] = 0x0061, -- MATHEMATICAL BOLD SCRIPT SMALL A
 [0x1D4EB] = 0x0062, -- MATHEMATICAL BOLD SCRIPT SMALL B
 [0x1D4EC] = 0x0063, -- MATHEMATICAL BOLD SCRIPT SMALL C
 [0x1D4ED] = 0x0064, -- MATHEMATICAL BOLD SCRIPT SMALL D
 [0x1D4EE] = 0x0065, -- MATHEMATICAL BOLD SCRIPT SMALL E
 [0x1D4EF] = 0x0066, -- MATHEMATICAL BOLD SCRIPT SMALL F
 [0x1D4F0] = 0x0067, -- MATHEMATICAL BOLD SCRIPT SMALL G
 [0x1D4F1] = 0x0068, -- MATHEMATICAL BOLD SCRIPT SMALL H
 [0x1D4F2] = 0x0069, -- MATHEMATICAL BOLD SCRIPT SMALL I
 [0x1D4F3] = 0x006A, -- MATHEMATICAL BOLD SCRIPT SMALL J
 [0x1D4F4] = 0x006B, -- MATHEMATICAL BOLD SCRIPT SMALL K
 [0x1D4F5] = 0x006C, -- MATHEMATICAL BOLD SCRIPT SMALL L
 [0x1D4F6] = 0x006D, -- MATHEMATICAL BOLD SCRIPT SMALL M
 [0x1D4F7] = 0x006E, -- MATHEMATICAL BOLD SCRIPT SMALL N
 [0x1D4F8] = 0x006F, -- MATHEMATICAL BOLD SCRIPT SMALL O
 [0x1D4F9] = 0x0070, -- MATHEMATICAL BOLD SCRIPT SMALL P
 [0x1D4FA] = 0x0071, -- MATHEMATICAL BOLD SCRIPT SMALL Q
 [0x1D4FB] = 0x0072, -- MATHEMATICAL BOLD SCRIPT SMALL R
 [0x1D4FC] = 0x0073, -- MATHEMATICAL BOLD SCRIPT SMALL S
 [0x1D4FD] = 0x0074, -- MATHEMATICAL BOLD SCRIPT SMALL T
 [0x1D4FE] = 0x0075, -- MATHEMATICAL BOLD SCRIPT SMALL U
 [0x1D4FF] = 0x0076, -- MATHEMATICAL BOLD SCRIPT SMALL V
 [0x1D500] = 0x0077, -- MATHEMATICAL BOLD SCRIPT SMALL W
 [0x1D501] = 0x0078, -- MATHEMATICAL BOLD SCRIPT SMALL X
 [0x1D502] = 0x0079, -- MATHEMATICAL BOLD SCRIPT SMALL Y
 [0x1D503] = 0x007A, -- MATHEMATICAL BOLD SCRIPT SMALL Z
 [0x1D504] = 0x0061, -- MATHEMATICAL FRAKTUR CAPITAL A
 [0x1D505] = 0x0062, -- MATHEMATICAL FRAKTUR CAPITAL B
 [0x1D506] = {status="disallowed"}, -- NA   <reserved-1D506>
 [0x1D507] = 0x0064, -- MATHEMATICAL FRAKTUR CAPITAL D
 [0x1D508] = 0x0065, -- MATHEMATICAL FRAKTUR CAPITAL E
 [0x1D509] = 0x0066, -- MATHEMATICAL FRAKTUR CAPITAL F
 [0x1D50A] = 0x0067, -- MATHEMATICAL FRAKTUR CAPITAL G
 -- NA   <reserved-1D4A7>..<reserved-1D4A8>
 [0x1D4A7] = {status="disallowed"},
 [0x1D4A8] = {status="disallowed"},
 [0x1D50D] = 0x006A, -- MATHEMATICAL FRAKTUR CAPITAL J
 [0x1D50E] = 0x006B, -- MATHEMATICAL FRAKTUR CAPITAL K
 [0x1D50F] = 0x006C, -- MATHEMATICAL FRAKTUR CAPITAL L
 [0x1D510] = 0x006D, -- MATHEMATICAL FRAKTUR CAPITAL M
 [0x1D511] = 0x006E, -- MATHEMATICAL FRAKTUR CAPITAL N
 [0x1D512] = 0x006F, -- MATHEMATICAL FRAKTUR CAPITAL O
 [0x1D513] = 0x0070, -- MATHEMATICAL FRAKTUR CAPITAL P
 [0x1D514] = 0x0071, -- MATHEMATICAL FRAKTUR CAPITAL Q
 [0x1D515] = {status="disallowed"}, -- NA   <reserved-1D515>
 [0x1D516] = 0x0073, -- MATHEMATICAL FRAKTUR CAPITAL S
 [0x1D517] = 0x0074, -- MATHEMATICAL FRAKTUR CAPITAL T
 [0x1D518] = 0x0075, -- MATHEMATICAL FRAKTUR CAPITAL U
 [0x1D519] = 0x0076, -- MATHEMATICAL FRAKTUR CAPITAL V
 [0x1D51A] = 0x0077, -- MATHEMATICAL FRAKTUR CAPITAL W
 [0x1D51B] = 0x0078, -- MATHEMATICAL FRAKTUR CAPITAL X
 [0x1D51C] = 0x0079, -- MATHEMATICAL FRAKTUR CAPITAL Y
 [0x1D51D] = {status="disallowed"}, -- NA   <reserved-1D51D>
 [0x1D51E] = 0x0061, -- MATHEMATICAL FRAKTUR SMALL A
 [0x1D51F] = 0x0062, -- MATHEMATICAL FRAKTUR SMALL B
 [0x1D520] = 0x0063, -- MATHEMATICAL FRAKTUR SMALL C
 [0x1D521] = 0x0064, -- MATHEMATICAL FRAKTUR SMALL D
 [0x1D522] = 0x0065, -- MATHEMATICAL FRAKTUR SMALL E
 [0x1D523] = 0x0066, -- MATHEMATICAL FRAKTUR SMALL F
 [0x1D524] = 0x0067, -- MATHEMATICAL FRAKTUR SMALL G
 [0x1D525] = 0x0068, -- MATHEMATICAL FRAKTUR SMALL H
 [0x1D526] = 0x0069, -- MATHEMATICAL FRAKTUR SMALL I
 [0x1D527] = 0x006A, -- MATHEMATICAL FRAKTUR SMALL J
 [0x1D528] = 0x006B, -- MATHEMATICAL FRAKTUR SMALL K
 [0x1D529] = 0x006C, -- MATHEMATICAL FRAKTUR SMALL L
 [0x1D52A] = 0x006D, -- MATHEMATICAL FRAKTUR SMALL M
 [0x1D52B] = 0x006E, -- MATHEMATICAL FRAKTUR SMALL N
 [0x1D52C] = 0x006F, -- MATHEMATICAL FRAKTUR SMALL O
 [0x1D52D] = 0x0070, -- MATHEMATICAL FRAKTUR SMALL P
 [0x1D52E] = 0x0071, -- MATHEMATICAL FRAKTUR SMALL Q
 [0x1D52F] = 0x0072, -- MATHEMATICAL FRAKTUR SMALL R
 [0x1D530] = 0x0073, -- MATHEMATICAL FRAKTUR SMALL S
 [0x1D531] = 0x0074, -- MATHEMATICAL FRAKTUR SMALL T
 [0x1D532] = 0x0075, -- MATHEMATICAL FRAKTUR SMALL U
 [0x1D533] = 0x0076, -- MATHEMATICAL FRAKTUR SMALL V
 [0x1D534] = 0x0077, -- MATHEMATICAL FRAKTUR SMALL W
 [0x1D535] = 0x0078, -- MATHEMATICAL FRAKTUR SMALL X
 [0x1D536] = 0x0079, -- MATHEMATICAL FRAKTUR SMALL Y
 [0x1D537] = 0x007A, -- MATHEMATICAL FRAKTUR SMALL Z
 [0x1D538] = 0x0061, -- MATHEMATICAL DOUBLE-STRUCK CAPITAL A
 [0x1D539] = 0x0062, -- MATHEMATICAL DOUBLE-STRUCK CAPITAL B
 [0x1D53A] = {status="disallowed"}, -- NA   <reserved-1D53A>
 [0x1D53B] = 0x0064, -- MATHEMATICAL DOUBLE-STRUCK CAPITAL D
 [0x1D53C] = 0x0065, -- MATHEMATICAL DOUBLE-STRUCK CAPITAL E
 [0x1D53D] = 0x0066, -- MATHEMATICAL DOUBLE-STRUCK CAPITAL F
 [0x1D53E] = 0x0067, -- MATHEMATICAL DOUBLE-STRUCK CAPITAL G
 [0x1D53F] = {status="disallowed"}, -- NA   <reserved-1D53F>
 [0x1D540] = 0x0069, -- MATHEMATICAL DOUBLE-STRUCK CAPITAL I
 [0x1D541] = 0x006A, -- MATHEMATICAL DOUBLE-STRUCK CAPITAL J
 [0x1D542] = 0x006B, -- MATHEMATICAL DOUBLE-STRUCK CAPITAL K
 [0x1D543] = 0x006C, -- MATHEMATICAL DOUBLE-STRUCK CAPITAL L
 [0x1D544] = 0x006D, -- MATHEMATICAL DOUBLE-STRUCK CAPITAL M
 [0x1D545] = {status="disallowed"}, -- NA   <reserved-1D545>
 [0x1D546] = 0x006F, -- MATHEMATICAL DOUBLE-STRUCK CAPITAL O
 -- NA   <reserved-1D50B>..<reserved-1D50C>
 [0x1D50B] = {status="disallowed"},
 [0x1D50C] = {status="disallowed"},
 [0x1D54A] = 0x0073, -- MATHEMATICAL DOUBLE-STRUCK CAPITAL S
 [0x1D54B] = 0x0074, -- MATHEMATICAL DOUBLE-STRUCK CAPITAL T
 [0x1D54C] = 0x0075, -- MATHEMATICAL DOUBLE-STRUCK CAPITAL U
 [0x1D54D] = 0x0076, -- MATHEMATICAL DOUBLE-STRUCK CAPITAL V
 [0x1D54E] = 0x0077, -- MATHEMATICAL DOUBLE-STRUCK CAPITAL W
 [0x1D54F] = 0x0078, -- MATHEMATICAL DOUBLE-STRUCK CAPITAL X
 [0x1D550] = 0x0079, -- MATHEMATICAL DOUBLE-STRUCK CAPITAL Y
 [0x1D551] = {status="disallowed"}, -- NA   <reserved-1D551>
 [0x1D552] = 0x0061, -- MATHEMATICAL DOUBLE-STRUCK SMALL A
 [0x1D553] = 0x0062, -- MATHEMATICAL DOUBLE-STRUCK SMALL B
 [0x1D554] = 0x0063, -- MATHEMATICAL DOUBLE-STRUCK SMALL C
 [0x1D555] = 0x0064, -- MATHEMATICAL DOUBLE-STRUCK SMALL D
 [0x1D556] = 0x0065, -- MATHEMATICAL DOUBLE-STRUCK SMALL E
 [0x1D557] = 0x0066, -- MATHEMATICAL DOUBLE-STRUCK SMALL F
 [0x1D558] = 0x0067, -- MATHEMATICAL DOUBLE-STRUCK SMALL G
 [0x1D559] = 0x0068, -- MATHEMATICAL DOUBLE-STRUCK SMALL H
 [0x1D55A] = 0x0069, -- MATHEMATICAL DOUBLE-STRUCK SMALL I
 [0x1D55B] = 0x006A, -- MATHEMATICAL DOUBLE-STRUCK SMALL J
 [0x1D55C] = 0x006B, -- MATHEMATICAL DOUBLE-STRUCK SMALL K
 [0x1D55D] = 0x006C, -- MATHEMATICAL DOUBLE-STRUCK SMALL L
 [0x1D55E] = 0x006D, -- MATHEMATICAL DOUBLE-STRUCK SMALL M
 [0x1D55F] = 0x006E, -- MATHEMATICAL DOUBLE-STRUCK SMALL N
 [0x1D560] = 0x006F, -- MATHEMATICAL DOUBLE-STRUCK SMALL O
 [0x1D561] = 0x0070, -- MATHEMATICAL DOUBLE-STRUCK SMALL P
 [0x1D562] = 0x0071, -- MATHEMATICAL DOUBLE-STRUCK SMALL Q
 [0x1D563] = 0x0072, -- MATHEMATICAL DOUBLE-STRUCK SMALL R
 [0x1D564] = 0x0073, -- MATHEMATICAL DOUBLE-STRUCK SMALL S
 [0x1D565] = 0x0074, -- MATHEMATICAL DOUBLE-STRUCK SMALL T
 [0x1D566] = 0x0075, -- MATHEMATICAL DOUBLE-STRUCK SMALL U
 [0x1D567] = 0x0076, -- MATHEMATICAL DOUBLE-STRUCK SMALL V
 [0x1D568] = 0x0077, -- MATHEMATICAL DOUBLE-STRUCK SMALL W
 [0x1D569] = 0x0078, -- MATHEMATICAL DOUBLE-STRUCK SMALL X
 [0x1D56A] = 0x0079, -- MATHEMATICAL DOUBLE-STRUCK SMALL Y
 [0x1D56B] = 0x007A, -- MATHEMATICAL DOUBLE-STRUCK SMALL Z
 [0x1D56C] = 0x0061, -- MATHEMATICAL BOLD FRAKTUR CAPITAL A
 [0x1D56D] = 0x0062, -- MATHEMATICAL BOLD FRAKTUR CAPITAL B
 [0x1D56E] = 0x0063, -- MATHEMATICAL BOLD FRAKTUR CAPITAL C
 [0x1D56F] = 0x0064, -- MATHEMATICAL BOLD FRAKTUR CAPITAL D
 [0x1D570] = 0x0065, -- MATHEMATICAL BOLD FRAKTUR CAPITAL E
 [0x1D571] = 0x0066, -- MATHEMATICAL BOLD FRAKTUR CAPITAL F
 [0x1D572] = 0x0067, -- MATHEMATICAL BOLD FRAKTUR CAPITAL G
 [0x1D573] = 0x0068, -- MATHEMATICAL BOLD FRAKTUR CAPITAL H
 [0x1D574] = 0x0069, -- MATHEMATICAL BOLD FRAKTUR CAPITAL I
 [0x1D575] = 0x006A, -- MATHEMATICAL BOLD FRAKTUR CAPITAL J
 [0x1D576] = 0x006B, -- MATHEMATICAL BOLD FRAKTUR CAPITAL K
 [0x1D577] = 0x006C, -- MATHEMATICAL BOLD FRAKTUR CAPITAL L
 [0x1D578] = 0x006D, -- MATHEMATICAL BOLD FRAKTUR CAPITAL M
 [0x1D579] = 0x006E, -- MATHEMATICAL BOLD FRAKTUR CAPITAL N
 [0x1D57A] = 0x006F, -- MATHEMATICAL BOLD FRAKTUR CAPITAL O
 [0x1D57B] = 0x0070, -- MATHEMATICAL BOLD FRAKTUR CAPITAL P
 [0x1D57C] = 0x0071, -- MATHEMATICAL BOLD FRAKTUR CAPITAL Q
 [0x1D57D] = 0x0072, -- MATHEMATICAL BOLD FRAKTUR CAPITAL R
 [0x1D57E] = 0x0073, -- MATHEMATICAL BOLD FRAKTUR CAPITAL S
 [0x1D57F] = 0x0074, -- MATHEMATICAL BOLD FRAKTUR CAPITAL T
 [0x1D580] = 0x0075, -- MATHEMATICAL BOLD FRAKTUR CAPITAL U
 [0x1D581] = 0x0076, -- MATHEMATICAL BOLD FRAKTUR CAPITAL V
 [0x1D582] = 0x0077, -- MATHEMATICAL BOLD FRAKTUR CAPITAL W
 [0x1D583] = 0x0078, -- MATHEMATICAL BOLD FRAKTUR CAPITAL X
 [0x1D584] = 0x0079, -- MATHEMATICAL BOLD FRAKTUR CAPITAL Y
 [0x1D585] = 0x007A, -- MATHEMATICAL BOLD FRAKTUR CAPITAL Z
 [0x1D586] = 0x0061, -- MATHEMATICAL BOLD FRAKTUR SMALL A
 [0x1D587] = 0x0062, -- MATHEMATICAL BOLD FRAKTUR SMALL B
 [0x1D588] = 0x0063, -- MATHEMATICAL BOLD FRAKTUR SMALL C
 [0x1D589] = 0x0064, -- MATHEMATICAL BOLD FRAKTUR SMALL D
 [0x1D58A] = 0x0065, -- MATHEMATICAL BOLD FRAKTUR SMALL E
 [0x1D58B] = 0x0066, -- MATHEMATICAL BOLD FRAKTUR SMALL F
 [0x1D58C] = 0x0067, -- MATHEMATICAL BOLD FRAKTUR SMALL G
 [0x1D58D] = 0x0068, -- MATHEMATICAL BOLD FRAKTUR SMALL H
 [0x1D58E] = 0x0069, -- MATHEMATICAL BOLD FRAKTUR SMALL I
 [0x1D58F] = 0x006A, -- MATHEMATICAL BOLD FRAKTUR SMALL J
 [0x1D590] = 0x006B, -- MATHEMATICAL BOLD FRAKTUR SMALL K
 [0x1D591] = 0x006C, -- MATHEMATICAL BOLD FRAKTUR SMALL L
 [0x1D592] = 0x006D, -- MATHEMATICAL BOLD FRAKTUR SMALL M
 [0x1D593] = 0x006E, -- MATHEMATICAL BOLD FRAKTUR SMALL N
 [0x1D594] = 0x006F, -- MATHEMATICAL BOLD FRAKTUR SMALL O
 [0x1D595] = 0x0070, -- MATHEMATICAL BOLD FRAKTUR SMALL P
 [0x1D596] = 0x0071, -- MATHEMATICAL BOLD FRAKTUR SMALL Q
 [0x1D597] = 0x0072, -- MATHEMATICAL BOLD FRAKTUR SMALL R
 [0x1D598] = 0x0073, -- MATHEMATICAL BOLD FRAKTUR SMALL S
 [0x1D599] = 0x0074, -- MATHEMATICAL BOLD FRAKTUR SMALL T
 [0x1D59A] = 0x0075, -- MATHEMATICAL BOLD FRAKTUR SMALL U
 [0x1D59B] = 0x0076, -- MATHEMATICAL BOLD FRAKTUR SMALL V
 [0x1D59C] = 0x0077, -- MATHEMATICAL BOLD FRAKTUR SMALL W
 [0x1D59D] = 0x0078, -- MATHEMATICAL BOLD FRAKTUR SMALL X
 [0x1D59E] = 0x0079, -- MATHEMATICAL BOLD FRAKTUR SMALL Y
 [0x1D59F] = 0x007A, -- MATHEMATICAL BOLD FRAKTUR SMALL Z
 [0x1D5A0] = 0x0061, -- MATHEMATICAL SANS-SERIF CAPITAL A
 [0x1D5A1] = 0x0062, -- MATHEMATICAL SANS-SERIF CAPITAL B
 [0x1D5A2] = 0x0063, -- MATHEMATICAL SANS-SERIF CAPITAL C
 [0x1D5A3] = 0x0064, -- MATHEMATICAL SANS-SERIF CAPITAL D
 [0x1D5A4] = 0x0065, -- MATHEMATICAL SANS-SERIF CAPITAL E
 [0x1D5A5] = 0x0066, -- MATHEMATICAL SANS-SERIF CAPITAL F
 [0x1D5A6] = 0x0067, -- MATHEMATICAL SANS-SERIF CAPITAL G
 [0x1D5A7] = 0x0068, -- MATHEMATICAL SANS-SERIF CAPITAL H
 [0x1D5A8] = 0x0069, -- MATHEMATICAL SANS-SERIF CAPITAL I
 [0x1D5A9] = 0x006A, -- MATHEMATICAL SANS-SERIF CAPITAL J
 [0x1D5AA] = 0x006B, -- MATHEMATICAL SANS-SERIF CAPITAL K
 [0x1D5AB] = 0x006C, -- MATHEMATICAL SANS-SERIF CAPITAL L
 [0x1D5AC] = 0x006D, -- MATHEMATICAL SANS-SERIF CAPITAL M
 [0x1D5AD] = 0x006E, -- MATHEMATICAL SANS-SERIF CAPITAL N
 [0x1D5AE] = 0x006F, -- MATHEMATICAL SANS-SERIF CAPITAL O
 [0x1D5AF] = 0x0070, -- MATHEMATICAL SANS-SERIF CAPITAL P
 [0x1D5B0] = 0x0071, -- MATHEMATICAL SANS-SERIF CAPITAL Q
 [0x1D5B1] = 0x0072, -- MATHEMATICAL SANS-SERIF CAPITAL R
 [0x1D5B2] = 0x0073, -- MATHEMATICAL SANS-SERIF CAPITAL S
 [0x1D5B3] = 0x0074, -- MATHEMATICAL SANS-SERIF CAPITAL T
 [0x1D5B4] = 0x0075, -- MATHEMATICAL SANS-SERIF CAPITAL U
 [0x1D5B5] = 0x0076, -- MATHEMATICAL SANS-SERIF CAPITAL V
 [0x1D5B6] = 0x0077, -- MATHEMATICAL SANS-SERIF CAPITAL W
 [0x1D5B7] = 0x0078, -- MATHEMATICAL SANS-SERIF CAPITAL X
 [0x1D5B8] = 0x0079, -- MATHEMATICAL SANS-SERIF CAPITAL Y
 [0x1D5B9] = 0x007A, -- MATHEMATICAL SANS-SERIF CAPITAL Z
 [0x1D5BA] = 0x0061, -- MATHEMATICAL SANS-SERIF SMALL A
 [0x1D5BB] = 0x0062, -- MATHEMATICAL SANS-SERIF SMALL B
 [0x1D5BC] = 0x0063, -- MATHEMATICAL SANS-SERIF SMALL C
 [0x1D5BD] = 0x0064, -- MATHEMATICAL SANS-SERIF SMALL D
 [0x1D5BE] = 0x0065, -- MATHEMATICAL SANS-SERIF SMALL E
 [0x1D5BF] = 0x0066, -- MATHEMATICAL SANS-SERIF SMALL F
 [0x1D5C0] = 0x0067, -- MATHEMATICAL SANS-SERIF SMALL G
 [0x1D5C1] = 0x0068, -- MATHEMATICAL SANS-SERIF SMALL H
 [0x1D5C2] = 0x0069, -- MATHEMATICAL SANS-SERIF SMALL I
 [0x1D5C3] = 0x006A, -- MATHEMATICAL SANS-SERIF SMALL J
 [0x1D5C4] = 0x006B, -- MATHEMATICAL SANS-SERIF SMALL K
 [0x1D5C5] = 0x006C, -- MATHEMATICAL SANS-SERIF SMALL L
 [0x1D5C6] = 0x006D, -- MATHEMATICAL SANS-SERIF SMALL M
 [0x1D5C7] = 0x006E, -- MATHEMATICAL SANS-SERIF SMALL N
 [0x1D5C8] = 0x006F, -- MATHEMATICAL SANS-SERIF SMALL O
 [0x1D5C9] = 0x0070, -- MATHEMATICAL SANS-SERIF SMALL P
 [0x1D5CA] = 0x0071, -- MATHEMATICAL SANS-SERIF SMALL Q
 [0x1D5CB] = 0x0072, -- MATHEMATICAL SANS-SERIF SMALL R
 [0x1D5CC] = 0x0073, -- MATHEMATICAL SANS-SERIF SMALL S
 [0x1D5CD] = 0x0074, -- MATHEMATICAL SANS-SERIF SMALL T
 [0x1D5CE] = 0x0075, -- MATHEMATICAL SANS-SERIF SMALL U
 [0x1D5CF] = 0x0076, -- MATHEMATICAL SANS-SERIF SMALL V
 [0x1D5D0] = 0x0077, -- MATHEMATICAL SANS-SERIF SMALL W
 [0x1D5D1] = 0x0078, -- MATHEMATICAL SANS-SERIF SMALL X
 [0x1D5D2] = 0x0079, -- MATHEMATICAL SANS-SERIF SMALL Y
 [0x1D5D3] = 0x007A, -- MATHEMATICAL SANS-SERIF SMALL Z
 [0x1D5D4] = 0x0061, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL A
 [0x1D5D5] = 0x0062, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL B
 [0x1D5D6] = 0x0063, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL C
 [0x1D5D7] = 0x0064, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL D
 [0x1D5D8] = 0x0065, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL E
 [0x1D5D9] = 0x0066, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL F
 [0x1D5DA] = 0x0067, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL G
 [0x1D5DB] = 0x0068, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL H
 [0x1D5DC] = 0x0069, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL I
 [0x1D5DD] = 0x006A, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL J
 [0x1D5DE] = 0x006B, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL K
 [0x1D5DF] = 0x006C, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL L
 [0x1D5E0] = 0x006D, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL M
 [0x1D5E1] = 0x006E, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL N
 [0x1D5E2] = 0x006F, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL O
 [0x1D5E3] = 0x0070, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL P
 [0x1D5E4] = 0x0071, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL Q
 [0x1D5E5] = 0x0072, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL R
 [0x1D5E6] = 0x0073, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL S
 [0x1D5E7] = 0x0074, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL T
 [0x1D5E8] = 0x0075, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL U
 [0x1D5E9] = 0x0076, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL V
 [0x1D5EA] = 0x0077, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL W
 [0x1D5EB] = 0x0078, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL X
 [0x1D5EC] = 0x0079, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL Y
 [0x1D5ED] = 0x007A, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL Z
 [0x1D5EE] = 0x0061, -- MATHEMATICAL SANS-SERIF BOLD SMALL A
 [0x1D5EF] = 0x0062, -- MATHEMATICAL SANS-SERIF BOLD SMALL B
 [0x1D5F0] = 0x0063, -- MATHEMATICAL SANS-SERIF BOLD SMALL C
 [0x1D5F1] = 0x0064, -- MATHEMATICAL SANS-SERIF BOLD SMALL D
 [0x1D5F2] = 0x0065, -- MATHEMATICAL SANS-SERIF BOLD SMALL E
 [0x1D5F3] = 0x0066, -- MATHEMATICAL SANS-SERIF BOLD SMALL F
 [0x1D5F4] = 0x0067, -- MATHEMATICAL SANS-SERIF BOLD SMALL G
 [0x1D5F5] = 0x0068, -- MATHEMATICAL SANS-SERIF BOLD SMALL H
 [0x1D5F6] = 0x0069, -- MATHEMATICAL SANS-SERIF BOLD SMALL I
 [0x1D5F7] = 0x006A, -- MATHEMATICAL SANS-SERIF BOLD SMALL J
 [0x1D5F8] = 0x006B, -- MATHEMATICAL SANS-SERIF BOLD SMALL K
 [0x1D5F9] = 0x006C, -- MATHEMATICAL SANS-SERIF BOLD SMALL L
 [0x1D5FA] = 0x006D, -- MATHEMATICAL SANS-SERIF BOLD SMALL M
 [0x1D5FB] = 0x006E, -- MATHEMATICAL SANS-SERIF BOLD SMALL N
 [0x1D5FC] = 0x006F, -- MATHEMATICAL SANS-SERIF BOLD SMALL O
 [0x1D5FD] = 0x0070, -- MATHEMATICAL SANS-SERIF BOLD SMALL P
 [0x1D5FE] = 0x0071, -- MATHEMATICAL SANS-SERIF BOLD SMALL Q
 [0x1D5FF] = 0x0072, -- MATHEMATICAL SANS-SERIF BOLD SMALL R
 [0x1D600] = 0x0073, -- MATHEMATICAL SANS-SERIF BOLD SMALL S
 [0x1D601] = 0x0074, -- MATHEMATICAL SANS-SERIF BOLD SMALL T
 [0x1D602] = 0x0075, -- MATHEMATICAL SANS-SERIF BOLD SMALL U
 [0x1D603] = 0x0076, -- MATHEMATICAL SANS-SERIF BOLD SMALL V
 [0x1D604] = 0x0077, -- MATHEMATICAL SANS-SERIF BOLD SMALL W
 [0x1D605] = 0x0078, -- MATHEMATICAL SANS-SERIF BOLD SMALL X
 [0x1D606] = 0x0079, -- MATHEMATICAL SANS-SERIF BOLD SMALL Y
 [0x1D607] = 0x007A, -- MATHEMATICAL SANS-SERIF BOLD SMALL Z
 [0x1D608] = 0x0061, -- MATHEMATICAL SANS-SERIF ITALIC CAPITAL A
 [0x1D609] = 0x0062, -- MATHEMATICAL SANS-SERIF ITALIC CAPITAL B
 [0x1D60A] = 0x0063, -- MATHEMATICAL SANS-SERIF ITALIC CAPITAL C
 [0x1D60B] = 0x0064, -- MATHEMATICAL SANS-SERIF ITALIC CAPITAL D
 [0x1D60C] = 0x0065, -- MATHEMATICAL SANS-SERIF ITALIC CAPITAL E
 [0x1D60D] = 0x0066, -- MATHEMATICAL SANS-SERIF ITALIC CAPITAL F
 [0x1D60E] = 0x0067, -- MATHEMATICAL SANS-SERIF ITALIC CAPITAL G
 [0x1D60F] = 0x0068, -- MATHEMATICAL SANS-SERIF ITALIC CAPITAL H
 [0x1D610] = 0x0069, -- MATHEMATICAL SANS-SERIF ITALIC CAPITAL I
 [0x1D611] = 0x006A, -- MATHEMATICAL SANS-SERIF ITALIC CAPITAL J
 [0x1D612] = 0x006B, -- MATHEMATICAL SANS-SERIF ITALIC CAPITAL K
 [0x1D613] = 0x006C, -- MATHEMATICAL SANS-SERIF ITALIC CAPITAL L
 [0x1D614] = 0x006D, -- MATHEMATICAL SANS-SERIF ITALIC CAPITAL M
 [0x1D615] = 0x006E, -- MATHEMATICAL SANS-SERIF ITALIC CAPITAL N
 [0x1D616] = 0x006F, -- MATHEMATICAL SANS-SERIF ITALIC CAPITAL O
 [0x1D617] = 0x0070, -- MATHEMATICAL SANS-SERIF ITALIC CAPITAL P
 [0x1D618] = 0x0071, -- MATHEMATICAL SANS-SERIF ITALIC CAPITAL Q
 [0x1D619] = 0x0072, -- MATHEMATICAL SANS-SERIF ITALIC CAPITAL R
 [0x1D61A] = 0x0073, -- MATHEMATICAL SANS-SERIF ITALIC CAPITAL S
 [0x1D61B] = 0x0074, -- MATHEMATICAL SANS-SERIF ITALIC CAPITAL T
 [0x1D61C] = 0x0075, -- MATHEMATICAL SANS-SERIF ITALIC CAPITAL U
 [0x1D61D] = 0x0076, -- MATHEMATICAL SANS-SERIF ITALIC CAPITAL V
 [0x1D61E] = 0x0077, -- MATHEMATICAL SANS-SERIF ITALIC CAPITAL W
 [0x1D61F] = 0x0078, -- MATHEMATICAL SANS-SERIF ITALIC CAPITAL X
 [0x1D620] = 0x0079, -- MATHEMATICAL SANS-SERIF ITALIC CAPITAL Y
 [0x1D621] = 0x007A, -- MATHEMATICAL SANS-SERIF ITALIC CAPITAL Z
 [0x1D622] = 0x0061, -- MATHEMATICAL SANS-SERIF ITALIC SMALL A
 [0x1D623] = 0x0062, -- MATHEMATICAL SANS-SERIF ITALIC SMALL B
 [0x1D624] = 0x0063, -- MATHEMATICAL SANS-SERIF ITALIC SMALL C
 [0x1D625] = 0x0064, -- MATHEMATICAL SANS-SERIF ITALIC SMALL D
 [0x1D626] = 0x0065, -- MATHEMATICAL SANS-SERIF ITALIC SMALL E
 [0x1D627] = 0x0066, -- MATHEMATICAL SANS-SERIF ITALIC SMALL F
 [0x1D628] = 0x0067, -- MATHEMATICAL SANS-SERIF ITALIC SMALL G
 [0x1D629] = 0x0068, -- MATHEMATICAL SANS-SERIF ITALIC SMALL H
 [0x1D62A] = 0x0069, -- MATHEMATICAL SANS-SERIF ITALIC SMALL I
 [0x1D62B] = 0x006A, -- MATHEMATICAL SANS-SERIF ITALIC SMALL J
 [0x1D62C] = 0x006B, -- MATHEMATICAL SANS-SERIF ITALIC SMALL K
 [0x1D62D] = 0x006C, -- MATHEMATICAL SANS-SERIF ITALIC SMALL L
 [0x1D62E] = 0x006D, -- MATHEMATICAL SANS-SERIF ITALIC SMALL M
 [0x1D62F] = 0x006E, -- MATHEMATICAL SANS-SERIF ITALIC SMALL N
 [0x1D630] = 0x006F, -- MATHEMATICAL SANS-SERIF ITALIC SMALL O
 [0x1D631] = 0x0070, -- MATHEMATICAL SANS-SERIF ITALIC SMALL P
 [0x1D632] = 0x0071, -- MATHEMATICAL SANS-SERIF ITALIC SMALL Q
 [0x1D633] = 0x0072, -- MATHEMATICAL SANS-SERIF ITALIC SMALL R
 [0x1D634] = 0x0073, -- MATHEMATICAL SANS-SERIF ITALIC SMALL S
 [0x1D635] = 0x0074, -- MATHEMATICAL SANS-SERIF ITALIC SMALL T
 [0x1D636] = 0x0075, -- MATHEMATICAL SANS-SERIF ITALIC SMALL U
 [0x1D637] = 0x0076, -- MATHEMATICAL SANS-SERIF ITALIC SMALL V
 [0x1D638] = 0x0077, -- MATHEMATICAL SANS-SERIF ITALIC SMALL W
 [0x1D639] = 0x0078, -- MATHEMATICAL SANS-SERIF ITALIC SMALL X
 [0x1D63A] = 0x0079, -- MATHEMATICAL SANS-SERIF ITALIC SMALL Y
 [0x1D63B] = 0x007A, -- MATHEMATICAL SANS-SERIF ITALIC SMALL Z
 [0x1D63C] = 0x0061, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL A
 [0x1D63D] = 0x0062, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL B
 [0x1D63E] = 0x0063, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL C
 [0x1D63F] = 0x0064, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL D
 [0x1D640] = 0x0065, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL E
 [0x1D641] = 0x0066, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL F
 [0x1D642] = 0x0067, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL G
 [0x1D643] = 0x0068, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL H
 [0x1D644] = 0x0069, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL I
 [0x1D645] = 0x006A, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL J
 [0x1D646] = 0x006B, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL K
 [0x1D647] = 0x006C, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL L
 [0x1D648] = 0x006D, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL M
 [0x1D649] = 0x006E, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL N
 [0x1D64A] = 0x006F, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL O
 [0x1D64B] = 0x0070, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL P
 [0x1D64C] = 0x0071, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL Q
 [0x1D64D] = 0x0072, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL R
 [0x1D64E] = 0x0073, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL S
 [0x1D64F] = 0x0074, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL T
 [0x1D650] = 0x0075, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL U
 [0x1D651] = 0x0076, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL V
 [0x1D652] = 0x0077, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL W
 [0x1D653] = 0x0078, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL X
 [0x1D654] = 0x0079, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL Y
 [0x1D655] = 0x007A, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL Z
 [0x1D656] = 0x0061, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL A
 [0x1D657] = 0x0062, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL B
 [0x1D658] = 0x0063, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL C
 [0x1D659] = 0x0064, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL D
 [0x1D65A] = 0x0065, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL E
 [0x1D65B] = 0x0066, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL F
 [0x1D65C] = 0x0067, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL G
 [0x1D65D] = 0x0068, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL H
 [0x1D65E] = 0x0069, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL I
 [0x1D65F] = 0x006A, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL J
 [0x1D660] = 0x006B, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL K
 [0x1D661] = 0x006C, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL L
 [0x1D662] = 0x006D, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL M
 [0x1D663] = 0x006E, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL N
 [0x1D664] = 0x006F, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL O
 [0x1D665] = 0x0070, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL P
 [0x1D666] = 0x0071, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL Q
 [0x1D667] = 0x0072, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL R
 [0x1D668] = 0x0073, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL S
 [0x1D669] = 0x0074, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL T
 [0x1D66A] = 0x0075, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL U
 [0x1D66B] = 0x0076, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL V
 [0x1D66C] = 0x0077, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL W
 [0x1D66D] = 0x0078, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL X
 [0x1D66E] = 0x0079, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL Y
 [0x1D66F] = 0x007A, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL Z
 [0x1D670] = 0x0061, -- MATHEMATICAL MONOSPACE CAPITAL A
 [0x1D671] = 0x0062, -- MATHEMATICAL MONOSPACE CAPITAL B
 [0x1D672] = 0x0063, -- MATHEMATICAL MONOSPACE CAPITAL C
 [0x1D673] = 0x0064, -- MATHEMATICAL MONOSPACE CAPITAL D
 [0x1D674] = 0x0065, -- MATHEMATICAL MONOSPACE CAPITAL E
 [0x1D675] = 0x0066, -- MATHEMATICAL MONOSPACE CAPITAL F
 [0x1D676] = 0x0067, -- MATHEMATICAL MONOSPACE CAPITAL G
 [0x1D677] = 0x0068, -- MATHEMATICAL MONOSPACE CAPITAL H
 [0x1D678] = 0x0069, -- MATHEMATICAL MONOSPACE CAPITAL I
 [0x1D679] = 0x006A, -- MATHEMATICAL MONOSPACE CAPITAL J
 [0x1D67A] = 0x006B, -- MATHEMATICAL MONOSPACE CAPITAL K
 [0x1D67B] = 0x006C, -- MATHEMATICAL MONOSPACE CAPITAL L
 [0x1D67C] = 0x006D, -- MATHEMATICAL MONOSPACE CAPITAL M
 [0x1D67D] = 0x006E, -- MATHEMATICAL MONOSPACE CAPITAL N
 [0x1D67E] = 0x006F, -- MATHEMATICAL MONOSPACE CAPITAL O
 [0x1D67F] = 0x0070, -- MATHEMATICAL MONOSPACE CAPITAL P
 [0x1D680] = 0x0071, -- MATHEMATICAL MONOSPACE CAPITAL Q
 [0x1D681] = 0x0072, -- MATHEMATICAL MONOSPACE CAPITAL R
 [0x1D682] = 0x0073, -- MATHEMATICAL MONOSPACE CAPITAL S
 [0x1D683] = 0x0074, -- MATHEMATICAL MONOSPACE CAPITAL T
 [0x1D684] = 0x0075, -- MATHEMATICAL MONOSPACE CAPITAL U
 [0x1D685] = 0x0076, -- MATHEMATICAL MONOSPACE CAPITAL V
 [0x1D686] = 0x0077, -- MATHEMATICAL MONOSPACE CAPITAL W
 [0x1D687] = 0x0078, -- MATHEMATICAL MONOSPACE CAPITAL X
 [0x1D688] = 0x0079, -- MATHEMATICAL MONOSPACE CAPITAL Y
 [0x1D689] = 0x007A, -- MATHEMATICAL MONOSPACE CAPITAL Z
 [0x1D68A] = 0x0061, -- MATHEMATICAL MONOSPACE SMALL A
 [0x1D68B] = 0x0062, -- MATHEMATICAL MONOSPACE SMALL B
 [0x1D68C] = 0x0063, -- MATHEMATICAL MONOSPACE SMALL C
 [0x1D68D] = 0x0064, -- MATHEMATICAL MONOSPACE SMALL D
 [0x1D68E] = 0x0065, -- MATHEMATICAL MONOSPACE SMALL E
 [0x1D68F] = 0x0066, -- MATHEMATICAL MONOSPACE SMALL F
 [0x1D690] = 0x0067, -- MATHEMATICAL MONOSPACE SMALL G
 [0x1D691] = 0x0068, -- MATHEMATICAL MONOSPACE SMALL H
 [0x1D692] = 0x0069, -- MATHEMATICAL MONOSPACE SMALL I
 [0x1D693] = 0x006A, -- MATHEMATICAL MONOSPACE SMALL J
 [0x1D694] = 0x006B, -- MATHEMATICAL MONOSPACE SMALL K
 [0x1D695] = 0x006C, -- MATHEMATICAL MONOSPACE SMALL L
 [0x1D696] = 0x006D, -- MATHEMATICAL MONOSPACE SMALL M
 [0x1D697] = 0x006E, -- MATHEMATICAL MONOSPACE SMALL N
 [0x1D698] = 0x006F, -- MATHEMATICAL MONOSPACE SMALL O
 [0x1D699] = 0x0070, -- MATHEMATICAL MONOSPACE SMALL P
 [0x1D69A] = 0x0071, -- MATHEMATICAL MONOSPACE SMALL Q
 [0x1D69B] = 0x0072, -- MATHEMATICAL MONOSPACE SMALL R
 [0x1D69C] = 0x0073, -- MATHEMATICAL MONOSPACE SMALL S
 [0x1D69D] = 0x0074, -- MATHEMATICAL MONOSPACE SMALL T
 [0x1D69E] = 0x0075, -- MATHEMATICAL MONOSPACE SMALL U
 [0x1D69F] = 0x0076, -- MATHEMATICAL MONOSPACE SMALL V
 [0x1D6A0] = 0x0077, -- MATHEMATICAL MONOSPACE SMALL W
 [0x1D6A1] = 0x0078, -- MATHEMATICAL MONOSPACE SMALL X
 [0x1D6A2] = 0x0079, -- MATHEMATICAL MONOSPACE SMALL Y
 [0x1D6A3] = 0x007A, -- MATHEMATICAL MONOSPACE SMALL Z
 [0x1D6A4] = 0x0131, -- MATHEMATICAL ITALIC SMALL DOTLESS I
 [0x1D6A5] = 0x0237, -- MATHEMATICAL ITALIC SMALL DOTLESS J
 -- NA   <reserved-1D547>..<reserved-1D549>
 [0x1D547] = {status="disallowed"},
 [0x1D548] = {status="disallowed"},
 [0x1D549] = {status="disallowed"},
 [0x1D6A8] = 0x03B1, -- MATHEMATICAL BOLD CAPITAL ALPHA
 [0x1D6A9] = 0x03B2, -- MATHEMATICAL BOLD CAPITAL BETA
 [0x1D6AA] = 0x03B3, -- MATHEMATICAL BOLD CAPITAL GAMMA
 [0x1D6AB] = 0x03B4, -- MATHEMATICAL BOLD CAPITAL DELTA
 [0x1D6AC] = 0x03B5, -- MATHEMATICAL BOLD CAPITAL EPSILON
 [0x1D6AD] = 0x03B6, -- MATHEMATICAL BOLD CAPITAL ZETA
 [0x1D6AE] = 0x03B7, -- MATHEMATICAL BOLD CAPITAL ETA
 [0x1D6AF] = 0x03B8, -- MATHEMATICAL BOLD CAPITAL THETA
 [0x1D6B0] = 0x03B9, -- MATHEMATICAL BOLD CAPITAL IOTA
 [0x1D6B1] = 0x03BA, -- MATHEMATICAL BOLD CAPITAL KAPPA
 [0x1D6B2] = 0x03BB, -- MATHEMATICAL BOLD CAPITAL LAMDA
 [0x1D6B3] = 0x03BC, -- MATHEMATICAL BOLD CAPITAL MU
 [0x1D6B4] = 0x03BD, -- MATHEMATICAL BOLD CAPITAL NU
 [0x1D6B5] = 0x03BE, -- MATHEMATICAL BOLD CAPITAL XI
 [0x1D6B6] = 0x03BF, -- MATHEMATICAL BOLD CAPITAL OMICRON
 [0x1D6B7] = 0x03C0, -- MATHEMATICAL BOLD CAPITAL PI
 [0x1D6B8] = 0x03C1, -- MATHEMATICAL BOLD CAPITAL RHO
 [0x1D6B9] = 0x03B8, -- MATHEMATICAL BOLD CAPITAL THETA SYMBOL
 [0x1D6BA] = 0x03C3, -- MATHEMATICAL BOLD CAPITAL SIGMA
 [0x1D6BB] = 0x03C4, -- MATHEMATICAL BOLD CAPITAL TAU
 [0x1D6BC] = 0x03C5, -- MATHEMATICAL BOLD CAPITAL UPSILON
 [0x1D6BD] = 0x03C6, -- MATHEMATICAL BOLD CAPITAL PHI
 [0x1D6BE] = 0x03C7, -- MATHEMATICAL BOLD CAPITAL CHI
 [0x1D6BF] = 0x03C8, -- MATHEMATICAL BOLD CAPITAL PSI
 [0x1D6C0] = 0x03C9, -- MATHEMATICAL BOLD CAPITAL OMEGA
 [0x1D6C1] = 0x2207, -- MATHEMATICAL BOLD NABLA
 [0x1D6C2] = 0x03B1, -- MATHEMATICAL BOLD SMALL ALPHA
 [0x1D6C3] = 0x03B2, -- MATHEMATICAL BOLD SMALL BETA
 [0x1D6C4] = 0x03B3, -- MATHEMATICAL BOLD SMALL GAMMA
 [0x1D6C5] = 0x03B4, -- MATHEMATICAL BOLD SMALL DELTA
 [0x1D6C6] = 0x03B5, -- MATHEMATICAL BOLD SMALL EPSILON
 [0x1D6C7] = 0x03B6, -- MATHEMATICAL BOLD SMALL ZETA
 [0x1D6C8] = 0x03B7, -- MATHEMATICAL BOLD SMALL ETA
 [0x1D6C9] = 0x03B8, -- MATHEMATICAL BOLD SMALL THETA
 [0x1D6CA] = 0x03B9, -- MATHEMATICAL BOLD SMALL IOTA
 [0x1D6CB] = 0x03BA, -- MATHEMATICAL BOLD SMALL KAPPA
 [0x1D6CC] = 0x03BB, -- MATHEMATICAL BOLD SMALL LAMDA
 [0x1D6CD] = 0x03BC, -- MATHEMATICAL BOLD SMALL MU
 [0x1D6CE] = 0x03BD, -- MATHEMATICAL BOLD SMALL NU
 [0x1D6CF] = 0x03BE, -- MATHEMATICAL BOLD SMALL XI
 [0x1D6D0] = 0x03BF, -- MATHEMATICAL BOLD SMALL OMICRON
 [0x1D6D1] = 0x03C0, -- MATHEMATICAL BOLD SMALL PI
 [0x1D6D2] = 0x03C1, -- MATHEMATICAL BOLD SMALL RHO
 -- NA   <reserved-1D6A6>..<reserved-1D6A7>
 [0x1D6A6] = {status="disallowed"},
 [0x1D6A7] = {status="disallowed"},
 [0x1D6D5] = 0x03C4, -- MATHEMATICAL BOLD SMALL TAU
 [0x1D6D6] = 0x03C5, -- MATHEMATICAL BOLD SMALL UPSILON
 [0x1D6D7] = 0x03C6, -- MATHEMATICAL BOLD SMALL PHI
 [0x1D6D8] = 0x03C7, -- MATHEMATICAL BOLD SMALL CHI
 [0x1D6D9] = 0x03C8, -- MATHEMATICAL BOLD SMALL PSI
 [0x1D6DA] = 0x03C9, -- MATHEMATICAL BOLD SMALL OMEGA
 [0x1D6DB] = 0x2202, -- MATHEMATICAL BOLD PARTIAL DIFFERENTIAL
 [0x1D6DC] = 0x03B5, -- MATHEMATICAL BOLD EPSILON SYMBOL
 [0x1D6DD] = 0x03B8, -- MATHEMATICAL BOLD THETA SYMBOL
 [0x1D6DE] = 0x03BA, -- MATHEMATICAL BOLD KAPPA SYMBOL
 [0x1D6DF] = 0x03C6, -- MATHEMATICAL BOLD PHI SYMBOL
 [0x1D6E0] = 0x03C1, -- MATHEMATICAL BOLD RHO SYMBOL
 [0x1D6E1] = 0x03C0, -- MATHEMATICAL BOLD PI SYMBOL
 [0x1D6E2] = 0x03B1, -- MATHEMATICAL ITALIC CAPITAL ALPHA
 [0x1D6E3] = 0x03B2, -- MATHEMATICAL ITALIC CAPITAL BETA
 [0x1D6E4] = 0x03B3, -- MATHEMATICAL ITALIC CAPITAL GAMMA
 [0x1D6E5] = 0x03B4, -- MATHEMATICAL ITALIC CAPITAL DELTA
 [0x1D6E6] = 0x03B5, -- MATHEMATICAL ITALIC CAPITAL EPSILON
 [0x1D6E7] = 0x03B6, -- MATHEMATICAL ITALIC CAPITAL ZETA
 [0x1D6E8] = 0x03B7, -- MATHEMATICAL ITALIC CAPITAL ETA
 [0x1D6E9] = 0x03B8, -- MATHEMATICAL ITALIC CAPITAL THETA
 [0x1D6EA] = 0x03B9, -- MATHEMATICAL ITALIC CAPITAL IOTA
 [0x1D6EB] = 0x03BA, -- MATHEMATICAL ITALIC CAPITAL KAPPA
 [0x1D6EC] = 0x03BB, -- MATHEMATICAL ITALIC CAPITAL LAMDA
 [0x1D6ED] = 0x03BC, -- MATHEMATICAL ITALIC CAPITAL MU
 [0x1D6EE] = 0x03BD, -- MATHEMATICAL ITALIC CAPITAL NU
 [0x1D6EF] = 0x03BE, -- MATHEMATICAL ITALIC CAPITAL XI
 [0x1D6F0] = 0x03BF, -- MATHEMATICAL ITALIC CAPITAL OMICRON
 [0x1D6F1] = 0x03C0, -- MATHEMATICAL ITALIC CAPITAL PI
 [0x1D6F2] = 0x03C1, -- MATHEMATICAL ITALIC CAPITAL RHO
 [0x1D6F3] = 0x03B8, -- MATHEMATICAL ITALIC CAPITAL THETA SYMBOL
 [0x1D6F4] = 0x03C3, -- MATHEMATICAL ITALIC CAPITAL SIGMA
 [0x1D6F5] = 0x03C4, -- MATHEMATICAL ITALIC CAPITAL TAU
 [0x1D6F6] = 0x03C5, -- MATHEMATICAL ITALIC CAPITAL UPSILON
 [0x1D6F7] = 0x03C6, -- MATHEMATICAL ITALIC CAPITAL PHI
 [0x1D6F8] = 0x03C7, -- MATHEMATICAL ITALIC CAPITAL CHI
 [0x1D6F9] = 0x03C8, -- MATHEMATICAL ITALIC CAPITAL PSI
 [0x1D6FA] = 0x03C9, -- MATHEMATICAL ITALIC CAPITAL OMEGA
 [0x1D6FB] = 0x2207, -- MATHEMATICAL ITALIC NABLA
 [0x1D6FC] = 0x03B1, -- MATHEMATICAL ITALIC SMALL ALPHA
 [0x1D6FD] = 0x03B2, -- MATHEMATICAL ITALIC SMALL BETA
 [0x1D6FE] = 0x03B3, -- MATHEMATICAL ITALIC SMALL GAMMA
 [0x1D6FF] = 0x03B4, -- MATHEMATICAL ITALIC SMALL DELTA
 [0x1D700] = 0x03B5, -- MATHEMATICAL ITALIC SMALL EPSILON
 [0x1D701] = 0x03B6, -- MATHEMATICAL ITALIC SMALL ZETA
 [0x1D702] = 0x03B7, -- MATHEMATICAL ITALIC SMALL ETA
 [0x1D703] = 0x03B8, -- MATHEMATICAL ITALIC SMALL THETA
 [0x1D704] = 0x03B9, -- MATHEMATICAL ITALIC SMALL IOTA
 [0x1D705] = 0x03BA, -- MATHEMATICAL ITALIC SMALL KAPPA
 [0x1D706] = 0x03BB, -- MATHEMATICAL ITALIC SMALL LAMDA
 [0x1D707] = 0x03BC, -- MATHEMATICAL ITALIC SMALL MU
 [0x1D708] = 0x03BD, -- MATHEMATICAL ITALIC SMALL NU
 [0x1D709] = 0x03BE, -- MATHEMATICAL ITALIC SMALL XI
 [0x1D70A] = 0x03BF, -- MATHEMATICAL ITALIC SMALL OMICRON
 [0x1D70B] = 0x03C0, -- MATHEMATICAL ITALIC SMALL PI
 [0x1D70C] = 0x03C1, -- MATHEMATICAL ITALIC SMALL RHO
 -- MATHEMATICAL BOLD SMALL FINAL SIGMA..MATHEMATICAL BOLD SMALL SIGMA
 [0x1D6D3] = 0x03C3,
 [0x1D6D4] = 0x03C3,
 [0x1D70F] = 0x03C4, -- MATHEMATICAL ITALIC SMALL TAU
 [0x1D710] = 0x03C5, -- MATHEMATICAL ITALIC SMALL UPSILON
 [0x1D711] = 0x03C6, -- MATHEMATICAL ITALIC SMALL PHI
 [0x1D712] = 0x03C7, -- MATHEMATICAL ITALIC SMALL CHI
 [0x1D713] = 0x03C8, -- MATHEMATICAL ITALIC SMALL PSI
 [0x1D714] = 0x03C9, -- MATHEMATICAL ITALIC SMALL OMEGA
 [0x1D715] = 0x2202, -- MATHEMATICAL ITALIC PARTIAL DIFFERENTIAL
 [0x1D716] = 0x03B5, -- MATHEMATICAL ITALIC EPSILON SYMBOL
 [0x1D717] = 0x03B8, -- MATHEMATICAL ITALIC THETA SYMBOL
 [0x1D718] = 0x03BA, -- MATHEMATICAL ITALIC KAPPA SYMBOL
 [0x1D719] = 0x03C6, -- MATHEMATICAL ITALIC PHI SYMBOL
 [0x1D71A] = 0x03C1, -- MATHEMATICAL ITALIC RHO SYMBOL
 [0x1D71B] = 0x03C0, -- MATHEMATICAL ITALIC PI SYMBOL
 [0x1D71C] = 0x03B1, -- MATHEMATICAL BOLD ITALIC CAPITAL ALPHA
 [0x1D71D] = 0x03B2, -- MATHEMATICAL BOLD ITALIC CAPITAL BETA
 [0x1D71E] = 0x03B3, -- MATHEMATICAL BOLD ITALIC CAPITAL GAMMA
 [0x1D71F] = 0x03B4, -- MATHEMATICAL BOLD ITALIC CAPITAL DELTA
 [0x1D720] = 0x03B5, -- MATHEMATICAL BOLD ITALIC CAPITAL EPSILON
 [0x1D721] = 0x03B6, -- MATHEMATICAL BOLD ITALIC CAPITAL ZETA
 [0x1D722] = 0x03B7, -- MATHEMATICAL BOLD ITALIC CAPITAL ETA
 [0x1D723] = 0x03B8, -- MATHEMATICAL BOLD ITALIC CAPITAL THETA
 [0x1D724] = 0x03B9, -- MATHEMATICAL BOLD ITALIC CAPITAL IOTA
 [0x1D725] = 0x03BA, -- MATHEMATICAL BOLD ITALIC CAPITAL KAPPA
 [0x1D726] = 0x03BB, -- MATHEMATICAL BOLD ITALIC CAPITAL LAMDA
 [0x1D727] = 0x03BC, -- MATHEMATICAL BOLD ITALIC CAPITAL MU
 [0x1D728] = 0x03BD, -- MATHEMATICAL BOLD ITALIC CAPITAL NU
 [0x1D729] = 0x03BE, -- MATHEMATICAL BOLD ITALIC CAPITAL XI
 [0x1D72A] = 0x03BF, -- MATHEMATICAL BOLD ITALIC CAPITAL OMICRON
 [0x1D72B] = 0x03C0, -- MATHEMATICAL BOLD ITALIC CAPITAL PI
 [0x1D72C] = 0x03C1, -- MATHEMATICAL BOLD ITALIC CAPITAL RHO
 [0x1D72D] = 0x03B8, -- MATHEMATICAL BOLD ITALIC CAPITAL THETA SYMBOL
 [0x1D72E] = 0x03C3, -- MATHEMATICAL BOLD ITALIC CAPITAL SIGMA
 [0x1D72F] = 0x03C4, -- MATHEMATICAL BOLD ITALIC CAPITAL TAU
 [0x1D730] = 0x03C5, -- MATHEMATICAL BOLD ITALIC CAPITAL UPSILON
 [0x1D731] = 0x03C6, -- MATHEMATICAL BOLD ITALIC CAPITAL PHI
 [0x1D732] = 0x03C7, -- MATHEMATICAL BOLD ITALIC CAPITAL CHI
 [0x1D733] = 0x03C8, -- MATHEMATICAL BOLD ITALIC CAPITAL PSI
 [0x1D734] = 0x03C9, -- MATHEMATICAL BOLD ITALIC CAPITAL OMEGA
 [0x1D735] = 0x2207, -- MATHEMATICAL BOLD ITALIC NABLA
 [0x1D736] = 0x03B1, -- MATHEMATICAL BOLD ITALIC SMALL ALPHA
 [0x1D737] = 0x03B2, -- MATHEMATICAL BOLD ITALIC SMALL BETA
 [0x1D738] = 0x03B3, -- MATHEMATICAL BOLD ITALIC SMALL GAMMA
 [0x1D739] = 0x03B4, -- MATHEMATICAL BOLD ITALIC SMALL DELTA
 [0x1D73A] = 0x03B5, -- MATHEMATICAL BOLD ITALIC SMALL EPSILON
 [0x1D73B] = 0x03B6, -- MATHEMATICAL BOLD ITALIC SMALL ZETA
 [0x1D73C] = 0x03B7, -- MATHEMATICAL BOLD ITALIC SMALL ETA
 [0x1D73D] = 0x03B8, -- MATHEMATICAL BOLD ITALIC SMALL THETA
 [0x1D73E] = 0x03B9, -- MATHEMATICAL BOLD ITALIC SMALL IOTA
 [0x1D73F] = 0x03BA, -- MATHEMATICAL BOLD ITALIC SMALL KAPPA
 [0x1D740] = 0x03BB, -- MATHEMATICAL BOLD ITALIC SMALL LAMDA
 [0x1D741] = 0x03BC, -- MATHEMATICAL BOLD ITALIC SMALL MU
 [0x1D742] = 0x03BD, -- MATHEMATICAL BOLD ITALIC SMALL NU
 [0x1D743] = 0x03BE, -- MATHEMATICAL BOLD ITALIC SMALL XI
 [0x1D744] = 0x03BF, -- MATHEMATICAL BOLD ITALIC SMALL OMICRON
 [0x1D745] = 0x03C0, -- MATHEMATICAL BOLD ITALIC SMALL PI
 [0x1D746] = 0x03C1, -- MATHEMATICAL BOLD ITALIC SMALL RHO
 -- MATHEMATICAL ITALIC SMALL FINAL SIGMA..MATHEMATICAL ITALIC SMALL SIGMA
 [0x1D70D] = 0x03C3,
 [0x1D70E] = 0x03C3,
 [0x1D749] = 0x03C4, -- MATHEMATICAL BOLD ITALIC SMALL TAU
 [0x1D74A] = 0x03C5, -- MATHEMATICAL BOLD ITALIC SMALL UPSILON
 [0x1D74B] = 0x03C6, -- MATHEMATICAL BOLD ITALIC SMALL PHI
 [0x1D74C] = 0x03C7, -- MATHEMATICAL BOLD ITALIC SMALL CHI
 [0x1D74D] = 0x03C8, -- MATHEMATICAL BOLD ITALIC SMALL PSI
 [0x1D74E] = 0x03C9, -- MATHEMATICAL BOLD ITALIC SMALL OMEGA
 [0x1D74F] = 0x2202, -- MATHEMATICAL BOLD ITALIC PARTIAL DIFFERENTIAL
 [0x1D750] = 0x03B5, -- MATHEMATICAL BOLD ITALIC EPSILON SYMBOL
 [0x1D751] = 0x03B8, -- MATHEMATICAL BOLD ITALIC THETA SYMBOL
 [0x1D752] = 0x03BA, -- MATHEMATICAL BOLD ITALIC KAPPA SYMBOL
 [0x1D753] = 0x03C6, -- MATHEMATICAL BOLD ITALIC PHI SYMBOL
 [0x1D754] = 0x03C1, -- MATHEMATICAL BOLD ITALIC RHO SYMBOL
 [0x1D755] = 0x03C0, -- MATHEMATICAL BOLD ITALIC PI SYMBOL
 [0x1D756] = 0x03B1, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL ALPHA
 [0x1D757] = 0x03B2, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL BETA
 [0x1D758] = 0x03B3, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL GAMMA
 [0x1D759] = 0x03B4, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL DELTA
 [0x1D75A] = 0x03B5, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL EPSILON
 [0x1D75B] = 0x03B6, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL ZETA
 [0x1D75C] = 0x03B7, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL ETA
 [0x1D75D] = 0x03B8, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL THETA
 [0x1D75E] = 0x03B9, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL IOTA
 [0x1D75F] = 0x03BA, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL KAPPA
 [0x1D760] = 0x03BB, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL LAMDA
 [0x1D761] = 0x03BC, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL MU
 [0x1D762] = 0x03BD, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL NU
 [0x1D763] = 0x03BE, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL XI
 [0x1D764] = 0x03BF, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL OMICRON
 [0x1D765] = 0x03C0, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL PI
 [0x1D766] = 0x03C1, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL RHO
 [0x1D767] = 0x03B8, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL THETA SYMBOL
 [0x1D768] = 0x03C3, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL SIGMA
 [0x1D769] = 0x03C4, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL TAU
 [0x1D76A] = 0x03C5, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL UPSILON
 [0x1D76B] = 0x03C6, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL PHI
 [0x1D76C] = 0x03C7, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL CHI
 [0x1D76D] = 0x03C8, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL PSI
 [0x1D76E] = 0x03C9, -- MATHEMATICAL SANS-SERIF BOLD CAPITAL OMEGA
 [0x1D76F] = 0x2207, -- MATHEMATICAL SANS-SERIF BOLD NABLA
 [0x1D770] = 0x03B1, -- MATHEMATICAL SANS-SERIF BOLD SMALL ALPHA
 [0x1D771] = 0x03B2, -- MATHEMATICAL SANS-SERIF BOLD SMALL BETA
 [0x1D772] = 0x03B3, -- MATHEMATICAL SANS-SERIF BOLD SMALL GAMMA
 [0x1D773] = 0x03B4, -- MATHEMATICAL SANS-SERIF BOLD SMALL DELTA
 [0x1D774] = 0x03B5, -- MATHEMATICAL SANS-SERIF BOLD SMALL EPSILON
 [0x1D775] = 0x03B6, -- MATHEMATICAL SANS-SERIF BOLD SMALL ZETA
 [0x1D776] = 0x03B7, -- MATHEMATICAL SANS-SERIF BOLD SMALL ETA
 [0x1D777] = 0x03B8, -- MATHEMATICAL SANS-SERIF BOLD SMALL THETA
 [0x1D778] = 0x03B9, -- MATHEMATICAL SANS-SERIF BOLD SMALL IOTA
 [0x1D779] = 0x03BA, -- MATHEMATICAL SANS-SERIF BOLD SMALL KAPPA
 [0x1D77A] = 0x03BB, -- MATHEMATICAL SANS-SERIF BOLD SMALL LAMDA
 [0x1D77B] = 0x03BC, -- MATHEMATICAL SANS-SERIF BOLD SMALL MU
 [0x1D77C] = 0x03BD, -- MATHEMATICAL SANS-SERIF BOLD SMALL NU
 [0x1D77D] = 0x03BE, -- MATHEMATICAL SANS-SERIF BOLD SMALL XI
 [0x1D77E] = 0x03BF, -- MATHEMATICAL SANS-SERIF BOLD SMALL OMICRON
 [0x1D77F] = 0x03C0, -- MATHEMATICAL SANS-SERIF BOLD SMALL PI
 [0x1D780] = 0x03C1, -- MATHEMATICAL SANS-SERIF BOLD SMALL RHO
 -- MATHEMATICAL BOLD ITALIC SMALL FINAL SIGMA..MATHEMATICAL BOLD ITALIC SMALL SIGMA
 [0x1D747] = 0x03C3,
 [0x1D748] = 0x03C3,
 [0x1D783] = 0x03C4, -- MATHEMATICAL SANS-SERIF BOLD SMALL TAU
 [0x1D784] = 0x03C5, -- MATHEMATICAL SANS-SERIF BOLD SMALL UPSILON
 [0x1D785] = 0x03C6, -- MATHEMATICAL SANS-SERIF BOLD SMALL PHI
 [0x1D786] = 0x03C7, -- MATHEMATICAL SANS-SERIF BOLD SMALL CHI
 [0x1D787] = 0x03C8, -- MATHEMATICAL SANS-SERIF BOLD SMALL PSI
 [0x1D788] = 0x03C9, -- MATHEMATICAL SANS-SERIF BOLD SMALL OMEGA
 [0x1D789] = 0x2202, -- MATHEMATICAL SANS-SERIF BOLD PARTIAL DIFFERENTIAL
 [0x1D78A] = 0x03B5, -- MATHEMATICAL SANS-SERIF BOLD EPSILON SYMBOL
 [0x1D78B] = 0x03B8, -- MATHEMATICAL SANS-SERIF BOLD THETA SYMBOL
 [0x1D78C] = 0x03BA, -- MATHEMATICAL SANS-SERIF BOLD KAPPA SYMBOL
 [0x1D78D] = 0x03C6, -- MATHEMATICAL SANS-SERIF BOLD PHI SYMBOL
 [0x1D78E] = 0x03C1, -- MATHEMATICAL SANS-SERIF BOLD RHO SYMBOL
 [0x1D78F] = 0x03C0, -- MATHEMATICAL SANS-SERIF BOLD PI SYMBOL
 [0x1D790] = 0x03B1, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL ALPHA
 [0x1D791] = 0x03B2, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL BETA
 [0x1D792] = 0x03B3, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL GAMMA
 [0x1D793] = 0x03B4, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL DELTA
 [0x1D794] = 0x03B5, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL EPSILON
 [0x1D795] = 0x03B6, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL ZETA
 [0x1D796] = 0x03B7, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL ETA
 [0x1D797] = 0x03B8, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL THETA
 [0x1D798] = 0x03B9, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL IOTA
 [0x1D799] = 0x03BA, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL KAPPA
 [0x1D79A] = 0x03BB, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL LAMDA
 [0x1D79B] = 0x03BC, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL MU
 [0x1D79C] = 0x03BD, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL NU
 [0x1D79D] = 0x03BE, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL XI
 [0x1D79E] = 0x03BF, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL OMICRON
 [0x1D79F] = 0x03C0, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL PI
 [0x1D7A0] = 0x03C1, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL RHO
 [0x1D7A1] = 0x03B8, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL THETA SYMBOL
 [0x1D7A2] = 0x03C3, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL SIGMA
 [0x1D7A3] = 0x03C4, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL TAU
 [0x1D7A4] = 0x03C5, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL UPSILON
 [0x1D7A5] = 0x03C6, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL PHI
 [0x1D7A6] = 0x03C7, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL CHI
 [0x1D7A7] = 0x03C8, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL PSI
 [0x1D7A8] = 0x03C9, -- MATHEMATICAL SANS-SERIF BOLD ITALIC CAPITAL OMEGA
 [0x1D7A9] = 0x2207, -- MATHEMATICAL SANS-SERIF BOLD ITALIC NABLA
 [0x1D7AA] = 0x03B1, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL ALPHA
 [0x1D7AB] = 0x03B2, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL BETA
 [0x1D7AC] = 0x03B3, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL GAMMA
 [0x1D7AD] = 0x03B4, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL DELTA
 [0x1D7AE] = 0x03B5, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL EPSILON
 [0x1D7AF] = 0x03B6, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL ZETA
 [0x1D7B0] = 0x03B7, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL ETA
 [0x1D7B1] = 0x03B8, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL THETA
 [0x1D7B2] = 0x03B9, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL IOTA
 [0x1D7B3] = 0x03BA, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL KAPPA
 [0x1D7B4] = 0x03BB, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL LAMDA
 [0x1D7B5] = 0x03BC, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL MU
 [0x1D7B6] = 0x03BD, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL NU
 [0x1D7B7] = 0x03BE, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL XI
 [0x1D7B8] = 0x03BF, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL OMICRON
 [0x1D7B9] = 0x03C0, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL PI
 [0x1D7BA] = 0x03C1, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL RHO
 -- MATHEMATICAL SANS-SERIF BOLD SMALL FINAL SIGMA..MATHEMATICAL SANS-SERIF BOLD SMALL SIGMA
 [0x1D781] = 0x03C3,
 [0x1D782] = 0x03C3,
 [0x1D7BD] = 0x03C4, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL TAU
 [0x1D7BE] = 0x03C5, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL UPSILON
 [0x1D7BF] = 0x03C6, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL PHI
 [0x1D7C0] = 0x03C7, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL CHI
 [0x1D7C1] = 0x03C8, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL PSI
 [0x1D7C2] = 0x03C9, -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL OMEGA
 [0x1D7C3] = 0x2202, -- MATHEMATICAL SANS-SERIF BOLD ITALIC PARTIAL DIFFERENTIAL
 [0x1D7C4] = 0x03B5, -- MATHEMATICAL SANS-SERIF BOLD ITALIC EPSILON SYMBOL
 [0x1D7C5] = 0x03B8, -- MATHEMATICAL SANS-SERIF BOLD ITALIC THETA SYMBOL
 [0x1D7C6] = 0x03BA, -- MATHEMATICAL SANS-SERIF BOLD ITALIC KAPPA SYMBOL
 [0x1D7C7] = 0x03C6, -- MATHEMATICAL SANS-SERIF BOLD ITALIC PHI SYMBOL
 [0x1D7C8] = 0x03C1, -- MATHEMATICAL SANS-SERIF BOLD ITALIC RHO SYMBOL
 [0x1D7C9] = 0x03C0, -- MATHEMATICAL SANS-SERIF BOLD ITALIC PI SYMBOL
 -- MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL FINAL SIGMA..MATHEMATICAL SANS-SERIF BOLD ITALIC SMALL SIGMA
 [0x1D7BB] = 0x03C3,
 [0x1D7BC] = 0x03C3,
 -- MATHEMATICAL BOLD CAPITAL DIGAMMA..MATHEMATICAL BOLD SMALL DIGAMMA
 [0x1D7CA] = 0x03DD,
 [0x1D7CB] = 0x03DD,
 [0x1D7CE] = 0x0030, -- MATHEMATICAL BOLD DIGIT ZERO
 [0x1D7CF] = 0x0031, -- MATHEMATICAL BOLD DIGIT ONE
 [0x1D7D0] = 0x0032, -- MATHEMATICAL BOLD DIGIT TWO
 [0x1D7D1] = 0x0033, -- MATHEMATICAL BOLD DIGIT THREE
 [0x1D7D2] = 0x0034, -- MATHEMATICAL BOLD DIGIT FOUR
 [0x1D7D3] = 0x0035, -- MATHEMATICAL BOLD DIGIT FIVE
 [0x1D7D4] = 0x0036, -- MATHEMATICAL BOLD DIGIT SIX
 [0x1D7D5] = 0x0037, -- MATHEMATICAL BOLD DIGIT SEVEN
 [0x1D7D6] = 0x0038, -- MATHEMATICAL BOLD DIGIT EIGHT
 [0x1D7D7] = 0x0039, -- MATHEMATICAL BOLD DIGIT NINE
 [0x1D7D8] = 0x0030, -- MATHEMATICAL DOUBLE-STRUCK DIGIT ZERO
 [0x1D7D9] = 0x0031, -- MATHEMATICAL DOUBLE-STRUCK DIGIT ONE
 [0x1D7DA] = 0x0032, -- MATHEMATICAL DOUBLE-STRUCK DIGIT TWO
 [0x1D7DB] = 0x0033, -- MATHEMATICAL DOUBLE-STRUCK DIGIT THREE
 [0x1D7DC] = 0x0034, -- MATHEMATICAL DOUBLE-STRUCK DIGIT FOUR
 [0x1D7DD] = 0x0035, -- MATHEMATICAL DOUBLE-STRUCK DIGIT FIVE
 [0x1D7DE] = 0x0036, -- MATHEMATICAL DOUBLE-STRUCK DIGIT SIX
 [0x1D7DF] = 0x0037, -- MATHEMATICAL DOUBLE-STRUCK DIGIT SEVEN
 [0x1D7E0] = 0x0038, -- MATHEMATICAL DOUBLE-STRUCK DIGIT EIGHT
 [0x1D7E1] = 0x0039, -- MATHEMATICAL DOUBLE-STRUCK DIGIT NINE
 [0x1D7E2] = 0x0030, -- MATHEMATICAL SANS-SERIF DIGIT ZERO
 [0x1D7E3] = 0x0031, -- MATHEMATICAL SANS-SERIF DIGIT ONE
 [0x1D7E4] = 0x0032, -- MATHEMATICAL SANS-SERIF DIGIT TWO
 [0x1D7E5] = 0x0033, -- MATHEMATICAL SANS-SERIF DIGIT THREE
 [0x1D7E6] = 0x0034, -- MATHEMATICAL SANS-SERIF DIGIT FOUR
 [0x1D7E7] = 0x0035, -- MATHEMATICAL SANS-SERIF DIGIT FIVE
 [0x1D7E8] = 0x0036, -- MATHEMATICAL SANS-SERIF DIGIT SIX
 [0x1D7E9] = 0x0037, -- MATHEMATICAL SANS-SERIF DIGIT SEVEN
 [0x1D7EA] = 0x0038, -- MATHEMATICAL SANS-SERIF DIGIT EIGHT
 [0x1D7EB] = 0x0039, -- MATHEMATICAL SANS-SERIF DIGIT NINE
 [0x1D7EC] = 0x0030, -- MATHEMATICAL SANS-SERIF BOLD DIGIT ZERO
 [0x1D7ED] = 0x0031, -- MATHEMATICAL SANS-SERIF BOLD DIGIT ONE
 [0x1D7EE] = 0x0032, -- MATHEMATICAL SANS-SERIF BOLD DIGIT TWO
 [0x1D7EF] = 0x0033, -- MATHEMATICAL SANS-SERIF BOLD DIGIT THREE
 [0x1D7F0] = 0x0034, -- MATHEMATICAL SANS-SERIF BOLD DIGIT FOUR
 [0x1D7F1] = 0x0035, -- MATHEMATICAL SANS-SERIF BOLD DIGIT FIVE
 [0x1D7F2] = 0x0036, -- MATHEMATICAL SANS-SERIF BOLD DIGIT SIX
 [0x1D7F3] = 0x0037, -- MATHEMATICAL SANS-SERIF BOLD DIGIT SEVEN
 [0x1D7F4] = 0x0038, -- MATHEMATICAL SANS-SERIF BOLD DIGIT EIGHT
 [0x1D7F5] = 0x0039, -- MATHEMATICAL SANS-SERIF BOLD DIGIT NINE
 [0x1D7F6] = 0x0030, -- MATHEMATICAL MONOSPACE DIGIT ZERO
 [0x1D7F7] = 0x0031, -- MATHEMATICAL MONOSPACE DIGIT ONE
 [0x1D7F8] = 0x0032, -- MATHEMATICAL MONOSPACE DIGIT TWO
 [0x1D7F9] = 0x0033, -- MATHEMATICAL MONOSPACE DIGIT THREE
 [0x1D7FA] = 0x0034, -- MATHEMATICAL MONOSPACE DIGIT FOUR
 [0x1D7FB] = 0x0035, -- MATHEMATICAL MONOSPACE DIGIT FIVE
 [0x1D7FC] = 0x0036, -- MATHEMATICAL MONOSPACE DIGIT SIX
 [0x1D7FD] = 0x0037, -- MATHEMATICAL MONOSPACE DIGIT SEVEN
 [0x1D7FE] = 0x0038, -- MATHEMATICAL MONOSPACE DIGIT EIGHT
 [0x1D7FF] = 0x0039, -- MATHEMATICAL MONOSPACE DIGIT NINE
 -- NA   <reserved-1D7CC>..<reserved-1D7CD>
 [0x1D7CC] = {status="disallowed"},
 [0x1D7CD] = {status="disallowed"},
 -- SIGNWRITING AIR BLOW SMALL ROTATIONS..SIGNWRITING BREATH EXHALE
 [0x1DA37] = {status="disallowed"},
 [0x1DA38] = {status="disallowed"},
 [0x1DA39] = {status="disallowed"},
 [0x1DA3A] = {status="disallowed"},
 [0x1DA75] = {status="valid"}, -- SIGNWRITING UPPER BODY TILTING FROM HIP JOINTS
 [0x1DA84] = {status="valid"}, -- SIGNWRITING LOCATION HEAD NECK
 [0x1DAA0] = {status="disallowed"}, -- NA   <reserved-1DAA0>
 -- SIGNWRITING FILL MODIFIER-2..SIGNWRITING FILL MODIFIER-6
 [0x1DA9B] = {status="valid"},
 [0x1DA9C] = {status="valid"},
 [0x1DA9D] = {status="valid"},
 [0x1DA9E] = {status="valid"},
 [0x1DA9F] = {status="valid"},
 [0x1E007] = {status="disallowed"}, -- NA   <reserved-1E007>
 -- NA   <reserved-1E019>..<reserved-1E01A>
 [0x1E019] = {status="disallowed"},
 [0x1E01A] = {status="disallowed"},
 [0x1E022] = {status="disallowed"}, -- NA   <reserved-1E022>
 [0x1E025] = {status="disallowed"}, -- NA   <reserved-1E025>
 -- COMBINING GLAGOLITIC LETTER YU..COMBINING GLAGOLITIC LETTER SMALL YUS
 [0x1E023] = {status="valid"},
 [0x1E024] = {status="valid"},
 -- COMBINING GLAGOLITIC LETTER YO..COMBINING GLAGOLITIC LETTER FITA
 [0x1E026] = {status="valid"},
 [0x1E027] = {status="valid"},
 [0x1E028] = {status="valid"},
 [0x1E029] = {status="valid"},
 [0x1E02A] = {status="valid"},
 -- NA   <reserved-1E8C5>..<reserved-1E8C6>
 [0x1E8C5] = {status="disallowed"},
 [0x1E8C6] = {status="disallowed"},
 [0x1E900] = 0x1E922, -- ADLAM CAPITAL LETTER ALIF
 [0x1E901] = 0x1E923, -- ADLAM CAPITAL LETTER DAALI
 [0x1E902] = 0x1E924, -- ADLAM CAPITAL LETTER LAAM
 [0x1E903] = 0x1E925, -- ADLAM CAPITAL LETTER MIIM
 [0x1E904] = 0x1E926, -- ADLAM CAPITAL LETTER BA
 [0x1E905] = 0x1E927, -- ADLAM CAPITAL LETTER SINNYIIYHE
 [0x1E906] = 0x1E928, -- ADLAM CAPITAL LETTER PE
 [0x1E907] = 0x1E929, -- ADLAM CAPITAL LETTER BHE
 [0x1E908] = 0x1E92A, -- ADLAM CAPITAL LETTER RA
 [0x1E909] = 0x1E92B, -- ADLAM CAPITAL LETTER E
 [0x1E90A] = 0x1E92C, -- ADLAM CAPITAL LETTER FA
 [0x1E90B] = 0x1E92D, -- ADLAM CAPITAL LETTER I
 [0x1E90C] = 0x1E92E, -- ADLAM CAPITAL LETTER O
 [0x1E90D] = 0x1E92F, -- ADLAM CAPITAL LETTER DHA
 [0x1E90E] = 0x1E930, -- ADLAM CAPITAL LETTER YHE
 [0x1E90F] = 0x1E931, -- ADLAM CAPITAL LETTER WAW
 [0x1E910] = 0x1E932, -- ADLAM CAPITAL LETTER NUN
 [0x1E911] = 0x1E933, -- ADLAM CAPITAL LETTER KAF
 [0x1E912] = 0x1E934, -- ADLAM CAPITAL LETTER YA
 [0x1E913] = 0x1E935, -- ADLAM CAPITAL LETTER U
 [0x1E914] = 0x1E936, -- ADLAM CAPITAL LETTER JIIM
 [0x1E915] = 0x1E937, -- ADLAM CAPITAL LETTER CHI
 [0x1E916] = 0x1E938, -- ADLAM CAPITAL LETTER HA
 [0x1E917] = 0x1E939, -- ADLAM CAPITAL LETTER QAAF
 [0x1E918] = 0x1E93A, -- ADLAM CAPITAL LETTER GA
 [0x1E919] = 0x1E93B, -- ADLAM CAPITAL LETTER NYA
 [0x1E91A] = 0x1E93C, -- ADLAM CAPITAL LETTER TU
 [0x1E91B] = 0x1E93D, -- ADLAM CAPITAL LETTER NHA
 [0x1E91C] = 0x1E93E, -- ADLAM CAPITAL LETTER VA
 [0x1E91D] = 0x1E93F, -- ADLAM CAPITAL LETTER KHA
 [0x1E91E] = 0x1E940, -- ADLAM CAPITAL LETTER GBE
 [0x1E91F] = 0x1E941, -- ADLAM CAPITAL LETTER ZAL
 [0x1E920] = 0x1E942, -- ADLAM CAPITAL LETTER KPO
 [0x1E921] = 0x1E943, -- ADLAM CAPITAL LETTER SHA
 -- NA   <reserved-1E94B>..<reserved-1E94F>
 [0x1E94B] = {status="disallowed"},
 [0x1E94C] = {status="disallowed"},
 [0x1E94D] = {status="disallowed"},
 [0x1E94E] = {status="disallowed"},
 [0x1E94F] = {status="disallowed"},
 -- NA   <reserved-1E95A>..<reserved-1E95D>
 [0x1E95A] = {status="disallowed"},
 [0x1E95B] = {status="disallowed"},
 [0x1E95C] = {status="disallowed"},
 [0x1E95D] = {status="disallowed"},
 -- ADLAM INITIAL EXCLAMATION MARK..ADLAM INITIAL QUESTION MARK
 [0x1E95E] = {status="disallowed"},
 [0x1E95F] = {status="disallowed"},
 [0x1EE00] = 0x0627, -- ARABIC MATHEMATICAL ALEF
 [0x1EE01] = 0x0628, -- ARABIC MATHEMATICAL BEH
 [0x1EE02] = 0x062C, -- ARABIC MATHEMATICAL JEEM
 [0x1EE03] = 0x062F, -- ARABIC MATHEMATICAL DAL
 [0x1EE04] = {status="disallowed"}, -- NA   <reserved-1EE04>
 [0x1EE05] = 0x0648, -- ARABIC MATHEMATICAL WAW
 [0x1EE06] = 0x0632, -- ARABIC MATHEMATICAL ZAIN
 [0x1EE07] = 0x062D, -- ARABIC MATHEMATICAL HAH
 [0x1EE08] = 0x0637, -- ARABIC MATHEMATICAL TAH
 [0x1EE09] = 0x064A, -- ARABIC MATHEMATICAL YEH
 [0x1EE0A] = 0x0643, -- ARABIC MATHEMATICAL KAF
 [0x1EE0B] = 0x0644, -- ARABIC MATHEMATICAL LAM
 [0x1EE0C] = 0x0645, -- ARABIC MATHEMATICAL MEEM
 [0x1EE0D] = 0x0646, -- ARABIC MATHEMATICAL NOON
 [0x1EE0E] = 0x0633, -- ARABIC MATHEMATICAL SEEN
 [0x1EE0F] = 0x0639, -- ARABIC MATHEMATICAL AIN
 [0x1EE10] = 0x0641, -- ARABIC MATHEMATICAL FEH
 [0x1EE11] = 0x0635, -- ARABIC MATHEMATICAL SAD
 [0x1EE12] = 0x0642, -- ARABIC MATHEMATICAL QAF
 [0x1EE13] = 0x0631, -- ARABIC MATHEMATICAL REH
 [0x1EE14] = 0x0634, -- ARABIC MATHEMATICAL SHEEN
 [0x1EE15] = 0x062A, -- ARABIC MATHEMATICAL TEH
 [0x1EE16] = 0x062B, -- ARABIC MATHEMATICAL THEH
 [0x1EE17] = 0x062E, -- ARABIC MATHEMATICAL KHAH
 [0x1EE18] = 0x0630, -- ARABIC MATHEMATICAL THAL
 [0x1EE19] = 0x0636, -- ARABIC MATHEMATICAL DAD
 [0x1EE1A] = 0x0638, -- ARABIC MATHEMATICAL ZAH
 [0x1EE1B] = 0x063A, -- ARABIC MATHEMATICAL GHAIN
 [0x1EE1C] = 0x066E, -- ARABIC MATHEMATICAL DOTLESS BEH
 [0x1EE1D] = 0x06BA, -- ARABIC MATHEMATICAL DOTLESS NOON
 [0x1EE1E] = 0x06A1, -- ARABIC MATHEMATICAL DOTLESS FEH
 [0x1EE1F] = 0x066F, -- ARABIC MATHEMATICAL DOTLESS QAF
 [0x1EE20] = {status="disallowed"}, -- NA   <reserved-1EE20>
 [0x1EE21] = 0x0628, -- ARABIC MATHEMATICAL INITIAL BEH
 [0x1EE22] = 0x062C, -- ARABIC MATHEMATICAL INITIAL JEEM
 [0x1EE23] = {status="disallowed"}, -- NA   <reserved-1EE23>
 [0x1EE24] = 0x0647, -- ARABIC MATHEMATICAL INITIAL HEH
 [0x1EE27] = 0x062D, -- ARABIC MATHEMATICAL INITIAL HAH
 [0x1EE28] = {status="disallowed"}, -- NA   <reserved-1EE28>
 [0x1EE29] = 0x064A, -- ARABIC MATHEMATICAL INITIAL YEH
 [0x1EE2A] = 0x0643, -- ARABIC MATHEMATICAL INITIAL KAF
 [0x1EE2B] = 0x0644, -- ARABIC MATHEMATICAL INITIAL LAM
 [0x1EE2C] = 0x0645, -- ARABIC MATHEMATICAL INITIAL MEEM
 [0x1EE2D] = 0x0646, -- ARABIC MATHEMATICAL INITIAL NOON
 [0x1EE2E] = 0x0633, -- ARABIC MATHEMATICAL INITIAL SEEN
 [0x1EE2F] = 0x0639, -- ARABIC MATHEMATICAL INITIAL AIN
 [0x1EE30] = 0x0641, -- ARABIC MATHEMATICAL INITIAL FEH
 [0x1EE31] = 0x0635, -- ARABIC MATHEMATICAL INITIAL SAD
 [0x1EE32] = 0x0642, -- ARABIC MATHEMATICAL INITIAL QAF
 [0x1EE33] = {status="disallowed"}, -- NA   <reserved-1EE33>
 [0x1EE34] = 0x0634, -- ARABIC MATHEMATICAL INITIAL SHEEN
 [0x1EE35] = 0x062A, -- ARABIC MATHEMATICAL INITIAL TEH
 [0x1EE36] = 0x062B, -- ARABIC MATHEMATICAL INITIAL THEH
 [0x1EE37] = 0x062E, -- ARABIC MATHEMATICAL INITIAL KHAH
 [0x1EE38] = {status="disallowed"}, -- NA   <reserved-1EE38>
 [0x1EE39] = 0x0636, -- ARABIC MATHEMATICAL INITIAL DAD
 [0x1EE3A] = {status="disallowed"}, -- NA   <reserved-1EE3A>
 [0x1EE3B] = 0x063A, -- ARABIC MATHEMATICAL INITIAL GHAIN
 -- NA   <reserved-1EE25>..<reserved-1EE26>
 [0x1EE25] = {status="disallowed"},
 [0x1EE26] = {status="disallowed"},
 [0x1EE42] = 0x062C, -- ARABIC MATHEMATICAL TAILED JEEM
 [0x1EE47] = 0x062D, -- ARABIC MATHEMATICAL TAILED HAH
 [0x1EE48] = {status="disallowed"}, -- NA   <reserved-1EE48>
 [0x1EE49] = 0x064A, -- ARABIC MATHEMATICAL TAILED YEH
 [0x1EE4A] = {status="disallowed"}, -- NA   <reserved-1EE4A>
 [0x1EE4B] = 0x0644, -- ARABIC MATHEMATICAL TAILED LAM
 [0x1EE4C] = {status="disallowed"}, -- NA   <reserved-1EE4C>
 [0x1EE4D] = 0x0646, -- ARABIC MATHEMATICAL TAILED NOON
 [0x1EE4E] = 0x0633, -- ARABIC MATHEMATICAL TAILED SEEN
 [0x1EE4F] = 0x0639, -- ARABIC MATHEMATICAL TAILED AIN
 [0x1EE50] = {status="disallowed"}, -- NA   <reserved-1EE50>
 [0x1EE51] = 0x0635, -- ARABIC MATHEMATICAL TAILED SAD
 [0x1EE52] = 0x0642, -- ARABIC MATHEMATICAL TAILED QAF
 [0x1EE53] = {status="disallowed"}, -- NA   <reserved-1EE53>
 [0x1EE54] = 0x0634, -- ARABIC MATHEMATICAL TAILED SHEEN
 -- NA   <reserved-1EE43>..<reserved-1EE46>
 [0x1EE43] = {status="disallowed"},
 [0x1EE44] = {status="disallowed"},
 [0x1EE45] = {status="disallowed"},
 [0x1EE46] = {status="disallowed"},
 [0x1EE57] = 0x062E, -- ARABIC MATHEMATICAL TAILED KHAH
 [0x1EE58] = {status="disallowed"}, -- NA   <reserved-1EE58>
 [0x1EE59] = 0x0636, -- ARABIC MATHEMATICAL TAILED DAD
 [0x1EE5A] = {status="disallowed"}, -- NA   <reserved-1EE5A>
 [0x1EE5B] = 0x063A, -- ARABIC MATHEMATICAL TAILED GHAIN
 [0x1EE5C] = {status="disallowed"}, -- NA   <reserved-1EE5C>
 [0x1EE5D] = 0x06BA, -- ARABIC MATHEMATICAL TAILED DOTLESS NOON
 [0x1EE5E] = {status="disallowed"}, -- NA   <reserved-1EE5E>
 [0x1EE5F] = 0x066F, -- ARABIC MATHEMATICAL TAILED DOTLESS QAF
 [0x1EE60] = {status="disallowed"}, -- NA   <reserved-1EE60>
 [0x1EE61] = 0x0628, -- ARABIC MATHEMATICAL STRETCHED BEH
 [0x1EE62] = 0x062C, -- ARABIC MATHEMATICAL STRETCHED JEEM
 [0x1EE63] = {status="disallowed"}, -- NA   <reserved-1EE63>
 [0x1EE64] = 0x0647, -- ARABIC MATHEMATICAL STRETCHED HEH
 -- NA   <reserved-1EE55>..<reserved-1EE56>
 [0x1EE55] = {status="disallowed"},
 [0x1EE56] = {status="disallowed"},
 [0x1EE67] = 0x062D, -- ARABIC MATHEMATICAL STRETCHED HAH
 [0x1EE68] = 0x0637, -- ARABIC MATHEMATICAL STRETCHED TAH
 [0x1EE69] = 0x064A, -- ARABIC MATHEMATICAL STRETCHED YEH
 [0x1EE6A] = 0x0643, -- ARABIC MATHEMATICAL STRETCHED KAF
 [0x1EE6B] = {status="disallowed"}, -- NA   <reserved-1EE6B>
 [0x1EE6C] = 0x0645, -- ARABIC MATHEMATICAL STRETCHED MEEM
 [0x1EE6D] = 0x0646, -- ARABIC MATHEMATICAL STRETCHED NOON
 [0x1EE6E] = 0x0633, -- ARABIC MATHEMATICAL STRETCHED SEEN
 [0x1EE6F] = 0x0639, -- ARABIC MATHEMATICAL STRETCHED AIN
 [0x1EE70] = 0x0641, -- ARABIC MATHEMATICAL STRETCHED FEH
 [0x1EE71] = 0x0635, -- ARABIC MATHEMATICAL STRETCHED SAD
 [0x1EE72] = 0x0642, -- ARABIC MATHEMATICAL STRETCHED QAF
 [0x1EE73] = {status="disallowed"}, -- NA   <reserved-1EE73>
 [0x1EE74] = 0x0634, -- ARABIC MATHEMATICAL STRETCHED SHEEN
 [0x1EE75] = 0x062A, -- ARABIC MATHEMATICAL STRETCHED TEH
 [0x1EE76] = 0x062B, -- ARABIC MATHEMATICAL STRETCHED THEH
 [0x1EE77] = 0x062E, -- ARABIC MATHEMATICAL STRETCHED KHAH
 [0x1EE78] = {status="disallowed"}, -- NA   <reserved-1EE78>
 [0x1EE79] = 0x0636, -- ARABIC MATHEMATICAL STRETCHED DAD
 [0x1EE7A] = 0x0638, -- ARABIC MATHEMATICAL STRETCHED ZAH
 [0x1EE7B] = 0x063A, -- ARABIC MATHEMATICAL STRETCHED GHAIN
 [0x1EE7C] = 0x066E, -- ARABIC MATHEMATICAL STRETCHED DOTLESS BEH
 [0x1EE7D] = {status="disallowed"}, -- NA   <reserved-1EE7D>
 [0x1EE7E] = 0x06A1, -- ARABIC MATHEMATICAL STRETCHED DOTLESS FEH
 [0x1EE7F] = {status="disallowed"}, -- NA   <reserved-1EE7F>
 [0x1EE80] = 0x0627, -- ARABIC MATHEMATICAL LOOPED ALEF
 [0x1EE81] = 0x0628, -- ARABIC MATHEMATICAL LOOPED BEH
 [0x1EE82] = 0x062C, -- ARABIC MATHEMATICAL LOOPED JEEM
 [0x1EE83] = 0x062F, -- ARABIC MATHEMATICAL LOOPED DAL
 [0x1EE84] = 0x0647, -- ARABIC MATHEMATICAL LOOPED HEH
 [0x1EE85] = 0x0648, -- ARABIC MATHEMATICAL LOOPED WAW
 [0x1EE86] = 0x0632, -- ARABIC MATHEMATICAL LOOPED ZAIN
 [0x1EE87] = 0x062D, -- ARABIC MATHEMATICAL LOOPED HAH
 [0x1EE88] = 0x0637, -- ARABIC MATHEMATICAL LOOPED TAH
 [0x1EE89] = 0x064A, -- ARABIC MATHEMATICAL LOOPED YEH
 [0x1EE8A] = {status="disallowed"}, -- NA   <reserved-1EE8A>
 [0x1EE8B] = 0x0644, -- ARABIC MATHEMATICAL LOOPED LAM
 [0x1EE8C] = 0x0645, -- ARABIC MATHEMATICAL LOOPED MEEM
 [0x1EE8D] = 0x0646, -- ARABIC MATHEMATICAL LOOPED NOON
 [0x1EE8E] = 0x0633, -- ARABIC MATHEMATICAL LOOPED SEEN
 [0x1EE8F] = 0x0639, -- ARABIC MATHEMATICAL LOOPED AIN
 [0x1EE90] = 0x0641, -- ARABIC MATHEMATICAL LOOPED FEH
 [0x1EE91] = 0x0635, -- ARABIC MATHEMATICAL LOOPED SAD
 [0x1EE92] = 0x0642, -- ARABIC MATHEMATICAL LOOPED QAF
 [0x1EE93] = 0x0631, -- ARABIC MATHEMATICAL LOOPED REH
 [0x1EE94] = 0x0634, -- ARABIC MATHEMATICAL LOOPED SHEEN
 [0x1EE95] = 0x062A, -- ARABIC MATHEMATICAL LOOPED TEH
 [0x1EE96] = 0x062B, -- ARABIC MATHEMATICAL LOOPED THEH
 [0x1EE97] = 0x062E, -- ARABIC MATHEMATICAL LOOPED KHAH
 [0x1EE98] = 0x0630, -- ARABIC MATHEMATICAL LOOPED THAL
 [0x1EE99] = 0x0636, -- ARABIC MATHEMATICAL LOOPED DAD
 [0x1EE9A] = 0x0638, -- ARABIC MATHEMATICAL LOOPED ZAH
 [0x1EE9B] = 0x063A, -- ARABIC MATHEMATICAL LOOPED GHAIN
 -- NA   <reserved-1EE65>..<reserved-1EE66>
 [0x1EE65] = {status="disallowed"},
 [0x1EE66] = {status="disallowed"},
 [0x1EEA1] = 0x0628, -- ARABIC MATHEMATICAL DOUBLE-STRUCK BEH
 [0x1EEA2] = 0x062C, -- ARABIC MATHEMATICAL DOUBLE-STRUCK JEEM
 [0x1EEA3] = 0x062F, -- ARABIC MATHEMATICAL DOUBLE-STRUCK DAL
 [0x1EEA4] = {status="disallowed"}, -- NA   <reserved-1EEA4>
 [0x1EEA5] = 0x0648, -- ARABIC MATHEMATICAL DOUBLE-STRUCK WAW
 [0x1EEA6] = 0x0632, -- ARABIC MATHEMATICAL DOUBLE-STRUCK ZAIN
 [0x1EEA7] = 0x062D, -- ARABIC MATHEMATICAL DOUBLE-STRUCK HAH
 [0x1EEA8] = 0x0637, -- ARABIC MATHEMATICAL DOUBLE-STRUCK TAH
 [0x1EEA9] = 0x064A, -- ARABIC MATHEMATICAL DOUBLE-STRUCK YEH
 [0x1EEAA] = {status="disallowed"}, -- NA   <reserved-1EEAA>
 [0x1EEAB] = 0x0644, -- ARABIC MATHEMATICAL DOUBLE-STRUCK LAM
 [0x1EEAC] = 0x0645, -- ARABIC MATHEMATICAL DOUBLE-STRUCK MEEM
 [0x1EEAD] = 0x0646, -- ARABIC MATHEMATICAL DOUBLE-STRUCK NOON
 [0x1EEAE] = 0x0633, -- ARABIC MATHEMATICAL DOUBLE-STRUCK SEEN
 [0x1EEAF] = 0x0639, -- ARABIC MATHEMATICAL DOUBLE-STRUCK AIN
 [0x1EEB0] = 0x0641, -- ARABIC MATHEMATICAL DOUBLE-STRUCK FEH
 [0x1EEB1] = 0x0635, -- ARABIC MATHEMATICAL DOUBLE-STRUCK SAD
 [0x1EEB2] = 0x0642, -- ARABIC MATHEMATICAL DOUBLE-STRUCK QAF
 [0x1EEB3] = 0x0631, -- ARABIC MATHEMATICAL DOUBLE-STRUCK REH
 [0x1EEB4] = 0x0634, -- ARABIC MATHEMATICAL DOUBLE-STRUCK SHEEN
 [0x1EEB5] = 0x062A, -- ARABIC MATHEMATICAL DOUBLE-STRUCK TEH
 [0x1EEB6] = 0x062B, -- ARABIC MATHEMATICAL DOUBLE-STRUCK THEH
 [0x1EEB7] = 0x062E, -- ARABIC MATHEMATICAL DOUBLE-STRUCK KHAH
 [0x1EEB8] = 0x0630, -- ARABIC MATHEMATICAL DOUBLE-STRUCK THAL
 [0x1EEB9] = 0x0636, -- ARABIC MATHEMATICAL DOUBLE-STRUCK DAD
 [0x1EEBA] = 0x0638, -- ARABIC MATHEMATICAL DOUBLE-STRUCK ZAH
 [0x1EEBB] = 0x063A, -- ARABIC MATHEMATICAL DOUBLE-STRUCK GHAIN
 -- NA   <reserved-1EE9C>..<reserved-1EEA0>
 [0x1EE9C] = {status="disallowed"},
 [0x1EE9D] = {status="disallowed"},
 [0x1EE9E] = {status="disallowed"},
 [0x1EE9F] = {status="disallowed"},
 [0x1EEA0] = {status="disallowed"},
 -- ARABIC MATHEMATICAL OPERATOR MEEM WITH HAH WITH TATWEEL..ARABIC MATHEMATICAL OPERATOR HAH WITH DAL
 [0x1EEF0] = {status="disallowed"},
 [0x1EEF1] = {status="disallowed"},
 -- NA   <reserved-1F02C>..<reserved-1F02F>
 [0x1F02C] = {status="disallowed"},
 [0x1F02D] = {status="disallowed"},
 [0x1F02E] = {status="disallowed"},
 [0x1F02F] = {status="disallowed"},
 -- NA   <reserved-1F0AF>..<reserved-1F0B0>
 [0x1F0AF] = {status="disallowed"},
 [0x1F0B0] = {status="disallowed"},
 [0x1F0BF] = {status="disallowed"}, -- PLAYING CARD RED JOKER
 [0x1F0C0] = {status="disallowed"}, -- NA   <reserved-1F0C0>
 [0x1F0D0] = {status="disallowed"}, -- NA   <reserved-1F0D0>
 [0x1F100] = {status="disallowed"}, -- DIGIT ZERO FULL STOP
 [0x1F101] = {status="disallowed"}, -- DIGIT ZERO COMMA
 [0x1F102] = {status="disallowed"}, -- DIGIT ONE COMMA
 [0x1F103] = {status="disallowed"}, -- DIGIT TWO COMMA
 [0x1F104] = {status="disallowed"}, -- DIGIT THREE COMMA
 [0x1F105] = {status="disallowed"}, -- DIGIT FOUR COMMA
 [0x1F106] = {status="disallowed"}, -- DIGIT FIVE COMMA
 [0x1F107] = {status="disallowed"}, -- DIGIT SIX COMMA
 [0x1F108] = {status="disallowed"}, -- DIGIT SEVEN COMMA
 [0x1F109] = {status="disallowed"}, -- DIGIT EIGHT COMMA
 [0x1F10A] = {status="disallowed"}, -- DIGIT NINE COMMA
 -- DINGBAT CIRCLED SANS-SERIF DIGIT ZERO..DINGBAT NEGATIVE CIRCLED SANS-SERIF DIGIT ZERO
 [0x1F10B] = {status="disallowed"},
 [0x1F10C] = {status="disallowed"},
 [0x1F110] = {status="disallowed"}, -- PARENTHESIZED LATIN CAPITAL LETTER A
 [0x1F111] = {status="disallowed"}, -- PARENTHESIZED LATIN CAPITAL LETTER B
 [0x1F112] = {status="disallowed"}, -- PARENTHESIZED LATIN CAPITAL LETTER C
 [0x1F113] = {status="disallowed"}, -- PARENTHESIZED LATIN CAPITAL LETTER D
 [0x1F114] = {status="disallowed"}, -- PARENTHESIZED LATIN CAPITAL LETTER E
 [0x1F115] = {status="disallowed"}, -- PARENTHESIZED LATIN CAPITAL LETTER F
 [0x1F116] = {status="disallowed"}, -- PARENTHESIZED LATIN CAPITAL LETTER G
 [0x1F117] = {status="disallowed"}, -- PARENTHESIZED LATIN CAPITAL LETTER H
 [0x1F118] = {status="disallowed"}, -- PARENTHESIZED LATIN CAPITAL LETTER I
 [0x1F119] = {status="disallowed"}, -- PARENTHESIZED LATIN CAPITAL LETTER J
 [0x1F11A] = {status="disallowed"}, -- PARENTHESIZED LATIN CAPITAL LETTER K
 [0x1F11B] = {status="disallowed"}, -- PARENTHESIZED LATIN CAPITAL LETTER L
 [0x1F11C] = {status="disallowed"}, -- PARENTHESIZED LATIN CAPITAL LETTER M
 [0x1F11D] = {status="disallowed"}, -- PARENTHESIZED LATIN CAPITAL LETTER N
 [0x1F11E] = {status="disallowed"}, -- PARENTHESIZED LATIN CAPITAL LETTER O
 [0x1F11F] = {status="disallowed"}, -- PARENTHESIZED LATIN CAPITAL LETTER P
 [0x1F120] = {status="disallowed"}, -- PARENTHESIZED LATIN CAPITAL LETTER Q
 [0x1F121] = {status="disallowed"}, -- PARENTHESIZED LATIN CAPITAL LETTER R
 [0x1F122] = {status="disallowed"}, -- PARENTHESIZED LATIN CAPITAL LETTER S
 [0x1F123] = {status="disallowed"}, -- PARENTHESIZED LATIN CAPITAL LETTER T
 [0x1F124] = {status="disallowed"}, -- PARENTHESIZED LATIN CAPITAL LETTER U
 [0x1F125] = {status="disallowed"}, -- PARENTHESIZED LATIN CAPITAL LETTER V
 [0x1F126] = {status="disallowed"}, -- PARENTHESIZED LATIN CAPITAL LETTER W
 [0x1F127] = {status="disallowed"}, -- PARENTHESIZED LATIN CAPITAL LETTER X
 [0x1F128] = {status="disallowed"}, -- PARENTHESIZED LATIN CAPITAL LETTER Y
 [0x1F129] = {status="disallowed"}, -- PARENTHESIZED LATIN CAPITAL LETTER Z
 [0x1F12A] = {0x3014, 0x0073, 0x3015}, -- TORTOISE SHELL BRACKETED LATIN CAPITAL LETTER S
 [0x1F12B] = 0x0063, -- CIRCLED ITALIC LATIN CAPITAL LETTER C
 [0x1F12C] = 0x0072, -- CIRCLED ITALIC LATIN CAPITAL LETTER R
 [0x1F12D] = {0x0063, 0x0064}, -- CIRCLED CD
 [0x1F12E] = {0x0077, 0x007A}, -- CIRCLED WZ
 [0x1F12F] = {status="disallowed"}, -- NA   <reserved-1F12F>
 [0x1F130] = 0x0061, -- SQUARED LATIN CAPITAL LETTER A
 [0x1F131] = 0x0062, -- SQUARED LATIN CAPITAL LETTER B
 [0x1F132] = 0x0063, -- SQUARED LATIN CAPITAL LETTER C
 [0x1F133] = 0x0064, -- SQUARED LATIN CAPITAL LETTER D
 [0x1F134] = 0x0065, -- SQUARED LATIN CAPITAL LETTER E
 [0x1F135] = 0x0066, -- SQUARED LATIN CAPITAL LETTER F
 [0x1F136] = 0x0067, -- SQUARED LATIN CAPITAL LETTER G
 [0x1F137] = 0x0068, -- SQUARED LATIN CAPITAL LETTER H
 [0x1F138] = 0x0069, -- SQUARED LATIN CAPITAL LETTER I
 [0x1F139] = 0x006A, -- SQUARED LATIN CAPITAL LETTER J
 [0x1F13A] = 0x006B, -- SQUARED LATIN CAPITAL LETTER K
 [0x1F13B] = 0x006C, -- SQUARED LATIN CAPITAL LETTER L
 [0x1F13C] = 0x006D, -- SQUARED LATIN CAPITAL LETTER M
 [0x1F13D] = 0x006E, -- SQUARED LATIN CAPITAL LETTER N
 [0x1F13E] = 0x006F, -- SQUARED LATIN CAPITAL LETTER O
 [0x1F13F] = 0x0070, -- SQUARED LATIN CAPITAL LETTER P
 [0x1F140] = 0x0071, -- SQUARED LATIN CAPITAL LETTER Q
 [0x1F141] = 0x0072, -- SQUARED LATIN CAPITAL LETTER R
 [0x1F142] = 0x0073, -- SQUARED LATIN CAPITAL LETTER S
 [0x1F143] = 0x0074, -- SQUARED LATIN CAPITAL LETTER T
 [0x1F144] = 0x0075, -- SQUARED LATIN CAPITAL LETTER U
 [0x1F145] = 0x0076, -- SQUARED LATIN CAPITAL LETTER V
 [0x1F146] = 0x0077, -- SQUARED LATIN CAPITAL LETTER W
 [0x1F147] = 0x0078, -- SQUARED LATIN CAPITAL LETTER X
 [0x1F148] = 0x0079, -- SQUARED LATIN CAPITAL LETTER Y
 [0x1F149] = 0x007A, -- SQUARED LATIN CAPITAL LETTER Z
 [0x1F14A] = {0x0068, 0x0076}, -- SQUARED HV
 [0x1F14B] = {0x006D, 0x0076}, -- SQUARED MV
 [0x1F14C] = {0x0073, 0x0064}, -- SQUARED SD
 [0x1F14D] = {0x0073, 0x0073}, -- SQUARED SS
 [0x1F14E] = {0x0070, 0x0070, 0x0076}, -- SQUARED PPV
 [0x1F14F] = {0x0077, 0x0063}, -- SQUARED WC
 -- NA   <reserved-1F10D>..<reserved-1F10F>
 [0x1F10D] = {status="disallowed"},
 [0x1F10E] = {status="disallowed"},
 [0x1F10F] = {status="disallowed"},
 [0x1F157] = {status="disallowed"}, -- NEGATIVE CIRCLED LATIN CAPITAL LETTER H
 [0x1F15F] = {status="disallowed"}, -- NEGATIVE CIRCLED LATIN CAPITAL LETTER P
 [0x1F16A] = {0x006D, 0x0063}, -- RAISED MC SIGN
 [0x1F16B] = {0x006D, 0x0064}, -- RAISED MD SIGN
 -- NA   <reserved-1F16C>..<reserved-1F16F>
 [0x1F16C] = {status="disallowed"},
 [0x1F16D] = {status="disallowed"},
 [0x1F16E] = {status="disallowed"},
 [0x1F16F] = {status="disallowed"},
 [0x1F179] = {status="disallowed"}, -- NEGATIVE SQUARED LATIN CAPITAL LETTER J
 [0x1F17A] = {status="disallowed"}, -- NEGATIVE SQUARED LATIN CAPITAL LETTER K
 [0x1F17F] = {status="disallowed"}, -- NEGATIVE SQUARED LATIN CAPITAL LETTER P
 -- NEGATIVE SQUARED LATIN CAPITAL LETTER L..NEGATIVE SQUARED LATIN CAPITAL LETTER M
      -- NEGATIVE SQUARED LATIN CAPITAL LETTER N..NEGATIVE SQUARED LATIN CAPITAL LETTER O
 [0x1F17B] = {status="disallowed"},
 [0x1F17C] = {status="disallowed"},
 [0x1F17D] = {status="disallowed"},
 [0x1F17E] = {status="disallowed"},
 [0x1F190] = {0x0064, 0x006A}, -- SQUARE DJ
 [0x1F200] = {0x307B, 0x304B}, -- SQUARE HIRAGANA HOKA
 [0x1F201] = {0x30B3, 0x30B3}, -- SQUARED KATAKANA KOKO
 [0x1F202] = 0x30B5, -- SQUARED KATAKANA SA
 [0x1F210] = 0x624B, -- SQUARED CJK UNIFIED IDEOGRAPH-624B
 [0x1F211] = 0x5B57, -- SQUARED CJK UNIFIED IDEOGRAPH-5B57
 [0x1F212] = 0x53CC, -- SQUARED CJK UNIFIED IDEOGRAPH-53CC
 [0x1F213] = 0x30C7, -- SQUARED KATAKANA DE
 [0x1F214] = 0x4E8C, -- SQUARED CJK UNIFIED IDEOGRAPH-4E8C
 [0x1F215] = 0x591A, -- SQUARED CJK UNIFIED IDEOGRAPH-591A
 [0x1F216] = 0x89E3, -- SQUARED CJK UNIFIED IDEOGRAPH-89E3
 [0x1F217] = 0x5929, -- SQUARED CJK UNIFIED IDEOGRAPH-5929
 [0x1F218] = 0x4EA4, -- SQUARED CJK UNIFIED IDEOGRAPH-4EA4
 [0x1F219] = 0x6620, -- SQUARED CJK UNIFIED IDEOGRAPH-6620
 [0x1F21A] = 0x7121, -- SQUARED CJK UNIFIED IDEOGRAPH-7121
 [0x1F21B] = 0x6599, -- SQUARED CJK UNIFIED IDEOGRAPH-6599
 [0x1F21C] = 0x524D, -- SQUARED CJK UNIFIED IDEOGRAPH-524D
 [0x1F21D] = 0x5F8C, -- SQUARED CJK UNIFIED IDEOGRAPH-5F8C
 [0x1F21E] = 0x518D, -- SQUARED CJK UNIFIED IDEOGRAPH-518D
 [0x1F21F] = 0x65B0, -- SQUARED CJK UNIFIED IDEOGRAPH-65B0
 [0x1F220] = 0x521D, -- SQUARED CJK UNIFIED IDEOGRAPH-521D
 [0x1F221] = 0x7D42, -- SQUARED CJK UNIFIED IDEOGRAPH-7D42
 [0x1F222] = 0x751F, -- SQUARED CJK UNIFIED IDEOGRAPH-751F
 [0x1F223] = 0x8CA9, -- SQUARED CJK UNIFIED IDEOGRAPH-8CA9
 [0x1F224] = 0x58F0, -- SQUARED CJK UNIFIED IDEOGRAPH-58F0
 [0x1F225] = 0x5439, -- SQUARED CJK UNIFIED IDEOGRAPH-5439
 [0x1F226] = 0x6F14, -- SQUARED CJK UNIFIED IDEOGRAPH-6F14
 [0x1F227] = 0x6295, -- SQUARED CJK UNIFIED IDEOGRAPH-6295
 [0x1F228] = 0x6355, -- SQUARED CJK UNIFIED IDEOGRAPH-6355
 [0x1F229] = 0x4E00, -- SQUARED CJK UNIFIED IDEOGRAPH-4E00
 [0x1F22A] = 0x4E09, -- SQUARED CJK UNIFIED IDEOGRAPH-4E09
 [0x1F22B] = 0x904A, -- SQUARED CJK UNIFIED IDEOGRAPH-904A
 [0x1F22C] = 0x5DE6, -- SQUARED CJK UNIFIED IDEOGRAPH-5DE6
 [0x1F22D] = 0x4E2D, -- SQUARED CJK UNIFIED IDEOGRAPH-4E2D
 [0x1F22E] = 0x53F3, -- SQUARED CJK UNIFIED IDEOGRAPH-53F3
 [0x1F22F] = 0x6307, -- SQUARED CJK UNIFIED IDEOGRAPH-6307
 [0x1F230] = 0x8D70, -- SQUARED CJK UNIFIED IDEOGRAPH-8D70
 [0x1F231] = 0x6253, -- SQUARED CJK UNIFIED IDEOGRAPH-6253
 [0x1F232] = 0x7981, -- SQUARED CJK UNIFIED IDEOGRAPH-7981
 [0x1F233] = 0x7A7A, -- SQUARED CJK UNIFIED IDEOGRAPH-7A7A
 [0x1F234] = 0x5408, -- SQUARED CJK UNIFIED IDEOGRAPH-5408
 [0x1F235] = 0x6E80, -- SQUARED CJK UNIFIED IDEOGRAPH-6E80
 [0x1F236] = 0x6709, -- SQUARED CJK UNIFIED IDEOGRAPH-6709
 [0x1F237] = 0x6708, -- SQUARED CJK UNIFIED IDEOGRAPH-6708
 [0x1F238] = 0x7533, -- SQUARED CJK UNIFIED IDEOGRAPH-7533
 [0x1F239] = 0x5272, -- SQUARED CJK UNIFIED IDEOGRAPH-5272
 [0x1F23A] = 0x55B6, -- SQUARED CJK UNIFIED IDEOGRAPH-55B6
 [0x1F23B] = 0x914D, -- SQUARED CJK UNIFIED IDEOGRAPH-914D
 [0x1F240] = {0x3014, 0x672C, 0x3015}, -- TORTOISE SHELL BRACKETED CJK UNIFIED IDEOGRAPH-672C
 [0x1F241] = {0x3014, 0x4E09, 0x3015}, -- TORTOISE SHELL BRACKETED CJK UNIFIED IDEOGRAPH-4E09
 [0x1F242] = {0x3014, 0x4E8C, 0x3015}, -- TORTOISE SHELL BRACKETED CJK UNIFIED IDEOGRAPH-4E8C
 [0x1F243] = {0x3014, 0x5B89, 0x3015}, -- TORTOISE SHELL BRACKETED CJK UNIFIED IDEOGRAPH-5B89
 [0x1F244] = {0x3014, 0x70B9, 0x3015}, -- TORTOISE SHELL BRACKETED CJK UNIFIED IDEOGRAPH-70B9
 [0x1F245] = {0x3014, 0x6253, 0x3015}, -- TORTOISE SHELL BRACKETED CJK UNIFIED IDEOGRAPH-6253
 [0x1F246] = {0x3014, 0x76D7, 0x3015}, -- TORTOISE SHELL BRACKETED CJK UNIFIED IDEOGRAPH-76D7
 [0x1F247] = {0x3014, 0x52DD, 0x3015}, -- TORTOISE SHELL BRACKETED CJK UNIFIED IDEOGRAPH-52DD
 [0x1F248] = {0x3014, 0x6557, 0x3015}, -- TORTOISE SHELL BRACKETED CJK UNIFIED IDEOGRAPH-6557
 -- NA   <reserved-1F23C>..<reserved-1F23F>
 [0x1F23C] = {status="disallowed"},
 [0x1F23D] = {status="disallowed"},
 [0x1F23E] = {status="disallowed"},
 [0x1F23F] = {status="disallowed"},
 [0x1F250] = 0x5F97, -- CIRCLED IDEOGRAPH ADVANTAGE
 [0x1F251] = 0x53EF, -- CIRCLED IDEOGRAPH ACCEPT
 [0x1F336] = {status="disallowed"}, -- HOT PEPPER
 [0x1F37D] = {status="disallowed"}, -- FORK AND KNIFE WITH PLATE
 [0x1F3C5] = {status="disallowed"}, -- SPORTS MEDAL
 [0x1F43F] = {status="disallowed"}, -- CHIPMUNK
 [0x1F440] = {status="disallowed"}, -- EYES
 [0x1F441] = {status="disallowed"}, -- EYE
 [0x1F4F8] = {status="disallowed"}, -- CAMERA WITH FLASH
 [0x1F4FF] = {status="disallowed"}, -- PRAYER BEADS
 [0x1F57A] = {status="disallowed"}, -- MAN DANCING
 [0x1F5A4] = {status="disallowed"}, -- BLACK HEART
 [0x1F600] = {status="disallowed"}, -- GRINNING FACE
 [0x1F611] = {status="disallowed"}, -- EXPRESSIONLESS FACE
 [0x1F615] = {status="disallowed"}, -- CONFUSED FACE
 [0x1F616] = {status="disallowed"}, -- CONFOUNDED FACE
 [0x1F617] = {status="disallowed"}, -- KISSING FACE
 [0x1F618] = {status="disallowed"}, -- FACE THROWING A KISS
 [0x1F619] = {status="disallowed"}, -- KISSING FACE WITH SMILING EYES
 [0x1F61A] = {status="disallowed"}, -- KISSING FACE WITH CLOSED EYES
 [0x1F61B] = {status="disallowed"}, -- FACE WITH STUCK-OUT TONGUE
 -- UNAMUSED FACE..PENSIVE FACE
 [0x1F612] = {status="disallowed"},
 [0x1F613] = {status="disallowed"},
 [0x1F614] = {status="disallowed"},
 [0x1F61F] = {status="disallowed"}, -- WORRIED FACE
 -- FACE WITH STUCK-OUT TONGUE AND WINKING EYE..DISAPPOINTED FACE
 [0x1F61C] = {status="disallowed"},
 [0x1F61D] = {status="disallowed"},
 [0x1F61E] = {status="disallowed"},
 [0x1F62C] = {status="disallowed"}, -- GRIMACING FACE
 [0x1F62D] = {status="disallowed"}, -- LOUDLY CRYING FACE
 [0x1F634] = {status="disallowed"}, -- SLEEPING FACE
 [0x1F6D0] = {status="disallowed"}, -- PLACE OF WORSHIP
 -- OCTAGONAL SIGN..SHOPPING TROLLEY
      -- STUPA..PAGODA
 [0x1F6D1] = {status="disallowed"},
 [0x1F6D2] = {status="disallowed"},
 [0x1F6D3] = {status="disallowed"},
 [0x1F6D4] = {status="disallowed"},
 -- NA   <reserved-1F6ED>..<reserved-1F6EF>
 [0x1F6ED] = {status="disallowed"},
 [0x1F6EE] = {status="disallowed"},
 [0x1F6EF] = {status="disallowed"},
 -- NA   <reserved-1F80C>..<reserved-1F80F>
 [0x1F80C] = {status="disallowed"},
 [0x1F80D] = {status="disallowed"},
 [0x1F80E] = {status="disallowed"},
 [0x1F80F] = {status="disallowed"},
 -- NA   <reserved-1F90C>..<reserved-1F90F>
 [0x1F90C] = {status="disallowed"},
 [0x1F90D] = {status="disallowed"},
 [0x1F90E] = {status="disallowed"},
 [0x1F90F] = {status="disallowed"},
 [0x1F91F] = {status="disallowed"}, -- I LOVE YOU HAND SIGN
 [0x1F930] = {status="disallowed"}, -- PREGNANT WOMAN
 [0x1F93F] = {status="disallowed"}, -- NA   <reserved-1F93F>
 [0x1F94C] = {status="disallowed"}, -- CURLING STONE
 -- NA   <reserved-1F94D>..<reserved-1F94F>
 [0x1F94D] = {status="disallowed"},
 [0x1F94E] = {status="disallowed"},
 [0x1F94F] = {status="disallowed"},
 [0x1F9C0] = {status="disallowed"}, -- CHEESE WEDGE
 -- NA   <reserved-2B81E>..<reserved-2B81F>
 [0x2B81E] = {status="disallowed"},
 [0x2B81F] = {status="disallowed"},
 [0x2F800] = 0x4E3D, -- CJK COMPATIBILITY IDEOGRAPH-2F800
 [0x2F801] = 0x4E38, -- CJK COMPATIBILITY IDEOGRAPH-2F801
 [0x2F802] = 0x4E41, -- CJK COMPATIBILITY IDEOGRAPH-2F802
 [0x2F803] = 0x20122, -- CJK COMPATIBILITY IDEOGRAPH-2F803
 [0x2F804] = 0x4F60, -- CJK COMPATIBILITY IDEOGRAPH-2F804
 [0x2F805] = 0x4FAE, -- CJK COMPATIBILITY IDEOGRAPH-2F805
 [0x2F806] = 0x4FBB, -- CJK COMPATIBILITY IDEOGRAPH-2F806
 [0x2F807] = 0x5002, -- CJK COMPATIBILITY IDEOGRAPH-2F807
 [0x2F808] = 0x507A, -- CJK COMPATIBILITY IDEOGRAPH-2F808
 [0x2F809] = 0x5099, -- CJK COMPATIBILITY IDEOGRAPH-2F809
 [0x2F80A] = 0x50E7, -- CJK COMPATIBILITY IDEOGRAPH-2F80A
 [0x2F80B] = 0x50CF, -- CJK COMPATIBILITY IDEOGRAPH-2F80B
 [0x2F80C] = 0x349E, -- CJK COMPATIBILITY IDEOGRAPH-2F80C
 [0x2F80D] = 0x2063A, -- CJK COMPATIBILITY IDEOGRAPH-2F80D
 [0x2F80E] = 0x514D, -- CJK COMPATIBILITY IDEOGRAPH-2F80E
 [0x2F80F] = 0x5154, -- CJK COMPATIBILITY IDEOGRAPH-2F80F
 [0x2F810] = 0x5164, -- CJK COMPATIBILITY IDEOGRAPH-2F810
 [0x2F811] = 0x5177, -- CJK COMPATIBILITY IDEOGRAPH-2F811
 [0x2F812] = 0x2051C, -- CJK COMPATIBILITY IDEOGRAPH-2F812
 [0x2F813] = 0x34B9, -- CJK COMPATIBILITY IDEOGRAPH-2F813
 [0x2F814] = 0x5167, -- CJK COMPATIBILITY IDEOGRAPH-2F814
 [0x2F815] = 0x518D, -- CJK COMPATIBILITY IDEOGRAPH-2F815
 [0x2F816] = 0x2054B, -- CJK COMPATIBILITY IDEOGRAPH-2F816
 [0x2F817] = 0x5197, -- CJK COMPATIBILITY IDEOGRAPH-2F817
 [0x2F818] = 0x51A4, -- CJK COMPATIBILITY IDEOGRAPH-2F818
 [0x2F819] = 0x4ECC, -- CJK COMPATIBILITY IDEOGRAPH-2F819
 [0x2F81A] = 0x51AC, -- CJK COMPATIBILITY IDEOGRAPH-2F81A
 [0x2F81B] = 0x51B5, -- CJK COMPATIBILITY IDEOGRAPH-2F81B
 [0x2F81C] = 0x291DF, -- CJK COMPATIBILITY IDEOGRAPH-2F81C
 [0x2F81D] = 0x51F5, -- CJK COMPATIBILITY IDEOGRAPH-2F81D
 [0x2F81E] = 0x5203, -- CJK COMPATIBILITY IDEOGRAPH-2F81E
 [0x2F81F] = 0x34DF, -- CJK COMPATIBILITY IDEOGRAPH-2F81F
 [0x2F820] = 0x523B, -- CJK COMPATIBILITY IDEOGRAPH-2F820
 [0x2F821] = 0x5246, -- CJK COMPATIBILITY IDEOGRAPH-2F821
 [0x2F822] = 0x5272, -- CJK COMPATIBILITY IDEOGRAPH-2F822
 [0x2F823] = 0x5277, -- CJK COMPATIBILITY IDEOGRAPH-2F823
 [0x2F824] = 0x3515, -- CJK COMPATIBILITY IDEOGRAPH-2F824
 [0x2F825] = 0x52C7, -- CJK COMPATIBILITY IDEOGRAPH-2F825
 [0x2F826] = 0x52C9, -- CJK COMPATIBILITY IDEOGRAPH-2F826
 [0x2F827] = 0x52E4, -- CJK COMPATIBILITY IDEOGRAPH-2F827
 [0x2F828] = 0x52FA, -- CJK COMPATIBILITY IDEOGRAPH-2F828
 [0x2F829] = 0x5305, -- CJK COMPATIBILITY IDEOGRAPH-2F829
 [0x2F82A] = 0x5306, -- CJK COMPATIBILITY IDEOGRAPH-2F82A
 [0x2F82B] = 0x5317, -- CJK COMPATIBILITY IDEOGRAPH-2F82B
 [0x2F82C] = 0x5349, -- CJK COMPATIBILITY IDEOGRAPH-2F82C
 [0x2F82D] = 0x5351, -- CJK COMPATIBILITY IDEOGRAPH-2F82D
 [0x2F82E] = 0x535A, -- CJK COMPATIBILITY IDEOGRAPH-2F82E
 [0x2F82F] = 0x5373, -- CJK COMPATIBILITY IDEOGRAPH-2F82F
 [0x2F830] = 0x537D, -- CJK COMPATIBILITY IDEOGRAPH-2F830
 [0x2F834] = 0x20A2C, -- CJK COMPATIBILITY IDEOGRAPH-2F834
 [0x2F835] = 0x7070, -- CJK COMPATIBILITY IDEOGRAPH-2F835
 [0x2F836] = 0x53CA, -- CJK COMPATIBILITY IDEOGRAPH-2F836
 [0x2F837] = 0x53DF, -- CJK COMPATIBILITY IDEOGRAPH-2F837
 [0x2F838] = 0x20B63, -- CJK COMPATIBILITY IDEOGRAPH-2F838
 [0x2F839] = 0x53EB, -- CJK COMPATIBILITY IDEOGRAPH-2F839
 [0x2F83A] = 0x53F1, -- CJK COMPATIBILITY IDEOGRAPH-2F83A
 [0x2F83B] = 0x5406, -- CJK COMPATIBILITY IDEOGRAPH-2F83B
 [0x2F83C] = 0x549E, -- CJK COMPATIBILITY IDEOGRAPH-2F83C
 [0x2F83D] = 0x5438, -- CJK COMPATIBILITY IDEOGRAPH-2F83D
 [0x2F83E] = 0x5448, -- CJK COMPATIBILITY IDEOGRAPH-2F83E
 [0x2F83F] = 0x5468, -- CJK COMPATIBILITY IDEOGRAPH-2F83F
 [0x2F840] = 0x54A2, -- CJK COMPATIBILITY IDEOGRAPH-2F840
 [0x2F841] = 0x54F6, -- CJK COMPATIBILITY IDEOGRAPH-2F841
 [0x2F842] = 0x5510, -- CJK COMPATIBILITY IDEOGRAPH-2F842
 [0x2F843] = 0x5553, -- CJK COMPATIBILITY IDEOGRAPH-2F843
 [0x2F844] = 0x5563, -- CJK COMPATIBILITY IDEOGRAPH-2F844
 -- CJK COMPATIBILITY IDEOGRAPH-2F831..CJK COMPATIBILITY IDEOGRAPH-2F833
 [0x2F831] = 0x537F,
 [0x2F832] = 0x537F,
 [0x2F833] = 0x537F,
 [0x2F847] = 0x5599, -- CJK COMPATIBILITY IDEOGRAPH-2F847
 [0x2F848] = 0x55AB, -- CJK COMPATIBILITY IDEOGRAPH-2F848
 [0x2F849] = 0x55B3, -- CJK COMPATIBILITY IDEOGRAPH-2F849
 [0x2F84A] = 0x55C2, -- CJK COMPATIBILITY IDEOGRAPH-2F84A
 [0x2F84B] = 0x5716, -- CJK COMPATIBILITY IDEOGRAPH-2F84B
 [0x2F84C] = 0x5606, -- CJK COMPATIBILITY IDEOGRAPH-2F84C
 [0x2F84D] = 0x5717, -- CJK COMPATIBILITY IDEOGRAPH-2F84D
 [0x2F84E] = 0x5651, -- CJK COMPATIBILITY IDEOGRAPH-2F84E
 [0x2F84F] = 0x5674, -- CJK COMPATIBILITY IDEOGRAPH-2F84F
 [0x2F850] = 0x5207, -- CJK COMPATIBILITY IDEOGRAPH-2F850
 [0x2F851] = 0x58EE, -- CJK COMPATIBILITY IDEOGRAPH-2F851
 [0x2F852] = 0x57CE, -- CJK COMPATIBILITY IDEOGRAPH-2F852
 [0x2F853] = 0x57F4, -- CJK COMPATIBILITY IDEOGRAPH-2F853
 [0x2F854] = 0x580D, -- CJK COMPATIBILITY IDEOGRAPH-2F854
 [0x2F855] = 0x578B, -- CJK COMPATIBILITY IDEOGRAPH-2F855
 [0x2F856] = 0x5832, -- CJK COMPATIBILITY IDEOGRAPH-2F856
 [0x2F857] = 0x5831, -- CJK COMPATIBILITY IDEOGRAPH-2F857
 [0x2F858] = 0x58AC, -- CJK COMPATIBILITY IDEOGRAPH-2F858
 [0x2F859] = 0x214E4, -- CJK COMPATIBILITY IDEOGRAPH-2F859
 [0x2F85A] = 0x58F2, -- CJK COMPATIBILITY IDEOGRAPH-2F85A
 [0x2F85B] = 0x58F7, -- CJK COMPATIBILITY IDEOGRAPH-2F85B
 [0x2F85C] = 0x5906, -- CJK COMPATIBILITY IDEOGRAPH-2F85C
 [0x2F85D] = 0x591A, -- CJK COMPATIBILITY IDEOGRAPH-2F85D
 [0x2F85E] = 0x5922, -- CJK COMPATIBILITY IDEOGRAPH-2F85E
 [0x2F85F] = 0x5962, -- CJK COMPATIBILITY IDEOGRAPH-2F85F
 [0x2F860] = 0x216A8, -- CJK COMPATIBILITY IDEOGRAPH-2F860
 [0x2F861] = 0x216EA, -- CJK COMPATIBILITY IDEOGRAPH-2F861
 [0x2F862] = 0x59EC, -- CJK COMPATIBILITY IDEOGRAPH-2F862
 [0x2F863] = 0x5A1B, -- CJK COMPATIBILITY IDEOGRAPH-2F863
 [0x2F864] = 0x5A27, -- CJK COMPATIBILITY IDEOGRAPH-2F864
 [0x2F865] = 0x59D8, -- CJK COMPATIBILITY IDEOGRAPH-2F865
 [0x2F866] = 0x5A66, -- CJK COMPATIBILITY IDEOGRAPH-2F866
 [0x2F867] = 0x36EE, -- CJK COMPATIBILITY IDEOGRAPH-2F867
 [0x2F868] = {status="disallowed"}, -- CJK COMPATIBILITY IDEOGRAPH-2F868
 [0x2F869] = 0x5B08, -- CJK COMPATIBILITY IDEOGRAPH-2F869
 -- CJK COMPATIBILITY IDEOGRAPH-2F845..CJK COMPATIBILITY IDEOGRAPH-2F846
 [0x2F845] = 0x5584,
 [0x2F846] = 0x5584,
 [0x2F86C] = 0x219C8, -- CJK COMPATIBILITY IDEOGRAPH-2F86C
 [0x2F86D] = 0x5BC3, -- CJK COMPATIBILITY IDEOGRAPH-2F86D
 [0x2F86E] = 0x5BD8, -- CJK COMPATIBILITY IDEOGRAPH-2F86E
 [0x2F86F] = 0x5BE7, -- CJK COMPATIBILITY IDEOGRAPH-2F86F
 [0x2F870] = 0x5BF3, -- CJK COMPATIBILITY IDEOGRAPH-2F870
 [0x2F871] = 0x21B18, -- CJK COMPATIBILITY IDEOGRAPH-2F871
 [0x2F872] = 0x5BFF, -- CJK COMPATIBILITY IDEOGRAPH-2F872
 [0x2F873] = 0x5C06, -- CJK COMPATIBILITY IDEOGRAPH-2F873
 [0x2F874] = {status="disallowed"}, -- CJK COMPATIBILITY IDEOGRAPH-2F874
 [0x2F875] = 0x5C22, -- CJK COMPATIBILITY IDEOGRAPH-2F875
 [0x2F876] = 0x3781, -- CJK COMPATIBILITY IDEOGRAPH-2F876
 [0x2F877] = 0x5C60, -- CJK COMPATIBILITY IDEOGRAPH-2F877
 [0x2F878] = 0x5C6E, -- CJK COMPATIBILITY IDEOGRAPH-2F878
 [0x2F879] = 0x5CC0, -- CJK COMPATIBILITY IDEOGRAPH-2F879
 [0x2F87A] = 0x5C8D, -- CJK COMPATIBILITY IDEOGRAPH-2F87A
 [0x2F87B] = 0x21DE4, -- CJK COMPATIBILITY IDEOGRAPH-2F87B
 [0x2F87C] = 0x5D43, -- CJK COMPATIBILITY IDEOGRAPH-2F87C
 [0x2F87D] = 0x21DE6, -- CJK COMPATIBILITY IDEOGRAPH-2F87D
 [0x2F87E] = 0x5D6E, -- CJK COMPATIBILITY IDEOGRAPH-2F87E
 [0x2F87F] = 0x5D6B, -- CJK COMPATIBILITY IDEOGRAPH-2F87F
 [0x2F880] = 0x5D7C, -- CJK COMPATIBILITY IDEOGRAPH-2F880
 [0x2F881] = 0x5DE1, -- CJK COMPATIBILITY IDEOGRAPH-2F881
 [0x2F882] = 0x5DE2, -- CJK COMPATIBILITY IDEOGRAPH-2F882
 [0x2F883] = 0x382F, -- CJK COMPATIBILITY IDEOGRAPH-2F883
 [0x2F884] = 0x5DFD, -- CJK COMPATIBILITY IDEOGRAPH-2F884
 [0x2F885] = 0x5E28, -- CJK COMPATIBILITY IDEOGRAPH-2F885
 [0x2F886] = 0x5E3D, -- CJK COMPATIBILITY IDEOGRAPH-2F886
 [0x2F887] = 0x5E69, -- CJK COMPATIBILITY IDEOGRAPH-2F887
 [0x2F888] = 0x3862, -- CJK COMPATIBILITY IDEOGRAPH-2F888
 [0x2F889] = 0x22183, -- CJK COMPATIBILITY IDEOGRAPH-2F889
 [0x2F88A] = 0x387C, -- CJK COMPATIBILITY IDEOGRAPH-2F88A
 [0x2F88B] = 0x5EB0, -- CJK COMPATIBILITY IDEOGRAPH-2F88B
 [0x2F88C] = 0x5EB3, -- CJK COMPATIBILITY IDEOGRAPH-2F88C
 [0x2F88D] = 0x5EB6, -- CJK COMPATIBILITY IDEOGRAPH-2F88D
 [0x2F88E] = 0x5ECA, -- CJK COMPATIBILITY IDEOGRAPH-2F88E
 [0x2F88F] = 0x2A392, -- CJK COMPATIBILITY IDEOGRAPH-2F88F
 [0x2F890] = 0x5EFE, -- CJK COMPATIBILITY IDEOGRAPH-2F890
 -- CJK COMPATIBILITY IDEOGRAPH-2F86A..CJK COMPATIBILITY IDEOGRAPH-2F86B
 [0x2F86A] = 0x5B3E,
 [0x2F86B] = 0x5B3E,
 [0x2F893] = 0x8201, -- CJK COMPATIBILITY IDEOGRAPH-2F893
 -- CJK COMPATIBILITY IDEOGRAPH-2F891..CJK COMPATIBILITY IDEOGRAPH-2F892
 [0x2F891] = 0x22331,
 [0x2F892] = 0x22331,
 [0x2F896] = 0x38C7, -- CJK COMPATIBILITY IDEOGRAPH-2F896
 [0x2F897] = 0x232B8, -- CJK COMPATIBILITY IDEOGRAPH-2F897
 [0x2F898] = 0x261DA, -- CJK COMPATIBILITY IDEOGRAPH-2F898
 [0x2F899] = 0x5F62, -- CJK COMPATIBILITY IDEOGRAPH-2F899
 [0x2F89A] = 0x5F6B, -- CJK COMPATIBILITY IDEOGRAPH-2F89A
 [0x2F89B] = 0x38E3, -- CJK COMPATIBILITY IDEOGRAPH-2F89B
 [0x2F89C] = 0x5F9A, -- CJK COMPATIBILITY IDEOGRAPH-2F89C
 [0x2F89D] = 0x5FCD, -- CJK COMPATIBILITY IDEOGRAPH-2F89D
 [0x2F89E] = 0x5FD7, -- CJK COMPATIBILITY IDEOGRAPH-2F89E
 [0x2F89F] = 0x5FF9, -- CJK COMPATIBILITY IDEOGRAPH-2F89F
 [0x2F8A0] = 0x6081, -- CJK COMPATIBILITY IDEOGRAPH-2F8A0
 [0x2F8A1] = 0x393A, -- CJK COMPATIBILITY IDEOGRAPH-2F8A1
 [0x2F8A2] = 0x391C, -- CJK COMPATIBILITY IDEOGRAPH-2F8A2
 [0x2F8A3] = 0x6094, -- CJK COMPATIBILITY IDEOGRAPH-2F8A3
 [0x2F8A4] = 0x226D4, -- CJK COMPATIBILITY IDEOGRAPH-2F8A4
 [0x2F8A5] = 0x60C7, -- CJK COMPATIBILITY IDEOGRAPH-2F8A5
 [0x2F8A6] = 0x6148, -- CJK COMPATIBILITY IDEOGRAPH-2F8A6
 [0x2F8A7] = 0x614C, -- CJK COMPATIBILITY IDEOGRAPH-2F8A7
 [0x2F8A8] = 0x614E, -- CJK COMPATIBILITY IDEOGRAPH-2F8A8
 [0x2F8A9] = 0x614C, -- CJK COMPATIBILITY IDEOGRAPH-2F8A9
 [0x2F8AA] = 0x617A, -- CJK COMPATIBILITY IDEOGRAPH-2F8AA
 [0x2F8AB] = 0x618E, -- CJK COMPATIBILITY IDEOGRAPH-2F8AB
 [0x2F8AC] = 0x61B2, -- CJK COMPATIBILITY IDEOGRAPH-2F8AC
 [0x2F8AD] = 0x61A4, -- CJK COMPATIBILITY IDEOGRAPH-2F8AD
 [0x2F8AE] = 0x61AF, -- CJK COMPATIBILITY IDEOGRAPH-2F8AE
 [0x2F8AF] = 0x61DE, -- CJK COMPATIBILITY IDEOGRAPH-2F8AF
 [0x2F8B0] = 0x61F2, -- CJK COMPATIBILITY IDEOGRAPH-2F8B0
 [0x2F8B1] = 0x61F6, -- CJK COMPATIBILITY IDEOGRAPH-2F8B1
 [0x2F8B2] = 0x6210, -- CJK COMPATIBILITY IDEOGRAPH-2F8B2
 [0x2F8B3] = 0x621B, -- CJK COMPATIBILITY IDEOGRAPH-2F8B3
 [0x2F8B4] = 0x625D, -- CJK COMPATIBILITY IDEOGRAPH-2F8B4
 [0x2F8B5] = 0x62B1, -- CJK COMPATIBILITY IDEOGRAPH-2F8B5
 [0x2F8B6] = 0x62D4, -- CJK COMPATIBILITY IDEOGRAPH-2F8B6
 [0x2F8B7] = 0x6350, -- CJK COMPATIBILITY IDEOGRAPH-2F8B7
 [0x2F8B8] = 0x22B0C, -- CJK COMPATIBILITY IDEOGRAPH-2F8B8
 [0x2F8B9] = 0x633D, -- CJK COMPATIBILITY IDEOGRAPH-2F8B9
 [0x2F8BA] = 0x62FC, -- CJK COMPATIBILITY IDEOGRAPH-2F8BA
 [0x2F8BB] = 0x6368, -- CJK COMPATIBILITY IDEOGRAPH-2F8BB
 [0x2F8BC] = 0x6383, -- CJK COMPATIBILITY IDEOGRAPH-2F8BC
 [0x2F8BD] = 0x63E4, -- CJK COMPATIBILITY IDEOGRAPH-2F8BD
 [0x2F8BE] = 0x22BF1, -- CJK COMPATIBILITY IDEOGRAPH-2F8BE
 [0x2F8BF] = 0x6422, -- CJK COMPATIBILITY IDEOGRAPH-2F8BF
 [0x2F8C0] = 0x63C5, -- CJK COMPATIBILITY IDEOGRAPH-2F8C0
 [0x2F8C1] = 0x63A9, -- CJK COMPATIBILITY IDEOGRAPH-2F8C1
 [0x2F8C2] = 0x3A2E, -- CJK COMPATIBILITY IDEOGRAPH-2F8C2
 [0x2F8C3] = 0x6469, -- CJK COMPATIBILITY IDEOGRAPH-2F8C3
 [0x2F8C4] = 0x647E, -- CJK COMPATIBILITY IDEOGRAPH-2F8C4
 [0x2F8C5] = 0x649D, -- CJK COMPATIBILITY IDEOGRAPH-2F8C5
 [0x2F8C6] = 0x6477, -- CJK COMPATIBILITY IDEOGRAPH-2F8C6
 [0x2F8C7] = 0x3A6C, -- CJK COMPATIBILITY IDEOGRAPH-2F8C7
 [0x2F8C8] = 0x654F, -- CJK COMPATIBILITY IDEOGRAPH-2F8C8
 [0x2F8C9] = 0x656C, -- CJK COMPATIBILITY IDEOGRAPH-2F8C9
 [0x2F8CA] = 0x2300A, -- CJK COMPATIBILITY IDEOGRAPH-2F8CA
 [0x2F8CB] = 0x65E3, -- CJK COMPATIBILITY IDEOGRAPH-2F8CB
 [0x2F8CC] = 0x66F8, -- CJK COMPATIBILITY IDEOGRAPH-2F8CC
 [0x2F8CD] = 0x6649, -- CJK COMPATIBILITY IDEOGRAPH-2F8CD
 [0x2F8CE] = 0x3B19, -- CJK COMPATIBILITY IDEOGRAPH-2F8CE
 [0x2F8CF] = 0x6691, -- CJK COMPATIBILITY IDEOGRAPH-2F8CF
 [0x2F8D0] = 0x3B08, -- CJK COMPATIBILITY IDEOGRAPH-2F8D0
 [0x2F8D1] = 0x3AE4, -- CJK COMPATIBILITY IDEOGRAPH-2F8D1
 [0x2F8D2] = 0x5192, -- CJK COMPATIBILITY IDEOGRAPH-2F8D2
 [0x2F8D3] = 0x5195, -- CJK COMPATIBILITY IDEOGRAPH-2F8D3
 [0x2F8D4] = 0x6700, -- CJK COMPATIBILITY IDEOGRAPH-2F8D4
 [0x2F8D5] = 0x669C, -- CJK COMPATIBILITY IDEOGRAPH-2F8D5
 [0x2F8D6] = 0x80AD, -- CJK COMPATIBILITY IDEOGRAPH-2F8D6
 [0x2F8D7] = 0x43D9, -- CJK COMPATIBILITY IDEOGRAPH-2F8D7
 [0x2F8D8] = 0x6717, -- CJK COMPATIBILITY IDEOGRAPH-2F8D8
 [0x2F8D9] = 0x671B, -- CJK COMPATIBILITY IDEOGRAPH-2F8D9
 [0x2F8DA] = 0x6721, -- CJK COMPATIBILITY IDEOGRAPH-2F8DA
 [0x2F8DB] = 0x675E, -- CJK COMPATIBILITY IDEOGRAPH-2F8DB
 [0x2F8DC] = 0x6753, -- CJK COMPATIBILITY IDEOGRAPH-2F8DC
 [0x2F8DD] = 0x233C3, -- CJK COMPATIBILITY IDEOGRAPH-2F8DD
 [0x2F8DE] = 0x3B49, -- CJK COMPATIBILITY IDEOGRAPH-2F8DE
 [0x2F8DF] = 0x67FA, -- CJK COMPATIBILITY IDEOGRAPH-2F8DF
 [0x2F8E0] = 0x6785, -- CJK COMPATIBILITY IDEOGRAPH-2F8E0
 [0x2F8E1] = 0x6852, -- CJK COMPATIBILITY IDEOGRAPH-2F8E1
 [0x2F8E2] = 0x6885, -- CJK COMPATIBILITY IDEOGRAPH-2F8E2
 [0x2F8E3] = 0x2346D, -- CJK COMPATIBILITY IDEOGRAPH-2F8E3
 [0x2F8E4] = 0x688E, -- CJK COMPATIBILITY IDEOGRAPH-2F8E4
 [0x2F8E5] = 0x681F, -- CJK COMPATIBILITY IDEOGRAPH-2F8E5
 [0x2F8E6] = 0x6914, -- CJK COMPATIBILITY IDEOGRAPH-2F8E6
 [0x2F8E7] = 0x3B9D, -- CJK COMPATIBILITY IDEOGRAPH-2F8E7
 [0x2F8E8] = 0x6942, -- CJK COMPATIBILITY IDEOGRAPH-2F8E8
 [0x2F8E9] = 0x69A3, -- CJK COMPATIBILITY IDEOGRAPH-2F8E9
 [0x2F8EA] = 0x69EA, -- CJK COMPATIBILITY IDEOGRAPH-2F8EA
 [0x2F8EB] = 0x6AA8, -- CJK COMPATIBILITY IDEOGRAPH-2F8EB
 [0x2F8EC] = 0x236A3, -- CJK COMPATIBILITY IDEOGRAPH-2F8EC
 [0x2F8ED] = 0x6ADB, -- CJK COMPATIBILITY IDEOGRAPH-2F8ED
 [0x2F8EE] = 0x3C18, -- CJK COMPATIBILITY IDEOGRAPH-2F8EE
 [0x2F8EF] = 0x6B21, -- CJK COMPATIBILITY IDEOGRAPH-2F8EF
 [0x2F8F0] = 0x238A7, -- CJK COMPATIBILITY IDEOGRAPH-2F8F0
 [0x2F8F1] = 0x6B54, -- CJK COMPATIBILITY IDEOGRAPH-2F8F1
 [0x2F8F2] = 0x3C4E, -- CJK COMPATIBILITY IDEOGRAPH-2F8F2
 [0x2F8F3] = 0x6B72, -- CJK COMPATIBILITY IDEOGRAPH-2F8F3
 [0x2F8F4] = 0x6B9F, -- CJK COMPATIBILITY IDEOGRAPH-2F8F4
 [0x2F8F5] = 0x6BBA, -- CJK COMPATIBILITY IDEOGRAPH-2F8F5
 [0x2F8F6] = 0x6BBB, -- CJK COMPATIBILITY IDEOGRAPH-2F8F6
 [0x2F8F7] = 0x23A8D, -- CJK COMPATIBILITY IDEOGRAPH-2F8F7
 [0x2F8F8] = 0x21D0B, -- CJK COMPATIBILITY IDEOGRAPH-2F8F8
 [0x2F8F9] = 0x23AFA, -- CJK COMPATIBILITY IDEOGRAPH-2F8F9
 [0x2F8FA] = 0x6C4E, -- CJK COMPATIBILITY IDEOGRAPH-2F8FA
 [0x2F8FB] = 0x23CBC, -- CJK COMPATIBILITY IDEOGRAPH-2F8FB
 [0x2F8FC] = 0x6CBF, -- CJK COMPATIBILITY IDEOGRAPH-2F8FC
 [0x2F8FD] = 0x6CCD, -- CJK COMPATIBILITY IDEOGRAPH-2F8FD
 [0x2F8FE] = 0x6C67, -- CJK COMPATIBILITY IDEOGRAPH-2F8FE
 [0x2F8FF] = 0x6D16, -- CJK COMPATIBILITY IDEOGRAPH-2F8FF
 [0x2F900] = 0x6D3E, -- CJK COMPATIBILITY IDEOGRAPH-2F900
 [0x2F901] = 0x6D77, -- CJK COMPATIBILITY IDEOGRAPH-2F901
 [0x2F902] = 0x6D41, -- CJK COMPATIBILITY IDEOGRAPH-2F902
 [0x2F903] = 0x6D69, -- CJK COMPATIBILITY IDEOGRAPH-2F903
 [0x2F904] = 0x6D78, -- CJK COMPATIBILITY IDEOGRAPH-2F904
 [0x2F905] = 0x6D85, -- CJK COMPATIBILITY IDEOGRAPH-2F905
 [0x2F906] = 0x23D1E, -- CJK COMPATIBILITY IDEOGRAPH-2F906
 [0x2F907] = 0x6D34, -- CJK COMPATIBILITY IDEOGRAPH-2F907
 [0x2F908] = 0x6E2F, -- CJK COMPATIBILITY IDEOGRAPH-2F908
 [0x2F909] = 0x6E6E, -- CJK COMPATIBILITY IDEOGRAPH-2F909
 [0x2F90A] = 0x3D33, -- CJK COMPATIBILITY IDEOGRAPH-2F90A
 [0x2F90B] = 0x6ECB, -- CJK COMPATIBILITY IDEOGRAPH-2F90B
 [0x2F90C] = 0x6EC7, -- CJK COMPATIBILITY IDEOGRAPH-2F90C
 [0x2F90D] = 0x23ED1, -- CJK COMPATIBILITY IDEOGRAPH-2F90D
 [0x2F90E] = 0x6DF9, -- CJK COMPATIBILITY IDEOGRAPH-2F90E
 [0x2F90F] = 0x6F6E, -- CJK COMPATIBILITY IDEOGRAPH-2F90F
 [0x2F910] = 0x23F5E, -- CJK COMPATIBILITY IDEOGRAPH-2F910
 [0x2F911] = 0x23F8E, -- CJK COMPATIBILITY IDEOGRAPH-2F911
 [0x2F912] = 0x6FC6, -- CJK COMPATIBILITY IDEOGRAPH-2F912
 [0x2F913] = 0x7039, -- CJK COMPATIBILITY IDEOGRAPH-2F913
 [0x2F914] = 0x701E, -- CJK COMPATIBILITY IDEOGRAPH-2F914
 [0x2F915] = 0x701B, -- CJK COMPATIBILITY IDEOGRAPH-2F915
 [0x2F916] = 0x3D96, -- CJK COMPATIBILITY IDEOGRAPH-2F916
 [0x2F917] = 0x704A, -- CJK COMPATIBILITY IDEOGRAPH-2F917
 [0x2F918] = 0x707D, -- CJK COMPATIBILITY IDEOGRAPH-2F918
 [0x2F919] = 0x7077, -- CJK COMPATIBILITY IDEOGRAPH-2F919
 [0x2F91A] = 0x70AD, -- CJK COMPATIBILITY IDEOGRAPH-2F91A
 [0x2F91B] = 0x20525, -- CJK COMPATIBILITY IDEOGRAPH-2F91B
 [0x2F91C] = 0x7145, -- CJK COMPATIBILITY IDEOGRAPH-2F91C
 [0x2F91D] = 0x24263, -- CJK COMPATIBILITY IDEOGRAPH-2F91D
 [0x2F91E] = 0x719C, -- CJK COMPATIBILITY IDEOGRAPH-2F91E
 [0x2F91F] = {status="disallowed"}, -- CJK COMPATIBILITY IDEOGRAPH-2F91F
 [0x2F920] = 0x7228, -- CJK COMPATIBILITY IDEOGRAPH-2F920
 [0x2F921] = 0x7235, -- CJK COMPATIBILITY IDEOGRAPH-2F921
 [0x2F922] = 0x7250, -- CJK COMPATIBILITY IDEOGRAPH-2F922
 [0x2F923] = 0x24608, -- CJK COMPATIBILITY IDEOGRAPH-2F923
 [0x2F924] = 0x7280, -- CJK COMPATIBILITY IDEOGRAPH-2F924
 [0x2F925] = 0x7295, -- CJK COMPATIBILITY IDEOGRAPH-2F925
 [0x2F926] = 0x24735, -- CJK COMPATIBILITY IDEOGRAPH-2F926
 [0x2F927] = 0x24814, -- CJK COMPATIBILITY IDEOGRAPH-2F927
 [0x2F928] = 0x737A, -- CJK COMPATIBILITY IDEOGRAPH-2F928
 [0x2F929] = 0x738B, -- CJK COMPATIBILITY IDEOGRAPH-2F929
 [0x2F92A] = 0x3EAC, -- CJK COMPATIBILITY IDEOGRAPH-2F92A
 [0x2F92B] = 0x73A5, -- CJK COMPATIBILITY IDEOGRAPH-2F92B
 -- CJK COMPATIBILITY IDEOGRAPH-2F894..CJK COMPATIBILITY IDEOGRAPH-2F895
 [0x2F894] = 0x5F22,
 [0x2F895] = 0x5F22,
 [0x2F92E] = 0x7447, -- CJK COMPATIBILITY IDEOGRAPH-2F92E
 [0x2F92F] = 0x745C, -- CJK COMPATIBILITY IDEOGRAPH-2F92F
 [0x2F930] = 0x7471, -- CJK COMPATIBILITY IDEOGRAPH-2F930
 [0x2F931] = 0x7485, -- CJK COMPATIBILITY IDEOGRAPH-2F931
 [0x2F932] = 0x74CA, -- CJK COMPATIBILITY IDEOGRAPH-2F932
 [0x2F933] = 0x3F1B, -- CJK COMPATIBILITY IDEOGRAPH-2F933
 [0x2F934] = 0x7524, -- CJK COMPATIBILITY IDEOGRAPH-2F934
 [0x2F935] = 0x24C36, -- CJK COMPATIBILITY IDEOGRAPH-2F935
 [0x2F936] = 0x753E, -- CJK COMPATIBILITY IDEOGRAPH-2F936
 [0x2F937] = 0x24C92, -- CJK COMPATIBILITY IDEOGRAPH-2F937
 [0x2F938] = 0x7570, -- CJK COMPATIBILITY IDEOGRAPH-2F938
 [0x2F939] = 0x2219F, -- CJK COMPATIBILITY IDEOGRAPH-2F939
 [0x2F93A] = 0x7610, -- CJK COMPATIBILITY IDEOGRAPH-2F93A
 [0x2F93B] = 0x24FA1, -- CJK COMPATIBILITY IDEOGRAPH-2F93B
 [0x2F93C] = 0x24FB8, -- CJK COMPATIBILITY IDEOGRAPH-2F93C
 [0x2F93D] = 0x25044, -- CJK COMPATIBILITY IDEOGRAPH-2F93D
 [0x2F93E] = 0x3FFC, -- CJK COMPATIBILITY IDEOGRAPH-2F93E
 [0x2F93F] = 0x4008, -- CJK COMPATIBILITY IDEOGRAPH-2F93F
 [0x2F940] = 0x76F4, -- CJK COMPATIBILITY IDEOGRAPH-2F940
 [0x2F941] = 0x250F3, -- CJK COMPATIBILITY IDEOGRAPH-2F941
 [0x2F942] = 0x250F2, -- CJK COMPATIBILITY IDEOGRAPH-2F942
 [0x2F943] = 0x25119, -- CJK COMPATIBILITY IDEOGRAPH-2F943
 [0x2F944] = 0x25133, -- CJK COMPATIBILITY IDEOGRAPH-2F944
 [0x2F945] = 0x771E, -- CJK COMPATIBILITY IDEOGRAPH-2F945
 -- CJK COMPATIBILITY IDEOGRAPH-2F92C..CJK COMPATIBILITY IDEOGRAPH-2F92D
 [0x2F92C] = 0x3EB8,
 [0x2F92D] = 0x3EB8,
 [0x2F948] = 0x774A, -- CJK COMPATIBILITY IDEOGRAPH-2F948
 [0x2F949] = 0x4039, -- CJK COMPATIBILITY IDEOGRAPH-2F949
 [0x2F94A] = 0x778B, -- CJK COMPATIBILITY IDEOGRAPH-2F94A
 [0x2F94B] = 0x4046, -- CJK COMPATIBILITY IDEOGRAPH-2F94B
 [0x2F94C] = 0x4096, -- CJK COMPATIBILITY IDEOGRAPH-2F94C
 [0x2F94D] = 0x2541D, -- CJK COMPATIBILITY IDEOGRAPH-2F94D
 [0x2F94E] = 0x784E, -- CJK COMPATIBILITY IDEOGRAPH-2F94E
 [0x2F94F] = 0x788C, -- CJK COMPATIBILITY IDEOGRAPH-2F94F
 [0x2F950] = 0x78CC, -- CJK COMPATIBILITY IDEOGRAPH-2F950
 [0x2F951] = 0x40E3, -- CJK COMPATIBILITY IDEOGRAPH-2F951
 [0x2F952] = 0x25626, -- CJK COMPATIBILITY IDEOGRAPH-2F952
 [0x2F953] = 0x7956, -- CJK COMPATIBILITY IDEOGRAPH-2F953
 [0x2F954] = 0x2569A, -- CJK COMPATIBILITY IDEOGRAPH-2F954
 [0x2F955] = 0x256C5, -- CJK COMPATIBILITY IDEOGRAPH-2F955
 [0x2F956] = 0x798F, -- CJK COMPATIBILITY IDEOGRAPH-2F956
 [0x2F957] = 0x79EB, -- CJK COMPATIBILITY IDEOGRAPH-2F957
 [0x2F958] = 0x412F, -- CJK COMPATIBILITY IDEOGRAPH-2F958
 [0x2F959] = 0x7A40, -- CJK COMPATIBILITY IDEOGRAPH-2F959
 [0x2F95A] = 0x7A4A, -- CJK COMPATIBILITY IDEOGRAPH-2F95A
 [0x2F95B] = 0x7A4F, -- CJK COMPATIBILITY IDEOGRAPH-2F95B
 [0x2F95C] = 0x2597C, -- CJK COMPATIBILITY IDEOGRAPH-2F95C
 -- CJK COMPATIBILITY IDEOGRAPH-2F946..CJK COMPATIBILITY IDEOGRAPH-2F947
 [0x2F946] = 0x771F,
 [0x2F947] = 0x771F,
 [0x2F95F] = {status="disallowed"}, -- CJK COMPATIBILITY IDEOGRAPH-2F95F
 [0x2F960] = 0x4202, -- CJK COMPATIBILITY IDEOGRAPH-2F960
 [0x2F961] = 0x25BAB, -- CJK COMPATIBILITY IDEOGRAPH-2F961
 [0x2F962] = 0x7BC6, -- CJK COMPATIBILITY IDEOGRAPH-2F962
 [0x2F963] = 0x7BC9, -- CJK COMPATIBILITY IDEOGRAPH-2F963
 [0x2F964] = 0x4227, -- CJK COMPATIBILITY IDEOGRAPH-2F964
 [0x2F965] = 0x25C80, -- CJK COMPATIBILITY IDEOGRAPH-2F965
 [0x2F966] = 0x7CD2, -- CJK COMPATIBILITY IDEOGRAPH-2F966
 [0x2F967] = 0x42A0, -- CJK COMPATIBILITY IDEOGRAPH-2F967
 [0x2F968] = 0x7CE8, -- CJK COMPATIBILITY IDEOGRAPH-2F968
 [0x2F969] = 0x7CE3, -- CJK COMPATIBILITY IDEOGRAPH-2F969
 [0x2F96A] = 0x7D00, -- CJK COMPATIBILITY IDEOGRAPH-2F96A
 [0x2F96B] = 0x25F86, -- CJK COMPATIBILITY IDEOGRAPH-2F96B
 [0x2F96C] = 0x7D63, -- CJK COMPATIBILITY IDEOGRAPH-2F96C
 [0x2F96D] = 0x4301, -- CJK COMPATIBILITY IDEOGRAPH-2F96D
 [0x2F96E] = 0x7DC7, -- CJK COMPATIBILITY IDEOGRAPH-2F96E
 [0x2F96F] = 0x7E02, -- CJK COMPATIBILITY IDEOGRAPH-2F96F
 [0x2F970] = 0x7E45, -- CJK COMPATIBILITY IDEOGRAPH-2F970
 [0x2F971] = 0x4334, -- CJK COMPATIBILITY IDEOGRAPH-2F971
 [0x2F972] = 0x26228, -- CJK COMPATIBILITY IDEOGRAPH-2F972
 [0x2F973] = 0x26247, -- CJK COMPATIBILITY IDEOGRAPH-2F973
 [0x2F974] = 0x4359, -- CJK COMPATIBILITY IDEOGRAPH-2F974
 [0x2F975] = 0x262D9, -- CJK COMPATIBILITY IDEOGRAPH-2F975
 [0x2F976] = 0x7F7A, -- CJK COMPATIBILITY IDEOGRAPH-2F976
 [0x2F977] = 0x2633E, -- CJK COMPATIBILITY IDEOGRAPH-2F977
 [0x2F978] = 0x7F95, -- CJK COMPATIBILITY IDEOGRAPH-2F978
 [0x2F979] = 0x7FFA, -- CJK COMPATIBILITY IDEOGRAPH-2F979
 [0x2F97A] = 0x8005, -- CJK COMPATIBILITY IDEOGRAPH-2F97A
 [0x2F97B] = 0x264DA, -- CJK COMPATIBILITY IDEOGRAPH-2F97B
 [0x2F97C] = 0x26523, -- CJK COMPATIBILITY IDEOGRAPH-2F97C
 [0x2F97D] = 0x8060, -- CJK COMPATIBILITY IDEOGRAPH-2F97D
 [0x2F97E] = 0x265A8, -- CJK COMPATIBILITY IDEOGRAPH-2F97E
 [0x2F97F] = 0x8070, -- CJK COMPATIBILITY IDEOGRAPH-2F97F
 [0x2F980] = 0x2335F, -- CJK COMPATIBILITY IDEOGRAPH-2F980
 [0x2F981] = 0x43D5, -- CJK COMPATIBILITY IDEOGRAPH-2F981
 [0x2F982] = 0x80B2, -- CJK COMPATIBILITY IDEOGRAPH-2F982
 [0x2F983] = 0x8103, -- CJK COMPATIBILITY IDEOGRAPH-2F983
 [0x2F984] = 0x440B, -- CJK COMPATIBILITY IDEOGRAPH-2F984
 [0x2F985] = 0x813E, -- CJK COMPATIBILITY IDEOGRAPH-2F985
 [0x2F986] = 0x5AB5, -- CJK COMPATIBILITY IDEOGRAPH-2F986
 [0x2F987] = 0x267A7, -- CJK COMPATIBILITY IDEOGRAPH-2F987
 [0x2F988] = 0x267B5, -- CJK COMPATIBILITY IDEOGRAPH-2F988
 [0x2F989] = 0x23393, -- CJK COMPATIBILITY IDEOGRAPH-2F989
 [0x2F98A] = 0x2339C, -- CJK COMPATIBILITY IDEOGRAPH-2F98A
 [0x2F98B] = 0x8201, -- CJK COMPATIBILITY IDEOGRAPH-2F98B
 [0x2F98C] = 0x8204, -- CJK COMPATIBILITY IDEOGRAPH-2F98C
 [0x2F98D] = 0x8F9E, -- CJK COMPATIBILITY IDEOGRAPH-2F98D
 [0x2F98E] = 0x446B, -- CJK COMPATIBILITY IDEOGRAPH-2F98E
 [0x2F98F] = 0x8291, -- CJK COMPATIBILITY IDEOGRAPH-2F98F
 [0x2F990] = 0x828B, -- CJK COMPATIBILITY IDEOGRAPH-2F990
 [0x2F991] = 0x829D, -- CJK COMPATIBILITY IDEOGRAPH-2F991
 [0x2F992] = 0x52B3, -- CJK COMPATIBILITY IDEOGRAPH-2F992
 [0x2F993] = 0x82B1, -- CJK COMPATIBILITY IDEOGRAPH-2F993
 [0x2F994] = 0x82B3, -- CJK COMPATIBILITY IDEOGRAPH-2F994
 [0x2F995] = 0x82BD, -- CJK COMPATIBILITY IDEOGRAPH-2F995
 [0x2F996] = 0x82E6, -- CJK COMPATIBILITY IDEOGRAPH-2F996
 [0x2F997] = 0x26B3C, -- CJK COMPATIBILITY IDEOGRAPH-2F997
 [0x2F998] = 0x82E5, -- CJK COMPATIBILITY IDEOGRAPH-2F998
 [0x2F999] = 0x831D, -- CJK COMPATIBILITY IDEOGRAPH-2F999
 [0x2F99A] = 0x8363, -- CJK COMPATIBILITY IDEOGRAPH-2F99A
 [0x2F99B] = 0x83AD, -- CJK COMPATIBILITY IDEOGRAPH-2F99B
 [0x2F99C] = 0x8323, -- CJK COMPATIBILITY IDEOGRAPH-2F99C
 [0x2F99D] = 0x83BD, -- CJK COMPATIBILITY IDEOGRAPH-2F99D
 [0x2F99E] = 0x83E7, -- CJK COMPATIBILITY IDEOGRAPH-2F99E
 [0x2F99F] = 0x8457, -- CJK COMPATIBILITY IDEOGRAPH-2F99F
 [0x2F9A0] = 0x8353, -- CJK COMPATIBILITY IDEOGRAPH-2F9A0
 [0x2F9A1] = 0x83CA, -- CJK COMPATIBILITY IDEOGRAPH-2F9A1
 [0x2F9A2] = 0x83CC, -- CJK COMPATIBILITY IDEOGRAPH-2F9A2
 [0x2F9A3] = 0x83DC, -- CJK COMPATIBILITY IDEOGRAPH-2F9A3
 [0x2F9A4] = 0x26C36, -- CJK COMPATIBILITY IDEOGRAPH-2F9A4
 [0x2F9A5] = 0x26D6B, -- CJK COMPATIBILITY IDEOGRAPH-2F9A5
 [0x2F9A6] = 0x26CD5, -- CJK COMPATIBILITY IDEOGRAPH-2F9A6
 [0x2F9A7] = 0x452B, -- CJK COMPATIBILITY IDEOGRAPH-2F9A7
 [0x2F9A8] = 0x84F1, -- CJK COMPATIBILITY IDEOGRAPH-2F9A8
 [0x2F9A9] = 0x84F3, -- CJK COMPATIBILITY IDEOGRAPH-2F9A9
 [0x2F9AA] = 0x8516, -- CJK COMPATIBILITY IDEOGRAPH-2F9AA
 [0x2F9AB] = 0x273CA, -- CJK COMPATIBILITY IDEOGRAPH-2F9AB
 [0x2F9AC] = 0x8564, -- CJK COMPATIBILITY IDEOGRAPH-2F9AC
 [0x2F9AD] = 0x26F2C, -- CJK COMPATIBILITY IDEOGRAPH-2F9AD
 [0x2F9AE] = 0x455D, -- CJK COMPATIBILITY IDEOGRAPH-2F9AE
 [0x2F9AF] = 0x4561, -- CJK COMPATIBILITY IDEOGRAPH-2F9AF
 [0x2F9B0] = 0x26FB1, -- CJK COMPATIBILITY IDEOGRAPH-2F9B0
 [0x2F9B1] = 0x270D2, -- CJK COMPATIBILITY IDEOGRAPH-2F9B1
 [0x2F9B2] = 0x456B, -- CJK COMPATIBILITY IDEOGRAPH-2F9B2
 [0x2F9B3] = 0x8650, -- CJK COMPATIBILITY IDEOGRAPH-2F9B3
 [0x2F9B4] = 0x865C, -- CJK COMPATIBILITY IDEOGRAPH-2F9B4
 [0x2F9B5] = 0x8667, -- CJK COMPATIBILITY IDEOGRAPH-2F9B5
 [0x2F9B6] = 0x8669, -- CJK COMPATIBILITY IDEOGRAPH-2F9B6
 [0x2F9B7] = 0x86A9, -- CJK COMPATIBILITY IDEOGRAPH-2F9B7
 [0x2F9B8] = 0x8688, -- CJK COMPATIBILITY IDEOGRAPH-2F9B8
 [0x2F9B9] = 0x870E, -- CJK COMPATIBILITY IDEOGRAPH-2F9B9
 [0x2F9BA] = 0x86E2, -- CJK COMPATIBILITY IDEOGRAPH-2F9BA
 [0x2F9BB] = 0x8779, -- CJK COMPATIBILITY IDEOGRAPH-2F9BB
 [0x2F9BC] = 0x8728, -- CJK COMPATIBILITY IDEOGRAPH-2F9BC
 [0x2F9BD] = 0x876B, -- CJK COMPATIBILITY IDEOGRAPH-2F9BD
 [0x2F9BE] = 0x8786, -- CJK COMPATIBILITY IDEOGRAPH-2F9BE
 [0x2F9BF] = {status="disallowed"}, -- CJK COMPATIBILITY IDEOGRAPH-2F9BF
 [0x2F9C0] = 0x87E1, -- CJK COMPATIBILITY IDEOGRAPH-2F9C0
 [0x2F9C1] = 0x8801, -- CJK COMPATIBILITY IDEOGRAPH-2F9C1
 [0x2F9C2] = 0x45F9, -- CJK COMPATIBILITY IDEOGRAPH-2F9C2
 [0x2F9C3] = 0x8860, -- CJK COMPATIBILITY IDEOGRAPH-2F9C3
 [0x2F9C4] = 0x8863, -- CJK COMPATIBILITY IDEOGRAPH-2F9C4
 [0x2F9C5] = 0x27667, -- CJK COMPATIBILITY IDEOGRAPH-2F9C5
 [0x2F9C6] = 0x88D7, -- CJK COMPATIBILITY IDEOGRAPH-2F9C6
 [0x2F9C7] = 0x88DE, -- CJK COMPATIBILITY IDEOGRAPH-2F9C7
 [0x2F9C8] = 0x4635, -- CJK COMPATIBILITY IDEOGRAPH-2F9C8
 [0x2F9C9] = 0x88FA, -- CJK COMPATIBILITY IDEOGRAPH-2F9C9
 [0x2F9CA] = 0x34BB, -- CJK COMPATIBILITY IDEOGRAPH-2F9CA
 [0x2F9CB] = 0x278AE, -- CJK COMPATIBILITY IDEOGRAPH-2F9CB
 [0x2F9CC] = 0x27966, -- CJK COMPATIBILITY IDEOGRAPH-2F9CC
 [0x2F9CD] = 0x46BE, -- CJK COMPATIBILITY IDEOGRAPH-2F9CD
 [0x2F9CE] = 0x46C7, -- CJK COMPATIBILITY IDEOGRAPH-2F9CE
 [0x2F9CF] = 0x8AA0, -- CJK COMPATIBILITY IDEOGRAPH-2F9CF
 [0x2F9D0] = 0x8AED, -- CJK COMPATIBILITY IDEOGRAPH-2F9D0
 [0x2F9D1] = 0x8B8A, -- CJK COMPATIBILITY IDEOGRAPH-2F9D1
 [0x2F9D2] = 0x8C55, -- CJK COMPATIBILITY IDEOGRAPH-2F9D2
 [0x2F9D3] = 0x27CA8, -- CJK COMPATIBILITY IDEOGRAPH-2F9D3
 [0x2F9D4] = 0x8CAB, -- CJK COMPATIBILITY IDEOGRAPH-2F9D4
 [0x2F9D5] = 0x8CC1, -- CJK COMPATIBILITY IDEOGRAPH-2F9D5
 [0x2F9D6] = 0x8D1B, -- CJK COMPATIBILITY IDEOGRAPH-2F9D6
 [0x2F9D7] = 0x8D77, -- CJK COMPATIBILITY IDEOGRAPH-2F9D7
 [0x2F9D8] = 0x27F2F, -- CJK COMPATIBILITY IDEOGRAPH-2F9D8
 [0x2F9D9] = 0x20804, -- CJK COMPATIBILITY IDEOGRAPH-2F9D9
 [0x2F9DA] = 0x8DCB, -- CJK COMPATIBILITY IDEOGRAPH-2F9DA
 [0x2F9DB] = 0x8DBC, -- CJK COMPATIBILITY IDEOGRAPH-2F9DB
 [0x2F9DC] = 0x8DF0, -- CJK COMPATIBILITY IDEOGRAPH-2F9DC
 [0x2F9DD] = 0x208DE, -- CJK COMPATIBILITY IDEOGRAPH-2F9DD
 [0x2F9DE] = 0x8ED4, -- CJK COMPATIBILITY IDEOGRAPH-2F9DE
 [0x2F9DF] = 0x8F38, -- CJK COMPATIBILITY IDEOGRAPH-2F9DF
 [0x2F9E0] = 0x285D2, -- CJK COMPATIBILITY IDEOGRAPH-2F9E0
 [0x2F9E1] = 0x285ED, -- CJK COMPATIBILITY IDEOGRAPH-2F9E1
 [0x2F9E2] = 0x9094, -- CJK COMPATIBILITY IDEOGRAPH-2F9E2
 [0x2F9E3] = 0x90F1, -- CJK COMPATIBILITY IDEOGRAPH-2F9E3
 [0x2F9E4] = 0x9111, -- CJK COMPATIBILITY IDEOGRAPH-2F9E4
 [0x2F9E5] = 0x2872E, -- CJK COMPATIBILITY IDEOGRAPH-2F9E5
 [0x2F9E6] = 0x911B, -- CJK COMPATIBILITY IDEOGRAPH-2F9E6
 [0x2F9E7] = 0x9238, -- CJK COMPATIBILITY IDEOGRAPH-2F9E7
 [0x2F9E8] = 0x92D7, -- CJK COMPATIBILITY IDEOGRAPH-2F9E8
 [0x2F9E9] = 0x92D8, -- CJK COMPATIBILITY IDEOGRAPH-2F9E9
 [0x2F9EA] = 0x927C, -- CJK COMPATIBILITY IDEOGRAPH-2F9EA
 [0x2F9EB] = 0x93F9, -- CJK COMPATIBILITY IDEOGRAPH-2F9EB
 [0x2F9EC] = 0x9415, -- CJK COMPATIBILITY IDEOGRAPH-2F9EC
 [0x2F9ED] = 0x28BFA, -- CJK COMPATIBILITY IDEOGRAPH-2F9ED
 [0x2F9EE] = 0x958B, -- CJK COMPATIBILITY IDEOGRAPH-2F9EE
 [0x2F9EF] = 0x4995, -- CJK COMPATIBILITY IDEOGRAPH-2F9EF
 [0x2F9F0] = 0x95B7, -- CJK COMPATIBILITY IDEOGRAPH-2F9F0
 [0x2F9F1] = 0x28D77, -- CJK COMPATIBILITY IDEOGRAPH-2F9F1
 [0x2F9F2] = 0x49E6, -- CJK COMPATIBILITY IDEOGRAPH-2F9F2
 [0x2F9F3] = 0x96C3, -- CJK COMPATIBILITY IDEOGRAPH-2F9F3
 [0x2F9F4] = 0x5DB2, -- CJK COMPATIBILITY IDEOGRAPH-2F9F4
 [0x2F9F5] = 0x9723, -- CJK COMPATIBILITY IDEOGRAPH-2F9F5
 [0x2F9F6] = 0x29145, -- CJK COMPATIBILITY IDEOGRAPH-2F9F6
 [0x2F9F7] = 0x2921A, -- CJK COMPATIBILITY IDEOGRAPH-2F9F7
 [0x2F9F8] = 0x4A6E, -- CJK COMPATIBILITY IDEOGRAPH-2F9F8
 [0x2F9F9] = 0x4A76, -- CJK COMPATIBILITY IDEOGRAPH-2F9F9
 [0x2F9FA] = 0x97E0, -- CJK COMPATIBILITY IDEOGRAPH-2F9FA
 [0x2F9FB] = 0x2940A, -- CJK COMPATIBILITY IDEOGRAPH-2F9FB
 [0x2F9FC] = 0x4AB2, -- CJK COMPATIBILITY IDEOGRAPH-2F9FC
 [0x2F9FD] = 0x29496, -- CJK COMPATIBILITY IDEOGRAPH-2F9FD
 -- CJK COMPATIBILITY IDEOGRAPH-2F95D..CJK COMPATIBILITY IDEOGRAPH-2F95E
 [0x2F95D] = 0x25AA7,
 [0x2F95E] = 0x25AA7,
 [0x2FA00] = 0x9829, -- CJK COMPATIBILITY IDEOGRAPH-2FA00
 [0x2FA01] = 0x295B6, -- CJK COMPATIBILITY IDEOGRAPH-2FA01
 [0x2FA02] = 0x98E2, -- CJK COMPATIBILITY IDEOGRAPH-2FA02
 [0x2FA03] = 0x4B33, -- CJK COMPATIBILITY IDEOGRAPH-2FA03
 [0x2FA04] = 0x9929, -- CJK COMPATIBILITY IDEOGRAPH-2FA04
 [0x2FA05] = 0x99A7, -- CJK COMPATIBILITY IDEOGRAPH-2FA05
 [0x2FA06] = 0x99C2, -- CJK COMPATIBILITY IDEOGRAPH-2FA06
 [0x2FA07] = 0x99FE, -- CJK COMPATIBILITY IDEOGRAPH-2FA07
 [0x2FA08] = 0x4BCE, -- CJK COMPATIBILITY IDEOGRAPH-2FA08
 [0x2FA09] = 0x29B30, -- CJK COMPATIBILITY IDEOGRAPH-2FA09
 [0x2FA0A] = 0x9B12, -- CJK COMPATIBILITY IDEOGRAPH-2FA0A
 [0x2FA0B] = 0x9C40, -- CJK COMPATIBILITY IDEOGRAPH-2FA0B
 [0x2FA0C] = 0x9CFD, -- CJK COMPATIBILITY IDEOGRAPH-2FA0C
 [0x2FA0D] = 0x4CCE, -- CJK COMPATIBILITY IDEOGRAPH-2FA0D
 [0x2FA0E] = 0x4CED, -- CJK COMPATIBILITY IDEOGRAPH-2FA0E
 [0x2FA0F] = 0x9D67, -- CJK COMPATIBILITY IDEOGRAPH-2FA0F
 [0x2FA10] = 0x2A0CE, -- CJK COMPATIBILITY IDEOGRAPH-2FA10
 [0x2FA11] = 0x4CF8, -- CJK COMPATIBILITY IDEOGRAPH-2FA11
 [0x2FA12] = 0x2A105, -- CJK COMPATIBILITY IDEOGRAPH-2FA12
 [0x2FA13] = 0x2A20E, -- CJK COMPATIBILITY IDEOGRAPH-2FA13
 [0x2FA14] = 0x2A291, -- CJK COMPATIBILITY IDEOGRAPH-2FA14
 [0x2FA15] = 0x9EBB, -- CJK COMPATIBILITY IDEOGRAPH-2FA15
 [0x2FA16] = 0x4D56, -- CJK COMPATIBILITY IDEOGRAPH-2FA16
 [0x2FA17] = 0x9EF9, -- CJK COMPATIBILITY IDEOGRAPH-2FA17
 [0x2FA18] = 0x9EFE, -- CJK COMPATIBILITY IDEOGRAPH-2FA18
 [0x2FA19] = 0x9F05, -- CJK COMPATIBILITY IDEOGRAPH-2FA19
 [0x2FA1A] = 0x9F0F, -- CJK COMPATIBILITY IDEOGRAPH-2FA1A
 [0x2FA1B] = 0x9F16, -- CJK COMPATIBILITY IDEOGRAPH-2FA1B
 [0x2FA1C] = 0x9F3B, -- CJK COMPATIBILITY IDEOGRAPH-2FA1C
 [0x2FA1D] = 0x2A600, -- CJK COMPATIBILITY IDEOGRAPH-2FA1D
 -- CJK COMPATIBILITY IDEOGRAPH-2F9FE..CJK COMPATIBILITY IDEOGRAPH-2F9FF
 [0x2F9FE] = 0x980B,
 [0x2F9FF] = 0x980B,
 [0xE0000] = {status="disallowed"}, -- NA   <reserved-E0000>
 [0xE0001] = {status="disallowed"}, -- LANGUAGE TAG
}

setmetatable(tbl, {
  __index = function (t, x)
  if x <= 0x002C then
    -- <control-0000>..COMMA
    if x >= 0x0000 then
      return {status="disallowed"}
    end
  elseif x <= 0x0039 then
    -- DIGIT ZERO..DIGIT NINE
    if x >= 0x0030 then
      return {status="valid"}
    end
  elseif x <= 0x0040 then
    -- COLON..COMMERCIAL AT
    if x >= 0x003A then
      return {status="disallowed"}
    end
  elseif x <= 0x0060 then
    -- LEFT SQUARE BRACKET..GRAVE ACCENT
    if x >= 0x005B then
      return {status="disallowed"}
    end
  elseif x <= 0x007A then
    -- LATIN SMALL LETTER A..LATIN SMALL LETTER Z
    if x >= 0x0061 then
      return {status="valid"}
    end
  elseif x <= 0x009F then
    -- <control-0080>..<control-009F>
    if x >= 0x0080 then
      return {status="disallowed"}
    end
  elseif x <= 0x00A7 then
    -- INVERTED EXCLAMATION MARK..SECTION SIGN
    if x >= 0x00A1 then
      return {status="disallowed"}
    end
  elseif x <= 0x00F6 then
    -- LATIN SMALL LETTER A WITH GRAVE..LATIN SMALL LETTER O WITH DIAERESIS
    if x >= 0x00E0 then
      return {status="valid"}
    end
  elseif x <= 0x00FF then
    -- LATIN SMALL LETTER O WITH STROKE..LATIN SMALL LETTER Y WITH DIAERESIS
    if x >= 0x00F8 then
      return {status="valid"}
    end
  elseif x <= 0x01C3 then
    -- LATIN SMALL LETTER TONE FIVE..LATIN LETTER RETROFLEX CLICK
    if x >= 0x01BD then
      return {status="valid"}
    end
  elseif x <= 0x0239 then
    -- LATIN SMALL LETTER L WITH CURL..LATIN SMALL LETTER T WITH CURL
    -- LATIN SMALL LETTER DOTLESS J..LATIN SMALL LETTER QP DIGRAPH
    if x >= 0x0234 then
      return {status="valid"}
    end
  elseif x <= 0x02AF then
    -- LATIN SMALL LETTER TURNED A..LATIN SMALL LETTER TC DIGRAPH WITH CURL
    -- LATIN SMALL LETTER FENG DIGRAPH..LATIN LETTER BIDENTAL PERCUSSIVE
    -- LATIN SMALL LETTER TURNED H WITH FISHHOOK..LATIN SMALL LETTER TURNED H WITH FISHHOOK AND TAIL
    if x >= 0x0250 then
      return {status="valid"}
    end
  elseif x <= 0x02C1 then
    -- MODIFIER LETTER PRIME..MODIFIER LETTER REVERSED GLOTTAL STOP
    if x >= 0x02B9 then
      return {status="valid"}
    end
  elseif x <= 0x02D1 then
    -- MODIFIER LETTER CIRCUMFLEX ACCENT..MODIFIER LETTER HALF TRIANGULAR COLON
    if x >= 0x02C6 then
      return {status="valid"}
    end
  elseif x <= 0x02D7 then
    -- MODIFIER LETTER CENTRED RIGHT HALF RING..MODIFIER LETTER MINUS SIGN
    if x >= 0x02D2 then
      return {status="disallowed"}
    end
  elseif x <= 0x02EB then
    -- MODIFIER LETTER EXTRA-HIGH TONE BAR..MODIFIER LETTER EXTRA-LOW TONE BAR
    -- MODIFIER LETTER YIN DEPARTING TONE MARK..MODIFIER LETTER YANG DEPARTING TONE MARK
    if x >= 0x02E5 then
      return {status="disallowed"}
    end
  elseif x <= 0x02FF then
    -- MODIFIER LETTER LOW DOWN ARROWHEAD..MODIFIER LETTER LOW LEFT ARROW
    if x >= 0x02EF then
      return {status="disallowed"}
    end
  elseif x <= 0x033F then
    -- COMBINING GRAVE ACCENT..COMBINING DOUBLE OVERLINE
    if x >= 0x0300 then
      return {status="valid"}
    end
  elseif x <= 0x034E then
    -- COMBINING BRIDGE ABOVE..COMBINING UPWARDS ARROW BELOW
    if x >= 0x0346 then
      return {status="valid"}
    end
  elseif x <= 0x0361 then
    -- COMBINING RIGHT ARROWHEAD ABOVE..COMBINING RIGHT HALF RING ABOVE
    -- COMBINING DOT ABOVE RIGHT..COMBINING DOUBLE BREVE BELOW
    -- COMBINING DOUBLE BREVE..COMBINING DOUBLE MACRON BELOW
    -- COMBINING DOUBLE TILDE..COMBINING DOUBLE INVERTED BREVE
    if x >= 0x0350 then
      return {status="valid"}
    end
  elseif x <= 0x036F then
    -- COMBINING LATIN SMALL LETTER A..COMBINING LATIN SMALL LETTER X
    if x >= 0x0363 then
      return {status="valid"}
    end
  elseif x <= 0x03C1 then
    -- GREEK SMALL LETTER ALPHA WITH TONOS..GREEK SMALL LETTER RHO
    if x >= 0x03AC then
      return {status="valid"}
    end
  elseif x <= 0x03CE then
    -- GREEK SMALL LETTER SIGMA..GREEK SMALL LETTER OMEGA WITH TONOS
    if x >= 0x03C3 then
      return {status="valid"}
    end
  elseif x <= 0x044F then
    -- CYRILLIC SMALL LETTER A..CYRILLIC SMALL LETTER YA
    if x >= 0x0430 then
      return {status="valid"}
    end
  elseif x <= 0x045C then
    -- CYRILLIC SMALL LETTER IO..CYRILLIC SMALL LETTER KJE
    if x >= 0x0451 then
      return {status="valid"}
    end
  elseif x <= 0x055F then
    -- ARMENIAN APOSTROPHE..ARMENIAN ABBREVIATION MARK
    if x >= 0x055A then
      return {status="disallowed"}
    end
  elseif x <= 0x0586 then
    -- ARMENIAN SMALL LETTER AYB..ARMENIAN SMALL LETTER FEH
    if x >= 0x0561 then
      return {status="valid"}
    end
  elseif x <= 0x05A1 then
    -- HEBREW ACCENT ETNAHTA..HEBREW ACCENT PAZER
    if x >= 0x0591 then
      return {status="valid"}
    end
  elseif x <= 0x05B9 then
    -- HEBREW ACCENT MUNAH..HEBREW MARK MASORA CIRCLE
    -- HEBREW POINT SHEVA..HEBREW POINT HOLAM
    if x >= 0x05A3 then
      return {status="valid"}
    end
  elseif x <= 0x05CF then
    -- NA   <reserved-05C8>..<reserved-05CF>
    if x >= 0x05C8 then
      return {status="disallowed"}
    end
  elseif x <= 0x05EA then
    -- HEBREW LETTER ALEF..HEBREW LETTER TAV
    if x >= 0x05D0 then
      return {status="valid"}
    end
  elseif x <= 0x0603 then
    -- NA   <reserved-05F5>..<reserved-05FF>
    -- ARABIC NUMBER SIGN..ARABIC SIGN SAFHA
    if x >= 0x05F5 then
      return {status="disallowed"}
    end
  elseif x <= 0x061A then
    -- ARABIC SIGN SALLALLAHOU ALAYHE WASSALLAM..ARABIC SMALL HIGH TAH
    -- ARABIC SMALL HIGH LIGATURE ALEF WITH LAM WITH YEH..ARABIC SMALL KASRA
    if x >= 0x0610 then
      return {status="valid"}
    end
  elseif x <= 0x063F then
    -- ARABIC LETTER HAMZA..ARABIC LETTER GHAIN
    -- ARABIC LETTER KEHEH WITH TWO DOTS ABOVE..ARABIC LETTER FARSI YEH WITH THREE DOTS ABOVE
    if x >= 0x0621 then
      return {status="valid"}
    end
  elseif x <= 0x065E then
    -- ARABIC LETTER FEH..ARABIC SUKUN
    -- ARABIC MADDAH ABOVE..ARABIC HAMZA BELOW
    -- ARABIC SUBSCRIPT ALEF..ARABIC MARK NOON GHUNNA
    -- ARABIC ZWARAKAY..ARABIC FATHA WITH TWO DOTS
    if x >= 0x0641 then
      return {status="valid"}
    end
  elseif x <= 0x0669 then
    -- ARABIC-INDIC DIGIT ZERO..ARABIC-INDIC DIGIT NINE
    if x >= 0x0660 then
      return {status="valid"}
    end
  elseif x <= 0x0674 then
    -- ARABIC LETTER DOTLESS BEH..ARABIC LETTER DOTLESS QAF
    -- ARABIC LETTER SUPERSCRIPT ALEF..ARABIC LETTER HIGH HAMZA
    if x >= 0x066E then
      return {status="valid"}
    end
  elseif x <= 0x06BE then
    -- ARABIC LETTER TTEH..ARABIC LETTER LAM WITH THREE DOTS ABOVE
    -- ARABIC LETTER LAM WITH THREE DOTS BELOW..ARABIC LETTER NOON WITH DOT BELOW
    -- ARABIC LETTER NOON GHUNNA..ARABIC LETTER HEH DOACHASHMEE
    if x >= 0x0679 then
      return {status="valid"}
    end
  elseif x <= 0x06CE then
    -- ARABIC LETTER HEH WITH YEH ABOVE..ARABIC LETTER YEH WITH SMALL V
    if x >= 0x06C0 then
      return {status="valid"}
    end
  elseif x <= 0x06DC then
    -- ARABIC LETTER AE..ARABIC SMALL HIGH SEEN
    if x >= 0x06D5 then
      return {status="valid"}
    end
  elseif x <= 0x06E8 then
    -- ARABIC SMALL HIGH ROUNDED ZERO..ARABIC SMALL HIGH NOON
    if x >= 0x06DF then
      return {status="valid"}
    end
  elseif x <= 0x06FE then
    -- ARABIC EMPTY CENTRE LOW STOP..ARABIC SMALL LOW MEEM
    -- ARABIC LETTER DAL WITH INVERTED V..ARABIC LETTER REH WITH INVERTED V
    -- EXTENDED ARABIC-INDIC DIGIT ZERO..EXTENDED ARABIC-INDIC DIGIT NINE
    -- ARABIC LETTER SHEEN WITH DOT BELOW..ARABIC SIGN SINDHI POSTPOSITION MEN
    if x >= 0x06EA then
      return {status="valid"}
    end
  elseif x <= 0x070D then
    -- SYRIAC END OF PARAGRAPH..SYRIAC HARKLEAN ASTERISCUS
    if x >= 0x0700 then
      return {status="disallowed"}
    end
  elseif x <= 0x074A then
    -- SYRIAC LETTER ALAPH..SYRIAC LETTER TAW
    -- SYRIAC LETTER PERSIAN BHETH..SYRIAC LETTER PERSIAN DHALATH
    -- SYRIAC PTHAHA ABOVE..SYRIAC BARREKH
    if x >= 0x0710 then
      return {status="valid"}
    end
  elseif x <= 0x07B0 then
    -- SYRIAC LETTER SOGDIAN ZHAIN..SYRIAC LETTER SOGDIAN FE
    -- ARABIC LETTER BEH WITH THREE DOTS HORIZONTALLY BELOW..ARABIC LETTER SEEN WITH TWO DOTS VERTICALLY ABOVE
    -- ARABIC LETTER HAH WITH SMALL ARABIC LETTER TAH BELOW..ARABIC LETTER KAF WITH TWO DOTS ABOVE
    -- THAANA LETTER HAA..THAANA SUKUN
    if x >= 0x074D then
      return {status="valid"}
    end
  elseif x <= 0x07BF then
    -- NA   <reserved-07B2>..<reserved-07BF>
    if x >= 0x07B2 then
      return {status="disallowed"}
    end
  elseif x <= 0x07F5 then
    -- NKO DIGIT ZERO..NKO LOW TONE APOSTROPHE
    if x >= 0x07C0 then
      return {status="valid"}
    end
  elseif x <= 0x082D then
    -- SAMARITAN LETTER ALAF..SAMARITAN MARK NEQUDAA
    if x >= 0x0800 then
      return {status="valid"}
    end
  elseif x <= 0x083E then
    -- SAMARITAN PUNCTUATION NEQUDAA..SAMARITAN PUNCTUATION ANNAAU
    if x >= 0x0830 then
      return {status="disallowed"}
    end
  elseif x <= 0x085B then
    -- MANDAIC LETTER HALQA..MANDAIC GEMINATION MARK
    if x >= 0x0840 then
      return {status="valid"}
    end
  elseif x <= 0x086A then
    -- SYRIAC LETTER MALAYALAM NGA..SYRIAC LETTER MALAYALAM SSA
    if x >= 0x0860 then
      return {status="valid"}
    end
  elseif x <= 0x089F then
    -- NA   <reserved-086B>..<reserved-089F>
    if x >= 0x086B then
      return {status="disallowed"}
    end
  elseif x <= 0x08B4 then
    -- ARABIC LETTER JEEM WITH TWO DOTS ABOVE..ARABIC LETTER ROHINGYA YEH
    -- ARABIC LETTER LOW ALEF..ARABIC LETTER ZAIN WITH INVERTED V ABOVE
    -- ARABIC LETTER AIN WITH THREE DOTS BELOW..ARABIC LETTER KAF WITH DOT BELOW
    if x >= 0x08A2 then
      return {status="valid"}
    end
  elseif x <= 0x08BD then
    -- ARABIC LETTER BEH WITH SMALL MEEM ABOVE..ARABIC LETTER AFRICAN NOON
    if x >= 0x08B6 then
      return {status="valid"}
    end
  elseif x <= 0x08D3 then
    -- NA   <reserved-08BE>..<reserved-08D3>
    if x >= 0x08BE then
      return {status="disallowed"}
    end
  elseif x <= 0x08E1 then
    -- ARABIC SMALL HIGH WORD AR-RUB..ARABIC SMALL HIGH SIGN SAFHA
    if x >= 0x08D4 then
      return {status="valid"}
    end
  elseif x <= 0x08FE then
    -- ARABIC CURLY FATHA..ARABIC DAMMA WITH DOT
    if x >= 0x08E4 then
      return {status="valid"}
    end
  elseif x <= 0x094D then
    -- DEVANAGARI LETTER A..DEVANAGARI LETTER HA
    -- DEVANAGARI VOWEL SIGN OE..DEVANAGARI VOWEL SIGN OOE
    -- DEVANAGARI SIGN NUKTA..DEVANAGARI SIGN VIRAMA
    if x >= 0x0905 then
      return {status="valid"}
    end
  elseif x <= 0x096F then
    -- DEVANAGARI DIGIT ZERO..DEVANAGARI DIGIT NINE
    if x >= 0x0966 then
      return {status="valid"}
    end
  elseif x <= 0x0977 then
    -- DEVANAGARI SIGN HIGH SPACING DOT..DEVANAGARI LETTER CANDRA A
    -- DEVANAGARI LETTER OE..DEVANAGARI LETTER UUE
    if x >= 0x0971 then
      return {status="valid"}
    end
  elseif x <= 0x098C then
    -- BENGALI LETTER A..BENGALI LETTER VOCALIC L
    if x >= 0x0985 then
      return {status="valid"}
    end
  elseif x <= 0x09A8 then
    -- BENGALI LETTER O..BENGALI LETTER NA
    if x >= 0x0993 then
      return {status="valid"}
    end
  elseif x <= 0x09B0 then
    -- BENGALI LETTER PA..BENGALI LETTER RA
    if x >= 0x09AA then
      return {status="valid"}
    end
  elseif x <= 0x09C4 then
    -- BENGALI VOWEL SIGN AA..BENGALI VOWEL SIGN VOCALIC RR
    if x >= 0x09BE then
      return {status="valid"}
    end
  elseif x <= 0x09D6 then
    -- NA   <reserved-09CF>..<reserved-09D6>
    if x >= 0x09CF then
      return {status="disallowed"}
    end
  elseif x <= 0x09F1 then
    -- BENGALI DIGIT ZERO..BENGALI LETTER RA WITH LOWER DIAGONAL
    if x >= 0x09E6 then
      return {status="valid"}
    end
  elseif x <= 0x09FA then
    -- BENGALI RUPEE MARK..BENGALI ISSHAR
    if x >= 0x09F2 then
      return {status="disallowed"}
    end
  elseif x <= 0x0A0A then
    -- GURMUKHI LETTER A..GURMUKHI LETTER UU
    if x >= 0x0A05 then
      return {status="valid"}
    end
  elseif x <= 0x0A28 then
    -- GURMUKHI LETTER OO..GURMUKHI LETTER NA
    if x >= 0x0A13 then
      return {status="valid"}
    end
  elseif x <= 0x0A30 then
    -- GURMUKHI LETTER PA..GURMUKHI LETTER RA
    if x >= 0x0A2A then
      return {status="valid"}
    end
  elseif x <= 0x0A58 then
    -- NA   <reserved-0A52>..<reserved-0A58>
    if x >= 0x0A52 then
      return {status="disallowed"}
    end
  elseif x <= 0x0A65 then
    -- NA   <reserved-0A5F>..<reserved-0A65>
    if x >= 0x0A5F then
      return {status="disallowed"}
    end
  elseif x <= 0x0A74 then
    -- GURMUKHI DIGIT ZERO..GURMUKHI EK ONKAR
    if x >= 0x0A66 then
      return {status="valid"}
    end
  elseif x <= 0x0A80 then
    -- NA   <reserved-0A76>..<reserved-0A80>
    if x >= 0x0A76 then
      return {status="disallowed"}
    end
  elseif x <= 0x0A8B then
    -- GUJARATI LETTER A..GUJARATI LETTER VOCALIC R
    if x >= 0x0A85 then
      return {status="valid"}
    end
  elseif x <= 0x0AA8 then
    -- GUJARATI LETTER O..GUJARATI LETTER NA
    if x >= 0x0A93 then
      return {status="valid"}
    end
  elseif x <= 0x0AB0 then
    -- GUJARATI LETTER PA..GUJARATI LETTER RA
    if x >= 0x0AAA then
      return {status="valid"}
    end
  elseif x <= 0x0AC5 then
    -- GUJARATI SIGN NUKTA..GUJARATI VOWEL SIGN CANDRA E
    if x >= 0x0ABC then
      return {status="valid"}
    end
  elseif x <= 0x0ADF then
    -- NA   <reserved-0AD1>..<reserved-0ADF>
    if x >= 0x0AD1 then
      return {status="disallowed"}
    end
  elseif x <= 0x0AEF then
    -- GUJARATI DIGIT ZERO..GUJARATI DIGIT NINE
    if x >= 0x0AE6 then
      return {status="valid"}
    end
  elseif x <= 0x0AF8 then
    -- NA   <reserved-0AF2>..<reserved-0AF8>
    if x >= 0x0AF2 then
      return {status="disallowed"}
    end
  elseif x <= 0x0AFF then
    -- GUJARATI SIGN SUKUN..GUJARATI SIGN TWO-CIRCLE NUKTA ABOVE
    if x >= 0x0AFA then
      return {status="valid"}
    end
  elseif x <= 0x0B0C then
    -- ORIYA LETTER A..ORIYA LETTER VOCALIC L
    if x >= 0x0B05 then
      return {status="valid"}
    end
  elseif x <= 0x0B28 then
    -- ORIYA LETTER O..ORIYA LETTER NA
    if x >= 0x0B13 then
      return {status="valid"}
    end
  elseif x <= 0x0B30 then
    -- ORIYA LETTER PA..ORIYA LETTER RA
    if x >= 0x0B2A then
      return {status="valid"}
    end
  elseif x <= 0x0B43 then
    -- ORIYA SIGN NUKTA..ORIYA VOWEL SIGN VOCALIC R
    if x >= 0x0B3C then
      return {status="valid"}
    end
  elseif x <= 0x0B55 then
    -- NA   <reserved-0B4E>..<reserved-0B55>
    if x >= 0x0B4E then
      return {status="disallowed"}
    end
  elseif x <= 0x0B6F then
    -- ORIYA DIGIT ZERO..ORIYA DIGIT NINE
    if x >= 0x0B66 then
      return {status="valid"}
    end
  elseif x <= 0x0B77 then
    -- ORIYA FRACTION ONE QUARTER..ORIYA FRACTION THREE SIXTEENTHS
    if x >= 0x0B72 then
      return {status="disallowed"}
    end
  elseif x <= 0x0B81 then
    -- NA   <reserved-0B78>..<reserved-0B81>
    if x >= 0x0B78 then
      return {status="disallowed"}
    end
  elseif x <= 0x0B8A then
    -- TAMIL LETTER A..TAMIL LETTER UU
    if x >= 0x0B85 then
      return {status="valid"}
    end
  elseif x <= 0x0BB5 then
    -- TAMIL LETTER MA..TAMIL LETTER VA
    if x >= 0x0BAE then
      return {status="valid"}
    end
  elseif x <= 0x0BD6 then
    -- NA   <reserved-0BD1>..<reserved-0BD6>
    if x >= 0x0BD1 then
      return {status="disallowed"}
    end
  elseif x <= 0x0BE5 then
    -- NA   <reserved-0BD8>..<reserved-0BE5>
    if x >= 0x0BD8 then
      return {status="disallowed"}
    end
  elseif x <= 0x0BEF then
    -- TAMIL DIGIT ONE..TAMIL DIGIT NINE
    if x >= 0x0BE7 then
      return {status="valid"}
    end
  elseif x <= 0x0BFA then
    -- TAMIL NUMBER TEN..TAMIL NUMBER ONE THOUSAND
    -- TAMIL DAY SIGN..TAMIL NUMBER SIGN
    if x >= 0x0BF0 then
      return {status="disallowed"}
    end
  elseif x <= 0x0C0C then
    -- TELUGU LETTER A..TELUGU LETTER VOCALIC L
    if x >= 0x0C05 then
      return {status="valid"}
    end
  elseif x <= 0x0C28 then
    -- TELUGU LETTER O..TELUGU LETTER NA
    if x >= 0x0C12 then
      return {status="valid"}
    end
  elseif x <= 0x0C33 then
    -- TELUGU LETTER PA..TELUGU LETTER LLA
    if x >= 0x0C2A then
      return {status="valid"}
    end
  elseif x <= 0x0C44 then
    -- TELUGU VOWEL SIGN AA..TELUGU VOWEL SIGN VOCALIC RR
    if x >= 0x0C3E then
      return {status="valid"}
    end
  elseif x <= 0x0C54 then
    -- NA   <reserved-0C4E>..<reserved-0C54>
    if x >= 0x0C4E then
      return {status="disallowed"}
    end
  elseif x <= 0x0C6F then
    -- TELUGU DIGIT ZERO..TELUGU DIGIT NINE
    if x >= 0x0C66 then
      return {status="valid"}
    end
  elseif x <= 0x0C77 then
    -- NA   <reserved-0C70>..<reserved-0C77>
    if x >= 0x0C70 then
      return {status="disallowed"}
    end
  elseif x <= 0x0C7F then
    -- TELUGU FRACTION DIGIT ZERO FOR ODD POWERS OF FOUR..TELUGU SIGN TUUMU
    if x >= 0x0C78 then
      return {status="disallowed"}
    end
  elseif x <= 0x0C8C then
    -- KANNADA LETTER A..KANNADA LETTER VOCALIC L
    if x >= 0x0C85 then
      return {status="valid"}
    end
  elseif x <= 0x0CA8 then
    -- KANNADA LETTER O..KANNADA LETTER NA
    if x >= 0x0C92 then
      return {status="valid"}
    end
  elseif x <= 0x0CB3 then
    -- KANNADA LETTER PA..KANNADA LETTER LLA
    if x >= 0x0CAA then
      return {status="valid"}
    end
  elseif x <= 0x0CC4 then
    -- KANNADA SIGN NUKTA..KANNADA SIGN AVAGRAHA
    -- KANNADA VOWEL SIGN AA..KANNADA VOWEL SIGN VOCALIC RR
    if x >= 0x0CBC then
      return {status="valid"}
    end
  elseif x <= 0x0CD4 then
    -- NA   <reserved-0CCE>..<reserved-0CD4>
    if x >= 0x0CCE then
      return {status="disallowed"}
    end
  elseif x <= 0x0CDD then
    -- NA   <reserved-0CD7>..<reserved-0CDD>
    if x >= 0x0CD7 then
      return {status="disallowed"}
    end
  elseif x <= 0x0CEF then
    -- KANNADA DIGIT ZERO..KANNADA DIGIT NINE
    if x >= 0x0CE6 then
      return {status="valid"}
    end
  elseif x <= 0x0CFF then
    -- NA   <reserved-0CF3>..<reserved-0CFF>
    if x >= 0x0CF3 then
      return {status="disallowed"}
    end
  elseif x <= 0x0D0C then
    -- MALAYALAM LETTER A..MALAYALAM LETTER VOCALIC L
    if x >= 0x0D05 then
      return {status="valid"}
    end
  elseif x <= 0x0D28 then
    -- MALAYALAM LETTER O..MALAYALAM LETTER NA
    if x >= 0x0D12 then
      return {status="valid"}
    end
  elseif x <= 0x0D39 then
    -- MALAYALAM LETTER PA..MALAYALAM LETTER HA
    if x >= 0x0D2A then
      return {status="valid"}
    end
  elseif x <= 0x0D43 then
    -- MALAYALAM VOWEL SIGN AA..MALAYALAM VOWEL SIGN VOCALIC R
    if x >= 0x0D3E then
      return {status="valid"}
    end
  elseif x <= 0x0D5E then
    -- MALAYALAM FRACTION ONE ONE-HUNDRED-AND-SIXTIETH..MALAYALAM FRACTION ONE FIFTH
    if x >= 0x0D58 then
      return {status="disallowed"}
    end
  elseif x <= 0x0D6F then
    -- MALAYALAM DIGIT ZERO..MALAYALAM DIGIT NINE
    if x >= 0x0D66 then
      return {status="valid"}
    end
  elseif x <= 0x0D78 then
    -- MALAYALAM NUMBER TEN..MALAYALAM FRACTION THREE QUARTERS
    -- MALAYALAM FRACTION ONE SIXTEENTH..MALAYALAM FRACTION THREE SIXTEENTHS
    if x >= 0x0D70 then
      return {status="disallowed"}
    end
  elseif x <= 0x0D7F then
    -- MALAYALAM LETTER CHILLU NN..MALAYALAM LETTER CHILLU K
    if x >= 0x0D7A then
      return {status="valid"}
    end
  elseif x <= 0x0D96 then
    -- SINHALA LETTER AYANNA..SINHALA LETTER AUYANNA
    if x >= 0x0D85 then
      return {status="valid"}
    end
  elseif x <= 0x0DB1 then
    -- SINHALA LETTER ALPAPRAANA KAYANNA..SINHALA LETTER DANTAJA NAYANNA
    if x >= 0x0D9A then
      return {status="valid"}
    end
  elseif x <= 0x0DBB then
    -- SINHALA LETTER SANYAKA DAYANNA..SINHALA LETTER RAYANNA
    if x >= 0x0DB3 then
      return {status="valid"}
    end
  elseif x <= 0x0DC6 then
    -- SINHALA LETTER VAYANNA..SINHALA LETTER FAYANNA
    if x >= 0x0DC0 then
      return {status="valid"}
    end
  elseif x <= 0x0DD4 then
    -- SINHALA VOWEL SIGN AELA-PILLA..SINHALA VOWEL SIGN KETTI PAA-PILLA
    if x >= 0x0DCF then
      return {status="valid"}
    end
  elseif x <= 0x0DDF then
    -- SINHALA VOWEL SIGN GAETTA-PILLA..SINHALA VOWEL SIGN GAYANUKITTA
    if x >= 0x0DD8 then
      return {status="valid"}
    end
  elseif x <= 0x0DE5 then
    -- NA   <reserved-0DE0>..<reserved-0DE5>
    if x >= 0x0DE0 then
      return {status="disallowed"}
    end
  elseif x <= 0x0DEF then
    -- SINHALA LITH DIGIT ZERO..SINHALA LITH DIGIT NINE
    if x >= 0x0DE6 then
      return {status="valid"}
    end
  elseif x <= 0x0E00 then
    -- NA   <reserved-0DF5>..<reserved-0E00>
    if x >= 0x0DF5 then
      return {status="disallowed"}
    end
  elseif x <= 0x0E32 then
    -- THAI CHARACTER KO KAI..THAI CHARACTER SARA AA
    if x >= 0x0E01 then
      return {status="valid"}
    end
  elseif x <= 0x0E3A then
    -- THAI CHARACTER SARA I..THAI CHARACTER PHINTHU
    if x >= 0x0E34 then
      return {status="valid"}
    end
  elseif x <= 0x0E4E then
    -- THAI CHARACTER SARA E..THAI CHARACTER YAMAKKAN
    if x >= 0x0E40 then
      return {status="valid"}
    end
  elseif x <= 0x0E59 then
    -- THAI DIGIT ZERO..THAI DIGIT NINE
    if x >= 0x0E50 then
      return {status="valid"}
    end
  elseif x <= 0x0E80 then
    -- NA   <reserved-0E5C>..<reserved-0E80>
    if x >= 0x0E5C then
      return {status="disallowed"}
    end
  elseif x <= 0x0E93 then
    -- NA   <reserved-0E8E>..<reserved-0E93>
    if x >= 0x0E8E then
      return {status="disallowed"}
    end
  elseif x <= 0x0E9F then
    -- LAO LETTER NO..LAO LETTER FO SUNG
    if x >= 0x0E99 then
      return {status="valid"}
    end
  elseif x <= 0x0EB2 then
    -- LAO LETTER O..LAO VOWEL SIGN AA
    if x >= 0x0EAD then
      return {status="valid"}
    end
  elseif x <= 0x0EB9 then
    -- LAO VOWEL SIGN I..LAO VOWEL SIGN UU
    if x >= 0x0EB4 then
      return {status="valid"}
    end
  elseif x <= 0x0ECD then
    -- LAO TONE MAI EK..LAO NIGGAHITA
    if x >= 0x0EC8 then
      return {status="valid"}
    end
  elseif x <= 0x0ED9 then
    -- LAO DIGIT ZERO..LAO DIGIT NINE
    if x >= 0x0ED0 then
      return {status="valid"}
    end
  elseif x <= 0x0EFF then
    -- NA   <reserved-0EE0>..<reserved-0EFF>
    if x >= 0x0EE0 then
      return {status="disallowed"}
    end
  elseif x <= 0x0F0A then
    -- TIBETAN MARK GTER YIG MGO TRUNCATED A..TIBETAN MARK BKA- SHOG YIG MGO
    if x >= 0x0F01 then
      return {status="disallowed"}
    end
  elseif x <= 0x0F17 then
    -- TIBETAN MARK SHAD..TIBETAN ASTROLOGICAL SIGN SGRA GCAN -CHAR RTAGS
    if x >= 0x0F0D then
      return {status="disallowed"}
    end
  elseif x <= 0x0F1F then
    -- TIBETAN SIGN RDEL DKAR GCIG..TIBETAN SIGN RDEL DKAR RDEL NAG
    if x >= 0x0F1A then
      return {status="disallowed"}
    end
  elseif x <= 0x0F29 then
    -- TIBETAN DIGIT ZERO..TIBETAN DIGIT NINE
    if x >= 0x0F20 then
      return {status="valid"}
    end
  elseif x <= 0x0F34 then
    -- TIBETAN DIGIT HALF ONE..TIBETAN MARK BSDUS RTAGS
    if x >= 0x0F2A then
      return {status="disallowed"}
    end
  elseif x <= 0x0F68 then
    -- TIBETAN LETTER WA..TIBETAN LETTER A
    if x >= 0x0F5D then
      return {status="valid"}
    end
  elseif x <= 0x0F80 then
    -- TIBETAN VOWEL SIGN E..TIBETAN VOWEL SIGN REVERSED I
    if x >= 0x0F7A then
      return {status="valid"}
    end
  elseif x <= 0x0F92 then
    -- TIBETAN SIGN LCI RTAGS..TIBETAN SIGN GRU MED RGYINGS
    -- TIBETAN SIGN INVERTED MCHU CAN..TIBETAN SUBJOINED SIGN INVERTED MCHU CAN
    -- TIBETAN SUBJOINED LETTER KA..TIBETAN SUBJOINED LETTER GA
    if x >= 0x0F86 then
      return {status="valid"}
    end
  elseif x <= 0x0FB7 then
    -- TIBETAN SUBJOINED LETTER ZHA..TIBETAN SUBJOINED LETTER -A
    -- TIBETAN SUBJOINED LETTER YA..TIBETAN SUBJOINED LETTER HA
    if x >= 0x0FAE then
      return {status="valid"}
    end
  elseif x <= 0x0FC5 then
    -- TIBETAN KU RU KHA..TIBETAN SYMBOL RDO RJE
    if x >= 0x0FBE then
      return {status="disallowed"}
    end
  elseif x <= 0x0FCC then
    -- TIBETAN SYMBOL RDO RJE RGYA GRAM..TIBETAN SYMBOL NOR BU BZHI -KHYIL
    if x >= 0x0FC7 then
      return {status="disallowed"}
    end
  elseif x <= 0x0FDA then
    -- TIBETAN MARK BSKA- SHOG GI MGO RGYAN..TIBETAN MARK MNYAM YIG GI MGO RGYAN
    -- TIBETAN MARK NYIS TSHEG..TIBETAN MARK CLOSING BRDA RNYING YIG MGO SGAB MA
    -- RIGHT-FACING SVASTI SIGN..LEFT-FACING SVASTI SIGN WITH DOTS
    -- TIBETAN MARK LEADING MCHAN RTAGS..TIBETAN MARK TRAILING MCHAN RTAGS
    if x >= 0x0FD0 then
      return {status="disallowed"}
    end
  elseif x <= 0x0FFF then
    -- NA   <reserved-0FDB>..<reserved-0FFF>
    if x >= 0x0FDB then
      return {status="disallowed"}
    end
  elseif x <= 0x1021 then
    -- MYANMAR LETTER KA..MYANMAR LETTER A
    if x >= 0x1000 then
      return {status="valid"}
    end
  elseif x <= 0x1049 then
    -- MYANMAR VOWEL SIGN AA..MYANMAR VOWEL SIGN AI
    -- MYANMAR VOWEL SIGN MON II..MYANMAR VOWEL SIGN E ABOVE
    -- MYANMAR SIGN ANUSVARA..MYANMAR SIGN VIRAMA
    -- MYANMAR SIGN ASAT..MYANMAR LETTER GREAT SA
    -- MYANMAR DIGIT ZERO..MYANMAR DIGIT NINE
    if x >= 0x102C then
      return {status="valid"}
    end
  elseif x <= 0x104F then
    -- MYANMAR SIGN LITTLE SECTION..MYANMAR SYMBOL GENITIVE
    if x >= 0x104A then
      return {status="disallowed"}
    end
  elseif x <= 0x109D then
    -- MYANMAR LETTER SHA..MYANMAR VOWEL SIGN VOCALIC LL
    -- MYANMAR LETTER MON NGA..MYANMAR SHAN DIGIT NINE
    -- MYANMAR SIGN KHAMTI TONE-1..MYANMAR VOWEL SIGN AITON AI
    if x >= 0x1050 then
      return {status="valid"}
    end
  elseif x <= 0x10C5 then
    -- GEORGIAN CAPITAL LETTER AN..GEORGIAN CAPITAL LETTER HOE
    if x >= 0x10A0 then
      return {status="disallowed"}
    end
  elseif x <= 0x10FA then
    -- GEORGIAN LETTER AN..GEORGIAN LETTER FI
    -- GEORGIAN LETTER YN..GEORGIAN LETTER ELIFI
    -- GEORGIAN LETTER TURNED GAN..GEORGIAN LETTER AIN
    if x >= 0x10D0 then
      return {status="valid"}
    end
  elseif x <= 0x115E then
    -- HANGUL CHOSEONG KIYEOK..HANGUL CHOSEONG YEORINHIEUH
    -- HANGUL CHOSEONG KIYEOK-TIKEUT..HANGUL CHOSEONG TIKEUT-RIEUL
    if x >= 0x1100 then
      return {status="disallowed"}
    end
  elseif x <= 0x11FF then
    -- HANGUL JUNGSEONG A..HANGUL JUNGSEONG SSANGARAEA
    -- HANGUL JUNGSEONG A-EU..HANGUL JUNGSEONG O-YAE
    -- HANGUL JONGSEONG KIYEOK..HANGUL JONGSEONG YEORINHIEUH
    -- HANGUL JONGSEONG KIYEOK-NIEUN..HANGUL JONGSEONG SSANGNIEUN
    if x >= 0x1161 then
      return {status="disallowed"}
    end
  elseif x <= 0x1206 then
    -- ETHIOPIC SYLLABLE HA..ETHIOPIC SYLLABLE HO
    if x >= 0x1200 then
      return {status="valid"}
    end
  elseif x <= 0x1246 then
    -- ETHIOPIC SYLLABLE LA..ETHIOPIC SYLLABLE QO
    if x >= 0x1208 then
      return {status="valid"}
    end
  elseif x <= 0x1256 then
    -- ETHIOPIC SYLLABLE QHA..ETHIOPIC SYLLABLE QHO
    if x >= 0x1250 then
      return {status="valid"}
    end
  elseif x <= 0x1286 then
    -- ETHIOPIC SYLLABLE BA..ETHIOPIC SYLLABLE XO
    if x >= 0x1260 then
      return {status="valid"}
    end
  elseif x <= 0x12AE then
    -- ETHIOPIC SYLLABLE NA..ETHIOPIC SYLLABLE KO
    if x >= 0x1290 then
      return {status="valid"}
    end
  elseif x <= 0x12BE then
    -- ETHIOPIC SYLLABLE KXA..ETHIOPIC SYLLABLE KXO
    if x >= 0x12B8 then
      return {status="valid"}
    end
  elseif x <= 0x12CE then
    -- ETHIOPIC SYLLABLE WA..ETHIOPIC SYLLABLE WO
    if x >= 0x12C8 then
      return {status="valid"}
    end
  elseif x <= 0x12D6 then
    -- ETHIOPIC SYLLABLE PHARYNGEAL A..ETHIOPIC SYLLABLE PHARYNGEAL O
    if x >= 0x12D0 then
      return {status="valid"}
    end
  elseif x <= 0x12EE then
    -- ETHIOPIC SYLLABLE ZA..ETHIOPIC SYLLABLE YO
    if x >= 0x12D8 then
      return {status="valid"}
    end
  elseif x <= 0x130E then
    -- ETHIOPIC SYLLABLE DA..ETHIOPIC SYLLABLE GO
    if x >= 0x12F0 then
      return {status="valid"}
    end
  elseif x <= 0x131E then
    -- ETHIOPIC SYLLABLE GGA..ETHIOPIC SYLLABLE GGO
    if x >= 0x1318 then
      return {status="valid"}
    end
  elseif x <= 0x1346 then
    -- ETHIOPIC SYLLABLE THA..ETHIOPIC SYLLABLE TZO
    if x >= 0x1320 then
      return {status="valid"}
    end
  elseif x <= 0x135A then
    -- ETHIOPIC SYLLABLE FA..ETHIOPIC SYLLABLE FYA
    if x >= 0x1348 then
      return {status="valid"}
    end
  elseif x <= 0x137C then
    -- ETHIOPIC WORDSPACE..ETHIOPIC NUMBER TEN THOUSAND
    if x >= 0x1361 then
      return {status="disallowed"}
    end
  elseif x <= 0x138F then
    -- ETHIOPIC SYLLABLE SEBATBEIT MWA..ETHIOPIC SYLLABLE PWE
    if x >= 0x1380 then
      return {status="valid"}
    end
  elseif x <= 0x1399 then
    -- ETHIOPIC TONAL MARK YIZET..ETHIOPIC TONAL MARK KURT
    if x >= 0x1390 then
      return {status="disallowed"}
    end
  elseif x <= 0x139F then
    -- NA   <reserved-139A>..<reserved-139F>
    if x >= 0x139A then
      return {status="disallowed"}
    end
  elseif x <= 0x13F4 then
    -- CHEROKEE LETTER A..CHEROKEE LETTER YV
    if x >= 0x13A0 then
      return {status="valid"}
    end
  elseif x <= 0x166C then
    -- CANADIAN SYLLABICS E..CANADIAN SYLLABICS CARRIER TTSA
    if x >= 0x1401 then
      return {status="valid"}
    end
  elseif x <= 0x167F then
    -- CANADIAN SYLLABICS QAI..CANADIAN SYLLABICS NNGAA
    -- CANADIAN SYLLABICS WOODS-CREE THWEE..CANADIAN SYLLABICS BLACKFOOT W
    if x >= 0x166F then
      return {status="valid"}
    end
  elseif x <= 0x169A then
    -- OGHAM LETTER BEITH..OGHAM LETTER PEITH
    if x >= 0x1681 then
      return {status="valid"}
    end
  elseif x <= 0x16EA then
    -- RUNIC LETTER FEHU FEOH FE F..RUNIC LETTER X
    if x >= 0x16A0 then
      return {status="valid"}
    end
  elseif x <= 0x16F0 then
    -- RUNIC SINGLE PUNCTUATION..RUNIC BELGTHOR SYMBOL
    if x >= 0x16EB then
      return {status="disallowed"}
    end
  elseif x <= 0x16F8 then
    -- RUNIC LETTER K..RUNIC LETTER FRANKS CASKET AESC
    if x >= 0x16F1 then
      return {status="valid"}
    end
  elseif x <= 0x16FF then
    -- NA   <reserved-16F9>..<reserved-16FF>
    if x >= 0x16F9 then
      return {status="disallowed"}
    end
  elseif x <= 0x170C then
    -- TAGALOG LETTER A..TAGALOG LETTER YA
    if x >= 0x1700 then
      return {status="valid"}
    end
  elseif x <= 0x1714 then
    -- TAGALOG LETTER LA..TAGALOG SIGN VIRAMA
    if x >= 0x170E then
      return {status="valid"}
    end
  elseif x <= 0x171F then
    -- NA   <reserved-1715>..<reserved-171F>
    if x >= 0x1715 then
      return {status="disallowed"}
    end
  elseif x <= 0x1734 then
    -- HANUNOO LETTER A..HANUNOO SIGN PAMUDPOD
    if x >= 0x1720 then
      return {status="valid"}
    end
  elseif x <= 0x173F then
    -- NA   <reserved-1737>..<reserved-173F>
    if x >= 0x1737 then
      return {status="disallowed"}
    end
  elseif x <= 0x1753 then
    -- BUHID LETTER A..BUHID VOWEL SIGN U
    if x >= 0x1740 then
      return {status="valid"}
    end
  elseif x <= 0x175F then
    -- NA   <reserved-1754>..<reserved-175F>
    if x >= 0x1754 then
      return {status="disallowed"}
    end
  elseif x <= 0x176C then
    -- TAGBANWA LETTER A..TAGBANWA LETTER YA
    if x >= 0x1760 then
      return {status="valid"}
    end
  elseif x <= 0x177F then
    -- NA   <reserved-1774>..<reserved-177F>
    if x >= 0x1774 then
      return {status="disallowed"}
    end
  elseif x <= 0x17B3 then
    -- KHMER LETTER KA..KHMER INDEPENDENT VOWEL QAU
    if x >= 0x1780 then
      return {status="valid"}
    end
  elseif x <= 0x17D3 then
    -- KHMER VOWEL SIGN AA..KHMER SIGN BATHAMASAT
    if x >= 0x17B6 then
      return {status="valid"}
    end
  elseif x <= 0x17E9 then
    -- KHMER DIGIT ZERO..KHMER DIGIT NINE
    if x >= 0x17E0 then
      return {status="valid"}
    end
  elseif x <= 0x17EF then
    -- NA   <reserved-17EA>..<reserved-17EF>
    if x >= 0x17EA then
      return {status="disallowed"}
    end
  elseif x <= 0x17F9 then
    -- KHMER SYMBOL LEK ATTAK SON..KHMER SYMBOL LEK ATTAK PRAM-BUON
    if x >= 0x17F0 then
      return {status="disallowed"}
    end
  elseif x <= 0x17FF then
    -- NA   <reserved-17FA>..<reserved-17FF>
    if x >= 0x17FA then
      return {status="disallowed"}
    end
  elseif x <= 0x1805 then
    -- MONGOLIAN BIRGA..MONGOLIAN FOUR DOTS
    if x >= 0x1800 then
      return {status="disallowed"}
    end
  elseif x <= 0x1819 then
    -- MONGOLIAN DIGIT ZERO..MONGOLIAN DIGIT NINE
    if x >= 0x1810 then
      return {status="valid"}
    end
  elseif x <= 0x181F then
    -- NA   <reserved-181A>..<reserved-181F>
    if x >= 0x181A then
      return {status="disallowed"}
    end
  elseif x <= 0x1877 then
    -- MONGOLIAN LETTER A..MONGOLIAN LETTER MANCHU ZHA
    if x >= 0x1820 then
      return {status="valid"}
    end
  elseif x <= 0x187F then
    -- NA   <reserved-1878>..<reserved-187F>
    if x >= 0x1878 then
      return {status="disallowed"}
    end
  elseif x <= 0x18A9 then
    -- MONGOLIAN LETTER ALI GALI ANUSVARA ONE..MONGOLIAN LETTER ALI GALI DAGALGA
    if x >= 0x1880 then
      return {status="valid"}
    end
  elseif x <= 0x18F5 then
    -- CANADIAN SYLLABICS OY..CANADIAN SYLLABICS CARRIER DENTAL S
    if x >= 0x18B0 then
      return {status="valid"}
    end
  elseif x <= 0x18FF then
    -- NA   <reserved-18F6>..<reserved-18FF>
    if x >= 0x18F6 then
      return {status="disallowed"}
    end
  elseif x <= 0x191E then
    -- LIMBU VOWEL-CARRIER LETTER..LIMBU LETTER HA
    -- LIMBU LETTER GYAN..LIMBU LETTER TRA
    if x >= 0x1900 then
      return {status="valid"}
    end
  elseif x <= 0x192B then
    -- LIMBU VOWEL SIGN A..LIMBU SUBJOINED LETTER WA
    if x >= 0x1920 then
      return {status="valid"}
    end
  elseif x <= 0x193B then
    -- LIMBU SMALL LETTER KA..LIMBU SIGN SA-I
    if x >= 0x1930 then
      return {status="valid"}
    end
  elseif x <= 0x196D then
    -- LIMBU DIGIT ZERO..TAI LE LETTER AI
    if x >= 0x1946 then
      return {status="valid"}
    end
  elseif x <= 0x197F then
    -- NA   <reserved-1975>..<reserved-197F>
    if x >= 0x1975 then
      return {status="disallowed"}
    end
  elseif x <= 0x19AB then
    -- NEW TAI LUE LETTER HIGH QA..NEW TAI LUE LETTER LOW XVA
    -- NEW TAI LUE LETTER HIGH SUA..NEW TAI LUE LETTER LOW SUA
    if x >= 0x1980 then
      return {status="valid"}
    end
  elseif x <= 0x19C9 then
    -- NEW TAI LUE VOWEL SIGN VOWEL SHORTENER..NEW TAI LUE TONE MARK-2
    if x >= 0x19B0 then
      return {status="valid"}
    end
  elseif x <= 0x19CF then
    -- NA   <reserved-19CA>..<reserved-19CF>
    if x >= 0x19CA then
      return {status="disallowed"}
    end
  elseif x <= 0x19D9 then
    -- NEW TAI LUE DIGIT ZERO..NEW TAI LUE DIGIT NINE
    if x >= 0x19D0 then
      return {status="valid"}
    end
  elseif x <= 0x19FF then
    -- NEW TAI LUE SIGN LAE..NEW TAI LUE SIGN LAEV
    -- KHMER SYMBOL PATHAMASAT..KHMER SYMBOL DAP-PRAM ROC
    if x >= 0x19DE then
      return {status="disallowed"}
    end
  elseif x <= 0x1A1B then
    -- BUGINESE LETTER KA..BUGINESE VOWEL SIGN AE
    if x >= 0x1A00 then
      return {status="valid"}
    end
  elseif x <= 0x1A5E then
    -- TAI THAM LETTER HIGH KA..TAI THAM CONSONANT SIGN SA
    if x >= 0x1A20 then
      return {status="valid"}
    end
  elseif x <= 0x1A7C then
    -- TAI THAM SIGN SAKOT..TAI THAM SIGN KHUEN-LUE KARAN
    if x >= 0x1A60 then
      return {status="valid"}
    end
  elseif x <= 0x1A89 then
    -- TAI THAM COMBINING CRYPTOGRAMMIC DOT..TAI THAM HORA DIGIT NINE
    if x >= 0x1A7F then
      return {status="valid"}
    end
  elseif x <= 0x1A8F then
    -- NA   <reserved-1A8A>..<reserved-1A8F>
    if x >= 0x1A8A then
      return {status="disallowed"}
    end
  elseif x <= 0x1A99 then
    -- TAI THAM THAM DIGIT ZERO..TAI THAM THAM DIGIT NINE
    if x >= 0x1A90 then
      return {status="valid"}
    end
  elseif x <= 0x1A9F then
    -- NA   <reserved-1A9A>..<reserved-1A9F>
    if x >= 0x1A9A then
      return {status="disallowed"}
    end
  elseif x <= 0x1AA6 then
    -- TAI THAM SIGN WIANG..TAI THAM SIGN REVERSED ROTATED RANA
    if x >= 0x1AA0 then
      return {status="disallowed"}
    end
  elseif x <= 0x1AAD then
    -- TAI THAM SIGN KAAN..TAI THAM SIGN CAANG
    if x >= 0x1AA8 then
      return {status="disallowed"}
    end
  elseif x <= 0x1ABD then
    -- COMBINING DOUBLED CIRCUMFLEX ACCENT..COMBINING PARENTHESES BELOW
    if x >= 0x1AB0 then
      return {status="valid"}
    end
  elseif x <= 0x1AFF then
    -- NA   <reserved-1ABF>..<reserved-1AFF>
    if x >= 0x1ABF then
      return {status="disallowed"}
    end
  elseif x <= 0x1B4B then
    -- BALINESE SIGN ULU RICEM..BALINESE LETTER ASYURA SASAK
    if x >= 0x1B00 then
      return {status="valid"}
    end
  elseif x <= 0x1B59 then
    -- BALINESE DIGIT ZERO..BALINESE DIGIT NINE
    if x >= 0x1B50 then
      return {status="valid"}
    end
  elseif x <= 0x1B6A then
    -- BALINESE PANTI..BALINESE MUSICAL SYMBOL DANG GEDE
    if x >= 0x1B5A then
      return {status="disallowed"}
    end
  elseif x <= 0x1B73 then
    -- BALINESE MUSICAL SYMBOL COMBINING TEGEH..BALINESE MUSICAL SYMBOL COMBINING GONG
    if x >= 0x1B6B then
      return {status="valid"}
    end
  elseif x <= 0x1B7C then
    -- BALINESE MUSICAL SYMBOL RIGHT-HAND OPEN DUG..BALINESE MUSICAL SYMBOL LEFT-HAND OPEN PING
    if x >= 0x1B74 then
      return {status="disallowed"}
    end
  elseif x <= 0x1BF3 then
    -- SUNDANESE SIGN PANYECEK..SUNDANESE SIGN PAMAAEH
    -- SUNDANESE SIGN VIRAMA..SUNDANESE CONSONANT SIGN PASANGAN WA
    -- SUNDANESE LETTER KHA..SUNDANESE DIGIT NINE
    -- SUNDANESE AVAGRAHA..SUNDANESE LETTER FINAL M
    -- BATAK LETTER A..BATAK PANONGONAN
    if x >= 0x1B80 then
      return {status="valid"}
    end
  elseif x <= 0x1BFB then
    -- NA   <reserved-1BF4>..<reserved-1BFB>
    if x >= 0x1BF4 then
      return {status="disallowed"}
    end
  elseif x <= 0x1C37 then
    -- LEPCHA LETTER KA..LEPCHA SIGN NUKTA
    if x >= 0x1C00 then
      return {status="valid"}
    end
  elseif x <= 0x1C49 then
    -- LEPCHA DIGIT ZERO..LEPCHA DIGIT NINE
    if x >= 0x1C40 then
      return {status="valid"}
    end
  elseif x <= 0x1C7D then
    -- LEPCHA LETTER TTA..OL CHIKI AHAD
    if x >= 0x1C4D then
      return {status="valid"}
    end
  elseif x <= 0x1CBF then
    -- NA   <reserved-1C89>..<reserved-1CBF>
    if x >= 0x1C89 then
      return {status="disallowed"}
    end
  elseif x <= 0x1CC7 then
    -- SUNDANESE PUNCTUATION BINDU SURYA..SUNDANESE PUNCTUATION BINDU BA SATANGA
    if x >= 0x1CC0 then
      return {status="disallowed"}
    end
  elseif x <= 0x1CCF then
    -- NA   <reserved-1CC8>..<reserved-1CCF>
    if x >= 0x1CC8 then
      return {status="disallowed"}
    end
  elseif x <= 0x1CF6 then
    -- VEDIC SIGN YAJURVEDIC MIDLINE SVARITA..VEDIC SIGN ARDHAVISARGA
    -- VEDIC SIGN ROTATED ARDHAVISARGA..VEDIC SIGN UPADHMANIYA
    if x >= 0x1CD4 then
      return {status="valid"}
    end
  elseif x <= 0x1CFF then
    -- NA   <reserved-1CFA>..<reserved-1CFF>
    if x >= 0x1CFA then
      return {status="disallowed"}
    end
  elseif x <= 0x1D2B then
    -- LATIN LETTER SMALL CAPITAL A..CYRILLIC LETTER SMALL CAPITAL EL
    if x >= 0x1D00 then
      return {status="valid"}
    end
  elseif x <= 0x1D77 then
    -- LATIN SMALL LETTER B WITH MIDDLE TILDE..LATIN SMALL LETTER TURNED G
    if x >= 0x1D6C then
      return {status="valid"}
    end
  elseif x <= 0x1D9A then
    -- LATIN SMALL LETTER INSULAR G..LATIN SMALL LETTER EZH WITH RETROFLEX HOOK
    if x >= 0x1D79 then
      return {status="valid"}
    end
  elseif x <= 0x1DF9 then
    -- COMBINING DOTTED GRAVE ACCENT..COMBINING SUSPENSION MARK
    -- COMBINING MACRON-ACUTE..COMBINING LATIN SMALL LETTER R BELOW
    -- COMBINING BREVE-MACRON..COMBINING LATIN SMALL LETTER Z
    -- COMBINING LATIN SMALL LETTER ALPHA..COMBINING UP TACK ABOVE
    -- COMBINING KAVYKA ABOVE RIGHT..COMBINING WIDE INVERTED BRIDGE BELOW
    if x >= 0x1DC0 then
      return {status="valid"}
    end
  elseif x <= 0x1F07 then
    -- GREEK SMALL LETTER ALPHA WITH PSILI..GREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI
    if x >= 0x1F00 then
      return {status="valid"}
    end
  elseif x <= 0x1F15 then
    -- GREEK SMALL LETTER EPSILON WITH PSILI..GREEK SMALL LETTER EPSILON WITH DASIA AND OXIA
    if x >= 0x1F10 then
      return {status="valid"}
    end
  elseif x <= 0x1F27 then
    -- GREEK SMALL LETTER ETA WITH PSILI..GREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENI
    if x >= 0x1F20 then
      return {status="valid"}
    end
  elseif x <= 0x1F37 then
    -- GREEK SMALL LETTER IOTA WITH PSILI..GREEK SMALL LETTER IOTA WITH DASIA AND PERISPOMENI
    if x >= 0x1F30 then
      return {status="valid"}
    end
  elseif x <= 0x1F45 then
    -- GREEK SMALL LETTER OMICRON WITH PSILI..GREEK SMALL LETTER OMICRON WITH DASIA AND OXIA
    if x >= 0x1F40 then
      return {status="valid"}
    end
  elseif x <= 0x1F57 then
    -- GREEK SMALL LETTER UPSILON WITH PSILI..GREEK SMALL LETTER UPSILON WITH DASIA AND PERISPOMENI
    if x >= 0x1F50 then
      return {status="valid"}
    end
  elseif x <= 0x1F67 then
    -- GREEK SMALL LETTER OMEGA WITH PSILI..GREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENI
    if x >= 0x1F60 then
      return {status="valid"}
    end
  elseif x <= 0x200A then
    -- EN QUAD..HAIR SPACE
    if x >= 0x2000 then
      return {status="disallowed"}
    end
  elseif x <= 0x2023 then
    -- LEFT SINGLE QUOTATION MARK..TRIANGULAR BULLET
    if x >= 0x2018 then
      return {status="disallowed"}
    end
  elseif x <= 0x202E then
    -- LINE SEPARATOR..RIGHT-TO-LEFT OVERRIDE
    if x >= 0x2028 then
      return {status="disallowed"}
    end
  elseif x <= 0x2046 then
    -- UNDERTIE..RIGHT SQUARE BRACKET WITH QUILL
    if x >= 0x203F then
      return {status="disallowed"}
    end
  elseif x <= 0x2056 then
    -- TIRONIAN SIGN ET..BLACK RIGHTWARDS BULLET
    -- LOW ASTERISK..COMMERCIAL MINUS SIGN
    -- SWUNG DASH..INVERTED UNDERTIE
    -- FLOWER PUNCTUATION MARK..THREE DOT PUNCTUATION
    if x >= 0x204A then
      return {status="disallowed"}
    end
  elseif x <= 0x205E then
    -- FOUR DOT PUNCTUATION..VERTICAL FOUR DOTS
    if x >= 0x2058 then
      return {status="disallowed"}
    end
  elseif x <= 0x206F then
    -- LEFT-TO-RIGHT ISOLATE..POP DIRECTIONAL ISOLATE
    -- INHIBIT SYMMETRIC SWAPPING..NOMINAL DIGIT SHAPES
    if x >= 0x2066 then
      return {status="disallowed"}
    end
  elseif x <= 0x20A7 then
    -- EURO-CURRENCY SIGN..PESETA SIGN
    if x >= 0x20A0 then
      return {status="disallowed"}
    end
  elseif x <= 0x20B8 then
    -- KIP SIGN..DRACHMA SIGN
    -- GERMAN PENNY SIGN..PESO SIGN
    -- GUARANI SIGN..CEDI SIGN
    -- LIVRE TOURNOIS SIGN..TENGE SIGN
    if x >= 0x20AD then
      return {status="disallowed"}
    end
  elseif x <= 0x20CF then
    -- NA   <reserved-20C0>..<reserved-20CF>
    if x >= 0x20C0 then
      return {status="disallowed"}
    end
  elseif x <= 0x20EA then
    -- COMBINING LEFT HARPOON ABOVE..COMBINING LEFT RIGHT ARROW ABOVE
    -- COMBINING ENCLOSING SCREEN..COMBINING ENCLOSING KEYCAP
    -- COMBINING ENCLOSING UPWARD POINTING TRIANGLE..COMBINING LEFTWARDS ARROW OVERLAY
    if x >= 0x20D0 then
      return {status="disallowed"}
    end
  elseif x <= 0x20FF then
    -- NA   <reserved-20F1>..<reserved-20FF>
    if x >= 0x20F1 then
      return {status="disallowed"}
    end
  elseif x <= 0x222B then
    -- LEFTWARDS ARROW..UPWARDS WHITE ARROW FROM BAR
    -- UPWARDS WHITE ARROW ON PEDESTAL..UP DOWN WHITE ARROW
    -- RIGHT ARROW WITH SMALL CIRCLE..LEFT RIGHT OPEN-HEADED ARROW
    -- FOR ALL..INTEGRAL
    if x >= 0x2190 then
      return {status="disallowed"}
    end
  elseif x <= 0x225F then
    -- CLOCKWISE INTEGRAL..QUESTIONED EQUAL TO
    if x >= 0x2231 then
      return {status="disallowed"}
    end
  elseif x <= 0x226D then
    -- IDENTICAL TO..NOT EQUIVALENT TO
    if x >= 0x2261 then
      return {status="disallowed"}
    end
  elseif x <= 0x22FF then
    -- NEITHER LESS-THAN NOR EQUAL TO..DOWN RIGHT DIAGONAL ELLIPSIS
    -- ELEMENT OF WITH LONG HORIZONTAL STROKE..Z NOTATION BAG MEMBERSHIP
    if x >= 0x2270 then
      return {status="disallowed"}
    end
  elseif x <= 0x2328 then
    -- HOUSE..KEYBOARD
    if x >= 0x2302 then
      return {status="disallowed"}
    end
  elseif x <= 0x237A then
    -- ERASE TO THE LEFT..APL FUNCTIONAL SYMBOL ALPHA
    if x >= 0x232B then
      return {status="disallowed"}
    end
  elseif x <= 0x23E7 then
    -- SHOULDERED OPEN BOX..CLEAR SCREEN SYMBOL
    -- LEFT PARENTHESIS UPPER HOOK..return {status="SYMBOL"}
    -- EJECT SYMBOL..VERTICAL LINE EXTENSION
    -- METRICAL BREVE..FUSE
    -- TOP PARENTHESIS..ELECTRICAL INTERSECTION
    if x >= 0x237D then
      return {status="disallowed"}
    end
  elseif x <= 0x23FE then
    -- BLACK RIGHT-POINTING DOUBLE TRIANGLE..HOURGLASS WITH FLOWING SAND
    -- BLACK MEDIUM LEFT-POINTING TRIANGLE..BLACK CIRCLE FOR RECORD
    -- POWER SYMBOL..POWER SLEEP SYMBOL
    if x >= 0x23E9 then
      return {status="disallowed"}
    end
  elseif x <= 0x2426 then
    -- SYMBOL FOR NULL..SYMBOL FOR NEWLINE
    -- SYMBOL FOR DELETE FORM TWO..SYMBOL FOR SUBSTITUTE FORM TWO
    if x >= 0x2400 then
      return {status="disallowed"}
    end
  elseif x <= 0x243F then
    -- NA   <reserved-2427>..<reserved-243F>
    if x >= 0x2427 then
      return {status="disallowed"}
    end
  elseif x <= 0x244A then
    -- OCR HOOK..OCR DOUBLE BACKSLASH
    if x >= 0x2440 then
      return {status="disallowed"}
    end
  elseif x <= 0x245F then
    -- NA   <reserved-244B>..<reserved-245F>
    if x >= 0x244B then
      return {status="disallowed"}
    end
  elseif x <= 0x249B then
    -- DIGIT ONE FULL STOP..NUMBER TWENTY FULL STOP
    if x >= 0x2488 then
      return {status="disallowed"}
    end
  elseif x <= 0x24FE then
    -- NEGATIVE CIRCLED NUMBER ELEVEN..DOUBLE CIRCLED NUMBER TEN
    if x >= 0x24EB then
      return {status="disallowed"}
    end
  elseif x <= 0x2617 then
    -- BOX DRAWINGS LIGHT HORIZONTAL..RIGHT ONE EIGHTH BLOCK
    -- QUADRANT LOWER LEFT..QUADRANT UPPER RIGHT AND LOWER LEFT AND LOWER RIGHT
    -- BLACK SQUARE..LARGE CIRCLE
    -- WHITE SQUARE WITH UPPER LEFT QUADRANT..WHITE CIRCLE WITH UPPER RIGHT QUADRANT
    -- UPPER LEFT TRIANGLE..LOWER RIGHT TRIANGLE
    -- BLACK SUN WITH RAYS..SALTIRE
    -- UMBRELLA WITH RAIN DROPS..HOT BEVERAGE
    -- WHITE SHOGI PIECE..BLACK SHOGI PIECE
    if x >= 0x2500 then
      return {status="disallowed"}
    end
  elseif x <= 0x269C then
    -- BLACK LEFT POINTING INDEX..MUSIC SHARP SIGN
    -- WEST SYRIAC CROSS..EAST SYRIAC CROSS
    -- UNIVERSAL RECYCLING SYMBOL..PARTIALLY-RECYCLED PAPER SYMBOL
    -- PERMANENT PAPER SIGN..WHEELCHAIR SYMBOL
    -- DIE FACE-1..BLACK CIRCLE WITH TWO WHITE DOTS
    -- MONOGRAM FOR YANG..BLACK FLAG
    -- HAMMER AND PICK..FLEUR-DE-LIS
    if x >= 0x261A then
      return {status="disallowed"}
    end
  elseif x <= 0x26B1 then
    -- THREE LINES CONVERGING RIGHT..THREE LINES CONVERGING LEFT
    -- WARNING SIGN..HIGH VOLTAGE SIGN
    -- DOUBLED FEMALE SIGN..FUNERAL URN
    if x >= 0x269E then
      return {status="disallowed"}
    end
  elseif x <= 0x26CD then
    -- CERES..SESQUIQUADRATE
    -- SOCCER BALL..SQUARED KEY
    -- WHITE DRAUGHTS MAN..BLACK DRAUGHTS KING
    -- SNOWMAN WITHOUT SNOW..DISABLED CAR
    if x >= 0x26B3 then
      return {status="disallowed"}
    end
  elseif x <= 0x26E1 then
    -- PICK..RESTRICTED LEFT ENTRY-2
    if x >= 0x26CF then
      return {status="disallowed"}
    end
  elseif x <= 0x26FF then
    -- PENTAGRAM..INVERTED PENTAGRAM
    -- BLACK CROSS ON SHIELD..WHITE FLAG WITH HORIZONTAL MIDDLE BLACK STRIPE
    if x >= 0x26E4 then
      return {status="disallowed"}
    end
  elseif x <= 0x2727 then
    -- TELEPHONE LOCATION SIGN..ENVELOPE
    -- RAISED FIST..RAISED HAND
    -- VICTORY HAND..WHITE FOUR POINTED STAR
    if x >= 0x2706 then
      return {status="disallowed"}
    end
  elseif x <= 0x274B then
    -- STRESS OUTLINED WHITE STAR..HEAVY EIGHT TEARDROP-SPOKED PROPELLER ASTERISK
    if x >= 0x2729 then
      return {status="disallowed"}
    end
  elseif x <= 0x2755 then
    -- LOWER RIGHT DROP-SHADOWED WHITE SQUARE..UPPER RIGHT SHADOWED WHITE SQUARE
    -- BLACK QUESTION MARK ORNAMENT..WHITE EXCLAMATION MARK ORNAMENT
    if x >= 0x274F then
      return {status="disallowed"}
    end
  elseif x <= 0x27AF then
    -- LIGHT VERTICAL BAR..HEAVY DOUBLE COMMA QUOTATION MARK ORNAMENT
    -- HEAVY LOW SINGLE COMMA QUOTATION MARK ORNAMENT..HEAVY LOW DOUBLE COMMA QUOTATION MARK ORNAMENT
    -- CURVED STEM PARAGRAPH SIGN ORNAMENT..ROTATED FLORAL HEART BULLET
    -- MEDIUM LEFT PARENTHESIS ORNAMENT..MEDIUM RIGHT CURLY BRACKET ORNAMENT
    -- DINGBAT NEGATIVE CIRCLED DIGIT ONE..HEAVY WIDE-HEADED RIGHTWARDS ARROW
    -- HEAVY PLUS SIGN..HEAVY DIVISION SIGN
    -- HEAVY SOUTH EAST ARROW..NOTCHED LOWER RIGHT-SHADOWED WHITE RIGHTWARDS ARROW
    if x >= 0x2758 then
      return {status="disallowed"}
    end
  elseif x <= 0x27BE then
    -- NOTCHED UPPER RIGHT-SHADOWED WHITE RIGHTWARDS ARROW..OPEN-OUTLINED RIGHTWARDS ARROW
    if x >= 0x27B1 then
      return {status="disallowed"}
    end
  elseif x <= 0x27CA then
    -- THREE DIMENSIONAL ANGLE..RIGHT S-SHAPED BAG DELIMITER
    -- OR WITH DOT INSIDE..VERTICAL BAR WITH HORIZONTAL STROKE
    if x >= 0x27C0 then
      return {status="disallowed"}
    end
  elseif x <= 0x2A0B then
    -- SQUARED LOGICAL AND..SQUARED LOGICAL OR
    -- WHITE DIAMOND WITH CENTRED DOT..MATHEMATICAL RIGHT DOUBLE ANGLE BRACKET
    -- MATHEMATICAL LEFT WHITE TORTOISE SHELL BRACKET..MATHEMATICAL RIGHT FLATTENED PARENTHESIS
    -- UPWARDS QUADRUPLE ARROW..LONG RIGHTWARDS SQUIGGLE ARROW
    -- BRAILLE PATTERN BLANK..BRAILLE PATTERN DOTS-12345678
    -- RIGHTWARDS TWO-HEADED ARROW WITH VERTICAL STROKE..SUMMATION WITH INTEGRAL
    if x >= 0x27CE then
      return {status="disallowed"}
    end
  elseif x <= 0x2A73 then
    -- FINITE PART INTEGRAL..EQUALS SIGN ABOVE TILDE OPERATOR
    if x >= 0x2A0D then
      return {status="disallowed"}
    end
  elseif x <= 0x2ADB then
    -- EQUALS SIGN WITH TWO DOTS ABOVE AND TWO DOTS BELOW..TRANSVERSAL INTERSECTION
    if x >= 0x2A77 then
      return {status="disallowed"}
    end
  elseif x <= 0x2B73 then
    -- NONFORKING..N-ARY WHITE VERTICAL BAR
    -- NORTH EAST WHITE ARROW..UP DOWN BLACK ARROW
    -- RIGHTWARDS ARROW WITH TIP DOWNWARDS..SQUARE WITH BOTTOM HALF BLACK
    -- SQUARE WITH UPPER RIGHT DIAGONAL HALF BLACK..DOTTED SQUARE
    -- BLACK LARGE SQUARE..BLACK PENTAGON
    -- WHITE PENTAGON..HORIZONTAL BLACK HEXAGON
    -- BLACK LARGE CIRCLE..RIGHTWARDS ARROW ABOVE REVERSE TILDE OPERATOR
    -- DOWNWARDS TRIANGLE-HEADED ZIGZAG ARROW..SHORT BACKSLANTED SOUTH ARROW
    -- WHITE MEDIUM STAR..WHITE RIGHT-POINTING PENTAGON
    -- HEAVY LARGE CIRCLE..HEAVY CIRCLED SALTIRE
    -- SLANTED NORTH ARROW WITH HOOKED HEAD..DOWNWARDS TRIANGLE-HEADED ARROW TO BAR
    if x >= 0x2ADD then
      return {status="disallowed"}
    end
  elseif x <= 0x2B95 then
    -- NORTH WEST TRIANGLE-HEADED ARROW TO BAR..RIGHTWARDS BLACK ARROW
    if x >= 0x2B76 then
      return {status="disallowed"}
    end
  elseif x <= 0x2BB9 then
    -- THREE-D TOP-LIGHTED LEFTWARDS EQUILATERAL ARROWHEAD..UP ARROWHEAD IN A RECTANGLE BOX
    if x >= 0x2B98 then
      return {status="disallowed"}
    end
  elseif x <= 0x2BC8 then
    -- BALLOT BOX WITH LIGHT X..BLACK MEDIUM RIGHT-POINTING TRIANGLE CENTRED
    if x >= 0x2BBD then
      return {status="disallowed"}
    end
  elseif x <= 0x2BD1 then
    -- TOP HALF BLACK CIRCLE..UNCERTAINTY SIGN
    if x >= 0x2BCA then
      return {status="disallowed"}
    end
  elseif x <= 0x2BEB then
    -- NA   <reserved-2BD3>..<reserved-2BEB>
    if x >= 0x2BD3 then
      return {status="disallowed"}
    end
  elseif x <= 0x2BFF then
    -- NA   <reserved-2BF0>..<reserved-2BFF>
    if x >= 0x2BF0 then
      return {status="disallowed"}
    end
  elseif x <= 0x2C5E then
    -- GLAGOLITIC SMALL LETTER AZU..GLAGOLITIC SMALL LETTER LATINATE MYSLITE
    if x >= 0x2C30 then
      return {status="valid"}
    end
  elseif x <= 0x2C7B then
    -- LATIN SMALL LETTER HALF H..LATIN SMALL LETTER TAILLESS PHI
    -- LATIN SMALL LETTER E WITH NOTCH..LATIN LETTER SMALL CAPITAL TURNED E
    if x >= 0x2C76 then
      return {status="valid"}
    end
  elseif x <= 0x2CEA then
    -- COPTIC SYMBOL MI RO..COPTIC SYMBOL SHIMA SIMA
    if x >= 0x2CE5 then
      return {status="disallowed"}
    end
  elseif x <= 0x2CFF then
    -- COPTIC OLD NUBIAN FULL STOP..COPTIC MORPHOLOGICAL DIVIDER
    if x >= 0x2CF9 then
      return {status="disallowed"}
    end
  elseif x <= 0x2D25 then
    -- GEORGIAN SMALL LETTER AN..GEORGIAN SMALL LETTER HOE
    if x >= 0x2D00 then
      return {status="valid"}
    end
  elseif x <= 0x2D67 then
    -- TIFINAGH LETTER YA..TIFINAGH LETTER YAZZ
    -- TIFINAGH LETTER YE..TIFINAGH LETTER YO
    if x >= 0x2D30 then
      return {status="valid"}
    end
  elseif x <= 0x2D6E then
    -- NA   <reserved-2D68>..<reserved-2D6E>
    if x >= 0x2D68 then
      return {status="disallowed"}
    end
  elseif x <= 0x2D7E then
    -- NA   <reserved-2D71>..<reserved-2D7E>
    if x >= 0x2D71 then
      return {status="disallowed"}
    end
  elseif x <= 0x2D96 then
    -- ETHIOPIC SYLLABLE LOA..ETHIOPIC SYLLABLE GGWE
    if x >= 0x2D80 then
      return {status="valid"}
    end
  elseif x <= 0x2D9F then
    -- NA   <reserved-2D97>..<reserved-2D9F>
    if x >= 0x2D97 then
      return {status="disallowed"}
    end
  elseif x <= 0x2DA6 then
    -- ETHIOPIC SYLLABLE SSA..ETHIOPIC SYLLABLE SSO
    if x >= 0x2DA0 then
      return {status="valid"}
    end
  elseif x <= 0x2DAE then
    -- ETHIOPIC SYLLABLE CCA..ETHIOPIC SYLLABLE CCO
    if x >= 0x2DA8 then
      return {status="valid"}
    end
  elseif x <= 0x2DB6 then
    -- ETHIOPIC SYLLABLE ZZA..ETHIOPIC SYLLABLE ZZO
    if x >= 0x2DB0 then
      return {status="valid"}
    end
  elseif x <= 0x2DBE then
    -- ETHIOPIC SYLLABLE CCHA..ETHIOPIC SYLLABLE CCHO
    if x >= 0x2DB8 then
      return {status="valid"}
    end
  elseif x <= 0x2DC6 then
    -- ETHIOPIC SYLLABLE QYA..ETHIOPIC SYLLABLE QYO
    if x >= 0x2DC0 then
      return {status="valid"}
    end
  elseif x <= 0x2DCE then
    -- ETHIOPIC SYLLABLE KYA..ETHIOPIC SYLLABLE KYO
    if x >= 0x2DC8 then
      return {status="valid"}
    end
  elseif x <= 0x2DD6 then
    -- ETHIOPIC SYLLABLE XYA..ETHIOPIC SYLLABLE XYO
    if x >= 0x2DD0 then
      return {status="valid"}
    end
  elseif x <= 0x2DDE then
    -- ETHIOPIC SYLLABLE GYA..ETHIOPIC SYLLABLE GYO
    if x >= 0x2DD8 then
      return {status="valid"}
    end
  elseif x <= 0x2DFF then
    -- COMBINING CYRILLIC LETTER BE..COMBINING CYRILLIC LETTER IOTIFIED BIG YUS
    if x >= 0x2DE0 then
      return {status="valid"}
    end
  elseif x <= 0x2E2E then
    -- RIGHT ANGLE SUBSTITUTION MARKER..DOUBLE OBLIQUE HYPHEN
    -- INVERTED INTERROBANG..TILDE WITH RING ABOVE
    -- LEFT LOW PARAPHRASE BRACKET..RIGHT LOW PARAPHRASE BRACKET
    -- TILDE WITH DOT ABOVE..REVERSED QUESTION MARK
    if x >= 0x2E00 then
      return {status="disallowed"}
    end
  elseif x <= 0x2E49 then
    -- TURNED COMMA..THREE-EM DASH
    -- STENOGRAPHIC FULL STOP..DOUBLE LOW-REVERSED-9 QUOTATION MARK
    -- DASH WITH LEFT UPTURN..DOUBLE SUSPENSION MARK
    -- INVERTED LOW KAVYKA..DOUBLE STACKED COMMA
    if x >= 0x2E32 then
      return {status="disallowed"}
    end
  elseif x <= 0x2E7F then
    -- NA   <reserved-2E4A>..<reserved-2E7F>
    if x >= 0x2E4A then
      return {status="disallowed"}
    end
  elseif x <= 0x2E99 then
    -- CJK RADICAL REPEAT..CJK RADICAL RAP
    if x >= 0x2E80 then
      return {status="disallowed"}
    end
  elseif x <= 0x2EF2 then
    -- CJK RADICAL CIVILIAN..CJK RADICAL J-SIMPLIFIED TURTLE
    if x >= 0x2EA0 then
      return {status="disallowed"}
    end
  elseif x <= 0x2EFF then
    -- NA   <reserved-2EF4>..<reserved-2EFF>
    if x >= 0x2EF4 then
      return {status="disallowed"}
    end
  elseif x <= 0x2FFF then
    -- NA   <reserved-2FD6>..<reserved-2FEF>
    -- IDEOGRAPHIC DESCRIPTION CHARACTER LEFT TO RIGHT..IDEOGRAPHIC DESCRIPTION CHARACTER OVERLAID
    -- NA   <reserved-2FFC>..<reserved-2FFF>
    if x >= 0x2FD6 then
      return {status="disallowed"}
    end
  elseif x <= 0x3029 then
    -- LEFT ANGLE BRACKET..HANGZHOU NUMERAL NINE
    if x >= 0x3008 then
      return {status="disallowed"}
    end
  elseif x <= 0x3035 then
    -- HANGUL SINGLE DOT TONE MARK..VERTICAL KANA REPEAT MARK LOWER HALF
    if x >= 0x302E then
      return {status="disallowed"}
    end
  elseif x <= 0x3096 then
    -- HIRAGANA LETTER SMALL A..HIRAGANA LETTER VU
    -- HIRAGANA LETTER SMALL KA..HIRAGANA LETTER SMALL KE
    if x >= 0x3041 then
      return {status="valid"}
    end
  elseif x <= 0x30FE then
    -- KATAKANA LETTER SMALL A..KATAKANA VOICED ITERATION MARK
    if x >= 0x30A1 then
      return {status="valid"}
    end
  elseif x <= 0x312C then
    -- BOPOMOFO LETTER B..BOPOMOFO LETTER GN
    if x >= 0x3105 then
      return {status="valid"}
    end
  elseif x <= 0x31BA then
    -- BOPOMOFO LETTER BU..BOPOMOFO FINAL LETTER H
    -- BOPOMOFO LETTER GH..BOPOMOFO LETTER ZY
    if x >= 0x31A0 then
      return {status="valid"}
    end
  elseif x <= 0x31E3 then
    -- CJK STROKE T..CJK STROKE N
    -- CJK STROKE H..CJK STROKE Q
    if x >= 0x31C0 then
      return {status="disallowed"}
    end
  elseif x <= 0x31EF then
    -- NA   <reserved-31E4>..<reserved-31EF>
    if x >= 0x31E4 then
      return {status="disallowed"}
    end
  elseif x <= 0x31FF then
    -- KATAKANA LETTER SMALL KU..KATAKANA LETTER SMALL RO
    if x >= 0x31F0 then
      return {status="valid"}
    end
  elseif x <= 0x324F then
    -- CIRCLED NUMBER TEN ON BLACK SQUARE..CIRCLED NUMBER EIGHTY ON BLACK SQUARE
    if x >= 0x3248 then
      return {status="disallowed"}
    end
  elseif x <= 0x4DB5 then
    -- CJK UNIFIED IDEOGRAPH-3400..CJK UNIFIED IDEOGRAPH-4DB5
    if x >= 0x3400 then
      return {status="valid"}
    end
  elseif x <= 0x4DBF then
    -- NA   <reserved-4DB6>..<reserved-4DBF>
    if x >= 0x4DB6 then
      return {status="disallowed"}
    end
  elseif x <= 0x4DFF then
    -- HEXAGRAM FOR THE CREATIVE HEAVEN..HEXAGRAM FOR BEFORE COMPLETION
    if x >= 0x4DC0 then
      return {status="disallowed"}
    end
  elseif x <= 0x9FCB then
    -- CJK UNIFIED IDEOGRAPH-4E00..CJK UNIFIED IDEOGRAPH-9FA5
    -- CJK UNIFIED IDEOGRAPH-9FA6..CJK UNIFIED IDEOGRAPH-9FBB
    -- CJK UNIFIED IDEOGRAPH-9FBC..CJK UNIFIED IDEOGRAPH-9FC3
    -- CJK UNIFIED IDEOGRAPH-9FC4..CJK UNIFIED IDEOGRAPH-9FCB
    if x >= 0x4E00 then
      return {status="valid"}
    end
  elseif x <= 0x9FEA then
    -- CJK UNIFIED IDEOGRAPH-9FCD..CJK UNIFIED IDEOGRAPH-9FD5
    -- CJK UNIFIED IDEOGRAPH-9FD6..CJK UNIFIED IDEOGRAPH-9FEA
    if x >= 0x9FCD then
      return {status="valid"}
    end
  elseif x <= 0x9FFF then
    -- NA   <reserved-9FEB>..<reserved-9FFF>
    if x >= 0x9FEB then
      return {status="disallowed"}
    end
  elseif x <= 0xA48C then
    -- YI SYLLABLE IT..YI SYLLABLE YYR
    if x >= 0xA000 then
      return {status="valid"}
    end
  elseif x <= 0xA4B3 then
    -- YI RADICAL QOT..YI RADICAL GA
    -- YI RADICAL ZUP..YI RADICAL CYT
    -- YI RADICAL DDUR..YI RADICAL JO
    if x >= 0xA490 then
      return {status="disallowed"}
    end
  elseif x <= 0xA4C0 then
    -- YI RADICAL JJY..YI RADICAL SHAT
    if x >= 0xA4B5 then
      return {status="disallowed"}
    end
  elseif x <= 0xA4CF then
    -- NA   <reserved-A4C7>..<reserved-A4CF>
    if x >= 0xA4C7 then
      return {status="disallowed"}
    end
  elseif x <= 0xA4FD then
    -- LISU LETTER BA..LISU LETTER TONE MYA JEU
    if x >= 0xA4D0 then
      return {status="valid"}
    end
  elseif x <= 0xA60C then
    -- VAI SYLLABLE EE..VAI SYLLABLE LENGTHENER
    if x >= 0xA500 then
      return {status="valid"}
    end
  elseif x <= 0xA62B then
    -- VAI SYLLABLE NDOLE FA..VAI SYLLABLE NDOLE DO
    if x >= 0xA610 then
      return {status="valid"}
    end
  elseif x <= 0xA63F then
    -- NA   <reserved-A62C>..<reserved-A63F>
    if x >= 0xA62C then
      return {status="disallowed"}
    end
  elseif x <= 0xA67D then
    -- COMBINING CYRILLIC LETTER UKRAINIAN IE..COMBINING CYRILLIC LETTER OMEGA
    -- COMBINING CYRILLIC KAVYKA..COMBINING CYRILLIC PAYEROK
    if x >= 0xA674 then
      return {status="valid"}
    end
  elseif x <= 0xA6E5 then
    -- BAMUM LETTER A..BAMUM LETTER KI
    if x >= 0xA6A0 then
      return {status="valid"}
    end
  elseif x <= 0xA6EF then
    -- BAMUM LETTER MO..BAMUM LETTER KOGHOM
    if x >= 0xA6E6 then
      return {status="disallowed"}
    end
  elseif x <= 0xA6F7 then
    -- BAMUM NJAEMLI..BAMUM QUESTION MARK
    if x >= 0xA6F2 then
      return {status="disallowed"}
    end
  elseif x <= 0xA6FF then
    -- NA   <reserved-A6F8>..<reserved-A6FF>
    if x >= 0xA6F8 then
      return {status="disallowed"}
    end
  elseif x <= 0xA716 then
    -- MODIFIER LETTER CHINESE TONE YIN PING..MODIFIER LETTER EXTRA-LOW LEFT-STEM TONE BAR
    if x >= 0xA700 then
      return {status="disallowed"}
    end
  elseif x <= 0xA71F then
    -- MODIFIER LETTER DOT VERTICAL BAR..MODIFIER LETTER LOWER RIGHT CORNER ANGLE
    -- MODIFIER LETTER RAISED UP ARROW..MODIFIER LETTER LOW INVERTED EXCLAMATION MARK
    if x >= 0xA717 then
      return {status="valid"}
    end
  elseif x <= 0xA778 then
    -- LATIN SMALL LETTER DUM..LATIN SMALL LETTER UM
    if x >= 0xA771 then
      return {status="valid"}
    end
  elseif x <= 0xA7F6 then
    -- NA   <reserved-A7B8>..<reserved-A7F6>
    if x >= 0xA7B8 then
      return {status="disallowed"}
    end
  elseif x <= 0xA827 then
    -- LATIN EPIGRAPHIC LETTER REVERSED F..LATIN EPIGRAPHIC LETTER ARCHAIC M
    -- SYLOTI NAGRI LETTER A..SYLOTI NAGRI VOWEL SIGN OO
    if x >= 0xA7FB then
      return {status="valid"}
    end
  elseif x <= 0xA839 then
    -- NORTH INDIC FRACTION ONE QUARTER..NORTH INDIC QUANTITY MARK
    if x >= 0xA830 then
      return {status="disallowed"}
    end
  elseif x <= 0xA83F then
    -- NA   <reserved-A83A>..<reserved-A83F>
    if x >= 0xA83A then
      return {status="disallowed"}
    end
  elseif x <= 0xA873 then
    -- PHAGS-PA LETTER KA..PHAGS-PA LETTER CANDRABINDU
    if x >= 0xA840 then
      return {status="valid"}
    end
  elseif x <= 0xA87F then
    -- NA   <reserved-A878>..<reserved-A87F>
    if x >= 0xA878 then
      return {status="disallowed"}
    end
  elseif x <= 0xA8C4 then
    -- SAURASHTRA SIGN ANUSVARA..SAURASHTRA SIGN VIRAMA
    if x >= 0xA880 then
      return {status="valid"}
    end
  elseif x <= 0xA8CD then
    -- NA   <reserved-A8C6>..<reserved-A8CD>
    if x >= 0xA8C6 then
      return {status="disallowed"}
    end
  elseif x <= 0xA8D9 then
    -- SAURASHTRA DIGIT ZERO..SAURASHTRA DIGIT NINE
    if x >= 0xA8D0 then
      return {status="valid"}
    end
  elseif x <= 0xA8DF then
    -- NA   <reserved-A8DA>..<reserved-A8DF>
    if x >= 0xA8DA then
      return {status="disallowed"}
    end
  elseif x <= 0xA8F7 then
    -- COMBINING DEVANAGARI DIGIT ZERO..DEVANAGARI SIGN CANDRABINDU AVAGRAHA
    if x >= 0xA8E0 then
      return {status="valid"}
    end
  elseif x <= 0xA92D then
    -- KAYAH LI DIGIT ZERO..KAYAH LI TONE CALYA PLOPHU
    if x >= 0xA900 then
      return {status="valid"}
    end
  elseif x <= 0xA953 then
    -- REJANG LETTER KA..REJANG VIRAMA
    if x >= 0xA930 then
      return {status="valid"}
    end
  elseif x <= 0xA95E then
    -- NA   <reserved-A954>..<reserved-A95E>
    if x >= 0xA954 then
      return {status="disallowed"}
    end
  elseif x <= 0xA97C then
    -- HANGUL CHOSEONG TIKEUT-MIEUM..HANGUL CHOSEONG SSANGYEORINHIEUH
    if x >= 0xA960 then
      return {status="disallowed"}
    end
  elseif x <= 0xA9C0 then
    -- JAVANESE SIGN PANYANGGA..JAVANESE PANGKON
    if x >= 0xA980 then
      return {status="valid"}
    end
  elseif x <= 0xA9CD then
    -- JAVANESE LEFT RERENGGAN..JAVANESE TURNED PADA PISELEH
    if x >= 0xA9C1 then
      return {status="disallowed"}
    end
  elseif x <= 0xA9D9 then
    -- JAVANESE PANGRANGKEP..JAVANESE DIGIT NINE
    if x >= 0xA9CF then
      return {status="valid"}
    end
  elseif x <= 0xA9FE then
    -- MYANMAR LETTER SHAN GHA..MYANMAR LETTER TAI LAING BHA
    if x >= 0xA9E0 then
      return {status="valid"}
    end
  elseif x <= 0xAA36 then
    -- CHAM LETTER A..CHAM CONSONANT SIGN WA
    if x >= 0xAA00 then
      return {status="valid"}
    end
  elseif x <= 0xAA3F then
    -- NA   <reserved-AA37>..<reserved-AA3F>
    if x >= 0xAA37 then
      return {status="disallowed"}
    end
  elseif x <= 0xAA4D then
    -- CHAM LETTER FINAL K..CHAM CONSONANT SIGN FINAL H
    if x >= 0xAA40 then
      return {status="valid"}
    end
  elseif x <= 0xAA59 then
    -- CHAM DIGIT ZERO..CHAM DIGIT NINE
    if x >= 0xAA50 then
      return {status="valid"}
    end
  elseif x <= 0xAA76 then
    -- MYANMAR LETTER KHAMTI GA..MYANMAR LOGOGRAM KHAMTI HM
    if x >= 0xAA60 then
      return {status="valid"}
    end
  elseif x <= 0xAAC2 then
    -- MYANMAR LETTER AITON RA..MYANMAR SIGN PAO KAREN TONE
    -- MYANMAR SIGN TAI LAING TONE-2..MYANMAR LETTER SHWE PALAUNG SHA
    -- TAI VIET LETTER LOW KO..TAI VIET TONE MAI SONG
    if x >= 0xAA7A then
      return {status="valid"}
    end
  elseif x <= 0xAADA then
    -- NA   <reserved-AAC3>..<reserved-AADA>
    if x >= 0xAAC3 then
      return {status="disallowed"}
    end
  elseif x <= 0xAAEF then
    -- MEETEI MAYEK LETTER E..MEETEI MAYEK VOWEL SIGN AAU
    if x >= 0xAAE0 then
      return {status="valid"}
    end
  elseif x <= 0xAB00 then
    -- NA   <reserved-AAF7>..<reserved-AB00>
    if x >= 0xAAF7 then
      return {status="disallowed"}
    end
  elseif x <= 0xAB06 then
    -- ETHIOPIC SYLLABLE TTHU..ETHIOPIC SYLLABLE TTHO
    if x >= 0xAB01 then
      return {status="valid"}
    end
  elseif x <= 0xAB0E then
    -- ETHIOPIC SYLLABLE DDHU..ETHIOPIC SYLLABLE DDHO
    if x >= 0xAB09 then
      return {status="valid"}
    end
  elseif x <= 0xAB16 then
    -- ETHIOPIC SYLLABLE DZU..ETHIOPIC SYLLABLE DZO
    if x >= 0xAB11 then
      return {status="valid"}
    end
  elseif x <= 0xAB1F then
    -- NA   <reserved-AB17>..<reserved-AB1F>
    if x >= 0xAB17 then
      return {status="disallowed"}
    end
  elseif x <= 0xAB26 then
    -- ETHIOPIC SYLLABLE CCHHA..ETHIOPIC SYLLABLE CCHHO
    if x >= 0xAB20 then
      return {status="valid"}
    end
  elseif x <= 0xAB2E then
    -- ETHIOPIC SYLLABLE BBA..ETHIOPIC SYLLABLE BBO
    if x >= 0xAB28 then
      return {status="valid"}
    end
  elseif x <= 0xAB5A then
    -- LATIN SMALL LETTER BARRED ALPHA..LATIN SMALL LETTER Y WITH SHORT RIGHT LEG
    if x >= 0xAB30 then
      return {status="valid"}
    end
  elseif x <= 0xAB65 then
    -- LATIN SMALL LETTER SAKHA YAT..LATIN SMALL LETTER UO
    -- LATIN SMALL LETTER INVERTED ALPHA..GREEK LETTER SMALL CAPITAL OMEGA
    if x >= 0xAB60 then
      return {status="valid"}
    end
  elseif x <= 0xAB6F then
    -- NA   <reserved-AB66>..<reserved-AB6F>
    if x >= 0xAB66 then
      return {status="disallowed"}
    end
  elseif x <= 0xABEA then
    -- MEETEI MAYEK LETTER KOK..MEETEI MAYEK VOWEL SIGN NUNG
    if x >= 0xABC0 then
      return {status="valid"}
    end
  elseif x <= 0xABF9 then
    -- MEETEI MAYEK DIGIT ZERO..MEETEI MAYEK DIGIT NINE
    if x >= 0xABF0 then
      return {status="valid"}
    end
  elseif x <= 0xABFF then
    -- NA   <reserved-ABFA>..<reserved-ABFF>
    if x >= 0xABFA then
      return {status="disallowed"}
    end
  elseif x <= 0xD7A3 then
    -- HANGUL SYLLABLE GA..HANGUL SYLLABLE HIH
    if x >= 0xAC00 then
      return {status="valid"}
    end
  elseif x <= 0xD7AF then
    -- NA   <reserved-D7A4>..<reserved-D7AF>
    if x >= 0xD7A4 then
      return {status="disallowed"}
    end
  elseif x <= 0xD7C6 then
    -- HANGUL JUNGSEONG O-YEO..HANGUL JUNGSEONG ARAEA-E
    if x >= 0xD7B0 then
      return {status="disallowed"}
    end
  elseif x <= 0xD7FB then
    -- HANGUL JONGSEONG NIEUN-RIEUL..HANGUL JONGSEONG PHIEUPH-THIEUTH
    if x >= 0xD7CB then
      return {status="disallowed"}
    end
  elseif x <= 0xF8FF then
    -- NA   <reserved-D7FC>..<reserved-D7FF>
    -- <surrogate-D800>..<surrogate-DFFF>
    -- <private-use-E000>..<private-use-F8FF>
    if x >= 0xD7FC then
      return {status="disallowed"}
    end
  elseif x <= 0xFAFF then
    -- NA   <reserved-FADA>..<reserved-FAFF>
    if x >= 0xFADA then
      return {status="disallowed"}
    end
  elseif x <= 0xFB12 then
    -- NA   <reserved-FB07>..<reserved-FB12>
    if x >= 0xFB07 then
      return {status="disallowed"}
    end
  elseif x <= 0xFBC1 then
    -- ARABIC SYMBOL DOT ABOVE..ARABIC SYMBOL SMALL TAH BELOW
    if x >= 0xFBB2 then
      return {status="disallowed"}
    end
  elseif x <= 0xFBD2 then
    -- NA   <reserved-FBC2>..<reserved-FBD2>
    if x >= 0xFBC2 then
      return {status="disallowed"}
    end
  elseif x <= 0xFD4F then
    -- NA   <reserved-FD40>..<reserved-FD4F>
    if x >= 0xFD40 then
      return {status="disallowed"}
    end
  elseif x <= 0xFDEF then
    -- NA   <reserved-FDC8>..<reserved-FDCF>
    -- <noncharacter-FDD0>..<noncharacter-FDEF>
    if x >= 0xFDC8 then
      return {status="disallowed"}
    end
  elseif x <= 0xFE0F then
    -- VARIATION SELECTOR-1..VARIATION SELECTOR-16
    if x >= 0xFE00 then
      return {status='ignored'}
    end
  elseif x <= 0xFE1F then
    -- NA   <reserved-FE1A>..<reserved-FE1F>
    if x >= 0xFE1A then
      return {status="disallowed"}
    end
  elseif x <= 0xFE2F then
    -- COMBINING LIGATURE LEFT HALF..COMBINING DOUBLE TILDE RIGHT HALF
    -- COMBINING MACRON LEFT HALF..COMBINING CONJOINING MACRON
    -- COMBINING LIGATURE LEFT HALF BELOW..COMBINING CONJOINING MACRON BELOW
    -- COMBINING CYRILLIC TITLO LEFT HALF..COMBINING CYRILLIC TITLO RIGHT HALF
    if x >= 0xFE20 then
      return {status="valid"}
    end
  elseif x <= 0xFFFB then
    -- NA   <reserved-FFEF>..<reserved-FFF8>
    -- INTERLINEAR ANNOTATION ANCHOR..INTERLINEAR ANNOTATION TERMINATOR
    if x >= 0xFFEF then
      return {status="disallowed"}
    end
  elseif x <= 0x1000B then
    -- LINEAR B SYLLABLE B008 A..LINEAR B SYLLABLE B046 JE
    if x >= 0x10000 then
      return {status="valid"}
    end
  elseif x <= 0x10026 then
    -- LINEAR B SYLLABLE B036 JO..LINEAR B SYLLABLE B032 QO
    if x >= 0x1000D then
      return {status="valid"}
    end
  elseif x <= 0x1003A then
    -- LINEAR B SYLLABLE B060 RA..LINEAR B SYLLABLE B042 WO
    if x >= 0x10028 then
      return {status="valid"}
    end
  elseif x <= 0x1004D then
    -- LINEAR B SYLLABLE B020 ZO..LINEAR B SYLLABLE B091 TWO
    if x >= 0x1003F then
      return {status="valid"}
    end
  elseif x <= 0x1005D then
    -- LINEAR B SYMBOL B018..LINEAR B SYMBOL B089
    if x >= 0x10050 then
      return {status="valid"}
    end
  elseif x <= 0x1007F then
    -- NA   <reserved-1005E>..<reserved-1007F>
    if x >= 0x1005E then
      return {status="disallowed"}
    end
  elseif x <= 0x100FA then
    -- LINEAR B IDEOGRAM B100 MAN..LINEAR B IDEOGRAM VESSEL B305
    if x >= 0x10080 then
      return {status="valid"}
    end
  elseif x <= 0x10133 then
    -- AEGEAN NUMBER ONE..AEGEAN NUMBER NINETY THOUSAND
    if x >= 0x10107 then
      return {status="disallowed"}
    end
  elseif x <= 0x1018E then
    -- AEGEAN WEIGHT BASE UNIT..AEGEAN MEASURE THIRD SUBUNIT
    -- GREEK ACROPHONIC ATTIC ONE QUARTER..GREEK ZERO SIGN
    -- GREEK ONE QUARTER SIGN..GREEK SINUSOID SIGN
    -- GREEK INDICTION SIGN..NOMISMA SIGN
    if x >= 0x10137 then
      return {status="disallowed"}
    end
  elseif x <= 0x1019B then
    -- ROMAN SEXTANS SIGN..ROMAN CENTURIAL SIGN
    if x >= 0x10190 then
      return {status="disallowed"}
    end
  elseif x <= 0x101CF then
    -- NA   <reserved-101A1>..<reserved-101CF>
    if x >= 0x101A1 then
      return {status="disallowed"}
    end
  elseif x <= 0x101FC then
    -- PHAISTOS DISC SIGN PEDESTRIAN..PHAISTOS DISC SIGN WAVY BAND
    if x >= 0x101D0 then
      return {status="disallowed"}
    end
  elseif x <= 0x1027F then
    -- NA   <reserved-101FE>..<reserved-1027F>
    if x >= 0x101FE then
      return {status="disallowed"}
    end
  elseif x <= 0x1029C then
    -- LYCIAN LETTER A..LYCIAN LETTER X
    if x >= 0x10280 then
      return {status="valid"}
    end
  elseif x <= 0x102D0 then
    -- CARIAN LETTER A..CARIAN LETTER UUU3
    if x >= 0x102A0 then
      return {status="valid"}
    end
  elseif x <= 0x102DF then
    -- NA   <reserved-102D1>..<reserved-102DF>
    if x >= 0x102D1 then
      return {status="disallowed"}
    end
  elseif x <= 0x102FB then
    -- COPTIC EPACT DIGIT ONE..COPTIC EPACT NUMBER NINE HUNDRED
    if x >= 0x102E1 then
      return {status="disallowed"}
    end
  elseif x <= 0x1031E then
    -- OLD ITALIC LETTER A..OLD ITALIC LETTER UU
    if x >= 0x10300 then
      return {status="valid"}
    end
  elseif x <= 0x1032C then
    -- NA   <reserved-10324>..<reserved-1032C>
    if x >= 0x10324 then
      return {status="disallowed"}
    end
  elseif x <= 0x10340 then
    -- OLD ITALIC LETTER YE..OLD ITALIC LETTER SOUTHERN TSE
    -- GOTHIC LETTER AHSA..GOTHIC LETTER PAIRTHRA
    if x >= 0x1032D then
      return {status="valid"}
    end
  elseif x <= 0x10349 then
    -- GOTHIC LETTER RAIDA..GOTHIC LETTER OTHAL
    if x >= 0x10342 then
      return {status="valid"}
    end
  elseif x <= 0x1037A then
    -- OLD PERMIC LETTER AN..COMBINING OLD PERMIC LETTER SII
    if x >= 0x10350 then
      return {status="valid"}
    end
  elseif x <= 0x1039D then
    -- UGARITIC LETTER ALPA..UGARITIC LETTER SSU
    if x >= 0x10380 then
      return {status="valid"}
    end
  elseif x <= 0x103C3 then
    -- OLD PERSIAN SIGN A..OLD PERSIAN SIGN HA
    if x >= 0x103A0 then
      return {status="valid"}
    end
  elseif x <= 0x103CF then
    -- OLD PERSIAN SIGN AURAMAZDAA..OLD PERSIAN SIGN BUUMISH
    if x >= 0x103C8 then
      return {status="valid"}
    end
  elseif x <= 0x103D5 then
    -- OLD PERSIAN WORD DIVIDER..OLD PERSIAN NUMBER HUNDRED
    if x >= 0x103D0 then
      return {status="disallowed"}
    end
  elseif x <= 0x103FF then
    -- NA   <reserved-103D6>..<reserved-103FF>
    if x >= 0x103D6 then
      return {status="disallowed"}
    end
  elseif x <= 0x1049D then
    -- DESERET SMALL LETTER LONG I..DESERET SMALL LETTER ENG
    -- DESERET SMALL LETTER OI..OSMANYA LETTER OO
    if x >= 0x10428 then
      return {status="valid"}
    end
  elseif x <= 0x104A9 then
    -- OSMANYA DIGIT ZERO..OSMANYA DIGIT NINE
    if x >= 0x104A0 then
      return {status="valid"}
    end
  elseif x <= 0x104AF then
    -- NA   <reserved-104AA>..<reserved-104AF>
    if x >= 0x104AA then
      return {status="disallowed"}
    end
  elseif x <= 0x104FB then
    -- OSAGE SMALL LETTER A..OSAGE SMALL LETTER ZHA
    if x >= 0x104D8 then
      return {status="valid"}
    end
  elseif x <= 0x10527 then
    -- ELBASAN LETTER A..ELBASAN LETTER KHE
    if x >= 0x10500 then
      return {status="valid"}
    end
  elseif x <= 0x1052F then
    -- NA   <reserved-10528>..<reserved-1052F>
    if x >= 0x10528 then
      return {status="disallowed"}
    end
  elseif x <= 0x10563 then
    -- CAUCASIAN ALBANIAN LETTER ALT..CAUCASIAN ALBANIAN LETTER KIW
    if x >= 0x10530 then
      return {status="valid"}
    end
  elseif x <= 0x1056E then
    -- NA   <reserved-10564>..<reserved-1056E>
    if x >= 0x10564 then
      return {status="disallowed"}
    end
  elseif x <= 0x105FF then
    -- NA   <reserved-10570>..<reserved-105FF>
    if x >= 0x10570 then
      return {status="disallowed"}
    end
  elseif x <= 0x10736 then
    -- LINEAR A SIGN AB001..LINEAR A SIGN A664
    if x >= 0x10600 then
      return {status="valid"}
    end
  elseif x <= 0x1073F then
    -- NA   <reserved-10737>..<reserved-1073F>
    if x >= 0x10737 then
      return {status="disallowed"}
    end
  elseif x <= 0x10755 then
    -- LINEAR A SIGN A701 A..LINEAR A SIGN A732 JE
    if x >= 0x10740 then
      return {status="valid"}
    end
  elseif x <= 0x1075F then
    -- NA   <reserved-10756>..<reserved-1075F>
    if x >= 0x10756 then
      return {status="disallowed"}
    end
  elseif x <= 0x10767 then
    -- LINEAR A SIGN A800..LINEAR A SIGN A807
    if x >= 0x10760 then
      return {status="valid"}
    end
  elseif x <= 0x107FF then
    -- NA   <reserved-10768>..<reserved-107FF>
    if x >= 0x10768 then
      return {status="disallowed"}
    end
  elseif x <= 0x10805 then
    -- CYPRIOT SYLLABLE A..CYPRIOT SYLLABLE JA
    if x >= 0x10800 then
      return {status="valid"}
    end
  elseif x <= 0x10835 then
    -- CYPRIOT SYLLABLE KA..CYPRIOT SYLLABLE WO
    if x >= 0x1080A then
      return {status="valid"}
    end
  elseif x <= 0x10855 then
    -- IMPERIAL ARAMAIC LETTER ALEPH..IMPERIAL ARAMAIC LETTER TAW
    if x >= 0x10840 then
      return {status="valid"}
    end
  elseif x <= 0x1085F then
    -- IMPERIAL ARAMAIC SECTION SIGN..IMPERIAL ARAMAIC NUMBER TEN THOUSAND
    if x >= 0x10857 then
      return {status="disallowed"}
    end
  elseif x <= 0x10876 then
    -- PALMYRENE LETTER ALEPH..PALMYRENE LETTER TAW
    if x >= 0x10860 then
      return {status="valid"}
    end
  elseif x <= 0x1087F then
    -- PALMYRENE LEFT-POINTING FLEURON..PALMYRENE NUMBER TWENTY
    if x >= 0x10877 then
      return {status="disallowed"}
    end
  elseif x <= 0x1089E then
    -- NABATAEAN LETTER FINAL ALEPH..NABATAEAN LETTER TAW
    if x >= 0x10880 then
      return {status="valid"}
    end
  elseif x <= 0x108A6 then
    -- NA   <reserved-1089F>..<reserved-108A6>
    if x >= 0x1089F then
      return {status="disallowed"}
    end
  elseif x <= 0x108AF then
    -- NABATAEAN NUMBER ONE..NABATAEAN NUMBER ONE HUNDRED
    if x >= 0x108A7 then
      return {status="disallowed"}
    end
  elseif x <= 0x108DF then
    -- NA   <reserved-108B0>..<reserved-108DF>
    if x >= 0x108B0 then
      return {status="disallowed"}
    end
  elseif x <= 0x108F2 then
    -- HATRAN LETTER ALEPH..HATRAN LETTER QOPH
    if x >= 0x108E0 then
      return {status="valid"}
    end
  elseif x <= 0x10915 then
    -- PHOENICIAN LETTER ALF..PHOENICIAN LETTER TAU
    if x >= 0x10900 then
      return {status="valid"}
    end
  elseif x <= 0x1091B then
    -- PHOENICIAN NUMBER ONE..PHOENICIAN NUMBER ONE HUNDRED
    -- PHOENICIAN NUMBER TWO..PHOENICIAN NUMBER THREE
    if x >= 0x10916 then
      return {status="disallowed"}
    end
  elseif x <= 0x10939 then
    -- LYDIAN LETTER A..LYDIAN LETTER C
    if x >= 0x10920 then
      return {status="valid"}
    end
  elseif x <= 0x1097F then
    -- NA   <reserved-10940>..<reserved-1097F>
    if x >= 0x10940 then
      return {status="disallowed"}
    end
  elseif x <= 0x109B7 then
    -- MEROITIC HIEROGLYPHIC LETTER A..MEROITIC CURSIVE LETTER DA
    if x >= 0x10980 then
      return {status="valid"}
    end
  elseif x <= 0x109CF then
    -- MEROITIC CURSIVE NUMBER ONE..MEROITIC CURSIVE NUMBER SEVENTY
    if x >= 0x109C0 then
      return {status="disallowed"}
    end
  elseif x <= 0x109FF then
    -- MEROITIC CURSIVE NUMBER ONE HUNDRED..MEROITIC CURSIVE FRACTION TEN TWELFTHS
    if x >= 0x109D2 then
      return {status="disallowed"}
    end
  elseif x <= 0x10A13 then
    -- KHAROSHTHI VOWEL LENGTH MARK..KHAROSHTHI LETTER GHA
    if x >= 0x10A0C then
      return {status="valid"}
    end
  elseif x <= 0x10A33 then
    -- KHAROSHTHI LETTER NYA..KHAROSHTHI LETTER TTTHA
    if x >= 0x10A19 then
      return {status="valid"}
    end
  elseif x <= 0x10A47 then
    -- KHAROSHTHI DIGIT ONE..KHAROSHTHI NUMBER ONE THOUSAND
    if x >= 0x10A40 then
      return {status="disallowed"}
    end
  elseif x <= 0x10A4F then
    -- NA   <reserved-10A48>..<reserved-10A4F>
    if x >= 0x10A48 then
      return {status="disallowed"}
    end
  elseif x <= 0x10A58 then
    -- KHAROSHTHI PUNCTUATION DOT..KHAROSHTHI PUNCTUATION LINES
    if x >= 0x10A50 then
      return {status="disallowed"}
    end
  elseif x <= 0x10A5F then
    -- NA   <reserved-10A59>..<reserved-10A5F>
    if x >= 0x10A59 then
      return {status="disallowed"}
    end
  elseif x <= 0x10A7C then
    -- OLD SOUTH ARABIAN LETTER HE..OLD SOUTH ARABIAN LETTER THETH
    if x >= 0x10A60 then
      return {status="valid"}
    end
  elseif x <= 0x10A9C then
    -- OLD NORTH ARABIAN LETTER HEH..OLD NORTH ARABIAN LETTER ZAH
    if x >= 0x10A80 then
      return {status="valid"}
    end
  elseif x <= 0x10ABF then
    -- NA   <reserved-10AA0>..<reserved-10ABF>
    if x >= 0x10AA0 then
      return {status="disallowed"}
    end
  elseif x <= 0x10AC7 then
    -- MANICHAEAN LETTER ALEPH..MANICHAEAN LETTER WAW
    if x >= 0x10AC0 then
      return {status="valid"}
    end
  elseif x <= 0x10AE6 then
    -- MANICHAEAN LETTER ZAYIN..MANICHAEAN ABBREVIATION MARK BELOW
    if x >= 0x10AC9 then
      return {status="valid"}
    end
  elseif x <= 0x10AF6 then
    -- MANICHAEAN NUMBER ONE..MANICHAEAN PUNCTUATION LINE FILLER
    if x >= 0x10AEB then
      return {status="disallowed"}
    end
  elseif x <= 0x10AFF then
    -- NA   <reserved-10AF7>..<reserved-10AFF>
    if x >= 0x10AF7 then
      return {status="disallowed"}
    end
  elseif x <= 0x10B35 then
    -- AVESTAN LETTER A..AVESTAN LETTER HE
    if x >= 0x10B00 then
      return {status="valid"}
    end
  elseif x <= 0x10B3F then
    -- AVESTAN ABBREVIATION MARK..LARGE ONE RING OVER TWO RINGS PUNCTUATION
    if x >= 0x10B39 then
      return {status="disallowed"}
    end
  elseif x <= 0x10B55 then
    -- INSCRIPTIONAL PARTHIAN LETTER ALEPH..INSCRIPTIONAL PARTHIAN LETTER TAW
    if x >= 0x10B40 then
      return {status="valid"}
    end
  elseif x <= 0x10B5F then
    -- INSCRIPTIONAL PARTHIAN NUMBER ONE..INSCRIPTIONAL PARTHIAN NUMBER ONE THOUSAND
    if x >= 0x10B58 then
      return {status="disallowed"}
    end
  elseif x <= 0x10B72 then
    -- INSCRIPTIONAL PAHLAVI LETTER ALEPH..INSCRIPTIONAL PAHLAVI LETTER TAW
    if x >= 0x10B60 then
      return {status="valid"}
    end
  elseif x <= 0x10B7F then
    -- INSCRIPTIONAL PAHLAVI NUMBER ONE..INSCRIPTIONAL PAHLAVI NUMBER ONE THOUSAND
    if x >= 0x10B78 then
      return {status="disallowed"}
    end
  elseif x <= 0x10B91 then
    -- PSALTER PAHLAVI LETTER ALEPH..PSALTER PAHLAVI LETTER TAW
    if x >= 0x10B80 then
      return {status="valid"}
    end
  elseif x <= 0x10B98 then
    -- NA   <reserved-10B92>..<reserved-10B98>
    if x >= 0x10B92 then
      return {status="disallowed"}
    end
  elseif x <= 0x10BA8 then
    -- NA   <reserved-10B9D>..<reserved-10BA8>
    if x >= 0x10B9D then
      return {status="disallowed"}
    end
  elseif x <= 0x10BAF then
    -- PSALTER PAHLAVI NUMBER ONE..PSALTER PAHLAVI NUMBER ONE HUNDRED
    if x >= 0x10BA9 then
      return {status="disallowed"}
    end
  elseif x <= 0x10BFF then
    -- NA   <reserved-10BB0>..<reserved-10BFF>
    if x >= 0x10BB0 then
      return {status="disallowed"}
    end
  elseif x <= 0x10C48 then
    -- OLD TURKIC LETTER ORKHON A..OLD TURKIC LETTER ORKHON BASH
    if x >= 0x10C00 then
      return {status="valid"}
    end
  elseif x <= 0x10C7F then
    -- NA   <reserved-10C49>..<reserved-10C7F>
    if x >= 0x10C49 then
      return {status="disallowed"}
    end
  elseif x <= 0x10CBF then
    -- NA   <reserved-10CB3>..<reserved-10CBF>
    if x >= 0x10CB3 then
      return {status="disallowed"}
    end
  elseif x <= 0x10CF2 then
    -- OLD HUNGARIAN SMALL LETTER A..OLD HUNGARIAN SMALL LETTER US
    if x >= 0x10CC0 then
      return {status="valid"}
    end
  elseif x <= 0x10CF9 then
    -- NA   <reserved-10CF3>..<reserved-10CF9>
    if x >= 0x10CF3 then
      return {status="disallowed"}
    end
  elseif x <= 0x10CFF then
    -- OLD HUNGARIAN NUMBER ONE..OLD HUNGARIAN NUMBER ONE THOUSAND
    if x >= 0x10CFA then
      return {status="disallowed"}
    end
  elseif x <= 0x10E5F then
    -- NA   <reserved-10D00>..<reserved-10E5F>
    if x >= 0x10D00 then
      return {status="disallowed"}
    end
  elseif x <= 0x10E7E then
    -- RUMI DIGIT ONE..RUMI FRACTION TWO THIRDS
    if x >= 0x10E60 then
      return {status="disallowed"}
    end
  elseif x <= 0x10FFF then
    -- NA   <reserved-10E7F>..<reserved-10FFF>
    if x >= 0x10E7F then
      return {status="disallowed"}
    end
  elseif x <= 0x11046 then
    -- BRAHMI SIGN CANDRABINDU..BRAHMI VIRAMA
    if x >= 0x11000 then
      return {status="valid"}
    end
  elseif x <= 0x1104D then
    -- BRAHMI DANDA..BRAHMI PUNCTUATION LOTUS
    if x >= 0x11047 then
      return {status="disallowed"}
    end
  elseif x <= 0x11065 then
    -- BRAHMI NUMBER ONE..BRAHMI NUMBER ONE THOUSAND
    if x >= 0x11052 then
      return {status="disallowed"}
    end
  elseif x <= 0x1106F then
    -- BRAHMI DIGIT ZERO..BRAHMI DIGIT NINE
    if x >= 0x11066 then
      return {status="valid"}
    end
  elseif x <= 0x1107E then
    -- NA   <reserved-11070>..<reserved-1107E>
    if x >= 0x11070 then
      return {status="disallowed"}
    end
  elseif x <= 0x110BA then
    -- KAITHI SIGN CANDRABINDU..KAITHI SIGN NUKTA
    if x >= 0x11080 then
      return {status="valid"}
    end
  elseif x <= 0x110CF then
    -- NA   <reserved-110C2>..<reserved-110CF>
    if x >= 0x110C2 then
      return {status="disallowed"}
    end
  elseif x <= 0x110E8 then
    -- SORA SOMPENG LETTER SAH..SORA SOMPENG LETTER MAE
    if x >= 0x110D0 then
      return {status="valid"}
    end
  elseif x <= 0x110EF then
    -- NA   <reserved-110E9>..<reserved-110EF>
    if x >= 0x110E9 then
      return {status="disallowed"}
    end
  elseif x <= 0x110F9 then
    -- SORA SOMPENG DIGIT ZERO..SORA SOMPENG DIGIT NINE
    if x >= 0x110F0 then
      return {status="valid"}
    end
  elseif x <= 0x110FF then
    -- NA   <reserved-110FA>..<reserved-110FF>
    if x >= 0x110FA then
      return {status="disallowed"}
    end
  elseif x <= 0x11134 then
    -- CHAKMA SIGN CANDRABINDU..CHAKMA MAAYYAA
    if x >= 0x11100 then
      return {status="valid"}
    end
  elseif x <= 0x1113F then
    -- CHAKMA DIGIT ZERO..CHAKMA DIGIT NINE
    if x >= 0x11136 then
      return {status="valid"}
    end
  elseif x <= 0x1114F then
    -- NA   <reserved-11144>..<reserved-1114F>
    if x >= 0x11144 then
      return {status="disallowed"}
    end
  elseif x <= 0x11173 then
    -- MAHAJANI LETTER A..MAHAJANI SIGN NUKTA
    if x >= 0x11150 then
      return {status="valid"}
    end
  elseif x <= 0x1117F then
    -- NA   <reserved-11177>..<reserved-1117F>
    if x >= 0x11177 then
      return {status="disallowed"}
    end
  elseif x <= 0x111C4 then
    -- SHARADA SIGN CANDRABINDU..SHARADA OM
    if x >= 0x11180 then
      return {status="valid"}
    end
  elseif x <= 0x111D9 then
    -- SHARADA DIGIT ZERO..SHARADA DIGIT NINE
    if x >= 0x111D0 then
      return {status="valid"}
    end
  elseif x <= 0x111F4 then
    -- SINHALA ARCHAIC DIGIT ONE..SINHALA ARCHAIC NUMBER ONE THOUSAND
    if x >= 0x111E1 then
      return {status="disallowed"}
    end
  elseif x <= 0x111FF then
    -- NA   <reserved-111F5>..<reserved-111FF>
    if x >= 0x111F5 then
      return {status="disallowed"}
    end
  elseif x <= 0x11211 then
    -- KHOJKI LETTER A..KHOJKI LETTER JJA
    if x >= 0x11200 then
      return {status="valid"}
    end
  elseif x <= 0x11237 then
    -- KHOJKI LETTER NYA..KHOJKI SIGN SHADDA
    if x >= 0x11213 then
      return {status="valid"}
    end
  elseif x <= 0x1123D then
    -- KHOJKI DANDA..KHOJKI ABBREVIATION SIGN
    if x >= 0x11238 then
      return {status="disallowed"}
    end
  elseif x <= 0x1127F then
    -- NA   <reserved-1123F>..<reserved-1127F>
    if x >= 0x1123F then
      return {status="disallowed"}
    end
  elseif x <= 0x11286 then
    -- MULTANI LETTER A..MULTANI LETTER GA
    if x >= 0x11280 then
      return {status="valid"}
    end
  elseif x <= 0x1129D then
    -- MULTANI LETTER NYA..MULTANI LETTER BA
    if x >= 0x1128F then
      return {status="valid"}
    end
  elseif x <= 0x112A8 then
    -- MULTANI LETTER BHA..MULTANI LETTER RHA
    if x >= 0x1129F then
      return {status="valid"}
    end
  elseif x <= 0x112AF then
    -- NA   <reserved-112AA>..<reserved-112AF>
    if x >= 0x112AA then
      return {status="disallowed"}
    end
  elseif x <= 0x112EA then
    -- KHUDAWADI LETTER A..KHUDAWADI SIGN VIRAMA
    if x >= 0x112B0 then
      return {status="valid"}
    end
  elseif x <= 0x112F9 then
    -- KHUDAWADI DIGIT ZERO..KHUDAWADI DIGIT NINE
    if x >= 0x112F0 then
      return {status="valid"}
    end
  elseif x <= 0x112FF then
    -- NA   <reserved-112FA>..<reserved-112FF>
    if x >= 0x112FA then
      return {status="disallowed"}
    end
  elseif x <= 0x1130C then
    -- GRANTHA LETTER A..GRANTHA LETTER VOCALIC L
    if x >= 0x11305 then
      return {status="valid"}
    end
  elseif x <= 0x11328 then
    -- GRANTHA LETTER OO..GRANTHA LETTER NA
    if x >= 0x11313 then
      return {status="valid"}
    end
  elseif x <= 0x11330 then
    -- GRANTHA LETTER PA..GRANTHA LETTER RA
    if x >= 0x1132A then
      return {status="valid"}
    end
  elseif x <= 0x11344 then
    -- GRANTHA SIGN NUKTA..GRANTHA VOWEL SIGN VOCALIC RR
    if x >= 0x1133C then
      return {status="valid"}
    end
  elseif x <= 0x11356 then
    -- NA   <reserved-11351>..<reserved-11356>
    if x >= 0x11351 then
      return {status="disallowed"}
    end
  elseif x <= 0x11363 then
    -- GRANTHA SIGN PLUTA..GRANTHA VOWEL SIGN VOCALIC LL
    if x >= 0x1135D then
      return {status="valid"}
    end
  elseif x <= 0x1136C then
    -- COMBINING GRANTHA DIGIT ZERO..COMBINING GRANTHA DIGIT SIX
    if x >= 0x11366 then
      return {status="valid"}
    end
  elseif x <= 0x113FF then
    -- NA   <reserved-11375>..<reserved-113FF>
    if x >= 0x11375 then
      return {status="disallowed"}
    end
  elseif x <= 0x1144A then
    -- NEWA LETTER A..NEWA SIDDHI
    if x >= 0x11400 then
      return {status="valid"}
    end
  elseif x <= 0x11459 then
    -- NEWA DIGIT ZERO..NEWA DIGIT NINE
    if x >= 0x11450 then
      return {status="valid"}
    end
  elseif x <= 0x1147F then
    -- NA   <reserved-1145E>..<reserved-1147F>
    if x >= 0x1145E then
      return {status="disallowed"}
    end
  elseif x <= 0x114C5 then
    -- TIRHUTA ANJI..TIRHUTA GVANG
    if x >= 0x11480 then
      return {status="valid"}
    end
  elseif x <= 0x114CF then
    -- NA   <reserved-114C8>..<reserved-114CF>
    if x >= 0x114C8 then
      return {status="disallowed"}
    end
  elseif x <= 0x114D9 then
    -- TIRHUTA DIGIT ZERO..TIRHUTA DIGIT NINE
    if x >= 0x114D0 then
      return {status="valid"}
    end
  elseif x <= 0x1157F then
    -- NA   <reserved-114DA>..<reserved-1157F>
    if x >= 0x114DA then
      return {status="disallowed"}
    end
  elseif x <= 0x115B5 then
    -- SIDDHAM LETTER A..SIDDHAM VOWEL SIGN VOCALIC RR
    if x >= 0x11580 then
      return {status="valid"}
    end
  elseif x <= 0x115C0 then
    -- SIDDHAM VOWEL SIGN E..SIDDHAM SIGN NUKTA
    if x >= 0x115B8 then
      return {status="valid"}
    end
  elseif x <= 0x115D7 then
    -- SIDDHAM SIGN SIDDHAM..SIDDHAM END OF TEXT MARK
    -- SIDDHAM SECTION MARK WITH TRIDENT AND U-SHAPED ORNAMENTS..SIDDHAM SECTION MARK WITH CIRCLES AND FOUR ENCLOSURES
    if x >= 0x115C1 then
      return {status="disallowed"}
    end
  elseif x <= 0x115DD then
    -- SIDDHAM LETTER THREE-CIRCLE ALTERNATE I..SIDDHAM VOWEL SIGN ALTERNATE UU
    if x >= 0x115D8 then
      return {status="valid"}
    end
  elseif x <= 0x115FF then
    -- NA   <reserved-115DE>..<reserved-115FF>
    if x >= 0x115DE then
      return {status="disallowed"}
    end
  elseif x <= 0x11640 then
    -- MODI LETTER A..MODI SIGN ARDHACANDRA
    if x >= 0x11600 then
      return {status="valid"}
    end
  elseif x <= 0x1164F then
    -- NA   <reserved-11645>..<reserved-1164F>
    if x >= 0x11645 then
      return {status="disallowed"}
    end
  elseif x <= 0x11659 then
    -- MODI DIGIT ZERO..MODI DIGIT NINE
    if x >= 0x11650 then
      return {status="valid"}
    end
  elseif x <= 0x1165F then
    -- NA   <reserved-1165A>..<reserved-1165F>
    if x >= 0x1165A then
      return {status="disallowed"}
    end
  elseif x <= 0x1166C then
    -- MONGOLIAN BIRGA WITH ORNAMENT..MONGOLIAN TURNED SWIRL BIRGA WITH DOUBLE ORNAMENT
    if x >= 0x11660 then
      return {status="disallowed"}
    end
  elseif x <= 0x1167F then
    -- NA   <reserved-1166D>..<reserved-1167F>
    if x >= 0x1166D then
      return {status="disallowed"}
    end
  elseif x <= 0x116B7 then
    -- TAKRI LETTER A..TAKRI SIGN NUKTA
    if x >= 0x11680 then
      return {status="valid"}
    end
  elseif x <= 0x116BF then
    -- NA   <reserved-116B8>..<reserved-116BF>
    if x >= 0x116B8 then
      return {status="disallowed"}
    end
  elseif x <= 0x116C9 then
    -- TAKRI DIGIT ZERO..TAKRI DIGIT NINE
    if x >= 0x116C0 then
      return {status="valid"}
    end
  elseif x <= 0x116FF then
    -- NA   <reserved-116CA>..<reserved-116FF>
    if x >= 0x116CA then
      return {status="disallowed"}
    end
  elseif x <= 0x11719 then
    -- AHOM LETTER KA..AHOM LETTER JHA
    if x >= 0x11700 then
      return {status="valid"}
    end
  elseif x <= 0x1172B then
    -- AHOM CONSONANT SIGN MEDIAL LA..AHOM SIGN KILLER
    if x >= 0x1171D then
      return {status="valid"}
    end
  elseif x <= 0x11739 then
    -- AHOM DIGIT ZERO..AHOM DIGIT NINE
    if x >= 0x11730 then
      return {status="valid"}
    end
  elseif x <= 0x1173F then
    -- AHOM NUMBER TEN..AHOM SYMBOL VI
    if x >= 0x1173A then
      return {status="disallowed"}
    end
  elseif x <= 0x1189F then
    -- NA   <reserved-11740>..<reserved-1189F>
    if x >= 0x11740 then
      return {status="disallowed"}
    end
  elseif x <= 0x118E9 then
    -- WARANG CITI SMALL LETTER NGAA..WARANG CITI DIGIT NINE
    if x >= 0x118C0 then
      return {status="valid"}
    end
  elseif x <= 0x118F2 then
    -- WARANG CITI NUMBER TEN..WARANG CITI NUMBER NINETY
    if x >= 0x118EA then
      return {status="disallowed"}
    end
  elseif x <= 0x118FE then
    -- NA   <reserved-118F3>..<reserved-118FE>
    if x >= 0x118F3 then
      return {status="disallowed"}
    end
  elseif x <= 0x119FF then
    -- NA   <reserved-11900>..<reserved-119FF>
    if x >= 0x11900 then
      return {status="disallowed"}
    end
  elseif x <= 0x11A3E then
    -- ZANABAZAR SQUARE LETTER A..ZANABAZAR SQUARE CLUSTER-FINAL LETTER VA
    if x >= 0x11A00 then
      return {status="valid"}
    end
  elseif x <= 0x11A46 then
    -- ZANABAZAR SQUARE INITIAL HEAD MARK..ZANABAZAR SQUARE CLOSING DOUBLE-LINED HEAD MARK
    if x >= 0x11A3F then
      return {status="disallowed"}
    end
  elseif x <= 0x11A4F then
    -- NA   <reserved-11A48>..<reserved-11A4F>
    if x >= 0x11A48 then
      return {status="disallowed"}
    end
  elseif x <= 0x11A83 then
    -- SOYOMBO LETTER A..SOYOMBO LETTER KSSA
    if x >= 0x11A50 then
      return {status="valid"}
    end
  elseif x <= 0x11A99 then
    -- SOYOMBO CLUSTER-INITIAL LETTER RA..SOYOMBO SUBJOINER
    if x >= 0x11A86 then
      return {status="valid"}
    end
  elseif x <= 0x11ABF then
    -- NA   <reserved-11AA3>..<reserved-11ABF>
    if x >= 0x11AA3 then
      return {status="disallowed"}
    end
  elseif x <= 0x11AF8 then
    -- PAU CIN HAU LETTER PA..PAU CIN HAU GLOTTAL STOP FINAL
    if x >= 0x11AC0 then
      return {status="valid"}
    end
  elseif x <= 0x11BFF then
    -- NA   <reserved-11AF9>..<reserved-11BFF>
    if x >= 0x11AF9 then
      return {status="disallowed"}
    end
  elseif x <= 0x11C08 then
    -- BHAIKSUKI LETTER A..BHAIKSUKI LETTER VOCALIC L
    if x >= 0x11C00 then
      return {status="valid"}
    end
  elseif x <= 0x11C36 then
    -- BHAIKSUKI LETTER E..BHAIKSUKI VOWEL SIGN VOCALIC L
    if x >= 0x11C0A then
      return {status="valid"}
    end
  elseif x <= 0x11C40 then
    -- BHAIKSUKI VOWEL SIGN E..BHAIKSUKI SIGN AVAGRAHA
    if x >= 0x11C38 then
      return {status="valid"}
    end
  elseif x <= 0x11C4F then
    -- NA   <reserved-11C46>..<reserved-11C4F>
    if x >= 0x11C46 then
      return {status="disallowed"}
    end
  elseif x <= 0x11C59 then
    -- BHAIKSUKI DIGIT ZERO..BHAIKSUKI DIGIT NINE
    if x >= 0x11C50 then
      return {status="valid"}
    end
  elseif x <= 0x11C6C then
    -- BHAIKSUKI NUMBER ONE..BHAIKSUKI HUNDREDS UNIT MARK
    if x >= 0x11C5A then
      return {status="disallowed"}
    end
  elseif x <= 0x11C8F then
    -- MARCHEN LETTER KA..MARCHEN LETTER A
    if x >= 0x11C72 then
      return {status="valid"}
    end
  elseif x <= 0x11CA7 then
    -- MARCHEN SUBJOINED LETTER KA..MARCHEN SUBJOINED LETTER ZA
    if x >= 0x11C92 then
      return {status="valid"}
    end
  elseif x <= 0x11CB6 then
    -- MARCHEN SUBJOINED LETTER YA..MARCHEN SIGN CANDRABINDU
    if x >= 0x11CA9 then
      return {status="valid"}
    end
  elseif x <= 0x11CFF then
    -- NA   <reserved-11CB7>..<reserved-11CFF>
    if x >= 0x11CB7 then
      return {status="disallowed"}
    end
  elseif x <= 0x11D06 then
    -- MASARAM GONDI LETTER A..MASARAM GONDI LETTER E
    if x >= 0x11D00 then
      return {status="valid"}
    end
  elseif x <= 0x11D36 then
    -- MASARAM GONDI LETTER AU..MASARAM GONDI VOWEL SIGN VOCALIC R
    if x >= 0x11D0B then
      return {status="valid"}
    end
  elseif x <= 0x11D47 then
    -- MASARAM GONDI VOWEL SIGN AU..MASARAM GONDI RA-KARA
    if x >= 0x11D3F then
      return {status="valid"}
    end
  elseif x <= 0x11D4F then
    -- NA   <reserved-11D48>..<reserved-11D4F>
    if x >= 0x11D48 then
      return {status="disallowed"}
    end
  elseif x <= 0x11D59 then
    -- MASARAM GONDI DIGIT ZERO..MASARAM GONDI DIGIT NINE
    if x >= 0x11D50 then
      return {status="valid"}
    end
  elseif x <= 0x11FFF then
    -- NA   <reserved-11D5A>..<reserved-11FFF>
    if x >= 0x11D5A then
      return {status="disallowed"}
    end
  elseif x <= 0x12398 then
    -- CUNEIFORM SIGN A..CUNEIFORM SIGN ZUM
    -- CUNEIFORM SIGN KAP ELAMITE..CUNEIFORM SIGN UM TIMES ME
    if x >= 0x12000 then
      return {status="valid"}
    end
  elseif x <= 0x123FF then
    -- NA   <reserved-1239A>..<reserved-123FF>
    if x >= 0x1239A then
      return {status="disallowed"}
    end
  elseif x <= 0x1246E then
    -- CUNEIFORM NUMERIC SIGN TWO ASH..CUNEIFORM NUMERIC SIGN OLD ASSYRIAN ONE QUARTER
    -- CUNEIFORM NUMERIC SIGN ONE QUARTER GUR..CUNEIFORM NUMERIC SIGN NINE U VARIANT FORM
    if x >= 0x12400 then
      return {status="disallowed"}
    end
  elseif x <= 0x1247F then
    -- NA   <reserved-12475>..<reserved-1247F>
    if x >= 0x12475 then
      return {status="disallowed"}
    end
  elseif x <= 0x12543 then
    -- CUNEIFORM SIGN AB TIMES NUN TENU..CUNEIFORM SIGN ZU5 TIMES THREE DISH TENU
    if x >= 0x12480 then
      return {status="valid"}
    end
  elseif x <= 0x12FFF then
    -- NA   <reserved-12544>..<reserved-12FFF>
    if x >= 0x12544 then
      return {status="disallowed"}
    end
  elseif x <= 0x1342E then
    -- EGYPTIAN HIEROGLYPH A001..EGYPTIAN HIEROGLYPH AA032
    if x >= 0x13000 then
      return {status="valid"}
    end
  elseif x <= 0x143FF then
    -- NA   <reserved-1342F>..<reserved-143FF>
    if x >= 0x1342F then
      return {status="disallowed"}
    end
  elseif x <= 0x14646 then
    -- ANATOLIAN HIEROGLYPH A001..ANATOLIAN HIEROGLYPH A530
    if x >= 0x14400 then
      return {status="valid"}
    end
  elseif x <= 0x167FF then
    -- NA   <reserved-14647>..<reserved-167FF>
    if x >= 0x14647 then
      return {status="disallowed"}
    end
  elseif x <= 0x16A38 then
    -- BAMUM LETTER PHASE-A NGKUE MFON..BAMUM LETTER PHASE-F VUEQ
    if x >= 0x16800 then
      return {status="valid"}
    end
  elseif x <= 0x16A3F then
    -- NA   <reserved-16A39>..<reserved-16A3F>
    if x >= 0x16A39 then
      return {status="disallowed"}
    end
  elseif x <= 0x16A5E then
    -- MRO LETTER TA..MRO LETTER TEK
    if x >= 0x16A40 then
      return {status="valid"}
    end
  elseif x <= 0x16A69 then
    -- MRO DIGIT ZERO..MRO DIGIT NINE
    if x >= 0x16A60 then
      return {status="valid"}
    end
  elseif x <= 0x16ACF then
    -- NA   <reserved-16A70>..<reserved-16ACF>
    if x >= 0x16A70 then
      return {status="disallowed"}
    end
  elseif x <= 0x16AED then
    -- BASSA VAH LETTER ENNI..BASSA VAH LETTER I
    if x >= 0x16AD0 then
      return {status="valid"}
    end
  elseif x <= 0x16AFF then
    -- NA   <reserved-16AF6>..<reserved-16AFF>
    if x >= 0x16AF6 then
      return {status="disallowed"}
    end
  elseif x <= 0x16B36 then
    -- PAHAWH HMONG VOWEL KEEB..PAHAWH HMONG MARK CIM TAUM
    if x >= 0x16B00 then
      return {status="valid"}
    end
  elseif x <= 0x16B3F then
    -- PAHAWH HMONG SIGN VOS THOM..PAHAWH HMONG SIGN XYEEM FAIB
    if x >= 0x16B37 then
      return {status="disallowed"}
    end
  elseif x <= 0x16B4F then
    -- NA   <reserved-16B46>..<reserved-16B4F>
    if x >= 0x16B46 then
      return {status="disallowed"}
    end
  elseif x <= 0x16B59 then
    -- PAHAWH HMONG DIGIT ZERO..PAHAWH HMONG DIGIT NINE
    if x >= 0x16B50 then
      return {status="valid"}
    end
  elseif x <= 0x16B61 then
    -- PAHAWH HMONG NUMBER TENS..PAHAWH HMONG NUMBER TRILLIONS
    if x >= 0x16B5B then
      return {status="disallowed"}
    end
  elseif x <= 0x16B77 then
    -- PAHAWH HMONG SIGN VOS LUB..PAHAWH HMONG SIGN CIM NRES TOS
    if x >= 0x16B63 then
      return {status="valid"}
    end
  elseif x <= 0x16B8F then
    -- PAHAWH HMONG CLAN SIGN TSHEEJ..PAHAWH HMONG CLAN SIGN VWJ
    if x >= 0x16B7D then
      return {status="valid"}
    end
  elseif x <= 0x16EFF then
    -- NA   <reserved-16B90>..<reserved-16EFF>
    if x >= 0x16B90 then
      return {status="disallowed"}
    end
  elseif x <= 0x16F44 then
    -- MIAO LETTER PA..MIAO LETTER HHA
    if x >= 0x16F00 then
      return {status="valid"}
    end
  elseif x <= 0x16F4F then
    -- NA   <reserved-16F45>..<reserved-16F4F>
    if x >= 0x16F45 then
      return {status="disallowed"}
    end
  elseif x <= 0x16F7E then
    -- MIAO LETTER NASALIZATION..MIAO VOWEL SIGN NG
    if x >= 0x16F50 then
      return {status="valid"}
    end
  elseif x <= 0x16F8E then
    -- NA   <reserved-16F7F>..<reserved-16F8E>
    if x >= 0x16F7F then
      return {status="disallowed"}
    end
  elseif x <= 0x16F9F then
    -- MIAO TONE RIGHT..MIAO LETTER REFORMED TONE-8
    if x >= 0x16F8F then
      return {status="valid"}
    end
  elseif x <= 0x16FDF then
    -- NA   <reserved-16FA0>..<reserved-16FDF>
    if x >= 0x16FA0 then
      return {status="disallowed"}
    end
  elseif x <= 0x16FFF then
    -- NA   <reserved-16FE2>..<reserved-16FFF>
    if x >= 0x16FE2 then
      return {status="disallowed"}
    end
  elseif x <= 0x187EC then
    -- TANGUT IDEOGRAPH-17000..TANGUT IDEOGRAPH-187EC
    if x >= 0x17000 then
      return {status="valid"}
    end
  elseif x <= 0x187FF then
    -- NA   <reserved-187ED>..<reserved-187FF>
    if x >= 0x187ED then
      return {status="disallowed"}
    end
  elseif x <= 0x18AF2 then
    -- TANGUT COMPONENT-001..TANGUT COMPONENT-755
    if x >= 0x18800 then
      return {status="valid"}
    end
  elseif x <= 0x1AFFF then
    -- NA   <reserved-18AF3>..<reserved-1AFFF>
    if x >= 0x18AF3 then
      return {status="disallowed"}
    end
  elseif x <= 0x1B11E then
    -- KATAKANA LETTER ARCHAIC E..HIRAGANA LETTER ARCHAIC YE
    -- HENTAIGANA LETTER A-1..HENTAIGANA LETTER N-MU-MO-2
    if x >= 0x1B000 then
      return {status="valid"}
    end
  elseif x <= 0x1B16F then
    -- NA   <reserved-1B11F>..<reserved-1B16F>
    if x >= 0x1B11F then
      return {status="disallowed"}
    end
  elseif x <= 0x1B2FB then
    -- NUSHU CHARACTER-1B170..NUSHU CHARACTER-1B2FB
    if x >= 0x1B170 then
      return {status="valid"}
    end
  elseif x <= 0x1BBFF then
    -- NA   <reserved-1B2FC>..<reserved-1BBFF>
    if x >= 0x1B2FC then
      return {status="disallowed"}
    end
  elseif x <= 0x1BC6A then
    -- DUPLOYAN LETTER H..DUPLOYAN LETTER VOCALIC M
    if x >= 0x1BC00 then
      return {status="valid"}
    end
  elseif x <= 0x1BC7C then
    -- DUPLOYAN AFFIX LEFT HORIZONTAL SECANT..DUPLOYAN AFFIX ATTACHED TANGENT HOOK
    if x >= 0x1BC70 then
      return {status="valid"}
    end
  elseif x <= 0x1BC88 then
    -- DUPLOYAN AFFIX HIGH ACUTE..DUPLOYAN AFFIX HIGH VERTICAL
    if x >= 0x1BC80 then
      return {status="valid"}
    end
  elseif x <= 0x1BC8F then
    -- NA   <reserved-1BC89>..<reserved-1BC8F>
    if x >= 0x1BC89 then
      return {status="disallowed"}
    end
  elseif x <= 0x1BC99 then
    -- DUPLOYAN AFFIX LOW ACUTE..DUPLOYAN AFFIX LOW ARROW
    if x >= 0x1BC90 then
      return {status="valid"}
    end
  elseif x <= 0x1CFFF then
    -- NA   <reserved-1BCA4>..<reserved-1CFFF>
    if x >= 0x1BCA4 then
      return {status="disallowed"}
    end
  elseif x <= 0x1D0F5 then
    -- BYZANTINE MUSICAL SYMBOL PSILI..BYZANTINE MUSICAL SYMBOL GORGON NEO KATO
    if x >= 0x1D000 then
      return {status="disallowed"}
    end
  elseif x <= 0x1D0FF then
    -- NA   <reserved-1D0F6>..<reserved-1D0FF>
    if x >= 0x1D0F6 then
      return {status="disallowed"}
    end
  elseif x <= 0x1D126 then
    -- MUSICAL SYMBOL SINGLE BARLINE..MUSICAL SYMBOL DRUM CLEF-2
    if x >= 0x1D100 then
      return {status="disallowed"}
    end
  elseif x <= 0x1D15D then
    -- MUSICAL SYMBOL DOUBLE SHARP..MUSICAL SYMBOL WHOLE NOTE
    if x >= 0x1D12A then
      return {status="disallowed"}
    end
  elseif x <= 0x1D172 then
    -- MUSICAL SYMBOL COMBINING STEM..MUSICAL SYMBOL COMBINING FLAG-5
    if x >= 0x1D165 then
      return {status="disallowed"}
    end
  elseif x <= 0x1D17A then
    -- MUSICAL SYMBOL BEGIN BEAM..MUSICAL SYMBOL END PHRASE
    if x >= 0x1D173 then
      return {status="disallowed"}
    end
  elseif x <= 0x1D1BA then
    -- MUSICAL SYMBOL COMBINING ACCENT..MUSICAL SYMBOL SEMIBREVIS BLACK
    if x >= 0x1D17B then
      return {status="disallowed"}
    end
  elseif x <= 0x1D1E8 then
    -- MUSICAL SYMBOL LONGA PERFECTA REST..MUSICAL SYMBOL PES SUBPUNCTIS
    -- MUSICAL SYMBOL KIEVAN C CLEF..MUSICAL SYMBOL KIEVAN FLAT SIGN
    if x >= 0x1D1C1 then
      return {status="disallowed"}
    end
  elseif x <= 0x1D1FF then
    -- NA   <reserved-1D1E9>..<reserved-1D1FF>
    if x >= 0x1D1E9 then
      return {status="disallowed"}
    end
  elseif x <= 0x1D245 then
    -- GREEK VOCAL NOTATION SYMBOL-1..GREEK MUSICAL LEIMMA
    if x >= 0x1D200 then
      return {status="disallowed"}
    end
  elseif x <= 0x1D2FF then
    -- NA   <reserved-1D246>..<reserved-1D2FF>
    if x >= 0x1D246 then
      return {status="disallowed"}
    end
  elseif x <= 0x1D356 then
    -- MONOGRAM FOR EARTH..TETRAGRAM FOR FOSTERING
    if x >= 0x1D300 then
      return {status="disallowed"}
    end
  elseif x <= 0x1D35F then
    -- NA   <reserved-1D357>..<reserved-1D35F>
    if x >= 0x1D357 then
      return {status="disallowed"}
    end
  elseif x <= 0x1D371 then
    -- COUNTING ROD UNIT DIGIT ONE..COUNTING ROD TENS DIGIT NINE
    if x >= 0x1D360 then
      return {status="disallowed"}
    end
  elseif x <= 0x1D3FF then
    -- NA   <reserved-1D372>..<reserved-1D3FF>
    if x >= 0x1D372 then
      return {status="disallowed"}
    end
  elseif x <= 0x1D9FF then
    -- SIGNWRITING HAND-FIST INDEX..SIGNWRITING HEAD
    if x >= 0x1D800 then
      return {status="disallowed"}
    end
  elseif x <= 0x1DA36 then
    -- SIGNWRITING HEAD RIM..SIGNWRITING AIR SUCKING IN
    if x >= 0x1DA00 then
      return {status="valid"}
    end
  elseif x <= 0x1DA6C then
    -- SIGNWRITING MOUTH CLOSED NEUTRAL..SIGNWRITING EXCITEMENT
    if x >= 0x1DA3B then
      return {status="valid"}
    end
  elseif x <= 0x1DA74 then
    -- SIGNWRITING SHOULDER HIP SPINE..SIGNWRITING TORSO-FLOORPLANE TWISTING
    if x >= 0x1DA6D then
      return {status="disallowed"}
    end
  elseif x <= 0x1DA83 then
    -- SIGNWRITING LIMB COMBINATION..SIGNWRITING LOCATION DEPTH
    if x >= 0x1DA76 then
      return {status="disallowed"}
    end
  elseif x <= 0x1DA8B then
    -- SIGNWRITING LOCATION TORSO..SIGNWRITING PARENTHESIS
    if x >= 0x1DA85 then
      return {status="disallowed"}
    end
  elseif x <= 0x1DA9A then
    -- NA   <reserved-1DA8C>..<reserved-1DA9A>
    if x >= 0x1DA8C then
      return {status="disallowed"}
    end
  elseif x <= 0x1DAAF then
    -- SIGNWRITING ROTATION MODIFIER-2..SIGNWRITING ROTATION MODIFIER-16
    if x >= 0x1DAA1 then
      return {status="valid"}
    end
  elseif x <= 0x1DFFF then
    -- NA   <reserved-1DAB0>..<reserved-1DFFF>
    if x >= 0x1DAB0 then
      return {status="disallowed"}
    end
  elseif x <= 0x1E006 then
    -- COMBINING GLAGOLITIC LETTER AZU..COMBINING GLAGOLITIC LETTER ZHIVETE
    if x >= 0x1E000 then
      return {status="valid"}
    end
  elseif x <= 0x1E018 then
    -- COMBINING GLAGOLITIC LETTER ZEMLJA..COMBINING GLAGOLITIC LETTER HERU
    if x >= 0x1E008 then
      return {status="valid"}
    end
  elseif x <= 0x1E021 then
    -- COMBINING GLAGOLITIC LETTER SHTA..COMBINING GLAGOLITIC LETTER YATI
    if x >= 0x1E01B then
      return {status="valid"}
    end
  elseif x <= 0x1E7FF then
    -- NA   <reserved-1E02B>..<reserved-1E7FF>
    if x >= 0x1E02B then
      return {status="disallowed"}
    end
  elseif x <= 0x1E8C4 then
    -- MENDE KIKAKUI SYLLABLE M001 KI..MENDE KIKAKUI SYLLABLE M060 NYON
    if x >= 0x1E800 then
      return {status="valid"}
    end
  elseif x <= 0x1E8CF then
    -- MENDE KIKAKUI DIGIT ONE..MENDE KIKAKUI DIGIT NINE
    if x >= 0x1E8C7 then
      return {status="disallowed"}
    end
  elseif x <= 0x1E8D6 then
    -- MENDE KIKAKUI COMBINING NUMBER TEENS..MENDE KIKAKUI COMBINING NUMBER MILLIONS
    if x >= 0x1E8D0 then
      return {status="valid"}
    end
  elseif x <= 0x1E8FF then
    -- NA   <reserved-1E8D7>..<reserved-1E8FF>
    if x >= 0x1E8D7 then
      return {status="disallowed"}
    end
  elseif x <= 0x1E94A then
    -- ADLAM SMALL LETTER ALIF..ADLAM NUKTA
    if x >= 0x1E922 then
      return {status="valid"}
    end
  elseif x <= 0x1E959 then
    -- ADLAM DIGIT ZERO..ADLAM DIGIT NINE
    if x >= 0x1E950 then
      return {status="valid"}
    end
  elseif x <= 0x1EDFF then
    -- NA   <reserved-1E960>..<reserved-1EDFF>
    if x >= 0x1E960 then
      return {status="disallowed"}
    end
  elseif x <= 0x1EE41 then
    -- NA   <reserved-1EE3C>..<reserved-1EE41>
    if x >= 0x1EE3C then
      return {status="disallowed"}
    end
  elseif x <= 0x1EEEF then
    -- NA   <reserved-1EEBC>..<reserved-1EEEF>
    if x >= 0x1EEBC then
      return {status="disallowed"}
    end
  elseif x <= 0x1EFFF then
    -- NA   <reserved-1EEF2>..<reserved-1EFFF>
    if x >= 0x1EEF2 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F02B then
    -- MAHJONG TILE EAST WIND..MAHJONG TILE BACK
    if x >= 0x1F000 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F093 then
    -- DOMINO TILE HORIZONTAL BACK..DOMINO TILE VERTICAL-06-06
    if x >= 0x1F030 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F09F then
    -- NA   <reserved-1F094>..<reserved-1F09F>
    if x >= 0x1F094 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F0AE then
    -- PLAYING CARD BACK..PLAYING CARD KING OF SPADES
    if x >= 0x1F0A0 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F0BE then
    -- PLAYING CARD ACE OF HEARTS..PLAYING CARD KING OF HEARTS
    if x >= 0x1F0B1 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F0CF then
    -- PLAYING CARD ACE OF DIAMONDS..PLAYING CARD BLACK JOKER
    if x >= 0x1F0C1 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F0F5 then
    -- PLAYING CARD ACE OF CLUBS..PLAYING CARD WHITE JOKER
    -- PLAYING CARD FOOL..PLAYING CARD TRUMP-21
    if x >= 0x1F0D1 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F0FF then
    -- NA   <reserved-1F0F6>..<reserved-1F0FF>
    if x >= 0x1F0F6 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F156 then
    -- NEGATIVE CIRCLED LATIN CAPITAL LETTER A..NEGATIVE CIRCLED LATIN CAPITAL LETTER G
    if x >= 0x1F150 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F15E then
    -- NEGATIVE CIRCLED LATIN CAPITAL LETTER I..NEGATIVE CIRCLED LATIN CAPITAL LETTER O
    if x >= 0x1F158 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F169 then
    -- NEGATIVE CIRCLED LATIN CAPITAL LETTER Q..NEGATIVE CIRCLED LATIN CAPITAL LETTER Z
    if x >= 0x1F160 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F178 then
    -- NEGATIVE SQUARED LATIN CAPITAL LETTER A..NEGATIVE SQUARED LATIN CAPITAL LETTER I
    if x >= 0x1F170 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F18F then
    -- NEGATIVE SQUARED LATIN CAPITAL LETTER Q..NEGATIVE SQUARED LATIN CAPITAL LETTER Z
    -- CROSSED NEGATIVE SQUARED LATIN CAPITAL LETTER P..NEGATIVE SQUARED SA
    -- NEGATIVE SQUARED AB..NEGATIVE SQUARED WC
    if x >= 0x1F180 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F1AC then
    -- SQUARED CL..SQUARED VS
    -- SQUARED THREE D..SQUARED VOD
    if x >= 0x1F191 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F1E5 then
    -- NA   <reserved-1F1AD>..<reserved-1F1E5>
    if x >= 0x1F1AD then
      return {status="disallowed"}
    end
  elseif x <= 0x1F1FF then
    -- REGIONAL INDICATOR SYMBOL LETTER A..REGIONAL INDICATOR SYMBOL LETTER Z
    if x >= 0x1F1E6 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F20F then
    -- NA   <reserved-1F203>..<reserved-1F20F>
    if x >= 0x1F203 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F24F then
    -- NA   <reserved-1F249>..<reserved-1F24F>
    if x >= 0x1F249 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F25F then
    -- NA   <reserved-1F252>..<reserved-1F25F>
    if x >= 0x1F252 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F265 then
    -- ROUNDED SYMBOL FOR FU..ROUNDED SYMBOL FOR CAI
    if x >= 0x1F260 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F2FF then
    -- NA   <reserved-1F266>..<reserved-1F2FF>
    if x >= 0x1F266 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F335 then
    -- CYCLONE..SHOOTING STAR
    -- THERMOMETER..WIND BLOWING FACE
    -- HOT DOG..BURRITO
    -- CHESTNUT..CACTUS
    if x >= 0x1F300 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F37C then
    -- TULIP..BABY BOTTLE
    if x >= 0x1F337 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F3C4 then
    -- BOTTLE WITH POPPING CORK..POPCORN
    -- RIBBON..GRADUATION CAP
    -- HEART WITH TIP ON THE LEFT..ADMISSION TICKETS
    -- CAROUSEL HORSE..SURFER
    if x >= 0x1F37E then
      return {status="disallowed"}
    end
  elseif x <= 0x1F43E then
    -- TROPHY..SWIMMER
    -- WEIGHT LIFTER..RACING CAR
    -- CRICKET BAT AND BALL..TABLE TENNIS PADDLE AND BALL
    -- SNOW CAPPED MOUNTAIN..STADIUM
    -- HOUSE BUILDING..EUROPEAN CASTLE
    -- WHITE PENNANT..LABEL
    -- BADMINTON RACQUET AND SHUTTLECOCK..EMOJI MODIFIER FITZPATRICK TYPE-6
    -- RAT..PAW PRINTS
    if x >= 0x1F3C6 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F4F7 then
    -- EAR..CAMERA
    if x >= 0x1F442 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F4FE then
    -- VIDEO CAMERA..VIDEOCASSETTE
    -- FILM PROJECTOR..PORTABLE STEREO
    if x >= 0x1F4F9 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F579 then
    -- TWISTED RIGHTWARDS ARROWS..DOWN-POINTING SMALL RED TRIANGLE
    -- LOWER RIGHT SHADOWED WHITE CIRCLE..UPPER RIGHT SHADOWED WHITE CIRCLE
    -- CIRCLED CROSS POMMEE..NOTCHED LEFT SEMICIRCLE WITH THREE DOTS
    -- NOTCHED RIGHT SEMICIRCLE WITH THREE DOTS..DOVE OF PEACE
    -- KAABA..BOWL OF HYGIEIA
    -- CLOCK FACE ONE OCLOCK..CLOCK FACE TWELVE-THIRTY
    -- RIGHT SPEAKER..JOYSTICK
    if x >= 0x1F500 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F5A3 then
    -- LEFT HAND TELEPHONE RECEIVER..BLACK DOWN POINTING BACKHAND INDEX
    if x >= 0x1F57B then
      return {status="disallowed"}
    end
  elseif x <= 0x1F5FF then
    -- DESKTOP COMPUTER..WORLD MAP
    -- MOUNT FUJI..MOYAI
    if x >= 0x1F5A5 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F610 then
    -- GRINNING FACE WITH SMILING EYES..NEUTRAL FACE
    if x >= 0x1F601 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F62B then
    -- ANGRY FACE..DISAPPOINTED BUT RELIEVED FACE
    -- FROWNING FACE WITH OPEN MOUTH..ANGUISHED FACE
    -- FEARFUL FACE..TIRED FACE
    if x >= 0x1F620 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F633 then
    -- FACE WITH OPEN MOUTH..HUSHED FACE
    -- FACE WITH OPEN MOUTH AND COLD SWEAT..FLUSHED FACE
    if x >= 0x1F62E then
      return {status="disallowed"}
    end
  elseif x <= 0x1F6CF then
    -- DIZZY FACE..WEARY CAT FACE
    -- SLIGHTLY FROWNING FACE..SLIGHTLY SMILING FACE
    -- UPSIDE-DOWN FACE..FACE WITH ROLLING EYES
    -- FACE WITH NO GOOD GESTURE..PERSON WITH FOLDED HANDS
    -- NORTH WEST POINTING LEAF..REVERSE CHECKER BOARD
    -- ROCKET..LEFT LUGGAGE
    -- TRIANGLE WITH ROUNDED CORNERS..BED
    if x >= 0x1F635 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F6DF then
    -- NA   <reserved-1F6D5>..<reserved-1F6DF>
    if x >= 0x1F6D5 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F6EC then
    -- HAMMER AND WRENCH..AIRPLANE ARRIVING
    if x >= 0x1F6E0 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F6F8 then
    -- SATELLITE..PASSENGER SHIP
    -- SCOOTER..CANOE
    -- SLED..FLYING SAUCER
    if x >= 0x1F6F0 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F6FF then
    -- NA   <reserved-1F6F9>..<reserved-1F6FF>
    if x >= 0x1F6F9 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F773 then
    -- ALCHEMICAL SYMBOL FOR QUINTESSENCE..ALCHEMICAL SYMBOL FOR HALF OUNCE
    if x >= 0x1F700 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F77F then
    -- NA   <reserved-1F774>..<reserved-1F77F>
    if x >= 0x1F774 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F7D4 then
    -- BLACK LEFT-POINTING ISOSCELES RIGHT TRIANGLE..HEAVY TWELVE POINTED PINWHEEL STAR
    if x >= 0x1F780 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F7FF then
    -- NA   <reserved-1F7D5>..<reserved-1F7FF>
    if x >= 0x1F7D5 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F80B then
    -- LEFTWARDS ARROW WITH SMALL TRIANGLE ARROWHEAD..DOWNWARDS ARROW WITH LARGE TRIANGLE ARROWHEAD
    if x >= 0x1F800 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F847 then
    -- LEFTWARDS ARROW WITH SMALL EQUILATERAL ARROWHEAD..DOWNWARDS HEAVY ARROW
    if x >= 0x1F810 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F84F then
    -- NA   <reserved-1F848>..<reserved-1F84F>
    if x >= 0x1F848 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F859 then
    -- LEFTWARDS SANS-SERIF ARROW..UP DOWN SANS-SERIF ARROW
    if x >= 0x1F850 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F85F then
    -- NA   <reserved-1F85A>..<reserved-1F85F>
    if x >= 0x1F85A then
      return {status="disallowed"}
    end
  elseif x <= 0x1F887 then
    -- WIDE-HEADED LEFTWARDS LIGHT BARB ARROW..WIDE-HEADED SOUTH WEST VERY HEAVY BARB ARROW
    if x >= 0x1F860 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F88F then
    -- NA   <reserved-1F888>..<reserved-1F88F>
    if x >= 0x1F888 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F8AD then
    -- LEFTWARDS TRIANGLE ARROWHEAD..WHITE ARROW SHAFT WIDTH TWO THIRDS
    if x >= 0x1F890 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F8FF then
    -- NA   <reserved-1F8AE>..<reserved-1F8FF>
    if x >= 0x1F8AE then
      return {status="disallowed"}
    end
  elseif x <= 0x1F90B then
    -- CIRCLED CROSS FORMEE WITH FOUR DOTS..DOWNWARD FACING NOTCHED HOOK WITH DOT
    if x >= 0x1F900 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F91E then
    -- ZIPPER-MOUTH FACE..SIGN OF THE HORNS
    -- CALL ME HAND..HAND WITH INDEX AND MIDDLE FINGERS CROSSED
    if x >= 0x1F910 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F92F then
    -- FACE WITH COWBOY HAT..SNEEZING FACE
    -- FACE WITH ONE EYEBROW RAISED..SHOCKED FACE WITH EXPLODING HEAD
    if x >= 0x1F920 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F93E then
    -- BREAST-FEEDING..PALMS UP TOGETHER
    -- SELFIE..HANDBALL
    if x >= 0x1F931 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F94B then
    -- WILTED FLOWER..MARTIAL ARTS UNIFORM
    if x >= 0x1F940 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F96B then
    -- CROISSANT..PANCAKES
    -- DUMPLING..CANNED FOOD
    if x >= 0x1F950 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F97F then
    -- NA   <reserved-1F96C>..<reserved-1F97F>
    if x >= 0x1F96C then
      return {status="disallowed"}
    end
  elseif x <= 0x1F997 then
    -- CRAB..UNICORN FACE
    -- EAGLE..SQUID
    -- GIRAFFE FACE..CRICKET
    if x >= 0x1F980 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F9BF then
    -- NA   <reserved-1F998>..<reserved-1F9BF>
    if x >= 0x1F998 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F9CF then
    -- NA   <reserved-1F9C1>..<reserved-1F9CF>
    if x >= 0x1F9C1 then
      return {status="disallowed"}
    end
  elseif x <= 0x1F9E6 then
    -- FACE WITH MONOCLE..SOCKS
    if x >= 0x1F9D0 then
      return {status="disallowed"}
    end
  elseif x <= 0x1FFFF then
    -- NA   <reserved-1F9E7>..<reserved-1FFFD>
    -- <noncharacter-1FFFE>..<noncharacter-1FFFF>
    if x >= 0x1F9E7 then
      return {status="disallowed"}
    end
  elseif x <= 0x2A6D6 then
    -- CJK UNIFIED IDEOGRAPH-20000..CJK UNIFIED IDEOGRAPH-2A6D6
    if x >= 0x20000 then
      return {status="valid"}
    end
  elseif x <= 0x2A6FF then
    -- NA   <reserved-2A6D7>..<reserved-2A6FF>
    if x >= 0x2A6D7 then
      return {status="disallowed"}
    end
  elseif x <= 0x2B734 then
    -- CJK UNIFIED IDEOGRAPH-2A700..CJK UNIFIED IDEOGRAPH-2B734
    if x >= 0x2A700 then
      return {status="valid"}
    end
  elseif x <= 0x2B73F then
    -- NA   <reserved-2B735>..<reserved-2B73F>
    if x >= 0x2B735 then
      return {status="disallowed"}
    end
  elseif x <= 0x2B81D then
    -- CJK UNIFIED IDEOGRAPH-2B740..CJK UNIFIED IDEOGRAPH-2B81D
    if x >= 0x2B740 then
      return {status="valid"}
    end
  elseif x <= 0x2CEA1 then
    -- CJK UNIFIED IDEOGRAPH-2B820..CJK UNIFIED IDEOGRAPH-2CEA1
    if x >= 0x2B820 then
      return {status="valid"}
    end
  elseif x <= 0x2CEAF then
    -- NA   <reserved-2CEA2>..<reserved-2CEAF>
    if x >= 0x2CEA2 then
      return {status="disallowed"}
    end
  elseif x <= 0x2EBE0 then
    -- CJK UNIFIED IDEOGRAPH-2CEB0..CJK UNIFIED IDEOGRAPH-2EBE0
    if x >= 0x2CEB0 then
      return {status="valid"}
    end
  elseif x <= 0x2F7FF then
    -- NA   <reserved-2EBE1>..<reserved-2F7FF>
    if x >= 0x2EBE1 then
      return {status="disallowed"}
    end
  elseif x <= 0xDFFFF then
    -- NA   <reserved-2FA1E>..<reserved-2FFFD>
    -- <noncharacter-2FFFE>..<noncharacter-2FFFF>
    -- NA   <reserved-30000>..<reserved-3FFFD>
    -- <noncharacter-3FFFE>..<noncharacter-3FFFF>
    -- NA   <reserved-40000>..<reserved-4FFFD>
    -- <noncharacter-4FFFE>..<noncharacter-4FFFF>
    -- NA   <reserved-50000>..<reserved-5FFFD>
    -- <noncharacter-5FFFE>..<noncharacter-5FFFF>
    -- NA   <reserved-60000>..<reserved-6FFFD>
    -- <noncharacter-6FFFE>..<noncharacter-6FFFF>
    -- NA   <reserved-70000>..<reserved-7FFFD>
    -- <noncharacter-7FFFE>..<noncharacter-7FFFF>
    -- NA   <reserved-80000>..<reserved-8FFFD>
    -- <noncharacter-8FFFE>..<noncharacter-8FFFF>
    -- NA   <reserved-90000>..<reserved-9FFFD>
    -- <noncharacter-9FFFE>..<noncharacter-9FFFF>
    -- NA   <reserved-A0000>..<reserved-AFFFD>
    -- <noncharacter-AFFFE>..<noncharacter-AFFFF>
    -- NA   <reserved-B0000>..<reserved-BFFFD>
    -- <noncharacter-BFFFE>..<noncharacter-BFFFF>
    -- NA   <reserved-C0000>..<reserved-CFFFD>
    -- <noncharacter-CFFFE>..<noncharacter-CFFFF>
    -- NA   <reserved-D0000>..<reserved-DFFFD>
    -- <noncharacter-DFFFE>..<noncharacter-DFFFF>
    if x >= 0x2FA1E then
      return {status="disallowed"}
    end
  elseif x <= 0xE00FF then
    -- NA   <reserved-E0002>..<reserved-E001F>
    -- TAG SPACE..CANCEL TAG
    -- NA   <reserved-E0080>..<reserved-E00FF>
    if x >= 0xE0002 then
      return {status="disallowed"}
    end
  elseif x <= 0xE01EF then
    -- VARIATION SELECTOR-17..VARIATION SELECTOR-256
    if x >= 0xE0100 then
      return {status='ignored'}
    end
  elseif x <= 0xFFFFF then
    -- NA   <reserved-E01F0>..<reserved-EFFFD>
    -- <noncharacter-EFFFE>..<noncharacter-EFFFF>
    -- <private-use-F0000>..<private-use-FFFFD>
    -- <noncharacter-FFFFE>..<noncharacter-FFFFF>
    if x >= 0xE01F0 then
      return {status="disallowed"}
    end
  end
  end,
})

return _ENV
